/*
 * mod_throttle.c 2.08 for Apache 1.3
 *
 * Copyright 1999 by Anthony Howe.  All rights reserved.
 */

#define VERSION		"2.08"
#define NDEBUG		1
#define __USE_POSIX	1

#include <limits.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <pwd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <time.h>

#include <httpd.h>
#include <http_config.h>
#include <http_conf_globals.h>
#include <http_core.h>
#include <http_log.h>

#include <fcntl.h>
#include <sys/mman.h>

#ifndef _POSIX_SHARED_MEMORY_OBJECTS
#include <sys/ipc.h>
#include <sys/shm.h>
#endif

#ifndef MAX_THROTTLES
#define MAX_THROTTLES		1000
#endif

#ifndef DEFAULT_SLACK
#define DEFAULT_SLACK		3600
#endif

#ifndef DEFAULT_REFRESH
#define DEFAULT_REFRESH		60
#endif

#ifndef DEFAULT_MAXDELAY
#define DEFAULT_MAXDELAY	30
#endif

#ifndef WARN_GREEN
#define WARN_GREEN		50
#endif

#ifndef WARN_YELLOW
#define WARN_YELLOW		80
#endif

#ifndef WARN_RED
#define WARN_RED		95
#endif

#ifndef WARN_CRITICAL
#define WARN_CRITICAL		100
#endif

module throttle_module;

#define ALERT_LEVELS 	(sizeof alert_names / sizeof *alert_names)
static char *alert_names[] = { "green", "yellow", "red", "critical" };
static unsigned int default_alert[ALERT_LEVELS] = {
	WARN_GREEN, WARN_YELLOW, WARN_RED, WARN_CRITICAL
};

/*
 * Per host or user information.
 */
typedef struct throttle {
	uid_t uid;
	char *name;
	time_t started;
	unsigned int delay;
	unsigned long bytes_max;
	unsigned long bytes_sent;
	unsigned long highest;
	struct throttle *master;
} t_throttle;

static t_throttle *users;
static t_throttle *hosts;
static unsigned long slack;
static unsigned int refresh;
static unsigned int maxdelay;
static unsigned int alert[ALERT_LEVELS];

static struct {
	int id;
	pid_t pid;
	t_throttle *pool;
	unsigned int length;
} share;

static char unknown[] = "(unknown)";

#ifdef _POSIX_SHARED_MEMORY_OBJECTS

#define	my_exit(code)		exit(code)

#else

static void
my_exit(int code)
{
	if (0 < share.id)
		(void) shmctl(share.id, IPC_RMID, NULL);
	exit(code);
}

#endif

#ifdef _POSIX_SEMAPHORES
#include <semaphore.h>

static sem_t *sem;

static void
critical_cleanup(void *semp)
{
}

static void
critical_setup(server_rec *s, pool *p)
{
	sem = sem_open("/mod_throttle_sem", O_CREAT, 0666, 1);
	if (sem == (sem_t *) 0) {
		ap_log_error(APLOG_MARK, APLOG_EMERG, s, "sem_open() failed");
		my_exit(APEXIT_INIT);
	}

	ap_block_alarms();
	ap_register_cleanup(p, (void *) 0, critical_cleanup, ap_null_cleanup);
	ap_unblock_alarms();
}

static void
critical_begin(void)
{
	(void) sem_wait(sem);
}

static void
critical_end(void)
{
	(void) sem_post(sem);
}

#else /* Systme V system calls */

#include <sys/ipc.h>
#include <sys/sem.h>

#if defined(__GNU_LIBRARY__) && !defined(_SEM_SEMUN_UNDEFINED)
/* union semun is defined by including <sys/sem.h> */
#else
/* X/OPEN says we have to define it ourselves (twits). */
union semun {
	int val;			/* value for SETVAL */
	struct semid_ds *buf;		/* buffer for IPC_STAT, IPC_SET */
	unsigned short int *array;	/* array for GETALL, SETALL */
	struct seminfo *__buf;		/* buffer for IPC_INFO */
};
#endif

static struct {
	int id;
	struct sembuf on;
	struct sembuf off;
} sem;

static void
critical_cleanup(void *s)
{
	union semun ick;

	if (sem.id <= 0)
		return;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"removing semaphore #%d", sem.id
	);

	ick.val = 0;
	(void) semctl(sem.id, 0, IPC_RMID, ick);
}

static void
critical_setup(server_rec *s, pool *p)
{
	union semun ick;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"critical_setup()"
	);

	if ((sem.id = semget(0x61715f02, 1, IPC_CREAT|0777)) < 0) {
		perror("critical_setup(): semget() failed");
		my_exit(APEXIT_INIT);
	}

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"created semaphore #%d", sem.id
	);

	ick.val = 1;
	if (semctl(sem.id, 0, SETVAL, ick) < 0) {
		perror("critical_setup(): semctl(SETVAL) failed");
		my_exit(APEXIT_INIT);
	}

	ap_block_alarms();
	ap_register_cleanup(p, (void *) s, critical_cleanup, ap_null_cleanup);
	ap_unblock_alarms();

	sem.on.sem_num = 0;
	sem.on.sem_op = -1;
	sem.on.sem_flg = SEM_UNDO;
	sem.off.sem_num = 0;
	sem.off.sem_op = 1;
	sem.off.sem_flg = SEM_UNDO;
}

static void
critical_begin(void)
{
	errno = 0;
	if (semop(sem.id, &sem.on, 1) < 0) {
		perror("critical_begin() failed");
		exit(APEXIT_CHILDFATAL);
	}
}

static void
critical_end(void)
{
	errno = 0;
	if (semop(sem.id, &sem.off, 1) < 0) {
		perror("critical_end() failed");
		exit(APEXIT_CHILDFATAL);
	}
}
#endif

#ifdef _POSIX_SHARED_MEMORY_OBJECTS

static void
setup_shared_memory(server_rec *s, int size)
{
	int shm_fd;

	errno = 0;
	share.length = 0;
	share.pid = getpid();

	shm_fd = shm_open("/mod_throttle_shm", O_CREAT|O_RDWR, 0666);
	if (shm_fd < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, s,
			"shm_open('/mod_throttle.shm') failed"
		);
		my_exit(APEXIT_INIT);
	}

	share.pool = (t_throttle *) mmap(
		NULL, size, PROT_READ|PROT_WRITE, MAP_SHARED, shm_fd, 0
	);

	if (share.pool == MAP_FAILED) {
		ap_log_error(APLOG_MARK, APLOG_EMERG, s, "mmap() failed");
		my_exit(APEXIT_INIT);
	}

}

#else /* System V routines. */

static void
setup_shared_memory(server_rec *s, int size)
{
#ifdef MOVEBREAK
	char *obrk;
#endif
	errno = 0;
	share.length = 0;
	share.pid = getpid();

	if ((share.id = shmget(0x61715f01, size, IPC_CREAT|0666)) < 0) {
		ap_log_error(
			APLOG_MARK, APLOG_EMERG, s,
			"shmget(0x61715f01) failed"
		);
		my_exit(APEXIT_INIT);
	}

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"shared memory segment #%d", share.id
	);

/*
 * I found this in http_main.c, so I guess I should do likewise.
 */
#ifdef MOVEBREAK
	/*
	 * Some SysV systems place the shared segment WAY too close
	 * to the dynamic memory break point (sbrk(0)). This severely
	 * limits the use of malloc/sbrk in the program since sbrk will
	 * refuse to move past that point.
	 *
	 * To get around this, we move the break point "way up there",
	 * attach the segment and then move break back down. Ugly
	 */
	if ((obrk = sbrk(MOVEBREAK)) == (char *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_ERR, s, "sbrk() could not move break"
		);
	}
#endif

	share.pool = shmat(share.id, (char *) 0, 0);
#if 0
	(void) shmctl(share.id, IPC_RMID, NULL);
#endif
	if (share.pool == (t_throttle *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_NOERRNO|APLOG_EMERG, s,
			"shmat(%d) failed", share.id
		);
		my_exit(APEXIT_INIT);
	}

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"shared memory address %lx", share.pool
	);
#ifdef MOVEBREAK
	if (obrk != (char *) -1 && sbrk(-(MOVEBREAK)) == (char *) -1) {
		ap_log_error(
			APLOG_MARK, APLOG_ERR, s,
			"sbrk() could not move break back"
		);
	}
#endif
}

#endif /* _POSIX_SHARED_MEMORY_OBJECTS */

static void
cleanup_module(void *s)
{
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"cleanup_module()"
	);

	/* For a DSO module, the create_server() and init_module() are
	 * repeated twice within different process spaces.  The shared
	 * memory used for the throttle array will be allocated twice,
	 * though the first instance is immediately discarded when the
	 * process dies.
	 *
	 * For a static module, create_server() and init_module() are
	 * repeated twice within the SAME process.  We have to discard
	 * the first pass' bogus data in order to avoid duplicate
	 * entries, while keeping the assigned shared memory.  Simplest
	 * way to do that is just reset the throttle array length.
	 */
	share.length = 0;
}

static void
init_module(server_rec *s, pool *p)
{
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"init_module(\"%s\", %lx) %lx %d",
		s->server_hostname, p, share.pool, share.length
	);

	critical_setup(s, p);

	ap_block_alarms();
	ap_register_cleanup(p, (void *) s, cleanup_module, ap_null_cleanup);
	ap_unblock_alarms();
}

/*
 * I needed these at one point for debugging.
 */
static void
init_child(server_rec *s, pool *p)
{
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"init_child(\"%s\", %lx)", s->server_hostname, p
	);
}

static void
exit_child(server_rec *s, pool *p)
{
	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"exit_child(\"%s\", %lx)", s->server_hostname, p
	);
}

/*
 * Only one to a customer...
 */
static t_throttle *
alloc_throttle(server_rec *s)
{
	t_throttle *tp;

	if (MAX_THROTTLES <= share.length) {
		ap_log_error(
			APLOG_MARK, APLOG_NOERRNO|APLOG_EMERG, s,
			"too many throttles, max. %d", MAX_THROTTLES
		);
		my_exit(APEXIT_INIT);
	}

	/* Allocate shared memory the first time we allocate a throttle
	 * and initialise default values.
	 */
	if (share.pool == (t_throttle *) 0) {
		int i;

		slack = DEFAULT_SLACK;
		refresh = DEFAULT_REFRESH;
		maxdelay = DEFAULT_MAXDELAY;

		for (i = 0; i < ALERT_LEVELS; ++i)
			alert[i] = default_alert[i];

		/* We have to allocate shared memory in order to properly
		 * track bytes sent across ALL processes.  There is no
		 * "shared" pool support in Apache, so I've improvised some.
		 *
		 * Only allocate one large block of throttles since
		 * shared memory is a very limited resource on some
		 * systems that should not be wasted on little chunks.
		 */
		setup_shared_memory(s, MAX_THROTTLES * sizeof *share.pool);
	}

	tp = &share.pool[share.length++];

        ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"alloc'd %lx (%d)", tp, share.length-1
	);

	return tp;
}

static void *
create_server_throttle(pool *p, server_rec *s)
{
	t_throttle *tp;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"created_server_throttle(%lx, %s) %d %d %lx",
		p, s->server_hostname, share.id, share.length, share.pool
	);

	tp = alloc_throttle(s);
	tp->uid = 0;
	tp->delay = 0;
	tp->started = time((time_t *) 0) - slack;
	tp->highest = tp->bytes_sent = tp->bytes_max = 0;
	tp->name = s->server_hostname != (char *) 0
		? s->server_hostname : unknown;
	tp->master = (t_throttle *) 0;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, s,
		"exit created_server_throttle() \"%s\" %lx", tp->name, tp
	);

	return (void *) tp;
}

/*
 * ThrottleBps <bytes-per-second>
 */
static const char *
setBps(cmd_parms *cmd, void *ignore, char *num)
{
	t_throttle *tp = (t_throttle *) ap_get_module_config(
		cmd->server->module_config, &throttle_module
	);

	if (tp == (t_throttle *) 0)
		return (const char *) 0;

	tp->bytes_max = strtol(num, (char **) 0, 10);

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"server %lx %s bps %ld", tp, tp->name, tp->bytes_max
	);

	return (const char *) 0;
}

/*
 * ThrottleMaster <previous-virtual-host>
 */
static const char *
setMaster(cmd_parms *cmd, void *ignore, char *domain)
{
	t_throttle *mp, *tp = (t_throttle *) ap_get_module_config(
		cmd->server->module_config, &throttle_module
	);

	if (tp == (t_throttle *) 0)
		return (const char *) 0;

	/* Find prveiously declared <VirtualHost>. */
	for (mp = share.pool; mp != tp; ++mp) {
		if (strcmp(domain, mp->name) == 0) {
			/* The parent's throttle regulates all the children.
			 * We make a copy of the parent throttle limit for
			 * status display purposes.
			 */
			tp->bytes_max = mp->bytes_max;
			tp->master = mp;
			break;
		}
	}

	return (const char *) 0;
}

/*
 * ThrottleUser <user-id> <bytes-per-second>
 */
static const char *
setUser(cmd_parms *cmd, void *dummy, char *user, char *num)
{
	char *u;
	uid_t uid;
	t_throttle *tp;
	struct passwd *pw;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, cmd->server,
		"setUser \"%s\" %s", user, num
	);

	/* Does the user exist? */
	if (strtol(user, &u, 10) != 0 && *u == '\0') {
		uid = (uid_t) strtol(user, (char **) 0, 10);
		if ((pw = getpwuid(uid)) == (struct passwd *) 0)
			return (const char *) ap_pstrcat(
				cmd->pool, "Invalid user ID ", user, NULL
			);
	} else {
		if ((pw = getpwnam(user)) == (struct passwd *) 0)
			return (const char *) ap_pstrcat(
				cmd->pool, "Invalid user ID ", user, NULL
			);
	}

	tp = alloc_throttle(cmd->server);
	tp->master = (t_throttle *) 0;
	tp->started = time((time_t *) 0) - slack;
	tp->highest = tp->bytes_sent = 0;
	tp->delay = 0;
	tp->name = ap_pstrdup(cmd->pool, pw->pw_name);
	tp->bytes_max = strtol(num, (char **) 0, 10);
	tp->uid = pw->pw_uid;

	return (const char *) 0;
}

/*
 * ThrottleMaxDelay <seconds>
 */
static const char *
setMaxDelay(cmd_parms *cmd, void *dummy, char *arg)
{
	maxdelay = strtol(arg, (char **) 0, 10);

	return (const char *) 0;
}

/*
 * ThrottleSlack <seconds>
 */
static const char *
setSlack(cmd_parms *cmd, void *dummy, char *arg)
{
	slack = strtol(arg, (char **) 0, 10);

	return (const char *) 0;
}

/*
 * ThrottleRefresh <seconds>
 */
static const char *
setRefresh(cmd_parms *cmd, void *dummy, char *arg)
{
	refresh = (int) strtol(arg, (char **) 0, 10);

	return (const char *) 0;
}

/*
 * ThrottleIndicator <green | yellow | red | critical> <percentage>
 */
static const char *
setIndicator(cmd_parms *cmd, void *dummy, char *indicator, char *num)
{
	int i;

	for (i = 0; i < ALERT_LEVELS; ++i) {
		if (strcmp(indicator, alert_names[i]) == 0) {
			alert[i] = strtol(num, (char **) 0, 10);
			return (const char *) 0;
		}
	}

	return (const char *) ap_pstrcat(
		cmd->pool, "Invalid throttle indicator: ", indicator, NULL
	);
}

int
check_access(t_throttle *tp)
{
	unsigned long bps;
	time_t now = time((time_t *) 0);

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, NULL,
		"check_access \"%s\" %d", tp->name
	);

	/* Avoid rolling over our bps calculation. */
	if (86400 < (now - tp->started)) {
		/* prevent roll-over */
		critical_begin();
		tp->bytes_sent /= 2;
      		tp->started += (now - tp->started) / 2;
		critical_end();
	}

	bps = tp->bytes_sent / (now - tp->started);

	/* Keep track of the high water mark so that a webmaster can
	 * make adjustments.
	 */
	if (tp->highest < bps) {
		critical_begin();
		tp->highest = bps;
		critical_end();
	}

	if (tp->uid == 0 && tp->master != (t_throttle *) 0) {
		return check_access(tp->master);
	}

	if (tp->bytes_max <= 0) {
		return OK;
	}

	/* When we exceed our limit, incrementally increase the delay.
	 * Yet when fall below our limit, we can decrease the delay.
	 */
	critical_begin();
	if ((maxdelay <= 0 || tp->delay < maxdelay) && tp->bytes_max < bps)
		++tp->delay;
	else if (0 < tp->delay)
		--tp->delay;
	critical_end();

	/* Allow the delay to adjusted (hopefully downwards) before
	 * considering to refuse the connection.
	 */
	if (0 < maxdelay && maxdelay <= tp->delay)
		return HTTP_REQUEST_TIME_OUT;

	/* Muhahaha - we throttle the site if necessary. */
	if (0 < tp->delay)
		sleep(tp->delay);

	return OK;
}

#ifdef POLITE_RESPONSE
/*
 * Some people have complained that the 408 response given by
 * Apache is not very polite.
 */
int
busy(request_rec *r)
{
	ap_send_http_header(r);
	if (r->header_only)
		return;

	ap_rprintf(
		r,
		"<html>\n<body>\n<center>\n<h2>\n"
		"The requested web page is busy. "
		"<br>Please try again later.\n"
		"</h2>\n</center>\n</body>\n</html>\n"
	);
}
#endif

/*
 * Check whether we should throttle down a server.
 */
int
access_throttle(request_rec *r)
{
	int i;
	t_throttle *tp = (t_throttle *) ap_get_module_config(
		r->server->module_config, &throttle_module
	);

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		"access_throttle \"%s\"", tp->name
	);

	for (i = 0; i < share.length; ++i) {
		if (share.pool[i].uid == 0)
			continue;
		if (share.pool[i].uid == r->finfo.st_uid) {
#ifdef POLITE_RESPONSE
			if (check_access(&share.pool[i]) != OK)
				busy(r);
			return OK;
#else
			return check_access(&share.pool[i]);
#endif
		}
	}

#ifdef POLITE_RESPONSE
	if (check_access(tp) != OK)
		busy(r);
	return OK;
#else
	return check_access(tp);
#endif
}

void
track_bytes_sent(request_rec *r, t_throttle *tp, unsigned long sent)
{
	int i;
	t_throttle *up;

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, r->server,
		"track_bytes_sent \"%s\"", tp->name
	);

	for (i = 0; i < share.length; ++i) {
		if (share.pool[i].uid == 0)
			continue;
		if (share.pool[i].uid == r->finfo.st_uid) {
			critical_begin();
			share.pool[i].bytes_sent += r->bytes_sent;
			critical_end();
			return;
		}
	}

	critical_begin();
	tp->bytes_sent += sent;
	critical_end();

	/* Is another virtual host responsible? */
	if (tp->master != (t_throttle *) 0)
		track_bytes_sent(r, tp->master, sent);
}

/*
 * Track the total number of bytes sent.
 */
int
log_throttle(request_rec *orig)
{
	request_rec *r;

	t_throttle *tp = (t_throttle *) ap_get_module_config(
		orig->server->module_config, &throttle_module
	);

	ap_log_error(
		APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, orig->server,
		"log_throttle \"%s\"", tp->name
	);

	/* The final sub-request holds the actual number of bytes sent. */
	for (r = orig ; r->next != (request_rec *) 0; r = r->next)
		;

	track_bytes_sent(r, tp, r->bytes_sent);

	return OK;
}

/*
 * Display throttling status
 */
static int
throttle_handler(request_rec *r)
{
	int row;
	char *cmd;
	time_t now = time((time_t *) 0);

	static struct {
		char *tag, *etag;
	} fonts[] = {
		{ "<font color=\"#009900\">", "</font>" },
		{ "<font color=\"#ff9900\">", "</font>" },
		{ "<font color=\"#cc0000\">", "</font>" },
#ifdef BLINK
		{ "<blink><b><font color=\"#ff0000\">", "</font><b></blink>" },
#else
		{ "<b><font color=\"#ff0000\">", "</font><b>" },
#endif
	};


	t_throttle *tp = (t_throttle *) ap_get_module_config(
		r->server->module_config, &throttle_module
	);

	r->allowed = (1 << M_GET);
	if (r->method_number != M_GET)
		return DECLINED;

	r->content_type = "text/html";

	if (r->args != (char *) 0
	&& (cmd = strstr(r->args, "refresh")) != (char *) 0) {
		if (cmd[strlen("refresh")] == '=') {
			ap_table_set(
				r->headers_out, "Refresh",
				cmd + strlen("refresh") + 1
			);
		} else {
			ap_table_set(r->headers_out, "Refresh", "1");
		}
	} else {
		char num[8];
		sprintf(num, "%u", refresh);
		ap_table_set(r->headers_out, "Refresh", num);
	}

	ap_send_http_header(r);
	if (r->header_only)
		return OK;

	ap_rprintf(
		r,
		"<html>\n<head>\n<title>Throttle Status</title>\n</head>\n"
		"<body bgcolor=\"#ffffff\" text=\"#000000\">\n"
		"<center>\n"
		"<h3>Throttle Status List</h3>\n"
		"<table cellpadding=1 cellspacing=0 width=100%>\n"
		"<tr>\n"
			"\t<th>&nbsp;</th>\n"
			"\t<th rowspan=2>Server</th>\n"
			"\t<th colspan=2>Current</th>\n"
			"\t<th>Allowed</th>\n"
			"\t<th colspan=2>Highest</th>\n"
			"\t<th rowspan=2>Delay</th>\n"
		"</tr><tr>\n"
			"\t<th>&nbsp;</th>\n"
			"\t<th>%%</th>\n"
			"\t<th>Bytes/s</th>\n"
			"\t<th>Bytes/s</th>\n"
			"\t<th>Bytes/s</th>\n"
			"\t<th>GB/month</th>\n"
		"</tr>\n"
	);

	for (row = 0; row < share.length; ++row) {
		int lvl;
		unsigned long bps, percent;
		t_throttle *tp = &share.pool[row];

		bps = tp->bytes_sent / (now - tp->started);

		percent = 0;
		if (0 < tp->bytes_max)
			percent = bps * 100 / tp->bytes_max;

		for (lvl = 0 ; lvl < ALERT_LEVELS-1; ++lvl)
			if (percent < alert[lvl])
				break;

		ap_rprintf(r,
			"<tr align=right%s>\n"
				"\t<td>%d</td>\n"
				"\t<td align=left>&nbsp;",
			(row & 1) ? "" : " bgcolor=\"#cccccc\"", row
		);

		if (tp->uid == 0) {
			ap_rprintf(r,
				"<a href=\"http://%s/\">%s</a>",
				tp->name, tp->name
			);
		} else {
			ap_rprintf(r, "%s", tp->name);
		}

		ap_rprintf(r,
				"</td>\n"
				"\t<td><b>%s%lu%s</b></td>\n"
				"\t<td><b>%s%lu%s</b></td>\n"
				"\t<td><b>%s%lu%s</b></td>\n"
				"\t<td><b>%s%lu%s</b></td>\n"
				"\t<td><b>%s%6.2f%s</b></td>\n"
				"\t<td><b>%s%u%s</b></td>\n"
			"</tr>\n",
			fonts[lvl].tag, percent, fonts[lvl].etag,
			fonts[lvl].tag, bps, fonts[lvl].etag,
			fonts[lvl].tag, tp->bytes_max, fonts[lvl].etag,
			fonts[lvl].tag, tp->highest, fonts[lvl].etag,
			fonts[lvl].tag, tp->highest * 0.0026297, fonts[lvl].etag,
			fonts[lvl].tag, tp->delay, fonts[lvl].etag
		);
	}

    	ap_rprintf(
		r, "</table>\n<p><font size=-1>"
		"mod_throttle.c "
		VERSION
		"<br>Copyright 1999 by Anthony Howe.  All rights reserved."
		"</font></center>\n</body>\n</html>\n"
	);

	return OK;
}

handler_rec throttle_handlers[] = {
	{ "throttle-info", throttle_handler },
	{ NULL }
};

command_rec throttle_cmds[] = {
	{ "ThrottleMaxDelay", setMaxDelay, NULL, RSRC_CONF, TAKE1,
	"Set max. delay threshold in seconds before refusing connections." },

	{ "ThrottleIndicator", setIndicator, NULL, RSRC_CONF, TAKE2,
	"Set % threshold for green, yellow, red, and critical alerts." },

	{ "ThrottleSlack", setSlack, NULL, RSRC_CONF, TAKE1,
	"Startup slack in seconds to prevent initial overload." },

	{ "ThrottleUser", setUser, NULL, RSRC_CONF, TAKE2,
	"Max. bytes per second allowed for user, or zero to monitor." },

	{ "ThrottleBps", setBps, NULL, RSRC_CONF, TAKE1,
	"Max. bytes per second allowed for server, or zero to monitor." },

	{ "ThrottleMaster", setMaster, NULL, RSRC_CONF, TAKE1,
	"Throttle controlled from another virtual host." },

	{ "ThrottleRefresh", setRefresh, NULL, RSRC_CONF, TAKE1,
	"Refresh time in seconds, or use /throttle?refresh=sec" },

	{ NULL }
};

module throttle_module = {
	STANDARD_MODULE_STUFF,
	init_module,		/* initializer */
	NULL,	          	/* dir config creater */
	NULL,                  	/* dir merger --- default is to override */
	create_server_throttle,	/* server config */
	NULL,          		/* merge server config */
	throttle_cmds,         	/* command table */
	throttle_handlers,     	/* handlers */
	NULL,                  	/* filename translation */
	NULL,                  	/* check_user_id */
	NULL,                  	/* check auth */
	access_throttle,       	/* check access */
	NULL,                  	/* type_checker */
	NULL,                  	/* fixups */
	log_throttle,	     	/* logger */
#if MODULE_MAGIC_NUMBER >= 19970103
	NULL,                  	/* header parser */
#endif
#if MODULE_MAGIC_NUMBER >= 19970719
	init_child,            	/* child_init */
#endif
#if MODULE_MAGIC_NUMBER >= 19970728
	exit_child,            	/* child_exit */
#endif
#if MODULE_MAGIC_NUMBER >= 19970902
	NULL                   	/* post read-request */
#endif
};
