/* apt-spy (c) Steven Holmes, 2003. This software is licensed as detailed in the LICENSE file. */

void usage(void);
void version(void);

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>

#include "include/update.h"
#include "include/file.h"
#include "include/parse.h"
#include "include/benchmark.h"

/* Our version number. */
const char apt_spy_v[] = "v3.0-3";

/* Defaults */

/* The default area */
char d_area[] = "All";

/* Default file locations */
char d_out[] = "/etc/apt/sources.list";
char d_config[] = "/etc/apt-spy.conf";
char d_mirror[] = "/usr/share/apt-spy/README.mirrors.txt";

/* Default file to grab when benchmarking */
char d_file[] = "ls-lR";

/* Default update URL */
char d_update_url[] = "http://http.us.debian.org/debian/README.mirrors.txt";

int main(int argc, char *argv[])
{
	int c;
	char *cur_entry;		/* Entry we are benchmarking */

	char *distrib = NULL;		/* distrubtion to use. */
	char *mirror_list = NULL;	/* mirror list file */
	char *config_file = NULL;	/* configuraion file */
	char *proxy = NULL;		/* Proxy server to use */
	char *infile = NULL;		/* optional infile */
	char *outfile = NULL;		/* outfile name */
	char *area = NULL;		/* Area to test */
	char *grab_file = NULL;		/* File to grab */
	char *update_url = NULL;	/* URL to use for updating */
	char *country_list = NULL;	/* List of countries to b/m */
	FILE *infile_p, *outfile_p;	/* input/output file pointers */
	FILE *config_p;			/* config file pointer */
	FILE *mirror_p;			/* mirror list pointer */
	int timeout = 15;		/* time to benchmark each server */
	int toplist = 0;		/* Whether to write toplist. */

	/* Number of servers to test. If negative, test them all. */
	int test_number = -1;		

	/* Server information structures. */
	server_t current, best[BESTNUMBER];

	/* Parse options... */
	while((c = getopt(argc, argv, "a:c:d:e:f:i:m:o:p:s:t:u:w:vh")) != -1)
		switch(c) {
		/* Area to benchmark */
		case 'a':
			area = optarg;
			break;
		/* Distribution we'll write into apt-sources. */
		case 'd':
			distrib = optarg;
			break;
		/* Configuration file to use */
		case 'c':
			config_file = optarg;
			break;
		/* Number of servers to benchmark */
		case 'e':
			test_number = atoi(optarg);
			break;
		/* File, relative to Debian base, to grab from server. */
		case 'f':
			grab_file = optarg;
			break;
		/* User-specified list of servers to benchmark. */
		case 'i':
			infile = optarg;
			break;
		/* The list of mirrors */
		case 'm':
			mirror_list = optarg;
			break;
		/* The output file we use */
		case 'o':
			outfile = optarg;
			break;
		/* Proxy server we should use */
		case 'p':
			proxy = optarg;
			break;
		/* List of countries to benchmark */
		case 's':
			country_list = optarg;
			break;
		/* Time to bencmark each server for. */
		case 't':
			timeout = atoi(optarg);
			break;
		/* The URL we should update ourselves from */					
		case 'u':
			update_url = optarg;
			break;
		/* Should we write a list of the "top" servers? */
		case 'w':
			toplist = 1;
			outfile = optarg;
			break;
		case 'v':
			version();
			break;			
		/* Help!! */
		case 'h':
		default:
			usage();			/* display help */
		}
		
	argc -= optind;
	argv += optind;

	/* We require an area and distribution argument if we are not updating */
	if ((argc == 0) && (distrib == NULL))
		usage();

	/* Check for silly combination of country and area arguments */
	if ((area != NULL) && (country_list != NULL))
		usage();

	/* Setup default area argument */
	if ((area == NULL) && (country_list == NULL))
		area = d_area;

	/* Setup default file argument if none given */
	if (grab_file == NULL)
		grab_file = d_file;

	/* Open mirror file. We pass argc so it can tell if we're updating 
	   or not */
	mirror_p = select_mirror(mirror_list, argc);
	if (mirror_p == NULL) {
		perror("fopen");
		fprintf(stderr, "Error opening mirror file. Exiting.\n");
		exit(1);
	}

	/* the only possible argument now is "update", for updating the 
	   mirrors list */
	if (argc == 1) {
		if (strcmp(argv[0], "update") != 0)
		usage();

                /* If necessary, set update_url to default */
		if (update_url == NULL)
			update_url = d_update_url;

		if (update(mirror_p, update_url, proxy) != 0) {
			fprintf(stderr, "Update failed. Exiting.\n");
			exit(1);
		}
		printf("Update complete. Exiting.\n");
		exit(0);
	}
	
	/* argc should be 0. If not, there's something wrong. */
	if (argc != 0)
		usage();
	                                

	/* We open the infile. Either a temporary file, or a user-specified 
	   one. */
	infile_p = select_infile(infile);
	if (infile_p == NULL) {
		perror("tmpfile");
		fprintf(stderr, "Failed to open infile. Exiting.\n");
		exit(1);
	}

	/* Set up default if user hasn't specified an outfile */
	if (outfile == NULL)
		outfile = d_out;

	/* Check output file for accessibility */
	if (check_write_access(outfile) == 1) {
		fprintf(stderr, "Could not open outfile. Exiting.\n");
		exit(1);
	}

	/* Open config file */
	config_p = select_config(config_file);
	if (config_p == NULL) {
		perror("fopen");
		fprintf(stderr, "Error opening config file. Exiting.\n");
		exit(1);
	}

	/* Fill temporary file with useful stuff if it's not user-specified. */
	if (infile == NULL) {
		if (area != NULL) {
			if (build_area_file(config_p, infile_p, mirror_p, area) != 0) {
				fprintf(stderr, 
				"Error building area file. Exiting.\n");
				exit(1);
			}
		} else {
			if (build_country_file(config_p, infile_p, mirror_p, country_list) != 0) {
				fprintf(stderr,
				"Error building country file. Exiting.\n");
				exit(1);
			}
		}
	}
			
	
	/* Make sure we're at the beginning... */
	rewind(infile_p);

	/* Zero the "best" structure */
	for (c = 0; c < BESTNUMBER; c++)
		memset(&best[c], 0, sizeof(server_t));

	/* This is the main loop. It'll exit when we've exhausted the URL 
	   list or test_number is 0 */

	while (((cur_entry = next_entry(infile_p)) != NULL) && test_number != 0) {
		if (ferror(infile_p) != 0) {
			fprintf(stderr, "Error while reading input file\n");
			exit(1);
		}

		/* Turn entry into a pretty structure */
		tokenise(&current, cur_entry);

		/* Do the benchmark */
		if (benchmark(&current, proxy, timeout, grab_file) != 0) {
			fprintf(stderr, 
			"Error while performing benchmark. Exiting.\n");
			exit(1);
		}

		decide_best(&current, best);
		free(cur_entry);

		if (test_number > 0)
			--test_number;
	}

	/* Open the output file... */
	outfile_p = select_outfile(outfile);
	if (outfile_p == NULL) {
		perror("fopen");
		fprintf(stderr, "Error opening output file. Exiting.\n");
		exit(1);
	}

	/* If infile is NULL (not user specified), we just write the sources 
	   list ;) */
	if (toplist == 0) {
		if (write_list(outfile_p, best, distrib) != 0) {
			fprintf(stderr, "Error writing output file. Exiting.");
			exit(1);
		}
	}
	/* Else we write out the top 5 servers to a file */
	else {
		if (write_top(infile_p, outfile_p, best) != 0) {
			fprintf(stderr, 
			"Error writing top servers list. Exiting.");
			exit(1);
		}
	}
	/* We're all done */
	exit(0);
}

void usage()
{
	printf("Usage: apt-spy [options]\n"
		"options:\n"
		"-d distribution\tDebian distribution (ie, stable). Required unless updating.\n"
		"-a area\t\tArea to benchmark. (eg, Europe).\n"
		"-c config\tConfiguration file to use.\n"
		"-e number\tNumber of servers to benchmark before exiting.\n");
	printf("-f file\t\tFile to grab when benchmarking. (relative to Debian base).\n"
		"-i file\t\tSpecify input file. For use with the -w option.\n"
		"-m mirror-list\tMirror list to use, or mirror-list to update when updating.\n"
		"-o output-file\tWhere to put output.\n"
		"-p proxy\tProxy server to use. In format <server>:[port]\n"
		"-s country_list\tList of countries to benchmark. Cannot be used with -a.\n"
		"-t time\t\tTime to benchmark each server for. An approximate value only.\n"
		"-u update-URL\tURL to grab mirror-list from when updating.\n");
	printf("-w file\t\tOutput top 5 servers to file for use with -i.\n"
		"-v\t\tOutput a version number.\n"
		"-h\t\tDisplay this message.\n"
		"update\t\tUpdate the mirror list.\n");
	exit(0);
}

void version()
{
	printf("apt-spy %s\n", apt_spy_v);
	exit(0);
}
