# library-branches.sh: list branches in the revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "list branches in the revision library\\n"
                printf "usage: library-branches [options] [[archive]/category]"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " -r --reverse                  reverse the listing\\n"
                printf "\\n"
		printf "List all branches within a particular archive/category with\\n"
		printf "records in the revision library.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-find: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-find: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -*)                 printf "library-branches: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: library-branches [options] [[archive]/category]" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  catspec="$1"
  shift
fi

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library -e library-branches`"

if test ! -z "$catspec" ; then 
  larch valid-package-name -e library-branches --basename "$catspec"
  archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$catspec\"`"
  category="`larch parse-package-name --basename \"$catspec\"`"
else
  tree_default="`larch tree-version`"
  archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$tree_default\"`"
  category="`larch parse-package-name --basename \"$tree_default\"`"
fi

################################################################
# Do It
# 

cd "$library"

if test ! -d "$archive/$category" ; then
  printf "\\n" 1>&2
  printf "library-branches: category not found in library\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "  category: %s\\n" "$category" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

cd "$archive/$category"

for f in `ls | sort $reverse` ; do
  if larch valid-package-name "$f" ; then
    printf "%s\\n" "$f"
  fi
done

# tag: Tom Lord Sun Jan 20 01:03:41 2002 (library/library-branches.sh)
#
