# make-library-browser.sh: create or update a library browser
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "create or update a library browser\\n"
                printf "usage: make-library-browser [options] output-dir [limit]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
                printf " -t --title TITLE              set the page title\\n"
		printf " --dists FILE                  specify a distributions spec file\\n"
		printf " --configs FILE                specify a configurations spec file\\n"
                printf "\\n"
                printf "Create or update web pages describing a revision library.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

title=
dists=
configs=

quiet=--quiet
report=--report
verbose=
silent_opt=
quiet_opt=
report_opt=
verbose_opt=
debug_opt=
presume_lock=

while test $# -ne 0 ; do

  case "$1" in 

    ----presume-lock)	shift
    			presume_lock=----presume-lock
			;;

    --silent)	shift
    		quiet=
		report=
		verbose=
		silent_opt=--silent
		quiet_opt=
		report_opt=
		verbose_opt=
		;;

    --quiet)	shift
    		quiet=--quiet
		report=
		verbose=
		silent_opt=
		quiet_opt=--quiet
		report_opt=
		verbose_opt=
		;;

    --report)	shift
    		quiet=--quiet
		report=--report
		verbose=
		silent_opt=
		quiet_opt=
		report_opt=--report
		verbose_opt=
		;;

    --verbose)	shift
    		quiet=--quiet
		report=--report
		verbose=--verbose
		silent_opt=
		quiet_opt=
		report_opt=
		verbose_opt=--verbose
		;;

    --debug)	shift
    		larch heading "make-library-browser: debugging output enabled\\n"
    		set -x
		debug_opt=--debug
		;;

    -t|--title)         shift
                        if test $# -eq 0 ; then
                          printf "make-library-browser: -t and --title require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        title="$1"
                        shift
                        ;;

    --dists)         	shift
                        if test $# -eq 0 ; then
                          printf "make-library-browser: --dists require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        dists="$1"
                        shift
                        ;;

    --configs)         	shift
                        if test $# -eq 0 ; then
                          printf "make-library-browser: --configs require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        configs="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;
			
    -*)                 printf "make-library-browser: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test "(" $# -lt 1 ")" -o "(" $# -gt 2 ")" ; then
  printf "usage: make-library-browser [options] output-dir [limit]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

output=$1
shift

if test $# -ne 0 ; then
  limit="$1"
  shift
else
  limit=
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"

if test ! -z "$dists" ; then
  cd "`dirname \"$dists\"`"
  dists="`pwd`/`basename \"$dists\"`"
  cd "$here"
fi

if test ! -z "$configs" ; then
  cd "`dirname \"$configs\"`"
  configs="`pwd`/`basename \"$configs\"`"
  cd "$here"
fi

cd "$output"
output="`pwd`"

library="`larch my-revision-library`"

if test -z "$title" ; then
  title="Library Browser"
fi

if test ! -z "$limit" ; then

  larch valid-package-name -e make-library-browser --basename --tolerant "$limit"

  limit_archive="`larch parse-package-name -R \"$archroot\" -A \"$archive\" --arch \"$limit\"`"
  limit_category="`larch parse-package-name --basename \"$limit\"`"

  limit_branch=
  limit_version=

  if larch valid-package-name --tolerant "$limit" ; then

    limit_branch="`larch parse-package-name \"$limit\"`"

    if larch valid-package-name --vsn "$limit" ; then

      limit_version="`larch parse-package-name --package-version \"$limit\"`"

     fi
  fi

fi

cd "$output"

if test ! -e "=browser" ; then
  printf "\\n" 1>&2
  printf "make-library-browser: output directory is not a browser\\n" 1>&2
  printf "\\n" 1>&2
  printf "  output: %s\\n" "$output" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

if test -z "$presume_lock" ; then 

  if test ! -z "$quiet" ; then
    larch heading "make-library-browser -- restarting with file lock"
  fi

  exec with-file-lock =browser \
	larch make-library-browser ----presume-lock \
				  $silent_opt $quiet_opt $report_opt $verbose_opt $debug_opt \
				  --title "$title" --dists "$dists" --configs "$configs" \
				  "$output" "$limit"

fi

################################################################
# Greetings
# 
if test ! -z "$quiet" ; then
  larch heading "make-library-browser\\n"
  printf "arguments: %s\\n" "$command_line" | fold -w 60 | larch body-indent
  larch heading --sub "make-library-browser start time: %s\\n" "`date`"
  larch heading --sub "output dir: %s\\n" "$output"
  larch heading --sub "library: %s\\n" "`larch my-revision-library`"
  if test ! -z "$limit" ; then
    larch heading --sub "limit: %s\\n" "$limit"
  fi
fi

################################################################
# Start the Index
# 
# 

html_quote()
{
  sed -e '{
	    s/&/\&amp;/g
	    s/</\&lt;/g
	    s/>/\&gt;/g
	    s/"/\&quot;/g
	  }'
}

sed_subst_quote()
{
  sed -e '{
  	    s/\\/\\\\/g
	    s,/,\\/,g
	  }'
}


rm -f library-root
ln -s "$library" library-root

larch library-archives > ,,archives

if test ! -z "$dists" ; then

  if test ! -z "$quiet" ; then
    larch heading --sub "computing distributions history\\n"
  fi

  mkdir -p dists

  cat "$dists" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | awk -v uniq="$$" \
        -v default_archive="`larch my-default-archive`" \
	'{
           dist_name = $1;
           config = $2;
           package = $3;

	   cache_file = "dists/=cache." dist_name;
	   cache_file_tmp = "dists/,,=cache." dist_name "." uniq;
	   cache_package_rev = "";

	   history_cmd = "larch config-history --reverse " config " " package;
	   latest_package_rev = "";

	   if ((0 < (getline < cache_file)) && ($1 == dist_name))
	     {
	       cache_dist_name = $1;
	       cache_package_rev = $2;
	       latest_cmd = "larch library-revisions --reverse --full " package " | head -n 1";
	       latest_cmd | getline latest_package_rev;
	       close(latest_cmd);
	       arch = package;
	       sub("/.*", "", arch);
	       if (arch == package)
	         arch = default_archive;
	       latest_package_rev = arch "/" latest_package_rev;
	     }
	   
	   if ((cache_package_rev != "") && (cache_package_rev == latest_package_rev))
	     {
	       while (getline < cache_file)
	         {
		   print $0;
		 }
	       close (cache_file);
	     }
	   else
	     {
	       close(cache_file);
	       print dist_name " " latest_package_rev > cache_file_tmp;

               while (history_cmd | getline)
                 {
                   package_vsn = $1;
		   included_rev = $2;
		   ("larch distribution-name --dist " dist_name " " package_vsn) | getline distribution;
		   printf("%s:%s\n", included_rev, distribution);
		   printf("%s:%s\n", included_rev, distribution) > cache_file_tmp;
		 }
		 close (cache_file_tmp);
		 system ("rm -f " cache_file);
		 system ("mv " cache_file_tmp " " cache_file);
             }

	   close(history_cmd);
         }' \
  > ,,distributions

fi

if test ! -z "$quiet" ; then
  larch heading --sub "building web pages\\n"
fi


# archive -> categories -> branches -> versions 	 	    -> revisions
#						-> creators
#						-> files
#
#   Y		G	      B		  R	     Y/P/S	     ?
#
archive_page_bg="#515131"
archive_table_bg="#e6e6b1"
archive_headers_bg="#bebebe"

distributions_page_bg="#9e9eff"
distributions_table_bg="#befefe"
distributions_headers_bg="#bebebe"

configs_page_bg="#9edebe"
configs_table_bg="#befefe"
configs_headers_bg="#bebebe"

distribution_page_bg="#8e2e8e"
distribution_table_bg="#febefe"
distribution_headers_bg="#bebebe"

categories_page_bg="#206020"
categories_table_bg="#befebe"
categories_headers_bg="#bebebe"

branches_page_bg="#5e5eaf"
branches_table_bg="#befefe"
branches_headers_bg="#bebebe"

versions_page_bg="#8e1e1e"
versions_table_bg="#febebe"
versions_headers_bg="#bebebe"

creators_page_bg="#103030"
creators_table_bg="#befebe"
creators_headers_bg="#bebebe"

revisions_page_bg="#818181"
revisions_table_bg="#f6e6f6"
revisions_headers_bg="#bebebe"
revisions_notes_color="#4e1010"

files_page_bg="#81f181"
files_table_bg="#e6e6f6"


html_title="`printf \"%s\" \"$title\" | html_quote`"
sed_title="`printf \"%s\" \"$html_title\" | sed_subst_quote`"

# index.html, browser-top.html, browser-right.html, browser-display.html
# 
sed -e "s/@TITLE@/$sed_title/g" "$ARCH_SUBCMD_ROOT/web/browser.html" > index.html
cp  "$ARCH_SUBCMD_ROOT/web/browser-top.html" browser-top.html
cp "$ARCH_SUBCMD_ROOT/web/browser-display.html" browser-display.html
cp "$ARCH_SUBCMD_ROOT/web/browser-help.html" browser-help.html
cp "$ARCH_SUBCMD_ROOT/web/browser-help-categories.html" browser-help-categories.html
cp "$ARCH_SUBCMD_ROOT/web/browser-help-branches.html" browser-help-branches.html
cp "$ARCH_SUBCMD_ROOT/web/browser-help-versions.html" browser-help-versions.html
cp "$ARCH_SUBCMD_ROOT/web/browser-help-creators.html" browser-help-creators.html

cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
| sed -e "s/@TITLE@/$sed_title/" \
      -e "s/@TOP@//" \
      -e "s/@ARCHIVE@//" \
      -e "s/@CATEGORY@//" \
      -e "s/@BRANCH@//" \
      -e "s/@VERSION@//" \
      -e "s/@HELPFILE@/browser-help.html/" \
> browser-nav.html

cat "$ARCH_SUBCMD_ROOT/web/browser-right.html" \
| sed -e "s/bgcolor=\"#bebebe\"/bgcolor=\"$categories_page_bg\"/" \
> browser-right.html

# browser-left.html -- the archive index
#

printf "<HTML>\\n" > browser-left.html
printf "\\n" >> browser-left.html
printf "<HEAD><TITLE>library catalog</TITLE></HEAD>\\n" >> browser-left.html
printf "\\n" >> browser-left.html
printf "<BODY bgcolor=\"%s\">\\n" "$archive_page_bg" >> browser-left.html
printf "\\n" >> browser-left.html

larch library-archives \
| awk -v title="cataloged archives" \
      -v library_root="library-root" \
      -v right_root="./repos" \
      -v link_stem="categories" \
      -v back="" \
      -v page_bg="$archive_page_bg" \
      -v table_bg="$archive_table_bg" \
      -v headers_bg="$archive_headers_bg" \
      -f "$ARCH_SUBCMD_ROOT/web/left-table.awk" \
>> browser-left.html


if test ! -z "$configs" ; then
  printf "\\n" >> browser-left.html
  printf "<HR>\\n" >> browser-left.html
  printf "\\n" >> browser-left.html

  cat "$configs" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | awk '{ print $1 }' \
  | awk -v title="cataloged configurations" \
	-v right_root="./configs" \
        -v link_stem="contents" \
        -v page_bg="$configs_page_bg" \
        -v table_bg="$configs_table_bg" \
        -v headers_bg="$configs_headers_bg" \
	-f "$ARCH_SUBCMD_ROOT/web/left-table.awk" \
  >> browser-left.html
fi

if test ! -z "$dists" ; then

  printf "\\n" >> browser-left.html
  printf "<HR>\\n" >> browser-left.html
  printf "\\n" >> browser-left.html

  cat "$dists" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | awk '{ print $1 " " $4 }' \
  | awk -v title="cataloged distributions" \
	-v right_root="./dists" \
        -v link_stem="contents" \
        -v page_bg="$distributions_page_bg" \
        -v table_bg="$distributions_table_bg" \
        -v headers_bg="$distributions_headers_bg" \
	-v about=1 \
	-f "$ARCH_SUBCMD_ROOT/web/left-table.awk" \
  >> browser-left.html
fi


printf "\\n" >> browser-left.html
printf "</BODY>\\n" >> browser-left.html
printf "\\n" >> browser-left.html
printf "</HTML>\\n" >> browser-left.html
printf "\\n" >> browser-left.html

if test ! -z "$dists" ; then
  mkdir -p dists
  
  cd "dists"
  
  cat "$dists" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | sort -k 1,1 \
  > ,,dists.sorted

  for distribution in `cat "$dists" | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces | awk '{print $1}'` ; do
  
    if test ! -z "$quiet" ; then
      larch heading --sub --sub "building distribution web pages: %s\\n" "$distribution"
    fi

    mkdir -p "$output/dists/$distribution"
    cd "$output/dists/$distribution"

    package="`printf '%s\n' \"$distribution\" | join -o 2.3 -1 1 -2 1 - ../,,dists.sorted | head -n 1`"
    config="`printf '%s\n' \"$distribution\" | join -o 2.2 -1 1 -2 1 - ../,,dists.sorted | head -n 1`"
  
    cat "$output/,,distributions" \
    | awk -v dist="$distribution" \
          '{
              d = $1;
              sub(".*:", "", d);
              name = substr(d, 1, length(dist));
              suffix = substr(d, 1 + length(dist));
  
              if ((name == dist) && ((suffix == "") || (match(suffix, "^-[[:digit:]]+\.[[:digit:]]+(pre[[:digit:]]+|\.[[:digit:]]+)?$"))))
                {
                  print $0;
                }
            }' \
    | awk -v title="$distribution distributions" \
          -v browser_root="../.." \
          -v page_bg="$distribution_page_bg" \
          -v table_bg="$distribution_table_bg" \
          -v headers_bg="$distribution_headers_bg" \
	  -v package="$package" \
	  -v config="$config" \
          -f "$ARCH_SUBCMD_ROOT/web/distribution-right-index.awk" \
    > contents-right.html
  
  done

fi

if test ! -z "$configs" ; then

  mkdir -p "$output/configs"
  
  cd "$output/configs"
  
  cat "$configs" \
  | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
  | sort -k 1,1 \
  > ,,configs.sorted


  for configuration in `cat "$configs" | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces | awk '{print $1}'` ; do
  
    if test ! -z "$quiet" ; then
      larch heading --sub --sub "building configuration web pages: %s\\n" "$configuration"
    fi
  
    mkdir -p "$output/configs/$configuration"
    cd "$output/configs/$configuration"

    package="`printf '%s\n' \"$configuration\" | join -o 2.2 -1 1 -2 1 - ../,,configs.sorted | head -n 1`"
    config="`printf '%s\n' \"$configuration\" | join -o 2.3 -1 1 -2 1 - ../,,configs.sorted | head -n 1`"
  
    archive="`larch parse-package-name --arch \"$package\"`"
    package="`larch parse-package-name --non-arch \"$package\"`"
    revision="`larch library-revisions --reverse --full \"$archive/$package\" | head -n 1`"
    category="`larch parse-package-name --basename \"$revision\"`"
    branch="`larch parse-package-name \"$revision\"`"
    version="`larch parse-package-name --package-version \"$revision\"`"
    lvl="`larch parse-package-name --lvl \"$revision\"`"

    file="contents-right.html"

    printf "<HTML>\\n" > "$file"
    printf "\\n" >> "$file"
    printf "<HEAD><TITLE>%s</TITLE></HEAD>\\n" "configuration $config" >> "$file"
    printf "\\n" >> "$file"
    printf "<BODY text=\"#00ff00\" link=\"#aaffaa\" alink=\"#eeeeff\" vlink=\"#aaffaa\" bgcolor=\"#000000\">\\n" >> "$file"
    printf "\\n" >> "$file"
    printf "\\n" >> "$file"

    printf "<p align=\"center\"><b>%s</b>\\n" "$configuration" >> "$file"
    printf "\\n" >> "$file"
    printf "<p>" >> "$file"
    printf "<small>root package:</small>\\n" >> "$file"
    printf "<a target=\"display\" href=\"../../repos/%s/cats/%s/branches/%s/vsns/%s/patches-full.d/loc-%s.html#%s\">\\n" \
	   "$archive" "$category" "$branch" "$version" "$lvl" "$lvl" \
    >> "$file"
    printf "<code>%s</code>\\n" "$archive/$revision" >> "$file"
    printf "</a>\\n" >> "$file"
    printf "<br>\\n" >> "$file"
    printf "<small>config name:</small>\\n" >> "$file"
    printf "<a target=\"display\" type=\"text/plain\" href=\"../../library-root/%s/%s/%s/%s/%s/configs/%s\">\\n" \
	   "$archive" "$category" "$branch" "$version" "$revision" "$config" \
    >> "$file"
    printf "<code>%s</code>\\n" "$config" >> "$file"
    printf "</a>\\n" >> "$file"
    printf "<hr>\\n" >> "$file"

    printf "\\n" >> "$file"
    printf "\\n" >> "$file"
    printf "<table width=\"100%%\" cellspacing=\"0\" cellpadding=\"0\" border=\"3\">\\n" >> "$file"
    printf "\\n" >> "$file"

    cat "`larch library-find \"$archive/$revision\"`/configs/$config" \
    | larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
    | awk \
	  '{
	      loc = $1;
	      pkg = $2;

	      cmd = "larch library-revisions --reverse --full " pkg " | head -n 1";
	      cmd | getline rev;
	      close(cmd);

	      a = pkg;
	      sub("/.*", "", a);

	      r = rev;
	      sub(".*/", r);

	      v = r;
	      sub("--[^-]*-[^-]*$", "", v);

	      b = v;
	      sub("--[^-]*$", "", b);

	      c = b;
	      sub("--.*$", "", c);

	      l = r;
	      sub(".*--", "", l);

	      print "<tr>"
	      print "<td>"
	      print "<font color=\"#00ff00\">"
	      print "<code>" loc "</code>"
	      print "</td>"
	      print "<td>"
	      print "<a target=\"display\" href=\"../../repos/" a "/cats/" c "/branches/" b "/vsns/" v "/patches-full.d/loc-" l ".html#" l "\">";
	      print "<code>" rev "</code></a>";
	      print "</td>"
	      print "</tr>"
	   }' \
    >> "$file"

    printf "</table>\\n" >> "$file"
    printf "</BODY>\\n" >> "$file"
    printf "</HEAD>\\n" >> "$file"
    printf "\\n" >> "$file"
    
  done

fi


cd "$output"
mkdir -p repos

if test ! -z "$limit_archive" ; then
  archive_list="$limit_archive"
else
  archive_list="`larch library-archives`"
fi

for archive in $archive_list  ; do

  if test ! -z "$quiet" ; then
    larch heading --sub --sub "archive: %s\\n" "$archive"
  fi

  mkdir -p "$output/repos/$archive"
  cd "$output/repos/$archive"

  archive_title="`printf \"%s index\" $archive | sed_subst_quote`"
  sed -e "s/@TITLE@/$archive_title/g" \
      -e 's/browser-top/at-top/g' \
      "$ARCH_SUBCMD_ROOT/web/browser.html" > index.html
  cp "$ARCH_SUBCMD_ROOT/web/browser-display.html" browser-display.html

  cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
  | sed -e "s/@TITLE@/$sed_title/" \
        -e "s/@TOP@/<a target=\"_top\" href=\"..\/..\/index.html\">top<\/a>/" \
        -e "s/@ARCHIVE@//" \
        -e "s/@CATEGORY@//" \
        -e "s/@BRANCH@//" \
        -e "s/@VERSION@//" \
        -e "s,@HELPFILE@,../../browser-help.html," \
  > browser-nav.html

  cat "$ARCH_SUBCMD_ROOT/web/browser-top.html" \
  | sed -e "s,browser-left,../../browser-left,g" \
        -e "s,browser-right,./categories-right,g" \
  > at-top.html

  larch library-categories "$archive" \
  | awk -v title="$archive categories" \
        -v top_root="./cats" \
	-v library_root="../../library-root/$archive" \
        -v page_bg="$categories_page_bg" \
        -v table_bg="$categories_table_bg" \
        -v headers_bg="$categories_headers_bg" \
	-f "$ARCH_SUBCMD_ROOT/web/right-index.awk" \
  > categories-right.html

  larch library-categories "$archive" \
  | awk -v title="$archive categories" \
        -v right_root="./cats" \
	-v link_stem=branches \
	-v library_root="../../library-root/$archive" \
        -v page_bg="$categories_page_bg" \
        -v table_bg="$categories_table_bg" \
        -v headers_bg="$categories_headers_bg" \
	-f "$ARCH_SUBCMD_ROOT/web/left-index.awk" \
  > categories-left.html


  if test ! -z "$limit_category" ; then
    category_list="$limit_category"
  else
    category_list="`larch library-categories $archive`"
  fi

  for category in $category_list ; do

    if test ! -z "$quiet" ; then
      larch heading --sub --sub --sub "category: %s\\n" "$category"
    fi

    mkdir -p "$output/repos/$archive/cats/$category"
    cd "$output/repos/$archive/cats/$category"

    category_title="`printf \"%s/%s index\" $archive/$category | sed_subst_quote`"
    sed -e "s/@TITLE@/$category_title/g" \
        -e 's/browser-top/at-top/g' \
        "$ARCH_SUBCMD_ROOT/web/browser.html" > index.html
    cp "$ARCH_SUBCMD_ROOT/web/browser-display.html" browser-display.html

    cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
    | sed -e "s/@TITLE@/$sed_title/" \
          -e "s/@TOP@/<a target=\"_top\" href=\"..\/..\/..\/..\/index.html\">top<\/a>/" \
          -e "s/@ARCHIVE@/archive: <a target=\"_top\" href=\"..\/..\/index.html\">$archive<\/a>/" \
          -e "s/@CATEGORY@//" \
          -e "s/@BRANCH@//" \
          -e "s/@VERSION@//" \
          -e "s,@HELPFILE@,../../../../browser-help-categories.html," \
    > browser-nav.html

    cat "$ARCH_SUBCMD_ROOT/web/browser-top.html" \
    | sed -e "s,browser-left,../../categories-left,g" \
          -e "s,browser-right,./branches-right,g" \
    > at-top.html

    larch library-branches "$archive/$category" \
    | awk -v title="$category branches" \
	  -v top_root="./branches" \
	  -v library_root="../../../../library-root/$archive/$category" \
          -v page_bg="$branches_page_bg" \
          -v table_bg="$branches_table_bg" \
          -v headers_bg="$branches_headers_bg" \
	  -f "$ARCH_SUBCMD_ROOT/web/right-index.awk" \
    > branches-right.html

    larch library-branches "$archive/$category" \
    | awk -v title="$category branches" \
	  -v right_root="./branches" \
	  -v link_stem=versions \
	  -v library_root="../../../../library-root/$archive/$category" \
          -v page_bg="$branches_page_bg" \
          -v table_bg="$branches_table_bg" \
          -v headers_bg="$branches_headers_bg" \
	  -f "$ARCH_SUBCMD_ROOT/web/left-index.awk" \
    > branches-left.html



    if test ! -z "$limit_branch" ; then
      branch_list="$limit_branch"
    else
      branch_list="`larch library-branches $archive/$category`"
    fi

    for branch in $branch_list ; do

      if test ! -z "$quiet" ; then
        larch heading --sub --sub --sub --sub "branch: %s\\n" "$branch"
      fi

      mkdir -p "$output/repos/$archive/cats/$category/branches/$branch"
      cd "$output/repos/$archive/cats/$category/branches/$branch"
  
      branch_title="`printf \"%s/%s index\" $archive/$branch | sed_subst_quote`"
      sed -e "s/@TITLE@/$branch_title/g" \
          -e 's/browser-top/at-top/g' \
          "$ARCH_SUBCMD_ROOT/web/browser.html" > index.html
      cp "$ARCH_SUBCMD_ROOT/web/browser-display.html" browser-display.html

      cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
      | sed -e "s/@TITLE@/$sed_title/" \
            -e "s/@TOP@/<a target=\"_top\" href=\"..\/..\/..\/..\/..\/..\/index.html\">top<\/a>/" \
            -e "s/@ARCHIVE@/archive: <a target=\"_top\" href=\"..\/..\/..\/..\/index.html\">$archive<\/a>/" \
            -e "s/@CATEGORY@/category: <a target=\"_top\" href=\"..\/..\/index.html\">$category<\/a>/" \
            -e "s/@BRANCH@//" \
            -e "s/@VERSION@//" \
            -e "s,@HELPFILE@,../../../../../../browser-help-branches.html," \
      > browser-nav.html

      cat "$ARCH_SUBCMD_ROOT/web/browser-top.html" \
      | sed -e "s,browser-left,../../branches-left,g" \
            -e "s,browser-right,./versions-right,g" \
      > at-top.html
  
      larch library-versions "$archive/$branch" \
      | awk -v title="$branch versions" \
	    -v details="patches-full.d/page-1.html" \
            -v top_root="./vsns" \
            -v library_root="../../../../../../library-root/$archive/$category/$branch" \
	    -v link_to_changelog=1 \
	    -v creators_root="./vsns" \
            -v page_bg="$versions_page_bg" \
            -v table_bg="$versions_table_bg" \
            -v headers_bg="$versions_headers_bg" \
            -f "$ARCH_SUBCMD_ROOT/web/right-index.awk" \
      > versions-right.html
  
      larch library-versions "$archive/$branch" \
      | awk -v title="$branch versions" \
	    -v details="patches-full.d/page-1.html" \
            -v right_root="./vsns" \
            -v link_stem=patches \
            -v library_root="../../../../../../library-root/$archive/$category/$branch" \
	    -v link_to_changelog=1 \
	    -v creators_root="./vsns" \
            -v page_bg="$versions_page_bg" \
            -v table_bg="$versions_table_bg" \
            -v headers_bg="$versions_headers_bg" \
            -f "$ARCH_SUBCMD_ROOT/web/left-index.awk" \
      > versions-left.html


      if test ! -z "$limit_version" ; then
        version_list="$limit_version"
      else
        version_list="`larch library-versions $archive/$branch`"
      fi

      for version in $version_list ; do

        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub "version: %s\\n" "$version"
        fi

        mkdir -p "$output/repos/$archive/cats/$category/branches/$branch/vsns/$version"
        cd "$output/repos/$archive/cats/$category/branches/$branch/vsns/$version"

        version_title="`printf \"%s/%s index\" $archive/$version | sed_subst_quote`"
        sed -e "s/@TITLE@/$version_title/g" \
            -e 's/browser-top/at-top/g' \
            "$ARCH_SUBCMD_ROOT/web/browser.html" > index.html
        cp "$ARCH_SUBCMD_ROOT/web/browser-display.html" browser-display.html

        cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
        | sed -e "s/@TITLE@/$sed_title/" \
              -e "s/@TOP@/<a target=\"_top\" href=\"..\/..\/..\/..\/..\/..\/..\/..\/index.html\">top<\/a>/" \
              -e "s/@ARCHIVE@/archive: <a target=\"_top\" href=\"..\/..\/..\/..\/..\/..\/index.html\">$archive<\/a>/" \
              -e "s/@CATEGORY@/category: <a target=\"_top\" href=\"..\/..\/..\/..\/index.html\">$category<\/a>/" \
              -e "s/@BRANCH@/branch: <a target=\"_top\" href=\"..\/..\/index.html\">$branch<\/a>/" \
              -e "s/@VERSION@//" \
              -e "s,@HELPFILE@,../../../../../../../../browser-help-versions.html," \
        > browser-nav.html

        cat "$ARCH_SUBCMD_ROOT/web/browser-top.html" \
        | sed -e "s,browser-left,../../versions-left,g" \
              -e "s,browser-right,./patches-right,g" \
        > at-top.html


        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "File History in: %s\\n" "$version"
        fi

	larch file-history "$archive/$version" > ,,file-history

	cat ,,file-history \
	| grep -E -e "^[^ \t]" \
	| awk -v n=0 \
	      '{
	         print $0 "\t" n;
		 ++n;
	       }' \
	| sort -k 1,1 \
	> ,,file-tag-index


	rm -rf file-details.d
	mkdir file-details.d
	cat ,,file-tag-index \
	| awk \
	    'BEGIN {
		     n = 0;
		     n_page = 0;
		     page = 1;
		     output_page = "file-details.d/page-contents." page;
	           }
	     {
	       tag = $1;
	       file_no = $2;

	       if (n_page < 20)
	         {
		   print tag > output_page;
		   ++n;
		   ++n_page;
		   system("ln -s page-" page ".html file-details.d/loc-" file_no ".html");
		 }
	       else
	         {
		   close(output_page);
		   n = 1;
		   n_page = 1;
		   page += 1;
		   output_page = "file-details.d/page-contents." page;
		   print tag > output_page;
		   system("ln -s page-" page ".html file-details.d/loc-" file_no ".html");
		 }
	     }
	     END   {
		     close(output_page);
		     for (x = 1; x <= page; ++x)
		       {
		         nav = "file-details.d/nav." x;

			 for (y = 1; y <= page; ++y)
			   {
			     if (x == y)
			       print "<a href=\"page-" y ".html\"><b>&lt;" y "&gt;</b></a>" > nav;
			     else
			       print "<a href=\"page-" y ".html\"><i>&lt;" y "&gt;</i></a>" > nav;
			   }
			 close(nav);
		       }
		   }'

        cd file-details.d

	for contents in page-contents.* ; do

	  page=page-${contents#*.}.html

          printf "<HTML>\\n" > "$page"
          printf "\\n" >> "$page"
          printf "<HEAD><TITLE>%s</TITLE></HEAD>\\n" "$version file details" >> "$page"
          printf "\\n" >> "$page"
          printf "<BODY bgcolor=\"%s\">\\n" "$files_page_bg" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"
 	  printf "<p align=\"center\"><small><a name="top-page"></a>" >> "$page"
	  cat nav.${contents#*.} >> "$page"
          printf "</small>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "<table border=\"3\" cellspacing=\"3\" cellpadding=\"0\" bgcolor=\"%s\" width=\"100%%\">\\n" "$revisions_table_bg" >> "$page"
          printf "<caption><font color=\"#fe005e\"><i>%s</i></font></caption>\\n" "$version file details" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"
  

          cat "../,,file-history" \
          | awk -v contents="$contents" \
		-v lib_vsn_prefix="../../../../../../../../../library-root/$archive/$category/$branch/$version/$version" \
                -f "$ARCH_SUBCMD_ROOT/web/file-details.awk" \
          >> "$page"
          
          printf "\\n" >> "$page"
          printf "</table>\\n" >> "$page"
          printf "\\n" >> "$page"
	  printf "<p align=\"center\"><small>" >> "$page"
	  cat nav.${contents#*.} >> "$page"
          printf "</small>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "</BODY>\\n" >> "$page"
          printf "</HTML>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"

	done

	cd ..


        printf "<HTML>\\n" > files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "<HEAD><TITLE>%s</TITLE></HEAD>\\n" "$version files by tag" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "<BODY bgcolor=\"%s\">\\n" "$files_page_bg" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "<table border=\"3\" cellspacing=\"3\" cellpadding=\"0\" bgcolor=\"%s\" width=\"100%%\">\\n" "$files_table_bg" >> files-by-tag.html
        printf "<caption><font color=\"#fe005e\"><i>%s</i></font></caption>\\n" "$version files by tag" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
	printf "<tr>\\n" >> files-by-tag.html
	printf "  <td>\\n" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "    <p align=\"right\">\\n" >> files-by-tag.html
        printf "    <a href=\"file-details.d/page-1.html\"><i>all file details</i></a>\\n" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
        printf "    <p>\\n" >> files-by-tag.html
        printf "      <code>\\n" >> files-by-tag.html
        printf "\\n" >> files-by-tag.html
	
	cat ,,file-tag-index \
	| html_quote \
	| awk \
	     '{
		print "<a href=\"file-details.d/loc-" $2 ".html#file-" $2 "\">" $1 "</a><br>";
	      }' \
	>> files-by-tag.html

	printf "      </code>\\n" >> files-by-tag.html
	printf "    </td>\\n" >> files-by-tag.html
	printf "  </tr>\\n" >> files-by-tag.html
	printf "</table>\\n" >> files-by-tag.html
	printf "</BODY>\\n" >> files-by-tag.html
	printf "</HTML>\\n" >> files-by-tag.html

        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "ChangeLog in: %s\\n" "$version"
        fi

	printf "<html>\\n" > changelog.html
	printf "<head>\\n" >> changelog.html
	printf "<title>%s</title>\\n" "ChangeLog for $archive/$version" >> changelog.html
	printf "</head>\\n" >> changelog.html
        printf "\\n" >> changelog.html
        printf "<body>\\n" >> changelog.html
        printf "\\n" >> changelog.html

        larch top larch heading --html "ChangeLog for %s\\n" "$archive/$version" >> changelog.html

        printf "<pre>\\n" >> changelog.html

        larch changelog --dir "`larch library-find \"$archive/$version\"`" "$archive/$version" \
        | html_quote \
        | awk -v archive="$archive" \
              -v version="$version" \
              -v browser_repos="../../../../../../.." \
              -v archive_list="$output/,,archives" \
          '
          function match_cl_ref()
          {
            return match($0, "^[ \t]*\\*[ \t]*" archivere "/" basere "(--" basere ")?--" vsnre "--" patchre "[ \t]*");
          }

          BEGIN {
  	 	  if (archive_list != "")
		    {
		      while (getline < archive_list)
		        {
			  is_indexed_archive[$0] = 1;
			}

		      close(archive_list);
		    }
                  await_summary = 0;
                  archivere="[-a-zA-Z0-9]+(\\.[-a-zA-Z0-9]*)*@[-a-zA-Z0-9.]*(--[-a-zA-Z0-9.]*)?";
                  basere="([[:alpha:]]([[:alnum:]]*(-[[:alnum:]]+)?)*)";
                  vsnre="([[:digit:]]+\\.[[:digit:]]+)";
                  patchre="(base-0|patch-[0-9]+|version-0|versionfix-[0-9]+)";
                }

          match($0, "^[12][0-9]") {
                                    current_patch = $0;
                                    sub("^.*\t", "", current_patch);
                                    gsub("[ \t]+", "", current_patch);
                                    await_summary = 1;

                                    print "<a name=\"" current_patch "\"></a>";
                                    sub(current_patch "$", "<a href=\"patches-full.d/loc-" current_patch ".html#" current_patch "\">" current_patch "</a>", $0);
                                  }

          (await_summary && match($0, "^[ \t]*Summary:")) {
                          					  sub("Summary:", \
								      "Summary: <a type=\"text/plain\"" \
										" href=\"reports/" current_patch "/=log.txt\">"\
										"<small>(full log entry)</small></a>", $0);
                                                                  await_summary = 0;
                                                                }

          match_cl_ref() {
                           ref_patch = $0;
                           sub("^[ \t]*\\*[ \t]*", "", ref_patch);
                           sub("[ \t]*$", "", ref_patch);
                           cl_lvl = ref_patch;
                           sub("^" archivere "/" basere "(--" basere ")?--" vsnre "--", "", cl_lvl);
                           cl_av = ref_patch;
                           sub("--" cl_lvl "$", "", cl_av);
			   cl_a = cl_av;
			   sub("/.*", "", cl_a);
			   cl_v = cl_av;
			   sub(".*/", "", cl_v);
			   cl_b = cl_v;
			   sub("--[^-]*(-[^-]+)*$", "", cl_b);
			   cl_c = cl_b;
			   sub("--.*", "", cl_c);
			   if (is_indexed_archive[cl_a])
			     {
			       cl_loc = browser_repos "/" cl_a "/cats/" cl_c "/branches/" cl_b "/vsns/" cl_v "/changelog.html#" cl_lvl;
                               $0 = "     * <a href=\"" cl_loc "\">" cl_av "--" cl_lvl "</a>";
			     }
                         }
          { print $0; }
          ' \
        >> changelog.html

        printf "</pre>\\n" >> changelog.html

        printf "\\n" >> changelog.html
        printf "</body>\\n" >> changelog.html
        printf "\\n" >> changelog.html


        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "patch sets in: %s\\n" "$version"
        fi

	vsndir="`pwd`"
	mkdir -p reports
	cd reports

	for lvl in `larch library-revisions "$archive/$version"` ; do
	  if test ! -d "$lvl" ; then

          if test ! -z "$quiet" ; then
            larch heading --sub --sub --sub --sub --sub --sub --sub "patch level: %s\\n" "$lvl"
          fi

	    rm -rf ",,$lvl"
	    mkdir ",,$lvl"
	    cd ",,$lvl"
	    ln -s `larch library-find "$archive/$version--$lvl"`/,,patch-set/* .
	    if test -e "patches" ; then 
	      larch patch-report --html . > =REPORT.html
	      larch patch-report . > =REPORT.txt
	      larch patch-details --notes "$output/,,distributions" --archive-list "$output/,,archives" . > =details.htmli
	    else
	      printf "\\n" > =REPORT.txt
	      printf "This revision has no patch set.\\n" >> =REPORT.txt
	      printf "\\n" >> =REPORT.txt
	      cp ./=REPORT.txt =REPORT.html
	      larch patch-details --notes "$output/,,distributions" --archive-list "$output/,,archives" . > =details.htmli
	    fi

            printf "<HTML>\\n" > files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "<HEAD><TITLE>%s</TITLE></HEAD>\\n" "$version--$lvl files by name" >> files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "<BODY bgcolor=\"%s\">\\n" "$files_page_bg" >> files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "<table border=\"3\" cellspacing=\"3\" cellpadding=\"0\" bgcolor=\"%s\" width=\"100%%\">\\n" "$revisions_table_bg" >> files-by-name.html
            printf "<caption><font color=\"#fe005e\"><i>%s</i></font></caption>\\n" "$version--$lvl files by name" >> files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "<tr>\\n" >> files-by-name.html
            printf "  <td>\\n" >> files-by-name.html
            printf "\\n" >> files-by-name.html
            printf "    <p>\\n" >> files-by-name.html
            printf "      <code><pre>\\n" >> files-by-name.html
            printf "\\n" >> files-by-name.html

	    cat "`larch library-find \"$archive/$version--$lvl\"`/,,index" \
	    | join -o 1.1,2.2 -1 2 -2 1 - ../../,,file-tag-index \
	    | sort -k 1,1 \
	    | awk -v library_root="../../../../../../../../../../library-root/$archive/$category/$branch/$version/$version--$lvl" \
	          '{
		     name = $1;
		     seq = $2;

		     sub("^./", "", name);

		     spaces = name;
		     gsub("[^/]", "", spaces);
		     gsub(".", " ", spaces);

		     print spaces "<a target=\"display\" href=\"" library_root "/" name "\">" name "</a> " \
			    "<a target=\"display\" href=\"../../file-details.d/loc-" seq ".html#file-" seq "\"><b>?</b></a><br>";
		   }' \
	    >> files-by-name.html

            printf "      </pre></code>\\n" >> files-by-name.html
            printf "    </td>\\n" >> files-by-name.html
            printf "  </tr>\\n" >> files-by-name.html
            printf "</table>\\n" >> files-by-name.html
            printf "</BODY>\\n" >> files-by-name.html
            printf "</HTML>\\n" >> files-by-name.html
            
	    cd ..
	    mv ",,$lvl" "$lvl"
	  fi
	done

	cd "$vsndir"

        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "patches in: %s\\n" "$version"
        fi

        larch library-revisions --reverse --summary --creator --date "$archive/$version" \
        | html_quote \
        | awk -v title="$version revisions" \
	      -v report_root="./reports" \
              -v library_root="../../../../../../../../library-root/$archive/$category/$branch/$version" \
	      -v archive="$archive" \
	      -v version="$version" \
              -v page_bg="$revisions_page_bg" \
              -v table_bg="$revisions_table_bg" \
              -v headers_bg="$revisions_headers_bg" \
	      -v revision_notes="$output/,,distributions" \
	      -v note_color="$revisions_notes_color" \
	      -v link_target="display" \
              -f "$ARCH_SUBCMD_ROOT/web/revisions-right-index.awk" \
        > patches-right.html

        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "patches details in: %s\\n" "$version"
        fi

	rm -rf patches-full.d
	mkdir patches-full.d
	larch library-revisions --reverse "$archive/$version" \
	| awk \
	    'BEGIN {
		     n = 0;
		     n_page = 0;
		     page = 1;
		     output_page = "patches-full.d/page-contents." page;
	           }
	     {
	       rev = $1;

	       if (n_page < 20)
	         {
		   print rev > output_page;
		   ++n;
		   ++n_page;
		   system("ln -s page-" page ".html patches-full.d/loc-" rev ".html");
		 }
	       else
	         {
		   close(output_page);
		   n = 1;
		   n_page = 1;
		   page += 1;
		   output_page = "patches-full.d/page-contents." page;
		   print rev > output_page;
		   system("ln -s page-" page ".html patches-full.d/loc-" rev ".html");
		 }
	     }
	     END   {
		     close(output_page);
		     for (x = 1; x <= page; ++x)
		       {
		         nav = "patches-full.d/nav." x;

			 for (y = 1; y <= page; ++y)
			   {
			     if (x == y)
			       print "<a href=\"page-" y ".html\"><b>&lt;" y "&gt;</b></a>" > nav;
			     else
			       print "<a href=\"page-" y ".html\"><i>&lt;" y "&gt;</i></a>" > nav;
			   }
			 close(nav);
		       }
		   }'


	cd patches-full.d

	for contents in page-contents.* ; do

	  page=page-${contents#*.}.html

          printf "<HTML>\\n" > "$page"
          printf "\\n" >> "$page"
          printf "<HEAD><TITLE>%s</TITLE></HEAD>\\n" "$version revisions" >> "$page"
          printf "\\n" >> "$page"
          printf "<BODY bgcolor=\"%s\">\\n" "$revisions_page_bg" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"
 	  printf "<p align=\"center\"><small><a name="top-of-page"></a>" >> "$page"
	  cat nav.${contents#*.} >> "$page"
          printf "</small>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "<table border=\"3\" cellspacing=\"3\" cellpadding=\"0\" bgcolor=\"%s\" width=\"100%%\">\\n" "$revisions_table_bg" >> "$page"
          printf "<caption><font color=\"#dedede\"><i>%s</i></font></caption>\\n" "$version revisions" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"
  
          cat "$contents" \
          | awk -v note_color="$revisions_notes_color" \
                -v browser="../../../../../../../.." \
                -v library_root="../../../../../../../../../library-root/$archive/$category/$branch/$version" \
                -v version="$version" \
                -f "$ARCH_SUBCMD_ROOT/web/details.awk" \
          >> "$page"
          
          printf "\\n" >> "$page"
          printf "</table>\\n" >> "$page"
          printf "\\n" >> "$page"
	  printf "<p align=\"center\"><small>" >> "$page"
	  cat nav.${contents#*.} >> "$page"
          printf "</small>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "</BODY>\\n" >> "$page"
          printf "</HTML>\\n" >> "$page"
          printf "\\n" >> "$page"
          printf "\\n" >> "$page"

	done
	cd ..
	
        if test ! -z "$quiet" ; then
          larch heading --sub --sub --sub --sub --sub --sub "creators in: %s\\n" "$version"
        fi

	mkdir -p creators
	rm -f ,,creators

        creator_title="`printf \"%s/%s creators\" $archive/$version | sed_subst_quote`"
        sed -e "s/@TITLE@/$creator_title/g" \
            -e 's/browser-top/creator-top/g' \
            -e 's/browser-nav/creator-nav/g' \
            "$ARCH_SUBCMD_ROOT/web/browser.html" > creator-index.html

        cat  "$ARCH_SUBCMD_ROOT/web/browser-nav.html" \
        | sed -e "s/@TITLE@/$sed_title/" \
              -e "s/@TOP@/<a target=\"_top\" href=\"..\/..\/..\/..\/..\/..\/..\/..\/index.html\">top<\/a>/" \
              -e "s/@ARCHIVE@/archive: <a target=\"_top\" href=\"..\/..\/..\/..\/..\/..\/index.html\">$archive<\/a>/" \
              -e "s/@CATEGORY@/category: <a target=\"_top\" href=\"..\/..\/..\/..\/index.html\">$category<\/a>/" \
              -e "s/@BRANCH@/branch: <a target=\"_top\" href=\"..\/..\/index.html\">$branch<\/a>/" \
              -e "s/@VERSION@/version: <a target=\"_top\" href=\".\/index.html\">$version<\/a>/" \
              -e "s,@HELPFILE@,../../../../../../../../browser-help-creators.html," \
        > creator-nav.html

	cat "$ARCH_SUBCMD_ROOT/web/browser-right.html" \
	| sed -e "s/bgcolor=\"#bebebe\"/bgcolor=\"$revisions_page_bg\"/" \
	> browser-right.html

        cat "$ARCH_SUBCMD_ROOT/web/browser-top.html" \
        | sed -e "s,browser-left,./creators-left,g" \
        > creator-top.html

        for creator in `larch library-revisions --creator "$archive/$version" \
		        | grep -E "^[[:space:]]" \
		        | sed -e 's/.*<//' -e 's/>$//' \
		        | sort -u` ; do

          if test ! -z "$quiet" ; then
            larch heading --sub --sub --sub --sub --sub "creator: %s\\n" "$creator"
          fi

          larch library-revisions --reverse --summary --creator --date "$archive/$version" \
          | html_quote \
          | awk -v title="$version revisions" \
	        -v report_root="../reports" \
                -v library_root="../../../../../../../../library-root/$archive/$category/$branch/$version" \
	        -v archive="$archive" \
	        -v version="$version" \
		-v creator_emph="$creator" \
                -v page_bg="$revisions_page_bg" \
                -v table_bg="$revisions_table_bg" \
                -v headers_bg="$revisionsa_headers_bg" \
                -v revision_notes="$output/,,distributions" \
	        -v note_color="$revisions_notes_color" \
		-v link_target="display" \
                -f "$ARCH_SUBCMD_ROOT/web/revisions-right-index.awk" \
          > "creators/${creator}-right.html"

	  printf "%s\\n" "$creator" >> ,,creators

	done

	cat ,,creators \
	| html_quote \
	| awk -v title="$version creators" \
	      -v right_root="./creators" \
	      -v library_root="" \
              -v page_bg="$creators_page_bg" \
              -v table_bg="$creators_table_bg" \
              -v headers_bg="$creators_headers_bg" \
	      -f "$ARCH_SUBCMD_ROOT/web/left-index.awk" \
	> creators-left.html

	cat ,,creators \
	| html_quote \
	| awk -v title="$version creators" \
	      -v top_root="." \
	      -v top_stem="creators" \
	      -v library_root="" \
              -v page_bg="$creators_page_bg" \
              -v table_bg="$creators_table_bg" \
              -v headers_bg="$creators_headers_bg" \
	      -f "$ARCH_SUBCMD_ROOT/web/right-index.awk" \
	> creators-right.html



      done

    done  


  done


done



if test ! -z "$quiet" ; then
  larch heading --sub "make-library-browser finish time: %s\\n" "`date`"
fi

# tag: Tom Lord Wed Jan  9 02:47:02 2002 (web/make-library-browser.sh)
#
