/*
 * File:        vectortest.c
 * Copyright:   (c) 2004-2007 Lawrence Livermore National Security, LLC
 * Revision:    @(#) $Revision: 6171 $
 * Date:        $Date: 2007-10-08 16:39:28 -0700 (Mon, 08 Oct 2007) $
 * Description: Contract Test C client
 *
 */
#include <stdio.h>
#include <math.h>
#include <string.h>
#include "sidlAsserts.h"
/*
 * Enable once new enforcement features fully supported.
 *
#include "sidl_EnfPolicy.h"
*/
#include "sidl_String.h"
#include "synch.h"
#include "vector_Utils.h"
#include "vector_Exception.h"
#include "sidl_Exception.h"

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif

static const int NO_EXC_EXPECTED = 0;
static const int EXC_EXPECTED    = 1;
static const int DBZ_EXPECTED    = 2;
static const int NVE_EXPECTED    = 3;
static const int PRE_EXPECTED    = 4;
static const int POST_EXPECTED   = 5;

static const int PROPER_BEHAVIOR = 0;
static const int BAD_LEVEL_1     = 1;
static const int BAD_LEVEL_2     = 2;
static const int BAD_LEVEL_3     = 3;
static const int BAD_LEVEL_4     = 4;

void declare_part(synch_RegOut tracker,  int * part_no ) {
  sidl_BaseInterface exception = NULL;
  synch_RegOut_startPart(tracker, ++(*part_no), &exception);
  if (exception) {
    sidl_BaseInterface throwaway_exception;
    sidl_BaseInterface exception2 = NULL;
    synch_RegOut_forceFailure(tracker, &exception2);
    if (exception2) {
      puts("TEST_RESULT FAIL\n");
      sidl_BaseInterface_deleteRef(exception2, &throwaway_exception);
    }
    sidl_BaseInterface_deleteRef(exception, &throwaway_exception);
  }
}

void end_part( synch_RegOut tracker, int part_no,
               enum synch_ResultType__enum result) {
  sidl_BaseInterface exception = NULL;
  synch_RegOut_endPart(tracker, part_no, result, &exception);
  if (exception) {
    sidl_BaseInterface throwaway_exception;
    sidl_BaseInterface exception2 = NULL;
    synch_RegOut_forceFailure(tracker, &exception2);
    if (exception2) {
      puts("TEST_RESULT FAIL\n");
      sidl_BaseInterface_deleteRef(exception2, &throwaway_exception);
    }
    sidl_BaseInterface_deleteRef(exception, &throwaway_exception);
  }
}

static void printExceptionNote(sidl_SIDLException _ex)
{
  if (_ex != NULL) {
    sidl_BaseInterface throwaway_exception;
    char* msg = NULL;

    msg = sidl_SIDLException_getNote(_ex, &throwaway_exception);
    printf("  %s\n", msg);
    sidl_String_free(msg);
  }
}

#define CHECK(FUNC,COMMENT) {\
  declare_part(tracker, &part_no); \
  synch_RegOut_writeComment(tracker, COMMENT, &exception); \
  result = (FUNC) ? synch_ResultType_PASS : synch_ResultType_FAIL; \
  end_part(tracker, part_no, result); \
}

#define EXCEPTION_CHECKS(EXC_VAR, EXC_EXPECT, RES, RES_EXPECT, RET) {\
  if ((EXC_VAR) != NULL) {\
    if (SIDL_CATCH((EXC_VAR), "vector.NegativeValueException")) {\
      if (EXC_EXPECT == NVE_EXPECTED) { RET = TRUE; } else { RET = FALSE; } \
      CLEANUP(EXC_VAR, RET)\
    } else if (SIDL_CATCH((EXC_VAR), "vector.DivideByZeroException")) {\
      if (EXC_EXPECT == DBZ_EXPECTED) { RET = TRUE; } else { RET = FALSE; }\
      CLEANUP(EXC_VAR, RET)\
    } else if (SIDL_CATCH((EXC_VAR), "vector.Exception")) {\
      if (EXC_EXPECT == EXC_EXPECTED) { RET = TRUE; } else { RET = FALSE; }\
      CLEANUP(EXC_VAR, RET)\
    } else if (SIDL_CATCH((EXC_VAR), "sidl.PreViolation")) {\
      if (EXC_EXPECT == PRE_EXPECTED) { RET = TRUE; } else { RET = FALSE; }\
      CLEANUP(EXC_VAR, RET)\
    } else if (SIDL_CATCH((EXC_VAR), "sidl.PostViolation")) {\
      if (EXC_EXPECT == POST_EXPECTED) { RET = TRUE; } else { RET = FALSE; }\
      CLEANUP(EXC_VAR, RET)\
    } else {\
      goto EXIT;\
    }\
  }\
}

#define RESULT_CHECKS(EXC_VAR, EXC_EXPECT, RES, RES_EXPECT) {\
  int _ret = FALSE;\
  EXCEPTION_CHECKS(EXC_VAR, EXC_EXPECT, RES, RES_EXPECT, _ret) \
  if ( ((EXC_VAR) == NULL) && (EXC_EXPECT == NO_EXC_EXPECTED) ) {\
    if (RES == RES_EXPECT) { _ret = TRUE; } else { _ret = FALSE; } \
    CLEANUP(EXC_VAR, _ret)\
  }\
}

#define RESULT_CHECKSD(EXC_VAR, EXC_EXPECT, RES, RES_EXPECT, TOL) {\
  int _ret = FALSE;\
  EXCEPTION_CHECKS(EXC_VAR, EXC_EXPECT, RES, RES_EXPECT, _ret) \
  if ( ((EXC_VAR) == NULL) && (EXC_EXPECT == NO_EXC_EXPECTED) ) {\
    if ( (fabs((double)RES) - fabs((double)RES_EXPECT)) <= (double)TOL ) { \
      _ret = TRUE; \
    } else { \
      _ret = FALSE; \
    } \
    CLEANUP(EXC_VAR, _ret)\
  }\
}

#define CLEAR(EX_VAR) { \
  if ((EX_VAR) != NULL) { \
    sidl_BaseInterface _tae; \
    sidl_BaseInterface_deleteRef((sidl_BaseInterface)(EX_VAR), &_tae); \
    (EX_VAR) = NULL; \
  }  \
}

#define CLEANUP(EXC_VAR, RET) {\
  if ((EXC_VAR) != NULL) {\
    sidl_BaseInterface _tae = NULL; \
    sidl_SIDLException _cue = sidl_SIDLException__cast(EXC_VAR, &_tae); \
    printExceptionNote(_cue); \
    if (_cue) sidl_SIDLException_deleteRef(_cue, &_tae); \
    CLEAR(EXC_VAR)\
  }\
  return RET;\
}


int runIsZero(struct sidl_double__array* v, double tol, int res, int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_isZero(v, tol, (sidl_BaseInterface*)(&_ex));
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runIsUnit(struct sidl_double__array* v, double tol, int res, int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_isUnit(v, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runAreEqual(struct sidl_double__array* u, 
                struct sidl_double__array* v, double tol, int res, 
                int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_areEqual(u, v, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runAreOrthogonal(struct sidl_double__array* u, 
                     struct sidl_double__array* v, double tol, int res, 
                     int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_areOrthogonal(u, v, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runSchwarzHolds(struct sidl_double__array* u, 
                    struct sidl_double__array* v, double tol, int res, 
                    int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_schwarzHolds(u, v, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runTriangleInequalityHolds(struct sidl_double__array* u, 
                               struct sidl_double__array* v, double tol, 
                               int res, int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_triangleInequalityHolds(u, v, tol, 
        (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runNorm(struct sidl_double__array* u, double tol, int badLvl, int res, 
            int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_norm(u, tol, badLvl, (sidl_BaseInterface*)(&_ex));
  RESULT_CHECKSD(_ex, expectExc, x, res, tol)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


int runDot(struct sidl_double__array* u, struct sidl_double__array* v, 
           double tol, int badLvl, int res, int expectExc)
{
  int x;
  sidl_BaseInterface _ex = NULL;

  x = vector_Utils_dot(u, v, tol, badLvl, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, x, res)

  EXIT:;
    CLEANUP(_ex, FALSE)
}


/* NOTE: Tolerance is only needed to check the result of the call. */
int runProduct(double a, struct sidl_double__array* u, double tol,
               int badLvl, struct sidl_double__array* res, int okay, 
               int expectExc)
{
  struct sidl_double__array* x = NULL;
  sidl_BaseInterface         _ex = NULL;
  int                        ok = FALSE;

  x  = vector_Utils_product(a, u, badLvl, (sidl_BaseInterface*)(&_ex));
  if (_ex != NULL) {
    RESULT_CHECKS(_ex, expectExc, okay, okay)
  }
  ok = vector_Utils_areEqual(x, res, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, ok, okay)

  EXIT:;
    if (x != NULL) sidl_double__array_deleteRef(x);
    CLEANUP(_ex, FALSE)
}


/* NOTE: Tolerance is only needed to check the result of the call. */
int runNegate(struct sidl_double__array* u, double tol, int badLvl,
              struct sidl_double__array* res, int okay, int expectExc)
{
  struct sidl_double__array* x = NULL;
  sidl_BaseInterface         _ex = NULL;
  int                        ok = FALSE;

  x  = vector_Utils_negate(u, badLvl, (sidl_BaseInterface*)(&_ex));
  if (_ex != NULL) {
    RESULT_CHECKS(_ex, expectExc, okay, okay)
  }
  ok = vector_Utils_areEqual(x, res, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, ok, okay)

  EXIT:;
    if (x != NULL) sidl_double__array_deleteRef(x);
    CLEANUP(_ex, FALSE)
}


int runNormalize(struct sidl_double__array* u, double tol, int badLvl,
                 struct sidl_double__array* res, int okay, int expectExc)
{
  struct sidl_double__array* x   = NULL;
  sidl_BaseInterface         _ex = NULL;
  int                        ok  = FALSE;

  x  = vector_Utils_normalize(u, tol, badLvl, (sidl_BaseInterface*)(&_ex));
  if (_ex != NULL) {
    RESULT_CHECKS(_ex, expectExc, okay, okay)
  }
  ok = vector_Utils_areEqual(x, res, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, ok, okay)

  EXIT:;
    if (x != NULL) sidl_double__array_deleteRef(x);
    CLEANUP(_ex, FALSE)
}


int runSum(struct sidl_double__array* u, struct sidl_double__array* v, 
           double tol, int badLvl, struct sidl_double__array* res, 
           int okay, int expectExc)
{
  struct sidl_double__array* x = NULL;
  sidl_BaseInterface         _ex = NULL;
  int                        ok = FALSE;

  x = vector_Utils_sum(u, v, badLvl, (sidl_BaseInterface*)(&_ex)); 
  if (_ex != NULL) {
    RESULT_CHECKS(_ex, expectExc, okay, okay)
  }
  ok = vector_Utils_areEqual(x, res, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, ok, okay)

  EXIT:;
    if (x != NULL) sidl_double__array_deleteRef(x);
    CLEANUP(_ex, FALSE)
}


int runDiff(struct sidl_double__array* u, struct sidl_double__array* v, 
           double tol, int badLvl, struct sidl_double__array* res, int okay, 
           int expectExc)
{
  struct sidl_double__array* x = NULL;
  sidl_BaseInterface         _ex = NULL;
  int                        ok = FALSE;

  x = vector_Utils_diff(u, v, badLvl, (sidl_BaseInterface*)(&_ex)); 
  if (_ex != NULL) {
    RESULT_CHECKS(_ex, expectExc, ok, okay)
  }
  ok = vector_Utils_areEqual(x, res, tol, (sidl_BaseInterface*)(&_ex)); 
  RESULT_CHECKS(_ex, expectExc, ok, okay)

  EXIT:;
    if (x != NULL) sidl_double__array_deleteRef(x);
    CLEANUP(_ex, FALSE)
}


/*************************** main ***************************/

int main(int argc, char**argv)
{ 
  static char*       statsFile = "VUUtils.stats";
  sidl_BaseInterface exception = NULL;

  enum synch_ResultType__enum result  = synch_ResultType_PASS; 
  int                         part_no = 0;


  int i, j;
  int    max_size  = 6;
  double sqrt_size = sqrt(max_size);
  double tol       = 1.0e-9;
  double ntol      = -1.0e-9;
  double val       = 1.0/sqrt_size;
  double nval      = -1.0/sqrt_size;
  double *cva      = (double *)malloc(max_size*sizeof(double));
  double *cva1     = (double *)malloc((max_size+1)*sizeof(double));
  double *cna      = (double *)malloc(max_size*sizeof(double));

  struct sidl_double__array *t  = NULL;
  struct sidl_double__array *u  = NULL;
  struct sidl_double__array *u1 = NULL;
  struct sidl_double__array *nu = NULL;
  struct sidl_double__array *z  = NULL;
  struct sidl_double__array *n  = NULL;  /* The null array! */

  synch_RegOut tracker = synch_RegOut__create(&exception); 
  SIDL_CHECK(exception);

  for (i=0; i < max_size; i++) {
    cva[i]  = val;
    cva1[i] = val;
    cna[i]  = nval;
  }
  cva1[max_size] = val;

  u = sidl_double__array_create1d(max_size);
  memcpy(sidl_double__array_first(u), cva, (size_t)(max_size*sizeof(double)));

  u1 = sidl_double__array_create1d(max_size+1);
  memcpy(sidl_double__array_first(u1), cva1,
         (size_t)((max_size+1)*sizeof(double)));

  nu = sidl_double__array_create1d(max_size);
  memcpy(sidl_double__array_first(nu), cna, (size_t)(max_size*sizeof(double)));

  z  = sidl_double__array_create1d(max_size);
  memset(sidl_double__array_first(z), 0, (size_t)(max_size*sizeof(double)));

  t  = sidl_double__array_create2dCol(max_size, max_size);
  for (i=0; i < max_size; i++) {
    for (j=0; j < max_size; j++) {
       sidl_double__array_set2(t, i, j, val); 
    }
  }
  sidl_double__array_set1(u1, max_size, val);

  (void) argc;
  (void) argv;
  synch_RegOut_setExpectations(tracker, 128, &exception); SIDL_CHECK(exception);

  printf("\nCOMMENT: *** ENABLE FULL CONTRACT CHECKING ***\n\n");
/*
 * Enable once new enforcement features fully supported.
 *
  sidl_EnfPolicy_enforceAll(&exception); SIDL_CHECK(exception);
 */
  vector_Utils__set_checking_static(CHECK_ALL_TYPES | CHECK_ALWAYS, 0, 0,
                                    &exception); SIDL_CHECK(exception);

  /* isZero set */
  CHECK(runIsZero(z, tol, TRUE, NO_EXC_EXPECTED), 
   "checking if zero vector is zero")
  CHECK(runIsZero(u, tol, FALSE, NO_EXC_EXPECTED), 
   "checking if unit vector not zero")
  CHECK(runIsZero(n, tol, TRUE, PRE_EXPECTED), 
   "passing isZero a null array")
  CHECK(runIsZero(t, tol, TRUE, PRE_EXPECTED), 
   "passing isZero a 2D array")
  CHECK(runIsZero(z, ntol, TRUE, PRE_EXPECTED), 
   "passing isZero a negative tolerance")

  /* isUnit set */
  CHECK(runIsUnit(u, tol, TRUE, NO_EXC_EXPECTED), 
   "checking if unit vector is unit")
  CHECK(runIsUnit(z, tol, FALSE, NO_EXC_EXPECTED), 
   "checking if zero vector is not unit")
  CHECK(runIsUnit(n, tol, TRUE, PRE_EXPECTED), 
   "passing isUnit a null array")
  CHECK(runIsUnit(t, tol, TRUE, PRE_EXPECTED), 
   "passing isUnit a 2D array")
  CHECK(runIsUnit(u, ntol, TRUE, PRE_EXPECTED), 
   "passing isUnit a negative tolerance")

  /* areEqual set */
  CHECK(runAreEqual(u, z, tol, FALSE, NO_EXC_EXPECTED), 
   "checking that unit and zero vecs are not equal")
  CHECK(runAreEqual(u, u, tol, TRUE, NO_EXC_EXPECTED), 
   "checking that unit vec is equal to itself")
  CHECK(runAreEqual(n, u, tol, TRUE, PRE_EXPECTED), 
   "passing areEqual null 1st array")
  CHECK(runAreEqual(t, u, tol, TRUE, PRE_EXPECTED), 
   "passing areEqual 2D 1st array")
  CHECK(runAreEqual(u, n, tol, TRUE, PRE_EXPECTED), 
   "passing areEqual null 2nd array")
  CHECK(runAreEqual(u, t, tol, TRUE, PRE_EXPECTED), 
   "passing areEqual a 2D 2nd array")
  CHECK(runAreEqual(u, u1, tol, TRUE, PRE_EXPECTED), 
   "passing areEqual different sized arrays")
  CHECK(runAreEqual(u, u, ntol, TRUE, PRE_EXPECTED), 
   "passing areEqual a negative tolerance")

  /* areOrthogonal set */
  CHECK(runAreOrthogonal(u, z, tol, TRUE, NO_EXC_EXPECTED), 
   "checking that unit and zero vecs are orthogonal")
  CHECK(runAreOrthogonal(u, nu, tol, FALSE, NO_EXC_EXPECTED), 
   "checking that unit and negative unit vecs are not orthogonal")
  CHECK(runAreOrthogonal(n, u, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal null 1st array")
  CHECK(runAreOrthogonal(t, u, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal a 2D 1st array")
  CHECK(runAreOrthogonal(u, n, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal null 2nd array")
  CHECK(runAreOrthogonal(u, t, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal a 2D 2nd array")
  CHECK(runAreOrthogonal(u, u1, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal different sized unit arrays")
  CHECK(runAreOrthogonal(u, u, ntol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal a negative tolerance")
  CHECK(runAreOrthogonal(t, t, tol, TRUE, PRE_EXPECTED), 
   "passing areOrthogonal 2D arrays in both args")

  /* schwarzHolds set */
  CHECK(runSchwarzHolds(u, z, tol, TRUE, NO_EXC_EXPECTED), 
   "checking that Schwarz holds for unit and zero vectors")
  CHECK(runSchwarzHolds(n, z, tol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds a null 1st array")
  CHECK(runSchwarzHolds(t, z, tol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds a 2D 1st array")
  CHECK(runSchwarzHolds(z, n, tol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds a null 2nd array")
  CHECK(runSchwarzHolds(u, t, tol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds a 2D 2nd array")
  CHECK(runSchwarzHolds(u, u1, tol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds different sized unit arrays")
  CHECK(runSchwarzHolds(u, z, ntol, TRUE, PRE_EXPECTED), 
   "passing schwarzHolds a negative tolerance")

  /* triangleInequalityHolds set */
  CHECK(runTriangleInequalityHolds(u, z, tol, TRUE, NO_EXC_EXPECTED), 
   "checking that Triangle Inequality holds for unit and zero vectors")
  CHECK(runTriangleInequalityHolds(n, u, tol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds a null 1st array")
  CHECK(runTriangleInequalityHolds(t, u, tol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds a 2D 1st array")
  CHECK(runTriangleInequalityHolds(u, n, tol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds a null 2nd array")
  CHECK(runTriangleInequalityHolds(u, t, tol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds a 2D 2nd array")
  CHECK(runTriangleInequalityHolds(u, u1, tol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds different sized unit vectors")
  CHECK(runTriangleInequalityHolds(u, u, ntol, TRUE, PRE_EXPECTED), 
   "passing triangleInequalityHolds a negative tolerance")

  /* norm set */
  CHECK(runNorm(u, tol, PROPER_BEHAVIOR, 1.0, NO_EXC_EXPECTED), 
   "checking unit vector norm to be 1.0")
  CHECK(runNorm(n, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing norm a null vector")
  CHECK(runNorm(t, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing norm a 2D array")
  CHECK(runNorm(u, ntol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing norm a negative tolerance")
  CHECK(runNorm(u, tol, BAD_LEVEL_1, -5.0, POST_EXPECTED), 
   "passing norm level 1 badness (i.e., negative result)")
  CHECK(runNorm(z, tol, BAD_LEVEL_2, 5.0, POST_EXPECTED), 
   "passing norm level 2 badness (i.e., result is positive) but zero vector")
  CHECK(runNorm(u, tol, BAD_LEVEL_3, 0.0, POST_EXPECTED), 
   "passing norm level 3 badness (i.e., result is zero) but non-zero vector")

  /* dot set */
  CHECK(runDot(u, z, tol, PROPER_BEHAVIOR, 0.0, NO_EXC_EXPECTED), 
   "checking that dot of unit and zero vectors is 0.0")
  CHECK(runDot(n, u, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a null 1st array")
  CHECK(runDot(t, u, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a 2D 1st array")
  CHECK(runDot(u, n, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a null 2nd array")
  CHECK(runDot(u, t, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a 2D 2nd array")
  CHECK(runDot(u, u1, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot different sized unit vectors")
  CHECK(runDot(u, u, ntol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a negative tolerance")
  CHECK(runDot(t, t, tol, PROPER_BEHAVIOR, 0.0, PRE_EXPECTED), 
   "passing dot a 2D arrays in both args")
  CHECK(runDot(u, u, tol, BAD_LEVEL_1, -5.0, POST_EXPECTED), 
   "passing dot level 1 badness (i.e., negative result) but u=v")
  CHECK(runDot(z, z, tol, BAD_LEVEL_2, 5.0, POST_EXPECTED), 
   "passing dot level 2 badness (i.e., positive result) but u=v=0")

  /* product set */
  CHECK(runProduct(1.0, u, tol, PROPER_BEHAVIOR, u, TRUE, NO_EXC_EXPECTED), 
   "checking that product of 1.0 and unit vector is unit vector")
  CHECK(runProduct(2.0, u, tol, PROPER_BEHAVIOR, u, FALSE, NO_EXC_EXPECTED), 
   "checking that product of 2.0 and unit vector is not unit vector")
  CHECK(runProduct(0.0, n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing product a null array")
  CHECK(runProduct(1.0, t, tol, PROPER_BEHAVIOR, t, TRUE, PRE_EXPECTED), 
   "passing product a 2D array")
  CHECK(runProduct(1.0, u, tol, BAD_LEVEL_1, u, TRUE, POST_EXPECTED), 
   "passing product level 1 badness (i.e., null result)")
  CHECK(runProduct(1.0, u, tol, BAD_LEVEL_2, u, TRUE, POST_EXPECTED), 
   "passing product level 2 badness (i.e., 2D array result)")
  CHECK(runProduct(1.0, u, tol, BAD_LEVEL_3, u, TRUE, POST_EXPECTED), 
   "passing product level 3 badness (i.e., result size mismatch)")

  /* negate set */
  CHECK(runNegate(u, tol, PROPER_BEHAVIOR, nu, TRUE, NO_EXC_EXPECTED), 
   "checking that negate of the unit vector is it's negative")
  CHECK(runNegate(u, tol, PROPER_BEHAVIOR, u, FALSE, NO_EXC_EXPECTED), 
   "checking that negate of the unit vector is not the unit vector")
  CHECK(runNegate(n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing negate a null array")
  CHECK(runNegate(t, tol, PROPER_BEHAVIOR, nu, TRUE, PRE_EXPECTED), 
   "passing negate a 2D array")
  CHECK(runNegate(u, tol, BAD_LEVEL_1, nu, TRUE, POST_EXPECTED), 
   "passing product level 1 badness (i.e., null result)")
  CHECK(runNegate(u, tol, BAD_LEVEL_2, nu, TRUE, POST_EXPECTED), 
   "passing product level 2 badness (i.e., 2D array result)")
  CHECK(runNegate(u, tol, BAD_LEVEL_3, nu, TRUE, POST_EXPECTED), 
   "passing product level 3 badness (i.e., result size mismatch)")

  /* normalize set */
  CHECK(runNormalize(u, tol, PROPER_BEHAVIOR, u, TRUE, NO_EXC_EXPECTED), 
       "checking that normalize of the unit vector is itself")
  CHECK(runNormalize(u, tol, PROPER_BEHAVIOR, nu, FALSE, NO_EXC_EXPECTED), 
       "checking that normalize of the unit vector is not it's negative")
  CHECK(runNormalize(z, tol, PROPER_BEHAVIOR, z, FALSE, DBZ_EXPECTED), 
       "checking that normalize of the zero vector raises DBZ exception")
  CHECK(runNormalize(n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
       "passing normalize a null array")
  CHECK(runNormalize(t, tol, PROPER_BEHAVIOR, u, TRUE, PRE_EXPECTED), 
       "passing normalize a 2D array")
  CHECK(runNormalize(u, ntol, PROPER_BEHAVIOR, u, TRUE, PRE_EXPECTED), 
       "passing normalize a negative tolerance using the unit vector")
  CHECK(runNormalize(u, tol, BAD_LEVEL_1, u, TRUE, POST_EXPECTED), 
       "passing normalize a bad lvl 1 (i.e., null result)")
  CHECK(runNormalize(u, tol, BAD_LEVEL_2, u, TRUE, POST_EXPECTED), 
       "passing normalize a bad lvl 2 (i.e., 2D result)")
  CHECK(runNormalize(u, tol, BAD_LEVEL_3, u, TRUE, POST_EXPECTED), 
       "passing normalize a bad lvl 3 (i.e., result size mismatch)")

  /* sum set (NOTE: tolerance not relevant to sum() API.) */
  CHECK(runSum(u, z, tol, PROPER_BEHAVIOR, u, TRUE, NO_EXC_EXPECTED), 
   "checking that sum of unit and zero vectors is the unit vector")
  CHECK(runSum(u, z, tol, PROPER_BEHAVIOR, nu, FALSE, NO_EXC_EXPECTED), 
   "checking that sum of unit and zero is not negative of the unit vector")
  CHECK(runSum(n, z, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing sum a null 1st array")
  CHECK(runSum(t, n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing sum a 2D 1st array")
  CHECK(runSum(u, n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing sum a null 2nd array")
  CHECK(runSum(u, t, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing sum a 2D as second")
  CHECK(runSum(u, u1, tol, PROPER_BEHAVIOR, z, TRUE, PRE_EXPECTED), 
   "passing sum different sized unit vectors")
  CHECK(runSum(u, z, tol, BAD_LEVEL_1, u, TRUE, POST_EXPECTED), 
   "passing sum a bad level 1 (i.e., null result)")
  CHECK(runSum(u, z, tol, BAD_LEVEL_2, u, TRUE, POST_EXPECTED), 
   "passing sum a bad level 2 (i.e., 2D result)")
  CHECK(runSum(u, z, tol, BAD_LEVEL_3, u, TRUE, POST_EXPECTED), 
   "passing sum a bad level 3 (i.e., result size mismatch)")

  /* diff set (NOTE: tolerance not relevant to diff() API.) */
  CHECK(runDiff(z, u, tol, PROPER_BEHAVIOR, nu, TRUE, NO_EXC_EXPECTED), 
   "checking that diff of zero and unit vector is the neg unit vector")
  CHECK(runDiff(u, z, tol, PROPER_BEHAVIOR, u, TRUE, NO_EXC_EXPECTED), 
   "checking that diff of unit and zero vectors is the unit vector")
  CHECK(runDiff(u, z, tol, PROPER_BEHAVIOR, nu, FALSE, NO_EXC_EXPECTED), 
   "checking that diff of unit and zero vectors is not the neg unit vector")
  CHECK(runDiff(n, u, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing diff a null 1st array")
  CHECK(runDiff(t, u, tol, PROPER_BEHAVIOR, u, TRUE, PRE_EXPECTED), 
   "passing diff a 2D 1st array")
  CHECK(runDiff(u, n, tol, PROPER_BEHAVIOR, n, TRUE, PRE_EXPECTED), 
   "passing diff a null 2nd array")
  CHECK(runDiff(u, t, tol, PROPER_BEHAVIOR, u, TRUE, PRE_EXPECTED), 
   "passing diff a 2D 2nd array")
  CHECK(runDiff(u, u1, tol, PROPER_BEHAVIOR, u, TRUE, PRE_EXPECTED), 
   "passing diff different sized vectors")
  CHECK(runDiff(z, u, tol, BAD_LEVEL_1, nu, TRUE, POST_EXPECTED), 
   "passing diff bad level 1 (i.e., null result)")
  CHECK(runDiff(z, u, tol, BAD_LEVEL_2, nu, TRUE, POST_EXPECTED), 
   "passing diff bad level 2 (i.e., 2D result)")
  CHECK(runDiff(z, u, tol, BAD_LEVEL_3, nu, TRUE, POST_EXPECTED), 
   "passing diff bad level 3 (i.e., result size mismatch)")

  vector_Utils__dump_stats_static(statsFile, &exception);SIDL_CHECK(exception);

  /****************************************************************
   * Now check preconditions only.  Only need three checks:
   *   1) successful execution;
   *   2) precondition violation that is not caught but is
   *      okay anyway; and
   *   3) postcondition violation that is caught.
   ****************************************************************/
  printf("\nCOMMENT: *** ENABLE PRECONDITION ENFORCEMENT ONLY ***\n\n");
/*
 * Enable once new enforcement features fully supported.
 *
  sidl_EnfPolicy_enforceAll(sidl_ContractClass_PRECONDS,
       &exception); SIDL_CHECK(exception);
 */
  vector_Utils__set_checking_static(CHECK_PRECONDITIONS|CHECK_ALWAYS,0,0,
                                    &exception); SIDL_CHECK(exception);

  CHECK(runDot(u, z, tol, PROPER_BEHAVIOR, 0.0, NO_EXC_EXPECTED),
   "checking that dot of unit and zero vectors is 0.0")
  CHECK(runDot(u, u, ntol, PROPER_BEHAVIOR, 1.0, PRE_EXPECTED),
   "passing dot a negative tolerance")
  CHECK(runDot(u, u, tol, BAD_LEVEL_1, -5.0, NO_EXC_EXPECTED),
   "passing dot level 1 badness (i.e., negative result) but u=v")

  vector_Utils__dump_stats_static(statsFile, &exception);SIDL_CHECK(exception);

  /****************************************************************
   * Now check postconditions only.  Only need three checks:
   *   1) successful execution;
   *   2) precondition violation that gets caught; and
   *   3) postcondition violation that is not caught.
   ****************************************************************/
  printf("\nCOMMENT: *** ENABLE POSTCONDITION ENFORCEMENT ONLY ***\n\n");
/*
 * Enable once new enforcement features fully supported.
 *
  sidl_EnfPolicy_enforceAll(sidl_ContractClass_POSTCONDS,
                           &exception); SIDL_CHECK(exception);
 */
  vector_Utils__set_checking_static(CHECK_POSTCONDITIONS|CHECK_ALWAYS,0,0, 
                                    &exception); SIDL_CHECK(exception);

  CHECK(runDot(u, z, tol, PROPER_BEHAVIOR, 0.0, NO_EXC_EXPECTED),
   "checking that dot of unit and zero vectors is 0.0")
  CHECK(runDot(u, u, ntol, PROPER_BEHAVIOR, 1.0, NO_EXC_EXPECTED),
   "passing dot a negative tolerance")
  CHECK(runDot(u, u, tol, BAD_LEVEL_1, -5.0, POST_EXPECTED),
   "passing dot level 1 badness (i.e., negative result) but u=v")

  vector_Utils__dump_stats_static(statsFile, &exception);SIDL_CHECK(exception);

  /****************************************************************
   * Now make sure contract violations are not caught when contract
   * enforcement turned off.  Do this for each type of violation
   * for every method.
   ****************************************************************/
  printf("\nCOMMENT: *** DISABLE ALL CONTRACT ENFORCEMENT ***\n\n");
/*
 * Enable once new enforcement features fully supported.
 *
  sidl_EnfPolicy_enforceNone(&exception); SIDL_CHECK(exception);
 */
  vector_Utils__set_checking_static(0, 0, 0, &exception);SIDL_CHECK(exception);

  CHECK(runIsZero(n, tol, FALSE, NO_EXC_EXPECTED),
   "passing isZero a null array - would be precondition violation")
  CHECK(runIsUnit(n, tol, FALSE, NO_EXC_EXPECTED),
   "passing isUnit a null array - would be precondition violation")
  CHECK(runAreEqual(n, u, tol, FALSE, NO_EXC_EXPECTED),
   "passing areEqual null 1st array - would be precondition violation")
  CHECK(runAreOrthogonal(n, u, tol, FALSE, NO_EXC_EXPECTED),
   "passing areOrthogonal null 1st array - would be precondition violation")
  CHECK(runSchwarzHolds(n, z, tol, FALSE, NO_EXC_EXPECTED),
   "passing schwarzHolds a null 1st array - would be precondition violation")
  CHECK(runTriangleInequalityHolds(n, u, tol, FALSE, NO_EXC_EXPECTED),
   "passing triangleInequalityHolds a null 1st array - would be pre violation")
  CHECK(runNorm(n, tol, PROPER_BEHAVIOR, 0.0, NO_EXC_EXPECTED),
   "passing norm a null vector - would be precondition violation")
  CHECK(runNorm(u, tol, BAD_LEVEL_1, -5.0, NO_EXC_EXPECTED),
   "passing norm level 1 badness (i.e., negative result) - would be post vio.")
  CHECK(runDot(n, u, tol, PROPER_BEHAVIOR, 0.0, NO_EXC_EXPECTED),
   "passing dot a null 1st array - would be precondition violation")
  CHECK(runDot(u, u, tol, BAD_LEVEL_1, -5.0, NO_EXC_EXPECTED),
   "passing dot level 1 badness (i.e., negative result) but u=v - post vio.")
  CHECK(runProduct(0.0, n, tol, PROPER_BEHAVIOR, n, FALSE, NO_EXC_EXPECTED),
   "passing product a null array - would be precondition violation")
  CHECK(runProduct(1.0, u, tol, BAD_LEVEL_1, u, FALSE, NO_EXC_EXPECTED),
   "passing product level 1 badness (i.e., null result) - would be post vio.")
  CHECK(runNegate(n, tol, PROPER_BEHAVIOR, n, FALSE, NO_EXC_EXPECTED),
   "passing negate a null array - would be precondition violation")
  CHECK(runNegate(u, tol, BAD_LEVEL_1, nu, FALSE, NO_EXC_EXPECTED),
   "passing product level 1 badness (i.e., null result) - would be post vio.")
  CHECK(runNormalize(n, tol, PROPER_BEHAVIOR, n, FALSE, NO_EXC_EXPECTED),
   "passing normalize a null array - would be precondition violation")
  CHECK(runNormalize(u, tol, BAD_LEVEL_1, u, FALSE, NO_EXC_EXPECTED),
   "passing normalize a bad lvl 1 (i.e., null result) - would be post vio.")
  CHECK(runSum(n, z, tol, PROPER_BEHAVIOR, n, FALSE, NO_EXC_EXPECTED),
   "passing sum a null 1st array - would be precondition violation")
  CHECK(runSum(u, z, tol, BAD_LEVEL_1, u, FALSE, NO_EXC_EXPECTED),
   "passing sum a bad level 1 (i.e., null result) - would be post violation")
  CHECK(runDiff(n, u, tol, PROPER_BEHAVIOR, n, FALSE, NO_EXC_EXPECTED),
   "passing diff a null 1st array - would be precondition violation")
  CHECK(runDiff(z, u, tol, BAD_LEVEL_1, nu, FALSE, NO_EXC_EXPECTED),
   "passing diff bad level 1 (i.e., null result) - would be post violation")

  vector_Utils__dump_stats_static(statsFile, &exception);SIDL_CHECK(exception);

  sidl__array_deleteRef((struct sidl__array*)t);
  sidl__array_deleteRef((struct sidl__array*)u);
  sidl__array_deleteRef((struct sidl__array*)nu);
  sidl__array_deleteRef((struct sidl__array*)u1);
  sidl__array_deleteRef((struct sidl__array*)z);
  
  
  synch_RegOut_close(tracker, &exception); SIDL_CHECK(exception);
  synch_RegOut_deleteRef(tracker, &exception); SIDL_CHECK(exception);

  return 0;
 EXIT:
  {
    sidl_BaseInterface throwaway_exception = NULL;
    if (tracker) {
      sidl_BaseInterface exception2 = NULL;
      synch_RegOut_forceFailure(tracker, &exception2);
      if (exception2) {
        puts("TEST_RESULT FAIL\n");
        sidl_BaseInterface_deleteRef(exception2, &throwaway_exception);
      }
      synch_RegOut_deleteRef(tracker, &throwaway_exception);
    }
    sidl_BaseInterface_deleteRef(exception, &throwaway_exception);
    return -1;
  }
}
