/*
 * Description  Extremely simple HTTP net-fetch client-transport.
 * CVS          $Id: http-fetch.c,v 1.22 2000/06/14 02:29:37 karlheg Exp $
 * Author       Marcel Harkema <marcel@debian.org>
 *
 * Copyright (C) 1999, 2000 Marcel Harkema
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <time.h>
#include <unistd.h>

#if defined (_TESTING_)
  #define _DEBUG_
#endif

#if defined (_TESTING_) && defined (_DEBUG_)
#include <newt.h>
#endif

#include "dbootstrap.h"
#include "util.h"
#include "lang.h"

#include "net-fetch.h"
#include "http-fetch.h"

static int
nf_http_set_nonblock(int desc, int value)
{
  int oldflags = fcntl(desc, F_GETFL, 0);

  if (oldflags == -1)
    return -1;

  if (value != 0)
    oldflags |= O_NONBLOCK;
  else
    oldflags &= ~O_NONBLOCK;

  return fcntl(desc, F_SETFL, oldflags);
}


static int
nf_http_write(int sock, char *buf)
{
  int r, offset, len;

  len = strlen(buf);
  offset = 0;

  do {
    r = write(sock, buf + offset, len - offset);
    if (r < 0) {
      if (errno == 0
       || errno == EINTR
       || errno == EAGAIN || errno == EWOULDBLOCK) {
        usleep(25000);
        continue;
      }
      close(sock);
      perrorBox("nf_http_write :: write");
      return -1;
    }
    else {
      offset += r;
    }
  } while (offset < len);

  return 0;
}


static int
nf_http_read(int sock, char *buf, int buflen, int *eof)
{
  int            r, offset;
  fd_set         rfds;
  char           in;
  struct timeval tv;

  offset = 0;
  *eof = 0;

  if (buflen <= 0)
    return 0;

  do {
    r = read(sock, buf + offset, buflen - offset);

    if (r == 0) {
      *eof = 1;
      break;
    }
    else if (r < 0) {
      if (errno != 0
       && errno != EINTR
       && errno != EAGAIN && errno != EWOULDBLOCK) {
        perrorBox("nf_http_read :: read");
        return -1;
      }
      usleep(25000);
      continue;
    }
    else {
      offset += r;
      buf[offset] = '\0';
      FD_ZERO(&rfds);
      FD_SET(0, &rfds);
      tv.tv_sec = tv.tv_usec = 0;
      if (select(1, &rfds, NULL, NULL, &tv)) {
        if (read(0, &in, 1) > 0 && (in == '\n' || in == '\r' || in == ' ' || in == '\003')) {
          if (yesNoBox(_("Are you sure you want to cancel this download?"), _("Cancel download")) == DLG_YES) {
            return -2;
          }
        }
      }
    }
  }
  while (r >= 0 && buflen - offset > 0);

#if defined (_TESTING_) && defined (_DEBUG_)
  /* File transfer via localhost is too fast, slow it down so I can
   * look at the fetch window for a while. */
  // usleep(50000);
#endif

  return offset;
}

struct gethost_env_s {
  char * con_host;
  struct hostent *hst;
};

static void
nf_http_fetchfile_gethostbyname (void *result)
{
  struct gethost_env_s *res = (struct gethost_env_s *) result;
  res->hst = gethostbyname (res->con_host);
  return;
}

struct connect_env_s {
  int sock;
  struct sockaddr *svr;
  size_t svr_size;
  int status;
};

static void
nf_http_fetchfile_connect (void *result)
{
  struct connect_env_s *res = (struct connect_env_s *) result;
  res->status = connect (res->sock, res->svr, res->svr_size);
  return;
}

int
nf_http_fetchfile(char               *server_host,
                  unsigned short int  server_port,
                  char               *proxy_host,
                  unsigned short int  proxy_port,
                  char               *remote_path,
                  char               *remote_filename,
                  char               *local_filename)
{
  int                 sock;
  struct sockaddr_in  svr;
  struct hostent     *hst = NULL;
  char               *con_host;
  int                 con_port;
  int                 use_proxy;
  char                buf[65336];
  FILE               *fp = NULL;
  int                 seen_reply_header, seen_http_ok;
  int                 offset;
  unsigned long       content_length;
  unsigned long       bytes_read;
  struct stat         statbuf;
  char                sbuf[MAXPATHLEN + 1024];
  int                 doresume = 0;
  unsigned long       oldsize = 0;

#if defined (_TESTING_) && defined (_DEBUG_)
  int curline;
#endif

  if (proxy_host != NULL
      && strcmp (proxy_host, "")
      && strcmp (proxy_host, "none")
      && proxy_port > 0) {
    con_host  = proxy_host;
    con_port  = proxy_port;
    use_proxy = 1;
  }
  else {
    con_host  = server_host;
    con_port  = server_port;
    use_proxy = 0;
  }

#if defined (_TESTING_) && defined (_DEBUG_)
  newtDrawRootText (2, 2, "Connecting ... [socket]     ");
  newtRefresh();
#endif

  if ((sock = socket (PF_INET, SOCK_STREAM, IPPROTO_TCP)) < 0) {
    perrorBox ("nf_http_fetchfile :: socket");
    return -1;
  }

#if defined (_TESTING_) && defined (_DEBUG_)
  newtDrawRootText (2, 2, "Connecting ... [dns]        ");
  newtRefresh();
#endif

  svr.sin_family = AF_INET;
  svr.sin_port   = htons (con_port);

  snprintf (sbuf, sizeof (sbuf) - 1,
	   _("Resolving %s via DNS ...  If there is an error it will eventually time out and return control."),
	   con_host);

  {
    struct gethost_env_s result = { con_host, NULL };
    int threw = do_interruptable (sbuf,
				  &nf_http_fetchfile_gethostbyname,
				  (void *) &result);
    if (result.hst == NULL) {
      if (threw) {
	snprintf(sbuf, sizeof (sbuf) - 1,
		 "nf_http_fetchfile :: Interrupted by operator.\n");
      }
      else {
	snprintf(sbuf, sizeof (sbuf) - 1,
		 "nf_http_fetchfile :: %s couldn't be resolved, %s (h_errno=%d).\n",
		 con_host, hstrerror(h_errno), h_errno);
      }
      close (sock);
      problemBox (sbuf, _("Problem"));
      return -1;
    }
    hst = result.hst;
  }

#if defined (_TESTING_) && defined (_DEBUG_)
  newtDrawRootText (2, 2, "Connecting ... [connect]    ");
  newtRefresh();
#endif

  svr.sin_addr = *(struct in_addr *) hst->h_addr;
  snprintf (sbuf, sizeof (sbuf) - 1,
	    _("Connecting to %s ...  If there is an error it will eventually time out and return control."),
	    con_host);
  {
    struct connect_env_s result = { sock, (struct sockaddr *) &svr, sizeof (svr), 0 };
    int threw = do_interruptable (sbuf,
				  &nf_http_fetchfile_connect,
				  (void *) &result);
    if (threw) {
      close (sock);
      problemBox (_("Interrupted by operator."), _("Problem"));
      return -1;
    }
    /* @@ Karlheg: how can this ever return EINPROGRESS when the sock is not nonblock? */
    if (result.status && errno != EINPROGRESS) {
      close (sock);
      perrorBox ("nf_http_fetchfile :: connect");
      return -1;
    }
  }

#if defined (_TESTING_) && defined (_DEBUG_)
  newtDrawRootText (2, 2, "Connecting ... [non-block]  ");
  newtRefresh();
#endif

  snprintf (sbuf, sizeof (sbuf) - 1,
	    _("Requesting %s ...  If there is an error it will eventually time out and return control."),
	    remote_filename);
  pleaseWaitBox (sbuf);

  /* @@ Marcel TODO: connect sequence unfinished. finish EINPROGRESS handling. */

  if (nf_http_set_nonblock(sock, 1) < 0) {
    close (sock);
    boxPopWindow();		/* Requesting <remote_filename> ... */
    perrorBox ("nf_http_fetchfile :: nf_http_set_nonblock");
    return -1;
  }

  /* changed \n\n to \r\n to better match the spec -randolph */
  if (use_proxy) {
    snprintf (buf, sizeof (buf) - 1,
	      "GET http://%s:%d/%s/%s HTTP/1.1\r\nHost: %s:%d\r\n",
	      server_host, server_port, remote_path, remote_filename,
	      server_host, server_port);
  }
  else {
    snprintf (buf, sizeof (buf) - 1,
	      "GET /%s/%s HTTP/1.1\r\nHost: %s:%d\r\n",
	      remote_path, remote_filename,
	      server_host, server_port);
  }

  /* see if the file already exists; if so, try to resume - randolph */
  if (stat (target_path (local_filename), &statbuf) >= 0) {
    boxPopWindow();		/* Requesting <remote_filename> ... */
    if (yesNoBox (_("Previous download detected. Attempt to resume?"),
		  _("Resume download?")) == DLG_YES) {
      char tmpbuf[256];
      snprintf (tmpbuf, sizeof (tmpbuf) - 1,
		"Range: bytes=%ld-\r\n", statbuf.st_size);
      strcat (buf, tmpbuf);
      oldsize = statbuf.st_size;
#if defined (_TESTING_) && defined (_DEBUG_)
      newtDrawRootText (2, 2, "Attempting to resume download  ");
      newtRefresh();
#endif
    }
    pleaseWaitBox (sbuf);	/* Assumes that sbuf hasn't been changed!! */
  }

  /* End the request header */
  strcat (buf, "\r\n");
  
  if (nf_http_write (sock, buf) < 0) {
    close (sock);
    boxPopWindow();		/* Requesting <remote_filename> ... */
    return -1;
  }

#if defined (_TESTING_) && defined (_DEBUG_)
  newtDrawRootText (2, 2, buf);
  newtRefresh();
  curline = 5;
#endif

  buf[0]            = '\0';
  seen_reply_header = 0;
  offset            = 0;
  content_length    = 0;
  seen_http_ok      = 0;
  bytes_read        = 0;

  boxPopWindow();		/* Requesting <remote_filename> ... */

  for (;;) {
    int r, eof;

    if ((r = nf_http_read (sock, buf + offset, sizeof (buf) - 2 - offset, &eof)) < 0)
      break;

    offset += r;

    if (! seen_reply_header) {
      do {
	char      *cp;
	const int  hdr_size = 2048;
	char       header[hdr_size];
	int        hdr_len;
	int        hdr_copied;
	
	if (eof)
	  break;
	
	if ((cp = strstr (buf, "\r\n")) == NULL)
	  break;
	
	hdr_len                = (cp - buf) + 2;
	hdr_copied             = hdr_len > hdr_size ? hdr_size : hdr_len;
	memcpy (header, buf, hdr_copied);
	header[hdr_copied - 2] = '\0';
	
#if defined (_TESTING_) && defined (_DEBUG_)
	newtDrawRootText (2, curline, header);
	newtRefresh();
	curline++;
#endif
	
	memmove (buf, buf + hdr_len, offset - hdr_len);
	offset -= hdr_len;
	if (! strncmp (header, "HTTP/", 5)) {
	  if (strstr (header, " 200 ")) {
	    seen_http_ok = 1;
	    doresume     = 0;
	    bytes_read   = 0;
	    oldsize      = 0;
	  }
	  if (strstr (header, " 206 ")) /* partial downloads */ {
	    seen_http_ok = 1;
	    doresume     = 1;
	    bytes_read   = oldsize;
	  }
	  if (strstr (header, " 404 "))
	    problemBox (header, "Error 404");
	}
	if (! strncasecmp (header, "Content-Length: ", 16)) {
	  unsigned long progress_size = content_length = strtoul (header + 16, NULL, 0);
	  if (doresume) progress_size += oldsize;
	  nf_progress_open (remote_filename, (off_t) progress_size);
	}
	if (!strcmp(header, "")) {
	  seen_reply_header = 1;
	  break;
	}
      } while (offset > 0);
    }

    if (seen_reply_header) {
      if (fp == NULL &&
	  (fp = fopen (target_path (local_filename), (doresume ? "a" : "w"))) == NULL) {
	snprintf (sbuf, sizeof (sbuf) - 1,
		  "nf_http_fetchfile :: couldn't open file %s for writing\n",
		  target_path (local_filename));
	close (sock);
	problemBox (sbuf, _("Problem"));
	return -1;
      }
      if (seen_http_ok) {
	while (offset) {
	  r = fwrite (buf, 1, offset, fp);
	  if (r < 0) {
	    snprintf (sbuf, sizeof (sbuf) - 1,
		      "nf_http_fetchfile :: error writing to output file %s\n",
		      target_path (local_filename));
	    problemBox (sbuf, _("Problem"));
	    break;
	  }
	  else {
	    offset -= r;
	    bytes_read += r;
	    nf_progress_update (bytes_read);
	  }
	}
      }
      else {
	break;
      }
    }
    if (eof)
      break;
  }

  nf_progress_close();

  close (sock);
  fclose (fp);

  if (! seen_reply_header || ! seen_http_ok) {
    snprintf (sbuf, sizeof (sbuf) - 1,
	      "nf_http_fetchfile :: file download failed, HTTP OK not received\n");
    problemBox (sbuf, _("Problem"));
    return -1;
  }

  if (stat (target_path (local_filename), &statbuf) != 0) {
    snprintf (sbuf, sizeof (sbuf) - 1,
	      "nf_http_fetchfile :: couldn't stat %s\n",
	      target_path(local_filename));
    perrorBox (sbuf);
    return -1;
  }

  if ((statbuf.st_size - oldsize) != content_length) {
    snprintf (sbuf, sizeof (sbuf) - 1,
	      "nf_http_fetchfile :: file download failed, content-length is %ld bytes, file size is %ld bytes\n",
	      content_length, statbuf.st_size);
    problemBox (sbuf, _("Problem"));
    return -1;
  }

  return 0;
}
