#! /bin/bash
# Boot Disk maker. Bruce Perens, July-September 1995
# Eric Delaunay, 1998 [silo boot]
# Enrique Zanardi, 1998
# This is free software under the GNU General Public License.

. ./common.sh
umask 022

#-----------------------------------------------
# Write (or don't write) the install.sh script
#
write_installsh () {

    if [ "x$noinstall" = x ]; then

	m4 -D__ERROR_MESS__=$scripts/messages/$language/install.sh.txt \
	   -D__version__=$kverfl -D__arch__=$arch \
	   <$scripts/install.sh >$mnt/install.sh
	chmod 755 $mnt/install.sh
    fi
}

#-----------------------------------------------
# write SILO boot infos
#
write_silo () {
    disksize=`echo $blocks | sed -e 's/\([0-9]\)\([0-9][0-9]\)[0-9]/\1.\2/'`
    cat > $mnt/silo.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux.bin
label=linux
append="load_ramdisk=1 prompt_ramdisk=1 disksize=$disksize"
read-write
EOF
    write_installsh
}


#-----------------------------------------------
# write PowerPC boot infos
#
# For the moment only with quik
#
write_powerpc () {
    mkdir $mnt/etc
    mkdir $mnt/boot
    cat > $mnt/etc/quik.conf << EOF
partition=1
message=/debian.txt
root=/dev/fd0
timeout=600
image=/linux.bin
label=Linux
append="load_ramdisk=1 prompt_ramdisk=1"
read-write
EOF
    write_installsh
}

#-----------------------------------------------
# write SysLinux boot infos
#
write_syslinux () {
    local flavor
    relax
	
    flavor=${subarch:+"flavor=$subarch"}
    [ -f "$sfont" ] && cp $sfont $mnt
    # Question: shouldn't 1440 be '$blocks'
    # font.psf is copied to the disk by the user (see scripts/dos/setlang.bat)
    disksize=`echo 1440 | sed -e 's/\([0-9]\)\([0-9][0-9]\)[0-9]/\1.\2/'`
    { [ -f "$sfont" ] && echo FONT $sfont || echo FONT font.psf; cat $scripts/syslinux.cfg; } |\
	m4 -D__size${blocks}__ -D__disksize__="disksize=${disksize}" \
	   -D__flavor__="${flavor}" | $nlfmt >$mnt/syslinux.cfg
    if [ "$language" = "ja" ]; then
	sed -e 's/vga=normal/vga=normal video=vga16:on/' $mnt/syslinux.cfg > ./syslinux.cfg.ja
	mv ./syslinux.cfg.ja $mnt/syslinux.cfg
    fi
    sed -e s/__arch__/$arch/ \
	<$scripts/rdev.sh >$mnt/rdev.sh
    chmod 755 $mnt/rdev.sh
    write_installsh
}

#-----------------------------------------------
# write m68k info
#
write_m68kinfo () {
    write_installsh
}

#-----------------------------------------------
# Alpha: add aboot bootloader and configuration files for aboot and milo
#
write_aboot () {
    if test -n "$APB"; then
        cp apb.exe $palcode $mnt/
        cp apb_install.txt $mnt/apb.txt
        cp apb_README.txt $mnt/up1000.txt
        cat > $mnt/apb.cfg <<EOF
debian_install|bootl $palcode linux root=/dev/fd0 load_ramdisk=1
EOF
    else
        # generic disk, put both milo and aboot config on it
        umount $mnt
        e2writeboot $floppy /boot/bootlx
        mount -o loop $fsopts -t $fstype $floppy $mnt
        #
        # for aboot a compressed kernel needs the suffix gz
        # (not strictly true, but it's useful)
        #
        mv $mnt/linux.bin $mnt/linux.gz
        ln -s linux.gz $mnt/linux.bin
        mkdir $mnt/etc
        cat  > $mnt/etc/aboot.conf << EOF
#
# Bootparameters for Debian-Installation 
#
# Installation-Boot with:  boot dva0 -fl 0
# Rescue-Boot with:        boot dva0 -fi linux.gz -fl root=/dev/...
#
# from SRM-Console
#
0:linux.gz ro root=/dev/fd0 load_ramdisk=1
EOF
        cat > $mnt/milo.cfg << EOF
#
# MILO-Configuration for Debian-Boot
#
# Boot with:  boot fd0:
#
timeout=100
 
image=fd0:linux.bin
label=install
root=/dev/fd0
append="load_ramdisk=1 prompt_ramdisk=1"
EOF

    fi
    write_installsh
}



#-----------------------------------------------
# write VMELILO boot infos
#
write_vmelilo () {
    mkdir $mnt/etc
    mkdir $mnt/boot

    # determine which loop device the image is mounted on
    loopdev=$(mount | perl -ne 'm|$floppy.*loop=(/dev/loop[0-9]+)| && print $1;')

    if [ "$blocks" = "1440" ]; then
	cat > $mnt/etc/vmelilo.conf << EOF
prompt  = true
default = install
message = /debian.txt
root    = /dev/sda
image   = /linux.bin
read-write

[boot]
label   = Install
append  = load_ramdisk=1 prompt_ramdisk=1

[boot]
label   = Rescue
append  = load_ramdisk=1 prompt_ramdisk=1 single
EOF
    else
	cat > $mnt/etc/vmelilo.conf << EOF
prompt  = true
default = install
message = /debian.txt
root    = /dev/ram0
image   = /linux.bin
read-write

[boot]
label   = Install
ramdisk = /root.bin

[boot]
label   = Rescue
ramdisk = /root.bin
append  = single
EOF
    fi
    case $subarch in
	bvme6000)
            vmearch=bvme
            ;;
	mvme16x|mvme147)
            vmearch=mvme
            ;;
    esac
    vmelilo -f -a $vmearch -b /dev/null -w $mnt -d $loopdev
    write_installsh
}

#-----------------------------------------------

# write elilo boot infos
#
write_elilo () {
    info "installing ia64 bootloader"
    tmp=${tmpdir}/boot-floppies/elilo.$$
    info "downloading elilo"
    elilopkg=`grab_paths elilo`
    debug "extracting $elilopkg"
    rm -rf $tmp
    dpkg --extract $elilopkg $tmp
    cp $tmp/usr/lib/elilo/elilo.efi $mnt
    rm -rf $tmp
    mkdir -p $mnt/efi/boot
    cp $mnt/elilo.efi $mnt/efi/boot/bootia64.efi
    echo "elilo linux" > $mnt/install.nsh
    cat > $mnt/elilo.conf << EOF
delay=20

image=linux.bin
	label=linux
	read-only
	initrd=root.bin
	root=/dev/ram
EOF

    write_installsh
}

# Print a usage message and exit if the argument count is wrong.
if [ $# -lt 7 ]; then
    echo "usage: $0 archive subarch-or-flavor kernel-version root-image blocks debian-version system language [sfont] [noinstall]" 1>&2
    cat 1>&2 << EOF

        archive: the directory where Debian binary packages will be downloaded
        subarch-or-flavor: subarchitecture or kernel flavor, or "" for none
        kernel-version: version of the kernel
	root-image: a compressed disk image to load in ramdisk and mount as root
	blocks: the size of the floppy in 1K blocks, use 720, 1200, 1440, or 2880
	debian-version: version of the Debian release
#	system: name of the target system
	language: language for the boot messages and help files
	sfont: name of the font loaded by syslinux, if any
	noinstall: make this non-blank to make this a boot disk only, optional
EOF

    exit -1
fi


# location of the package archive
archive=$1

# subarchitecture or flavor
subarch=$2

kver=$3

# location of the root filesystem image
rootimage=$4

# number of blocks per floppy
blocks=$5

# Debian version
debianversion=$6

# system
system=$7

# language for the boot messages and help files
language=$8

# font loaded by syslinux, if any
sfont=$9

# set to non blank if you want this to be for booting only
noinstall=$9

# charset for the boot messages and help files
# FIXME: this should be a runtime parameter
charset=latin1

testroot

# block device for the floppy disk
floppy=$(tempfile -p fimg -m 644 -d /var/tmp)

mnt=${tmpdir}/boot-floppies/mnt.rescue

# home of the scripts used by this script
scripts=`pwd`/scripts/rescue

# make sure the kernel package is available
if [ ! -d $archive ]; then
    error "can't find archive '$archive'"
fi

# make sure language is valid
if [ ! -d $scripts/messages/$language ]; then
    error "invalid language '$language', no dir $scripts/messages/$language"
fi



# Try to unmount anything that should not be mounted. Aborted runs of this
# script may have left things mounted.
(umount /mnt; umount $mnt; umount $floppy; true)2>/dev/null >/dev/null

make_tmpdir $mnt

# zero the entire disk, so that when we compress the raw disk image,
# unused blocks will compress well
if [ "$arch" != sparc ]; then
	info "making loop filesystem in $floppy, size ${blocks}k"
	dd if=/dev/zero of=$floppy bs=1k count=$blocks
fi

## FIXME: what is this for?
DEBIAN_KERNEL_IMAGE=`basename $archive .deb | sed -e s/_$arch\$//`
#case $arch in
#    alpha) SUBARCH=`expr $subarch : 'alpha_\(.*\)'`;;
#esac

umount $floppy >/dev/null 2>&1 || true

info "formatting and conditioning the new image"
case "$arch" in
    alpha)
        case $SUBARCH in
            nautilus)
                mkdosfs -r 112 -F 12 $floppy $blocks
                APB=1
                palcode=up1000.pal
                fstype=msdos
                fsopts="-o fat=12"
                nlfmt="cat -"
                ;;
            *)
                mke2fs -O none -F -q $floppy $blocks
                fstype=ext2
                nlfmt="cat -"       #  UNIX newline convention
                ;;
        esac
        ;;
    arm)
        # netwinder doesn't use this, but we need it to install kernel
        # other sustems will need more help
        mke2fs -F -q $floppy $blocks
        fstype=ext2
        nlfmt='cat -'       # UNIX newline convention
        ;;
    i386)
        if [ -f /usr/lib/syslinux/img$blocks""k.gz ]; then
            gunzip < /usr/lib/syslinux/img$blocks""k.gz > $floppy
        elif [ -f $scripts/img$blocks""k.gz ]; then
            gunzip < $scripts/img$blocks""k.gz > $floppy
        else
            error "can't find $blocks KB floppy image"
        fi
        syslinux $floppy    # Use the "fast" loader.
        fstype=msdos
        if [ -f "$sfont" ]; then   # MSDOS newline convention
            nlfmt=sed\ s/$/$'\r'/
        else
            nlfmt="recode $charset..ibmpc"
        fi
        fsopts="-o fat=12"
        ;;
    ia64)
	mkdosfs -r 112 -F 16 $floppy $blocks
	fstype=vfat
	nlfmt='cat -'
	;;
    m68k)
        case "$subarch" in
            amiga|atari)
                mkdosfs -r 112 -F 12 $floppy $blocks
                fstype=msdos
                fsopts="-o fat=12"
                nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
                ;;
            mac)
                # dhd: We are not actually going to use this to boot, and I can't
	        # seem to get HFS or msdos images to transfer reliably over NFS,
		# so we will use ext2 here.  This is still suboptimal, dbootstrap
	        # should be able to install a "loose" kernel... work in progress
                mke2fs -F -q $floppy $blocks
		fstype=ext2
		nlfmt="recode $charset..applemac"    # Does not matter really
                ;;
            bvme6000|mvme16x|mvme147)
                # uses ext2 formatted floppies & VMELILO boot loader
                mke2fs -F -q $floppy $blocks
                fstype=ext2
                nlfmt='cat -'       # UNIX newline convention
                ;;
        esac
        ;;
    mips|mipsel)
       mke2fs -F -q $floppy $blocks
       fstype=ext2
       nlfmt='cat -'       # UNIX newline convention
       ;;
    powerpc)
        case "$subarch" in
            apus|chrp)
                mkdosfs -r 112 -F 12 $floppy $blocks
                fstype=msdos
                fsopts="-o fat=12"
                nlfmt="recode $charset..ibmpc"       # MSDOS newline convention
                ;;
            mbx)
                error "don't know what to do for MBX"
                ;;
            pmac|newpmac)
                mke2fs -F -q $floppy $blocks		# just for resc1440; 
                fstype=ext2				# really want HFS for boot floppy
                nlfmt="recode $charset..applemac"	# Mac newline convention
                ;;
	    prep)
		# use ext2 formatted floppies for now, a port of milo or
		# quik is not far off
		mke2fs -F -q $floppy $blocks
		fstype=ext2
                nlfmt='cat -'       			# UNIX newline convention
            esac
            ;;
    sparc)
        # [ED] sparc uses ext2 formatted floppies & SILO boot loader. The
	# silo images are floppies with a Sun Disklabel & SILO installed.
	# They come with the silo package, for our convienience.
	#
	tmp=${tmpdir}/boot-floppies/silo_proto.$$
	if [ ${subarch} = "sun4dm-pci" ]; then
            silokver="-sun4cdm"
	else
            silokver="-${subarch}"
	fi
	info "downloading silo"
	silopkg=`grab_paths silo`
	debug "extracting $silopkg"
	rm -rf $tmp
	dpkg --extract $silopkg $tmp
	gunzip < $tmp/usr/lib/silo/silo${blocks}k${silokver}.gz > $floppy
	rm -rf $tmp
	fstype=ext2
	nlfmt='cat -'	# UNIX newline convention
        ;;
    hppa)
	# hppa uses ext2 formatted floppies.  We can't boot from floppies
        # so this is just a way to fit in the typical boot method
	mke2fs -F -q $floppy $blocks
	fstype=ext2
	nlfmt='cat -'	# UNIX newline convention
	;;
    s390)
	# s390 uses ext2 formatted floppies.  We can't boot from floppies
        # so this is just a way to fit in the typical boot method
	mke2fs -F -q $floppy $blocks
	fstype=ext2
	nlfmt='cat -'	# UNIX newline convention
	;;
esac

sync
relax
sync

# mount the disk image to install additional files
info "mounting new disk image at $mnt"
mount -o loop $fsopts -t $fstype $floppy $mnt

# for atari, copy bootloader and linux.bin on disk
if [ "$subarch" = atari ]; then
    info "installing atari bootloader"
    mkdir $mnt/auto
    # hardcoded bootargs; kernel is in \, bootstrap in \auto
    cat <<EOF > $mnt/auto/bootargs
-s -k \linux.bin -r \root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
    cp m68k-specials/bootstra.ttp $mnt/auto/bootstra.prg
    # both bootstrap and kernel/ramdisk in \
    cat <<EOF > $mnt/bootargs
-s -k linux.bin -r root.bin root=/dev/ram load_ramdisk=1 video=keep
EOF
    cp m68k-specials/bootstra.ttp $mnt/bootstra.ttp
fi

##
## install kernel
##

relax
	
case "$arch/$subarch" in
    powerpc/*pmac)
	info "compressing and installing kernel 'linux$subarch.bin'"
	gzip -9f linux$subarch.bin
	cp linux$subarch.bin.gz $mnt/linux.gz
	;;
    hppa*|mips*)		
	info "compressing and installing kernel 'linux$subarch.bin'"
	gzip -9cf linux$subarch.bin > $mnt/linux.gz
	;;
    *)
	info "installing kernel 'linux$subarch.bin'"
	cp linux$subarch.bin $mnt/linux.bin
	;;
esac

relax

# only i386 can use rdev
if [ "$arch" = i386 ]; then
    debug "running rdev on kernel"
    /usr/sbin/rdev -R $mnt/linux.bin 1
    if [ "$blocks" = "2880" ]; then
	/usr/sbin/rdev -r $mnt/linux.bin 0
    else
	/usr/sbin/rdev -r $mnt/linux.bin 49152
    fi
    /usr/sbin/rdev -v $mnt/linux.bin -1
    /usr/sbin/rdev $mnt/linux.bin /dev/ram0
fi

# copy root.bin to 2880k floppies
if [ \( "$arch" = i386 -a "$blocks" = "2880" \) -o \
     \( "$arch/$subarch" = "m68k/bvme6000" -a "$blocks" = "2880" \) -o \
     \( "$arch" = "powerpc" -a "$subarch" != "newpmac" -a "$blocks" = "2880" \) ]; then
    info "copying root filesystem ($rootimage) into filesystem"
    cp $rootimage $mnt/root.bin
elif [ "$arch" = ia64 ]; then
    info "copying root filesystem ($rootimage) into filesystem"
    cp $rootimage $mnt/root.bin
else
    info "rescue image too small to include root.bin"
fi	

# alpha and sun4u can't afford the extra space needed
if [ "$arch" != alpha -a \
     \( "$arch" != "sparc" -o "$subarch" != "sun4u" \) ]; then
    info "copying in sys_map$subarch and config$subarch"
    cp sys_map${subarch}.gz $mnt/sys_map.gz
    cp config${subarch}.gz $mnt/config.gz
fi

relax

info "copying help messages"

suffix=".$arch"                 # can be overriden below

case "$arch/$subarch" in
    */bvme6000|*/mvme16x|*/mvme147)
        suffix=.vme
	files=debian.txt
	;;
    powerpc/*|sparc/*|alpha/*|ia64/*)
	files="debian.txt readme.txt"
	;;
    *)
	files="debian.txt readme.txt f1.txt f2.txt f3.txt f4.txt f5.txt f6.txt f7.txt f8.txt f9.txt f10.txt" 
	;;
esac

# cosmetics, create a version variable with or withour flavor
if [ -z "$subarch" ]; then
   export kverfl=$kver
else
   export kverfl=$kver-$subarch
fi

for i in $files; do
    file=$scripts/messages/$language/$i$suffix
    if [ ! -f $file ]; then
	file=$scripts/messages/$language/$i
    fi
    if [ ! -f $file ]; then
        error "can't find file '$file'"
    fi
    debug "m4 processing $file"
    m4 -EP -D__kernel_version__=$kverfl \
           -D__kernel_image__=$DEBIAN_KERNEL_IMAGE \
	   -D__size${blocks}__ \
	   -D__debianversion__=$debianversion \
	   -D__arch__=$arch $file \
		| $nlfmt > $mnt/$i
    testexit
done

info "installing boot loader"
case "$arch" in
    sparc)
	write_silo
        ;;
    alpha)
        write_aboot
        ;;
    powerpc)
	case $subarch in
	    apus|bbox|chrp|mbx|prep)
		# no bootloader yet, write out install.sh like m68k
		write_m68kinfo
                ;;
	    pmac|newpmac)
		# boot from vmlinux.coff installer for now.
		# write_powerpc
		write_m68kinfo
                ;;
        esac
        ;;
    m68k)
	case $subarch in
	    bvme6000|mvme16x|mvme147)
		write_vmelilo
                ;;
	    atari|amiga|mac)
		# these LILOs not implemented yet, but need to write install.sh
		echo "Not writing syslinux for $subarch !"
		write_m68kinfo
                ;;
	esac
        ;;
    i386)
	write_syslinux
        ;;
    ia64)
	write_elilo
	;;
    arm)
	write_m68kinfo
        ;;
    hppa)
	write_m68kinfo
	;;
    mips)
	write_m68kinfo
	;;
    mipsel)
	write_installsh
	;;
    s390)
	write_installsh
	;;
esac

if [ "x$noinstall" = x ]; then
    echo rescue >$mnt/type.txt || error "Couldn't write type.txt"
else
    echo boot >$mnt/type.txt || error "Couldn't write type.txt"
fi

# do a few tests
[ -f $mnt/linux.bin -o -f $mnt/linux.gz ] || error "can't find kernel"
[ -f $mnt/install.sh -o "x$noinstall" != x ] || error "can't find install.sh"
[ -x $mnt/install.sh -o "x$noinstall" != x ] || error "install.sh is not excutable"

relax

info "unmounting, cleaning up"
sync
umount $mnt

# sometimes with new kernels (on building machine), the umount is not quick
# enough and the build fails. Trying to compensate with multiple tries and
# sync's
sync
sleep 1
umount $mnt  > /dev/null 2>&1 || umount $mnt  > /dev/null 2>&1 || true
sync
sleep 1

rmdir $mnt
rm -fr /var/tmp/extract-tmp-$$ resc${blocks}${subarch}.bin
mv $floppy resc${blocks}${subarch}.bin

info "rescue image complete:" `ls -l resc${blocks}${subarch}.bin`

# make root disk for i386, 1200k or 1440k
if [ "$arch" = i386 -a $blocks -le 2800 ]; then
    info "making detached root disk root${blocks}${subarch}.bin from $rootimage"
    dd if=/dev/zero of=root${blocks}${subarch}.bin bs=1k count=$blocks
    dd if=$rootimage of=root${blocks}${subarch}.bin bs=1k count=$blocks conv=notrunc
    rootimagesz="$(ls -l $rootimage | awk '{print $5}')"
    floppysz="$(ls -l root${blocks}${subarch}.bin | awk '{print $5}')"
    if [ "$rootimagesz" -gt "$floppysz" ] ; then
        error "$rootimage is larger than root${blocks}${subarch}.bin ($rootimagesz > $floppysz)"
    fi
elif  [ "$arch" = powerpc -a $blocks = 1440 ]; then
    # don't create separate root disk for newpmac; they're exactly the same
    tsubarch="$subarch"
    if [ "$tsubarch" = newpmac ]; then
	tsubarch=pmac
    fi
    info "making detached root disk root${blocks}${tsubarch}.bin from $rootimage"
    if [ -n "$i18n_low_space" ] ; then
       info "lowspace flag is set, using rootmini.bin as rootfs image"
       rootimage=rootmini.bin
    fi
    dd if=/dev/zero of=root${blocks}${tsubarch}.bin bs=1k count=$blocks
    dd if=$rootimage of=root${blocks}${tsubarch}.bin bs=1k count=$blocks conv=notrunc
    rootimagesz="$(ls -l $rootimage | awk '{print $5}')"
    floppysz="$(ls -l root${blocks}${tsubarch}.bin | awk '{print $5}')"
    if [ "$rootimagesz" -gt "$floppysz" ] ; then
        error "$rootimage is larger than root${blocks}${tsubarch}.bin ($rootimagesz > $floppysz)"
    fi
elif  [ "$arch" = m68k -a $blocks = 1440 ]; then
    info "making detached root disk root${blocks}${subarch}.bin from $rootimage"
    # we seem to have dropped 720, rootmac.bin doesn't quite fit)
    dd if=/dev/zero of=root${blocks}${subarch}.bin bs=1k count=$blocks
    dd if=$rootimage of=root${blocks}${subarch}.bin bs=1k count=$blocks conv=notrunc
    rootimagesz="$(ls -l $rootimage | awk '{print $5}')"
    floppysz="$(ls -l root${blocks}${subarch}.bin | awk '{print $5}')"
    if [ "$rootimagesz" -gt "$floppysz" ] ; then
	error "$rootimage is larger than root${blocks}${subarch}.bin ($rootimagesz > $floppysz)"
    fi
elif [ "$arch" = alpha -a "${subarch}" = "generic" ]; then
    info "padding root${blocks}${subarch}.bin from ${rootimage} to prevent IO errors"
    dd if=/dev/zero of=root${blocks}${subarch}.bin bs=1k count=$blocks
    dd if=$rootimage of=root${blocks}${subarch}.bin bs=1k count=$blocks conv=notrunc
    rootimagesz="$(ls -l $rootimage | awk '{print $5}')"
    floppysz="$(ls -l root${blocks}${subarch}.bin | awk '{print $5}')"
    if [ "$rootimagesz" -gt "$floppysz" ] ; then
        error "$rootimage is larger than root${blocks}${subarch}.bin ($rootimagesz > $floppysz)"
    fi
fi

exit 0
