/* (C) 2001 Nemosoft Unv. (nemosoft@smcc.demon.nl) */
/**
  \brief Graphical representation of values with a solid or segmented bar

  This is a nice widget that displays a range of values as a LED bar, just
  like your stereo installation or mixer panel. It is therefor quite suited
  as a level meter for an audio application, but it's certainly not limited
  to that.

  Features:
  - solid or block bar
  - different colors for each block/line segment
  - can be orientated to each of 4 directions
  - peak hold display

  The widget has two mode of operandi: solid or block segment; the latter
  looks more like a real LED bar that's on most audio devices. The solid bar
  mode represents the value more accurately. For the rest, both modes work
  similar.

  The bar is divided into a user-defined number of segments; in block mode,
  this also defines the number of blocks. Each segment/block can be assigned
  a unique color. Each segment has the same width, this is calculated when
  the number of segments is set.

  CLedBar has 4 directions in which it can 'grow': North, East, South, West.
  Finally, the CLedBar features a peak-hold function: with this on the
  highest value remains visible in the bar; after a user-defined time
  the peak hold is automatically cleared.

  At creation time you set the minimum and maximum value for the range; any
  values outside this range are clipped. The barlength is linear to the
  value you set; if you want a logarithmic display you should calculate and
  scale the value before you set it in this class.
*/

#include <assert.h>

#include <qapplication.h>
#include <qpainter.h>
#include <qpixmap.h>

#include "LedBar.h"

/**
  \brief Constructor
  \param orientation One of \ref Orientation; determins direction in which the bar grows
  \param min_range The minimum value
  \param max_range The maximum value
  \param solid_bar If TRUE, the bar will be painted solid
  \param parent The widget's parent object
  \param name The widget's name
*/

CLedBar::CLedBar(Orientation orientation, int min_range, int max_range, BarMode bar_mode, QWidget *parent, const char *name)
   : QWidget(parent, name)
{
   assert(min_range < max_range);
   BarOrientation = orientation;
   Minimum = min_range;
   Maximum = max_range;
   Range = Maximum - Minimum;
   m_BarMode = bar_mode;

   Value = 0;
   Segments = 1;
   Margin = 0;
   ColorOfSegment[0] = blue;
   setBackgroundMode(NoBackground); // flicker-free drawing; we paint everything ourselves
   CalcBoundaries();
}



// private

/* Guard in a multhithreaded environment; otherwise, this isn't a problem */
void CLedBar::BusyLock()
{
#ifdef QT_THREAD_SUPPORT
   Busy.lock();
#endif
}

void CLedBar::BusyUnlock()
{
#ifdef QT_THREAD_SUPPORT
   Busy.unlock();
#endif
}


/**
  Should be called with BusyLock()
*/
void CLedBar::CalcBoundaries()
{
   int i, half;

   if (BarOrientation == East || BarOrientation == West) {
     BarLength = width();  // horizontal
     BarWidth = height() - 2 * Margin;
   }
   else {
     BarLength = height(); // vertical
     BarWidth = width() - 2 * Margin;
   }

   /* Recalculate boundaries of segments; we calculate one boundary extra
      for proper end value of the last segment
    */
   for (i = 0; i <= Segments; i++)
      StartOfSegment[i] = (BarLength * i) / Segments;
   /* Calculate trigger level for segments; this is only used for block mode;
      the value is placed halfway the segment.
    */
   half = Range / (Segments * 2);
   for (i = 0; i < Segments; i++)
      ValueForSegment[i] = Minimum + half + (i * Range) / Segments;
}

// protected

void CLedBar::resizeEvent(QResizeEvent *e)
{
   QWidget::resizeEvent(e);
   BusyLock();
   DrawingArea.resize(size());
   CalcBoundaries();
   BusyUnlock();
}

void CLedBar::paintEvent(QPaintEvent *)
{
   QPainter draw;
   int i;

   BusyLock();
   draw.begin(&DrawingArea); // begin painting in pixmap
   DrawingArea.fill(black);
   switch (m_BarMode)
   {
     case SolidBar:
     {
       int part, b, w;

       // How many pixels do we need to paint?
       part = ((Value - Minimum) * BarLength) / Range;
       for (i = 0; i < Segments; i++) {
          b = StartOfSegment[i];
          w = -1;
          if (part > StartOfSegment[i + 1]) // paint whole segment
            w = StartOfSegment[i + 1] - StartOfSegment[i];
          else if (part > StartOfSegment[i]) // last, partial segment
            w = part - StartOfSegment[i];
          if (w > 0) {
            switch(BarOrientation) {
              case East:  draw.fillRect(b, Margin, w, BarWidth, ColorOfSegment[i]); break;
              case West:  draw.fillRect(BarLength - b - w, Margin, w, BarWidth, ColorOfSegment[i]); break;
              case North: draw.fillRect(Margin, BarLength - b - w, BarWidth, w, ColorOfSegment[i]); break;
              case South: draw.fillRect(Margin, b, BarWidth, w, ColorOfSegment[i]); break;
            }
          }
       } // ..for
     }
     break;

     case SegmentBar:
     {
       int NewSeg, SegBeg, SegLen, val;

       NewSeg = -1;
       val = Value; // saves a mutex lock in setValue()
       for (i = 0; i < Segments; i++) {
          if (val > ValueForSegment[i])
            NewSeg = i + 1; // would be painted
       }
       for (i = 0; i < NewSeg; i++)
       {
          SegLen = StartOfSegment[i + 1] - StartOfSegment[i] - 2 * Margin;
          switch(BarOrientation)
          {
            case East:
              SegBeg = StartOfSegment[i] + Margin;
              draw.fillRect(SegBeg, Margin, SegLen, BarWidth, ColorOfSegment[i]);
              break;
            case West:
              SegBeg = BarLength - StartOfSegment[i + 1] + Margin;
              draw.fillRect(SegBeg, Margin, SegLen, BarWidth, ColorOfSegment[i]);
              break;
            case North:
              SegBeg = BarLength - StartOfSegment[i + 1] + Margin;
              draw.fillRect(Margin, SegBeg, BarWidth, SegLen, ColorOfSegment[i]);
              break;
            case South:
              SegBeg = StartOfSegment[i] + Margin;
              draw.fillRect(Margin, SegBeg, BarWidth, SegLen, ColorOfSegment[i]);
              break;
          }  // ..switch
       } // ..for
     }
     break;
   }
   draw.end();
   BusyUnlock();

   // Now copy image to screen, eliminating flicker
   draw.begin(this);
   draw.drawPixmap(0, 0, DrawingArea); // copy to screen
   draw.end();
}

// public


void CLedBar::SetBarMode(BarMode bar_mode)
{
  BusyLock();
  m_BarMode = bar_mode;
  BusyUnlock();
}


/**
  \brief Set the number of semgents and their respective colors
  \param segment Number of segments; must be between 1 and 20
  \param colors An array with the colors for each segment

  With this function you set the number of segments/block for this
  ledbar, and assign colors to each segment.

  By default, the bar has 1 segment, coloured blue.
*/
void CLedBar::SetSegments(int segments, const QColor *colors)
{
   if (segments < 1)
     return;
   if (segments > 20)
     segments = 20;
   if (colors == NULL)
     return;

   BusyLock();
   Segments = segments;
   CalcBoundaries();
   // Copy colours
   for (int i = 0; i < Segments; i++)
      ColorOfSegment[i] = colors[i];
   BusyUnlock();
}

void CLedBar::SetMargin(int width)
{
   if (width < 0)
     return;
   BusyLock();
   Margin = width;
   CalcBoundaries();
   BusyUnlock();
}

// public signals

void CLedBar::setValue(int value)
{
   if (value < Minimum)
     Value = Minimum;
   else if (value > Maximum)
     Value = Maximum;
   else
     Value = value;
   update();
}
