/*
 * KON2 - Kanji ON Console -
 * Copyright (C) 1992-1996 Takashi MANABE (manabe@papilio.tutics.tut.ac.jp)
 *
 * CCE - Console Chinese Environment -
 * Copyright (C) 1998-1999 Rui He (herui@cs.duke.edu)
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY TAKASHI MANABE ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE TERRENCE R. LAMBERT BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#if defined(linux)
#include	<termio.h>
#elif defined(__FreeBSD__)
#include	<termios.h>
#include	<machine/console.h>
#define termio termios
#endif
#include	<signal.h>
#include	<errno.h>
#if defined(linux)
#include	<sys/vt.h>
#include	<sys/kd.h>
#endif
#include	<sys/time.h>
#include	<sys/ioctl.h>
#include	<sys/file.h>
#include	<sys/wait.h>

#include	<getcap.h>
#include	<defs.h>
#include	<errors.h>
#include	<vc.h>
#include	<vt.h>
#include	<child.h>
#include	<mouse.h>
#include 	<key.h>
#include        <pwd.h>
#include        <utmp.h>
#include        <grp.h>
#include        <sys/stat.h>

#ifdef __FreeBSD__
#define TCSETA     TIOCSETA
#define TCGETA     TIOCGETA
#define SIGCLD     SIGCHLD
#define XCASE 0
#endif

#define NR_TERMS    10
int numTerms = 0;
int termMask[NR_TERMS];

ConInfo *curCon = NULL;

static struct termio	oldTio;

int VtNum;  /* current virtual terminal */ 

struct initInfo {
    bool display;                        /* display initialized */
   // bool utmp;                           /* utmp set */
    bool termio;                         /* termio saved */
};

static struct initInfo init;

static void TermClose(ConInfo *con);
static void SetSignalHandlers(void);

/*************************************************************************
 *                      Terminal Routines                                *
 *************************************************************************/

static void CleanUp(void)
{
    if (init.display)
        ConsoleCleanup();
  
    //if (init.utmp)
    //    ResetUtmp(ptyName);

    //if (init.socket)
    //	SocketKill(sockFd);

    if (init.termio)
        ioctl(0, TCSETA, &oldTio);

    signal(SIGCHLD, SIG_DFL);  
    signal(SIGHUP, SIG_DFL);
    signal(SIGTERM, SIG_DFL);
    signal(SIGSEGV, SIG_DFL);
    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);

    ChildCleanup();
    InputCleanup();
    KeymapCleanup();
 
    FontDetach(TRUE);  // remove loaded font

    fprintf(stderr,"CCE> Finished without core dump :) \r\n\r\n");
}

static void ExitTerm(int signum)
{
    /*
    FILE *fp;

    fp = fopen("/tmp/cce-tmp","wb");
    fprintf(fp,"Exit Term Entered!, signal no = %d %s\n", signum,
       sys_siglist[signum]);

    */

    fatal("%s\n",sys_siglist[signum]);

   //fclose(fp);
}

static fd_set origReadFds;
static int maxFds = 0;

/* MouseSetRfd/MouseResetRfd during EnterVC/LeaveVC */
void MouseSetRfd(int mfd)
{
    if (mfd > 0) FD_SET(mfd, &origReadFds);
    
    if (mfd >= maxFds)
        maxFds = mfd + 1;
}

void MouseResetRfd(int mfd)
{
    if (mfd > 0) FD_CLR(mfd, &origReadFds);
}

static void	ConsoleHandler(void)
{
    static u_char	buff[BUFSIZ + 1];
    fd_set readFds;
    int	j,k, nRead;
    struct timeval tv;
    ConInfo *con;

    if (mInfo.has_mouse && mouseFd > 0)
	MouseSetRfd(mouseFd);
    else
	MouseResetRfd(mouseFd);

    /* Note: we use timeout on select call even if cursor blink is off
       because of screen saver and mouse cursor timeout. */

    while (curCon && numTerms > 0)   // forever looping
    {
	int	v;
	
	do   // wait until something happens
 	{
	    /* Idle loop. */
	    PollCursor(FALSE);
	    readFds = origReadFds;
	    tv.tv_sec = 0;
	    tv.tv_usec = 100000; /* 0.1 sec */
	    v = select(maxFds, &readFds, NULL, NULL, &tv);
	} 
	while (v == 0 || (v < 0 && (errno == EINTR || mouseFd < 0)));

	if (v < 0) PerrorExit("select");

        /* input from different terminal's masterPty, output via VtWrite */
        for(k = numTerms, con = curCon; k > 0; k--)
        {
	    if (FD_ISSET(con->masterPty, &readFds)) 
            {
	        if ((nRead = read(con->masterPty, buff, BUFSIZ)) > 0)
                {
		    if (text_mode) 
		        write(1, buff, nRead);
                    else 
                    {
		        VtWrite(con, buff, nRead);
		        if (con == curCon) TextRefresh(curCon);
		    }
	        }
            }
           
            con = con->nextCon;
            if (con->prevCon->terminate)
            {
                 FD_CLR(con->prevCon->masterPty, &origReadFds);
                 TermClose(con->prevCon);
            }
	}

        /* some input from stdin, process the normail key input */
	if (curCon && FD_ISSET(0, &readFds)) 
        {
	    nRead = read(0, buff, BUFSIZ);
       
            for(j=0; j <nRead; j++)
  		ProcessNormalModeKey(curCon->masterPty,buff[j]);

            /* Call HZFilter here, input keys -> output Chinese Chars,
              write to masterPty */

	    PollCursor(TRUE);
	}

 /*   // no socket input
	if (FD_ISSET(sockFd, &readFds)) 
             SocketInterface(sockFd);
 */
        /* some input from mouse */
	if (mInfo.has_mouse && curCon && mouseFd > 0) 
        {
	    if (FD_ISSET(mouseFd, &readFds)) 
            {
		nRead = read(mouseFd, buff, BUFSIZ);
		if (nRead > 0) MouseGetPacket(buff, nRead);
		PollCursor(TRUE);
	    }
	}
    } 
}


/* Do initialization before reading config file */
void	TermInit()
{
    init.display = init.termio = FALSE;

    text_mode = TRUE;

    /* Initialize subsystems. */
    
    CapInit();
    ChildInit();
    MouseInit();

    InputInit();
    FontInit();
    KeymapInit();

    VtInit();
    ConsoleInit();
}

void ChangeTerm(int direction)
{
    if (direction)
        curCon = curCon->nextCon;
    else
        curCon = curCon->prevCon;
    TextRepaintAll(curCon);
}

static void TermClose(ConInfo *con)
{
    sigset_t mask;

    if (con == NULL) return;

    sigemptyset(&mask);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_BLOCK, &mask, NULL);

    if (con != con->nextCon) /* is all console is close? */
    {
        con->prevCon->nextCon = con->nextCon;
        con->nextCon->prevCon = con->prevCon;
    }
    else curCon = NULL;

    numTerms--;
    termMask[con->window] = 0;

    sigprocmask(SIG_UNBLOCK, &mask, NULL);
    if (curCon == con)
       ChangeTerm(1);
    close(con->masterPty);
    VtCleanup(con);
  //  SafeFree(&con);
}

void TermOpen(void)
{
    ConInfo *con;
    sigset_t mask;
    int i;

    if (numTerms < NR_TERMS)
    {
         con = malloc(sizeof(ConInfo));
         if (con == NULL)
	 {
             error("Open New TTY failed\n");
             return;
         }     
         
         if (TermFork(con))  // parent process
         {
             sigemptyset(&mask);
             sigaddset(&mask, SIGCHLD);
             sigprocmask(SIG_BLOCK, &mask, NULL);

             if (curCon)
             {
                  // insert con into the double linked list
                  con->prevCon = curCon;
                  con->nextCon = curCon->nextCon;
                  curCon->nextCon->prevCon = con;
                  curCon->nextCon = con;
             }
             else curCon = con->nextCon = con->prevCon = con;
             numTerms++;
             
             for(i = 0; i < NR_TERMS; i++)
               if (termMask[i] == 0)
               {
                  con->window = i;
                  termMask[i] = 1;
                  break;
               }  // find an unused window
            
             sigprocmask(SIG_UNBLOCK,&mask, NULL);

             curCon = con;  // activate current
             VtStart(con);
             FD_SET(con->masterPty,&origReadFds);
             if (con->masterPty >= maxFds)
		maxFds = con->masterPty+1;
         }
    }
}

int TermFork(ConInfo *con)
{
    char  ls, ln;
#if defined(__FreeBSD__)
    int devtty;
#endif

   // 1. Looking for unused PTY/TTY Master/Slave

    /* Open PTY(master) from [pqrs][0-F], in fact p-z is ok? */
    /* MasterPty:  pty[p-z][0-f] pty[a-e][0-f]  16*16=256
       SlaveTty:   tty[p-z][0-f] tty[a-e][0-f]  16*16=256 */

    for (ls = 'p'; ls <= 's'; ls ++)
    {
        for (ln = 0; ln <= 0xF; ln ++)
        {
            sprintf(con->ptyName, "/dev/pty%1c%1x", ls, ln);
            if ((con->masterPty = open(con->ptyName, O_RDWR)) >= 0) break;
        }
        if (con->masterPty >= 0) break;
    }

    if (con->masterPty < 0)
    {
        message("can not get master pty\r\n");
        PerrorExit(con->ptyName);
    }
    con->ptyName[5] = 't';   /* slave tty */

        /* fork handler */
    if ((con->childPid = fork()) < 0)
        PerrorExit("fork");
    else if (con->childPid == 0)  // in child process
    {
        int     efd, slavePty;
        FILE    *errfp;

        efd = dup(2);
        errfp = fdopen(efd, "w");

        /* I'm child, Make me process leader */
        setsid();

#if defined(__FreeBSD__)
        if ((devtty = open("/dev/tty",O_RDWR|O_NDELAY)) >= 0)
        {
            ioctl(devtty, TIOCNOTTY, (char *)0);
            /* void TTY association */
            close(devtty);
        }
#endif

        /* Open TTY(slave) */
        if ((slavePty = open(con->ptyName, O_RDWR)) < 0)
            PerrorExit(con->ptyName);

        close(con->masterPty);

        /* Set old tio to TTY */
        ioctl(slavePty, TCSETA, &oldTio);
#ifdef __FreeBSD__
        ioctl(slavePty, TIOCSCTTY, (char *)0);
        /* /usr/include/sys/ttycom.h   becoming controlling tty */
#endif

        /* Set std??? to pty, dup2(oldfd,newfd) */
        dup2(slavePty, 0);
        dup2(slavePty, 1);
        dup2(slavePty, 2);

        /* added */
        close(slavePty);

        ChildStart(errfp);  // execute the shell
    }
  
    return 1; // parent process
}

/* set the console to raw mode */

void SetConsoleRawMode(void)
{
    struct termio       newTio;

    /* Get old tio of 0 */
    ioctl(0, TCGETA, &oldTio);
    init.termio = TRUE;

     /* Set new tio of 0 */
    newTio = oldTio;

    newTio.c_lflag &= ~(ECHO|ISIG|ICANON|XCASE);
    newTio.c_iflag &= 0;
    newTio.c_cflag |= CS8;
    newTio.c_oflag &= ~(OPOST);
    newTio.c_cc[VMIN] = 1;
    newTio.c_cc[VTIME] = 0;

#if defined(__FreeBSD__)
    newTio.c_cc[VDISCARD] = _POSIX_VDISABLE;
    newTio.c_cc[VLNEXT] = _POSIX_VDISABLE;
    newTio.c_cc[VSTART] = _POSIX_VDISABLE;
    newTio.c_cc[VSTOP] = _POSIX_VDISABLE;
    newTio.c_cc[VINTR] = _POSIX_VDISABLE;
    newTio.c_cc[VSUSP] = _POSIX_VDISABLE;
    newTio.c_cc[VDSUSP] = _POSIX_VDISABLE;
    newTio.c_cc[VQUIT] = _POSIX_VDISABLE;
#endif
#ifdef linux
    newTio.c_line = 0;
#endif

    ioctl(0, TCSETA, &newTio);

    /*   // error ???
    if (tcsetattr(0, TCSAFLUSH, p) < 0)
    {
        fatal("Set Console Termios Failed\n"); 
    }
    */
}

void	TermStart(void)
{
    if (mInfo.has_mouse) 
    {
	mouseFd = MouseStart();
	if (mouseFd == -1)  fprintf(stderr,"Mouse Init failed!\n");
    }

  //  chown("/dev/tty0", getuid(), getgid());
    /* I don't know why we need this statement. Just comment it */

//    SetUtmp(ptyName);  // now slave tty name
 //   init.utmp = TRUE;

    atexit(CleanUp);  /* program exit cleanup */

    SetConsoleRawMode();
    SetSignalHandlers();
   
    FD_ZERO(&origReadFds);
    FD_SET(0, &origReadFds);  // from console

    TermOpen();

    FontAttach();
    ConsoleStart();  // switch to graph mode 
    init.display = TRUE;

    ConsoleHandler();   // main loop event
}

// mark it as terminate status
void SigChildHandler(int signo)
{
  int status;
  ConInfo *con;

  con = curCon;
  do
  {
     if (waitpid(con->childPid,&status, WNOHANG))
       con->terminate = 1;
     con = con->nextCon;
  } while (con != curCon );

  // Re-set SIGCLD Handler - fixed by Holly Lee. Sep 20, 1999 in Shanghai
  signal(SIGCHLD, SigChildHandler);
}

static void SetSignalHandlers(void)
{
/* catch serious signal and do clean up before exit */
/*
  act.sa_handler = ExitTerm;
  act.sa_flags = 0;
  sigemptyset(&act.sa_mask);
  sigaction(SIGHUP, &act, NULL);
  sigaction(SIGINT, &act, NULL);
  sigaction(SIGQUIT, &act, NULL);
  sigaction(SIGILL, &act, NULL);
  sigaction(SIGABRT, &act, NULL);
  sigaction(SIGIOT, &act, NULL);
  sigaction(SIGBUS, &act, NULL);
  sigaction(SIGFPE, &act, NULL);
  sigaction(SIGTERM, &act, NULL);
*/
     signal(SIGCHLD, SigChildHandler); 
      /*SIGCHLD Child status has changed (POSIX).*/      

     signal(SIGHUP, ExitTerm);
     signal(SIGTERM, ExitTerm);
     signal(SIGSEGV, ExitTerm);

     /*  Do we need to handle these signal? 
     signal(SIGINT, ExitTerm);
     signal(SIGQUIT, ExitTerm);
     signal(SIGILL, ExitTerm);
     signal(SIGABRT, ExitTerm);
     signal(SIGIOT, ExitTerm);
     signal(SIGBUS, ExitTerm);
     */
}

/*********************************************************************
 *                        utmp set/reset                             *
 *            Not  necessary to process the utmp record              *
 *********************************************************************/

static int ttyGid;

void	SetUtmp(char *tty)
{
#ifdef linux
	struct utmp	utmp;
	struct passwd	*pw;
	struct group	*ttygrp;
	char	*tn;

	pw = getpwuid(getuid());
	tn = rindex(tty, '/') + 1;
	memset((char *)&utmp, 0, sizeof(utmp));
	strncpy(utmp.ut_id, tn + 3, sizeof(utmp.ut_id));
	utmp.ut_type = DEAD_PROCESS;
	setutent();
	getutid(&utmp);
	utmp.ut_type = USER_PROCESS;
	utmp.ut_pid = getpid();
	strncpy(utmp.ut_line, tn, sizeof(utmp.ut_line));
	strncpy(utmp.ut_user, pw->pw_name, sizeof(utmp.ut_user));
	time(&(utmp.ut_time));
	pututline(&utmp);
	endutent();
	if ((ttygrp = getgrnam("tty")) != NULL)
		ttyGid = ttygrp->gr_gid;
	else
		ttyGid = -1;
	chmod(tty, 0622);
	chown(tty, getuid(), ttyGid);
#endif
}

void	ResetUtmp(char *tty)
{
#ifdef linux
	struct utmp	utmp, *utp;
	char	*tn;

	tn = rindex(tty, '/') + 4;
	memset((char *)&utmp, 0, sizeof(utmp));
	strncpy(utmp.ut_id, tn, sizeof(utmp.ut_id));
	utmp.ut_type = USER_PROCESS;
	setutent();
	utp = getutid(&utmp);
	utp->ut_type = DEAD_PROCESS;
	memset(utp->ut_user, 0, sizeof(utmp.ut_user));
	utp->ut_type = DEAD_PROCESS;
	time(&(utp->ut_time));
	pututline(utp);
	endutent();
	chmod(tty, 0600);
	chown(tty, 0, ttyGid);
#endif
}
