/*
	$Id: displaycard.h,v 1.1.1.1 2000/04/09 12:17:59 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

//! component="Display 2D"

#ifndef header_displaycard
#define header_displaycard

class CL_VidMode;
class CL_Palette;
#include "cliprect.h"

class CL_InputSourceProvider;
class CL_DisplayCard
//: The display card class.
// <p>This class represents a display card in ClanLib. It is mainly used to
// set the videomode, flip the display and do simple graphics operations
// such as filled rectangles or drawing lines.</p>
//
// <p>If you're always only using one display card at the time, you may want
// to consider using the static functions in CL_Display instead. They do
// exactly the same, but operates on a selected display card instead
// (default the primary card). This saves you from passing around a pointer
// to the displaycard.</p>
{
public:
	virtual const std::list<CL_VidMode*> &get_videomodes()=0;
//: Get list of videomodes available on this card.
//!retval: Returns the list of videomodes.

	virtual std::string get_name()=0;
//: Get name of the card.
//!retval: Returns the card name.
	
	virtual int get_total_memory()=0;
//: Get videomemory available.
//!retval: Videomemory available, or -1 if unknown.
	
	virtual int get_card_no()=0;
//: Get card number.
//!retval: Card number.

	virtual ~CL_DisplayCard() { ; }

	virtual void flip_display(bool sync=false)=0;
//: Flips the front and backbuffer. Everything is normally drawn to the 
//: backbuffer, and flip_display() needs to be called before it can be seen 
//: onto the screen.
//!param: sync - VSync on/off
	
	virtual void clear_display(float red=0, float green=0, float blue=0, float alpha=1)=0;
//: Clears backbuffer with the specified color.
//!param: red - red component of the color.
//!param: green - green component of the color.
//!param: blue - blue component of the color.
//!param: alpha - alpha (transparency) component of the color.

	virtual void set_palette(CL_Palette *palette)=0;
//: Set system palette on this card.
//: <br>
//: If the display card is in a non-palettelized mode, this will be the palette 
//: used when surfaceproviders doesn't specify a palette themself.
//!param: palette - Palette to use as new system palette.
	
	virtual CL_Palette *get_palette()=0;
//: Return the current system palette on this card.
//!retval: The system palette.

	virtual void set_videomode(CL_VidMode *mode);
//: Change the display card's video mode.
//!param: mode - videomode to be changed to.

	virtual void set_videomode(
		int width,
		int height,
		int bpp,
		bool fullscreen = true,
		bool allow_resize = false)=0;
//: Change the display card's video mode.
//!param: width - width in pixels of the new video mode.
//!param: height - height in pixels of the new video mode.
//!param: bpp - Bits per pixel. The depth of the new video mode. (8, 16, 24, 32)

	virtual bool is_initialized()=0;
//: Returns true if the display card is in a video mode.
//!retval: true if initialized.

	virtual CL_Target* get_target()=0;
//: Return either NULL or the Target that is blitted to (aka Framebuffer)
//:!retval: NULL if Framebuffer access is not possible, CL_Target if it is

	virtual int get_width()=0;
//: Returns the width of the current video mode.
//!retval: Width of video mode.

	virtual int get_height()=0;
//: Returns the height of the current video mode.
//!retval: Height of video mode.

	virtual int get_bpp()=0;
//: Returns the depth of the current video mode.
//!retval: Depth of video mode.

	virtual void push_clip_rect()=0;
//: Pushes the current clipping rectangle onto the cliprect stack.

	virtual void push_clip_rect(const CL_ClipRect &rect)=0;
//: Pushes the current clipping rectangle onto the cliprect stack. It then clips 
//: the passed rectangle 'rect' with the current one, and uses the result as the 
//: new clipping rect.
//!param: rect - The new clipping rectangle to be with the old one and then used.
 
	virtual CL_ClipRect get_clip_rect()=0;
//: Returns the current clipping rectangle.
//!retval: The current clipping rectangle.
 
	virtual void set_clip_rect(const CL_ClipRect &rect)=0;
//: Sets the current clipping rectangle. This is an absolute set, so it doesn't
//: get clipped with the previous one.

	virtual void pop_clip_rect()=0;
//: Pop the clipping rectangle last pushed onto the stack.

	virtual void fill_rect(int x1, int y1, int x2, int y2, float r, float g, float b, float a)=0;
//: Draw a filled rectangle from ('x1', 'y1') to ('x2', 'y2') using the color
//: ('r', 'g', 'b', 'a').
//!param: x1 - Leftmost x-coordinate.
//!param: y1 - Upper y-coordinate.
//!param: x2 - Rightmost x-coordinate.
//!param: y2 - Lower y-coordinate.
//!param: r - Red component of the filled color.
//!param: g - Green component of the filled color.
//!param: b - Blue component of the filled color.
//!param: a - Alpha component of the filled color.

	virtual void draw_line(int x1, int y1, int x2, int y2, float r, float g, float b, float a)=0;
//: Draw a line from ('x1', 'y1') to ('x2', 'y2') using the color
//: ('r', 'g', 'b', 'a').
//!param: x1 - Leftmost x-coordinate. //FIXME
//!param: y1 - Upper y-coordinate.
//!param: x2 - Rightmost x-coordinate.
//!param: y2 - Lower y-coordinate.
//!param: r - Red component of the filled color.
//!param: g - Green component of the filled color.
//!param: b - Blue component of the filled color.
//!param: a - Alpha component of the filled color.
	
	virtual void sync_buffers()=0;
//: Copies the contents of the frontbuffer to all other buffers (usually just
//: the backbuffer). This ensures that all buffers contain the same image.
};

#endif
