
#include "Core/precomp.h"

#ifdef USE_NETWORK

#include <API/Core/System/error.h>
#include <API/Core/System/mutex.h>
#include <API/Core/Network/netcomputer.h>
#include <API/Core/Network/netgroup.h>
#include <API/Core/Network/netmessage.h>
#include <API/Core/System/thread.h>
#include <API/Core/System/cl_assert.h>
#include <API/Core/System/system.h>
#include <Core/Network/Generic/network_delivery_impl.h>
#include <Core/Network/Generic/netgame_client.h>
#include <Core/Network/Generic/netgame_generic.h>
#include <Core/IOData/Generic/outputsource_memory_generic.h>
#include <Core/IOData/Generic/inputsource_memory_generic.h>

CL_NetGame_Client::CL_NetGame_Client(
	int ip_addr,
	int port,
	std::string &game_id,
	CL_Network_Generic *network)
: CL_NetGame_Generic(network)
{
	mutex = CL_Mutex::create();

	tcp_connection = network->provider->create_tcp_connection(
		ip_addr,
		port);

	if (tcp_connection == NULL)
	{
		throw CL_Error("Could not connect to host.");
	}

	udp_connection = NULL; // still no support for udp in client.

	// client code currently only knows about the server.
	all.computers.push_back(&server);

	// do "hello to you too" handshake.
	CL_ConnectionPacket p = tcp_connection->receive();
	while (p.size == 0) // wait until we get a hello packet.
	{
		network->provider->wait_for_connection_data(mutex);
		p = tcp_connection->receive();
	}
	
	CL_InputSource_MemoryGeneric input(p.data, p.size, true);
	if (input.read_int32() != Packet_Hello)
	{
		throw CL_Error("Protocol error. Didn't get a Hello.");
	}
	our_id = input.read_int32();

	int size = input.read_int32();
	if (size > 1000)
		throw CL_Error("Protocol error. Game ID size above 1000.");

	char *str = new char[size+1];
	str[size] = 0; // null-terminate.
	input.read(str, size);
	
	if (game_id != str) // not same game_id. probably an other clanlib game.
	{
		delete[] str;
		throw CL_Error("Wrong netgame id.");
	}

	delete[] str;
	
	// say "HelloToYouToo" to officially join the netgame:
	CL_OutputSource_MemoryGeneric output;
	output.write_int32(Packet_Hello_ToYouToo);
	tcp_connection->send(
		CL_ConnectionPacket(
			output.get_data(),
			output.size()));

	exit_thread = false;
	thread = CL_Thread::create(this);
	thread->start();
}

CL_NetGame_Client::~CL_NetGame_Client()
{
	exit_thread = true;
	thread->wait();
	delete thread;

	delete tcp_connection;
	delete udp_connection;

	// clean up connections to clients:
	for (
		std::list<CL_NetChannelQueue_Client*>::iterator it = netchannels.begin();
		it != netchannels.end();
		it++)
	{
		delete *it;
	}
/*
	should be enabled when the client knows about other clients...

	while (leave_queue.empty() == false)
	{
		delete leave_queue.pop();
	}
*/

	delete mutex;
}

const CL_NetComputer *CL_NetGame_Client::get_server() const
{
	return &server;
}

const CL_NetGroup *CL_NetGame_Client::get_all() const
{
	return &all;
}

bool CL_NetGame_Client::peek(int channel) const
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *queue = find_queue(channel);
	if (queue != NULL)
	{
		return !queue->empty();
	}

	return false;
}

CL_NetMessage CL_NetGame_Client::receive(int channel, int timeout)
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *queue = find_queue(channel);
	if (queue != NULL)
	{
		while (queue->empty() && timeout > 0) 
		{
			int s = timeout >= 20 ? 20 : timeout;
			CL_System::sleep(s);
			timeout -= s;
		}
		if (queue->empty()) throw CL_Error("No message to receive!");
		CL_NetMessage msg = queue->front(); queue->pop();
		return msg;
	}
	
	throw CL_Error("No message to receive!");
	CL_NetMessage netmsg;
	return netmsg;
}

void CL_NetGame_Client::send(
	const int dest_channel,
	const CL_NetComputer *dest,
	const CL_NetMessage &message,
	bool reliable /*= true*/)
{
	CL_NetGroup group;
	group.computers.push_back((CL_NetComputer *) dest);

	send(
		dest_channel,
		&group,
		message,
		reliable);
}

void CL_NetGame_Client::send(
	const int dest_channel,
	const CL_NetGroup *dest,
	const CL_NetMessage &message,
	bool reliable /*= true*/)
{
	CL_MutexSection mutex_section(mutex);

	static bool warning = true;
	if (warning && reliable == false)
	{
		cl_info(info_network, "cannot send data unreliable (udp): not implemented yet!");
		warning = false;
	}
	
	CL_NetChannelQueue_Client *queue = find_queue(dest_channel);

	if (queue == NULL)
		throw CL_Error("No write access to netchannel.");

	if ((queue->access & ACCESS_CHANNEL_WRITE) != ACCESS_CHANNEL_WRITE)
		throw CL_Error("No write access to netchannel.");

	// permissions ok, send message:
	CL_OutputSource_MemoryGeneric output;
	output.write_int32(Packet_NetChannel_Message_ToServer);
	output.write_int32(dest_channel);

	// todo: write the computer id's on destination computers here.

	output.write_int32(message.data.size());
	output.write(message.data.data(), message.data.size());

	CL_ConnectionPacket packet;
	packet.size = output.size();
	packet.data = output.get_data();

	tcp_connection->send(packet);
}

CL_NetComputer *CL_NetGame_Client::receive_computer_leave()
{
	return NULL;
}

const CL_NetComputer *CL_NetGame_Client::receive_computer_join()
{
	return NULL;
}

const CL_NetComputer *CL_NetGame_Client::receive_computer_rejoin()
{
	return NULL;
}

bool CL_NetGame_Client::receive_game_closed()
{
	CL_MutexSection mutex_section(mutex);
	return tcp_connection->connection_lost();
}

int CL_NetGame_Client::access_status(int channel) const
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *queue = find_queue(channel);
	return (queue == NULL) ? 0 : queue->access;
}

bool CL_NetGame_Client::is_writable(int channel) const
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *queue = find_queue(channel);
	if (queue == NULL) return false;

	return (queue->access & ACCESS_CHANNEL_WRITE) == ACCESS_CHANNEL_WRITE;
}

bool CL_NetGame_Client::is_readable(int channel) const
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *queue = find_queue(channel);
	if (queue == NULL) return false;

	return (queue->access & ACCESS_CHANNEL_READ) == ACCESS_CHANNEL_READ;
}

// Client side only:
int CL_NetGame_Client::receive_access_changed()
{
	CL_MutexSection mutex_section(mutex);

	if (access_queue.empty()) return -1;
	int id = access_queue.front()->channel_id;
	access_queue.pop();
	return id;
}

// Server side only:
void CL_NetGame_Client::set_access(
	int /*channel*/,
	const CL_NetComputer * /*computer*/,
	int access_rights /*=ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE*/)
{
	throw CL_Error("We are not the network server. Cannot change access.");
}

void CL_NetGame_Client::set_access(
	int /*channel*/,
	const CL_NetGroup * /*group*/,
	int access_rights /*=ACCESS_CHANNEL_READ|ACCESS_CHANNEL_WRITE*/)
{
	throw CL_Error("We are not the network server. Cannot change access.");
}

void CL_NetGame_Client::keep_alive()
{
	CL_MutexSection mutex_section(mutex);
	if (exit_thread) return;

	if (tcp_connection->connection_lost())
	{
		network->provider->remove_connection(tcp_connection);
		return;
	}

	while (tcp_connection->peek())
	{
		CL_ConnectionPacket msg = tcp_connection->receive();
			
		CL_InputSource_MemoryGeneric input(msg.data, msg.size, true);
		switch (input.read_int32())
		{
		case Packet_NetChannel_AccessChange:
			{
				CL_NetChannelQueue_Client *queue = create_queue(input.read_int32());
				queue->access = input.read_int32();
			}
			break;
		
		case Packet_NetChannel_Message_ToClient:
			{
				CL_NetChannelQueue_Client *queue = create_queue(input.read_int32());
				CL_NetMessage game_msg;
				int size = input.read_int32();
				char *data = new char[size];
				game_msg.from = &server;
				input.read(data, size);
				game_msg.data.append(data, size);
				queue->push(game_msg);
			}
			break;

		default:
			cl_info(info_network, "Network Protocol error!");
			cl_assert(false);
		}
	}
}

CL_NetChannelQueue_Client *CL_NetGame_Client::find_queue(int netchannel) const
{
	CL_MutexSection mutex_section(mutex);

	// Note: This really should be a STL map, not a list. Too lazy to fix it now -- mbn.
	for (
		std::list<CL_NetChannelQueue_Client*>::const_iterator it = netchannels.begin();
		it != netchannels.end();
		it++)
	{
		if ((*it)->channel_id == netchannel) return *it;
	}

	return NULL;
}

CL_NetChannelQueue_Client *CL_NetGame_Client::create_queue(int netchannel)
{
	CL_MutexSection mutex_section(mutex);

	CL_NetChannelQueue_Client *found = find_queue(netchannel);
	if (found != NULL) return found;
	
	CL_NetChannelQueue_Client *c = new CL_NetChannelQueue_Client;
	c->channel_id = netchannel;
	c->access = 0;

	netchannels.push_back(c);

	return c;
}

void CL_NetGame_Client::run()
{
	while (exit_thread == false)
	{
		keep_alive();
		network->provider->wait_for_connection_data(mutex);
	}
}

#endif
