/*
	$Id: network_delivery_winsocket.cpp,v 1.1.1.1 2000/04/09 12:18:01 mbn Exp $

	------------------------------------------------------------------------
	ClanLib, the platform independent game SDK.

	This library is distributed under the GNU LIBRARY GENERAL PUBLIC LICENSE
	version 2. See COPYING for details.

	For a total list of contributers see CREDITS.

	------------------------------------------------------------------------
*/

#include "Core/precomp.h"

#ifdef USE_NETWORK

#include "API/Core/System/cl_assert.h"
#include <API/Core/System/mutex.h>
#include <Core/Network/Generic/network_delivery_impl.h>
#include <Core/Network/WinSockets/network_delivery_winsocket.h>

#define NETWORK_MAGIC 0x16042104

/******************************************************************************
					Win32 Socket implementation
******************************************************************************/

#define AVAIL_BUFFER_SIZE 16*1024

CL_Win32Socket::CL_Win32Socket()
{
	is_connection_lost = false;
	sock = INVALID_SOCKET;

	recv_state = expect_magic;
	cur_message = NULL;
	cur_message_size = -1;

	avail_buffers.push_back(new char[AVAIL_BUFFER_SIZE]);
	avail_buffer_pos = 0;
	avail_buffer_end = 0;
}

CL_Win32Socket::~CL_Win32Socket()
{
	if (sock != INVALID_SOCKET) closesocket(sock);

	// <tm> code:
	std::list<char *>::iterator counter=avail_buffers.begin();
	while (counter!=avail_buffers.end())
	{
		delete[] *counter;
	}
	avail_buffers.clear();
}

bool CL_Win32Socket::read_avail()
{
	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	if (retval == SOCKET_ERROR || retval == 0) 
	{
		return false;
	}

	int data_gotten = ::recv(
		sock,
		(avail_buffers.back()+avail_buffer_end),
		AVAIL_BUFFER_SIZE-avail_buffer_end,
		0);

	if (data_gotten == 0 || data_gotten == SOCKET_ERROR)
	{
		is_connection_lost = true;
		return false;
	}

	avail_buffer_end += data_gotten;

	if (avail_buffer_end == AVAIL_BUFFER_SIZE) // expand data avail buffer
	{
		avail_buffers.push_back(new char[AVAIL_BUFFER_SIZE]);
		avail_buffer_end = 0;
		read_avail(); // go get the rest!
	}

	return true;
}

bool CL_Win32Socket::require_avail(int size)
{
	if (avail_buffers.front() == avail_buffers.back())
	{
		if (avail_buffer_end-avail_buffer_pos >= size) 
		{
			return true;
		}
		return false;
	}
	
	std::list<char *>::iterator counter=avail_buffers.begin();

	int total_avail = 0;
	while (counter!=avail_buffers.end())
	{
		int available = AVAIL_BUFFER_SIZE;
		if ((*counter) == avail_buffers.front()) available -= avail_buffer_pos;
		else if ((*counter) == avail_buffers.back()) available = avail_buffer_end;

		total_avail += available;
		if (total_avail >= size) return true;
	}
	return false;
}

void CL_Win32Socket::get_avail(void *buf, int size)
{
	char *buf2 = (char *) buf;

	while (true)
	{
		int available = AVAIL_BUFFER_SIZE - avail_buffer_pos;
		if (avail_buffers.front() == avail_buffers.back())
			available = avail_buffer_end-avail_buffer_pos;

		if (available > size) available = size;

		memcpy(buf2, avail_buffers.front()+avail_buffer_pos, available);

		avail_buffer_pos += available;
		if (avail_buffer_pos == AVAIL_BUFFER_SIZE)
		{
			delete[] avail_buffers.front();
			avail_buffers.erase(avail_buffers.begin());

			avail_buffer_pos = 0;
		}

		size -= available;
		buf2 += available;
		if (size == 0) return;
	}
}

bool CL_Win32Socket::init_socket(SOCKET init_socket)
{
	if (init_socket != INVALID_SOCKET)
	{
		sock = init_socket;
		write_int(NETWORK_MAGIC);
		return true;
	}
	else
	{
		sock = socket(AF_INET, SOCK_STREAM, 0);
		return (sock != INVALID_SOCKET);
	}
}

bool CL_Win32Socket::peek()
{
	if (sock == INVALID_SOCKET) return false;

	read_avail();

	switch (recv_state)
	{
	case expect_magic:
		{
			if (!require_avail(4)) return false;
			int found_magic;
			get_avail(&found_magic, sizeof(int));
			found_magic = ntohl(found_magic);
			if (found_magic != NETWORK_MAGIC)
			{
				cl_info(4, "WRONG NETWORK MAGIC!");
				is_connection_lost=true;
			}
			else recv_state = expect_packet_size;
		}

	case expect_packet_size:
		{
			if (!require_avail(4)) return false;
			get_avail(&cur_message_size, sizeof(int));
			cur_message_size = ntohl(cur_message_size);
			recv_state = expect_packet_data;
		}

	case expect_packet_data:
		{
			if (!require_avail(cur_message_size)) return false;
			cur_message = new char[cur_message_size];
			get_avail(cur_message, cur_message_size);
			recv_state = packet_finished;
		}

	case packet_finished:
		return true;

	default:
		return false;
	};

	return false;
}

CL_ConnectionPacket CL_Win32Socket::receive()
{
	if (peek() == false) // go away!
	{
		CL_ConnectionPacket ret;
		ret.size = 0;
		ret.data = NULL;
		return ret;
	}

	CL_ConnectionPacket ret;
	ret.size = cur_message_size;
	ret.data = cur_message;

	recv_state = expect_packet_size;

	return ret;
}

void CL_Win32Socket::send(CL_ConnectionPacket message)
{
	write_int(message.size);
	write_data(message.data, message.size);
}

bool CL_Win32Socket::connection_lost()
{
	return is_connection_lost;
}

unsigned long CL_Win32Socket::read_int()
{
	unsigned char data[4];
	int size = 4;
	int read = 0;
	while (read < 4)
	{
		int val = recv(sock, (char *) data+read, 4-read, 0);
		if (val == 0 || val == SOCKET_ERROR) 
		{
			is_connection_lost = true;
			return 0;
		}

		read += val;
	}

	return ntohl(*((unsigned long *) data));
}

void CL_Win32Socket::read_data(void *dest, int size_data)
{
	int total_read = 0;
	while (total_read < size_data)
	{
		int bytes_read = recv( sock,
						((char *) dest)+total_read, 
						size_data-total_read, 
						0);
		if (bytes_read == 0 || bytes_read == SOCKET_ERROR)
		{
			is_connection_lost = true;
			return;
		}
			
		total_read += bytes_read;
	}
}

void CL_Win32Socket::write_int(unsigned long value)
{
	unsigned long write_num = htonl(value);

	write_data(&write_num, sizeof(unsigned long));
}

void CL_Win32Socket::write_data(void *data, unsigned int size)
{
	unsigned int bytes_send = 0;
	while (bytes_send < size)
	{
		int b = ::send(sock, ((char *) data)+bytes_send, size-bytes_send, 0);
		if (b == SOCKET_ERROR)
		{
			is_connection_lost = true;
			return;
		}

		bytes_send += b;
	}
}

bool CL_Win32Socket::try_connect(unsigned long remote_ip_network_format, int port)
{
	sockaddr_in sock_addr;
	sock_addr.sin_family = AF_INET;
	sock_addr.sin_addr.s_addr = remote_ip_network_format;
	sock_addr.sin_port = htons(port);
	
	int res = ::connect(sock, (sockaddr *) &sock_addr, sizeof(sockaddr_in));
	if (!res)
	{
		write_int(NETWORK_MAGIC);
		return true;
	}
	else 
	{
		std::cout << "WSA ERROR: " << strerror(WSAGetLastError()) << std::endl;
		return false;
	}
}

/******************************************************************************
					CL_Win32AcceptSocket implementation
******************************************************************************/

CL_Win32AcceptSocket::CL_Win32AcceptSocket()
{
	port = -1;
}

CL_Win32AcceptSocket::~CL_Win32AcceptSocket()
{
}

bool CL_Win32AcceptSocket::bind(int _port)
{
	if (!init_socket()) return false;

	sockaddr_in addr;
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(_port);

	// Bind the socket to the specified port
	int err = ::bind(sock, (sockaddr *) &addr, sizeof(sockaddr_in));	
	if (err == SOCKET_ERROR) return false;

	// Call getsockname to determine allocated port-number,
	// since socket might have been created with _port==0 (any)
	int len = sizeof(sockaddr_in);
	err = getsockname(sock, (sockaddr *) &addr, &len);
	if (err == SOCKET_ERROR) return false;

	port = ntohs(addr.sin_port);
	
	// Make the socket listen for incoming connection attempts
	err = listen(sock, 64);
	if (err == SOCKET_ERROR) return false;

	return true;
}

bool CL_Win32AcceptSocket::peek()
{
	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(sock+1, &rfds, NULL, NULL, &timeout);
	cl_assert(retval != SOCKET_ERROR);

	if (retval == 0) return false;
	else return true;
}

CL_Win32Socket *CL_Win32AcceptSocket::accept()
{
	if (peek())
	{
		SOCKET res_sock = ::accept(sock, NULL, NULL);
		cl_assert(res_sock != INVALID_SOCKET);

		CL_Win32Socket *ret = new CL_Win32Socket;
		ret->init_socket(res_sock);

		return ret;
	}
	else
	{
		return NULL;
	}
}

bool CL_Win32AcceptSocket::try_connect(unsigned long remote_ip_network_format, int port)
{
	cl_assert(false);

	return false;
}

CL_ConnectionPacket CL_Win32AcceptSocket::receive()
{
	cl_assert(false);
	CL_ConnectionPacket ret;
	return ret;
}

void CL_Win32AcceptSocket::send(CL_ConnectionPacket message)
{
	cl_assert(false);
}

bool CL_Win32AcceptSocket::connection_lost()
{
	cl_assert(false);
	return false;
}

/******************************************************************************
					CL_UDPConnection_Win32 implementation
******************************************************************************/

CL_UDPConnection_Win32::CL_UDPConnection_Win32()
{
	sock = INVALID_SOCKET;
	port = -1;
}

CL_UDPConnection_Win32::~CL_UDPConnection_Win32()
{
	if (sock != INVALID_SOCKET) closesocket(sock);
}

bool CL_UDPConnection_Win32::bind(unsigned int _port)
{
	sock = socket(AF_INET, SOCK_DGRAM, 0);
	if (sock == INVALID_SOCKET) return false;

	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(_port);

	int res = ::bind(sock, (sockaddr *) &addr, sizeof(addr));
	cl_assert(res != SOCKET_ERROR);
	if (res == SOCKET_ERROR) return false;

	// Call getsockname to determine allocated port-number,
	// since socket might have been created with _port==0 (any)
	int len = sizeof(sockaddr_in);
	res = getsockname(sock, (sockaddr *) &addr, &len);
	cl_assert(res != SOCKET_ERROR);
	if (res == SOCKET_ERROR) return false;

	port = ntohs(addr.sin_port);

	int enable = 1;
	res = setsockopt(sock, SOL_SOCKET, SO_BROADCAST, (const char *) &enable, sizeof(int));
	cl_assert(res != SOCKET_ERROR);
	if (res == SOCKET_ERROR) return false;

	return true;
}

bool CL_UDPConnection_Win32::peek()
{
	if (sock == INVALID_SOCKET) return false;

	fd_set rfds;

	FD_ZERO(&rfds);
	FD_SET(sock, &rfds);

	timeval timeout;
	memset(&timeout, 0, sizeof(timeval));

	int retval = select(0, &rfds, NULL, NULL, &timeout);
	if (retval == SOCKET_ERROR) return false;

	return (retval > 0);
}

CL_UDPConnectionPacket CL_UDPConnection_Win32::receive()
{
	char *receive_buf = new char[8096];

	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	int len_addr = sizeof(addr);
	int res = ::recvfrom(sock, receive_buf, 8096, 0, (sockaddr *) &addr, &len_addr);
	cl_assert(res != SOCKET_ERROR);

	CL_UDPConnectionPacket ret;
	ret.data = receive_buf;
	ret.size = res;
	ret.ip_addr = addr.sin_addr.s_addr;
	ret.port = ntohs(addr.sin_port);

	return ret;
}

void CL_UDPConnection_Win32::send(CL_UDPConnectionPacket message)
{
	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_BROADCAST /*message.ip_addr*/;
	addr.sin_port = htons(message.port);

	cl_assert(message.size < 2000);

	int res = ::sendto(sock, (char *) message.data, message.size, 0, (sockaddr *) &addr, sizeof(sockaddr_in));
	cl_assert(res != SOCKET_ERROR);
}

void CL_UDPConnection_Win32::broadcast(CL_UDPConnectionPacket message)
{
	sockaddr_in addr;
	memset(&addr, 0, sizeof(sockaddr_in));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_BROADCAST;
	addr.sin_port = htons(message.port);
	int res = ::sendto(sock, (char *) message.data, message.size, 0, (sockaddr *) &addr, sizeof(sockaddr_in));
	cl_assert(res != SOCKET_ERROR);
}

unsigned int CL_UDPConnection_Win32::get_port()
{
	return port;
}

/******************************************************************************
					Win32 PipeConnection implementation
******************************************************************************/
/*
CL_Win32PipeConnection::CL_Win32PipeConnection(HANDLE _read, HANDLE _write)
{
	read = _read;
	write = _write;
	is_connection_lost = false;
}

CL_Win32PipeConnection::~CL_Win32PipeConnection()
{
	CloseHandle(read);
	CloseHandle(write);
}

bool CL_Win32PipeConnection::peek()
{
	unsigned long data_avail;
	if (!PeekNamedPipe(read, NULL, 0, NULL, &data_avail, NULL))
	{
		is_connection_lost = true;
		return false;
	}

	return (data_avail > 0);
}

CL_ConnectionPacket CL_Win32PipeConnection::receive()
{
	unsigned long size_data = read_int();

	unsigned char *data = new unsigned char[size_data];
	read_data(data, size_data);

	CL_ConnectionPacket ret;
	ret.size = size_data;
	ret.data = data;

	return ret;
}

void CL_Win32PipeConnection::send(CL_ConnectionPacket message)
{
	write_int(message.size);
	write_data(message.data, message.size);
}

bool CL_Win32PipeConnection::connection_lost()
{
	return is_connection_lost;
}

unsigned long CL_Win32PipeConnection::read_int()
{
	unsigned long total_read = 0;
	unsigned char data[4];

	while (total_read < 4)
	{
		unsigned long bytes_read;
		int res = ReadFile(read, data+total_read, 4-total_read, &bytes_read, NULL); 
		if (res == 0)
		{
			is_connection_lost = true;
			return 0;
		}
			
		total_read += bytes_read;
	}

	return ntohl(*((unsigned long *) data));
}

void CL_Win32PipeConnection::read_data(void *dest, int size_data)
{
	int total_read = 0;
	while (total_read < size_data)
	{
		unsigned long bytes_read;
		int res = ReadFile(
						read, 
						((unsigned char *) dest)+total_read, 
						size_data-total_read, 
						&bytes_read, 
						NULL); 
		if (res == 0)
		{
			is_connection_lost = true;
			return;
		}
			
		total_read += bytes_read;
	}
}

void CL_Win32PipeConnection::write_int(unsigned long value)
{
	unsigned long write_num = htonl(value);

	write_data(&write_num, sizeof(unsigned long));
}

inline void CL_Win32PipeConnection::write_data(void *data, unsigned int size)
{
	unsigned long total_written = 0;
	while (total_written < size)
	{
		unsigned long bytes_written;
		int res = WriteFile(
					write, 
					((unsigned char *) data)+total_written, 
					size-total_written, 
					&bytes_written, 
					NULL); 

		if (res == 0)
		{
			is_connection_lost = true;
			return;
		}

		total_written += bytes_written;
	}
}
*/

#endif // USE_NETWORK


