extern "C" {
#include <complearn/complearn.h>
#include <qsearch/qsearch.h>
}

#include <iostream>
#include <GL/glut.h>
#include <GL/glu.h>
#include <math.h>
#include <QGLViewer/vec.h>
#include <QGLViewer/camera.h>
#include <ui/mwgen.h>
#include <ui/sbviewer.moc>
#include <qfiledialog.h>
#include <qlistbox.h>
#include <qlabel.h>
#include <qaction.h>
#include <ui/mwgen.h>

const double myPI = atan(1.0)*4;

const double SCENERADIUS = 120;
const double EYEDIST = SCENERADIUS*0.25;
const double LIGHTSCALE = 2000;
const double SPACESHRINK = 8e-2;

const double SPHERESHRINK = 0.4;
const double LEAFSHRINK = 0.2;
const double SPHERERADIUS = 4.0;
const int SPHERESLICES = 20;
const int SPHERESTACKS = 16;
const double SPHERESMALLFACTOR = 0.75;
const double SPRINGCYLRAD = SPHERERADIUS * 0.2;
const double CYLDETAILFACTOR = 0.25;
const double UNIVERSESPEED = 1e7;

SBViewer::SBViewer(int argc, char **argv)
{
  _win = 0;
  do_init(argc, argv);
}

void SBViewer::do_init(int argc, char **argv)
{
  if (argc > 1)
    _dist_fname = strdup(argv[1]);
  else
    _dist_fname = strdup("distmatrix.clb");
  _tree_fname = 0;
  if (argc > 2)
    _tree_fname = argv[2];
  _tf = TimeFrame(UNIVERSESPEED);
  _labels_on = true;
  _wireframe_on = false;
  _rc = complearn_environment_load_compressor_named("bzlib");
  if (_rc == 0)
    complearn_environment_load_compressor_named("zlib");
  if (_rc == 0)
    complearn_environment_load_compressor_named(complearn_environment_compressor_list()[0]);
}

#define li (0.10)
GLfloat light_diffuse[] = { li, li, li, 1.0 };
GLfloat light_specular[] = { li, li, li, 1.0 };
#undef li

#define LP(x) (x * LIGHTSCALE)
GLfloat light_position1[] = { LP(8.0), LP(10.0), LP(1.0), LP(0.0) };
GLfloat light_position2[] = { LP(10.0), LP(-10.0), LP(10.0), LP(0.0) };
GLfloat light_position3[] = { LP(-10.0), LP(1.0), LP(9.0), LP(0.0) };
GLfloat light_position4[] = { LP(10.0), LP(-10.0), LP(-10.0), LP(0.0) };
#undef LP
GLfloat model_ambient[] = { 0.03, 0.03, 0.03, 1.0 };

GLfloat wire_diff[] = { 0.5, 0.5, 0.5, 1.0 };
GLfloat wire_spec[] = { 0.00, 0.00, 0.00, 1.0 };
GLfloat wire_shine[] = { 1.0 };
GLfloat matball_diff[] = { 0.11, 0.11, 0.3, 1.0 };
GLfloat matball_spec[] = { 0.01, 0.01, 0.31, 1.0 };
GLfloat matball_shine[] = { 1.0 };

GLfloat matkern_diff[] = { 0.1, 0.02, 0.0, 1.0 };
GLfloat matkern_spec[] = { 0.09, 0.02, 0.02, 1.0 };
GLfloat matkern_shine[] = { 2.0 };

GLfloat matspring_diff[] = { 0.10, 0.10, 0, 1.0 };
GLfloat matspring_spec[] = { 0.10, 0.10, 0, 1.0 };
GLfloat matspring_shine[] = { 8.0 };

void SBViewer::make_lights(void)
{
  glLightfv(GL_LIGHT0, GL_POSITION, light_position1);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT0, GL_SPECULAR, light_specular);

  glLightfv(GL_LIGHT1, GL_POSITION, light_position2);
  glLightfv(GL_LIGHT1, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT1, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT2, GL_POSITION, light_position3);
  glLightfv(GL_LIGHT2, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT2, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT3, GL_POSITION, light_position4);
  glLightfv(GL_LIGHT3, GL_DIFFUSE, light_diffuse);
  glLightfv(GL_LIGHT3, GL_SPECULAR, light_specular);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, model_ambient);

  glEnable(GL_LIGHT0);
  glEnable(GL_LIGHT1);
  glEnable(GL_LIGHT2);
  glEnable(GL_LIGHT3);

}

void SBViewer::set_light_mode(void)
{
    if (_wireframe_on) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    setForegroundColor(QColor(255,255,255));
    setBackgroundColor(QColor(0,0,0));
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_LIGHTING);
    glDisable( GL_CULL_FACE );
    glEnable(GL_COLOR_MATERIAL);
  }
  else {
    glPolygonMode(GL_FRONT, GL_FILL);
    glShadeModel( GL_SMOOTH );
    glEnable( GL_CULL_FACE );
    glDepthFunc(GL_LEQUAL);
    glCullFace( GL_BACK );
    glFrontFace( GL_CCW );
    setBackgroundColor(QColor(0,0,0));
    setForegroundColor(QColor(255,255,255));
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_COLOR_MATERIAL);
    glEnable(GL_LIGHTING);
  }
  _light_mode_already_set = true;
}

void SBViewer::draw()
{
  if (!_made_quadrics)
    make_quadrics();
  if (!_light_mode_already_set)
    set_light_mode();

  _tf.set_time();
  if (_sbs == 0)
    return;

  gsl_matrix *smat = _smtw->get_springy_matrix();
    if (smat) {
//      printf("Adjusting springs...\n");
      _sbs->adjust_to_springy_matrix(smat);
      gsl_matrix_free(smat);
    } else {
//      printf("no springs...\n");
    }
//    _sbs->dosimstep();
    if (_wireframe_on) {
    glDisable(GL_LIGHTING);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, wire_diff);
    glMaterialfv(GL_FRONT, GL_SPECULAR, wire_spec);
    glMaterialfv(GL_FRONT, GL_SHININESS, wire_shine);
  }
  if (!_wireframe_on) {
    glEnable(GL_LIGHTING);
    glMaterialfv(GL_FRONT, GL_DIFFUSE, matspring_diff);
    glMaterialfv(GL_FRONT, GL_SPECULAR, matspring_spec);
    glMaterialfv(GL_FRONT, GL_SHININESS, matspring_shine);
  }

  int i, j;
  for (i = 0; i < _sbs->get_ball_count(); i += 1)
    for (j = i + 1; j < _sbs->get_ball_count(); j += 1) {
      if (_sbs->is_spring_connecting(i,j))
        draw_spring(i,j);
    }
  int leaf_cutoff = (_sbs->get_ball_count() + 2) / 2;
  for (i = 0; i < _sbs->get_ball_count(); i += 1)
    draw_sphere(i,  i >= leaf_cutoff);
  char buf[255];
  float src[3] = {-2.0*SPACESHRINK*10, 2*SPACESHRINK*10, SPACESHRINK*10};
  char *gstr = (char *) "   ";
  char *mstr = (char *) "   ";
  if (_sbs->get_gravity())
    gstr = (char *) "(g)";
  if (_sbs->get_motion())
    mstr = (char *) "(m)";
  sprintf(buf, "%s %s %s %s nodes: %d  seq: %08d  score: %3.9f", (_lm == 0) ? "[   ]" : "[mat]", (_rc == 0)?"(none)":real_compressor_name(_rc)->str, mstr, gstr, _smtw->get_leaf_node_count(), _smtw->get_seqno(), _smtw->get_score());
  sc->getProjectedCoordinatesOf(src, src);
  glDisable(GL_LIGHTING);
  //drawText(src[0], src[1], buf);
  drawText(50,40, buf);
  glEnable(GL_LIGHTING);
  return;
}

void SBViewer::make_quadrics()
{
  int argc = 1;
  char *argv[2];
  argv[0] = (char *) "sv";
  argv[1] = 0;
  int LoD = 1;
  glutInit(&argc, argv);
  _sphere_quadric = gluNewQuadric();
  _qspringcyl = gluNewQuadric();
  _smallkspherequad = gluNewQuadric();
  set_light_mode();
  make_lights();
      gluQuadricDrawStyle(_sphere_quadric, GLU_FILL);
    gluQuadricNormals(_sphere_quadric, GLU_SMOOTH);
    glNewList(1, GL_COMPILE);  /* create leaf sphere display list */
    glPushMatrix();
  glScalef(LEAFSHRINK,LEAFSHRINK,LEAFSHRINK);
    gluSphere(_sphere_quadric, /* radius */ SPHERERADIUS, /* slices */ SPHERESLICES*LoD,  /* stacks */ SPHERESTACKS*LoD
);
    glPopMatrix();
    glEndList();
    glNewList(3, GL_COMPILE);  /* create kernel sphere display list */
    gluSphere(_smallkspherequad, /* radius */ SPHERERADIUS, /* slices */ (int) (SPHERESLICES * SPHERESMALLFACTOR*LoD),  /* stacks */ (int) (SPHERESTACKS * SPHERESMALLFACTOR *LoD)
);
    glEndList();
    glPushMatrix();
    glNewList(4, GL_COMPILE); /* create spring cylinder display list */
    glPushMatrix();
    glTranslatef(0,0,-0.5);
  glScalef(SPHERESHRINK,SPHERESHRINK,0.24);
    gluCylinder(_qspringcyl, SPRINGCYLRAD, SPRINGCYLRAD, 1.06*SPHERERADIUS, SPHERESTACKS*LoD, (int) (SPHERESTACKS*CYLDETAILFACTOR*LoD));
    glPopMatrix();
    glEndList();
    glPopMatrix();
  _made_quadrics = true;
}


void SBViewer::init()
{
  double t1;

  _wireframe_on = false;
  _made_quadrics = false;

  _sbs = 0;
  _lm = 0;
  FILE *fp = fopen(_dist_fname, "rb");
  if (fp != 0) {
    fclose(fp);
    GString *dmblock = complearn_read_whole_file(_dist_fname);
    if (dmblock == NULL) {
      std::cerr << "Cannot read distance matrix " << _dist_fname << std::endl;
      exit(1);
    }
    _lm = complearn_load_any_matrix(dmblock);
  }
  GString *treeblock = 0;
  QSearchTree *inittree = 0;
  if (_tree_fname != 0) {
    treeblock = complearn_read_whole_file(_tree_fname);
    if (treeblock == NULL) {
      std::cerr << "Cannot read tree dot file " << _tree_fname << std::endl;
      exit(1);
    }
    inittree = qsearch_tree_new(_lm->mat->size1);
    qsearch_tree_read_from_dot(inittree, treeblock, _lm);
  }
  if (_lm) {
    _sbs = new SpringBallSystem(_tf, _lm->mat->size1*2-2);
    if (inittree != 0)
      _smtw = new SpringyMatrixTreeWatcher(inittree);
    else {
      _tm = qsearch_treemaster_new(_lm->mat);
      _smtw = new SpringyMatrixTreeWatcher(_tm);
    }
  }
  t1 = _tf.get_time();
  sc = camera();
  sc->setSceneRadius(SCENERADIUS);
  sc->setPosition(qglviewer::Vec(EYEDIST, EYEDIST, EYEDIST));
  lookTowardsCOM();
  if (_sbs) {
    _sbs->dosimloopthread();
  }
  startAnimation();
  help();
}

void SBViewer::lookTowardsCOM(void)
{
  double m[3];
  if (_sbs) {
    _sbs->getCOM(m);
    sc->lookAt(qglviewer::Vec(m[0],m[1],m[2]));
  }
}

void SBViewer::toggleWireframe(void)
{
  _wireframe_on = !_wireframe_on;
}

void SBViewer::toggleLabels(void)
{
  _labels_on = !_labels_on;
}


void SBViewer::keyPressEvent(QKeyEvent *q)
{
  switch (q->ascii()) {
    case 'g': _sbs->toggle_gravity(); break;
    case 'm': _sbs->toggle_motion(); break;
    case 'l': toggleLabels(); break;
    case 'w': toggleWireframe();
  _light_mode_already_set = false; break;
    default: QGLViewer::keyPressEvent(q);
//   case 'c': _sbs->toggle_coulomb(); break;
//    case 'b': destroySpring(); break;
  }
}

void SBViewer::draw_spring(int i1, int i2)
{
  double posx1, posx2, posy1, posy2, posz1, posz2;
  double p1[3], p2[3];
  _sbs->get_ball_position(i1, p1);
  _sbs->get_ball_position(i2, p2);
  posx1 =SPACESHRINK*p1[0]; posy1 =SPACESHRINK*p1[1]; posz1 =SPACESHRINK*p1[2];
  posx2 = SPACESHRINK*p2[0]; posy2 =SPACESHRINK*p2[1];posz2= SPACESHRINK*p2[2];
    double x, y, z;
  double ang1, ang2;
  double len, xzlen;
  glPushMatrix();
  glTranslatef( (posx1+posx2)/2, (posy1+posy2)/2, (posz1+posz2)/2 );
  x = posx2 - posx1; y = posy2 - posy1; z = posz2 - posz1;
  xzlen = sqrt(x*x+z*z);
  len = sqrt(x*x+y*y+z*z);
  ang2 = atan2(y, xzlen) * 180 / myPI;
  ang1 = -atan2(x, -z) * 180 / myPI;
  glRotatef(ang1, 0, 1, 0);
  glRotatef(ang2, 1, 0, 0);
  glScalef(1, 1, 0.9*len);
  glCallList(4);
  glPopMatrix();
}

void SBViewer::draw_sphere(int whichid, int issmall)
{
  double p[3];
  if (!_wireframe_on) {
    if (issmall) {
    glMaterialfv(GL_FRONT, GL_DIFFUSE, matkern_diff);
    glMaterialfv(GL_FRONT, GL_SPECULAR, matkern_spec);
    glMaterialfv(GL_FRONT, GL_SHININESS, matkern_shine);
    } else {
    glMaterialfv(GL_FRONT, GL_DIFFUSE, matball_diff);
    glMaterialfv(GL_FRONT, GL_SPECULAR, matball_spec);
    glMaterialfv(GL_FRONT, GL_SHININESS, matball_shine);
    }
  }
  _sbs->get_ball_position(whichid, p);
  glPushMatrix();
  glTranslatef( SPACESHRINK*p[0], SPACESHRINK*p[1], SPACESHRINK*p[2] );
  float src[3] = { SPACESHRINK*p[0], SPACESHRINK*p[1], SPACESHRINK*p[2] };
  sc->getProjectedCoordinatesOf(src, src);
  if (issmall)
    glScalef(SPHERESMALLFACTOR,SPHERESMALLFACTOR,SPHERESMALLFACTOR);
  glCallList(1);
  glPopMatrix();
  if (_labels_on && !issmall) {
    glMaterialfv(GL_FRONT, GL_DIFFUSE, wire_diff);
    glMaterialfv(GL_FRONT, GL_SPECULAR, wire_spec);
    glMaterialfv(GL_FRONT, GL_SHININESS, wire_shine);
    glDisable(GL_LIGHTING);
    drawText(src[0], src[1], _lm->labels1[whichid]);
    glEnable(GL_LIGHTING);
  }
}

QString SBViewer::helpString() const
{
  QString text("<h2>QSearch CompLearn GUI DEMO</h2>");
  text += "Use the mouse to move the tree around.<br>";
  text += "Press g to toggle alternating gravity.<br>";
  text += "Press ESC to quit.<br>";
  text += "Press m to toggle motion.<br>";
  text += "Use mouse wheel to zoom in and out.<br>";
  text += "Use both left and right buttons held and drag to zoom in/out.<br>";
  return text;
}
SBViewer::SBViewer(QWidget *qw, const char *str) : QGLViewer(qw, str)
{
  char *fakeargv[2];
  fakeargv[0] = strdup("clgui");
  fakeargv[1] = 0;
  int fakeargc = 1;
  _win = (MWGen *) qw;
  do_init(fakeargc, fakeargv);
  free(fakeargv[0]);
}

static QLabel *simpleLabel(QWidget *th, int x, int *yptr, const char *str) {
  QLabel *lab = new QLabel(th, str);
  lab->setGeometry(x, *yptr, th->width(), 20);
  lab->setText(str);
  *yptr += 15;
  return lab;
}

OkDialog::OkDialog(const char *msg) {
  setCaption("Ok");
  resize(600,80);
  int x=20, y=20;
  simpleLabel(this, x,&y,"Successful Operation");
  simpleLabel(this, x,&y,msg);
  show();
}

ErrorDialog::ErrorDialog(const char *msg) {
  setCaption("Error");
  resize(600,80);
  int x=20, y=20;
  simpleLabel(this, x,&y,"CompLearn Error");
  simpleLabel(this, x,&y,msg);
  show();
}

HelpAboutDialog::HelpAboutDialog(void) {
  setCaption("About");
  resize(400,80);
  int x=20, y=20;
  simpleLabel(this, x,&y,"CompLearn GUI Demonstration");
  simpleLabel(this, x,&y,"by Rudi Cilibrasi <cilibrar@cilibrar.com>");
  simpleLabel(this, x,&y,"see http://complearn.org/ for more info");
  show();
}

HelpDialog::HelpDialog(void) {
  setCaption("Help");
  resize(400,80);
  int x=20, y=20;
  simpleLabel(this, x,&y,"To use this program, first make a matrix.");
  simpleLabel(this, x,&y,"Next, calculate a tree from it.");
  simpleLabel(this, x,&y,"see http://complearn.org/ for more info");
  show();
}

void SBViewer::fileChangeCompressor(void) {
  new CompressionDialog(&_rc);
}

void SBViewer::fileHelp(void) {
  new HelpDialog();
}

void SBViewer::distmatOpen(void) {
  QString s = QFileDialog::getOpenFileName(
                    ".",
                    "(*.clb)",
                    this,
                    "Open Distance Matrix"
                    "Choose a file" );
  const char *fname = s.data();
  if (fname == 0)
    return;
  if (fname != 0)
    readAndAnalyzeDistanceMatrix(fname);
}
void SBViewer::readAndAnalyzeDistanceMatrix(const char *fname) {
  GString *dmblock = complearn_read_whole_file(fname);
  if (dmblock == NULL) {
    new ErrorDialog("Cannot read distance matrix");
    return;
  }
  _lm = complearn_load_any_matrix(dmblock);
  if (_sbs)
    _sbs->stopsimloop();
  _sbs = new SpringBallSystem(_tf, _lm->mat->size1*2-2);
  if (_tm) {
    qsearch_treemaster_stop_search(_tm);
  }
  _tm = qsearch_treemaster_new(_lm->mat);
  _smtw = new SpringyMatrixTreeWatcher(_tm);
  _sbs->dosimloopthread();
}

void SBViewer::treeSave(void) {
  if (_tm == 0) {
    new ErrorDialog("You must calculate a tree first.");
    return;
  }
  QString s = QFileDialog::getSaveFileName(
                    ".",
                    "(*.dot)",
                    this,
                    "Save Tree"
                    "Choose a file" );
  const char *fname = s.data();
  if (fname == 0)
    return;
  if (strstr(fname, ".dot") == 0) {
    char *nfn = (char *) calloc(strlen(fname)+8, 1);
    sprintf(nfn, "%s.dot", fname);
    fname = nfn;
  }
  FILE *fp = fopen(fname, "wb");
  GString *gs = complearn_matrix_prettyprint_clb(_lm);
  fwrite(gs->str, gs->len, 1, fp);
  fclose(fp);
  g_string_free(gs, TRUE);
  new OkDialog("Distance Matrix Saved");
}

void SBViewer::treeOpen(void) {
  if (_lm == 0) {
    new ErrorDialog("Must load correspond matrix first.");
    return;
  }
  QString s = QFileDialog::getOpenFileName(
                    ".",
                    "(*.dot)",
                    this,
                    "Open Tree"
                    "Choose tree to view" );
  if (s.data() == 0)
    return;
  _tree_fname = strdup(s.data());
  GString *treeblock = 0;
  QSearchTree *inittree = 0;
  treeblock = complearn_read_whole_file(_tree_fname);
  if (treeblock == NULL) {
    std::cerr << "Cannot read tree dot file " << _tree_fname << std::endl;
    exit(1);
  }
  inittree = qsearch_tree_new(_lm->mat->size1);
  qsearch_tree_read_from_dot(inittree, treeblock, _lm);
  if (_sbs)
    _sbs->stopsimloop();
  if (_tm) {
    qsearch_treemaster_stop_search(_tm);
  }
  _sbs = new SpringBallSystem(_tf, _lm->mat->size1*2-2);
  _smtw = new SpringyMatrixTreeWatcher(inittree);
  _sbs->dosimloopthread();
}

void SBViewer::distmatNew(void) {
  static int haveDoneMatrix;
  if (haveDoneMatrix) {
    new ErrorDialog("Sorry, only one new matrix per program invocation.");
    return;
  }
  haveDoneMatrix = 1;
  QString s = QFileDialog::getExistingDirectory(
                    ".",
                    this,
                    "Choose Directory",
                    "Which directory to analyze" );
  const char *dname = s.data();
  if (dname == 0)
    return;
  GSList *dblocks;
  dblocks = complearn_read_directory_of_files(dname);
  int len = g_slist_length(dblocks);
  if (_rc == 0) {
    new ErrorDialog("No compressor available, sorry.");
    return;
  }
  if (len < 4) {
    new ErrorDialog("You must have at least four files to analyze.");
    return;
  }
  if (len > 300) {
    new ErrorDialog("No more than 300 files are allowed to analyze");
    return;
  }
  complearn_ncd_load_compressor(real_compressor_name(_rc)->str);
  complearn_ncd_set_directory_mode();
  complearn_ncd_add_argument(dname);
  complearn_ncd_add_argument(dname);
  complearn_ncd_set_binary_output_mode();
  gchar *outfname = strdup("/tmp/distmat");
  complearn_ncd_set_output_filestem(complearn_ncd_top(), outfname);
  complearn_ncd_set_write_mode();
  complearn_ncd_do_double_list();
  free(outfname);
  outfname = strdup("/tmp/distmat.clb");
  readAndAnalyzeDistanceMatrix(outfname);
  free(outfname);
}

void SBViewer::distmatSave(void) {
  if (_lm == 0) {
    new ErrorDialog("You must make a distance matrix first.");
    return;
  }
  QString s = QFileDialog::getSaveFileName(
                    ".",
                    "(*.clb)",
                    this,
                    "Save Distance Matrix"
                    "Choose a file" );
  const char *fname = s.data();
  if (fname == 0)
    return;
  if (strstr(fname, ".clb") == 0) {
    char *nfn = (char *) calloc(strlen(fname)+8, 1);
    sprintf(nfn, "%s.clb", fname);
    fname = nfn;
  }
  FILE *fp = fopen(fname, "wb");
  GString *gs = complearn_matrix_prettyprint_clb(_lm);
  fwrite(gs->str, gs->len, 1, fp);
  fclose(fp);
  g_string_free(gs, TRUE);
  new OkDialog("Distance Matrix Saved");
}

void SBViewer::treeCalculate(void) {
  if (_sbs != 0) {
    new ErrorDialog("A tree is already calculating before your eyes.");
    return;
  }
  if (_lm == 0) {
    new ErrorDialog("Cannot calculate tree without distance matrix.");
    return;
  }
}

void SBViewer::fileAbout(void) {
  new HelpAboutDialog();
  return;
/*
  if (_tm != 0) {
    qsearch_treemaster_stop_search(_tm);
    _tm = 0;
  }
  if (_sbs != 0) {
    _sbs->stopsimloop();
    _sbs = 0;
  }
QString s = QFileDialog::getOpenFileName(
                    "/home",
                    "Images (*.png *.xpm *.jpg)",
                    this,
                    "open file dialog"
                    "Choose a file" );
*/
}
