# Written by John Hoffman
# Modified by Cameron Dale
# see LICENSE.txt for license information
#
# $Id: T2T.py 266 2007-08-18 02:06:35Z camrdale-guest $

"""Tracker to tracker connection management.

@type logger: C{logging.Logger}
@var logger: the logger to send all log messages to for this module
@type R_0: C{function}
@var R_0: a function that always returns 0
@type R_1: C{function}
@var R_1: a function that always returns 0

"""

from Rerequester import Rerequester
from urllib import quote
from threading import Event
from random import randrange
from string import lower
import sys, logging
import __init__

logger = logging.getLogger('DebTorrent.BT1.T2T')

R_0 = lambda: 0
R_1 = lambda: 1

class T2TConnection:
    """A single tracker to tracker connection for a single torrent.
    
    Creates a L{Rerequester.Rerequester} object but does not start it running,
    instead all requesting is controlled from this class.
    
    @type tracker: C{string}
    @ivar tracker: the tracker address to contact
    @type interval: C{int}
    @ivar interval: original seconds between outgoing tracker announces
    @type hash: C{string}
    @ivar hash: the info hash of the torrent
    @type operatinginterval: C{int}
    @ivar operatinginterval: current seconds between outgoing tracker announces
    @type peers: C{int}
    @ivar peers: number of peers to get in a tracker announce
    @type rawserver: L{DebTorrent.RawServer.RawServer}
    @ivar rawserver: the server instance to use
    @type disallow: C{method}
    @ivar disallow: method to call to disallow a tracker address
    @type isdisallowed: C{method}
    @ivar isdisallowed: method to call to check if a tracker address is disallowed
    @type active: C{boolean}
    @ivar active: whether the connection is active
    @type busy: C{boolean}
    @ivar busy: not used
    @type errors: C{int}
    @ivar errors: the number of errors that have occurred since the last 
        successful request
    @type rejected: C{int}
    @ivar rejected: the number of rejections that have occurred since the last 
        successful request
    @type trackererror: C{boolean}
    @ivar trackererror: not used
    @type peerlists: C{list} of C{list} of (C{string}, C{string}, C{int})
    @ivar peerlists: the last 10 announce peer data of peer ID, IP address, and port
    @type rerequester: L{Rerequester.Rerequester}
    @ivar rerequester: the tracker rerequester instance
    @type lastsuccessful: C{boolean}
    @ivar lastsuccessful: whether the last request was successful
    @type newpeerdata: C{list} of (C{string}, C{string}, C{int})
    @ivar newpeerdata: the list of peer data of peer ID, IP address, and port
    
    """
    
    def __init__(self, myid, tracker, hash, interval, peers, timeout,
                     rawserver, disallow, isdisallowed):
        """Initialize the instance and schedule a request.
        
        @type myid: C{string}
        @param myid: the peer ID to send to the tracker
        @type tracker: C{string}
        @param tracker: the tracker address to contact
        @type hash: C{string}
        @param hash: the info hash of the torrent
        @type interval: C{int}
        @param interval: seconds between outgoing tracker announces
        @type peers: C{int}
        @param peers: number of peers to get in a tracker announce
        @type timeout: C{int}
        @param timeout: number of seconds to wait before assuming that a
            tracker connection has timed out
        @type rawserver: L{DebTorrent.RawServer.RawServer}
        @param rawserver: the server instance to use
        @type disallow: C{method}
        @param disallow: method to call to disallow a tracker address
        @type isdisallowed: C{method}
        @param isdisallowed: method to call to check if a tracker address is disallowed
        
        """
        
        self.tracker = tracker
        self.interval = interval
        self.hash = hash
        self.operatinginterval = interval
        self.peers = peers
        self.rawserver = rawserver
        self.disallow = disallow
        self.isdisallowed = isdisallowed
        self.active = True
        self.busy = False
        self.errors = 0
        self.rejected = 0
        self.trackererror = False
        self.peerlists = []
        cfg = { 'min_peers': peers,
                'max_initiate': peers,
                'rerequest_interval': interval,
                'http_timeout': timeout }
        self.rerequester = Rerequester( 0, myid, hash, [[tracker]], cfg,
            rawserver.add_task, rawserver.add_task, self.errorfunc,
            self.addtolist, R_0, R_1, R_0, R_0, R_0, R_0,
            Event() )

        if self.isactive():
            rawserver.add_task(self.refresh, randrange(int(self.interval/10), self.interval))
                                        # stagger announces

    def isactive(self):
        """Check if the tracker connection has been disallowed.
        
        @rtype: C{boolean}
        @return: whether the connection is active
        
        """
        
        if self.isdisallowed(self.tracker):    # whoops!
            self.deactivate()
        return self.active
            
    def deactivate(self):
        """Deactive the connection."""
        self.active = False

    def refresh(self):
        """Request new peer data from the tracker."""
        if not self.isactive():
            return
        self.lastsuccessful = True
        self.newpeerdata = []
        logger.info('contacting '+self.tracker+' for info_hash='+quote(self.hash))
        self.rerequester.snoop(self.peers, self.callback)

    def callback(self):
        """Process the returned peer data from the tracker."""
        self.busy = False
        if self.lastsuccessful:
            self.errors = 0
            self.rejected = 0
            if self.rerequester.announce_interval > (3*self.interval):
                # I think I'm stripping from a regular tracker; boost the number of peers requested
                self.peers = int(self.peers * (self.rerequester.announce_interval / self.interval))
            self.operatinginterval = self.rerequester.announce_interval
            logger.info(self.tracker+' with info_hash='+quote(self.hash)+' returned '+str(len(self.newpeerdata))+' peers')
            self.peerlists.append(self.newpeerdata)
            self.peerlists = self.peerlists[-10:]  # keep up to the last 10 announces
        if self.isactive():
            self.rawserver.add_task(self.refresh, self.operatinginterval)

    def addtolist(self, peers):
        """Save the returned peer data from the tracker for later processing.
        
        @type peers: C{list} of ((C{string}, C{int}), C{string}, C{boolean})
        @param peers: the list of IP address, port, peer ID, and whether to encrypt
        
        """
        
        for peer in peers:
            self.newpeerdata.append((peer[1],peer[0][0],peer[0][1]))
        
    def errorfunc(self, r):
        """Process an error that occurred.
        
        @type r: C{string}
        @param r: the error message
        
        """
        
        self.lastsuccessful = False
        logger.info(self.tracker+' with info_hash='+quote(self.hash)+' gives error: "'+r+'"')
        if r == self.rerequester.rejectedmessage + 'disallowed':   # whoops!
            logger.info(' -- disallowed - deactivating')
            self.deactivate()
            self.disallow(self.tracker)   # signal other torrents on this tracker
            return
        if lower(r[:8]) == 'rejected': # tracker rejected this particular torrent
            self.rejected += 1
            if self.rejected == 3:     # rejected 3 times
                logger.info(' -- rejected 3 times - deactivating')
                self.deactivate()
            return
        self.errors += 1
        if self.errors >= 3:                         # three or more errors in a row
            self.operatinginterval += self.interval  # lengthen the interval
            logger.info(' -- lengthening interval to '+str(self.operatinginterval)+' seconds')

    def harvest(self):
        """Retrieve the saved list of peers from this tracker connection.
        
        @rtype: C{list} of (C{string}, C{string}, C{int})
        @return: the list of peer data of peer ID, IP address, and port
        
        """
        
        x = []
        for list in self.peerlists:
            x += list
        self.peerlists = []
        return x


class T2TList:
    """A list of all tracker to tracker connections.
    
    @type enabled: C{boolean}
    @ivar enabled: whether to enable multitracker operation
    @type trackerid: C{string}
    @ivar trackerid: this tracker's ID
    @type interval: C{int}
    @ivar interval: seconds between outgoing tracker announces
    @type maxpeers: C{int}
    @ivar maxpeers: number of peers to get in a tracker announce
    @type timeout: C{int}
    @ivar timeout: number of seconds to wait before assuming that a
        tracker connection has timed out
    @type rawserver: L{DebTorrent.RawServer.RawServer}
    @ivar rawserver: the server instance to use
    @type list: {C{string}: {C{string}: L{T2TConnection}, ...}, ...}
    @ivar list: keys are the tracker addresses, values are dictionaries with
        torrent info hashes as keys and the connection to that tracker for
        that torrent as values
    @type torrents: {C{string}: [L{T2TConnection}, ...], ...}
    @ivar torrents: keys are the info hashes, values are a list of the
        tracker connections for that torrent
    @type disallowed: {C{string}: C{boolean}, ...}
    @ivar disallowed: keys are the tracker addresses, values are True if the
        tracker is disallowing this tracker
    @type oldtorrents: C{list} of L{T2TConnection}
    @ivar oldtorrents: deactivated connections that are kept in case threads
        try to access them
    
    """
    
    def __init__(self, enabled, trackerid, interval, maxpeers, timeout, rawserver):
        """Initialize the instance.
        
        @type enabled: C{boolean}
        @param enabled: whether to enable multitracker operation
        @type trackerid: C{string}
        @param trackerid: this tracker's ID
        @type interval: C{int}
        @param interval: seconds between outgoing tracker announces
        @type maxpeers: C{int}
        @param maxpeers: number of peers to get in a tracker announce
        @type timeout: C{int}
        @param timeout: number of seconds to wait before assuming that a
            tracker connection has timed out
        @type rawserver: L{DebTorrent.RawServer.RawServer}
        @param rawserver: the server instance to use
        
        """
        
        self.enabled = enabled
        self.trackerid = trackerid
        self.interval = interval
        self.maxpeers = maxpeers
        self.timeout = timeout
        self.rawserver = rawserver
        self.list = {}
        self.torrents = {}
        self.disallowed = {}
        self.oldtorrents = []

    def parse(self, allowed_list):
        """Parse a list of allowed torrents and enable any new ones.
        
        @type allowed_list: C{dictionary}
        @param allowed_list: keys are info hashes, values are the torrent data
        
        """
        
        if not self.enabled:
            return

        # step 1:  Create a new list with all tracker/torrent combinations in allowed_dir        
        newlist = {}
        for hash, data in allowed_list.items():
            if data.has_key('announce-list'):
                for tier in data['announce-list']:
                    for tracker in tier:
                        self.disallowed.setdefault(tracker, False)
                        newlist.setdefault(tracker, {})
                        newlist[tracker][hash] = None # placeholder
                            
        # step 2:  Go through and copy old data to the new list.
        # if the new list has no place for it, then it's old, so deactivate it
        for tracker, hashdata in self.list.items():
            for hash, t2t in hashdata.items():
                if not newlist.has_key(tracker) or not newlist[tracker].has_key(hash):
                    t2t.deactivate()                # this connection is no longer current
                    self.oldtorrents += [t2t]
                        # keep it referenced in case a thread comes along and tries to access.
                else:
                    newlist[tracker][hash] = t2t
            if not newlist.has_key(tracker):
                self.disallowed[tracker] = False    # reset when no torrents on it left

        self.list = newlist
        newtorrents = {}

        # step 3:  If there are any entries that haven't been initialized yet, do so.
        # At the same time, copy all entries onto the by-torrent list.
        for tracker, hashdata in newlist.items():
            for hash, t2t in hashdata.items():
                if t2t is None:
                    hashdata[hash] = T2TConnection(self.trackerid, tracker, hash,
                                        self.interval, self.maxpeers, self.timeout,
                                        self.rawserver, self._disallow, self._isdisallowed)
                newtorrents.setdefault(hash,[])
                newtorrents[hash] += [hashdata[hash]]
                
        self.torrents = newtorrents

    def _disallow(self,tracker):
        """Disallow all connections from contacting a tracker.
        
        @type tracker: C{string}
        @param tracker: the tracker address to disallow
        
        """
        
        self.disallowed[tracker] = True

    def _isdisallowed(self,tracker):
        """Check if a tracker has been disallowed.
        
        @type tracker: C{string}
        @param tracker: the tracker address to check
        @rtype: C{boolean}
        @return: whether the tracker has been disallowed
        
        """
        
        return self.disallowed[tracker]

    def harvest(self,hash):
        """Harvest a list of peers from all tracker's for a torrent.
        
        @type hash: C{string}
        @param hash: the info hash of the torrent to get peers for
        @rtype: C{list} of (C{string}, C{string}, C{int})
        @return: the list of peer data of peer ID, IP address, and port
        
        """
        
        harvest = []
        if self.enabled:
            for t2t in self.torrents[hash]:
                harvest += t2t.harvest()
        return harvest
