/* main.c
 * sets up the GUI and connects the main callback functions.
 *
 * for Denemo, a gtk+ frontend to GNU Lilypond
 * (c) 1999, 2000, 2001 Matthew Hiller
 */

#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <unistd.h>
#include <gtk/gtk.h>
#ifndef G_OS_WIN32
# include <sys/wait.h>
#endif

#include "commandfuncs.h"
#include "config.h"
#include "draw.h"		/* Which includes gtk.h */
#include "datastructures.h"
#include "dialogs.h"
#include "exportmudela.h"
#include "file.h"
#include "gcs.h"
#include "kbd-custom.h"
#include "kbd-interface.h"
#include "keyresponses.h"
#include "contexts.h"
#include "help.h"
#include "midi.h"
#include "mousing.h"
#include "moveviewport.h"
#include "prefops.h"
#include "scoreops.h"
#include "selectops.h"
#include "staffops.h"
#include "utils.h"
#include "lilydirectives.h"
#include "dynamic.h"
#include "changenotehead.h"
#include "articulations.h"
#include "analysis_highlighting.h"
#include "print.h"

#define INITIAL_WIDTH 1100
#define INITIAL_HEIGHT 500

/* signal handler to be invoked when child processes _exit() without
 * having to wait for them */

/* Code by Erik Mouw, taken directly from the gtk+ FAQ */

void
sigchld_handler (gint num)
{
  sigset_t set, oldset;
  pid_t pid;
  gint status, exitstatus;

#ifndef G_OS_WIN32
  /* block other incoming SIGCHLD signals */
  sigemptyset (&set);
  sigaddset (&set, SIGCHLD);
  sigprocmask (SIG_BLOCK, &set, &oldset);

  /* wait for child */
  while ((pid = waitpid ((pid_t) - 1, &status, WNOHANG)) > 0)
    {
      if (WIFEXITED (status))
	{
	  exitstatus = WEXITSTATUS (status);

	  fprintf (stderr,
		   _("Parent: child exited, pid = %d, exit status = %d\n"),
		   (int) pid, exitstatus);
	}
      else if (WIFSIGNALED (status))
	{
	  exitstatus = WTERMSIG (status);

	  fprintf (stderr,
		   _("Parent: child terminated by signal %d, pid = %d\n"),
		   exitstatus, (int) pid);
	}
      else if (WIFSTOPPED (status))
	{
	  exitstatus = WSTOPSIG (status);

	  fprintf (stderr,
		   _("Parent: child stopped by signal %d, pid = %d\n"),
		   exitstatus, (int) pid);
	}
      else
	{
	  fprintf (stderr,
		   _("Parent: child exited magically, pid = %d\n"),
		   (int) pid);
	}
    }

  /* re-install the signal handler (some systems need this) */
  signal (SIGCHLD, sigchld_handler);

  /* and unblock it */
  sigemptyset (&set);
  sigaddset (&set, SIGCHLD);
  sigprocmask (SIG_UNBLOCK, &set, &oldset);
#endif
}

/* when invoked (via signal delete_event or file->quit), terminates
 * the application.  */

void
closeit (GtkWidget * widget, gpointer data)
{
  gtk_main_quit ();
}

void
close_application (GtkWidget * widget, GdkEvent * event, gpointer data)
{
  confirmbox (data, GTK_SIGNAL_FUNC (closeit));
}

void
closewrapper (gpointer data, guint callback_action, GtkWidget * widget)
{
  confirmbox (data, GTK_SIGNAL_FUNC (closeit));
}

GtkItemFactoryEntry menu_items[] = {
  {N_("/_File"), "<control>a", NULL, 0, "<Branch>"},
  {N_("/File/"), "<control>b", NULL, 0, "<Tearoff>"},
  {N_("/File/_New"), NULL, file_newwrapper, 0, NULL},
  {N_("/File/_Open"), NULL, file_openwrapper, 0, NULL},
  {N_("/File/_Save"), NULL, file_savewrapper, 0, NULL},
  {N_("/File/Save _As"), NULL, file_saveaswrapper, 0, NULL},
  {N_("/File/sep1"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/Set _Headers"), NULL, header_change, 0, NULL},
  {N_("/File/sep3"), NULL, NULL, 0, "<Separator>"},
  {N_("/File/_Print"), NULL, print, 0, NULL},
  {N_("/File/_Quit"), NULL, closewrapper, 0, NULL},
  {N_("/_Edit"), NULL, NULL, 0, "<Branch>"},
  {N_("/Edit/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Edit/_Copy"), NULL, copywrapper, 0, NULL},
  {N_("/Edit/C_ut"), NULL, cutwrapper, 0, NULL},
  {N_("/Edit/_Paste"), NULL, pastewrapper, 0, NULL},
  {N_("/Edit/Save Selection"), NULL, saveselwrapper, 0, NULL},
  {N_("/Edit/sep1"), NULL, NULL, 0, "<Separator>"},
  {N_("/Edit/P_references"), NULL, preferences_change, 0, NULL},
  {N_("/Edit/_Keyboard"), NULL, configure_keyboard_dialog, 0, NULL},
  {N_("/_Staff"), NULL, NULL, 0, "<Branch>"},
  {N_("/Staff/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Staff/Add New Staff _Before Current Staff"), NULL, newstaff,
   BEFORE, NULL},
  {N_("/Staff/Add New Staff _After Current Staff"), NULL, newstaff,
   AFTER, NULL},
  {N_("/Staff/Add New Staff in _Initial Position"), NULL, newstaff,
   FIRST, NULL},
  {N_("/Staff/Add New Staff in _Last Position"), NULL, newstaff,
   LAST, NULL},
  {N_("/Staff/sep1"), NULL, NULL, 0, "<Separator>"},
  {N_("/Staff/Add _Voice to Current Staff"), NULL, newstaff, NEWVOICE, NULL},
  {N_("/Staff/sep2"), NULL, NULL, 0, "<Separator>"},
  {N_("/Staff/Staff _Properties"), NULL, staff_properties_change, 0, NULL},
  {N_("/_Clef"), NULL, NULL, 0, "<Branch>"},
  {N_("/Clef/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Clef/Set _Initial Clef"), NULL, clef_change, CHANGEINITIAL,
   NULL},
  {N_("/Clef/Insert Clef _Change"), NULL, clef_change, INSERT, NULL},
  {N_("/_Key"), NULL, NULL, 0, "<Branch>"},
  {N_("/Key/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Key/Set _Initial Key Signature"), NULL,
   key_change, CHANGEINITIAL, NULL},
  {N_("/Key/Insert Key Signature _Change"), NULL, key_change,
   INSERT, NULL},
  {N_("/_Time"), NULL, NULL, 0, "<Branch>"},
  {N_("/Time/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Time/Set _Initial Time Signature"), NULL,
   timesig_change, CHANGEINITIAL, NULL},
  {N_("/Time/Insert Time Signature _Change"), NULL,
   timesig_change, INSERT, NULL},
  {N_("/_Other"), NULL, NULL, 0, "<Branch>"},
  {N_("/Other/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Other/Change _Notehead"), NULL, set_notehead, 0, NULL},
  {N_("/Other/Insert _Stemming Directive"), NULL, stem_directive_insert, 0,
   NULL},
  {N_("/Other/Insert _Dynamic"), NULL, insert_dynamic, 0, NULL},
  {N_("/Other/Insert _Articulation"), NULL, NULL, 0, "<Branch>"},
  
    {N_("/Other/Insert _Articulation/General"), NULL, insert_articulation,
   GENERAL, NULL},
  {N_("/Other/Insert _Articulation/String"), NULL, insert_articulation,
   STRING, NULL},
  {N_("/Other/Insert _Articulation/Organ"), NULL, insert_articulation, ORGAN,
   NULL},
  {N_("/_Display"), NULL, NULL, 0, "<Branch>"},
  {N_("/Display/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/_Display/Change _Measure Width"), NULL, score_mwidth_change, 0,
   NULL},
  {N_("/_Display/Change _Space between Staffs"), NULL, score_staffspace_change,
   0, NULL},
  {N_("/_Navigation"), NULL, NULL, 0, "<Branch>"},
  {N_("/Navigation/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Navigation/Go _To Measure"), NULL, tomeasurenum, 0, NULL},
  {N_("/Navigation/To _Beginning of Score"), NULL, tohome, 0, NULL},
  {N_("/Navigation/To _End of Score"), NULL, toend, 0, NULL},
  {N_("/_Playback"), NULL, NULL, 0, "<Branch>"},
  {N_("/Playback/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Playback/_Play"), NULL, playback, 0, NULL},
  {N_("/Playback/P_roperties"), NULL, playback_properties_change, 0, NULL},
  {N_("/Analysis/"), NULL, NULL, 0, "<Branch>"},
  {N_("/Analysis/"), NULL, NULL, 0, "<Tearoff>"},
  {N_("/Analysis/Read Analysis Results"), NULL, read_resultsfile, 0, NULL},
  {N_("/Analysis/Highlight"), NULL, highlight, 0, NULL},
  {N_("/Analysis/Unhighlight"), NULL, unhighlight, 0, NULL},
  {N_("/Analysis/Analyse"), NULL, perform_analysis, 0, NULL},
  {N_("/_Help"), NULL, NULL, 0, "<LastBranch>"},
  {N_("/Help/"), NULL, NULL, 0, "<Tearoff>"},
  /*  {N_("/Help/List _Keybindings"), NULL, keybindings, 0, NULL}, */
  {N_("/_Help/_About"), NULL, about, 0, NULL}
};

gint n_menu_items = (sizeof (menu_items) / sizeof (GtkItemFactoryEntry));

int
main (int argc, char *argv[])
{
  struct scoreinfo *si = g_malloc (sizeof (struct scoreinfo));
  struct prefinfo *prefs;
  GtkWidget *main_vbox, *menubar, *score_and_scroll_hbox;
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint opts;
  static struct option long_options[] = {
    {"help", 0, NULL, 'h'},
    {"version", 0, NULL, 'v'}
  };
  gchar *helptext = g_strconcat (_("\nGNU Denemo version "), VERSION ".\n\n",
				 _("usage: denemo [OPTION... [FILE]\n\n\
Run denemo, opening save file FILE\n\n\
Denemo is a graphical music notation editor.  It produces save files\n\
in GNU Lilypond input format (suitable for immediate typesetting with GNU\n\
Lilypond) and Adam Tee's JTF file format. Denemo is part of the GNU\n\
project.\n\n\
Options:\n\
  -h,--help                 print this help and exit\n\
  -v,--version              print version number and exit\n\n\n\
Report bugs to bug-denemo@gnu.org\n"), NULL);
  gchar *copytext
    = _("(c) 1999, 2000, 2001 Matthew Hiller, Adam Tee, and others\n\n\n\
This program is provided with absolutely NO WARRANTY; see\n\
the file COPYING for details.\n\n\
This software may be redistributed and modified under the\n\
terms of the GNU General Public License; again, see the file\n\
COPYING for details.\n\n");
  gchar *win32text
    = _("Warning: the win32 port of Denemo is as yet incomplete.\n\
Most prominently, playback and options do not work.\n\n");

  gtk_init (&argc, &argv);

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

#ifndef G_OS_WIN32
  while ((opts = getopt_long (argc, argv, "hv", long_options, NULL)) != -1)
    {
      if (opts == 'h')
	{
	  printf (helptext);
	  exit (0);
	}
      else if (opts == 'v')
	{
	  printf (_("\nGNU Denemo version "));
	  printf (VERSION ".\n\n");
	  printf (copytext);
	  exit (0);
	}
    }
#endif

  printf (_("\nGNU Denemo, a gtk+ frontend for GNU Lilypond\n"));
  printf (copytext);

#ifdef G_OS_WIN32
  printf (win32text);
#endif
  g_free (helptext);

  /* Initialize preferences */
  prefs = initprefs ();

  /* Do a midi init */
  midi_init ();

  /* Initialize the staff structure */
  init_score (si);
  si->prefs = prefs;
  newstaff (si, INITIAL, NULL);
  si->pixmap = NULL;

  /* Initialize the GUI */

  si->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy (GTK_WINDOW (si->window), TRUE, TRUE, FALSE);

  main_vbox = gtk_vbox_new (FALSE, 1);
  gtk_container_border_width (GTK_CONTAINER (main_vbox), 1);
  gtk_container_add (GTK_CONTAINER (si->window), main_vbox);
  gtk_widget_show (main_vbox);

  /* This part is taken more-or-less directly from the gtk+ tutorial */
  accel_group = gtk_accel_group_new ();
  item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>",
				       accel_group);
  gtk_item_factory_create_items (item_factory, n_menu_items, menu_items, si);
  /* This also sets si as the  callback data for all the functions in the 
   * menubar, which is precisely what we want. */
  gtk_accel_group_attach (accel_group, GTK_OBJECT (si->window));
  menubar = gtk_item_factory_get_widget (item_factory, "<main>");
  gtk_box_pack_start (GTK_BOX (main_vbox), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);

  score_and_scroll_hbox = gtk_hbox_new (FALSE, 1);
  gtk_box_pack_start (GTK_BOX (main_vbox), score_and_scroll_hbox, TRUE, TRUE,
		      0);
  gtk_widget_show (score_and_scroll_hbox);

  si->scorearea = gtk_drawing_area_new ();
  gtk_drawing_area_size (GTK_DRAWING_AREA (si->scorearea),
			 INITIAL_WIDTH, INITIAL_HEIGHT);
  gtk_box_pack_start (GTK_BOX (score_and_scroll_hbox), si->scorearea, TRUE,
		      TRUE, 0);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "expose_event",
		      GTK_SIGNAL_FUNC (scorearea_expose_event), si);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "configure_event",
		      GTK_SIGNAL_FUNC (scorearea_configure_event), si);
  gtk_signal_connect (GTK_OBJECT (si->scorearea), "button_release_event",
		      GTK_SIGNAL_FUNC (scorearea_button_release), si);
  gtk_widget_set_events (si->scorearea, (GDK_EXPOSURE_MASK
					 | GDK_LEAVE_NOTIFY_MASK
					 | GDK_BUTTON_PRESS_MASK
					 | GDK_BUTTON_RELEASE_MASK));
  gtk_widget_show (si->scorearea);

  si->vadjustment = gtk_adjustment_new (1.0, 1.0, 2.0, 1.0, 4.0, 1.0);
  gtk_signal_connect (GTK_OBJECT (si->vadjustment), "value_changed",
		      GTK_SIGNAL_FUNC (vertical_scroll), si);
  si->vscrollbar = gtk_vscrollbar_new (GTK_ADJUSTMENT (si->vadjustment));
  gtk_box_pack_start (GTK_BOX (score_and_scroll_hbox), si->vscrollbar, FALSE,
		      TRUE, 0);
  gtk_widget_show (si->vscrollbar);

  si->hadjustment = gtk_adjustment_new (1.0, 1.0, 2.0, 1.0, 4.0, 1.0);
  gtk_signal_connect (GTK_OBJECT (si->hadjustment), "value_changed",
		      GTK_SIGNAL_FUNC (horizontal_scroll), si);
  si->hscrollbar = gtk_hscrollbar_new (GTK_ADJUSTMENT (si->hadjustment));
  gtk_box_pack_start (GTK_BOX (main_vbox), si->hscrollbar, FALSE, TRUE, 0);
  gtk_widget_show (si->hscrollbar);

  gtk_signal_connect (GTK_OBJECT (si->window), "delete_event",
		      (GtkSignalFunc) close_application, si);
  gtk_signal_connect (GTK_OBJECT (si->window), "key_press_event",
		      (GtkSignalFunc) scorearea_keypress_event, si);
  gtk_widget_show (si->window);

  /* Now that the window's shown, initialize the gcs */

  gcs_init (si->window->window);

  /* And open a file, if it was specified on the command line. Note
   * that this had to be done after the window was created, otherwise
   * there wouldn't have been a titlebar to set. Also note that
   * a blank score is created whether or not a load was specified.
   * This is done this way because the load could bomb out. */

#ifndef G_OS_WIN32
  if (optind < argc)
    open_for_real (argv[optind], si);
#endif

  /* Similarly, the keymap should be initialized after the
     only once si->window is shown, as it may pop up an advisory
     dialog. */

  si->prefs->the_keymap = init_keymap ();

  /* Set up the signal handler */
#ifndef G_OS_WIN32
  signal (SIGCHLD, sigchld_handler);
#endif

  /* Now launch into the main gtk event loop and we're all set */
  gtk_main ();
  return 0;
}
