/*
 * dmx_display.c
 * Copyright (C) Michael Stickel <michael@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sched.h>

#include "LEDbar.h"

#include <X11/StringDefs.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>

#include <dmx/dmx.h>

#define COLOR(r,g,b) ((r<<16)|(g<<8)|b)

#define BORDER(a)     XtNhSpace, (a), XtNvSpace, (a)
#define BORDERSIZE(a) XtNborderWidth,(a)
#define BORDER0       BORDER(0), BORDERSIZE(0)

#define VERTICAL   XtNorientation, XtorientVertical
#define HORIZONTAL XtNorientation, XtorientHorizontal

static int CHANNELS_PER_ROW=50;
static int ROWS=2;
static int TOTAL_CHANNELS=0;
static int offset=0;

static int           dmx_fd = -1;
static XtAppContext  app_context;
static Widget        *led;

static dmx_t *olddata;
static dmx_t *data;

static void CHECK(void *p)
{
  if(p==NULL)
    {
      fprintf(stderr, "could not create widget/alloc memory\n");
      exit(1);
    }
}

static void compute_dmx_data (XtPointer closure, int *source, XtInputId  *id)
{
  int i;

  if(lseek (dmx_fd, offset, SEEK_SET) < 0)
    {
      fprintf(stderr, "could not seek : %s\n", strerror(errno));
      exit(1);
    }
  if(read (dmx_fd, data, TOTAL_CHANNELS) < 0)
    {
      fprintf(stderr, "could not read : %s\n", strerror(errno));
      exit(1);
    }

  for (i=0; i<TOTAL_CHANNELS; i++)
    if (data[i]!=olddata[i])
      {
	XtVaSetValues(led[i], XtNvalue, data[i]/5, NULL);
	olddata[i] = data[i];
      }
}

int main (int argc, const char **argv)
{
  Widget toplevel;
  Widget mframe;
  Pixel backgroundcolor=COLOR(0x06,0x20,0x30);
  Pixel channelcolor=COLOR(0xcc,0xcc,0xdd);
  int i,row,col;
  int ledwidth=16, ledheight=100;
  const char *arg=NULL;
  const char *dmx_name = DMXdev(&argc, argv);

  dmx_fd = open (dmx_name, O_RDONLY);
  if (dmx_fd<0)
    {
      printf ("can not open %s\n", dmx_name);
      return 1;
    }

  if(DMXgetarg(&argc, argv, "--ledwidth", 1, &arg)==1)
    ledwidth=atoi(arg);
  if(ledwidth<1)
    ledwidth=1;

  if(DMXgetarg(&argc, argv, "--ledheight", 1, &arg)==1)
    ledheight=atoi(arg);
  if(ledheight<1)
    ledheight=1;

  if(DMXgetarg(&argc, argv, "--cols", 1, &arg)==1)
    CHANNELS_PER_ROW=atoi(arg);
  if(CHANNELS_PER_ROW<1)
    CHANNELS_PER_ROW=1;

  if(DMXgetarg(&argc, argv, "--rows", 1, &arg)==1)
    ROWS=atoi(arg);
  if(ROWS<1)
    ROWS=1;

  if(DMXgetarg(&argc, argv, "--offset", 1, &arg)==1)
    offset=atoi(arg);
  if(offset<0)
    offset=0;

  TOTAL_CHANNELS=CHANNELS_PER_ROW * ROWS;
  led=calloc(TOTAL_CHANNELS, sizeof(Widget));
  CHECK(led);
  olddata=calloc(TOTAL_CHANNELS, sizeof(dmx_t));
  CHECK(olddata);
  data=calloc(TOTAL_CHANNELS, sizeof(dmx_t));
  CHECK(data);

  toplevel = XtAppInitialize (&app_context, "DMXDisplay", NULL, 0, &argc, (char**)argv,
                              NULL, NULL,0);
  CHECK(toplevel);

  /* the main frame */
  mframe = XtVaCreateManagedWidget ("mainframe", boxWidgetClass, toplevel,
				    BORDER0,
				    VERTICAL,
                                    XtNbackground, backgroundcolor,
                                    NULL);
  CHECK(mframe);

  if (geteuid() == 0)
    {
      struct sched_param sparam;
      sparam.sched_priority = sched_get_priority_max(SCHED_RR);
      sched_setscheduler(0, SCHED_RR, &sparam);
      setuid(getuid());
      printf("using soft-realtime-scheduling\n");
    }
  else
    printf("NOT using soft-realtime-scheduling\n");

  i=0;
  for (row=0; row<ROWS; row++)
    {
      Widget ledrow = XtVaCreateManagedWidget ("Row", boxWidgetClass, mframe,
					       BORDER0,
					       HORIZONTAL,
					       XtNbackground, backgroundcolor,
					       NULL);
      CHECK(ledrow);

      /* the  LED-Bars */
      for (col=0; col < CHANNELS_PER_ROW; col++)
        {
          Widget w, l;
          char num_str[2];
	  num_str[0]=((i+1 + offset)%10)+'0';
	  num_str[1]=0;

          w = XtVaCreateManagedWidget ("LEDbox", boxWidgetClass, ledrow,
				       BORDER0,
				       VERTICAL,
                                       XtNbackground, backgroundcolor,
                                       NULL);
	  CHECK(w);

	  l = XtVaCreateManagedWidget ("label", labelWidgetClass, w,
				       XtNlabel, num_str,
				       XtNwidth, ledwidth+2,  XtNheight, 15,
				       XtNjustify, XtJustifyCenter,
				       BORDERSIZE(0),
				       XtNbackground, channelcolor,
				       NULL);
	  CHECK(l);

	  led[i] = XtVaCreateManagedWidget ("LED", xmsLEDbarWidgetClass, w,
					    XtNnumboxes, 50,
					    XtNvalue, 0,
					    XtNwidth, ledwidth,  XtNheight, ledheight,
					    XtNbackground, backgroundcolor,
					    BORDERSIZE(0),
					    NULL);
	  CHECK(led[i]);

	  i++;
	}
    }

  XtRealizeWidget (toplevel);
  XtAppAddInput(app_context, dmx_fd, (XtPointer)XtInputReadMask, compute_dmx_data, NULL);
  XtAppMainLoop (app_context);
  return 0;
}
