package org.apache.maven.doxia.module.xdoc;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.maven.doxia.sink.Sink;
import org.apache.maven.doxia.sink.AbstractSinkTest;
import org.apache.maven.doxia.util.HtmlTools;

import java.io.StringWriter;
import java.io.Writer;
import org.apache.maven.doxia.sink.SinkEventAttributeSet;

/**
 * @author <a href="mailto:jason@maven.org">Jason van Zyl</a>
 * @version $Id: XdocSinkTest.java 779565 2009-05-28 11:24:35Z vsiveton $
 * @since 1.0
 */
public class XdocSinkTest
    extends AbstractSinkTest
{
    /** {@inheritDoc} */
    protected String outputExtension()
    {
        return "xml";
    }

    /** {@inheritDoc} */
    protected Sink createSink( Writer writer )
    {
        return new XdocSink( writer, "UTF-8" );
    }

    /** {@inheritDoc} */
    protected boolean isXmlSink()
    {
        return true;
    }

    /** {@inheritDoc} */
    protected String getTitleBlock( String title )
    {
        return "<title>" + title + "</title>";
    }

    /** {@inheritDoc} */
    protected String getAuthorBlock( String author )
    {
        return author;
    }

    /** {@inheritDoc} */
    protected String getDateBlock( String date )
    {
        return date;
    }

    /** {@inheritDoc} */
    protected String getHeadBlock()
    {
        return "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
            + "<document xmlns=\"http://maven.apache.org/XDOC/2.0\" "
            + "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
            + "xsi:schemaLocation=\"http://maven.apache.org/XDOC/2.0 http://maven.apache.org/xsd/xdoc-2.0.xsd\">"
            + "<properties></properties>";
    }

    /** {@inheritDoc} */
    protected String getBodyBlock()
    {
        return "<body></body></document>";
    }

    /** {@inheritDoc} */
    protected String getSectionTitleBlock( String title )
    {
        return title;
    }

    /** {@inheritDoc} */
    protected String getSection1Block( String title )
    {
        return "<section name=\"" + title + "\"></section>";
    }

    /** {@inheritDoc} */
    protected String getSection2Block( String title )
    {
        return "<subsection name=\"" + title + "\"></subsection>";
    }

    /** {@inheritDoc} */
    protected String getSection3Block( String title )
    {
        return "<h4>" + title + "</h4>";
    }

    /** {@inheritDoc} */
    protected String getSection4Block( String title )
    {
        return "<h5>" + title + "</h5>";
    }

    /** {@inheritDoc} */
    protected String getSection5Block( String title )
    {
        return "<h6>" + title + "</h6>";
    }

    /** {@inheritDoc} */
    protected String getListBlock( String item )
    {
        return "<ul><li>" + item + "</li></ul>";
    }

    /** {@inheritDoc} */
    protected String getNumberedListBlock( String item )
    {
        return "<ol style=\"list-style-type: lower-roman\"><li>" + item + "</li></ol>";
    }

    /** {@inheritDoc} */
    protected String getDefinitionListBlock( String definum, String definition )
    {
        return "<dl><dt>" + definum + "</dt><dd>" + definition + "</dd></dl>";
    }

    /** {@inheritDoc} */
    protected String getFigureBlock( String source, String caption )
    {
        return "<img src=\"" + source + "\" alt=\"" + caption + "\" />";
    }

    /** {@inheritDoc} */
    protected String getTableBlock( String cell, String caption )
    {
        return "<table align=\"center\" border=\"0\"><caption>" + caption
                + "</caption><tr valign=\"top\"><td align=\"center\">" + cell + "</td></tr></table>";
    }

    /** {@inheritDoc} */
    protected String getParagraphBlock( String text )
    {
        return "<p>" + text + "</p>";
    }

    /** {@inheritDoc} */
    protected String getVerbatimBlock( String text )
    {
        return "<source>" + text + "</source>";
    }

    /** {@inheritDoc} */
    protected String getHorizontalRuleBlock()
    {
        return "<hr />";
    }

    /** {@inheritDoc} */
    protected String getPageBreakBlock()
    {
        return "<!-- PB -->";
    }

    /** {@inheritDoc} */
    protected String getAnchorBlock( String anchor )
    {
        return "<a name=\"" + anchor + "\">" + anchor + "</a>";
    }

    /** {@inheritDoc} */
    protected String getLinkBlock( String link, String text )
    {
        return "<a href=\"" + link + "\">" + text + "</a>";
    }

    /** {@inheritDoc} */
    protected String getItalicBlock( String text )
    {
        return "<i>" + text + "</i>";
    }

    /** {@inheritDoc} */
    protected String getBoldBlock( String text )
    {
        return "<b>" + text + "</b>";
    }

    /** {@inheritDoc} */
    protected String getMonospacedBlock( String text )
    {
        return "<tt>" + text + "</tt>";
    }

    /** {@inheritDoc} */
    protected String getLineBreakBlock()
    {
        return "<br />";
    }

    /** {@inheritDoc} */
    protected String getNonBreakingSpaceBlock()
    {
        return "&#160;";
    }

    /** {@inheritDoc} */
    protected String getTextBlock( String text )
    {
        // TODO: need to be able to retreive those from outside the sink
        return HtmlTools.escapeHTML( text );
    }

    /** {@inheritDoc} */
    protected String getRawTextBlock( String text )
    {
        return "~,_=,_-,_+,_*,_[,_],_<,_>,_{,_},_\\";
    }

    /**
     * Test verbatim.
     */
    public void testBoxedVerbatim()
    {
        Writer writer =  new StringWriter();
        XdocSink sink = null;

        try
        {
            sink = new XdocSink( writer );

            sink.verbatim( null );
            sink.verbatim_();
            sink.verbatim( SinkEventAttributeSet.BOXED );
            sink.verbatim_();
            sink.verbatim( new SinkEventAttributeSet( new String[] {SinkEventAttributeSet.WIDTH, "20%"} ) );
            sink.verbatim_();
        }
        finally
        {
            sink.close();
        }

        assertEquals( "<pre></pre><source></source><pre width=\"20%\"></pre>", writer.toString() );
    }

    /**
     * Test link.
     */
    public void testLinkWithTarget()
    {
        Writer writer =  new StringWriter();
        XdocSink sink = null;

        try
        {
            sink = new XdocSink( writer );

            sink.link( "name", (String) null );
            sink.link_();
            sink.link( "name", "nirvana" );
            sink.link_();
        }
        finally
        {
            sink.close();
        }

        assertEquals( "<a href=\"name\"></a><a href=\"name\" target=\"nirvana\"></a>", writer.toString() );
    }

    /** {@inheritDoc} */
    protected String getCommentBlock( String text )
    {
        return "<!-- Simple comment with - - - - -->";
    }
}
