/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/*************************************************************
*
*     file:        meanint.c
*
*     Purpose:    Implements routines for integral of mean
*                 curvature over a surface. Computation is 
*                 exact, as the integral can be defined for a
*                 polyhedral surface as edge dihedral angle
*                 times edge length.  Mean curvature is average
*                 of sectional curvatures, not sum.
*
*                 Implemented as general quantity.  Uses info
*                 passed in qinfo structure.
*                 qinfo has vertices: tail,head,left wing, right wing
*                 qinfo has sides: edge,left from tail, right from tail
*     Basic formulas:
* 
*     cos(theta) =              <A,B><A,C> - <A,A><B,C>
*                      _________________________________________________
*                      sqrt(<A,A><B,B>-<A,B>^2)sqrt((<A,A><C,C>-<A,C>^2)
*
*     sin(theta) =             <A x B, C> ||A||
*                      _________________________________________________
*                      sqrt(<A,A><B,B>-<A,B>^2)sqrt((<A,A><C,C>-<A,C>^2)
*
*     tan(theta) =      <A x B, C> ||A||
*                      _______________________
*                      <A,B><A,C> - <A,A><B,C>
*
*     Will use qinfo.ss to hold side products.
*/

#include "include.h"

/*********************************************************************
*
* function: mean_int_init()
*
* purpose:  Check illegalities
*
*/

void mean_int_init(mode,mi)
int mode;
struct method_instance *mi;
{
  if ( web.dimension != 2 )
     kb_error(1594,"mean_curvature_integral method only for 2D facets.\n",RECOVERABLE);

  if ( web.modeltype != LINEAR )
     kb_error(1595,"mean_curvature_integral method only for LINEAR model.\n",RECOVERABLE);


  if ( everything_quantities_flag && mean_curv_int_flag )
     GEN_QUANT(mean_curv_int_quantity_num)->modulus = 
        globals(mean_curvature_param)->value.real;
}

/**************************************************************
*  
*  Function: mean_int_value()
*
*  Purpose:  Computes contribution of one edge. 
*/

REAL mean_int_value(e_info)
struct qinfo *e_info;
{ REAL sinq,cosq; /* proprotional to sin and cos of angle */
  REAL theta; /* the angle */
  REAL len; /* length of side */
  REAL vol; /* volume of parallelpiped of sides */
  int sign = inverted(get_fe_facet(get_edge_fe(e_info->id))) ? 1 : -1;
  
  mat_tsquare(e_info->sides[0],e_info->ss,3,SDIM); /* side products */
  len = sqrt(e_info->ss[0][0]);
  vol = triple_prod(e_info->sides[0][0],e_info->sides[0][1],e_info->sides[0][2]);
  if ( vol == 0.0 ) return vol;
  sinq = len*vol;
  cosq = e_info->ss[0][1]*e_info->ss[0][2] - e_info->ss[0][0]*e_info->ss[1][2];
  theta = atan2(sinq,cosq);
  return sign*theta*len/2;
}

/**************************************************************
*  
*  Function: mean_int_gradient()
*
*  Purpose:  Computes contribution of one edge to gradient. 
*/

REAL mean_int_gradient(e_info)
struct qinfo *e_info;
{ REAL sinq,cosq; /* proprotional to sin and cos of angle */
  REAL dsinq,dcosq; /* derivatives */
  REAL theta;
  REAL dtheta;
  REAL len,dlen;
  REAL vol,dvol;
  int i,j;
  REAL g;
  int sign = inverted(get_fe_facet(get_edge_fe(e_info->id))) ? 1 : -1;
  
  for ( i = 0 ; i < 4 ; i++ )
    for ( j = 0 ; j < SDIM ; j++ )
      e_info->grad[i][j] = 0.0;

  mat_tsquare(e_info->sides[0],e_info->ss,3,SDIM); /* side products */
  len = sqrt(e_info->ss[0][0]);
  vol = triple_prod(e_info->sides[0][0],e_info->sides[0][1],e_info->sides[0][2]);
  if ( vol == 0.0 )  return 0.0;
  sinq = len*vol;
  cosq = e_info->ss[0][1]*e_info->ss[0][2] - e_info->ss[0][0]*e_info->ss[1][2];
  theta = atan2(sinq,cosq);

  /* derivatives of stuff, side by side */
  /* doing partial derivatives in each coordinate */
  for ( i = 0 ; i < SDIM ; i++ )
    { /* main edge */
      dlen = e_info->sides[0][0][i]/len;
      dvol = e_info->sides[0][1][(i+1)%3]*e_info->sides[0][2][(i+2)%3]
                - e_info->sides[0][1][(i+2)%3]*e_info->sides[0][2][(i+1)%3];
      dsinq = dlen*vol + len*dvol;
      dcosq = e_info->sides[0][1][i]*e_info->ss[0][2]
              + e_info->ss[0][1]*e_info->sides[0][2][i]
            - 2*e_info->sides[0][0][i]*e_info->ss[1][2];
      dtheta = (cosq*dsinq-dcosq*sinq)/(sinq*sinq+cosq*cosq);
      g = sign*(dtheta*len + theta*dlen)/2;
      e_info->grad[1][i] += g;
      e_info->grad[0][i] -= g;

      /* left edge */
      dvol = e_info->sides[0][2][(i+1)%3]*e_info->sides[0][0][(i+2)%3]
                - e_info->sides[0][2][(i+2)%3]*e_info->sides[0][0][(i+1)%3];
      dsinq = len*dvol;
      dcosq = e_info->sides[0][0][i]*e_info->ss[0][2]
                    - e_info->ss[0][0]*e_info->sides[0][2][i];
      dtheta = (cosq*dsinq-dcosq*sinq)/(sinq*sinq+cosq*cosq);
      g = sign*dtheta*len/2;
      e_info->grad[2][i] += g;
      e_info->grad[0][i] -= g;

      /* right edge */
      dvol = e_info->sides[0][0][(i+1)%3]*e_info->sides[0][1][(i+2)%3]
                - e_info->sides[0][0][(i+2)%3]*e_info->sides[0][1][(i+1)%3];
      dsinq = len*dvol;
      dcosq = e_info->sides[0][0][i]*e_info->ss[0][1]
                 - e_info->ss[0][0]*e_info->sides[0][1][i];
      dtheta = (cosq*dsinq-dcosq*sinq)/(sinq*sinq+cosq*cosq);
      g = sign*dtheta*len/2;
      e_info->grad[3][i] += g;
      e_info->grad[0][i] -= g;
    }
  return sign*theta*len/2;
}
