/* $Id: main.c,v 1.8 2009-01-27 16:17:18 potyra Exp $ 
 *
 * Copyright (C) 2006-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/time.h>
#include <assert.h>
#include <errno.h>
#include <getopt.h>

#include "launch_client.h"
#include "launch_comm.h"

#include "expect.h"

#include "usb_dev.h"

#include "usb_bridge.h"
#include "usb_bridge_config.h"

#include "sig_usb.h"
#include "umutil.h"
#include "bridge.h"
#include "glue-log.h"

/* ********************* DEFINITIONS ************************** */

/* ****************** GLOBAL VARIABLES ************************ */

static const char *prog_name;
static struct usb_bridge_config_s config;
static const char *config_path;

volatile int loop;			/* continue the event loop */
volatile int reload_bridge_config;	/* bridge configuration has changed, reload */

/* ******************* IMPLEMENTATION ************************* */

static void
sig_sighup(int signum)
{
	assert(signum == SIGHUP);
#if UB_DEBUGMASK & UB_DEBUG_CIM
	faum_log(FAUM_LOG_DEBUG, UB_LOG_TYPE, UB_LOG_NAME,
		"%s: SIGHUP received, preparing to reload bridge configuration\n",
		__FUNCTION__);
#endif
	reload_bridge_config = 1;
}

static void
sig_sigpipe(int signum)
{
	assert(signum == SIGPIPE);
}

/* SIGTERM, SIGINT */
static void
sig_shutdown(int signum)
{
	faum_log(FAUM_LOG_DEBUG, UB_LOG_TYPE, UB_LOG_NAME,
		"%s: shutdown requested\n", prog_name);
	loop = 0;
}

/*
 * load usb bridge configuration
 */
static void
usb_bridge_load_config(const char *path)
{
	int errors = 0;
	struct cfg *cfg;

	cfg = cfg_open(path, USB_BRIDGE_CONFIG, 0, "port.usb");
	if (cfg == NULL) {
		faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
			"%s: opening usbbus bridge config for bridge-usb failed in %s\n",
			prog_name, path);
		errors++;
	} else {
		cim_read_config_from_cfg(cfg, &config.bcc_usb, &errors);
		cfg_close(cfg);
	}

	if (errors) {
		faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
			"%s: disabling bridge-usb with configuration path '%s', "
			"too many errors while reading configuration\n",
			prog_name, path);
		return -1;
	}
	return 0;
}

void
expect_handle_event(fd_set *rfds) {
	int ret;

	if (!FD_ISSET(0, rfds)) {
		return;
	}

	ret = launch_comm_recv(0);
	if (ret <= 0) {
		faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
			"%s: expect channel closed (expect terminated)\n",
			prog_name);
		loop = 0;
		return;
	}

	switch (launch_comm_msgtype()) {
	case LC_TERM:
		faum_log(FAUM_LOG_DEBUG, UB_LOG_TYPE, UB_LOG_NAME,
				"%s: expect told me to terminate\n",
				prog_name);
		loop = 0;
		break;
	default:
		faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
				"%s: unexpected launch_comm message type %d\n",
				prog_name, launch_comm_msgtype());
		break;
	}
}

static void
event_loop(int expect)
{
	fd_set rfds;
	int max_set_fd;
	int ret;

	loop = 1;
	reload_bridge_config = 0;
	while (loop) {
		FD_ZERO(&rfds);

		usb_handle_event_prepare(&rfds);

		if (expect) {
			FD_SET(0, &rfds);
		}

		/* Get highest file descriptor number. */
		/* FIXME: do we need to do this every time? */
		for (max_set_fd = FD_SETSIZE - 1; 0 <= max_set_fd; max_set_fd--) {
			if (FD_ISSET(max_set_fd, &rfds)) {
				break;
			}
		}

		ret = select(max_set_fd + 1, &rfds, NULL, NULL, NULL);

		if (reload_bridge_config) {
			reload_bridge_config = 0;

#if UB_DEBUGMASK & UB_DEBUG_CIM
			faum_log(FAUM_LOG_DEBUG, UB_LOG_TYPE, UB_LOG_NAME,
				"%s: reconfiguring...",
				prog_name);
#endif

			usb_bridge_load_config(config_path);
			usb_reconfig();

#if UB_DEBUGMASK & UB_DEBUG_CIM
			faum_cont(FAUM_LOG_DEBUG, " done.");
#endif

			continue;
		}

		if (ret != -1) {
			usb_handle_event(&rfds);

			if (expect) {
				expect_handle_event(&rfds);
			}
		} else if (errno != EINTR) {
			perror(__FUNCTION__);
		}
	}
}

static void
usb_bridge_create(struct usb_device_selection *devselect)
{
	usb_create(prog_name, &config, devselect);
}

static void
usb_bridge_destroy(void)
{
	usb_destroy();
}

static int
configure(char *filename, char *data)
{
	/* FIXME: use config_path here? */
	if (strcmp(filename, USB_BRIDGE_CONFIG "-0/generics")
	 && strcmp(filename, USB_BRIDGE_CONFIG "-0/port.usb")) {
		faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
			"%s: launcher addressed illegal config file named '%s'\n",
			prog_name, filename);
		return -1;
	}

	return launch_client_create_configfile(filename, data);
}

static struct option long_options[] = {
	{ "help",	0, NULL, 'h' },
	{ "launched",	0, NULL, 'l' },
	{ "config-dir", 1, NULL, 'B' },
	{ "force",	0, NULL, 'f' },
	{ NULL,		0, NULL, 0 }
};

void
cmdline_parse(int argc, char **argv,
	int *expect, int *launched,
	struct usb_device_selection *devselect) {
	int cmd, option_index = 0;

	*launched = *expect = 0;
	
	devselect->addr_selected =
		devselect->id_selected =
		devselect->force = 0;

	while (1) {
		cmd = getopt_long(argc, argv, "hlB:s:d:f", long_options, &option_index);

		/* end of option list? */
		if (cmd == -1) {
			break;
		}

		switch (cmd) {
		case 'h':	/* Print help and exit.  */
			fprintf(stderr,
				"Purpose:\n"
				"  FAUmachine's USB bridge\n"
				"\n"
				"Usage: %s [OPTIONS]...\n"
				"   -h        --help                Print help and exit\n"
				"   -BSTRING  --config-dir=STRING   node configuration directory (default='./')\n"
				"   -l        --launched            used by expect (default=off)\n"
				"   -s [bus]:[devnum]               attach USB device at bus:device (decimal)\n"
				"   -d [vendor]:[product]           attach USB device with vendor:product id (hex)\n"
				"   -f        --force               try to detach kernel driver if interface is unavailable\n",
				argv[0]);
			exit(EXIT_SUCCESS);

		case 'l':
			*expect = 1;
			*launched = 1;
			break;

		case 'B':
			config_path = strdup(optarg);
			if (config_path == NULL) {
				fprintf(stderr, "%s: not enough memory\n", prog_name);
				exit(EXIT_FAILURE);
			}
			break;

		case 's':
			devselect->addr_selected = 1;
			if (sscanf(optarg, "%d:%d",
				&devselect->addr_bus,
				&devselect->addr_dev) != 2) {
				fprintf(stderr, "%s: failed parsing -s parameter %s\n",
					prog_name, optarg);
				exit(EXIT_FAILURE);
			}

			break;

		case 'd':
			devselect->id_selected = 1;
			if (sscanf(optarg, "%x:%x",
				&devselect->id_vendor,
				&devselect->id_product) != 2) {
				fprintf(stderr, "%s: failed parsing -d parameter %s\n",
					prog_name, optarg);
				exit(EXIT_FAILURE);
			}
			break;

		case 'f':
			devselect->force = 1;
			break;

		case '?':
			exit(EXIT_FAILURE);
		}
	}

	if (!devselect->addr_selected
	 && !devselect->id_selected) {
		fprintf(stderr, "%s: either -s or -d must be provided\n",
			prog_name);
		exit(EXIT_FAILURE);
	}
}

int
main(int argc, char **argv) {
	int expect, launched;
	struct usb_device_selection devselect;

	prog_name = argv[0];
	config_path = ".";

	cmdline_parse(argc, argv, &expect, &launched, &devselect);

	if (launched) {
		int ret;

		ret = launch_client_init(&prog_name, 0, 1, configure);
		/* FIXME: check configuration here */
		switch (ret) {
		case LC_CREATE:
			break;
		case LC_TERM:
			exit(0);
		default:
			faum_log(FAUM_LOG_ERROR, UB_LOG_TYPE, UB_LOG_NAME,
				"%s: cannot initialize node from launcher\n",
				prog_name);
			exit(1);
		}

	}

	signal(SIGINT, sig_shutdown);
	signal(SIGTERM, sig_shutdown);
	signal(SIGHUP, sig_sighup);

	cim_client_install_handler(SIGPIPE, sig_sigpipe);

	usb_bridge_load_config(config_path);

	usb_bridge_create(&devselect);

	if (launched) {
		launch_client_state_waiting();
		launch_client_wait_for_start();
	}

	event_loop(expect);

	usb_bridge_destroy();

	return 0;
}
