/*
 * Copyright (C) 2004-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/*
 * Info Structure CD:
 *
 * Power Calibration Area (Diameter 15-22.35mmm):
 *
 * Program Memory Area (Diameter 22.35-23mm):
 *
 * Lead-In (first Lead-In: Diameter 23-25mm):
 *	Size: first lead-in: at least 1min
 *	      next lead-ins: at least 1min
 *	TOC: each entry repeated at least 3 times
 *
 * Program Area (25-...mm):
 *
 * Lead-Out:
 *	Size: first lead-out: at least 1:30min
 *	      next lead-outs: at least 0:30min
 *
 * End (Diameter 59[.5]mm)
 */

#define PMA_SIZE	(60*75)
#define TOC_SIZE	(60*75)

#define CD_BLOCKSIZE	(2352 + 10 + 10)

#define COMP_(x)	media_gen_cd_ ## x

#include <assert.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "glue.h"

#include "conv_cd.h"
#include "conv_zero.h"
#include "sig_boolean.h"
#include "media_gen_cd.h"

struct cpssp {
	char name[1024];
	enum { TYPE_CD_ROM, TYPE_CD_R, TYPE_CD_RW } media_type;
	unsigned int size;
	const char *iso_image;
	const char *cue_sheet;

	void *media;

	/* Faults */
	struct {
		enum type { UNUSED, BLOCK } type;
		int32_t blkno;
		struct sig_boolean *sig;
		unsigned int val;
	} fault[32];
};

static int
COMP_(atip)(void *_cpssp, struct cd_atip *atip)
{
	/* See: http://www.instantinfo.de/detailcdr.php?ID=1337 */

	atip->writing_power = 7; /* 12mw */
	atip->reference_speed = 3; /* 8x */
	atip->unrestricted_use = 1; /* GEMA-Bit, don't care */
	atip->media_type = 1; /* 0: CD-R, 1: CD-RW */
	atip->start_of_leadin_m = 0x61;
	atip->start_of_leadin_s = 0x0f;
	atip->start_of_leadin_f = 0x11;
	atip->last_possible_start_of_leadout_m = 79;
	atip->last_possible_start_of_leadout_s = 57;
	atip->last_possible_start_of_leadout_f = 73;

	return 0;
}

static int
COMP_(read)(
	void *_cpssp,
	int32_t blk,
	uint8_t *data,
	uint8_t *psub,
	uint8_t *qsub
)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t buf[CD_BLOCKSIZE];
	unsigned int i;
	int ret;

	/*
	 * Check for unreadable sectors.
	 */
	for (i = 0; i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]); i++) {
		if (cpssp->fault[i].type == BLOCK
		 && cpssp->fault[i].blkno == blk
		 && cpssp->fault[i].val) {
			/* Block unreadable. */
			return -1;
		}
	}

	blk += PMA_SIZE + TOC_SIZE;

	if (blk < 0 || PMA_SIZE + TOC_SIZE + cpssp->size <= blk) {
		return 0;
	}
	// blk += 1; /* Skip magic. */ /* FIXME */

	ret = storage_read(cpssp->media, buf, CD_BLOCKSIZE,
			(unsigned long long) blk * CD_BLOCKSIZE);
	assert(ret == CD_BLOCKSIZE);

	memcpy(data, buf            , 2352);
	memcpy(psub, buf + 2352     , 10);
	memcpy(qsub, buf + 2352 + 10, 10);

	return qsub[0] != 0;
}

static int
COMP_(write)(
	void *_cpssp,
	int32_t blk,
	const uint8_t *data,
	const uint8_t *psub,
	const uint8_t *qsub
)
{
	struct cpssp *cpssp = _cpssp;
	uint8_t buf[CD_BLOCKSIZE];
	int ret;

	blk += PMA_SIZE + TOC_SIZE;

	if (blk < 0 || PMA_SIZE + TOC_SIZE + cpssp->size <= blk) {
		return 0;
	}
	// blk += 1; /* Skip magic. */ /* FIXME */

	if (cpssp->iso_image
	 || cpssp->cue_sheet) {
		/* CD is read-only. */
		return 0;

	}

	memcpy(buf            , data, 2352);
	memcpy(buf + 2352     , psub, 10);
	memcpy(buf + 2352 + 10, qsub, 10);

	ret = storage_write(cpssp->media, buf, CD_BLOCKSIZE,
			(unsigned long long) blk * CD_BLOCKSIZE);
	assert(ret == CD_BLOCKSIZE);

	return 1;
}

static int
COMP_(port)(struct cpssp *cpssp, const char *port, int32_t *blknop)
{
	char *port2;

	if (strncmp(port, "block_fault", strlen("block_fault")) == 0) {
		port += strlen("block_fault");
	} else {
		return 1;
	}

	if (*port == '/') {
		port++;
	} else {
		return 1;
	}

	*blknop = strtol(port, &port2, 0);
	port = port2;

	if (*port != '\0') {
		return 1;
	}

	return 0;
}

static void
COMP_(fault_block_set)(void *_cpssp, unsigned int i, int val)
{
	struct cpssp *cpssp = _cpssp;

	assert(0 <= val && val <= 1);
	assert(cpssp->fault[i].type == BLOCK);
	cpssp->fault[i].val = val;
}

static void
COMP_(fault_block_set0)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 0, val);
}

static void
COMP_(fault_block_set1)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 1, val);
}

static void
COMP_(fault_block_set2)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 2, val);
}
static void
COMP_(fault_block_set3)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 3, val);
}

static void
COMP_(fault_block_set4)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 4, val);
}

static void
COMP_(fault_block_set5)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 5, val);
}

static void
COMP_(fault_block_set6)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 6, val);
}

static void
COMP_(fault_block_set7)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 7, val);
}

static void
COMP_(fault_block_set8)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 8, val);
}

static void
COMP_(fault_block_set9)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 9, val);
}

static void
COMP_(fault_block_set10)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 10, val);
}

static void
COMP_(fault_block_set11)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 11, val);
}

static void
COMP_(fault_block_set12)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 12, val);
}

static void
COMP_(fault_block_set13)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 13, val);
}

static void
COMP_(fault_block_set14)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 14, val);
}

static void
COMP_(fault_block_set15)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 15, val);
}

static void
COMP_(fault_block_set16)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 16, val);
}

static void
COMP_(fault_block_set17)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 17, val);
}

static void
COMP_(fault_block_set18)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 18, val);
}

static void
COMP_(fault_block_set19)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 19, val);
}

static void
COMP_(fault_block_set20)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 20, val);
}

static void
COMP_(fault_block_set21)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 21, val);
}

static void
COMP_(fault_block_set22)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 22, val);
}

static void
COMP_(fault_block_set23)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 23, val);
}

static void
COMP_(fault_block_set24)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 24, val);
}

static void
COMP_(fault_block_set25)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 25, val);
}

static void
COMP_(fault_block_set26)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 26, val);
}

static void
COMP_(fault_block_set27)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 27, val);
}

static void
COMP_(fault_block_set28)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 28, val);
}

static void
COMP_(fault_block_set29)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 29, val);
}

static void
COMP_(fault_block_set30)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 30, val);
}

static void
COMP_(fault_block_set31)(void *_cpssp, unsigned int val)
{
	COMP_(fault_block_set)(_cpssp, 31, val);
}

static void
COMP_(connect)(void *_cpssp, const char *port, void *sig)
{
	static const struct sig_boolean_funcs fault_block_funcs[] = {
		{ .set = COMP_(fault_block_set0), },
		{ .set = COMP_(fault_block_set1), },
		{ .set = COMP_(fault_block_set2), },
		{ .set = COMP_(fault_block_set3), },
		{ .set = COMP_(fault_block_set4), },
		{ .set = COMP_(fault_block_set5), },
		{ .set = COMP_(fault_block_set6), },
		{ .set = COMP_(fault_block_set7), },
		{ .set = COMP_(fault_block_set8), },
		{ .set = COMP_(fault_block_set9), },
		{ .set = COMP_(fault_block_set10), },
		{ .set = COMP_(fault_block_set11), },
		{ .set = COMP_(fault_block_set12), },
		{ .set = COMP_(fault_block_set13), },
		{ .set = COMP_(fault_block_set14), },
		{ .set = COMP_(fault_block_set15), },
		{ .set = COMP_(fault_block_set16), },
		{ .set = COMP_(fault_block_set17), },
		{ .set = COMP_(fault_block_set18), },
		{ .set = COMP_(fault_block_set19), },
		{ .set = COMP_(fault_block_set20), },
		{ .set = COMP_(fault_block_set21), },
		{ .set = COMP_(fault_block_set22), },
		{ .set = COMP_(fault_block_set23), },
		{ .set = COMP_(fault_block_set24), },
		{ .set = COMP_(fault_block_set25), },
		{ .set = COMP_(fault_block_set26), },
		{ .set = COMP_(fault_block_set27), },
		{ .set = COMP_(fault_block_set28), },
		{ .set = COMP_(fault_block_set29), },
		{ .set = COMP_(fault_block_set30), },
		{ .set = COMP_(fault_block_set31), },
	};
	struct cpssp *cpssp = _cpssp;
	int32_t blkno;
	unsigned int i;

	assert(sizeof(fault_block_funcs) / sizeof(fault_block_funcs[0])
			== sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));

	if (COMP_(port)(cpssp, port, &blkno)) {
		return;
	}

	/* Lookup unused entry. */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));

		if (cpssp->fault[i].type == UNUSED) {
			break;
		}
	}

	/* Add entry. */
	cpssp->fault[i].type = BLOCK;
	cpssp->fault[i].blkno = blkno;
	cpssp->fault[i].val = 0;

	cpssp->fault[i].sig = sig;

	sig_boolean_connect_in(cpssp->fault[i].sig, cpssp,
			&fault_block_funcs[i]);
}

static void
COMP_(disconnect)(void *_cpssp, const char *port)
{
	struct cpssp *cpssp = _cpssp;
	int32_t blkno;
	unsigned int i;

	if (COMP_(port)(cpssp, port, &blkno)) {
		return;
	}

	/* Lookup entry. */
	for (i = 0; ; i++) {
		assert(i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]));

		if (cpssp->fault[i].blkno == blkno) {
			break;
		}
	}

	// sig_boolean_disconnect_in(cpssp->fault[i].sig, cpssp);

	/* Disable entry. */
	cpssp->fault[i].type = UNUSED;
}

void *
COMP_(create)(
	const char *name,
	const char *media,
	const char *size,
	const char *iso,
	const char *cue,
	struct sig_manage *port_manage,
	struct sig_magneto_optical *port_connect
)
{
	static const struct sig_manage_funcs manage_funcs = {
		.connect = COMP_(connect),
		.disconnect = COMP_(disconnect),
	};
	static const struct sig_magneto_optical_funcs connect_funcs = {
		.atip = COMP_(atip),
		.read = COMP_(read),
		.write = COMP_(write),
	};
	struct cpssp *cpssp;
	const char *image;
	void *(*conv_open)(const char *, uint64_t);
	int (*conv_close)(void *);
	int64_t (*conv_read)(void *, void *, uint64_t, uint64_t);
	char path[1024];
	// char buf[CD_BLOCKSIZE];
	unsigned int i;

	if (! media) media = "CD-RW";
	if (! size) size = "360000"; /* 80*60*75 */

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	system_name_push(name);

	/* name */
	assert(strlen(system_path()) < sizeof(cpssp->name));
	strcpy(cpssp->name, system_path());

	/* media */
	if (strcmp(media, "CD-ROM") == 0) {
		cpssp->media_type = TYPE_CD_ROM;
	} else if (strcmp(media, "CD-R") == 0) {
		cpssp->media_type = TYPE_CD_R;
	} else if (strcmp(media, "CD-RW") == 0) {
		cpssp->media_type = TYPE_CD_RW;
	} else {
		assert(0); /* FIXME */
	}

	/* size */
	cpssp->size = atoi(size);

	/* contents */
	cpssp->iso_image = iso;
	cpssp->cue_sheet = cue;

	if (iso) {
		image = iso;
		conv_open = conv_cd_from_iso_open;
		conv_close = conv_cd_from_iso_close;
		conv_read = conv_cd_from_iso_read;
	} else if (cue) {
		image = cue;
		conv_open = conv_cd_from_cue_open;
		conv_close = conv_cd_from_cue_close;
		conv_read = conv_cd_from_cue_read;
	} else {
		image = NULL;
		conv_open = conv_zero_open;
		conv_close = conv_zero_close;
		conv_read = conv_zero_read;
	}

	assert(strlen(system_path()) + strlen(".media") < sizeof(path));
	sprintf(path, "%s.media", system_path());
	cpssp->media = storage_create(path,
			(uint64_t) (/*1 +*/ PMA_SIZE + TOC_SIZE + cpssp->size) * CD_BLOCKSIZE,
			image, conv_open, conv_close, conv_read);
	assert(cpssp->media);

	// memset(buf, 0, sizeof(buf)); FIXME
	// strcpy(buf, "FAUmachine CD/DVD Image\n");
	// storage_write(&cpssp->media, buf, sizeof(buf), 0ULL);

	for (i = 0; i < sizeof(cpssp->fault) / sizeof(cpssp->fault[0]); i++) {
		cpssp->fault[i].type = UNUSED;
	}

	/* Call */
	sig_manage_connect(port_manage, cpssp, &manage_funcs);
	sig_magneto_optical_connect(port_connect, cpssp, &connect_funcs);

	system_name_pop();

	return cpssp;
}

void
COMP_(destroy)(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;
	int ret;

	ret = storage_destroy(cpssp->media);
	assert(0 <= ret);

	shm_free(cpssp);
}

void
COMP_(suspend)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fComp);
}

void
COMP_(resume)(void *_cpssp, FILE *fComp)
{
	struct cpssp *cpssp = _cpssp;

	void *savemedia = cpssp->media;
	const char *saveiso_image = cpssp->iso_image;
	const char *savecue_sheet = cpssp->cue_sheet;

	generic_resume(cpssp, sizeof(*cpssp), fComp);

	cpssp->media = savemedia;
	cpssp->iso_image = saveiso_image;
	cpssp->cue_sheet = savecue_sheet;

	fprintf(stdout, "\t\t\ttoDo: Handle faults\n", 0);
}
