/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#define INCLUDE
#include "arch_potentiometer.c"
#undef INCLUDE

#include "glue.h"
#include "oscilloscope.h"

#define VALUE_BUFFER_SIZE 4096

#define SCREEN_WIDTH		240
#define SCREEN_HEIGHT		192
#define SCREEN_ZERO_LINE 	92
#define GRID_STEP		24
#define RING_BUFFER_SIZE 	8000
#define HOLD_OFF_TICKS 		240

/*
	TODO if trigger too high -> auto mode -> unsteady display
	TODO indicate trigger cursor
	TODO display current values
	TODO adjust screen size and gridstep values
*/

struct cpssp{
	struct sig_std_logic *port_b;
	struct sig_opt_rgb *port_monitor;
	double voltage_value_buffer[VALUE_BUFFER_SIZE];
	int ring_buffer[RING_BUFFER_SIZE];
	int ring_buffer_current_pos;
	int ring_buffer_trigger_pos;
	int trigger_sample_count;
	bool trigger_passed;
	bool in_hold_off;
	int hold_off_tick_count;
	int vertical_offset;
	double current_value_buffer[VALUE_BUFFER_SIZE];
	int current_buffer_pos;
	bool pixel_buffer_voltage[SCREEN_HEIGHT][SCREEN_WIDTH];
	bool pixel_buffer_current[SCREEN_HEIGHT][SCREEN_WIDTH];
	int current_voltage;
	int current_current;
	int current_screen_x;
	int trigger_value;
	int voltage_amplification;
	int voltage_time_scale;
	bool reset;
	void *sec_div_poti;
	void *volts_div_poti;
	void *trigger_poti;
	void *vert_pos_poti;
	int poti_trigger_val;
	int poti_vertical_pos_val;

#define STATE

#define NAME		poti_trigger
#define NAME_(x)	poti_trigger_ ## x
#define SNAME		"poti_trigger"
#include "arch_potentiometer.c"
#undef SNAME
#undef NAME_
#undef NAME

#define NAME		poti_volt_div
#define	NAME_(x)	poti_volt_div_ ## x
#define SNAME 		"poti_volt_div"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#define NAME		poti_sec_div
#define	NAME_(x)	poti_sec_div_ ## x
#define SNAME 		"poti_sec_div"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#define NAME		poti_vertical_pos
#define NAME_(x)	poti_vertical_pos_ ## x
#define SNAME		"poti_vertical_pos"
#include "arch_potentiometer.c"
#undef SNAME
#undef NAME_
#undef NAME
#undef STATE
};

static void
update_pixel_buffer(struct cpssp *cpssp)
{
	int row, column, start_pos, pos;
	start_pos = cpssp->ring_buffer_trigger_pos - (SCREEN_WIDTH / 2);
	if (start_pos < 0) {
		start_pos = RING_BUFFER_SIZE + start_pos;
	}
	start_pos = start_pos % RING_BUFFER_SIZE;

	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = 0; column < SCREEN_WIDTH; column++) {
			pos = (start_pos + column) % RING_BUFFER_SIZE;
			if (cpssp->ring_buffer[pos] == row) {
				cpssp->pixel_buffer_voltage[row][column] = true;
			} else {
				cpssp->pixel_buffer_voltage[row][column] = false;
			}
		}
	}
}

static void
draw_pixels(void *_cpssp)
{
	int row, column;
	struct cpssp *cpssp;

	cpssp = _cpssp;
	sig_opt_rgb_size_set(cpssp->port_monitor, cpssp, SCREEN_WIDTH, SCREEN_HEIGHT);
	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = 0; column < SCREEN_WIDTH; column++) {
			if (cpssp->pixel_buffer_voltage[row][column] == true) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 255, 255, 255);
			} else if (column % GRID_STEP == 0 || row % GRID_STEP == 0) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 100, 100, 100);
			} else {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 0, 0, 0);
			}
		}
	}
	sig_opt_rgb_sync(cpssp->port_monitor, cpssp);
	time_call_after(TIME_HZ, draw_pixels, cpssp);
}

static void
on_timer_tick(void *_cpssp)
{
	struct cpssp *cpssp;
	int on_screen_value_y_voltage;

	cpssp = _cpssp;
	on_screen_value_y_voltage = SCREEN_ZERO_LINE - (((cpssp->current_voltage / 100) * cpssp->voltage_amplification)) + cpssp->vertical_offset;
	cpssp->ring_buffer[cpssp->ring_buffer_current_pos] = on_screen_value_y_voltage;
	cpssp->ring_buffer_current_pos = (cpssp->ring_buffer_current_pos + 1) % RING_BUFFER_SIZE;

	if (cpssp->in_hold_off == true) {
		cpssp->hold_off_tick_count++;
		if (cpssp->hold_off_tick_count == HOLD_OFF_TICKS) {
			cpssp->in_hold_off = false;
			cpssp->hold_off_tick_count = 0;
		}
	}
	if (cpssp->trigger_passed == true) {
		cpssp->trigger_sample_count++;
		if (cpssp->trigger_sample_count == SCREEN_WIDTH) {
			cpssp->trigger_sample_count = 0;
			cpssp->trigger_passed = false;
			update_pixel_buffer(cpssp);
			/* draw_pixels_triggered(cpssp); */
		}
	}
	time_call_after(TIME_HZ / cpssp->voltage_time_scale , &on_timer_tick, cpssp);
}

static void
on_voltage_changed(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp;
	int mV, mA;

	if (val == -1) {
		return;
	}
	cpssp = _cpssp;
	mV = SIG_mV(val);
	/* mA = SIG_mA(val); actually correct, but no correct current is sent at the moment */
	mA = mV * 2;
	if (cpssp->current_buffer_pos < VALUE_BUFFER_SIZE) {
		cpssp->voltage_value_buffer[cpssp->current_buffer_pos++] = mV;
		cpssp->current_value_buffer[cpssp->current_buffer_pos++] = mA;
	} else {
		/* TODO tbd */
	}

	if (cpssp->trigger_value > cpssp->current_voltage
	 && mV >= cpssp->trigger_value
	 && cpssp->in_hold_off == false
	 && cpssp->trigger_passed == false) {
		/* rising edge */
		cpssp->trigger_passed = true;
		cpssp->ring_buffer_trigger_pos = cpssp->ring_buffer_current_pos;
	}
	cpssp->current_voltage = mV;
	cpssp->current_current = mA;
}

static void
poti_trigger_mid_out_set(struct cpssp *cpssp, unsigned int val)
{
	int mV;

	if (val == -1) {
		return;
	}

	mV = SIG_mV(val);
	if (mV < 503) {
		mV = 503; /* test. creates image instantly */
	}
	cpssp->trigger_value = mV;
	/* fprintf(stderr, "setting trigger to %d\n", cpssp->trigger_value); */
	cpssp->in_hold_off = true;
}

static void
poti_volt_div_mid_out_set(struct cpssp *cpssp, unsigned int val)
{
	int mV;

	mV = SIG_mV(val) * 0.002;
	if (mV < 1) {
		mV = 1;
	}
	cpssp->voltage_amplification = mV;
	/* fprintf(stderr, "amp %d\n", mV); */
	cpssp->in_hold_off = true;
}

static void
poti_sec_div_mid_out_set(struct cpssp *cpssp, unsigned int val)
{
	int mV;

	mV = SIG_mV(val) * 0.2;
	if (mV < 100) {
		mV = 100; /* temporary fix to avoid animation stop */
	}
	cpssp->voltage_time_scale = mV;
	/* fprintf(stderr, "time scale %d\n", mV); */
	cpssp->in_hold_off = true;
}

static void
poti_vertical_pos_mid_out_set(struct cpssp *cpssp, unsigned int val)
{
	int mV;

	mV = SIG_mV(val) / 100;
	cpssp->vertical_offset = mV;
	/* fprintf(stderr, "vertical offset %d\n", mV); */
	cpssp->in_hold_off = true;
}

#define BEHAVIOR

#define NAME		poti_trigger
#define	NAME_(x)	poti_trigger_ ## x
#define SNAME 		"poti_trigger"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#define NAME		poti_volt_div
#define	NAME_(x)	poti_volt_div_ ## x
#define SNAME 		"poti_volt_div"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#define NAME		poti_sec_div
#define	NAME_(x)	poti_sec_div_ ## x
#define SNAME 		"poti_sec_div"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#define NAME		poti_vertical_pos
#define	NAME_(x)	poti_vertical_pos_ ## x
#define SNAME 		"poti_vertical_pos"
#include "arch_potentiometer.c"
#undef	SNAME
#undef	NAME_
#undef	NAME

#undef BEHAVIOR

static void
oscilloscope_trigger_adj_set(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_trigger_adj_set(cpssp, val);
}

static void
oscilloscope_volt_div_adj_set(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_volt_div_adj_set(cpssp, val);
}

static void
oscilloscope_sec_div_adj_set(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_sec_div_adj_set(cpssp, val);
}

static void
oscilloscope_vertical_pos_adj_set(void *_cpssp, int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_vertical_pos_adj_set(cpssp, val);
}

static void
oscilloscope_gnd_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_trigger_right_in_set(cpssp, val);
	poti_volt_div_right_in_set(cpssp, val);
	poti_sec_div_right_in_set(cpssp, val);
	poti_vertical_pos_right_in_set(cpssp, val);
}

static void
oscilloscope_vcc_set(void *_cpssp, unsigned int val)
{
	struct cpssp *cpssp = _cpssp;

	poti_trigger_left_in_set(cpssp, val);
	poti_volt_div_left_in_set(cpssp, val);
	poti_sec_div_left_in_set(cpssp, val);
	poti_vertical_pos_left_in_set(cpssp, val);
}

void *
oscilloscope_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_a,
	struct sig_std_logic *port_b,
	struct sig_opt_rgb *port_monitor,
	struct sig_std_logic *port_gnd,
	struct sig_std_logic *port_vcc,
	struct sig_integer *port_adj_c,
	struct sig_integer *port_adj_d,
	struct sig_integer *port_adj_e,
	struct sig_integer *port_adj_f
)
{
	static const struct sig_std_logic_funcs gnd_func = {
		.std_logic_set = oscilloscope_gnd_set,
	};
	static const struct sig_std_logic_funcs vcc_func = {
		.std_logic_set = oscilloscope_vcc_set,
	};
	static const struct sig_std_logic_funcs port_func = {
		.std_logic_set = on_voltage_changed,
	};
	static const struct sig_integer_funcs trigger_adj_funcs = {
		.set = oscilloscope_trigger_adj_set,
	};
	static const struct sig_integer_funcs volt_div_adj_funcs = {
		.set = oscilloscope_volt_div_adj_set,
	};
	static const struct sig_integer_funcs sec_div_adj_funcs = {
		.set = oscilloscope_sec_div_adj_set,
	};
	static const struct sig_integer_funcs vertical_pos_adj_funcs = {
		.set = oscilloscope_vertical_pos_adj_set,
	};
	struct cpssp *cpssp;

	cpssp = malloc(sizeof(*cpssp));
	assert(cpssp);

	poti_trigger_create(cpssp);
	poti_volt_div_create(cpssp);
	poti_sec_div_create(cpssp);
	poti_vertical_pos_create(cpssp);

	cpssp->port_b = port_b;
	cpssp->port_monitor = port_monitor;
	cpssp->current_buffer_pos = 0;
	cpssp->reset = true;
	cpssp->voltage_time_scale = 100;
	/* start ring buffer experiment */
	cpssp->in_hold_off = true;
	cpssp->trigger_passed = false;
	cpssp->ring_buffer_current_pos = 0;
	cpssp->ring_buffer_trigger_pos = 0;
	/* end experiment ring buffer experiment */

	/* Out */
	sig_std_logic_connect_out(cpssp->port_b, cpssp, SIG_STD_LOGIC_Z);

	/* In */
	sig_std_logic_connect_in(port_gnd, cpssp, &gnd_func);
	sig_std_logic_connect_in(port_vcc, cpssp, &vcc_func);
	sig_std_logic_connect_in(port_a, cpssp, &port_func);
	sig_integer_connect_in(port_adj_c, cpssp, &trigger_adj_funcs);
	sig_integer_connect_in(port_adj_d, cpssp, &volt_div_adj_funcs);
	sig_integer_connect_in(port_adj_e, cpssp, &sec_div_adj_funcs);
	sig_integer_connect_in(port_adj_f, cpssp, &vertical_pos_adj_funcs);

	time_call_after(TIME_HZ / 10, on_timer_tick, cpssp);
	/* time_call_after(TIME_HZ, draw_pixels_triggered, cpssp); */
	time_call_after(TIME_HZ, draw_pixels, cpssp);

	return cpssp;
}


void
oscilloscope_destroy(void *_cpssp)
{
	poti_trigger_destroy(_cpssp);
	poti_volt_div_destroy(_cpssp);
	poti_sec_div_destroy(_cpssp);
	poti_vertical_pos_destroy(_cpssp);
	struct cpssp *cpssp = _cpssp;
	free(cpssp);
}

void
oscilloscope_suspend(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
oscilloscope_resume(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;

	generic_resume(cpssp, sizeof(*cpssp), fp);
}

#if 0
/* draws black pixels, grid lines and values  */
/* probably obsolete */
void
draw_pixels_trigger(struct cpssp *cpssp)
{
	int row, column;
	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = 0; column < SCREEN_WIDTH; column++) {
			if (cpssp->voltage_ring_buffer[column] == row) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, cpssp->voltage_ring_buffer[column], 255, 0, 0);
			} else if (column % GRID_STEP == 0 || row % GRID_STEP == 0) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 1, 1, 1);
			} else {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 0, 0, 0);
			}
		}
	}
}
#endif

#if 0

/* updates graphical display only when the image is drawn completely */
/* probably obsolete */

static void
on_timer_tick(void *_cpssp)
{
	struct cpssp *cpssp;
	int on_screen_value_y_voltage;
	int on_screen_value_y_current;

	cpssp = _cpssp;

	/* on_screen_value_y_voltage = SCREEN_ZERO_LINE - ((cpssp->current_voltage / 100) * cpssp->voltage_amplification);  */
	on_screen_value_y_voltage = SCREEN_ZERO_LINE - ((cpssp->current_voltage / 100));
	on_screen_value_y_current = SCREEN_ZERO_LINE - (cpssp->current_current / 100);

	if (on_screen_value_y_voltage < 0) {
		/* do nothing */
	} else {
		cpssp->voltage_ring_buffer[cpssp->voltage_current_ring_buffer_pos] = on_screen_value_y_voltage;
	}

	if (cpssp->voltage_current_ring_buffer_pos == SCREEN_WIDTH - 1) {
		/* TODO draw values or wait for screen width to 'fill' */
		/* draw_pixels_trigger(cpssp); */
		cpssp->voltage_current_ring_buffer_pos = 0;
	} else {
		cpssp->voltage_current_ring_buffer_pos++;
	}

	if (cpssp->voltage_trigger_passed == true) {
		cpssp->voltage_trigger_sample_count++;
		if (cpssp->voltage_trigger_sample_count == (SCREEN_WIDTH / 2)) {
			draw_pixels_trigger(cpssp);
			cpssp->voltage_trigger_sample_count = 0;
			cpssp->voltage_trigger_passed = false;
		}
	}
	time_call_after(TIME_HZ / cpssp->voltage_time_scale, &on_timer_tick, cpssp);
}

#endif

#if 0

/* creates keithley-like graphical update */

static void
on_timer_tick(void *_cpssp)
{
	struct cpssp *cpssp;
	int on_screen_value_x;
	int on_screen_value_y_voltage;
	int on_screen_value_y_current;

	cpssp = _cpssp;

	if (cpssp->current_voltage < cpssp->trigger_value ) {
		cpssp->current_screen_x++; /* correct? */
		time_call_after(TIME_HZ / 100, &on_timer_tick, cpssp);
		return;
	}
	if (SCREEN_WIDTH <= cpssp->current_screen_x) {
		cpssp->current_screen_x = 0; /* reset cursor */
		cpssp->reset = true;
		draw_pixels(cpssp);
		wipe_screen(cpssp);
	} else if (cpssp->current_screen_x % GRID_STEP == 0) {
		draw_pixels(cpssp);
	}

	on_screen_value_x = cpssp->current_screen_x;
	on_screen_value_y_voltage = SCREEN_ZERO_LINE - ((cpssp->current_voltage / 100) * cpssp->voltage_amplification );
	on_screen_value_y_current = SCREEN_ZERO_LINE - (cpssp->current_current / 100);
	if (on_screen_value_y_voltage < 0) {
		/* do nothing */
	} else {
		cpssp->pixel_buffer_voltage[on_screen_value_y_voltage][on_screen_value_x] = true;
	}
	cpssp->pixel_buffer_current[on_screen_value_y_current][on_screen_value_x] = true;
	cpssp->current_screen_x++;
	time_call_after(TIME_HZ / cpssp->voltage_time_scale, &on_timer_tick, cpssp);
}

#endif

#if 0
/* draws value pixels */

void
draw_pixels(struct cpssp *cpssp)
{
	int row, column, startColumn, endColumn;
	if (cpssp->reset == true) {
		startColumn = 0;
		endColumn = SCREEN_WIDTH;
		cpssp->reset = false;
	} else {
		startColumn = cpssp->current_screen_x - GRID_STEP;
		endColumn = cpssp->current_screen_x;
	}
	startColumn = 0;
	endColumn = SCREEN_WIDTH;
	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = startColumn; column < endColumn; column++) {
			if (cpssp->pixel_buffer_voltage[row][column] == true) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 255, 0, 0);
			}
			if (cpssp->pixel_buffer_current[row][column] == true) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 0, 0, 255);
			}
		}
	}
}

/* draws black pixels and grid lines */

void
wipe_screen(struct cpssp *cpssp)
{
	int row, column;
	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = 0; column < SCREEN_WIDTH; column++) {
			cpssp->pixel_buffer_voltage[row][column] = false;
			cpssp->pixel_buffer_current[row][column] = false;
			if (column % GRID_STEP == 0 || row % GRID_STEP == 0) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 1, 1, 1);
			} else {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 0, 0, 0);
			}
		}
	}
}

void
draw_pixels_triggered(struct cpssp *cpssp)
{
	int row, column, start_pos, pos;
	sig_opt_rgb_size_set(cpssp->port_monitor, cpssp, SCREEN_WIDTH, SCREEN_HEIGHT);
	start_pos = cpssp->ring_buffer_trigger_pos - (SCREEN_WIDTH / 2);
	if (start_pos < 0) {
		start_pos = RING_BUFFER_SIZE + start_pos;
	}
	start_pos = start_pos % RING_BUFFER_SIZE;

	for(row = 0; row < SCREEN_HEIGHT; row++) {
		for(column = 0; column < SCREEN_WIDTH; column++) {
			pos = (start_pos + column) % RING_BUFFER_SIZE;
			if (cpssp->ring_buffer[pos] == row) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, cpssp->ring_buffer[pos], 255, 255, 255);
			} else if (column % GRID_STEP == 0 || row % GRID_STEP == 0) {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 100, 100, 100);
			} else {
				sig_opt_rgb_pixel_set(cpssp->port_monitor, cpssp, column, row, 0, 0, 0);
			}
		}
	}

	sig_opt_rgb_sync(cpssp->port_monitor, cpssp);
	/* time_call_after(TIME_HZ, draw_pixels_triggered, cpssp); */
}
#endif
