/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  File-Roller
 *
 *  Copyright (C) 2001 The Free Software Foundation, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include "file-utils.h"


typedef struct {
	gchar * start_from;
	GnomeVFSURI * uri;
	GList * files;
	GnomeVFSDirectoryFilter * filter;
} WCSearchData;



static WCSearchData *
wc_search_data_new (const gchar *directory)
{
	WCSearchData *data;
	gchar *escaped;

	data = g_new (WCSearchData, 1);

	data->start_from = g_strdup (directory);

	escaped = gnome_vfs_escape_path_string (directory);
	data->uri = gnome_vfs_uri_new (escaped);
	g_free (escaped);

	data->files = NULL;
	data->filter = NULL;

	return data;
}


static void
wc_search_data_free (WCSearchData *data)
{
	if (data == NULL)
		return;

	if (data->start_from) {
		g_free (data->start_from);
		data->start_from = NULL;
	}
	
	if (data->uri != NULL)
		gnome_vfs_uri_unref (data->uri);

	g_free (data);
}


static gboolean 
wc_visit_cb (const gchar * rel_path,
	     GnomeVFSFileInfo * info,
	     gboolean recursing_will_loop,
	     gpointer callback_data,
	     gboolean * recurse)
{
	WCSearchData *data = callback_data;

	if (gnome_vfs_directory_filter_apply (data->filter, info)) {
		data->files = g_list_prepend (data->files, 
					      g_strdup (rel_path));
	}

	*recurse = ! recursing_will_loop;

	return TRUE;
}


GList *
get_wildcard_file_list (gchar *directory, 
			const gchar *filter_pattern, 
			gboolean recursive,
			gboolean follow_links,
			gboolean same_fs,
			gboolean no_backup_files,
			gboolean no_dot_files,
			gboolean ignorecase)
{
	WCSearchData *                 data;
	GnomeVFSDirectoryFilterOptions filter_options;
	GnomeVFSDirectoryFilter *      dir_filter;
	GnomeVFSDirectoryFilterOptions dir_filter_options;
	GnomeVFSResult                 result;
	GnomeVFSFileInfoOptions        info_options; 
	GnomeVFSDirectoryVisitOptions  visit_options;
	GList *                        list;

	data = wc_search_data_new (directory);

	/* file filter */

	filter_options = (GNOME_VFS_DIRECTORY_FILTER_DEFAULT
			  | GNOME_VFS_DIRECTORY_FILTER_NODIRS
			  | GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR
			  | GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR);
	if (no_backup_files)
		filter_options |= GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES;
	if (no_dot_files)
		filter_options |= GNOME_VFS_DIRECTORY_FILTER_NODOTFILES;
	if (ignorecase)
		filter_options |= GNOME_VFS_DIRECTORY_FILTER_IGNORECASE;

	data->filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN, filter_options, filter_pattern);

	/* dir filter */

	dir_filter_options = (GNOME_VFS_DIRECTORY_FILTER_DEFAULT 
			      | GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR 
			      | GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR);
	if (! recursive)
		dir_filter_options |= GNOME_VFS_DIRECTORY_FILTER_NODIRS;
	
	dir_filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN, dir_filter_options, "*");

	/* info options. */

	info_options = GNOME_VFS_FILE_INFO_DEFAULT;
	if (follow_links)
		info_options |= GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	visit_options =  GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK;
	if (same_fs)
		visit_options |= GNOME_VFS_DIRECTORY_VISIT_SAMEFS;

	result = gnome_vfs_directory_visit_uri (
		data->uri,
		info_options,
		dir_filter,
		visit_options,
		wc_visit_cb,
		data);

	gnome_vfs_directory_filter_destroy (dir_filter);
	gnome_vfs_directory_filter_destroy (data->filter);
	list = data->files;
	wc_search_data_free (data);

	return list;
}


/* -- get_directory_file_list -- */


typedef struct {
	gchar * directory;
	gchar * base_dir;
	GnomeVFSURI * uri;
	GList * files;
	GnomeVFSDirectoryFilter * filter;
} DirSearchData;



static DirSearchData *
dir_search_data_new (const gchar *base_dir,
		     const gchar *directory)
{
	DirSearchData *data;
	gchar *escaped;
	gchar *full_path;

	data = g_new (DirSearchData, 1);

	data->directory = g_strdup (directory);
	data->base_dir = g_strdup (base_dir);

	full_path = g_strconcat (base_dir, "/", directory, NULL);
	escaped = gnome_vfs_escape_path_string (full_path);
	data->uri = gnome_vfs_uri_new (escaped);
	g_free (escaped);
	g_free (full_path);

	data->files = NULL;
	data->filter = NULL;

	return data;
}


static void
dir_search_data_free (DirSearchData *data)
{
	if (data == NULL)
		return;

	if (data->directory) {
		g_free (data->directory);
		data->directory = NULL;
	}

	if (data->base_dir) {
		g_free (data->base_dir);
		data->base_dir = NULL;
	}
	
	if (data->uri != NULL)
		gnome_vfs_uri_unref (data->uri);

	g_free (data);
}


static gboolean 
dir_visit_cb (const gchar * rel_path,
	      GnomeVFSFileInfo * info,
	      gboolean recursing_will_loop,
	      gpointer callback_data,
	      gboolean * recurse)
{
	DirSearchData *data = callback_data;

	if (gnome_vfs_directory_filter_apply (data->filter, info)) 
		data->files = g_list_prepend (data->files, 
					      g_strconcat (data->directory,
							   "/",
							   rel_path,
							   NULL));

	*recurse = ! recursing_will_loop;

	return TRUE;
}


GList *
get_directory_file_list (const gchar *directory,
			 const gchar *base_dir)
{
	DirSearchData *                data;
	GnomeVFSDirectoryFilterOptions filter_options;
	GnomeVFSDirectoryFilter *      dir_filter;
	GnomeVFSDirectoryFilterOptions dir_filter_options;
	GnomeVFSResult                 result;
	GnomeVFSFileInfoOptions        info_options; 
	GnomeVFSDirectoryVisitOptions  visit_options;
	GList *                        list;

	data = dir_search_data_new (base_dir, directory);

	/* file filter */

	filter_options = (GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR
			  | GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR
			  | GNOME_VFS_DIRECTORY_FILTER_NODIRS
			  | GNOME_VFS_DIRECTORY_FILTER_NODOTFILES
			  | GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES);
	
	data->filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN, filter_options, "*");

	/* dir filter */

	dir_filter_options = (GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR 
			      | GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR);

	dir_filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN, dir_filter_options, "*");

	/* options. */

	info_options = GNOME_VFS_FILE_INFO_FOLLOW_LINKS;

	visit_options = (GNOME_VFS_DIRECTORY_VISIT_LOOPCHECK
			 | GNOME_VFS_DIRECTORY_VISIT_SAMEFS);

	result = gnome_vfs_directory_visit_uri (data->uri,
						info_options,
						dir_filter,
						visit_options,
						dir_visit_cb,
						data);

	gnome_vfs_directory_filter_destroy (dir_filter);
	gnome_vfs_directory_filter_destroy (data->filter);
	list = data->files;
	dir_search_data_free (data);

	return list;
}
