# lib/foomatic/DB_perl_xml.pm
# ---------------------------
#
# This is a legacy version of DB.pm with the old Perl-XML/Grove XML
# handling and on-disk cache, it can be used with foomatic-combo-xml.c or 
# without (very slow and memory-consuming). It is not needed by a standard
# Foomatic installation.
#
# To use it, either replace al occurences of "DB" in the Foomatic script(s)
# by "DB_perl_xml" or move this file to lib/Foomatic/DB.pm and replace all
# occurences of "DB_perl_xml" in this file by "DB".
#
# If you want to use the Makefile to build Foomatic with this file,
# uncomment all cache-related lines in Makefile and makeDefaults (do a
# case-independent search for the term "cache" in these files).
#
# Here some hints about the old Perl-XML/Grove system from the old
# Foomatic 1.1 README file:
#
# Foomatic::GrovePath
# -------------------
#
# This is a slightly modified version of the standard XML::Grove::Path,
# which kept blowing up on me.  The silly thing die'd for XML paths that
# don't exist.  This one doesn't.
#
#
# Dependencies
# ------------
# use Storable;              # Already in Debian libstorable-perl
# use XML::Grove;            # Already in Debian
# use XML::Grove::Builder;
# use XML::Grove::AsString;
# use XML::Parser::PerlSAX;
# use XML::Grove::Factory;
# use XML::Grove::AsCanonXML;
# use XML::Twig;
#
# These came in, and required beyond Debian potato stuff, the following
# files from CPAN (Foomatic is the printer database for Mandrake 8.1 and
# newer, so all these are already available):
#
# Storable-1.0.12.tar.gz     # already in Debian
# DateManip-5.39.tar.gz
# Parse-Yapp-1.04.tar.gz
# XML-Grove-0.46alpha.tar.gz # already in Debian
# XML-Dumper-0.4.tar.gz
# XML-Encoding-1.01.tar.gz
# XML-Parser-2.29.tar.gz     # includes expat; easy to install
# XML-Parser.2.30.tar.gz     # doesn't include expat; harder to install
# XML-Twig-2.02.tar.gz
# libxml-enno-1.02.tar.gz
# libxml-perl-0.07.tar.gz    # Debian has, but out of date?
#
# I experimented with XML::DOM, but the API was created by lunatics or
# something, and the resulting data structure is impossible to work with
# sensibly in Perl: it's gratuitously self-referential and stores all
# sorts of stuff in $i[0][1][3][5][2][9] array things for no good
# reason.
#
# I also experimented with several pretty-print methods for XML; the
# current one works if there are no comments, and only mucks up slightly
# on comments.  The current scheme involves comverting back and forth
# from xml a few times to get it pretty.  This is better than
# libxml-enno XML::Filter::Reindent, which reverses all the tags and
# skips elements.  Argh!
#
# Special Grove-specific API functions
# ------------------------------------
#
# get_overview_grove
#
#   This returns an overview listing of all printers, with ids, makes,
#   models, functionaliy, drivers, etc.  This function returns a Grove.
#
# get_combo_data_grove
#
#   This returns the combined printer/driver data for a particular
#   combination.  Arguments are driver, printer id, force; where force
#   is 1 to recompile for sure and not included or zero to take from the
#   cache.
#
#   This function returns a Perl XML::Grove structure.
#
#   The "combo" operation is less trival than it looks; be careful when
#   messing with it.  It took me (Grant) hours to get right...
#
# get_printer_grove
# get_driver_grove
#
#   These return the information in the printer info or driver info
#   database files as a Grove.
#
# grove_*
# pretty_xml
#
#   Various utility functions for XML and Grove data manipulation.  I'll
#   probably rearrange and derive a personal flavour of Grove so that
#   these will be methods on Groves instead of random utility functions
#   like this.
#
# All the rest is as in lib/Foomatic/DB.pm and you can read in the README
# file about it.

package Foomatic::DB_perl_xml;
use Exporter;
@ISA = qw(Exporter);

# TODO make the grove stuff a proper module, etc
@EXPORT_OK = qw(normalizename comment_filter
		grove_from_filename 
		grove_from_xml
		grove_prune
		grove_clone
		grove_pathval
		grove_to_xml
		pretty_xml 
		get_overview
		getexecdocs
		);

use Foomatic::Defaults qw(:DEFAULT $DEBUG);
use Data::Dumper;
use Storable qw/nstore retrieve/;
use XML::Grove;
use XML::Grove::Builder;
use Foomatic::GrovePath;	# fixed XML::Grove::Path;
use XML::Grove::AsString;
use XML::Parser::PerlSAX;
use XML::Grove::Factory;
use XML::Grove::AsCanonXML;
#use XML::Grove::PerlSAX;
use XML::Twig;			# for pretty print
use POSIX;                      # for rounding integers

my $ver = '$Revision: 1.2 $ ';

# constructor for Foomatic::DB
sub new {
    my $type = shift(@_);
    my $this = bless {@_}, $type;
    return $this;
}

# A map from the database's internal one-letter types to English
my %driver_types = ('F' => 'Filter',
		    'P' => 'Postscript',
		    'U' => 'Ghostscript Uniprint',
		    'G' => 'Ghostscript');

# List of driver names
sub get_driverlist {
    my ($this) = @_;
    return $this->_get_xml_filelist('source/driver');
}

# List of printer id's
sub get_printerlist {
    my ($this) = @_;
    return $this->_get_xml_filelist('source/printer');
}

sub get_overview {
    my ($this, $compile) = @_;

    if ((!defined($this->{'overview'}))
	or (defined($compile) and $compile)) {

	my @over = ();
	
	my $ogrove = $this->get_overview_grove($compile);
	my $plist = $ogrove->at_path('/overview');
	my $pgrove;
	for $pgrove (@{$plist->{Contents}}) {
	    next if ($pgrove->{Name} ne 'printer');
	    
	    my %p;
	    
	    $p{'id'} = grove_pathval($pgrove, '/id');
	    $p{'functionality'} = grove_pathval($pgrove, '/functionality');
	    $p{'make'} = grove_pathval($pgrove, '/make');
	    $p{'model'} = grove_pathval($pgrove, '/model');
	    $p{'driver'} = grove_pathval($pgrove, '/driver');
	    $p{'unverified'} = ($pgrove->at_path('/unverified') ? 1 : 0);
	    
	    my @drivers = ();
	    my $dlist = $pgrove->at_path('/drivers');
	    my $delem;
	    for $delem (@{$dlist->{Contents}}) {
		next if ($delem->{Name} ne 'driver');
		push (@drivers, grove_pathval($delem, '/'));
	    }
	    
	    $p{'drivers'} = \@drivers;
	    
	    push (@over, \%p);
	}
	
	$this->{'overview'} = \@over;
    }

    return $this->{'overview'};
}

sub get_overview_xml {
    my ($this, $compile) = @_;

### CACHE ###
    if ((-r "$cachedir/compiled/overview.xml") && (!$compile))
    {
	return `cat '$cachedir/compiled/overview.xml'`;
    }

### NO CACHE ###
    if (-x "$bindir/foomatic-combo-xml") {
	# The C code is already used in "get_overview_grove", but the XML
	# prettyfying needs so long, that we directly generate the XML here.

### CACHE ###
	if ($cachedir)
	{
	    system "umask 0002; mkdir -p '$cachedir/compiled/'; chmod -R 775 '$cachedir/compiled/'";
	    system "umask 0002; '$bindir/foomatic-combo-xml' -O -l '$libdir' > '$cachedir/compiled/overview.xml'";
	    return `cat '$cachedir/compiled/overview.xml'`;
	}
	else 
### NO CACHE ###
	{
	    open( FCX, "$bindir/foomatic-combo-xml -O -l '$libdir'|")
	      or die "Can't execute $bindir/foomatic-combo-xml -O -l '$libdir'";
	    $_ = <FCX>;
	    close FCX;
	    return $_;
	}
    } else {
	return pretty_xml(grove_to_xml(get_overview_grove(@_)));
    }
}

sub get_overview_grove {
    my ($this, $compile) = @_;

    my $retval = undef;

### CACHE ###
    if (((!-r "$cachedir/compiled/overview.xml") || ($compile)) &&
	(-x "$bindir/foomatic-combo-xml")) {
	# We have C backend "foomatic-combo-xml"
	# Make the needed directory
	system "umask 0002; mkdir -p '$cachedir/compiled/'; chmod -R 775 '$cachedir/compiled/'";
	# Generate the XML overview file and put it into the cache
	system "umask 0002; '$bindir/foomatic-combo-xml' -O -l '$libdir' > '$cachedir/compiled/overview.xml'";
    }

### NO CACHE ###
    if (!$compile) {
	$retval = $this->_get_object_grove('compiled/overview');
    }

    if (! defined($retval)) {

	print STDERR "Compiling overview...\n" if $DEBUG;
	
	my @plist;
	my $p;
	for $p ($this->get_printerlist()) {
	    my ($make, $model, $func, $unver, $driver, @dlist);
	    
	    my $pgrove = $this->get_printer_grove($p);
	    $make = grove_pathval($pgrove, '/printer/make');
	    $model = grove_pathval($pgrove, '/printer/model');
	    $driver = grove_pathval($pgrove, '/printer/driver');
	    $func = grove_pathval($pgrove, '/printer/functionality');
	    $unverified = grove_pathval($pgrove, '/printer/unverified');

	    my $autoxml = '';
	    my $auto = $pgrove->at_path('/printer/autodetect');
	    if (defined($auto)) {
		$autoxml = pretty_xml(grove_to_xml($auto));
	    }

	    my $d;
	    for $d ($this->_get_drivers_for_printer($p)) {
		push (@dlist, "      <driver>$d</driver>\n");
	    }
	    
	    push (@plist,
		  "  <printer>\n",
		  "    <id>$p</id>\n",
		  "    <make>$make</make>\n",
		  "    <model>$model</model>\n");

	    push (@plist,
		  "    <functionality>$func</functionality>\n")
		if defined($func);

	    push (@plist,
		  "    <unverified>$func</unverified>\n")
		if defined($unverified);

	    push (@plist,
		  "    <driver>$driver</driver>\n")
		if defined($driver);

	    push (@plist, $autoxml);

	    push (@plist,
		  "    <drivers>\n",
		  @dlist,
		  "    </drivers>\n",
		  "  </printer>\n");

	}

	# Now save the thing
	$this->_set_object_xml('compiled/overview',
			       join('',
				    "<overview>\n",
				    @plist,
				    "</overview>\n")
			       ,1 ### CACHE ###
			       );


	# Now read back to get a grove
	$retval = $this->_get_object_grove('compiled/overview');

    }

    return $retval;
}


# The combo_data is the new form of what was once the ".foo" or "dat"
# structure.  It is essentially everything related to operating a
# specific printer with a particular driver.  Lots of cruft gets
# pruned, and you get a thing structured thusly:
#
# /foomatic/printer     - Various general printer details
# /foomatic/driver      - How to execute the driver, etc
# /foomatic/options     - Options for submission with a job

# The main function is get_combo_data_grove(), which returns a Perl
# Grove of the thing from a driver id and printer id.  You can also
# call the _xml form to get it returned as XML text.

sub get_combo_data_xml {
    return pretty_xml(grove_to_xml(get_combo_data_grove(@_)));
}

sub uncache_combo_data_grove {
    my ($this,$drv,$poid) = @_;

    delete $this->{'combo-data-grove-cache'}{$drv}{$poid};
}

sub get_combo_data_grove {
    my ($this,$drv,$poid,$compile) = @_;

    my $objid = "compiled/combo/$drv/$poid";
    my $retval = undef;

### CACHE ###
    if ($cachedir)
    {
	if (((!-r "$cachedir/compiled/combo/$drv/$poid.xml") ||
	     (-z "$cachedir/compiled/combo/$drv/$poid.xml") ||
	     ($compile)) &&
	    (-x "$bindir/foomatic-combo-xml")) {
	    # We have C backend "foomatic-combo-xml"
	    # Make the needed directory
	    system "umask 0002; mkdir -p '$cachedir/compiled/combo/$drv'; chmod -R 775 '$cachedir/compiled/'";
	    # Generate the XML combo file and put it into the cache
	    system "umask 0002; '$bindir/foomatic-combo-xml' -p '$poid' -d '$drv' -l '$libdir' > '$cachedir/compiled/combo/$drv/$poid.xml'";
	}
    }

### NO CACHE ###
    # Try to load from file or memory cache
    if (!$compile) {
	$retval = $this->_get_object_grove("compiled/combo/$drv/$poid");
    }

    # Otherwise, build anew
    if (!defined($retval)) {
    
	#print STDERR "Compiling combo data for $drv and $poid...\n";
	
	# First, get printer and driver info
	my $printer = $this->get_printer_grove($poid);
	my $driver = $this->get_driver_grove($drv);

	die "Unable to load printer $printer - does it exist?\n"
	    if !defined($printer);
	die "Unable to load driver $driver - does it exist?\n"
	    if !defined($driver);
	
	# Second, verify that this driver works on this printer
	my $p;
	my $found=undef;
	my $prnlist = $driver->at_path("/driver/printers");
	for $p (@{$prnlist->{'Contents'}}) {
	    next if ( (!defined ($p->{'Name'}))
		      or $p->{'Name'} ne 'printer');
	    
	    my $id = grove_pathval($p, 'id');
	    if ($id eq "printer/$poid") {
		$found=$p;
		last;
	    }
	}
	if (!defined($found)) {
	    warn "Printer $poid doesn't work with driver $drv!\n";
	    return undef;
	}
	
	# Get printer make and model for constraint application
	my ($make, $model) = (grove_pathval($printer,'/printer/make'),
			      grove_pathval($printer,'/printer/model'));
	
	# Now form a new document with the printer and driver info, pruned
	# a bit, plus a new options section listing the options that
	# apply...
	
	my $options = new XML::Grove::Element('Name' => 'options');
	my $foo = new XML::Grove::Element('Name' => 'foomatic',
					  'Contents' => [ grove_clone($printer->at_path("/printer")),
							  grove_clone($driver->at_path("/driver")),
							  $options ] );
	my $doc = new XML::Grove::Document('Contents' => [ $foo ]);
	
	# Leave in only this printer in driver/printers, for it may have
	# useful comments!
	
	my $prnhandle = $foo->at_path('/driver/printers');
	$prnhandle->{Contents} = [ grove_clone($found) ];
	
	# Examine every single option in the world.  See if the
	# constraints let it apply; if so, keep the one constraint that
	# works (it has the default value in) and add the option into the
	# "options" section.
	
	opendir DIR, "$libdir/db/source/opt" 
	    or die "Can't open $libdir/db/source/opt\n";
	my $fname;
	while($fname=readdir(DIR)) {
	    next if ($fname !~ m!^([\-\_\.\w\d]+)\.xml$!);
	    my $id = $1;
	    my $o_orig = $this->_get_object_grove("source/opt/$id");

#	    print STDERR "...considering option $id\n";

	    my $cons = $o_orig->at_path('/option/constraints');
	    my $c;
	    if ($cons) {
		my $clist = $this->_parse_constraints_list("opt/$id", $cons);
		$c = $this->_check_constraints_grove($clist, "opt/$id",
						     $drv, $make, $model,
						     "printer/$poid");
	    }
	    my $sense = 'false';	# false unless constrained otherwise
	    if ($c) {
		$sense = $c->{Attributes}{'sense'};
	    }
	    
	    # debug
	    if (0) {
		if ($sense eq 'true') {
		    print STDERR "option $id applies!\n";
		} else {
		    print STDERR "option $id doesn't apply.\n";
		}
	    }
	    
	    if ($sense eq 'true') {
		# OK, this option applies.  Make a copy, add to $options
		my $o = grove_clone($o_orig);
		push (@{$options->{'Contents'}}, $o);
		
		# 0: These constraints are not useful?
		if (0) {
		    # replace all the constraints with a copy of the one that applies
		    $o->at_path('constraints')->{'Contents'} = [ $c ];
		} else {
		    grove_prune($o, '/', 'constraints');
		}

		# Put in the arg_defval
		my $adefval = $c->at_path('arg_defval');
		die "No default value in opt/$id\n"
		    if (!defined($adefval));
		push (@{$o->{'Contents'}}, $adefval);

		# Now, if enum, do constraints on enum_vals, too
		if ('enum' eq $o->{'Attributes'}->{'type'}) {
		    my @ev_list;
		    my $ev;
		    my $evref = $o->at_path('/enum_vals');

		    if (!defined($evref)) {
			warn "No <enum_vals> in opt/$id!\n";
		    } else {

			my @enum_vals = @{$evref->{'Contents'}};
			
			for $ev (@enum_vals) {
			    next if ($ev->{'Name'} ne 'enum_val');
			    
			    my $ev_cons = $ev->at_path('constraints');
			    
			    my $ev_c = undef;
			    if ($ev_cons) {
				my $ev_id = $ev->{'Attributes'}{'id'};
				
				my $ev_clist = $this->_parse_constraints_list("opt/$id:ev/$ev_id",
									      $ev_cons);
				$ev_c = $this->_check_constraints_grove($ev_clist, 
									"opt/$id:ev/$ev_id",
									$drv, $make, $model,
									"printer/$poid");
			    }
			    
			    my $ev_sense = 'true'; # true unless constrained otherwise
			    if ($ev_c) {
				$ev_sense = $ev_c->{Attributes}{'sense'};
			    }
			    
			    if ($ev_sense eq 'true') { 
				# Various doctoring of constraints in the output
				
				# Remove all the constraints
				delete $ev_cons->{'Contents'};
				
				# 0: These constraints are not useful?
				if (0 and $ev_cons and $ev_c) {
				    # There was a constraint that made us apply; include
				    $ev_cons->{'Contents'} = [ $ev_c ];
				} else {
				    # There is no constraint; remove the section entirely
				    grove_prune($ev, '/', 'constraints');
				}
				
				# Now note this as an applicable enum_val
				push (@ev_list, $ev);
			    }
			}
			
			# Replace enum_vals from o with @ev_list
			# This is kosher; $o is a copy
			my $oevref = $o->at_path('enum_vals');
			if (defined($oevref)) {
			    $oevref->{'Contents'} = \@ev_list;
			} else {
			    warn "This should never happen (opt/$id)\n";
			}
		    }
		}
	    }
	    # Flush memory cache to lower the memory usage and to solve
	    # problem of foomatic-configure not exiting when the Foomatic
	    # data of GIMP-Print is installed.
	    delete $this->{'combo-data-grove-cache'};
	    delete $this->{'constraints-cache'};
	    delete $this->{'illegal-constraint-cache'};
	    delete $this->{'grovecache'};	    
	}

	# Cache the thing, both in memory and on disk
	$retval = $doc;

	# Try to save if permissions allow...
	$this->_set_object_xml("compiled/combo/$drv/$poid", 
			       pretty_xml(grove_to_xml($doc)),
			       1);   # cache!
    }

    # Return the Grove
    return $retval;
}

# Take a .../constraints list and parse the sucker
sub _parse_constraints_list {
    my ($this, $objid, $constraints) = @_;

    my $retval = $this->{'constraints-cache'}{$objid};

    if (!defined($retval)) {

	my @clist = ();
	my $c;
	for $c (@{$constraints->{'Contents'}}) {
	    
	    # skip whitespace, comment, etc nodes...
	    next if ((!defined($c->{'Name'}))
		     or ( $c->{'Name'} ne 'constraint') );
	    
	    # Get values for this constraint from the Grove
	    my $c_make = grove_pathval($c, '/make');
	    my $c_model = grove_pathval($c, '/model');
	    my $c_poid = grove_pathval($c, '/printer');
	    my $c_driver = grove_pathval($c, '/driver');
	    my $c_sense = ( $c->{Attributes}{'sense'} eq 'true' ? 1 : 0);
	    
	    my $c_item = { 'make' => $c_make,
			   'model' => $c_model,
			   'poid' => $c_poid,
			   'driver' => $c_driver,
			   'sense' => $c_sense,
			   'xml' => grove_to_xml($c) };

	    push (@clist, $c_item);

	}    

	$retval = $this->{'constraints-cache'}{$objid} = \@clist;
    }

    return $retval;
}

# Tricky internal function; used to apply options and choices when
# computing a printer/driver combo...
sub _check_constraints_grove {
    # Find the most specific constraint for a thing return undef if
    # false sense or not found, or ref to constraint struct

    my ($this, $constraints, $objectid, $driver, $make, $model, $poid) = @_;

    # $constraints is an XML::Grove::Element for <constraints>...</constraints>
    die "\n" if (!defined($constraints));

    my $winner=undef;
    my $c;
    for $c (@{$constraints}) {

	# Get values for this constraint from the list thing
	my $c_make = $c->{'make'};
	my $c_model = $c->{'model'};
	my $c_poid = $c->{'poid'};
	my $c_driver = $c->{'driver'};
	my $c_sense = $c->{'sense'};
	
	if (! ($c_make or $c_model or $c_driver or $c_poid)) {
	    warn "Illegal null constraint in object '$objectid'!\n"
		unless ($this->{'illegal-constraint-cache'}{$objectid}++);
	    next;
	}
	
	if (($c_make or $c_model) and $c_poid) {
	    warn "Both printer id and make/model in constraint '$objectid'!\n"
		unless ($this->{'illegal-constraint-cache'}{$objectid}++);
	    next;
	}

	if (0) {
	    print STDERR ("  in object $objectid ponder cons('$c_make'?$make,'$c_model'?$model,",
			  "'$c_driver'?$driver,'$c_poid'?$poid '$c_sense')\n");
	}
	
	# if make matches, $p match grade 1.  
	# if model matches, $p match grade 2
	# no information, $p == 0
	# mismatch, $p = -1
	my $p=0;		# printer match
	my $d=0;		# driver match; -1==mismatch, 1==match, 0==no info

	# The per-printer constraining can happen by poid or by
	# a make[/model] pair
	if (defined($poid) and defined($c_poid)) {
	    if ($c_poid eq $poid) {
		$p = 2;		# exact match
	    } else {
		$p = -1		# mismatch
	    }
	} elsif (defined($make) and defined($c_make)) {
	    # We have a make and a requested make, so it can't be zero.
	    # You can't request or constraint by model only!
	    if ($c_make eq $make) {
		$p = 1;		# tentative match
		if (defined($model) and defined($c_model)) {
		    if ($c_model eq $model) {
			$p = 2;	# exact match
		    } else {
			$p = -1; # mismatch on make and model
		    }
		}
	    } else {
		$p = -1;	# mismatch on make alone
	    }
	} 
	
	# if the constraint's driver is null or if it matches, then
	# the driver matches
	if (defined($driver) and defined($c_driver)) {
	    # We have driver info, and a requested driver, so $d can't be 0
	    if ($c_driver eq $driver) {
		$d = 1;
	    } else {
		$d = -1;
	    }
	}
	
	# Hang onto the various scores, along with the constraint
	my $contender = { 'c' => $c,
			  'p' => $p,
			  'd' => $d };
	
#	print STDERR "         ...score p=$p d=$d\n";
	
	# if any sort of match...
	if (($p > 0 or $d > 0) and ($p != -1 and $d != -1)) {

	    # Does this beat our best match to date?
	    if (! defined($winner)) {
		# any sort of match with no winner thus far wins
		$winner = $contender;
	    } else {
		my $win = 0;
		if ($p >= $winner->{'p'}  and  $d >= $winner->{'d'}) {
		    # They're equal or better in both categories
		    $win=1;
		} elsif ($p == 2) {
		    # A specific printer always wins
		    $win=1;
		}
		
		if ($win) {
		    $winner = $contender;
		}
	    }
	}
    }

    # Return the winning constraint
    my $retval = undef;

    if (defined($winner)) {
	$retval = grove_from_xml($winner->{'c'}{'xml'});
	if (defined($retval)) {
	    $retval = $retval->{Contents}[0];
	} else {
	    warn "Defined winner, but no grove (object $objectid)?!\n";
	    print STDERR Dumper($winner);
	}
    }

    return $retval;
}

# Various fetching things.  These work through cache, operated by the
# _get_object functions (way at the bottom of this file).
sub get_printer_grove {
    my ($this, $poid) = @_;
    return $this->_get_object_grove("source/printer/$poid");
}
sub get_driver_grove {
    my ($this, $drv) = @_;
    return $this->_get_object_grove("source/driver/$drv");
}
sub get_printer_xml {
    my ($this, $poid) = @_;
    return $this->_get_object_xml("source/printer/$poid", 1);
}
sub get_driver_xml {
    my ($this, $drv) = @_;
    return $this->_get_object_xml("source/driver/$drv", 1);
}


# Utility query function sorts of things:

sub get_printers_for_driver {
    my ($this, $driver) = @_;

    if (!defined($this->{'printers-for-driver'}{$driver})) {
	my @retval;

	my $dgrove = $this->get_driver_grove($driver);
	do { warn "Unable to get driver grove for driver $driver\n";
	     return undef } unless $dgrove;

	my $prnlist = $dgrove->at_path("/driver/printers");
	
	for $p (@{$prnlist->{'Contents'}}) {
	    next if ( (!defined ($p->{'Name'}))
		      or $p->{'Name'} ne 'printer');
	    
	    my $id = grove_pathval($p, 'id');
	    $id =~ s!^printer\/!!;
	    push (@retval, $id);
	}

	# ref OK; retval is instantly out of scope, anyway...
	$this->{'printers-for-driver'}{$driver} = \@retval;
    }

    return @{$this->{'printers-for-driver'}{$driver}};
}

# Routine lookup; just examine the overview
# See also the "real" one: _get_drivers_for_printer()
sub get_drivers_for_printer {
    my ($this, $printer) = @_;

    my @drivers = ();

    my $over = $this->get_overview();

    my $p;
    for $p (@{$over}) {
	if ($p->{'id'} eq $printer) {
	    return @{$p->{'drivers'}};
	}
    }

    return undef;
}

# This is a bitch; we have to search everywhere then invert
sub _get_drivers_for_printer {
    my ($this, $printer) = @_;

    if (!defined($this->{'drivers-for-printer'})) {
	my $driver;
	for $driver ($this->get_driverlist()) {
	    my @printers = $this->get_printers_for_driver($driver);
	    for (@printers) {
		push (@{$this->{'drivers-for-printer'}{$_}}, $driver);
	    }
	}
    }
    
    return @{$this->{'drivers-for-printer'}{$printer}};
}

# This function should apply a more or less senseful order to the argument
# names so that they appear kink of sorted in frontends
sub sortargs ($$) {

    # All sorting done case-insensitive and characters which are not a letter
    # or number are taken out!!

    # List of typical option names to appear at first
    # The terms must fit to the beginning of the line, terms which must fit
    # exactly must have '\$' in the end.
    my @standardopts = (
			# Options which appear in the "General" group in CUPS
			# and similar media handling options
			"pagesize",
			"papersize",
			"mediasize",
			"inputslot",
			"papersource",
			"mediasource",
			"sheetfeeder",
			"mediafeed",
			"paperfeed",
			"manualfeed",
			"manual",
			"outputtray",
			"outputslot",
			"outtray",
			"faceup",
			"facedown",
			"mediatype",
			"papertype",
			"mediaweight",
			"paperweight",
			"duplex",
			"sides",
			"binding",
			"tumble",
			"notumble",
			"media",
			"paper",
			# Other hardware options
			"inktype",
			"ink",
			# Page choice/ordering options
			"pageset",
			"pagerange",
			"pages",
			"nup",
			"numberup",
			# Printout quality, colour/bw
			"resolution",
			"gsresolution",
			"hwresolution",
			"quality",
			"printquality",
			"printoutquality",
			"bitsperpixel",
			"photomode",
			"photo",
			"colormode",
			"colourmode",
			"color",
			"colour",
			"grayscale",
			"gray",
			"monochrome",
			"mono",
			"blackonly",
			"colormodel",
			"colourmodel",
			"processcolormodel",
			"processcolourmodel",
			"printcolors",
			"printcolours",
			"outputtype",
			"outputmode",
			"printingmode",
			"printoutmode",
			"printmode",
			"mode",
			"imagetype",
			"imagemode",
			"image",
			"dithering",
			"dither",
			"halftoning",
			"halftone",
			"floydsteinberg",
			"ret\$",
			"cret\$",
			"photoret\$",
			# Adjustments
			"gammacorrection",
			"gammacorr",
			"gammageneral",
			"mastergamma",
			"stpgamma",
			"gammablack",
			"gammacyan",
			"gammamagenta",
			"gammayellow",
			"gamma",
			"density",
			"stpdensity",
			"hpljdensity",
			"tonerdensity",
			"inkdensity",
			"brightness",
			"stpbrightness",
			"saturation",
			"stpsaturation",
			"hue",
			"stphue",
			"tint",
			"stptint",
			"contrast",
			"stpcontrast",
			"black",
			"stpblack",
			"cyan",
			"stpcyan",
			"magenta",
			"stpmagenta",
			"yellow",
			"stpyellow"
			);
    # Bring the two option names into a standard form to compare them
    # in a better way
    my $first = normalizename(lc($_[0]));
    $first =~ s/[\W_]//g;
    my $second = normalizename(lc($_[1]));
    $second =~ s/[\W_]//g;
    # Check whether they are in the @standardopts list
    my $i;
    for ($i = 0; $i <= $#standardopts; $i++) {
	my $firstinlist = ($first =~ /^$standardopts[$i]/);
	my $secondinlist = ($second =~ /^$standardopts[$i]/);
	if (($firstinlist) && (!$secondinlist)) {return -1};
	if (($secondinlist) && (!$firstinlist)) {return 1};
    }

    # None of the search terms in the list, compare the standard-formed strings
    my $compare = ( $first cmp $second );
    if ($compare != 0) {return $compare};

    # No other criteria fullfilled, compare the original input strings
    return $_[0] cmp $_[1];
}

sub sortvals ($$) {

    # All sorting done case-insensitive and characters which are not a letter
    # or number are taken out!!

    # List of typical choice names to appear at first
    # The terms must fit to the beginning of the line, terms which must fit
    # exactly must have '\$' in the end.
    my @standardvals = (
			# Paper sizes
			#"letter",
			#"legal",
			#"a000004",
			);
    # Bring the two option names into a standard form to compare them
    # in a better way
    my $first = normalizename(lc($_[0]));
    $first =~ s/[\W_]//g;
    my $second = normalizename(lc($_[1]));
    $second =~ s/[\W_]//g;
    # Check whether they are in the @standardopts list
    my $i;
    for ($i = 0; $i <= $#standardvals; $i++) {
	my $firstinlist = ($first =~ /^$standardvals[$i]/);
	my $secondinlist = ($second =~ /^$standardvals[$i]/);
	if (($firstinlist) && (!$secondinlist)) {return -1};
	if (($secondinlist) && (!$firstinlist)) {return 1};
    }

    # None of the search terms in the list, compare the standard-formed strings
    my $compare = ( $first cmp $second );
    if ($compare != 0) {return $compare};

    # No other criteria fullfilled, compare the original input strings
    return $_[0] cmp $_[1];
}

# Take driver/pid arguments, call get_combo_data_grove, and whip up
# an old-fashioned .foo Perl hash structure, so that the existing
# backends and FOO->SPOOLER DATA functions can still work...
sub getdat {
    my ($this, $drv, $poid, $force) = @_;

    my %dat;			# Our purpose in life...

    my $c = $this->get_combo_data_grove($drv, $poid, $force);
    
    do { warn "Cannot load driver $drv for printer $poid!\n";
	 return undef; } unless defined ($c);

    ## From driver info
    $dat{'driver'} = $drv;
    $dat{'url'} = grove_pathval($c, '/foomatic/driver/url');

    if ($c->at_path('/foomatic/driver/execution/filter')) {
	$dat{'type'} = 'F';
    } elsif ($c->at_path('/foomatic/driver/execution/uniprint')) {
	$dat{'type'} = 'U';
    } elsif ($c->at_path('/foomatic/driver/execution/ghostscript')) {
	$dat{'type'} = 'G';
    } elsif ($c->at_path('/foomatic/driver/execution/postscript')) {
	$dat{'type'} = 'P';
    }

    $dat{'cmd'} = grove_pathval($c, '/foomatic/driver/execution/prototype');
    $dat{'comment'} = grove_pathval($c, '/foomatic/driver/comments/en');

    ## From printer info

    $dat{'make'} = grove_pathval($c, '/foomatic/printer/make');
    $dat{'model'} = grove_pathval($c, '/foomatic/printer/model');
    $dat{'pjl'} = grove_pathval($c, '/foomatic/printer/lang/pjl');
    $dat{'pnp_mfg'} = grove_pathval($c, '/foomatic/printer/autodetect/parallel/manufacturer');
    $dat{'pnp_cmd'} = grove_pathval($c, '/foomatic/printer/autodetect/parallel/commandset');
    $dat{'pnp_des'} = grove_pathval($c, '/foomatic/printer/autodetect/parallel/description');
    $dat{'pnp_mdl'} = grove_pathval($c, '/foomatic/printer/autodetect/parallel/model');    
    my $cset = grove_pathval($c, '/foomatic/printer/lang/text/charset');
    $dat{'ascii'} = (($cset eq 'us-ascii'  or  $cset eq 'iso-8859-1')
		     ? 1 : 0);
    $dat{'color'} = $c->at_path('/foomatic/printer/mechanism/color') ? 1 : 0;

    ## From opts
    my $maxspot = 'A';
    my $o;
    for $o (@{$c->at_path('/foomatic/options')->{Contents}}) {
	next if ($o->{Name} ne 'option'); # skip whitespace, etc

	my $arg;
	# First get base data in
	$arg->{'type'} = $o->{Attributes}{'type'};
	$arg->{'idx'} = $o->{Attributes}{'id'};
	if ($o->at_path('/arg_execution/arg_substitution')) {
	    $arg->{'style'} = 'C'; # command-line
	} elsif ($o->at_path('/arg_execution/arg_postscript')) {
	    $arg->{'style'} = 'G'; # gs/ps
	} elsif ($o->at_path('/arg_execution/arg_pjl')) {
	    $arg->{'style'} = 'J'; # pjl
	}
	if ($o->at_path('/arg_execution/arg_required')) {
	    $arg->{'required'} = 1;
	}
	$arg->{'name'} = grove_pathval($o, '/arg_shortname/en');
	$arg->{'comment'} = grove_pathval($o, '/arg_longname/en');
	$arg->{'proto'} = grove_pathval($o, '/arg_execution/arg_proto');
	$arg->{'spot'} = grove_pathval($o, '/arg_execution/arg_spot');
	$arg->{'order'} = grove_pathval($o, '/arg_execution/arg_order');
	$maxspot = $arg->{'spot'} if ($arg->{'spot'} gt $maxspot);
	
	# for future reference
	my $defval = grove_pathval($o, '/arg_defval');

	## Then do type-specifics
	if ($arg->{'type'} eq 'enum') {

	    my $vg;
	    for $vg (@{$o->at_path('/enum_vals')->{Contents}}) {
		next if ($vg->{Name} ne 'enum_val');
		my $v;

		$v->{'idx'} = $vg->{Attributes}{'id'};
		$v->{'value'} = grove_pathval($vg, '/ev_shortname/en');
		$v->{'comment'} = grove_pathval($vg, '/ev_longname/en');
		if (!$v->{'comment'}) {
		    # Must have something for PPDs etc.
		    $v->{'comment'} = $v->{'value'};
		}
		$v->{'driverval'} =  grove_pathval($vg, '/ev_driverval');
		if (!defined($v->{'driverval'})) {
		    # Ought to have something to pass to the driver ;)
		    $v->{'driverval'} = $v->{'value'};
		}

		if ($v->{'idx'} eq $defval) {
		    $arg->{'default'} = $v->{'value'};
		}

		$arg->{'vals_byname'}->{$v->{'value'}} = $v;
	    }
	    
	    # Record enum_vals in an array
	    # Values sorted with "sortvals" function
	    my @sortedvalslist = sort sortvals keys(%{$arg->{'vals_byname'}});
	    my $i;
	    for $i (@sortedvalslist) {
		my $val = $arg->{'vals_byname'}{$i};
		push (@{$arg->{'vals'}}, $val);
	    }
	    
	} elsif ($arg->{'type'} eq 'bool') {
	    $arg->{'name_true'} = 
		grove_pathval($o, '/arg_shortname/en');
	    $arg->{'name_false'} = 
		grove_pathval($o, '/arg_shortname_false/en');
	    if (!defined($arg->{'name_false'})) {
		$arg->{'name_false'} = ("Not " . $arg->{'name_true'});
	    }
	    $arg->{'default'} = $defval ? 1 : '0';
	} elsif ($arg->{'type'} eq 'int' or $arg->{'type'} eq 'float') {
	    $arg->{'max'} = grove_pathval($o, '/arg_max');
	    $arg->{'min'} = grove_pathval($o, '/arg_min');
	    $arg->{'default'} = $defval;
	}

	# remember this argument
	$dat{'args_byname'}{$arg->{'name'}} = $arg;
    }

    $dat{'maxspot'} = $maxspot;

    # Options sorted with "sortargs" function
    my @sortedarglist = sort sortargs keys(%{$dat{'args_byname'}});
    my $an;
    for $an (@sortedarglist) {
	my $arg = $dat{'args_byname'}{$an};
	push (@{$dat{'args'}}, $arg);
    }

    $dat{'compiled-at'} = localtime(time());
    $dat{'timestamp'} = time();

    my $user = `whoami`; chomp $user;
    my $host = `hostname`; chomp $host;

    $dat{'compiled-by'} = "$user\@$host";

    $dat{'id'} = $poid;

    # Funky one-at-a-time cache thing
    $this->{'dat'} = \%dat;

    return \%dat;
}



###################
# MagicFilter with LPRng
#
# getmfdata() returns a magicfilter 2 printer m4 def file

sub getmfdata {
    my ($this) = @_;
    die "you must call getdat first\n" if (!$this->{'dat'});

    my $dat = $this->{'dat'};
    my $driver = $dat->{'driver'};

    my $make = $dat->{'make'};
    my $model = $dat->{'model'};
    my $color = ($dat->{'color'} ? 'true' : 'false');
    my $text = ($dat->{'ascii'} ? 'true' : 'false');
    my $filename = "$make-$model-$driver";
    $filename =~ s![ /]!_!g;
    
    my $tag = $this->{'tag'};
    my $tagfilen;
    if ($tag) {
	$tagfilen = "-$tag";
    }

    push (@decl,
	  "#! \@MAGICFILTER\@\n",
	  "define(Vendor, `$make')dnl\n",
	  "define(Printer, `$model (via $driver driver)')dnl\n",
	  "define(FOOMATIC, `$libdir/data$tagfilen/$filename.foo')dnl\n",
	  "define(COLOR, $color)dnl\n",
	  "define(TEXT, $text)dnl\n");

    return @decl;
}


###################
# PDQ
#
# getpdqdata() returns a PDQ driver description file.

my $pdqomaticversion = $ver;
my $enscriptcommand = 'mpage -o -1 -P- -';

sub getpdqdata {
    my ($this) = @_;
    die "you must call getdat first\n" if (!$this->{'dat'});

    my $dat = $this->{'dat'};
    my $driver = $dat->{'driver'};
    my $make = $dat->{'make'};
    my $model = $dat->{'model'};

    # Encase data for inclusion in the PDQ config file
    my @datablob;
    for(split('\n',$this->getascii())) {
	push(@datablob, "# COMDATA #$_\n");
    }
    
    # Construct structure with driver information
    my @declaration=undef;

    # Construct structure for searching the job whether it contains options
    # added by a PPD-driven client application
    my @searchjobforoptions;
    push (@searchjobforoptions, 
	  "    for opt in \`grep FoomaticOpt \$INPUT | cut -d \" \" -f 3\`; do\n",
	  "        option=\`echo \$opt | cut -d \"=\" -f 1\`\n",
	  "        value=\`echo \$opt | cut -d \"=\" -f 2\`\n",
	  "        case \"\$option\" in\n");

    # If we find only *ostScript style options, the job cannot contain
    # "%% FoomaticOpt" lines. Then we remove @searchjobforoptions
    # afterwards because we do not need to examine the job file.
    my $onlygsargs = 1;

    # First, compute the various option/value clauses
    for $arg (@{$dat->{'args'}}) {

	if ($arg->{'type'} eq 'enum') {
	    
	    my $com = $arg->{'comment'};
	    my $idx = $arg->{'idx'};
	    my $def = $p->{'arg_default'};
	    my $nam = $arg->{'name'};
	    $arg->{'varname'} = "EOPT_$idx";
	    $arg->{'varname'} =~ s![\-\/\.]!\_!g;
	    my $varn = $arg->{'varname'};
	    my $gsarg = 1 if ($arg->{'style'} eq 'G');

	    if (!$gsarg) {$onlygsargs = 0};

	    # No quotes, thank you.
	    $com =~ s!\"!\\\"!g;
	    
	    push(@driveropts,
		 "  option {\n",
		 "    var = \"$varn\"\n",
		 "    desc = \"$com\"\n");
	    
	    push(@searchjobforoptions,
		 "          $nam)\n",
		 "            case \"\$value\" in\n") unless $gsarg;
	
	    # get enumeration values for each enum arg
	    my ($ev, @vals, @valstmp);
	    for $ev (@{$arg->{'vals'}}) {
		my $choiceshortname = $ev->{'value'};
		my $choicename = "${nam}_${choiceshortname}";
		my $val = (defined($ev->{'driverval'}) 
			   ? $ev->{'driverval'} 
			   : $ev->{'value'});
		$val =~ s!\"!\\\"!g;
		my $com = $ev->{'comment'};
		
		# stick another choice on driveropts
		push(@valstmp,
		     "    choice \"$choicename\" {\n",
		     "      desc = \"$com\"\n",
		     "      value = \"$val\"\n",
		     "    }\n");
		push(@searchjobforoptions,
		     "              $choiceshortname)\n",
		     "                $varn=\"$val\"\n",
		     "                ;;\n") unless $gsarg;
	    }
	    
	    push(@driveropts,
		 "    default_choice \"" . $nam . "_" . $arg->{'default'} . 
		 "\"\n",
		 @valstmp,
		 "  }\n\n");

	    push(@searchjobforoptions,
		 "            esac\n",
		 "            ;;\n") unless $gsarg;
	    
	} elsif ($arg->{'type'} eq 'int' or $arg->{'type'} eq 'float') {
	    
	    my $com = $arg->{'comment'};
	    my $idx = $arg->{'idx'};
	    my $nam = $arg->{'name'};
	    my $max = $arg->{'max'};
	    my $min = $arg->{'min'};
	    $arg->{'varname'} = "OPT_$nam";
	    $arg->{'varname'} =~ s![\-\/\.]!\_!g;
	    my $varn = $arg->{'varname'};
	    my $legal = $arg->{'legal'} = "Minimum value: $min, Maximum value: $max";
	    my $gsarg = 1 if ($arg->{'style'} eq 'G');
	    
	    if (!$gsarg) {$onlygsargs = 0};

	    my $defstr = "";
	    if ($arg->{'default'}) {
		$defstr = sprintf("    def_value \"%s\"\n", 
				  $arg->{'default'});
	    }
	    
	    push(@driveropts,
		 "  argument {\n",
		 "    var = \"$varn\"\n",
		 "    desc = \"$nam\"\n",
		 $defstr,
		 "    help = \"$com $legal\"\n",
		 "  }\n\n");
	    
	    push(@searchjobforoptions,
		 "          $nam)\n",
		 "            $varn=\"\$value\"\n",
		 "            ;;\n") unless $gsarg;

	} elsif ($arg->{'type'} eq 'bool') {
	    
	    my $com = $arg->{'comment'};
	    my $tname = $arg->{'name_true'};
	    my $fname = $arg->{'name_false'};
	    my $idx = $arg->{'idx'};
	    $arg->{'legal'} = "Value is a boolean flag";
	    $arg->{'varname'} = "BOPT_$idx";
	    $arg->{'varname'} =~ s![\-\/\.]!\_!g;
	    my $varn = $arg->{'varname'};
	    my $proto = $arg->{'proto'}; 
	    my $gsarg = 1 if ($arg->{'style'} eq 'G');
	    
	    if (!$gsarg) {$onlygsargs = 0};

	    my $defstr = "";
	    if ($arg->{'default'}) {
		$defstr = sprintf("    default_choice \"%s\"\n", 
				  $arg->{'default'} ? "$tname" : "$fname");
	    } else {
		$defstr = sprintf("    default_choice \"%s\"\n", "$fname");
	    }
	    push(@driveropts,
		 "  option {\n",
		 "    var = \"$varn\"\n",
		 "    desc = \"$com\"\n",
		 $defstr,
		 "    choice \"$tname\" {\n",
		 "      desc = \"$tname\"\n",
		 "      value = \"TRUE\"\n",
		 "    }\n",
		 "    choice \"$fname\" {\n",
		 "      desc = \"$fname\"\n",
		 "      value = \"FALSE\"\n",
		 "    }\n",
		 "  }\n\n");

	    push(@searchjobforoptions,
		 "          $tname)\n",
		 "            case \"\$value\" in\n",
		 "              True)\n",
		 "                $varn=\"TRUE\"\n",
		 "                ;;\n",
		 "              False)\n",
		 "                $varn=\"FALSE\"\n",
		 "                ;;\n",
		 "            esac\n",
		 "            ;;\n") unless $gsarg;
	}
	
    }
    
    if ($onlygsargs) {
	@searchjobforoptions = ();
    } else {
	push (@searchjobforoptions, 
	      "        esac\n",
	      "    done\n\n");
    }

    ## Define the "docs" option to print the driver documentation page

    push(@driveropts,
	 "  option {\n",
	 "    var = \"DRIVERDOCS\"\n",
	 "    desc = \"Print driver usage information\"\n",
	 "    default_choice \"nodocs\"\n", 
	 "    choice \"docs\" {\n",
	 "      desc = \"Yes\"\n",
	 "      value = \"yes\"\n",
	 "    }\n",
	 "    choice \"nodocs\" {\n",
	 "      desc = \"No\"\n",
	 "      value = \"no\"\n",
	 "    }\n",
	 "  }\n\n");
    
    ## Now let's compute the postscript filter part
    my @drivfilter;
    push(@drivfilter,
	 "  language_driver postscript {\n",
	 "    # Various postscript tricks would go here\n",
	 "  }\n\n");

    ## Add ASCII to drivfilter!
    ## FIXME
    # Options: we do ascii, so just crlf fix it

    push (@drivfilter,
	  "  language_driver text {\n");

    # temporarily force slow-path ascii for stable release
    if (0 and $dat->{'ascii'}) {
	push(@drivfilter,
	     "\n",
             "     convert_exec {#!/bin/sh\n",
	     "\n",
	     "        sed 's/\$/\r/' \$INPUT > \$OUTPUT\n",
	     "        touch \$OUTPUT.ok\n",
	     "     }\n");
    } else {
	push(@drivfilter,
	     "
     convert_exec {#!/bin/sh

        cat \$INPUT | $enscriptcommand > \$OUTPUT
     }
");
    }

    push (@drivfilter,
	  "  }\n\n");

    ## Load the command line prototype, from which the final command line
    ## will be built.

    my $commandline = $dat->{'cmd'};

    ## Quote special characters so that they are not interpreted when
    ## PDQ builds the filter shell script, but only when PDQ executes it.

    $commandline =~ s!\\!\\\\!g;
    $commandline =~ s!\"!\\\"!g;
    $commandline =~ s!\$!\\\$!g;
        
    ## Now we go through all the options, ordered by the spots in the
    ## command line. The options will be stuffed into the right place
    ## depending on their type

    my @letters = qw/A B C D E F G H I J K L M Z/;
    for $spot (@letters) {
	if ($commandline =~ m!\%$spot!) {
	    
	  argument:
	    for $arg (sort { $a->{'order'} <=> $b->{'order'} } 
		      @{$dat->{'args'}}) {
		
		# Only do arguments that go in this spot
		next argument if ($arg->{'spot'} ne $spot);
		next argument if (($arg->{'style'} ne 'C') && 
				  ($arg->{'style'} ne 'G'));
		
		my $varname = $arg->{'varname'};
		my $cmd = $arg->{'proto'};
		my $comment = $arg->{'comment'};
		my $cmdvar = $arg->{'cmdvarname'} = "CMD_$varname";
		my $type = $arg->{'type'};
		my $gsarg = 1 if ($arg->{'style'} eq 'G');
		
		if ($type eq 'bool') {
		    
		    # If BOPT_whatever is true, the cmd is present.
		    # Otherwise this option is the empty string
		    push(@psfilter,
			 "      # $comment\n",
			 "      if [ \"x\${$varname}\" == 'xTRUE' ]; then\n",
			 "         $cmdvar=\'$cmd\'\n",
			 "      fi\n\n");
		    
		} elsif ($type eq 'int' or $type eq 'float'){
		    
		    # If [IF]OPT_whatever is non-null, put in the
		    # argument.  Otherwise this option is the empty
		    # string.  Error checking?
		    
		    my $fixedcmd = $cmd;
		    $fixedcmd =~ s!\%([^s])!\%\%$1!g;
		    if ($gsarg) {
			$fixedcmd =~ s!\"!\\\"!g;
		    } else {
			#$fixedcmd =~ s!([\\\"\$\;\,\!\&\<\>])!\\\\$1!g;
		    }
		    $fixedcmd = sprintf($fixedcmd, "\${$varname}");
		    
		    push(@psfilter,
			 "      # $comment\n",
			 "      # We aren't really checking for max/min.\n",
			 "      if [ \"x\${$varname}\" != 'x' ]; then\n",
			 "         $cmdvar=\"$fixedcmd\"\n",
			 "      fi\n\n");
		    
		} elsif ($type eq 'enum') {
		    
		    # If EOPT_whatever is non-null, put in the
		    # choice value.
		    
		    my $fixedcmd = $cmd;
		    $fixedcmd =~ s!\%([^s])!\%\%$1!g;
		    if ($gsarg) {
			$fixedcmd =~ s!\"!\\\"!g;
		    } else {
			#$fixedcmd =~ s!([\\\"\$\;\,\!\&\<\>])!\\\\$1!g;
		    }
		    $fixedcmd = sprintf($fixedcmd, "\${$varname}");
		    
		    push(@psfilter,
			 "      # $comment\n",
			 "      # We aren't really checking for legal vals.\n",
			 "      if [ \"x\${$varname}\" != 'x' ]; then\n",
			 "         $cmdvar=\"$fixedcmd\"\n",
			 "      fi\n\n");
		    
		} else {
		    
		    die "evil type!?\n";
		    
		}
		
		if (! $gsarg) {
		    # Insert the processed variable in the commandline
		    # just before the spot marker.
		    $commandline =~ s!\%$spot!\$$cmdvar\%$spot!;
		} else {
		    # Ghostscript/Postscript argument, prepend to job
		    push(@echoes, "echo \"\${$cmdvar}\"");
		}
	    }
	    
	    # Remove the letter marker from the commandline
	    $commandline =~ s!\%$spot!!;
	
	}
    }

    # Generate a driver documentation page which is printed when the user
    # uses the "docs" option.

    my $optstr = ("Specify each option as a -o/-a argument to pdq ie\n",
                  "% pdq -P printer -oDuplex_On -aTwo=2\n");
    
    push(@doctext, 
	 "Invokation summary for your $make $model printer as driven by\n",
	 "the $driver driver.\n",
	 "\n",
	 "$optstr\n",
	 "The following options are available for this printer:\n",
	 "\n");

    for $arg (@{$dat->{'args'}}) {
        my ($name,
            $required,
            $type,
            $comment,
            $spot,
            $default) = ($arg->{'name'},
                         $arg->{'required'},
                         $arg->{'type'},
                         $arg->{'comment'},
                         $arg->{'spot'},
                         $arg->{'default'});
 
        my $reqstr = ($required ? " required" : "n optional");
	push(@doctext,
	     "Option '$name':\n  A$reqstr $type argument.\n  $comment\n");
	push(@doctext,
	     "  This options corresponds to a PJL command.\n") if ($arg->{'style'} eq 'J');
 
        if ($type eq 'bool') {
            push(@doctext, "  Possible choices:\n");
	    my $tname = $arg->{'name_true'};
	    my $fname = $arg->{'name_false'};
	    push(@doctext, "   o -o${tname}: $tname\n");
	    push(@doctext, "   o -o${fname}: $fname\n");
	    my $defstr;
            if (defined($default)) {
                $defstr = ($default ? "$tname" : "$fname");
	    } else {
		$defstr = $fname;
	    }
	    push(@doctext, "  Default: $defstr\n");
	    push(@doctext, "  Example: -o$tname\n");
        } elsif ($type eq 'enum') {
            push(@doctext, "  Possible choices:\n");
            my $exarg;
            for (@{$arg->{'vals'}}) {
                my ($choice, $comment) = ($_->{'value'}, $_->{'comment'});
		push(@doctext, "   o -o${name}_$choice: $comment\n");
                $exarg=$choice;
            }
            if (defined($default)) {
                push(@doctext, "  Default: -o${name}_$default\n");
            }
            push(@doctext, "  Example: -o${name}_$exarg\n");
        } elsif ($type eq 'int' or $type eq 'float') {
            my ($max, $min) = ($arg->{'max'}, $arg->{'min'});
            my $exarg;
            if (defined($max)) {
                push(@doctext, "  Range: $min <= x <= $max\n");
                $exarg=$max;
            }
            if (defined($default)) {
                push(@doctext, "  Default: $default\n");
                $exarg=$default;
            }
            if (!$exarg) { $exarg=0; }
            push(@doctext, "  Example: -aOPT_$name=$exarg\n");
        }
 
	push(@doctext, "\n");
    }

    $docstr = join ("", @doctext);

    # Embed this text file as a "here document" in a shell script which makes
    # PostScript out of it because it will be passed through GhostScript and
    # GhostScript does not understand plain text

    $docstr = "cat <<EOF | $enscriptcommand\n" . $docstr . "\nEOF\n";

    # Execute command
    #
    # Spit out the command with all the post-processed arguments
    # stuffed in where the %A %B etc were.  Don't forget to deal
    # with the %Z normal gs option stuff.

    my $echostr = undef;
    if (scalar(@echoes)) {
	$echostr = join (";\\\n         ", @echoes);
    }

    $commandline =~ s!^\s*gs !\$gs !;
    $commandline =~ s!(\|\s*)gs !\|\$gs !;
    $commandline =~ s!(;\s*)gs !; \$gs !;

    # Important: the parantheses around "$commandline" allow the driver's
    # command line to be composed from various shell commands.
    push(@psfilter,
         "      gs=gs      # assume normal gs unless...\n",
         "      hash foomatic-gswrapper	&& gs='foomatic-gswrapper'\n",
	 "      if ! test -e \$INPUT.ok; then\n",
	 "        # Execute this puppy, already...\n",
	 ( defined($echostr) 
	   ? "        ($echostr;\\\n"
	   : "        ( \n"),
	 "         if [ \"x\$DRIVERDOCS\" == 'xyes' ]; then\n",
	 "           $docstr\n",
	 "         else\n",
	 "           cat \$INPUT\n",
	 "         fi\n",
	 "        ) | sh -c \"( $commandline )\"\\\n",
	 "            >> \$OUTPUT\n",
	 "        if ! test -e \$OUTPUT; then \n",
	 "           echo 'Error running Ghostscript; no output!'\n",
	 "           exit 1\n",
	 "        fi\n",
	 "      else\n",
	 "        ln -s \$INPUT \$OUTPUT\n",
	 "      fi\n\n");
    
    # OK, so much for the postscript_filter part.
    
    # Now let's compute the filter_exec script, which processes
    # all jobs right before sending.  Here is where we do PJL options.

    my (@pjlfilter, @pjlfilter_bot);
    if (defined($dat->{'pjl'})) {
	push(@pjlfilter, 
	     "    echo -ne '\33%-12345X' > \$OUTPUT\n",
	     "    echo '\@PJL JOB NAME=\"PDQ Print Job\"' >> \$OUTPUT\n");
	
      argument:
	for $arg (sort { $a->{'order'} <=> $b->{'order'} } 
		  @{$dat->{'args'}}) {
	    
	    # Only do PJL arguments 
	    next argument if ($arg->{'style'} ne 'J');
	    
	    my $varname = $arg->{'varname'};
	    my $cmd = $arg->{'proto'};
	    my $comment = $arg->{'comment'};
	    my $cmdvar = $arg->{'cmdvarname'} = "CMD_$varname";
	    my $type = $arg->{'type'};
	    
	    my $pjlcmd = sprintf($cmd, "\$$varname");
	    $pjlcmd =~ s!\"!\\\"!g;
	    $pjlcmd =~ s!\\!\\\\!g;
	    
	    if ($type eq 'bool') {
		
		push(@pjlfilter,
		     "      # $comment\n",
		     "      if [ \"x\${$varname}\" != 'x' ]; then\n",
		     "        if [ \"x\${$varname}\" == 'xTRUE' ]; then\n",
		     "          echo \"\@PJL $pjlcmd\" >> \$OUTPUT\n",
		     "        fi\n",
		     "      fi\n\n");
		
	    } elsif ($type eq 'int' or $type eq 'float'){
		
		push(@pjlfilter,
		     "      # $comment\n",
		     "      if [ \"x\${$varname}\" != 'x' ]; then\n",
		     "        echo \"\@PJL $pjlcmd\" >> \$OUTPUT\n",
		     "      fi\n\n");
		
	    } elsif ($type eq 'enum') {
		
		# If EOPT_whatever is non-null, put in the
		# choice value.
		
		push(@pjlfilter,
		     "      # $comment\n",
		     "      if [ \"x\${$varname}\" != 'x' ]; then\n",
		     "        echo \"\@PJL $pjlcmd\" >> \$OUTPUT\n",
		     "      fi\n\n");
		
	    } else {
		
		die "evil type!?\n";
		
	    }
	    
	    # Insert the processed variable in the commandline
	    # just before the spot marker.
	    $commandline =~ s!\%$spot!\$$cmdvar\%$spot!;
	}
	
	# Send the job, followed by the end of job command
	push(@pjlfilter_bot, 
	     "    echo -ne '\33%-12345X' >> \$OUTPUT\n",
	     "    echo '\@PJL RESET' >> \$OUTPUT\n",
	     "    echo '\@PJL EOJ' >> \$OUTPUT\n\n");
	
    }

    my $wwwhome = 'http://www.linuxprinting.org/show_driver.cgi';
    my $showurl = "$wwwhome?driver=$driver";
    my $notes = $dat->{'comment'};
    $notes =~ s!\"!\\\"!sg;
    my $pname = $dat->{'make'} . " " . $dat->{'model'};
    
    push (@body,
	  "  # This PDQ driver was generated automatically by pdq-o-matic.cgi from\n",
	  "  # information in the Printing HOWTO's compatibility database.  It uses\n",
	  "  # the $driver driver to drive a $pname.  \n",
	  "  #\n",
	  "  # For more information on this driver please see the HOWTO's $driver\n",
	  "  # driver database entry at \n",
	  "  # $showurl\n\n",
	  "  help \"$notes\"\n\n",
	  
#	      (  $dat->{'type'} eq 'G' or $dat->{'type'} eq 'U' ? 
#		 "  requires \"gs\"\n" : ""),
	  
	  "  # We need the $driver driver, but I haven't implemented requires yet.\n\n",
	  
	  @driveropts,

	  @drivfilter,
	  
	  "  filter_exec {\n",
	  @searchjobforoptions,
	  @pjlfilter,
	  @psfilter,
	  @pjlfilter_bot,
	  "  }\n"
	  );
    
    my $version = $dat->{'timestamp'};
    my ($smake, $smodel) = ($dat->{'make'}, $dat->{'model'});
    $smake =~ s/ /\-/g;
    $smodel =~ s/ /\-/g;
    my $name = "POM-$driver-$smake-$smodel-$version";
    $name =~ s! !\-!g;
    
    push (@declaration,
	  "# This is a PDQ driver declaration for the ", 
	  lc($driver_types{$dat->{'type'}}), " driver $driver.\n",
	  "# It was generated by pdq-o-matic.cgi version $pdqomaticversion\n\n",
	  "# You should append this file to your personal .printrc, the system\n",
	  "# /etc/printrc, or place it by itself in the systemwide /etc/pdq/drivers\n",
	  "# area.  Then run PDQ's new printer setup wizard.\n\n",
	  "driver \"$name\" {\n\n",
	  @body,
	  "}\n\n",
	  @datablob);
    
    return @declaration;
}

#################
# LPD and spooler-less printing stuff
#
# getlpddata() returns a data file which you can give to lpdomatic or
# directomatic

# Set when you change.  (Not used, but should be?)
my $lpdomaticversion = $ver;
sub getlpddata {

    my ($db) = @_;

    die "you need to call getdat first!\n" 
	if (!defined($db->{'dat'}));

    my $dat = $db->{'dat'};

    # Encase data for inclusion in FOO file
    my @datablob;
    for(split('\n',$db->getascii())) {
	push(@datablob, "$_\n");
    }	

    ## OK, now we have a whole structure named $dat about the
    ## calling of this driver.

    my ($make, $model, $driver, $poid) = ($dat->{'make'}, 
					  $dat->{'model'}, 
					  $dat->{'driver'},
					  $dat->{'id'});
    my @ppd;
    push(@ppd,
	 "# This is an LPD-O-Matic/Direct-O-Matic printer definition file for the\n",
	 "# $make $model printer using the $driver driver.\n",
	 "#\n",
	 "# It is designed to be used together with the lpdomatic or directomatic\n",
	 "# backend filter script.  For more information, see:\n#\n",
	 "# Documentation: http://www.linuxprinting.org/lpd-doc.html\n",
	 "#                http://www.linuxprinting.org/direct-doc.html\n",
	 "# Driver `$driver': http://www.linuxprinting.org/show_driver.cgi?driver=$driver\n",
	 "# $make $model: http://www.linuxprinting.org/show_printer.cgi?recnum=$poid\n\n",
	 
	 
	 "# \"\$postpipe\" is a command to pipe the printer data to somewhere on the\n",
	 "# network or, in case of Direct-O-Matic, to a local printer port (parallel,\n",
	 "# serial, or USB).  Uncomment/modify a line you like. For local printers\n",
	 "# under LPD-O-Matic this doesn't apply.\n",
	 "#\n",
	 "# Netware users might stick something here like:\n",
	 "#\n",
	 "# \$postpipe = '| nprint -U guest -S net -q foo1 -';\n",
	 "#\n",
	 "# Remote LPD printers should be done using rlpr.  The if= isn't run\n",
	 "# with any arguments locally, so you have to set up lpdomatic printing\n",
	 "# to a local printer on /dev/null, and set this to *really* send the\n",
	 "# job over the network.\n",
	 "#\n",
	 "# \$postpipe = '| rlpr -Premotequeue\@remotehost';\n",
	 "#\n",
	 "# Windows/SMB remote printers would use an smbprint command.\n",
	 "#\n",
	 "# Remote HP JetDirect network printers will usually work with either of:\n",
	 "#\n",
	 "# \$postpipe = '| nc -w 1 ipaddress 9100';\n",
	 "# \$postpipe = '| rlpr -Praw\@ipaddress';\n",
	 "#\n",
	 "# Note the \"-w 1\" in the \"nc\" command line, it makes \"nc\" exiting\n",
	 "# immediately after the data is tranferred to the printer.\n",
	 "#\n",
	 "# To print on local printers with Direct-O-Matic use the \"cat\" command:\n",
	 "#\n",
	 "# \$postpipe = '| cat > /dev/lp0';\n",
	 "# \$postpipe = '| cat > /dev/usb/lp0';\n",
	 "#\n",

	 "# Important is to remember the leading | symbol.\n\n",

	 @datablob
	 );
    
    return @ppd;

}


#####################
# CUPS stuff
#

## Set this whenever you change the getcupsppd code!!!!
# NOT USED!!!
#my $cupsomaticversion = $ver;

# Return a PPD for CUPS and the cupsomatic script.  Built from the
# standard data; you must call getdat() first.

# This function will probably removed later on and only PPD-O-Matic PPD
# files will be used.

sub getcupsppd {
    my ($db) = @_;
    die "you need to call getdat first!\n" 
	if (!defined($db->{'dat'}));

    # Encase data for inclusion in PPD file
    my @datablob;
    push(@datablob, 
"*% What follows is a dumped representation of the internal Perl data
*% structure representing one entry in the Linux Printing Database.
*% This is used by the backend filter to deal with the options. 
*%
");
    for(split('\n',$db->getascii())) {
	push(@datablob, "*% COMDATA #$_\n");
	}	

    # Construct various selectors for PPD file
    my @optionblob;
    
    my $dat = $db->{'dat'};
    
    for $arg (@{$dat->{'args'}}) {
	my $name = $arg->{'name'};
	my $type = $arg->{'type'};
	my $com  = $arg->{'comment'};
	my $default = $arg->{'default'};
	my $idx = $arg->{'idx'};
	
	if ($type eq 'enum') {
	    # Skip zero or one choice arguments (except "PageSize", a PPD
	    # file without "PageSize" will break the CUPS environment).
	    if ((1 < scalar(@{$arg->{'vals'}})) ||
		($name eq "PageSize")) {
		push(@optionblob,
		     sprintf("\n*OpenUI *%s/%s: PickOne\n", $name, $com),
		     sprintf("*Default%s: %s\n", 
			     $name,
			     (defined($default) ? $default : 'Unknown')));
		if (!defined($default)) {
		    my $whr = sprintf("%s %s driver %s",
				      $dat->{'make'},
				      $dat->{'model'},
				      $dat->{'driver'});
		    warn "undefined default for $idx/$name on a $whr\n";
		}
	    
		my $v;
		for $v (@{$arg->{'vals'}}) {
		    my $psstr = "";
		    
		    if ($arg->{'style'} eq 'G') {
			# Ghostscript argument; offer up ps for insertion
			$psstr = sprintf($arg->{'proto'}, 
					 (defined($v->{'driverval'})
					  ? $v->{'driverval'}
					  : $v->{'value'}));
		    }
		    push(@optionblob,
			 sprintf("*%s %s/%s: \"$psstr\"\n", 
				 $name, $v->{'value'}, $v->{'comment'}));
		}
		
		push(@optionblob,
		     sprintf("*CloseUI: *%s\n", $name));
		if ($name eq "PageSize") {
		    push (@optionblob, "\@\@PAPERDIM\@\@");
		}
	    }
	    
	} elsif ($type eq 'bool') {
	    my $name = $arg->{'name'};
	    my $namef = $arg->{'name_false'};
	    my $defstr = ($default ? 'True' : 'False');
	    my $psstr = "";
	    if ($arg->{'style'} eq 'G') {
		# Ghostscript argument
		$psstr = $arg->{'proto'};
	    }
	    if (!defined($default)) { 
		$defstr = 'Unknown';
	    }
	    push(@optionblob,
		 sprintf("\n*OpenUI *%s/%s: Boolean\n", $name, $com),
		 sprintf("*Default%s: $defstr\n", $name),
		 sprintf("*%s True/%s: \"$psstr\"\n", $name, $name),
		 sprintf("*%s False/%s: \"\"\n", $name, $namef),
		 sprintf("*CloseUI: *%s\n", $name));
	    
	} elsif ($type eq 'int') {
	    
	    # max, min, and a few in between?
	    
	} elsif ($type eq 'float') {
	    
	    # max, min, and a few in between?
	    
	}
	
    }

    my $paperdim;		# computed as side effect of PageSize
    if (! $dat->{'args_byname'}{'PageSize'} ) {
	
	# This is a problem, since CUPS segfaults on PPD files without
	# a default PageSize set.  Indeed, the PPD spec requires a
	# PageSize clause.
	
	# GhostScript does not understand "/PageRegion[...]", therefore
	# we use "/PageSize[...]" in the "*PageRegion" option here, in
	# addition, for most modern PostScript interpreters "PageRegion"
	# is the same as "PageSize".

	push(@optionblob, <<EOFPGSZ);

*% This is fake. We have no information on how to
*% set the pagesize for this driver in the database. To
*% prevent PPD users from blowing up, we must provide a
*% default pagesize value.

*OpenUI *PageSize/Media Size: PickOne
*OrderDependency: 10 AnySetup *PageSize
*DefaultPageSize: Letter
*PageSize Letter/Letter: "<</PageSize[612 792]/ImagingBBox null>>setpagedevice"
*PageSize Legal/Legal: "<</PageSize[612 1008]/ImagingBBox null>>setpagedevice"
*PageSize A4/A4: "<</PageSize[595 842]/ImagingBBox null>>setpagedevice"
*CloseUI: *PageSize

*OpenUI *PageRegion: PickOne
*OrderDependency: 10 AnySetup *PageRegion
*DefaultPageRegion: Letter
*PageRegion Letter/Letter: "<</PageSize[612 792]/ImagingBBox null>>setpagedevice"
*PageRegion Legal/Legal: "<</PageSize[612 1008]/ImagingBBox null>>setpagedevice"
*PageRegion A4/A4: "<</PageSize[595 842]/ImagingBBox null>>setpagedevice"
*CloseUI: *PageRegion

*DefaultImageableArea: Letter
*ImageableArea Letter/Letter:	"0 0 612 792"
*ImageableArea Legal/Legal:	"0 0 612 1008"
*ImageableArea A4/A4:	"0 0 595 842"

*DefaultPaperDimension: Letter
*PaperDimension Letter/Letter:	"612 792"
*PaperDimension Legal/Legal:	"612 1008"
*PaperDimension A4/A4:	"595 842"

EOFPGSZ

    } else {
	# We *do* have a page size argument; construct
	# PageRegion, ImageableArea, and PaperDimension clauses from it.
	# Arguably this is all backwards, but what can you do! ;)

	my @pageregion;
	my @imageablearea;
	my @paperdimension;

	push(@pageregion,
	     "*OpenUI *PageRegion: PickOne
*OrderDependency: 10 AnySetup *PageRegion
*DefaultPageRegion: $dat->{'args_byname'}{'PageSize'}{'default'}");
	push(@imageablearea, 
	     "*DefaultImageableArea: $dat->{'args_byname'}{'PageSize'}{'default'}");
	push(@paperdimension, 
	     "*DefaultPaperDimension: $dat->{'args_byname'}{'PageSize'}{'default'}");

	for (@{$dat->{'args_byname'}{'PageSize'}{'vals'}}) {
	    my $name = $_->{'value'}; # in a PPD, the value is the PPD 
	                              # name...
	    my $comment = $_->{'comment'};

	    # In modern PostScript interpreters "PageRegion" and "PageSize"
	    # are the same option, so we fill in the "PageRegion" the same
	    # way as the "PageSize" choices.
	    if ($dat->{'args_byname'}{'PageSize'}{'style'} eq 'G') {
		# Ghostscript argument; offer up ps for insertion
		$psstr = sprintf($dat->{'args_byname'}{'PageSize'}{'proto'},
				 (defined($_->{'driverval'})
				  ? $_->{'driverval'}
				  : $_->{'value'}));
	    } else {
		$psstr = "";
	    }
	    push(@pageregion,
		 sprintf("*PageRegion %s/%s: \"$psstr\"", 
			 $_->{'value'}, $_->{'comment'}));
	    # Here we have to fill in the absolute sizes of the papers. We
	    # consult a table when we could not read the sizes out of the
	    # choices of the "PageSize" option.
	    my $size = $_->{'driverval'};
	    my $value = $_->{'value'};
	    if ($size !~ /^\s*\d+\s+\d+\s*$/) { 
		# 2 positive integers separated by whitespace
		$size = getpapersize($value);
	    }
	    push(@imageablearea,
		 "*ImageableArea $name/$comment: \"0 0 $size\"");
	    push(@paperdimension,
		 "*PaperDimension $name/$comment: \"$size\"");
	}

	push(@pageregion,
	     "*CloseUI: *PageRegion");


	$paperdim = join("\n", 
			 ("", @pageregion, "", @imageablearea, "",
			  @paperdimension, ""));
    }

    my @others;

    # *pnpFoo are KUPS extensions.  There is actually a PPD ieee probe
    # string value already, but they didn't use that for whatever
    # reason...
    if (defined($dat->{'pnp_mfg'})) {
	push(@others, "*pnpManufacturer: \"", $dat->{'pnp_mfg'}, "\"\n");
	
    }
    if (defined($dat->{'pnp_mdl'})) {
	push(@others, "*pnpModel: \"", $dat->{'pnp_mdl'}, "\"\n");
	
    }
    if (defined($dat->{'pnp_cmd'})) {
	push(@others, "*pnpCmd: \"", $dat->{'pnp_cmd'}, "\"\n");
	
    }
    if (defined($dat->{'pnp_des'})) {
	push(@others, "*pnpDescr: \"", $dat->{'pnp_des'}, "\"\n");
    }
    
    my $headcomment =
"*% For information on using this, and to obtain the required backend
*% script, consult http://www.linuxprinting.org/cups-doc.html
*%
*% CUPS-O-MATIC generated this PPD file.  It is for use with the CUPS 
*% printing system and the \"cupsomatic\" backend filter script.  These
*% two files work together to support the use of arbitrary free
*% software drivers with CUPS, replete with basic support for
*% driver-provided options.";

    my $blob = join('',@datablob);
    my $opts = join('',@optionblob);
    my $otherstuff = join('',@others);
    $driver =~ m!(^(.{1,5}))!;
    my $shortname = uc($1);
    my $model = $dat->{'model'};
    my $make = $dat->{'make'};
    my $filename = join('-',($dat->{'make'},
			     $dat->{'model'},
			     $dat->{'driver'},
			     "cups"));;
    $filename =~ s![ /]!_!g;
    my $longname = "$filename.ppd";

    my $drivername = $dat->{'driver'};
    
    # evil special case.
    $drivername = "stp-4.0" if $drivername eq 'stp';

    my $nickname = "$make $model, Foomatic + $drivername";

    my $tmpl = get_tmpl();
    $tmpl =~ s!\@\@HEADCOMMENT\@\@!$headcomment!g;
    $tmpl =~ s!\@\@MODEL\@\@!$model!g;
    $tmpl =~ s!\@\@NICKNAME\@\@!$nickname!g;
    $tmpl =~ s!\@\@MAKE\@\@!$make!g;
    $tmpl =~ s!\@\@SAVETHISAS\@\@!$longname!g;
    $tmpl =~ s!\@\@NUMBER\@\@!$shortname!g;
    $tmpl =~ s!\@\@OTHERSTUFF\@\@!$otherstuff!g;
    $tmpl =~ s!\@\@OPTIONS\@\@!$opts!g;
    $tmpl =~ s!\@\@COMDATABLOB\@\@!$blob!g;
    #$tmpl =~ s!\@\@PAPERDIMENSION\@\@!$paperdim!g;
    $tmpl =~ s!\@\@PAPERDIMENSION\@\@!!g;
    $tmpl =~ s!\@\@PAPERDIM\@\@!$paperdim!g;
    
    return ($tmpl);
}


#####################
# Generic PPD stuff
#

## Set this whenever you change the getgenericppd code!!!!
# NOT USED!!!
#my $ppdomaticversion = $ver;

# Return a generic Adobe-compliant PPD for the filter scripts for all
# spoolers.  Built from the standard data; you must call getdat()
# first.
sub getgenericppd {
    my ($db) = @_;
    die "you need to call getdat first!\n" 
	if (!defined($db->{'dat'}));

    # Encase data for inclusion in PPD file
    my @datablob;
    if (1) {
	push(@datablob, 
"*% What follows is a dumped representation of the internal Perl data
*% structure representing one entry in the Linux Printing Database.
*% This can be used by frontends to give advanced features which are
*% beyond the possibilities which can be defined by Adobe-compliant PPDs.
*% The lines are comment lines, so that programs which require 
*% Adobe-compliant PPD files can handle this file. They simply ignore
*% this additional information.
*%
");
	for(split('\n',$db->getascii())) {
	    push(@datablob, "*% COMDATA #$_\n");
	    }	
    }

    # Construct various selectors for PPD file
    my @optionblob;
    
    my $dat = $db->{'dat'};
    
    for $arg (@{$dat->{'args'}}) {
	my $name = $arg->{'name'};
	my $type = $arg->{'type'};
	my $com  = $arg->{'comment'};
	my $default = $arg->{'default'};
	my $idx = $arg->{'idx'};
	
	if ($type eq 'enum') {
	    # Skip zero or one choice arguments (except "PageSize", a PPD
	    # file without "PageSize" will break the CUPS environment).
	    if ((1 < scalar(@{$arg->{'vals'}})) ||
		($name eq "PageSize")) {
		push(@optionblob,
		     sprintf("\n*OpenUI *%s/%s: PickOne\n", $name, $com),
		     sprintf("*Default%s: %s\n", 
			     $name,
			     (defined($default) ? $default : 'Unknown')));
		if (!defined($default)) {
		    my $whr = sprintf("%s %s driver %s",
				      $dat->{'make'},
				      $dat->{'model'},
				      $dat->{'driver'});
		    warn "undefined default for $idx/$name on a $whr\n";
		}
	    
		my $v;
		for $v (@{$arg->{'vals'}}) {
		    my $psstr = "";
		    
		    if ($arg->{'style'} eq 'G') {
			# Ghostscript argument; offer up ps for insertion
			$psstr = sprintf($arg->{'proto'}, 
					 (defined($v->{'driverval'})
					  ? $v->{'driverval'}
					  : $v->{'value'}));
		    } else {
			# Option setting directive for Foomatic filter
			# 8 "%" because of several "sprintf applied to it
			# In the end stay 2 "%" to have a PostScript comment
			$psstr = sprintf("%%%%%%%% FoomaticOpt: %s=%s",
					 $name, $v->{'value'});
		    }
		    push(@optionblob,
			 sprintf("*%s %s/%s: \"$psstr\"\n", 
				 $name, $v->{'value'}, $v->{'comment'}));
		}
		
		push(@optionblob,
		     sprintf("*CloseUI: *%s\n", $name));
		if ($name eq "PageSize") {
		    push (@optionblob, "\@\@PAPERDIM\@\@");
		}
	    }
	    
	} elsif ($type eq 'bool') {
	    my $name = $arg->{'name'};
	    my $namef = $arg->{'name_false'};
	    my $defstr = ($default ? 'True' : 'False');
	    my $psstr = "";
	    my $psstrf = "";
	    if ($arg->{'style'} eq 'G') {
		# Ghostscript argument
		$psstr = $arg->{'proto'};
	    } else {
		# Option setting directive for Foomatic filter
		# 8 "%" because of several "sprintf applied to it
		# In the end stay 2 "%" to have a PostScript comment
		$psstr = sprintf("%%%%%%%% FoomaticOpt: %s=True", $name);
		$psstrf = sprintf("%%%%%%%% FoomaticOpt: %s=False", $name);
	    }
	    if (!defined($default)) { 
		$defstr = 'Unknown';
	    }
	    push(@optionblob,
		 sprintf("\n*OpenUI *%s/%s: Boolean\n", $name, $com),
		 sprintf("*Default%s: $defstr\n", $name),
		 sprintf("*%s True/%s: \"$psstr\"\n", $name, $name),
		 sprintf("*%s False/%s: \"$psstrf\"\n", $name, $namef),
		 sprintf("*CloseUI: *%s\n", $name));
	    
	} elsif ($type eq 'int') {

	    # Real numerical options do not exist in the Adobe
	    # specification for PPD files. So we map the numerical
	    # options to enumerated options offering the minimum, the
	    # maximum, the default, and some values inbetween to the
	    # user.

	    my $min = $arg->{'min'};
	    my $max = $arg->{'max'};
	    my $second = $min + 1;
	    my $stepsize = 1;
	    if (($max - $min > 100) && ($name ne "Copies")) {
		# We don't want to have more than 1000 values, but when the
		# difference between min and max is more than 1000 we should
		# have at least 100 steps.
		my $mindesiredvalues = 10;
		my $maxdesiredvalues = 100;
		# Find the order of magnitude of the value range
		my $rangesize = $max - $min;
		my $log10 = log(10.0);
		my $rangeom = POSIX::floor(log($rangesize)/$log10);
		# Now find the step size
		my $trialstepsize = 10 ** $rangeom;
		my $numvalues = 0;
		while (($numvalues <= $mindesiredvalues) &&
		       ($trialstepsize > 2)) {
		    $trialstepsize /= 10;
		    $numvalues = $rangesize/$trialstepsize;
		}
		# Try to find a finer stepping
		$stepsize = $trialstepsize;
		$trialstepsize = $stepsize / 2;
		$numvalues = $rangesize/$trialstepsize;
		if ($numvalues <= $maxdesiredvalues) {
		    if ($stepsize > 20) { 
			$trialstepsize = $stepsize / 4;
			$numvalues = $rangesize/$trialstepsize;
		    }
		    if ($numvalues <= $maxdesiredvalues) {
			$trialstepsize = $stepsize / 5;
			$numvalues = $rangesize/$trialstepsize;
		    }
		    if ($numvalues <= $maxdesiredvalues) {
			$stepsize = $trialstepsize;
		    } else {
			$stepsize /= 2;
		    }
		}
		$numvalues = $rangesize/$stepsize;
		# We have the step size. Now we must find an appropriate
		# second value for the value list, so that it contains
		# the integer multiples of 10, 100, 1000, ...
		$second = $stepsize * POSIX::ceil($min / $stepsize);
		if ($second <= $min) {$second += $stepsize};
	    }
	    # Generate the choice list
	    my @choicelist;
	    push (@choicelist, $min);
	    if (($default < $second) && ($default > $min)) {
		push (@choicelist, $default);
	    }
	    my $item = $second;
	    while ($item < $max) {
		push (@choicelist, $item);
		if (($default < $item + $stepsize) && ($default > $item) &&
		    ($default < $max)) {
		    push (@choicelist, $default);
		}
		$item += $stepsize;
	    }
	    push (@choicelist, $max);

            # Add the option

	    # Skip zero or one choice arguments
	    if (1 < scalar(@choicelist)) {
		push(@optionblob,
		     sprintf("\n*OpenUI *%s/%s: PickOne\n", $name, $com),
		     sprintf("*Default%s: %s\n", 
			     $name,
			     (defined($default) ? $default : 'Unknown')));
		if (!defined($default)) {
		    my $whr = sprintf("%s %s driver %s",
				      $dat->{'make'},
				      $dat->{'model'},
				      $dat->{'driver'});
		    warn "undefined default for $idx/$name on a $whr\n";
		}
	    
		my $v;
		for $v (@choicelist) {
		    my $psstr = "";
		    
		    if ($arg->{'style'} eq 'G') {
			# Ghostscript argument; offer up ps for insertion
			$psstr = sprintf($arg->{'proto'}, $v);
		    } else {
			# Option setting directive for Foomatic filter
			# 8 "%" because of several "sprintf applied to it
			# In the end stay 2 "%" to have a PostScript comment
			$psstr = sprintf("%%%%%%%% FoomaticOpt: %s=%s",
					 $name, $v);
		    }
		    push(@optionblob,
			 sprintf("*%s %s/%s: \"$psstr\"\n", 
				 $name, $v, $v));
		}
		
		push(@optionblob,
		     sprintf("*CloseUI: *%s\n", $name));
	    }
	    
	} elsif ($type eq 'float') {
	    
	    my $min = $arg->{'min'};
	    my $max = $arg->{'max'};
	    # We don't want to have more than 500 values or less than 50
	    # values.
	    my $mindesiredvalues = 10;
	    my $maxdesiredvalues = 100;
	    # Find the order of magnitude of the value range
	    my $rangesize = $max - $min;
	    my $log10 = log(10.0);
	    my $rangeom = POSIX::floor(log($rangesize)/$log10);
	    # Now find the step size
	    my $trialstepsize = 10 ** $rangeom;
	    my $stepom = $rangeom; # Order of magnitude of stepsize,
	                           # needed for determining necessary number
	                           # of digits
	    my $numvalues = 0;
	    while ($numvalues <= $mindesiredvalues) {
		$trialstepsize /= 10;
		$stepom -= 1;
		$numvalues = $rangesize/$trialstepsize;
	    }
	    # Try to find a finer stepping
	    $stepsize = $trialstepsize;
	    my $stepsizeorig = $stepsize;
	    $trialstepsize = $stepsizeorig / 2;
	    $numvalues = $rangesize/$trialstepsize;
	    if ($numvalues <= $maxdesiredvalues) {
		$stepsize = $trialstepsize;
		$trialstepsize = $stepsizeorig / 4;
		$numvalues = $rangesize/$trialstepsize;
		if ($numvalues <= $maxdesiredvalues) {
		    $stepsize = $trialstepsize;
		    $trialstepsize = $stepsizeorig / 5;
		    $numvalues = $rangesize/$trialstepsize;
		    if ($numvalues <= $maxdesiredvalues) {
			$stepsize = $trialstepsize;
		    }
		}
	    }
	    $numvalues = $rangesize/$stepsize;
	    if ($stepsize < $stepsizeorig * 0.9) {$stepom -= 1;}
	    # Determine number of digits after the decimal point for
	    # formatting the output values.
	    my $digits = 0;
	    if ($stepom < 0) {
		$digits = - $stepom;
	    }
	    # We have the step size. Now we must find an appropriate
	    # second value for the value list, so that it contains
	    # the integer multiples of 10, 100, 1000, ...
	    $second = $stepsize * POSIX::ceil($min / $stepsize);
	    if ($second <= $min) {$second += $stepsize};
	    # Generate the choice list
	    my @choicelist;
	    my $choicestr =  sprintf("%.${digits}f", $min);
	    push (@choicelist, $choicestr);
	    if (($default < $second) && ($default > $min)) {
		$choicestr =  sprintf("%.${digits}f", $default);
		# Prevent values from entering twice because of rounding
		# inacuracy
		if ($choicestr ne $choicelist[$#choicelist]) {
		    push (@choicelist, $choicestr);
		}
	    }
	    my $item = $second;
	    my $i = 0;
	    while ($item < $max) {
		$choicestr =  sprintf("%.${digits}f", $item);
		# Prevent values from entering twice because of rounding
		# inacuracy
		if ($choicestr ne $choicelist[$#choicelist]) {
		    push (@choicelist, $choicestr);
		}
		if (($default < $item + $stepsize) && ($default > $item) &&
		    ($default < $max)) {
		    $choicestr =  sprintf("%.${digits}f", $default);
		    # Prevent values from entering twice because of rounding
		    # inacuracy
		    if ($choicestr ne $choicelist[$#choicelist]) {
			push (@choicelist, $choicestr);
		    }
		}
		$i += 1;
		$item = $second + $i * $stepsize;
	    }
	    $choicestr =  sprintf("%.${digits}f", $max);
	    # Prevent values from entering twice because of rounding
	    # inacuracy
	    if ($choicestr ne $choicelist[$#choicelist]) {
		push (@choicelist, $choicestr);
	    }

            # Add the option

	    # Skip zero or one choice arguments
	    if (1 < scalar(@choicelist)) {
		push(@optionblob,
		     sprintf("\n*OpenUI *%s/%s: PickOne\n", $name, $com),
		     sprintf("*Default%s: %s\n", 
			     $name,
			     (defined($default) ? 
			      sprintf("%.${digits}f", $default) : 'Unknown')));
		if (!defined($default)) {
		    my $whr = sprintf("%s %s driver %s",
				      $dat->{'make'},
				      $dat->{'model'},
				      $dat->{'driver'});
		    warn "undefined default for $idx/$name on a $whr\n";
		}
	    
		my $v;
		for $v (@choicelist) {
		    my $psstr = "";
		    if ($arg->{'style'} eq 'G') {
			# Ghostscript argument; offer up ps for insertion
			$psstr = sprintf($arg->{'proto'}, $v);
		    } else {
			# Option setting directive for Foomatic filter
			# 8 "%" because of several "sprintf applied to it
			# In the end stay 2 "%" to have a PostScript comment
			$psstr = sprintf("%%%%%%%% FoomaticOpt: %s=%s",
					 $name, $v);
		    }
		    push(@optionblob,
			 sprintf("*%s %s/%s: \"$psstr\"\n", 
				 $name, $v, $v));
		}
		
		push(@optionblob,
		     sprintf("*CloseUI: *%s\n", $name));
	    }
        }
    }

    my $paperdim = "";		# computed as side effect of PageSize
    if (! $dat->{'args_byname'}{'PageSize'} ) {
	
	# This is a problem, since CUPS segfaults on PPD files without
	# a default PageSize set.  Indeed, the PPD spec requires a
	# PageSize clause.
	
	# GhostScript does not understand "/PageRegion[...]", therefore
	# we use "/PageSize[...]" in the "*PageRegion" option here, in
	# addition, for most modern PostScript interpreters "PageRegion"
	# is the same as "PageSize".

	push(@optionblob, <<EOFPGSZ);

*% This is fake. We have no information on how to
*% set the pagesize for this driver in the database. To
*% prevent PPD users from blowing up, we must provide a
*% default pagesize value.

*OpenUI *PageSize/Media Size: PickOne
*OrderDependency: 10 AnySetup *PageSize
*DefaultPageSize: Letter
*PageSize Letter/Letter: "<</PageSize[612 792]/ImagingBBox null>>setpagedevice"
*PageSize Legal/Legal: "<</PageSize[612 1008]/ImagingBBox null>>setpagedevice"
*PageSize A4/A4: "<</PageSize[595 842]/ImagingBBox null>>setpagedevice"
*CloseUI: *PageSize

*OpenUI *PageRegion: PickOne
*OrderDependency: 10 AnySetup *PageRegion
*DefaultPageRegion: Letter
*PageRegion Letter/Letter: "<</PageSize[612 792]/ImagingBBox null>>setpagedevice"
*PageRegion Legal/Legal: "<</PageSize[612 1008]/ImagingBBox null>>setpagedevice"
*PageRegion A4/A4: "<</PageSize[595 842]/ImagingBBox null>>setpagedevice"
*CloseUI: *PageRegion

*DefaultImageableArea: Letter
*ImageableArea Letter/Letter:	"0 0 612 792"
*ImageableArea Legal/Legal:	"0 0 612 1008"
*ImageableArea A4/A4:	"0 0 595 842"

*DefaultPaperDimension: Letter
*PaperDimension Letter/Letter:	"612 792"
*PaperDimension Legal/Legal:	"612 1008"
*PaperDimension A4/A4:	"595 842"

EOFPGSZ

    } else {
	# We *do* have a page size argument; construct
	# PageRegion, ImageableArea, and PaperDimension clauses from it.
	# Arguably this is all backwards, but what can you do! ;)

	my @pageregion;
	my @imageablearea;
	my @paperdimension;

	push(@pageregion,
	     "*OpenUI *PageRegion: PickOne
*OrderDependency: 10 AnySetup *PageRegion
*DefaultPageRegion: $dat->{'args_byname'}{'PageSize'}{'default'}");
	push(@imageablearea, 
	     "*DefaultImageableArea: $dat->{'args_byname'}{'PageSize'}{'default'}");
	push(@paperdimension, 
	     "*DefaultPaperDimension: $dat->{'args_byname'}{'PageSize'}{'default'}");

	for (@{$dat->{'args_byname'}{'PageSize'}{'vals'}}) {
	    my $name = $_->{'value'}; # in a PPD, the value is the PPD 
	                              # name...
	    my $comment = $_->{'comment'};

	    # In modern PostScript interpreters "PageRegion" and "PageSize"
	    # are the same option, so we fill in the "PageRegion" the same
	    # way as the "PageSize" choices.
	    if ($dat->{'args_byname'}{'PageSize'}{'style'} eq 'G') {
		# Ghostscript argument; offer up ps for insertion
		$psstr = sprintf($dat->{'args_byname'}{'PageSize'}{'proto'},
				 (defined($_->{'driverval'})
				  ? $_->{'driverval'}
				  : $_->{'value'}));
	    } else {
		# Option setting directive for Foomatic filter
		# 8 "%" because of several "sprintf applied to it
		# In the end stay 2 "%" to have a PostScript comment
		$psstr = sprintf("%%%%%%%% FoomaticOpt: PageSize=%s",
				 $_->{'value'});
	    }
	    push(@pageregion,
		 sprintf("*PageRegion %s/%s: \"$psstr\"", 
			 $_->{'value'}, $_->{'comment'}));
	    # Here we have to fill in the absolute sizes of the papers. We
	    # consult a table when we could not read the sizes out of the
	    # choices of the "PageSize" option.
	    my $size = $_->{'driverval'};
	    my $value = $_->{'value'};
	    if ($size !~ /^\s*\d+\s+\d+\s*$/) { 
		# 2 positive integers separated by whitespace
		$size = getpapersize($value);
	    }
	    push(@imageablearea,
		 "*ImageableArea $name/$comment: \"0 0 $size\"");
	    push(@paperdimension,
		 "*PaperDimension $name/$comment: \"$size\"");
	}

	push(@pageregion,
	     "*CloseUI: *PageRegion");


	$paperdim = join("\n", 
			 ("", @pageregion, "", @imageablearea, "",
			  @paperdimension, ""));
    }

    my @others;

    my $headcomment =
"*% For information on using this, and to obtain the required backend
*% script, consult http://www.linuxprinting.org/ppd-doc.html
*%
*% PPD-O-MATIC generated this PPD file. It is for use with all programs 
*% and environments which use PPD files for dealing with printer capabilty
*% information. The printer must be configured with a Foomatic backend
*% filter script. This file and the backend filter script work together to
*% support PPD-controlled printer driver option access with arbitrary free 
*% software printer drivers and printing spoolers.";

    my $blob = join('',@datablob);
    my $opts = join('',@optionblob);
    my $otherstuff = join('',@others);
    $driver =~ m!(^(.{1,5}))!;
    my $shortname = uc($1);
    my $model = $dat->{'model'};
    my $make = $dat->{'make'};
    my $filename = join('-',($dat->{'make'},
			     $dat->{'model'},
			     $dat->{'driver'},
			     "ppd"));;
    $filename =~ s![ /]!_!g;
    my $longname = "$filename.ppd";

    my $drivername = $dat->{'driver'};
    
    # evil special case.
    $drivername = "stp-4.0" if $drivername eq 'stp';

    my $nickname = "$make $model, Foomatic + $drivername";

    my $tmpl = get_tmpl();
    $tmpl =~ s!\@\@HEADCOMMENT\@\@!$headcomment!g;
    $tmpl =~ s!\@\@MODEL\@\@!$model!g;
    $tmpl =~ s!\@\@NICKNAME\@\@!$nickname!g;
    $tmpl =~ s!\@\@MAKE\@\@!$make!g;
    $tmpl =~ s!\@\@SAVETHISAS\@\@!$longname!g;
    $tmpl =~ s!\@\@NUMBER\@\@!$shortname!g;
    $tmpl =~ s!\@\@OTHERSTUFF\@\@!$otherstuff!g;
    $tmpl =~ s!\@\@OPTIONS\@\@!$opts!g;
    $tmpl =~ s!\@\@COMDATABLOB\@\@!$blob!g;
    $tmpl =~ s!\@\@PAPERDIMENSION\@\@!!g;
    $tmpl =~ s!\@\@PAPERDIM\@\@!$paperdim!g;
    
    return ($tmpl);
}


# Utility function; returns content of a URL
sub getpage {
    my ($this, $url, $dontdie) = @_;

    use LWP::UserAgent;
    my $ua = LWP::UserAgent->new();
    $ua->agent("PHTDBPUB/$ver ($0)");
    $ua->timeout([30]);
    
    # should call ->proxy() here if needed...

    my $request = $ua->request(new HTTP::Request('GET', $url));

    if ($request->is_error()) {
	if ($dontdie) {
	    return undef;
	} else {
	    die ("http error: " . $request->status_line . "\n");
	}
    }

    my $page = $request->content;

    return $page;
}

# Prepare strings for being part of an HTML document by, converting
# "<" to "&lt;", ">" to "&gt;", and "&" to "&amp;"
sub htmlify {
    my $str = $_[0];
    $str =~ s!&!&amp;!g;
    $str =~ s/\</\&lt;/g;
    $str =~ s/\>/\&gt;/g;
    return $str;
}

# Get documentation for the printer/driver pair to print out. For
# "Execution Details" section of driver web pages of linuxprinting.org

sub getexecdocs {

    my ($this) = $_[0];

    my $dat = $this->{'dat'};

    my @docs;
    
    # Construct the proper command line.
    my $commandline = htmlify($dat->{'cmd'});

    if ($commandline eq "") {return ();}

    my @letters = qw/A B C D E F G H I J K L M Z/;
    my $spot;
    
    for $spot (@letters) {
	
	if($commandline =~ m!\%$spot!) {

	    my $arg;
	  argument:
	    for $arg (sort { $a->{'order'} <=> $b->{'order'} } 
		      @{$dat->{'args'}}) {
		
		# Only do arguments that go in this spot
		next argument if ($arg->{'spot'} ne $spot);
		# PJL arguments are not inserted at a spot in the command
		# line
		next argument if ($arg->{'style'} eq 'J');
		
		my $name = $arg->{'name'};
		my $varname = $arg->{'varname'};
		my $cmd = htmlify($arg->{'proto'});
		my $comment = htmlify($arg->{'comment'});
		my $placeholder = "</TT><I>&lt;$name&gt;</I><TT>";
		my $default = $arg->{'default'};
		my $type = $arg->{'type'};
		my $cmdvar = "";
		my $gsarg1 = "";
		my $gsarg2 = "";
		if ($arg->{'style'} eq 'G') {
		    $gsarg1 = ' -c "';
		    $gsarg2 = '"';
		    $cmd =~ s/\"/\\\"/g;
		}
		#my $leftbr = ($arg->{'required'} ? "" : "[");
		#my $rightbr = ($arg->{'required'} ? "" : "]");
		my $leftbr = "";
		my $rightbr = "";
	
		if ($type eq 'bool') {
		    $cmdvar = "$leftbr$gsarg1$cmd$gsarg2$rightbr";
		} elsif ($type eq 'int' or $type eq 'float') {
		    $cmdvar = sprintf("$leftbr$gsarg1$cmd$gsarg2$rightbr",$placeholder);
		} elsif ($type eq 'enum') {
		    my $val;
		    if ($val=valbyname($arg,$default)) {
			$cmdvar = sprintf("$leftbr$gsarg1$cmd$gsarg2$rightbr",
					  $placeholder);
		    }
		}
		
		# Insert the processed argument in the commandline
		# just before the spot marker.
		$cmdvar =~ s!^\[\ !\ \[!;
		$commandline =~ s!\%$spot!$cmdvar\%$spot!;
	    }
	    
	    # Remove the letter marker from the commandline
	    $commandline =~ s!\%$spot!!;
	    
	}
	
    }

    $dat->{'excommandline'} = $commandline;

    push(@docs, "<B>Command Line</B><P>");
    push(@docs, "<BLOCKQUOTE><TT>$commandline</TT></BLOCKQUOTE><P>");

    my ($arg, @doctmp);
    my @pjlcommands = ();
  argt:
    for $arg (sort { $a->{'order'} <=> $b->{'order'} } 
	      @{$dat->{'args'}}) {

	my $name = $arg->{'name'};
	my $cmd = htmlify($arg->{'proto'});
	my $comment = htmlify($arg->{'comment'});
	my $placeholder = "</TT><I>&lt;$name&gt;</I><TT>";
	if ($arg->{'style'} eq 'J') {
	    $cmd = "\@PJL $cmd";
	    push (@pjlcommands, sprintf($cmd, $placeholder));
	}

	my $default = htmlify($arg->{'default'});
	my $type = $arg->{'type'};
	
	my $required = ($arg->{'required'} ? " required" : "n optional");
	my $pjl = ($arg->{'style'} eq 'J' ? "PJL " : "");

	if ($type eq 'bool') {
	    my $name_false = $arg->{'name_false'};
	    push(@doctmp,
		 "<DL><DT><I>$name</I></DT>",
		 "<DD>A$required boolean ${pjl}argument meaning $name if present or $name_false if not.<BR>",
		 "$comment<BR>",
		 "Prototype: <TT>$cmd</TT><BR>",
		 "Default: ", $default ? "True" : "False",
		 "</DD></DL><P>"
		 );

	} elsif ($type eq 'int' or $type eq 'float') {
	    my $max = (defined($arg->{'max'}) ? $arg->{'max'} : "none");
	    my $min = (defined($arg->{'min'}) ? $arg->{'min'} : "none");
	    push(@doctmp,
		 "<DL><DT><I>$name</I></DT>",
		 "<DD>A$required $type ${pjl}argument.<BR>",
		 "$comment<BR>",
		 "Prototype: <TT>", sprintf($cmd, $placeholder),
		 "</TT><BR>",
		 "Default: <TT>$default</TT><BR>",
		 "Range: <TT>$min &lt;= $placeholder &lt;= $max</TT>",
		 "</DD></DL><P>"
		 );

	} elsif ($type eq 'enum') {
	    my ($val, $defstr);
	    my (@choicelist) = ();

	    for $val (@{$arg->{'vals'}}) {
		my ($value, $comment, $driverval) = 
		    ($val->{'value'},
		     $val->{'comment'},
		     $val->{'driverval'});

		if (defined($driverval)) {
		    if ($driverval eq "") {
			push(@choicelist,
			     "<LI>$value: $comment (<TT>$placeholder</TT> is left blank)</LI>");
		    } else {
			push(@choicelist,
			     "<LI>$value: $comment (<TT>$placeholder</TT> is '<TT>$driverval</TT>')</LI>");
		    }
		} else {
		    push(@choicelist,
			 "<LI>$value: $comment (<TT>$placeholder</TT> is '<TT>$value</TT>')</LI>");
		}
	    }

	    push(@doctmp,
		 "<DL><DT><I>$name</I></DT>",
		 "<DD>A$required enumerated choice ${pjl}argument.<BR>",
		 "$comment<BR>",
		 "Prototype: <TT>", sprintf($cmd, $placeholder),
		 "</TT><BR>",
		 "Default: $default",
		 "<UL>", 
		 join("",sort { normalizename($a) cmp normalizename($b) }
		      @choicelist), 
		 "</UL></DD></DL><P>"
		 );

	}
    }

    # Instructions for PJL commands
    if (($#pjlcommands > -1) && (defined($dat->{'pjl'}))) {
    #if (($#pjlcommands > -1)) {
	my @pjltmp;
	push(@pjltmp,
	     "PJL arguments are not put into the command line, they must be put into a PJL header which is prepended to the actual job data which is generated by the command line shown above and sent to the printer. Always when such a header is applied one has also to append a PJL command to close the job. So a complete job looks as follows:<BLOCKQUOTE>",
	     "<I>&lt;ESC&gt;</I>",
	     "<TT>%-12345X\@PJL JOB NAME=\"</TT>",
	     "<I>&lt;A job name&gt;</I>",
	     "<TT>\"</TT><BR>");
	for $command (@pjlcommands) {
	    push(@pjltmp,
		 "<TT>$command</TT><BR>");
	}
	push(@pjltmp,
	     "<I>&lt;The job data&gt;</I><BR>",
	     "<TT>\@PJL EOJ</TT></BLOCKQUOTE><P>",
	     "<I>&lt;ESC&gt;</I>",
	     ": This is the ",
	     "<I>ESC</I>",
	     " character, ASCII code 27.<BR>",
	     "<I>&lt;A job name&gt;</I>",
	     ": The job name can be chosen arbitrarily, some printers show it on their front panel displays.<P>",
	     "It is not required to give the PJL arguments, you can leave out some of them or you can even send only the job data without PJL header and PJL end-of-job mark.<P>");
	push(@docs, "<B>PJL</B><P>");
	push(@docs, @pjltmp);
    }

    push(@docs, "<B>Options</B><P>");

    push(@docs, @doctmp);

    return @docs;
   
}

# Get a shorter summary documentation thing.  This appears on the
# driver pages, for example.
#
# About as obsolete as getexecdocs...
sub get_summarydocs {
    my ($this) = $_[0];

    my $dat = $this->{'dat'};

    my @docs;

    for $arg (@{$dat->{'args'}}) {
	my ($name,
	    $required,
	    $type,
	    $comment,
	    $spot,
	    $default) = ($arg->{'name'},
			 $arg->{'required'},
			 $arg->{'type'},
			 $arg->{'comment'},
			 $arg->{'spot'},
			 $arg->{'default'});
	
	my $reqstr = ($required ? " required" : "n optional");
	push(@docs,
	     "Option `$name':\n  A$reqstr $type argument.\n  $comment\n");

	push(@docs,
	     "  This option corresponds to a PJL command.\n") 
	    if ($spot eq 'Y');
	
	if ($type eq 'bool') {
	    if (defined($default)) {
		my $defstr = ($default ? "True" : "False");
		push(@docs, "  Default: $defstr\n");
	    }
	    push(@docs, "  Example (true): `$name'\n");
	    push(@docs, "  Example (false): `no$name'\n");
	} elsif ($type eq 'enum') {
	    push(@docs, "  Possible choices:\n");
	    my $exarg;
	    for (@{$arg->{'vals'}}) {
		my ($choice, $comment) = ($_->{'value'}, $_->{'comment'});
		push(@docs, "   * $choice: $comment\n");
		$exarg=$choice;
	    }
	    if (defined($default)) {
		push(@docs, "  Default: $default\n");
	    }
	    push(@docs, "  Example: `$name=$exarg'\n");
	} elsif ($type eq 'int' or $type eq 'float') {
	    my ($max, $min) = ($arg->{'max'}, $arg->{'min'});
	    my $exarg;
	    if (defined($max)) {
		push(@docs, "  Range: $min <= x <= $max\n");
		$exarg=$max;
	    }
	    if (defined($default)) {
		push(@docs, "  Default: $default\n");
		$exarg=$default;
	    }
	    if (!$exarg) { $exarg=0; }
	    push(@docs, "  Example: `$name=$exarg'\n");
	}

	push(@docs, "\n");
    }

    return @docs;

}

# About as obsolete as the other docs functions.  Why on earth are
# there three, anyway?!
sub getdocs {
    my ($this) = $_[0];

    my $dat = $this->{'dat'};

    my @docs;

    for $arg (@{$dat->{'args'}}) {
	my ($name,
	    $required,
	    $type,
	    $comment,
	    $spot,
	    $default) = ($arg->{'name'},
			 $arg->{'required'},
			 $arg->{'type'},
			 $arg->{'comment'},
			 $arg->{'spot'},
			 $arg->{'default'});
	
	my $reqstr = ($required ? " required" : "n optional");
	push(@docs,
	     "Option `$name':\n  A$reqstr $type argument.\n  $comment\n");

	push(@docs,
	     "  This option corresponds to a PJL command.\n") 
	    if ($spot eq 'Y');
	
	if ($type eq 'bool') {
	    if (defined($default)) {
		my $defstr = ($default ? "True" : "False");
		push(@docs, "  Default: $defstr\n");
	    }
	    push(@docs, "  Example (true): `$name'\n");
	    push(@docs, "  Example (false): `no$name'\n");
	} elsif ($type eq 'enum') {
	    push(@docs, "  Possible choices:\n");
	    my $exarg;
	    for (@{$arg->{'vals'}}) {
		my ($choice, $comment) = ($_->{'value'}, $_->{'comment'});
		push(@docs, "   * $choice: $comment\n");
		$exarg=$choice;
	    }
	    if (defined($default)) {
		push(@docs, "  Default: $default\n");
	    }
	    push(@docs, "  Example: `$name=$exarg'\n");
	} elsif ($type eq 'int' or $type eq 'float') {
	    my ($max, $min) = ($arg->{'max'}, $arg->{'min'});
	    my $exarg;
	    if (defined($max)) {
		push(@docs, "  Range: $min <= x <= $max\n");
		$exarg=$max;
	    }
	    if (defined($default)) {
		push(@docs, "  Default: $default\n");
		$exarg=$default;
	    }
	    if (!$exarg) { $exarg=0; }
	    push(@docs, "  Example: `$name=$exarg'\n");
	}

	push(@docs, "\n");
    }

    return @docs;

}

# Find a choice value hash by name.
# Operates on old dat structure...
sub valbyname {
    my ($arg,$name) = @_;

    my $val;
    for $val (@{$arg->{'vals'}}) {
	return $val if (lc($name) eq lc($val->{'value'}));
    }

    return undef;
}

# replace numbers with fixed 6-digit number for ease of sorting
# ie: sort { normalizename($a) cmp normalizename($b) } @foo;
sub normalizename {
    my $n = $_[0];

    if ($n =~ m!(\d+)!) {
	my $num = sprintf("%06d", $1);
	$n =~ s!(\d+)!$num!;
    }
    return $n;
}


# PPD boilerplate template

sub get_tmpl_paperdimension {
    return <<ENDPDTEMPL;
*% Generic PaperDimension; evidently there was no normal PageSize argument

*DefaultPaperDimension: Letter
*PaperDimension Letter:	"612 792"
*PaperDimension Legal:	"612 1008"
*PaperDimension A4:	"595 842"
ENDPDTEMPL
}

sub get_tmpl {
    return <<ENDTMPL;
*PPD-Adobe: "4.3"
*%
\@\@HEADCOMMENT\@\@
*%
*% You may save this file as '\@\@SAVETHISAS\@\@'
*%
*%
*FormatVersion:	"4.3"
*FileVersion:	"1.1"
*LanguageVersion: English 
*LanguageEncoding: ISOLatin1
*PCFileName:	"COM\@\@NUMBER\@\@.PPD"
*Manufacturer:	"\@\@MAKE\@\@"
*Product:	"\@\@MODEL\@\@"
*cupsVersion:	1.0
*cupsManualCopies: True
*cupsModelNumber:  2
*cupsFilter:	"application/vnd.cups-postscript 0 cupsomatic"
*ModelName:     "\@\@MODEL\@\@"
*ShortNickName: "\@\@MODEL\@\@"
*NickName:      "\@\@NICKNAME\@\@"
*PSVersion:	"(3010.000) 550"
*PSVersion:	"(3010.000) 651"
*PSVersion:	"(3010.000) 652"
*LanguageLevel:	"3"
*ColorDevice:	True
*DefaultColorSpace: RGB
*FileSystem:	False
*Throughput:	"1"
*LandscapeOrientation: Plus90
*VariablePaperSize: False
*TTRasterizer:	Type42
\@\@OTHERSTUFF\@\@
 
\@\@OPTIONS\@\@

*% Generic boilerplate PPD stuff as standard PostScript fonts and so on

\@\@PAPERDIMENSION\@\@

*DefaultFont: Courier
*Font AvantGarde-Book: Standard "(001.006S)" Standard ROM
*Font AvantGarde-BookOblique: Standard "(001.006S)" Standard ROM
*Font AvantGarde-Demi: Standard "(001.007S)" Standard ROM
*Font AvantGarde-DemiOblique: Standard "(001.007S)" Standard ROM
*Font Bookman-Demi: Standard "(001.004S)" Standard ROM
*Font Bookman-DemiItalic: Standard "(001.004S)" Standard ROM
*Font Bookman-Light: Standard "(001.004S)" Standard ROM
*Font Bookman-LightItalic: Standard "(001.004S)" Standard ROM
*Font Courier: Standard "(002.004S)" Standard ROM
*Font Courier-Bold: Standard "(002.004S)" Standard ROM
*Font Courier-BoldOblique: Standard "(002.004S)" Standard ROM
*Font Courier-Oblique: Standard "(002.004S)" Standard ROM
*Font Helvetica: Standard "(001.006S)" Standard ROM
*Font Helvetica-Bold: Standard "(001.007S)" Standard ROM
*Font Helvetica-BoldOblique: Standard "(001.007S)" Standard ROM
*Font Helvetica-Narrow: Standard "(001.006S)" Standard ROM
*Font Helvetica-Narrow-Bold: Standard "(001.007S)" Standard ROM
*Font Helvetica-Narrow-BoldOblique: Standard "(001.007S)" Standard ROM
*Font Helvetica-Narrow-Oblique: Standard "(001.006S)" Standard ROM
*Font Helvetica-Oblique: Standard "(001.006S)" Standard ROM
*Font NewCenturySchlbk-Bold: Standard "(001.009S)" Standard ROM
*Font NewCenturySchlbk-BoldItalic: Standard "(001.007S)" Standard ROM
*Font NewCenturySchlbk-Italic: Standard "(001.006S)" Standard ROM
*Font NewCenturySchlbk-Roman: Standard "(001.007S)" Standard ROM
*Font Palatino-Bold: Standard "(001.005S)" Standard ROM
*Font Palatino-BoldItalic: Standard "(001.005S)" Standard ROM
*Font Palatino-Italic: Standard "(001.005S)" Standard ROM
*Font Palatino-Roman: Standard "(001.005S)" Standard ROM
*Font Symbol: Special "(001.007S)" Special ROM
*Font Times-Bold: Standard "(001.007S)" Standard ROM
*Font Times-BoldItalic: Standard "(001.009S)" Standard ROM
*Font Times-Italic: Standard "(001.007S)" Standard ROM
*Font Times-Roman: Standard "(001.007S)" Standard ROM
*Font ZapfChancery-MediumItalic: Standard "(001.007S)" Standard ROM
*Font ZapfDingbats: Special "(001.004S)" Standard ROM

\@\@COMDATABLOB\@\@
ENDTMPL
}

# Determine the paper width and height in points from a given paper size
# name. Used for the "PaperDimension" and "ImageableArea" entries in PPD
# files.
#
# The paper sizes in the list are all sizes known to GhostScript, all
# of GIMP-Print, all sizes of HPIJS, and some others found in the data
# of printer drivers.

sub getpapersize {
    my $papersize = lc(join('', @_));

    my $sizetable = {
	'lettersmall'      => '612 792',
	'letter'           => '612 792',
	'legal'            => '612 1008',
	'postcard'         => '283 416',
	'tabloid'          => '792 1224',
	'ledger'           => '1224 792',
	'tabloidextra'     => '864 1296',
	'superb'           => '936 1368',
	'statement'        => '396 612',
	'manual'           => '396 612',
	'halfletter'       => '396 612',
	'executive'        => '522 756',
	'archa'            => '648 864',
	'archb'            => '864 1296',
	'archc'            => '1296 1728',
	'archd'            => '1728 2592',
	'arche'            => '2592 3456',
	'b6-c4'            => '354 918',
	'c7-6'             => '229 459',
	'a4small'          => '595 842',
	'sra4'             => '637 907',
	'sra3'             => '907 1275',
	'sra2'             => '1275 1814',
	'sra1'             => '1814 2551',
	'sra0'             => '2551 3628',
	'ra4'              => '609 864',
	'ra3'              => '864 1218',
	'ra2'              => '1218 1729',
	'ra1'              => '1729 2437',
	'ra0'              => '2437 3458',
	'a10'              => '74 105',
	'a9'               => '105 148',
	'a8'               => '148 210',
	'a7'               => '210 297',
	'a6'               => '297 420',
	'a5'               => '420 595',
	'a4'               => '595 842',
	'a3'               => '842 1191',
	'a2'               => '1191 1684',
	'a1'               => '1684 2384',
	'a0'               => '2384 3370',
	'2a'               => '3370 4768',
	'4a'               => '4768 6749',
	'c10'              => '79 113',
	'c9'               => '113 161',
	'c8'               => '161 229',
	'c7'               => '229 323',
	'c6'               => '323 459',
	'c5'               => '459 649',
	'c4'               => '649 918',
	'c3'               => '918 1298',
	'c2'               => '1298 1836',
	'c1'               => '1836 2599',
	'c0'               => '2599 3676',
	'b10.*jis'         => '90 127',
	'b9.*jis'          => '127 180',
	'b8.*jis'          => '180 257',
	'b7.*jis'          => '257 362',
	'b6.*jis'          => '362 518',
	'b5.*jis'          => '518 727',
	'b4.*jis'          => '727 1029',
	'b3.*jis'          => '1029 1459',
	'b2.*jis'          => '1459 2063',
	'b1.*jis'          => '2063 2919',
	'b0.*jis'          => '2919 4127',
	'jis.*b10'         => '90 127',
	'jis.*b9'          => '127 180',
	'jis.*b8'          => '180 257',
	'jis.*b7'          => '257 362',
	'jis.*b6'          => '362 518',
	'jis.*b5'          => '518 727',
	'jis.*b4'          => '727 1029',
	'jis.*b3'          => '1029 1459',
	'jis.*b2'          => '1459 2063',
	'jis.*b1'          => '2063 2919',
	'jis.*b0'          => '2919 4127',
	'b10.*iso'         => '87 124',
	'b9.*iso'          => '124 175',
	'b8.*iso'          => '175 249',
	'b7.*iso'          => '249 354',
	'b6.*iso'          => '354 498',
	'b5.*iso'          => '498 708',
	'b4.*iso'          => '708 1000',
	'b3.*iso'          => '1000 1417',
	'b2.*iso'          => '1417 2004',
	'b1.*iso'          => '2004 2834',
	'b0.*iso'          => '2834 4008',
	'2b.*iso'          => '4008 5669',
	'4b.*iso'          => '5669 8016',
	'iso.*b10'         => '87 124',
	'iso.*b9'          => '124 175',
	'iso.*b8'          => '175 249',
	'iso.*b7'          => '249 354',
	'iso.*b6'          => '354 498',
	'iso.*b5'          => '498 708',
	'iso.*b4'          => '708 1000',
	'iso.*b3'          => '1000 1417',
	'iso.*b2'          => '1417 2004',
	'iso.*b1'          => '2004 2834',
	'iso.*b0'          => '2834 4008',
	'iso.*2b'          => '4008 5669',
	'iso.*4b'          => '5669 8016',
	'b10'              => '90 127',
	'b9'               => '127 180',
	'b8'               => '180 257',
	'b7'               => '257 362',
	'b6'               => '362 518',
	'b5'               => '518 727',
	'b4'               => '727 1029',
	'b3'               => '1029 1459',
	'b2'               => '1459 2063',
	'b1'               => '2063 2919',
	'b0'               => '2919 4127',
	'monarch'          => '279 540',
	'dl'               => '311 623',
	'com10'            => '297 684',
	'com.*10'          => '297 684',
	'hagaki'           => '283 420',
	'oufuku'           => '420 567',
	'kaku'             => '680 941',
	'long.*3'          => '340 666',
	'long.*4'          => '255 581',
	'flsa'             => '612 936',
	'flse'             => '648 936',
	'a2.*invit.*'      => '315 414',
	'photofullbleed'   => '298 440',
	'photo'            => '288 432',
	'roll'             => '612 0',
	'custom'           => '0 0'
	};

    # Remove prefixes which sometimes could appear
    $papersize =~ s/form_//;
    $papersize =~ s/na_//;

    # Check if we have a "<Width>x<Height>" format, assume the numbers are
    # given in inches
    if ($papersize =~ /(\d+)x(\d+)/) {
	my $w = $1 * 72;
	my $h = $2 * 72;
	return sprintf("%d %d", $w, $h);
    }

    # Check if we have a "w<Width>h<Height>" format, assume the numbers are
    # given in points
    if ($papersize =~ /w(\d+)h(\d+)/) {
	return "$1 $2";
    }

    # Check if we have a "w<Width>" format, assume roll paper with the given
    # width in points
    if ($papersize =~ /w(\d+)/) {
	return "$1 0";
    }

    # Check whether the paper size name is in the list above
    for $item (keys(%{$sizetable})) {
	if ($papersize =~ /$item/) {
	    return $sizetable->{$item};
	}
    }

    # This paper size is absolutely unknown, issue a warning
    warn "WARNING: Unknown paper size: $papersize!";
    return "0 0";
}

sub grove_from_xml {
    my ($xml) = @_;

    return undef if ! $xml;

    my $grove_builder = XML::Grove::Builder->new;
    my $parser = XML::Parser::PerlSAX->new ( Handler => $grove_builder );
    my $doc = $parser->parse ( Source => { String => $xml } );

    return $doc;
}

sub grove_from_filename {
    my ($filen) = @_;

    return undef if ! -f $filen;

    my $grove_builder = XML::Grove::Builder->new;
    my $parser = XML::Parser::PerlSAX->new ( Handler => $grove_builder );
    my $doc = $parser->parse ( Source => { System => $filen } );

    return $doc;
}

sub grove_to_xml {
    my $retval = $_[0]->as_canon_xml('Comments' => 1);
    $retval =~ s!<([\w =\"]+?)></\w+?>!<$1 />!g;
    return $retval;
}

sub pretty_xml {
    my $retval = $_[0];

    my $t = new XML::Twig(PrettyPrint => 'indented',
			  Comments => 'keep',
			  EmptyTags => 'normal');
    $t->parse($retval);
    $retval = $t->sprint() . "\n";

    return $retval;
}

# takes grove, a path, and an element name to cut off at that point
sub grove_prune {
    my ($grove, $path, $child) = (@_);
    my $subgrove;
    if ($path eq '/') {
	$subgrove = $grove;
    } else {
	$subgrove = $grove->at_path($path);
    }
    if (!defined($subgrove)) {
	warn "No such path $path in grove $grove passed to grove_prune!\n";
	return 0;
    }
    my @contents;
    for (@{$subgrove->{'Contents'}}) {
	push (@contents, $_)
	    unless ($_->{'Name'} eq $child);
    }
    $subgrove->{'Contents'} = \@contents;
    return 1;
}

# Clone a Grove element
sub grove_clone {
    my $grove = $_[0];

    # Formalizes into a document, so take first content thing to get
    # just the element back...
    return grove_from_xml(grove_to_xml($grove))->{'Contents'}[0];
}

# Get grove element value by path
sub grove_pathval {
    my ($grove, $path) = @_;
    
    my $retval = undef;
    if (defined($grove)) {
	my $elem = $grove->at_path($path);
	if (defined($elem)) {
	    $retval = $elem->as_string();
	}
    } else {
	warn "grove_pathval called on undefined \$grove for path '$path'!?\n";
    }
    
    return $retval;
}

# Get a parsed Grove struture for an XML object...
sub _get_object_grove {
    my ($this, $file, $dontwarn) = @_;

    print STDERR "Loading object $file\n" if $DEBUG;

    # First look at the in-memory cache
    if (!defined($this->{'grovecache'}{$file})) {

	print STDERR "  ...memory cache miss\n" if $DEBUG;

### CACHE ###
	if ($cachedir)
	{
	    # Then look at perl filecache
	    my $perlfile = "$cachedir/pcache/$file.perl";
	    my (@statperl) = stat $perlfile;
	    my (@statxml) = stat "$libdir/$file.xml";

	    # stat element 9 == mtime
	    if ($statperl[9] > $statxml[9]) {

		#print STDERR "  ...pcache hit\n";

		$this->{'grovecache'}{$file} = retrieve($perlfile)
		  or do {
		      warn "Error loading Storable $perlfile; using XML\n";
		      goto load_xml;
		  };

		if (0) {
		    # OK, Perl cachefile is newer
		    open PFILE, $perlfile or die "Unable to read $perlfile\n";
		    my (@pcontents) = <PFILE>;
		    close PFILE;
		    my $VAR1;
		    eval(join('',@pcontents)) 
		      or do {
			  #warn "Error in pfile $perlfile; reloading from xml\n";
			  goto load_xml;
		      };
		    $this->{'grovecache'}{$file} = $VAR1;
		}
	    } else {

	      load_xml:
		#print STDERR "  ...pcache miss; loading XML\n";

		# Then load and parse from XML file, and cache the result
		my $quiet = 1;
		$quiet = 0 if $file !~ m!compiled!; # when to complain...

		$this->{'grovecache'}{$file} = 
		  grove_from_xml($this->_get_object_xml($file, $quiet));

		if (defined($this->{'grovecache'}{$file})) {
		    # Write preparsed perlfile cache
		    my @path = split('/',$file); # mkdir -p
		    pop @path;
		    unshift (@path, 'pcache');
		    my $pelem;
		    my $dir = "$cachedir";
		    for $pelem (@path) {
			$dir = "$dir/$pelem";
			if (! -d $dir) {
			    umask 0002;
			    mkdir $dir, 0775 or warn "Could not mkdir $dir\n";
			}
		    }

		    umask 0002;
		    nstore ($this->{'grovecache'}{$file}, "$perlfile.$$")
		      or warn "Could not write $perlfile.$$\n";
		    rename "$perlfile.$$", "$perlfile";
		}
	    }
	}
	else
### NO CACHE ###
	{
	    $this->{'grovecache'}{$file} = 
	      grove_from_xml($this->_get_object_xml($file, $quiet));
	}
    }

    return $this->{'grovecache'}{$file};
}

# Load an XML object from the library
# You specify the relative file path (to .../db/), less the .xml on the end.
sub _get_object_xml {
    my ($this, $file, $quiet) = @_;

    open XML, "$libdir/db/$file.xml"
### CACHE ###
	or open XML, "$cachedir/$file.xml"
### NO CACHE ###
	    or do { warn "Cannot open file $libdir/db/$file.xml\n"
			if !$quiet;
		    return undef; };
    my $xml = join('', (<XML>));
    close XML;

    # This would be nice, but it runs Perl out of memory...
    if (0) {
	# Remove all useless whitespace before parse, to make a "neater" Grove
	$xml =~ s!\>([\s\n]*)?\<!\>\<!gs;
	#print STDERR $xml;
    }

    return $xml;
}

# Write an XML object from the library
# You specify the relative file path (to .../db/), less the .xml on the end.
sub _set_object_xml {
    my ($this, $file, $stuff, $cache) = @_;

    my $dir = "$libdir/db";
### CACHE ###
    if ($cachedir)
    {
	# mkdir -p
	my @path = split('/',$file);
	pop @path;			# drop filename
	my $pelem;
	my $dir = "$cachedir";
	for $pelem (@path) {
	    $dir = "$dir/$pelem";
	    if (! -d $dir) {
		umask 0002;
		mkdir $dir, 0775 or warn "Could not mkdir $dir\n";
	    }
	}

	$dir = $cachedir;
    }
### NO CACHE ###
    my $xfile = "$dir/$file.xml";
    umask 0002;
    open XML, ">$xfile.$$"
	or do { warn "Cannot write file $xfile.$$\n";
		return undef; };
    print XML $stuff;
    close XML;
    rename "$xfile.$$", $xfile
	or die "Cannot rename $xfile.$$ to $xfile\n";

    # Update cache
    delete $this->{'grovecache'}{$file};
    return 1;
}

# Get a list of XML filenames from a library directory.  These could then be
# read with _get_object_xml.
sub _get_xml_filelist {
    my ($this, $dir) = @_;

    if (!defined($this->{"names-$dir"})) {
	opendir DRV, "$libdir/db/$dir"
	    or die 'Cannot find source db for $dir\n';
	my $driverfile;
	while($driverfile = readdir(DRV)) {
	    next if ($driverfile !~ m!^(.+)\.xml$!);
	    push(@{$this->{"names-$dir"}}, $1);
	}
	closedir(DRV);
    }

    return @{$this->{"names-$dir"}};
}


# Return a Perl structure in eval-able ascii format
sub getascii {
    my ($this) = $_[0];
    if (! $this->{'dat'}) {
	$this->getdat();
    }
    
    local $Data::Dumper::Purity=1;
    local $Data::Dumper::Indent=1;

    # Encase data for inclusion in PPD file
    return Dumper($this->{'dat'});
}

# Return list of printer makes
sub _get_makes {
    my ($this) = @_;

    my @makes;
    my %seenmakes;
    my $p;
    for $p ($this->get_printerlist()) {
	my $pgrove = $this->get_printer_grove($p);
	my $make = grove_pathval($pgrove, '/printer/make');
	push (@makes, $make) if ! $seenmakes{$make};
    }

    return @makes;
}
sub get_makes {
    my ($this) = @_;

    my @makes;
    my %seenmakes;
    my $p;
    for $p (@{$this->get_overview()}) {
	my $make = $p->{'make'};
	push (@makes, $make) 
	    if ! $seenmakes{$make}++;
    }
	
    return @makes;
	
}
    
# get a list of model names from a make
sub get_models_by_make {
    my ($this, $wantmake) = @_;

    my $over = $this->get_overview();

    my @models;
    my %seenmakes;
    my $p;
    for $p (@{$over}) {
	push (@models, $p->{'model'}) 
	    if ($wantmake eq $p->{'make'});
    }

    return @models;
}

# get a printer id from a make/model
sub get_printer_from_make_model {
    my ($this, $wantmake, $wantmodel) = @_;

    my $over = $this->get_overview();
    my $p;
    for $p (@{$over}) {
	return $p->{'id'} if ($p->{'make'} eq $wantmake
			      and $p->{'model'} eq $wantmodel);
    }

    return undef;
}

sub get_javascript2 {

    my ($this) = @_;

    my @swit;
    my $mak;
    my $else = "";
    for $mak ($this->get_makes()) {
	push (@swit,
	      " $else if (make == \"$mak\") {\n");

	my $ct = 0;
	my $mod;
	for $mod (sort {normalizename($a) cmp normalizename($b) } 
		  $this->get_models_by_make($mak)) {
	    
	    my $p;
	    $p = $this->get_printer_from_make_model($mak, $mod);
	    if (defined($p)) {
		push (@swit,
		      "      o[i++]=new Option(\"$mod\", \"$p\");\n");
		$ct++;
	    }
	}

	if (!$ct) {
	    push(@swit,
		 "      o[i++]=new Option(\"No Printers\", \"0\");\n");
	}

	push (@swit,
	      "    }");
	$else = "else";
    }

    my $switch = join('',@swit);

    my $javascript = '
       function reflectMake(makeselector, modelselector) {
	 //
	 // This function is called when makeselector changes
	 // by an onchange thingy on the makeselector.
	 //

	 // Get the value of the OPTION that just changed
	 selected_value=makeselector.options[makeselector.selectedIndex].value;
	 // Get the text of the OPTION that just changed
	 make=makeselector.options[makeselector.selectedIndex].text;

	 o = new Array;
	 i=0;

     ' . $switch . '    if (i==0) {
	   alert("Error: that dropdown should do something, but it doesnt");
	 } else {
	   modelselector.length=o.length;
	   for (i=0; i < o.length; i++) {
	     modelselector.options[i]=o[i];
	   }
	   modelselector.options[0].selected=true;
	 }

       }
     ';

    return $javascript;
}

################################3
#################################

# A bunch of mostly or totally obsolete crap below here.

# Compute the union of a printer and a driver.
# Returns what was once called a '.foo' data structure.
# Horribly obsolete; exists just to illustrate what getdat should do
sub obsolete_getdat_internal {
    die "obsolete!\n";

    my ($this,$driver,$poid) = @_;
    my ($dbh) = $this->{'handle'};

    # Construct structure with driver information
    my @ppd;
    my %dat;
    my $ct = 0;			# flag; is there any info?
    
    my $p = $this->get_driver($driver);
    if(defined($p)) {

	$dat{'driver'} = $driver;
	$dat{'url'} =  $p->{'gs_driver_url'};
	$dat{'type'} = $p->{'driver_type'};
	$dat{'cmd'} = $p->{'prototype'};
	$dat{'comment'} = $p->{'drv_comment'};

	if ($dat{'cmd'}) {
	    # OK, we've got some information to work with
	    $ct = 1;
	}
	
    } else {
	
	# unknown driver; handle gracefully?
	die "unknown driver $driver\n";
	
    }
    
    # Fetch printer-specific information
    my ($make, $model) = (undef, undef);
    if (defined($poid)) {

	$p = $this->get_printer($poid);
	if (defined($p)) {
	    $make = $dat{'make'} = $p->{'make'};
	    $model = $dat{'model'} = $p->{'model'};
	    $dat{'pjl'} = $p->{'pjl'};
	    $dat{'pnp_mfg'} = $p->{'pnp_mfg'};
	    $dat{'pnp_cmd'} = $p->{'pnp_cmd'};
	    $dat{'pnp_des'} = $p->{'pnp_des'};
	    $dat{'pnp_mdl'} = $p->{'pnp_mdl'};
	    $dat{'ascii'} = $p->{'ascii'};
	    $dat{'color'} = $p->{'color'};
	}
    }
    
    if ($ct) {			# have we got any info?

	# Assemble various arguments that plug into the prototype

	my ($maxspot) = 'A';
	my ($next) = 'first';
	while(defined($p = $this->get_option('enum', $next))) {
	    $next = $p->{'next_idx'};

	    my $c = $this->check_constraints($p, $driver, $make, $model);
	    if ($c and $c->{'this'}) {
		my $arg;
		$arg->{'constraint'} = $c;
		$arg->{'style'} = $p->{'arg_style'};
		$arg->{'type'} = 'enum';
		$arg->{'proto'} = $p->{'arg_proto'};
		my $com = $arg->{'comment'} = $p->{'arg_comment'};
		my $idx = $arg->{'idx'} = $p->{'arg_idx'};
		my $def = $c->{'arg_defval'};
		my $nam = $arg->{'name'} = $p->{'arg_shortname'};
		$arg->{'required'} = $p->{'arg_required'};
		$arg->{'spot'} = $p->{'arg_spot'};
		$arg->{'order'} = $p->{'arg_order'};
		$maxspot = $arg->{'spot'} if ($arg->{'spot'} gt $maxspot);
	    
		# Now, find all the choices for this enum

		my ($ev, @vals, @valstmp);
		for $ev (@{$p->{'enum_vals'}}) {
		    my $vcon = $this->check_constraints($ev,   $driver, 
							$make, $model);
		    
		    # choice applies if no constraint, or if
		    # constraint is not false
		    if ((!defined($vcon)) or ($vcon->{'this'})) {
			my $v;
			$v->{'idx'} = $ev->{'ev_idx'};
			my $val = $v->{'value'} = $ev->{'ev_value'};
			my $dval = $v->{'driverval'} = $ev->{'ev_driverval'};
			my $com = $v->{'comment'} = $ev->{'ev_comment'};
		    
			# Stuff actual value of default into arg's default
			if ($v->{'idx'} == $def) {
			    $arg->{'default'} = $v->{'value'};
			}

			# We must have some sort of comment for PPD files
			if (!$com) {
			    $v->{'comment'} = $val;
			}

			push (@vals, $v);
			$arg->{'vals_byname'}->{$val} = $v;
			
		    }
		}

	    
		# Record all those enumerated values in the argument structure
		$arg->{'vals'} = \@vals;
	    
		# Remember this argument
		push(@{$dat{'args'}}, $arg);
	    }
	}
	
	# get integer args
	$next = 'first';
	while (defined($p = $this->get_option('int', $next))) {
	    $next = $p->{'next_idx'};

	    my $c = $this->check_constraints($p, $driver, $make, $model);
	    if ($c and $c->{'this'}) {
		my $arg;
		$arg->{'type'} = 'int';
		$arg->{'constraint'} = $c;
	    
		$arg->{'style'} = $p->{'arg_style'};
		$arg->{'proto'} = $p->{'arg_proto'};
		my $com = $arg->{'comment'} = $p->{'arg_comment'};
		my $idx = $arg->{'idx'} = $p->{'arg_idx'};
		my $nam = $arg->{'name'} = $p->{'arg_shortname'};
		$arg->{'default'} = $c->{'arg_defval'};
		$arg->{'required'} = $p->{'arg_required'};
		$arg->{'spot'} = $p->{'arg_spot'};
		$arg->{'order'} = $p->{'arg_order'};
		my $max = $arg->{'max'} = $p->{'arg_max'};
		my $min = $arg->{'min'} = $p->{'arg_min'};
	    
		$maxspot = $arg->{'spot'} if ($arg->{'spot'} gt $maxspot);
	    
		# Remember this argument
		push(@{$dat{'args'}}, $arg);
	    }
	}
	
	# get float args
	$next = 'first';
	while (defined($p = $this->get_option('float', $next))) {
	    $next = $p->{'next_idx'};

	    my $c = $this->check_constraints($p, $driver, $make, $model);
	    if ($c and $c->{'this'}) {
		my $arg;
		$arg->{'type'} = 'float';
		$arg->{'constraint'} = $c;
	    
		$arg->{'style'} = $p->{'arg_style'};
		$arg->{'proto'} = $p->{'arg_proto'};
		my $com = $arg->{'comment'} = $p->{'arg_comment'};
		my $idx = $arg->{'idx'} = $p->{'arg_idx'};
		my $nam = $arg->{'name'} = $p->{'arg_shortname'};
		$arg->{'default'} = $c->{'arg_defval'};
		$arg->{'required'} = $p->{'arg_required'};
		$arg->{'spot'} = $p->{'arg_spot'};
		$arg->{'order'} = $p->{'arg_order'};
		my $max = $arg->{'max'} = $p->{'arg_max'};
		my $min = $arg->{'min'} = $p->{'arg_min'};
	    
		$maxspot = $arg->{'spot'} if ($arg->{'spot'} gt $maxspot);
	    
		# Remember this argument
		push(@{$dat{'args'}}, $arg);
	    }
	}
	
	# get boolean args
	$next = 'first';
	while (defined($p = $this->get_option('bool', $next))) {
	    $next = $p->{'next_idx'};

	    my $c = $this->check_constraints($p, $driver, $make, $model);
	    if ($c and $c->{'this'}) {
		my $arg;
		$arg->{'type'} = 'bool';
		$arg->{'constraint'} = $c;
		$arg->{'style'} = $p->{'arg_style'};
		$arg->{'proto'} = $p->{'arg_proto'};
		my $com = $arg->{'comment'} = $p->{'arg_comment'};
		my $tname = $arg->{'name'} = $p->{'arg_shortname'};
		my $fname = $arg->{'name_false'} = $p->{'arg_shortname_false'};
		my $idx = $arg->{'idx'} = $p->{'arg_idx'};
		$arg->{'default'} = $c->{'arg_defval'};
		$arg->{'spot'} = $p->{'arg_spot'};
		$arg->{'order'} = $p->{'arg_order'};
	    
		$maxspot = $arg->{'spot'} if ($arg->{'spot'} gt $maxspot);
		
		# Remember this argument
		push(@{$dat{'args'}}, $arg);
	    }
	}

	$dat{'maxspot'} = $maxspot;
	$dat{'ct'} = $ct;

	# Generate hash full of refs to arguments by name
	my $arg;
	for $arg (@{$dat{'args'}}) {
	    my $name = $arg->{'name'};
	    $dat{'args_byname'}->{$name} = $arg;
	}
    }

    $dat{'timestamp'} = time();
    $dat{'id'} = $poid;

    $this->{'dat'} = undef;
    $this->{'dat'} = \%dat;

    return %dat;

}



# Public getdat.  Obtains a printer/driver data object from the
# filesystem, or from the web if the local copy is missing.  This data
# object is cached internally; other operations will operate on it.
#
# We're stupid; there's no cache in the filesystem.  To get a proper
# local copy of all the data, run snarf-library.
# 
# You can call this again to throw out the last data object and start
# working with a new one, if you care to write a multi-printer/driver
# sort of program.
#
sub obsolete_getdat {

    die "obsolete!\n";


    my ($this, $driver, $poid) = @_;

    if (defined($this->{'dat'})) {
	$this->{'dat'} = undef;
    }


    my ($tag) = $this->{'tag'};
    my ($tagclause, $tagfilen);
    if ($tag) {
	$tagclause = "&tag=$tag";
	$tagfilen = "-$tag";
    }

    my $filename = "$libdir/data$tagfilen/$driver-$poid.foo";
    if (-f $filename) {

	open FOO, $filename 
	    or die "Cannot open file $filename\n";
	my @page = <FOO>;
	close FOO;

	my $VAR1;
	eval join('', @page)
	    or die "Error in datablob file $filename!\n";
	
	$this->{'dat'} = $VAR1;

    } else {

	my $base = 'http://www.linuxprinting.org';
	my $url = "$base/get_data.cgi?driver=$driver&printer=$poid$tagclause";
	my $page = $this->getpage($url);
	
	my $VAR1;
	eval ($page) || die "Error in datablob!\n";
	
	$this->{'dat'} = $VAR1;

    }	

    return $this->{'dat'};
}

# Returns a data structure of make/model listings.  You get the name,
# pnp info, notes, func, a proofread bit, a list of drivers, and a
# record id number.

sub obsolete_getoverview {
    die "getoverview obsolete!\n";

    my ($this) = @_;

    if (defined($this->{'overview'})) {
	$this->{'overview'} = undef;
    }

    my ($tag) = $this->{'tag'};
    my ($tagclause, $tagfilen);
    if ($this->{'tag'}) {
	$tagclause = "?tag=$tag";
	$tagfilen = "-$tag";
    }

    my $filename = "$libdir/data$tagfilen/00-overview.foo";
    if (-f $filename) {

	open FOO, $filename 
	    or die "Cannot open file $filename\n";
	my @page = <FOO>;
	close FOO;

	my $VAR1;
	eval join('', @page)
	    or die "Error in overview file $filename!\n";
	
	$this->{'overview'} = $VAR1;

    } else {

	my $base = 'http://www.linuxprinting.org';
	my $url = "$base/get_over.cgi$tagclause";
	
	my $page = $this->getpage($url);
	
	eval ($page) || die "Error in datablob!\n";
	
	$this->{'overview'} = $VAR1;

    }

    return $this->{'overview'};
}

# Modify comments text to contain only what it should:
#
# <a>, <p>, <br> (<br> -> <p>)
#
sub comment_filter {
    my ($text) = @_;

    my $fake = ("INSERTFIXEDTHINGHERE" . sprintf("%06x", rand(1000000)));
    my %replacements;
    my $num = 1;

    # extract all the A href tags
    my $replace = "ANCHOR$fake$num";
    while ($text =~ 
	   s!(<\s*a\s+href\s*=\s*['"]([^'"]+)['"]\s*>)!$replace!i) {
	$replacements{$replace} = $1;
	$num++;
	$replace = "ANCHOR$fake$num";
    }

    # extract all the A tail tags
    my $replace = "ANCHORTAIL$fake$num";
    while ($text =~ 
	   s!(<\s*/\s*a\s*>)!$replace!i) {
	$replacements{$replace} = $1;
	$num++;
	$replace = "ANCHOR$fake$num";
    }

    # extract all the P tags
    $replace = "PARA$fake$num";
    while ($text =~ 
	   s!(<\s*p\s*>)!$replace!i) {

	$replacements{$replace} = $1;
	$num++;
	$replace = "PARA$fake$num";
    }

    # extract all the BR tags
    $replace = "PARA$fake$num";
    while ($text =~ 
	   s!(<\s*br\s*>)!$replace!i) {

	$replacements{$replace} = $1;
	$num++;
	$replace = "PARA$fake$num";
    }

    # Now it's just clean text; remove all tags and &foo;s
    $text =~ s!<[^>]+>! !g;
    $text =~ s!&amp;!&!g;
    $text =~ s!&lt;!<!g;
    $text =~ s!&gt;!>!g;
    $text =~ s!&[^;]+?;! !g;

    # Now rewrite into our teeny-html subset
    $text =~ s!&!&amp;!g;
    $text =~ s!<!&lt;!g;
    $text =~ s!>!&gt;!g;

    # And reinsert the few things we wanted to preserve
    for (keys(%replacements)) {
	my ($k, $r) = ($_, $replacements{$_});
	$text =~ s!$k!$r!;
    }

#    print STDERR "$text";

    return $text;
}

1;
