/*
  Copyright 2005, 2006, 2007 David Cad, Damien Stehl.

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; see the file COPYING.  If not, write to the Free
  Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA.

  This program implements ideas from the paper "Floating-point LLL Revisited", 
  by Phong Nguyen and Damien Stehl, in the Proceedings of Eurocrypt'2005, 
  Springer-Verlag; and was partly inspired by Shoup's NTL library: 
  http://www.shoup.net/ntl/

*/

#ifndef MATRIX_CPP
#define MATRIX_CPP

#include "matrix.h"

template<class ZT> ZZ_mat<ZT>::ZZ_mat(int rows,int cols)
{
  c=cols;
  r=rows;
  
  matrix=new Z_NR<ZT>*[r];
  
  for (int i=0;i<r;i++)
    matrix[i]=new Z_NR<ZT>[c];
}

template<class ZT> ZZ_mat<ZT>::~ZZ_mat()
{
  if (matrix)
    {
      for (int i=0;i<r;i++)
	if (matrix[i])
	  delete[] matrix[i];
      delete[] matrix;
    }
}

inline char nextNonBlankChar(char& ch)
{
  
  ch=getchar();
  while (ch==' '||ch=='\t'||ch=='\r'||ch=='\n')
    ch=getchar();
  return ch;
}

template<class ZT> int ZZ_mat<ZT>::getMaxExp()
{
  int max=0;
  for (int i=0;i<r;i++)
    for (int j=0;j<c;j++)
      {
	long int x;
	matrix[i][j].get_d_2exp(&x);
	if (max<x) max=x;
      }
  return max;
}

template<class ZT> int ZZ_mat<ZT>::read()
{
  char ch;
  
  nextNonBlankChar(ch);

  if (ch!='[')
    {
      cerr<< "[ expected instead of "<<ch<<"\n";
      return 1;
    }
  for (int i=0; i<r; i++)
    {
      nextNonBlankChar(ch);
      if (ch != '[')
	{
	  cerr <<"Error at row "<<i<< " '[' expected instead of "<<ch<<"\n";
	  return 1;
	}
      for (int j=0; j<c; j++)
	{
	  matrix[i][j].read();
	}
      
      nextNonBlankChar(ch);
      if (ch != ']')
	{
	  cerr<<"Error: ']' expected at line "<<i<<"\n";
	  return 1;
	}
    }

  nextNonBlankChar(ch);
  if (ch != ']')
    {
      cerr<<"Error: ']' expected\n";
      return 1;
    }

  return 0;
}


template<class ZT> inline Z_NR<ZT>& ZZ_mat<ZT>::Get (int i, int j)
{
#ifdef DEBUG
  //bound checking
  if (i<0 || i>=r || j<0 || j>=c)
    {
      cerr<<"bound checking error ! (z_nr) ("<<i<<","<<j<<")"<< endl;
      //sigsegv on purpose
      int x= *(int*)NULL;
    }
#endif
  return matrix[i][j];
}

template<class ZT> inline void ZZ_mat<ZT>::Set (int i, int j,Z_NR<ZT>& s)
{
#ifdef DEBUG
  //bound checking
  if (i<0 || i>=r || j<0 || j>=c)
    {
      cerr<<"bound checking error ! (z_nr) ("<<i<<","<<j<<")"<< endl;
      //sigsegv on purpose
      int x= *(int*)NULL;
    }
#endif
  matrix[i][j].set(s);
}


template<class ZT> int ZZ_mat<ZT>::getShift()
{
  int n = GetNumCols();
  int shift=0;
  for (int i=0;i<GetNumRows();i++)
    {
      int j;
      for (j=n-1;j>=0 && Get(i,j).sgn()==0;j--);  
      
      if (shift<j-i) shift=j-i;
      
    }

#ifdef VERBOSE
  cerr << "Shift  =  " << shift;
#endif
  return shift;
}


template<class ZT> void ZZ_mat<ZT>::print (int d,int n)
{
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
  cout << "[";
  for (int i=0;i<d;i++) 
    {
      cout<< "[";
      for (int j=0;j<n;j++)
	{
	  matrix[i][j].print();
	  cout<<" ";
	}
      cout << "]" << endl;
    }
  cout << "]" << endl; 
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();

}
template<class ZT> void ZZ_mat<ZT>::print ()
{
  print(r,c);
}



/*fp*/


template<class FT> FP_mat<FT>::FP_mat(int rows,int cols)
{
  c=cols;
  r=rows;
  
  matrix=new FP_NR<FT>*[r];
  
  for (int i=0;i<r;i++)
    matrix[i]=new FP_NR<FT>[c];
}

template<class FT> FP_mat<FT>::~FP_mat()
{
  if (matrix)
    {
      for (int i=0;i<r;i++)
	if (matrix[i])
	  delete[] matrix[i];
      delete[] matrix;
    }
}

template<class FT> inline FP_NR<FT>& FP_mat<FT>::Get (int i, int j)
{
#ifdef DEBUG
  //  printf ("c=%d r=%d\n", c, r);
  //bound checking
  if (i<0 || i>=r || j<0 || j>=c)
    {
      cerr<<"bound checking error (fp_nr)! ("<<i<<","<<j<<")"<< endl;
      //      abort();
    }
#endif
  return matrix[i][j];
}

template<class FT> inline void FP_mat<FT>::Set (int i, int j,FP_NR<FT>& s)
{
#ifdef DEBUG
  //bound checking
  if (i<0 || i>=r || j<0 || j>=c)
    {
      cerr<<"bound checking error (fp_nr)! ("<<i<<","<<j<<")"<< endl;
      //      abort();
    }
#endif
  matrix[i][j].set(s);
}
template<class FT> void FP_mat<FT>::print (int d,int n)
{
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
  cout<< "[";
  for (int i=0;i<d;i++) 
    {
      cout<< "[";
      for (int j=0;j<n;j++)
	{
	  matrix[i][j].print();
	  cout<<" ";
	}
      cout << "]" << endl;
    }
  cout << "]" << endl; 
  fflush(stderr);
  fflush(stdout);
  cout.flush();
  cerr.flush();
}
template<class FT> void FP_mat<FT>::print ()
{
  print(r,c);
}

template<class FT> inline FP_NR<FT>*& FP_mat<FT>::GetVec(int i)
{
  return matrix[i];
}
template<class ZT> inline Z_NR<ZT>*& ZZ_mat<ZT>::GetVec(int i)
{
  return matrix[i];
}

template<class ZT> inline void ZZ_mat<ZT>::gen_intrel(int bits)
{
  if (c!=r+1)
    {
      cerr<<"gen_intrel called on an ill-formed matrix"<<endl;
      return;
    }
  int i,j;
  for (i=0;i<r;i++)
    {
      matrix[i][0].randb(bits);
      for (j=1; j<=i; j++)
	matrix[i][j].set(0);
      matrix[i][i+1].set(1);
      for (j=i+2; j<c; j++)
	matrix[i][j].set(0);      
    }
}


template<class ZT> inline void ZZ_mat<ZT>::gen_simdioph(int bits,int bits2)
{
  if (c!=r)
    {
      cerr<<"gen_simdioph called on an ill-formed matrix"<<endl;
      return;
    }
  int i, j;

  matrix[0][0].set(1);
  matrix[0][0].mul_2exp(matrix[0][0], bits2);
  for (i=1; i<r; i++)
    matrix[0][i].randb(bits);
  for (i=1; i<r; i++)
    {
      for (j=1; j<i; j++)
	matrix[j][i].set(0);
      matrix[i][i].set(1);
      matrix[i][i].mul_2exp(matrix[i][i], bits);
      for (j=i+1; j<c; j++)
	matrix[j][i].set(0);
    }
}

template<class ZT> inline void ZZ_mat<ZT>::gen_uniform(int bits)
{
  for (int i=0;i<r;i++)for(int j=0;j<c;j++)matrix[i][j].randb(bits);
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ntrulike(int bits,int q)
{
  int i, j, k;
  int d=r/2;
  if (c!=r || c!=2*d) 
    {
      cerr<<"gen_ntrulike called on an ill-formed matrix"<<endl;
      return;
    }
  Z_NR<ZT> * h=new Z_NR<ZT>[d];

  for (i=0; i<d; i++)
    h[i].randb(bits);
  
  for (i=0; i<d; i++)
    {
      for (j=0; j<i; j++)
	matrix[i][j].set(0);
      matrix[i][i].set(1);
      for (j=i+1; j<d; j++)
	matrix[i][j].set(0);
    }

  for (i=d; i<r; i++)
    for (j=0; j<d; j++)
      matrix[i][j].set(0);

  for (i=d; i<r; i++)
    {
      for (j=d; j<i; j++)
	matrix[i][j].set(0);
      matrix[i][i].set(q);
      for (j=i+1; j<c; j++)
	matrix[i][j].set(0);
    }

  for (i=0; i<d; i++)
    for (j=d; j<c; j++)
      { 
	k = j+i;
	while (k>=d)k-=d;
	matrix[i][j].set(h[k]);
      }

  delete[] h;
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ntrulike2(int bits,int q)
{

  int i, j, k;
  
  int d=r/2;
  if (c!=r || c!=2*d) 
    {
      cerr<<"gen_ntrulike2 called on an ill-formed matrix"<<endl;
      return;
    }
  Z_NR<ZT> * h=new Z_NR<ZT>[d];
   
  for (i=0; i<d; i++)
    h[i].randb(bits);
  
  for (i=0; i<d; i++)
    for (j=0; j<c; j++)
      matrix[i][j].set(0);

  for (i=0; i<d; i++)
    matrix[i][i].set(q);


  for (i=d; i<r; i++)
    for (j=d; j<c; j++)
      matrix[i][j].set(0);
      
  for (i=d; i<c; i++)
    matrix[i][i].set(1);

  for (i=d; i<r; i++)
    for (j=0; j<d; j++)
      { 
	k = i+j;
	while (k>=d)k-=d;
	matrix[i][j].set(h[k]);
      }

  delete[] h;
}

template<class ZT> inline void ZZ_mat<ZT>::gen_ajtai(double alpha)
{
  int i, j, bits;
  Z_NR<ZT> ztmp;
  
  int d=r;
  if (c!=r) 
    {
      cerr<<"gen_ajtai called on an ill-formed matrix"<<endl;
      return;
    }

  for (i=0; i<d; i++)
    {
      bits = (int) pow((double) (2*d-i), alpha);
      ztmp.set(1);
      ztmp.mul_2exp(ztmp, bits);	  
      matrix[i][i].randm(ztmp);
      matrix[i][i].add_ui(matrix[i][i], 1);
       ztmp.div_2exp(matrix[i][i], 1);
      for (j=i+1; j<d; j++)
	{
	  matrix[j][i].randm(ztmp);
	  matrix[i][j].set(0);
	}
    }
}


#endif
