/*
 *      Copyright (C) 1997 Claus-Justus Heine

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *
 * $RCSfile: ftformat.h,v $
 * $Revision: 1.6 $
 * $Date: 2001/08/11 00:08:58 $
 *
 *      This file contains the definitions for the user level floppy
 *      tape formatting stuff for the QIC-40/80/3010/3020 floppy-tape
 *      driver "ftape" for Linux.
 */

#ifndef NR_FTAPE_BUFFERS
#define NR_FTAPE_BUFFERS 3
#endif

#define NR_ITEMS(x)     (int)(sizeof(x)/ sizeof(*x))

/* Each sector in a segment needs these four bytes to be send via dma.
 * As the fdc doesn't know about the sequential ordering of data on a tape,
 * one needs to translate the segment id and sector number of a TAPE segment
 * into something the fdc understands. In turn, the tape drive hardware knows
 * what to do with the data it gets from the fdc.
 *
 * cyl  gives the floppy track the sector belongs to
 * head gives the floppy r/w head this sector belongs to
 * sect gives the sector relative to the start of the floppy track
 * size is equal to 3.
 */

typedef struct {
    u_int8_t cyl; u_int8_t head; u_int8_t sect; u_int8_t size;
} format_sector __attribute__ ((packed));

typedef struct {
    format_sector sectors[FT_SECTORS_PER_SEGMENT];
} format_segment __attribute__ ((packed));

#define MAX_FLOPPY_SECTOR 128        /* number of sectors per track */
#define SEGMENTS_PER_FLOPPY_TRACK (MAX_FLOPPY_SECTOR/FT_SECTORS_PER_SEGMENT)
/* segments per floppy head. ftm == floppy track max. */
#define SPFH(ftm) (SEGMENTS_PER_FLOPPY_TRACK * ((ftm) + 1))

/*  Unaligned access.
 */
#if !WORDS_BIGENDIAN
# define PUT2(ptr, off, val) \
{ u_int16_t _val = val; memcpy((u_int8_t *)ptr + off, &_val, 2); }
# define PUT4(ptr, off, val) \
{ u_int32_t _val = val; memcpy((u_int8_t *)ptr + off, &_val, 4); }
# define PUT8(ptr, off, val) \
{ u_int64_t _val = val; memcpy((u_int8_t *)ptr + off, &_val, 8); }
# define GET2(ptr, off) \
({ u_int16_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 2); _res; })
# define GET4(ptr, off) \
({ u_int32_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 4); _res; })
# define GET8(ptr, off) \
({ u_int64_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 8); _res; })
#else
# if __GLIBC__ < 2 || __GLIBC_MINOR__ < 2 || !__GNUC__ || __GNUC__ < 2
#  error Only LITTLEENDIAN byte order supported, sorry
# endif
# define PUT2(ptr, off, val) \
{ u_int16_t _val = bswap_16(val); memcpy((u_int8_t *)ptr + off, &_val, 2); }
# define PUT4(ptr, off, val) \
{ u_int32_t _val = bswap_32(val); memcpy((u_int8_t *)ptr + off, &_val, 4); }
# define PUT8(ptr, off, val) \
{ u_int64_t _val = bswap_64(val); memcpy((u_int8_t *)ptr + off, &_val, 8); }
# define GET2(ptr, off) \
({ u_int16_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 2); bswap_16(_res); })
# define GET4(ptr, off) \
({ u_int32_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 4); bswap_32(_res); })
# define GET8(ptr, off) \
({ u_int64_t _res; memcpy(&_res, (u_int8_t *)ptr + off, 8); bswap_64(_res); })
#endif
/* count the number of bits set in mask
 */
static inline int count_ones(unsigned long mask)
{
    int bits;

    for (bits = 0; mask != 0; mask >>= 1) {
	if (mask & 1) {
	    ++bits;
	}
    }
    return bits;
}

/* old APIs. The numbers of the tape operation have not changed, but
 * the argument size has slightly been altered, so we need to define
 * different MTIOCFTFORMAT ioctls.
 *
 * The most recent API supported by ftformat always is defined in
 * $(top_srcdir)/include/ftape/linux/mtio.h
 */

/*  ftape-3.x API
 */

struct ftfmtparms_3 {
    unsigned char  ft_qicstd;   /* QIC-40/QIC-80/QIC-3010/QIC-3020 */
    unsigned char  ft_fmtcode;  /* Refer to the QIC specs */
    unsigned char  ft_fhm;      /* floppy head max */
    unsigned char  ft_ftm;      /* floppy track max */
    unsigned short ft_spt;      /* segments per track */
    unsigned short ft_tpc;      /* tracks per cartridge */
};

struct ftfmttrack_3 {
    unsigned int  ft_track;   /* track to format */
    unsigned char ft_gap3;    /* size of gap3, for FORMAT_TRK */
};

struct ftfmtstatus_3 {
    unsigned int  ft_segment;  /* segment currently being formatted */
};

struct ftfmtverify_3 {
    unsigned int  ft_segment;   /* segment to verify */
    unsigned long ft_bsm;       /* bsm as result of VERIFY cmd */
};

struct mtftformat_3 {
    unsigned int fmt_op;      /* operation to perform */
    union {
	struct ftfmtparms_3  fmt_parms;  /* format parameters */
	struct ftfmttrack_3  fmt_track;  /* ctrl while formatting */
	struct ftfmtstatus_3 fmt_status;
	struct ftfmtverify_3 fmt_verify; /* for verifying */ 
    } fmt_arg;
};

#define MTIOCFTFORMAT_3   _IOWR('m', 10, struct mtftformat_3)

/* ftape-4.02 API
 */
#define FTFMT_MAGIC_4           (4 << 16) /* old version of format API */

struct ftfmtparms_4 {
    unsigned char  ft_qicstd;   /* QIC-40/QIC-80/QIC-3010/QIC-3020 */
    unsigned char  ft_fmtcode;  /* Refer to the QIC specs */
    unsigned char  ft_fhm;      /* floppy head max */
    unsigned char  ft_ftm;      /* floppy track max */
    unsigned short ft_spt;      /* segments per track */
    unsigned short ft_tpc;      /* tracks per cartridge */
};

struct ftfmttrack_4 {
    unsigned int  ft_track;   /* track to format */
    unsigned char ft_gap3;    /* size of gap3, for FORMAT_TRK */
};

struct ftfmtstatus_4 {
    unsigned int  ft_segment;  /* segment currently being formatted */
};

struct ftfmtverify_4 {
    unsigned int  ft_segment;   /* segment to verify */
    unsigned long ft_bsm;       /* bsm as result of VERIFY cmd */
};

struct mtftformat_4 {
    unsigned int fmt_magic;   /* should be FTFMT_MAGIC */
    unsigned int fmt_op;      /* operation to perform */
    union {
	struct ftfmtparms_4  fmt_parms;  /* format parameters */
	struct ftfmttrack_4  fmt_track;  /* ctrl while formatting */
	struct ftfmtstatus_4 fmt_status;
	struct ftfmtverify_4 fmt_verify; /* for verifying */ 
    } fmt_arg;
};

#define MTIOCFTFORMAT_4   _IOWR('m', 10, struct mtftformat_4)

/* Now we play a dirty trick. We know that all changes in the
 * ftfmt... structures have altered field at the END of some
 * structures. So we construct an artificial worst case mtftformat
 * structure here. We call the API with the correct MTIOC ioctl, so
 * the size check performed by the kernel driver will succeed, and the
 * kernel driver will simply ignore the excess elements.
 */

struct ftfmtparms_virtual {
    unsigned char  ft_qicstd;   /* QIC-40/QIC-80/QIC-3010/QIC-3020 */
    unsigned char  ft_fmtcode;  /* Refer to the QIC specs */
    unsigned char  ft_fhm;      /* floppy head max */
    unsigned char  ft_ftm;      /* floppy track max */
    unsigned short ft_spt;      /* segments per track */
    unsigned short ft_tpc;      /* tracks per cartridge */
    /* added fields in for ftape-4.03 */
    unsigned char  ft_gap3;     /* size of gap3, for FORMAT_TRK */
    unsigned char  ft_ffb;      /* format filler byte */
};

struct ftfmttrack_virtual {
    unsigned int  ft_track;   /* track to format */
    /* removed in ftape-4.03 */
    unsigned char ft_gap3;    /* size of gap3, for FORMAT_TRK */
};

struct ftfmtstatus_virtual {
    unsigned int  ft_segment;  /* segment currently being formatted */
};

struct ftfmtverify_virtual {
    unsigned int  ft_segment;   /* segment to verify */
    unsigned long ft_bsm;       /* bsm as result of VERIFY cmd */
};

struct mtftformat_virtual {
    unsigned int fmt_magic;   /* should be FTFMT_MAGIC or FTFMT_MAGIC_4 */
    unsigned int fmt_op;      /* operation to perform */
    union {
	struct ftfmtparms_virtual  fmt_parms;  /* format parameters */
	struct ftfmttrack_virtual  fmt_track;  /* ctrl while formatting */
	struct ftfmtstatus_virtual fmt_status;
	struct ftfmtverify_virtual fmt_verify; /* for verifying */ 
    } fmt_arg;
};

#define MTIOCFTFORMAT_4   _IOWR('m', 10, struct mtftformat_4)



#ifndef HAVE_GETSUBOPT
extern int
getsubopt (char **optionp, const char *const *tokens, char **valuep);
#endif

/*
 * Local variables:
 *  version-control: t
 *  kept-new-versions: 5
 *  c-basic-offset: 4
 * End:
 */
