// ---------------------------------------------------------------------
// $Id: BruteForce.cc 154 2008-06-04 06:40:03Z daaugusto $
//
//   BruteForce.cc (created on Fri Nov 18 20:47:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2008 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, see <http://www.gnu.org/licenses/>.
//
// ---------------------------------------------------------------------

#include "BruteForce.hh"
#include "../util/Exception.hh"

#include <cmath>
#include <limits>

#include <iomanip>

// Round a given float number
#define ROUND(x) (x<0?ceil((x)-0.5):floor((x)+0.5))

// ---------------------------------------------------------------------
void
BruteForce::Evolve()
{
   // Prepare (free) the m_best pointer for the current generation
   Reset();

   if (m_params.m_verbose)
   {  
      cout << *this;

      const Params::UBigInt max_iter = 
         static_cast<Params::UBigInt>(ROUND(pow(2.0L,(int)m_files.size())));

      /* Is the input size able to be managed by the Brute Force search?

         Test the following conditions:
         - if max_iter == 0 then 'overflow error'
         - if max_iter != 2^n then 'overflow error'

         Note: log2(2^n) = n, where log2(x) = log(x)/log(2)
       */

      cout << "> Trying 2^" << m_files.size() << " combinations.";

      if (max_iter == 0 || m_files.size() !=
            static_cast<size_t>(ROUND(log(static_cast<long double>(max_iter))/log(2.0L)))) 
      {
         cout << " (so many iterations, how about GA search?)" << endl;
      }
      else cout << " (" << max_iter << " iterations)" << endl;
   }

   // allocate memory for the "global" m_best pointer
   m_best = new vector<bool>(m_files.size(),0);

   // Find the best solution and put it into m_best variable
   EnumerateAndTest();
}

// --------------------------------------------------------------------
void BruteForce::EnumerateAndTest()
{
   // Initialize/reset the temporary variables
   m_stop = false;
   m_best_score = numeric_limits<Params::Size_t>::max();
   m_candidate.resize(m_files.size());

   // Run the recursive function to enumerate and test candidate
   // solutions
   EnumerateAndTest(0);
}
   
// --------------------------------------------------------------------
void BruteForce::EnumerateAndTest(int index)
{
   if (index >= m_candidate.size()) return; // stop criterion

   for (int j=0; j<2; ++j) // j=0,1 means a binary vector (0s and 1s)
   {
      // Stop the recursion if a (perfect) solution was found.
      if (m_stop) return;

      m_candidate[index] = j;

      /* Calls recursively in order to change (swap) the next
       * positions in a chain fashion. */
      EnumerateAndTest(index+1);

      if (index == m_candidate.size()-1)
      {  
         /* A candidate solution (combination) has been built. Now it is
          * necessary to evaluate it. */
         Params::Size_t score = Evaluate(m_candidate);

         if (score >= 0 && score < m_best_score) 
         { 
            // copy 'candidate' to pointer 'm_best' (from Optimizer) 
            m_best->assign(m_candidate.begin(), m_candidate.end());
            m_best_score = score;

            if (m_params.m_verbose) 
            {
               cout << "> Best fit so far (diff): " 
                    << m_params.PrettySize(m_best_score);

               /* Show the search progress, i.e., how many of the search
                * space (2^input) were visited when a best solution has been found. 
                *
                * How it works?
                *
                * If the first position of m_candidate is '0' then the
                * search is under 50% (0^1 * 100); If it is '1', then the search is
                * above 50% (0.5^1 * 100). The next positions add more
                * information (precision). For instance, if the second
                * position is '1', the progress grows 0.5^2 (if it is
                * '0' then the progress grows nothing). So, the progress is
                * calculated as [0.5 or 0]^1 + [0.5 or 0]^2 + ... + [0.5 or 0]^n.
                *
                * At the end it is necessary to add a residual value 0.5^n in order
                * to ensure that progress sums up to 100%.
                */
               double progress = 0.0;
               for (int i=0; i<m_candidate.size(); ++i) 
                        progress += (m_candidate[i] ? pow(0.5, i+1) : 0.0);

               // adds the residual value
               progress += pow(0.5, (int)m_candidate.size());

               cout << " (Progress: " <<  fixed << setprecision(2) 
                    << progress*100.0 << "%)" << endl;
            }

            if (score <= 0.0) { m_stop = true; }; // found a perfect fit!
         }

      }
   }
}
// --------------------------------------------------------------------
ostream& 
BruteForce::Write(ostream& s) const 
{ 
   s << endl;
   s << "> -----------------------------------" << endl;
   s << "> Brute Force search"                  << endl;
   s << "> -----------------------------------" << endl;

   Optimizer::Write(s);

   s << endl << flush;

   return s;
}

// --------------------------------------------------------------------
