// ---------------------------------------------------------------------
// $Id: Exception.hh 151 2008-06-04 01:35:38Z daaugusto $
//
//   Exception.hh (created on Tue Aug 23 01:08:35 BRT 2005)
// 
//   Genetic Algorithm File Fitter (gaffitter)
//
//   Copyright (C) 2005-2008 Douglas A. Augusto
// 
// This file is part of gaffitter.
// 
// gaffitter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 3 of the License, or (at
// your option) any later version.
// 
// gaffitter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with gaffitter; if not, see <http://www.gnu.org/licenses/>.
//
// ---------------------------------------------------------------------

#ifndef exception_hh
#define exception_hh 

#include <string>
#include <sstream>
#include <limits>
#include <exception>

using namespace std;

//----------------------------------------------------------------------
#ifdef NDEBUG
const bool debug = false;
#else 
const bool debug = true;
#endif

class E_Exception;

//----------------------------------------------------------------------
/**
 * Exception base class.
 */
class E_Exception: std::exception { 
public:
  /**
   * Create an exception with an optional message "L" and an
   * error message "M".
   */
  E_Exception(const string& L = "", const string& M = "")
            : m_loc(L), m_msg(M) {}

  virtual ~E_Exception() throw() {};
  /** 
   * Return the user message. 
  */
  virtual const char * local() { return m_loc.c_str(); }
  /**
   * Return the error message.
   */
  virtual const char * what() const throw() { return m_msg.c_str(); }

protected:
  string m_loc;
  string m_msg;
};

//----------------------------------------------------------------------
inline ostream&
operator << (ostream& o, E_Exception& e)
{
   o << '\n' << "> Error: " << e.what() << endl;
   return o;
}

//-----------------------------------------------------------------------------
class E_NoInputFiles : public E_Exception {
public:  
   E_NoInputFiles ( const string& loc = "" )
      : E_Exception (loc, "No input to process") {};
};

//-----------------------------------------------------------------------------
class E_BigInput : public E_Exception {
public:  
   E_BigInput ( const string& loc = "" )
      : E_Exception (loc, "Too many input files/items") {};
};

//-----------------------------------------------------------------------------
class E_NoTarget : public E_Exception {
public:  
   E_NoTarget (const string& app = "gaffitter"): E_Exception ("", 
         "Missing '--target' (Try '"+app+" -h' for more information)") {};
};

//-----------------------------------------------------------------------------
/**
 * When a Max value is smaller than a given Min value.
 *
 * Usage: if (max<min) throw E_MaxMin<type>(min, max,"optional text");
 */
template<class T> class E_MaxMin : public E_Exception {
public:  
   E_MaxMin (T min, T max, const string& loc = "")
   { 
      ostringstream o;

      o << loc << "'max' (" << max << ") smaller than 'min' (" << min << ")" 
        << ", feasible range: ["<< numeric_limits<T>::min() << ":"
                                << numeric_limits<T>::max() << "]";

      m_msg = o.str();
   }
};

//-----------------------------------------------------------------------------
template<class X, class A> inline void Assert(A expression)
{
   if (debug && !expression) throw X();
} 

inline void Assert(bool expression)
{
   Assert<E_Exception>(expression);
}

//-----------------------------------------------------------------------------
#endif
