/*
 * Gnome-biff program
 * Version:  2.6
 * Copyright (C) 2000-2001  Nicolas Rougier
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#ifdef USE_GNOME
#  include <gnome.h>
#  include <applet-widget.h>
#endif
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "gbiff.h"
#include "applet.h"
#include "window.h"
#include "callbacks.h"
#include "protocol.h"
#include "interface.h"
#include "support.h"

/* Mailbox toplevel */
/********************************************************************************
 * Function: Callback on mailbox deletion
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show the applet in gnome taskbar
 *           Remove the poll timer and initiate a new one
 *           Hide the mailbox 
 * Return  : TRUE
 ********************************************************************************/
gboolean on_Mailbox_delete (GtkObject *object, GdkEvent *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  GtkEntry *editable = GTK_ENTRY (lookup_widget (biff->mailbox, "AddressEntry"));
  // Get the adress from the AddressEntry widget
  if ((biff->protocol == PROTOCOL_POP3) || (biff->protocol == PROTOCOL_IMAP4)) {
    // If no port has been supplied, then set the default port at the end of the address
    if (strchr (gtk_entry_get_text (editable), (int) ':') == NULL) {
      free (biff->address);
      biff->address = (char *) malloc (strlen (gtk_entry_get_text (editable))+1+4);
      if (biff->protocol == PROTOCOL_POP3)
	sprintf (biff->address, "%s:%.4d", gtk_entry_get_text (editable), DEFAULT_POP3_PORT);
      if (biff->protocol == PROTOCOL_IMAP4)
	sprintf (biff->address, "%s:%.4d", gtk_entry_get_text (editable), DEFAULT_IMAP4_PORT);
    }
    // Set the new address in the widget
    gtk_entry_set_text (editable, biff->address);
  }

#ifdef USE_GNOME
  gtk_widget_show (biff->applet);
#else
  gtk_widget_show (biff->window);
#endif
  if (biff->suspendMode == 0) {
    if (biff->polltag > -1)
      gtk_timeout_remove (biff->polltag);
    biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  }
  gtk_widget_hide (biff->mailbox);

#ifdef USE_GNOME
  /* Assume the user won't want us to pop up again */
  /* Not true if using POP3 however (password not saved) */
  if (biff->protocol != PROTOCOL_POP3)
    biff->optNoConfigure = 1;
  /* sync on preference close */
  applet_widget_sync_config (APPLET_WIDGET(biff->applet));
#endif

  return (TRUE);
}

/********************************************************************************
 * Function: Callback on change within title entry
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update title within biff structure
 *           Update label of applet
 *           Destroy and create a new popup (according to new parameters)
 * Return  : /
 ********************************************************************************/
void on_TitleEntry_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  if (biff->title != 0)
    free (biff->title);
  biff->title = (char *) strdup (gtk_entry_get_text (GTK_ENTRY(editable)));
  gtk_widget_hide (biff->popup);
  if (biff->suspendMode == 0) {
    if (biff->polltag == -1)
      biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  }
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
#ifdef USE_GNOME
  if (biff->applet != 0)
    gtk_label_set_text (GTK_LABEL (lookup_widget (biff->applet, "TitleLabel")), biff->title);
#else
  updateWindow (biff, -2);
#endif

}

/********************************************************************************
 * Function: Callback on file protocol toggle
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update protocol within biff structure
 *           Make user and password entries insensitive
 * Return  : /
 ********************************************************************************/
void on_ProtocolRadioFile_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordLabel"), FALSE);
    biff->protocol = PROTOCOL_FILE;
  }
}

/********************************************************************************
 * Function: Callback on Maildir protocol toggle
 * Input   : Gtk stuff + user_data which is a biff structure
 * Output  : Update protocol within biff structure
 *           Make user and password entries insensitive
 * Return  : /
 * Notes   : Added by Neil Muller, Febuary 2001
 *           Copied from on_ProtocolRadioFile_toggled
 ********************************************************************************/
void on_ProtocolRadioMaildir_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordLabel"), FALSE);
    biff->protocol = PROTOCOL_MAILDIR;
  }
}

/********************************************************************************
 * Function: Callback on pop3 protocol toggle
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update protocol within biff structure
 *           Make user and password entries sensitive
 * Return  : /
 ********************************************************************************/
void on_ProtocolRadioPop3_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserEntry"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordEntry"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordLabel"), TRUE);
    biff->protocol = PROTOCOL_POP3;
  }
}

/********************************************************************************
 * Function: Callback on imap4 protocol toggle
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update protocol within biff structure
 *           Make user and password entries sensitive
 * Return  : /
 ********************************************************************************/
void on_ProtocolRadioImap4_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserEntry"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordEntry"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordLabel"), TRUE);
    biff->protocol = PROTOCOL_IMAP4;
  }
}

/********************************************************************************
 * Function: Callback on mh protocol toggle
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update protocol within biff structure
 *           Make user and password entries sensitive
 * Return  : /
 ********************************************************************************/
void on_ProtocolRadioMh_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "UserLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordEntry"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PasswordLabel"), FALSE);
    biff->protocol = PROTOCOL_MH;
  }
}

/********************************************************************************
 * Function: Callback on change within user entry
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update user within biff structure
 * Return  : /
 ********************************************************************************/
void on_UserEntry_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  if (biff->user != 0)
    free (biff->user);
  biff->user = (char *) strdup (gtk_entry_get_text (GTK_ENTRY(editable)));
}

/********************************************************************************
 * Function: Callback on change within user entry
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update password within biff structure
 * Return  : /
 ********************************************************************************/
void on_PasswordEntry_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->password != 0)
    free (biff->password);
  biff->password = (char *) strdup (gtk_entry_get_text (GTK_ENTRY(editable)));
}

/********************************************************************************
 * Function: Callback on change within address entry
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update address within biff structure
 * Return  : /
 ********************************************************************************/
void on_AddressEntry_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->address != 0)
    free (biff->address);
  biff->address = (char *) strdup (gtk_entry_get_text (GTK_ENTRY(editable)));
}

/********************************************************************************
 * Function: Callback on address button clicked
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show file selection dialog
 * Return  : /
 ********************************************************************************/
void on_AddressButton_clicked (GtkButton  *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->selectionMode = ADDRESS_SELECTION_MODE;
  gtk_widget_show (biff->fileDialog);
}

/********************************************************************************
 * Function: Callback on change within polltime spin
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update polltime within biff structure
 * Return  : /
 ********************************************************************************/
void on_PollTimeSpin_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  biff->polltime = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
}

/********************************************************************************
 * Function: Callback on change within poptime spin
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update poptime within biff structure
 * Return  : /
 ********************************************************************************/
void on_PopTimeSpin_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->poptime = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
}

/********************************************************************************
 * Function: Callback on change within popup X coord.
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update popup X coord. within biff structure
 * Return  : /
 ********************************************************************************/
void on_PopupGeometryXSpin_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->popupX = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
}

/********************************************************************************
 * Function: Callback on change within popup Y coord.
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update popup Y coord. within biff structure
 * Return  : /
 ********************************************************************************/
void on_PopupGeometryYSpin_changed (GtkEditable *editable, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->popupY = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
}

/********************************************************************************
 * Function: Callback on change within window X coord.
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update popup X coord. within biff structure
 * Return  : /
 ********************************************************************************/
void on_WindowGeometryXSpin_changed (GtkEditable *editable, gpointer user_data) {
  int width, height, x, y;
  Biff *biff = (Biff *) user_data;

  biff->windowX = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
  gdk_window_get_size (biff->window->window, &width, &height);
  if (biff->windowX < 0)
    x = gdk_screen_width()  - width  + biff->windowX;
  else
    x = biff->windowX;
  if (biff->windowY < 0)
    y = gdk_screen_height() - height + biff->windowY; 
  else
    y = biff->windowY;
  gdk_window_move (biff->window->window, x, y);
  /*  printf ("Move window to (%d, %d)\n", x, y);*/
}

/********************************************************************************
 * Function: Callback on change within window Y coord.
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update popup Y coord. within biff structure
 * Return  : /
 ********************************************************************************/
void on_WindowGeometryYSpin_changed (GtkEditable *editable, gpointer user_data) {
  int width, height, x, y;
  Biff *biff = (Biff *) user_data;

  biff->windowY = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(editable));
  gdk_window_get_size (biff->window->window, &width, &height);
  if (biff->windowX < 0)
    x = gdk_screen_width()  - width  + biff->windowX;
  else
    x = biff->windowX;
  if (biff->windowY < 0)
    y = gdk_screen_height() - height + biff->windowY; 
  else
    y = biff->windowY;
  gdk_window_move (biff->window->window, x, y);
  /*  printf ("Move window to (%d, %d)\n", x, y);*/
}

/********************************************************************************
 * Function: Callback on no date toggled
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update no date option within biff structure
 *           Destroy and create a new popup (according to new parameters)
 * Return  : /
 ********************************************************************************/
void on_OptionsNoDateCheck_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  if (togglebutton->active)
    biff->optNoDate = 1;
  else
    biff->optNoDate = 0;
  gtk_widget_hide (biff->popup);
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
}

/********************************************************************************
 * Function: Callback on no popup toggled
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update no popup option in biff structure
 *           Make popup options sensitive or insensisitive
 *            according to toggle state
 * Return  : /
 ********************************************************************************/
void on_OptionsNoPopupCheck_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    biff->optNoPopup = 1;
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "OptionsNoDateCheck"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "OptionsNoTitleCheck"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "FontLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "FontButton"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopTimeLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopTimeSpin"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryXLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryYLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryXSpin"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryYSpin"), FALSE);
  }
  else {
    biff->optNoPopup = 0;
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "OptionsNoDateCheck"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "OptionsNoTitleCheck"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "FontLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "FontButton"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopTimeLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopTimeSpin"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryXLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryYLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryXSpin"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "PopupGeometryYSpin"), TRUE);
  }
}

/********************************************************************************
 * Function: Callback on no title toggled
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update no title option in biff structure
 *           Destroy and create a new popup (according to new parameters)
 * Return  : /
 ********************************************************************************/
void on_OptionsNoTitleCheck_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  if (togglebutton->active) 
    biff->optNoTitle = 1;
  else
    biff->optNoTitle = 0;
  gtk_widget_hide (biff->popup);
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
}

/********************************************************************************
 * Function: Callback on no date toggled
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update no sound option in biff structure
 *           Make sound options sensitive or insensisitive
 *            according to toggle state
 * Return  : /
 ********************************************************************************/
void on_OptionsNoSoundCheck_toggled (GtkToggleButton *togglebutton, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (togglebutton->active) {
    biff->optNoSound = 1;
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundLabel"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundButton"), FALSE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundPlayButton"), FALSE);
  }
  else {
    biff->optNoSound = 0;
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundLabel"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundButton"), TRUE);
    gtk_widget_set_sensitive (lookup_widget (biff->mailbox, "SoundPlayButton"), TRUE);
  }
}

/********************************************************************************
 * Function: Callback on sound button selection
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show the file selection dialog and set selection mode to sound selection
 * Return  : /
 ********************************************************************************/
void on_SoundButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->selectionMode = SOUND_SELECTION_MODE;
  gtk_widget_show (biff->fileDialog);
}

/********************************************************************************
 * Function: Callback on sound play button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Play the current sound using system command "play"
 * Return  : /
 ********************************************************************************/
void on_SoundPlayButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  char command[1048];
  
  /* Do not remove the 1024, it prevents buffer overflow */
  sprintf (command, "play %.1024s", biff->sound);
  system (command);
}

/********************************************************************************
 * Function: Callback on font selection
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show the font selection dialog 
 * Return  : /
 ********************************************************************************/
void on_FontButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  gtk_widget_show (biff->fontDialog);
}

/********************************************************************************
 * Function: Callback on mail pixmap selection
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show the file selection dialog and set selection mode to mail selection
 * Return  : FALSE
 ********************************************************************************/
gboolean on_MailEventBox_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->selectionMode = MAIL_SELECTION_MODE;
  gtk_widget_show (biff->fileDialog);
  return FALSE;
}

/********************************************************************************
 * Function: Callback on nomail pixmap selection
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show the file selection dialog and set selection mode to nomail selection
 * Return  : FALSE
 ********************************************************************************/
gboolean on_NoMailEventBox_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->selectionMode = NOMAIL_SELECTION_MODE;
  gtk_widget_show (biff->fileDialog);
  return FALSE;
}

/********************************************************************************
 * Function: Callback on OK button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Shows the applet (GNOME) or window (GTK)
 *           Lauch poll timer and hide mailbox properties
 * Return  : /
 ********************************************************************************/
void on_OkButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  GtkEntry *editable = GTK_ENTRY (lookup_widget (biff->mailbox, "AddressEntry"));
  // Get the adress from the AddressEntry widget
  if ((biff->protocol == PROTOCOL_POP3) || (biff->protocol == PROTOCOL_IMAP4)) {
    // If no port has been supplied, then set the default port at the end of the address
    if (strchr (gtk_entry_get_text (editable), (int) ':') == NULL) {
      free (biff->address);
      biff->address = (char *) malloc (strlen (gtk_entry_get_text (editable))+1+4);
      if (biff->protocol == PROTOCOL_POP3)
	sprintf (biff->address, "%s:%.4d", gtk_entry_get_text (editable), DEFAULT_POP3_PORT);
      if (biff->protocol == PROTOCOL_IMAP4)
	sprintf (biff->address, "%s:%.4d", gtk_entry_get_text (editable), DEFAULT_IMAP4_PORT);
    }
    // Set the new address in the widget
    gtk_entry_set_text (editable, biff->address);
  }
  
#ifdef USE_GNOME
  gtk_widget_show (biff->applet);
#else
  gtk_widget_show (biff->window);
#endif
  if (biff->suspendMode == 0) {
    if (biff->polltag > -1)
      gtk_timeout_remove (biff->polltag);
    biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  }
  gtk_widget_hide (biff->mailbox);

#ifdef USE_GNOME
  /* Assume the user won't want us to pop up again */
  /* Not true if POP3 is used (no password saved) */
  if (biff->protocol != PROTOCOL_POP3)
    biff->optNoConfigure = 1;
  /* sync on preference close */
  applet_widget_sync_config (APPLET_WIDGET(biff->applet));
#endif
}

/********************************************************************************
 * Function: Callback on Quit Button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Free gbiff stuff and quit
 * Return  : /
 ********************************************************************************/
void on_QuitButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  int i;

  free (biff->title);
  free (biff->address);
  free (biff->user);
  free (biff->password);
  free (biff->sound);
  free (biff->mailPixmapFilename);
  free (biff->nomailPixmapFilename);
  free (biff->font);
  for (i=0; i<biff->unread; i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  parse_ISO_8859 ((char *) -1);
  gtk_exit (0);
}

/********************************************************************************
 * Function: Callback on Suspend Button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Suspend automatic mail check
 * Return  : /
 ********************************************************************************/
void on_SuspendButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  
  if (biff->suspendMode == 0) {
    gtk_label_set_text (GTK_LABEL (GTK_BIN (button)->child), "Continue");
    biff->suspendMode = 1;
    if (biff->polltag > -1) {
      gtk_timeout_remove (biff->poptag);
      biff->poptag = -1;
    };
  }
  else {
    gtk_label_set_text (GTK_LABEL (GTK_BIN (button)->child), "Suspend");
    biff->suspendMode = 0;
    if (biff->polltag == -1)
      biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  }
}



/* Font selection dialog */
/********************************************************************************
 * Function: Callback on font selector delete event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Hide font selection
 * Return  : TRUE
 ********************************************************************************/
gboolean on_FontSelectionDialog_delete (GtkObject *object, GdkEvent *event, gpointer user_data) {
  gtk_widget_hide (GTK_WIDGET(object));
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on font selector OK button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set the font and hide font selector
 * Return  : /
 ********************************************************************************/
void on_FontSelectionDialogOkButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->font != 0)
    free (biff->font);
  biff->font = (char *) strdup (gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog)));
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog), biff->font);
  gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG (biff->fontDialog), _("gbiff 2.6 by Nicolas Rougier"));
  gtk_widget_hide (biff->popup);
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
  gtk_widget_hide (biff->fontDialog);
}

/********************************************************************************
 * Function: Callback on font selector cancel button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Reset to the default font and hide font selector
 * Return  : /
 ********************************************************************************/
void on_FontSelectionDialogCancelButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog), biff->font);
  gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG (biff->fontDialog), _("gbiff 2.6 by Nicolas Rougier"));
  gtk_widget_hide (biff->popup);
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
  gtk_widget_hide (biff->fontDialog);
}

/********************************************************************************
 * Function: Callback on font selector apply button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set the font and hide font selector
 * Return  : /
 ********************************************************************************/
void on_FontSelectionDialogApplyButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->font != 0)
    free (biff->font);
  biff->font = (char *) strdup (gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog)));
  gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG (biff->fontDialog), biff->font);
  gtk_font_selection_dialog_set_preview_text (GTK_FONT_SELECTION_DIALOG (biff->fontDialog), _("gbiff 2.6 by Nicolas Rougier"));
  gtk_widget_hide (biff->popup);
  gtk_widget_destroy (biff->popup);
  biff->popup = create_Popup (biff);
  biff->popupState = 0;
  gtk_widget_hide (biff->fontDialog);
}





/* File selection dialog */
/********************************************************************************
 * Function: Callback on file selector delete event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Hide font selector
 * Return  : TRUE
 ********************************************************************************/
gboolean on_FileSelectionDialog_delete (GtkObject *object, GdkEvent *event, gpointer user_data) {
  gtk_widget_hide (GTK_WIDGET(object));
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on file selector OK button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set gbiff stuff according to selection mode
 * Return  : /
 ********************************************************************************/
void on_FileSelectionDialogOkButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  GtkWidget *newPixmap;
  GdkColormap *colormap;

  if (biff->selectionMode == SOUND_SELECTION_MODE) {
    if (biff->sound != 0)
      free (biff->sound);
    biff->sound = (char *) strdup (gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
    gtk_label_set_text (GTK_LABEL(GTK_BIN(lookup_widget (biff->mailbox, "SoundButton"))->child), strrchr(biff->sound, '/')+1);
  }
  else if (biff->selectionMode == ADDRESS_SELECTION_MODE) {
    if (biff->address != 0)
      free (biff->address);
    biff->address = (char *) strdup (gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
    gtk_entry_set_text (GTK_ENTRY(lookup_widget (biff->mailbox, "AddressEntry")), biff->address);
  }
  else if (biff->selectionMode == MAIL_SELECTION_MODE) {
    gtk_container_remove (GTK_CONTAINER (lookup_widget(biff->mailbox, "MailPixmapFrame")), lookup_widget(biff->mailbox, "MailPixmap"));
    gtk_widget_destroy (lookup_widget(biff->mailbox, "MailPixmap"));
    
    /* Make a new pixmap for the mailbox */
    newPixmap = create_pixmap (biff->mailbox, gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
    gtk_widget_set_name (newPixmap, "MailPixmap");
    gtk_widget_ref (newPixmap);
    gtk_object_set_data_full (GTK_OBJECT (biff->mailbox), "MailPixmap", newPixmap,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (newPixmap);
    gtk_container_add (GTK_CONTAINER (lookup_widget (biff->mailbox, "MailPixmapFrame")), newPixmap);
    gtk_misc_set_padding (GTK_MISC (newPixmap), 0, 3);
    
#ifdef USE_GNOME
    colormap = gtk_widget_get_colormap (biff->applet);
#else
    colormap = gtk_widget_get_colormap (biff->window);
#endif
    biff->mailGdkPixmap =
      gdk_pixmap_colormap_create_from_xpm (NULL, colormap, NULL, NULL, gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
  }
  else if (biff->selectionMode == NOMAIL_SELECTION_MODE) {
    gtk_container_remove (GTK_CONTAINER (lookup_widget(biff->mailbox, "NoMailPixmapFrame")), lookup_widget(biff->mailbox, "NoMailPixmap"));
    gtk_widget_destroy (lookup_widget(biff->mailbox, "NoMailPixmap"));
   
    /* Make a new pixmap for the mailbox */
    newPixmap = create_pixmap (biff->mailbox, gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
    gtk_widget_set_name (newPixmap, "NoMailPixmap");
    gtk_widget_ref (newPixmap);
    gtk_object_set_data_full (GTK_OBJECT (biff->mailbox), "NoMailPixmap", newPixmap,
			      (GtkDestroyNotify) gtk_widget_unref);
    gtk_widget_show (newPixmap);
    gtk_container_add (GTK_CONTAINER (lookup_widget (biff->mailbox, "NoMailPixmapFrame")), newPixmap);
    gtk_misc_set_padding (GTK_MISC (newPixmap), 0, 3);

#ifdef USE_GNOME
    colormap = gtk_widget_get_colormap (biff->applet);
#else
    colormap = gtk_widget_get_colormap (biff->window);
#endif
    biff->nomailGdkPixmap =
      gdk_pixmap_colormap_create_from_xpm (NULL, colormap, NULL, NULL, gtk_file_selection_get_filename(GTK_FILE_SELECTION(biff->fileDialog)));
  }
  
  gtk_widget_hide (biff->fileDialog);
}

/********************************************************************************
 * Function: Callback on file selector Cancel button
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Hide file selector
 * Return  : /
 ********************************************************************************/
void on_FileSelectionDialogCancelButton_clicked (GtkButton *button, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  gtk_widget_hide (biff->fileDialog);
}


/* Popup */
/********************************************************************************
 * Function: Callback on popup button press event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Hide popup, remore pop timer, set poll timer
 * Return  : TRUE
 ********************************************************************************/
gboolean on_Popup_button_press_event (GtkWidget *widget, GdkEventCrossing *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  biff->popupState = 0;
  gtk_widget_hide (biff->popup);
  if (biff->poptag > -1) {
    gtk_timeout_remove (biff->poptag);
    biff->poptag = -1;
  }
  if (biff->suspendMode == 0) {
    if (biff->polltag == -1)
      biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  }
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on popup enter event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Remove pop timer
 * Return  : TRUE
 ********************************************************************************/
gboolean on_Popup_enter_notify_event (GtkWidget *widget, GdkEventCrossing *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->poptag > -1) {
    gtk_timeout_remove (biff->poptag);
    biff->poptag = -1;
  }
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on popup leave event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set pop timer
 * Return  : TRUE
 ********************************************************************************/
gboolean on_Popup_leave_notify_event (GtkWidget *widget, GdkEventCrossing *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->popupState == 1) {
    if (biff->poptag > -1)
      gtk_timeout_remove (biff->poptag);
    biff->poptag = gtk_timeout_add (biff->poptime*1000, on_TimeoutPop_event, biff);
  };
  return (TRUE);
}


/* Window */
/********************************************************************************
 * Function: Get current window position value and set window spins accordingly
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update window geom spins
 * Return  : TRUE
 ********************************************************************************/
gboolean on_Window_leave_notify_event (GtkWidget *widget, GdkEventCrossing *event, gpointer user_data) {
  int xroot, yroot, x, y, width, height;
  Biff *biff = (Biff *) user_data;
  GtkWidget *DA = lookup_widget (biff->window, "DrawingArea");

  gdk_window_get_size (biff->window->window, &width, &height);
  gdk_window_get_position (biff->window->window, &x, &y);
  gdk_window_get_root_origin (DA->window, &xroot, &yroot);

  // For some window manager there is a problem between the collected window position
  // and the actual position of the window:
  // gdk_window_get_position should return position widget without WM frame
  // and gdk_window_get_root_origin should return position widget with WM frame
  // So if it's not he case we would have a problem when we set window position
  // Waiting for the next version of gtk, we have to add a hack
  // In future versions, we may be able to get WM decoration size
#ifdef HACK_WM
    x -= WM_DECORATION_WIDTH;
    y -= WM_DECORATION_HEIGHT;
#endif


  if (biff->windowX < 0) 
    biff->windowX = - (gdk_screen_width()-width-x);
  else
    biff->windowX = x;
  if (biff->windowY < 0) 
    biff->windowY = - (gdk_screen_height()-height-y);
  else
    biff->windowY = y;

  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (biff->mailbox, "WindowGeometryXSpin")), biff->windowX);
  gtk_spin_button_set_value (GTK_SPIN_BUTTON(lookup_widget (biff->mailbox, "WindowGeometryYSpin")), biff->windowY);

  return (TRUE);
}

/********************************************************************************
 * Function: Callback on window expose event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update window
 * Return  : FALSE
 ********************************************************************************/
gboolean on_WindowDrawingArea_expose (GtkWidget *widget, GdkEventExpose *event, gpointer user_data) {
  updateWindow ((Biff *) user_data, -2);
  return (FALSE);
}

/********************************************************************************
 * Function: Callback on window configure event
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Update window
 * Return  : TRUE
 ********************************************************************************/
gboolean on_WindowDrawingArea_configure (GtkWidget *widget, GdkEventConfigure *event, gpointer user_data) {
  updateWindow ((Biff *) user_data, -2);
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on window button press
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Button 1 => get mail status and show mail headers if necessary
 *           Button 3 => show mailbox properties
 * Return  : FALSE
 ********************************************************************************/
gboolean on_WindowDrawingArea_button_press (GtkWidget *widget, GdkEventButton *event,  gpointer user_data) {
  if (event->button == 1)
    on_AppletEventBox_button_press_event (widget, event, user_data);
  else if (event->button == 3)
    on_Applet_PropertiesMenu_selection (widget, user_data);
  return (FALSE);
}

/********************************************************************************
 * Function: Callback on window button release
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set pop timer if popup is on
 * Return  : FALSE
 ********************************************************************************/
gboolean on_WindowDrawingArea_button_release (GtkWidget *widget, GdkEventButton *event,  gpointer user_data) {
  if (event->button == 1)
    on_AppletEventBox_button_release_event (widget, event, user_data);
  return (FALSE);
}


/* Applet */
/********************************************************************************
 * Function: Callback on applet remove menu
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Free gbiff stuff and quit
 * Return  : /
 ********************************************************************************/
gboolean on_Applet_remove_event (GtkWidget *widget, GdkEvent *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  int i;

  free (biff->title);
  free (biff->address);
  free (biff->user);
  free (biff->password);
  free (biff->sound);
  free (biff->mailPixmapFilename);
  free (biff->nomailPixmapFilename);
  free (biff->font);
  for (i=0; i<biff->unread; i++) {
    free (biff->headers[i][NUMBER_INDEX]);
    free (biff->headers[i][SENDER_INDEX]);
    free (biff->headers[i][FROM_INDEX]);
    free (biff->headers[i][DATE_INDEX]);
  }
  parse_ISO_8859 ((char *) -1);

  return (FALSE);
};

/********************************************************************************
 * Function: Callback on applet properties menu
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show properties box and reset poll anmd pop timer
 * Return  : /
 ********************************************************************************/
void on_Applet_PropertiesMenu_selection (GtkWidget *widget, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  gtk_widget_hide (biff->popup);
  gtk_widget_show (biff->mailbox);
  if (biff->polltag > -1) {
    gtk_timeout_remove (biff->polltag);
    biff->polltag = -1;
  }
  if (biff->poptag > -1) {
    gtk_timeout_remove (biff->poptag);
    biff->poptag = -1;
  };
}

/********************************************************************************
 * Function: Callback on user button press on applet
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Check mail status and get headers if necessary
 *           Show the popup and set timers stuff
 * Return  : FALSE
 ********************************************************************************/
gboolean on_AppletEventBox_button_press_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;
  int status = -1, i;
  GtkCList *list;
#ifdef USE_GNOME
  char unread[4];
#endif
  gint x, y, width, height;

  /* Single click on button 1 */
  if (event->button == 1) {
    /* First hide popup */
    gtk_widget_hide (biff->popup);
    
    /* If the popup was on display, then the event is a request to hide it */
    if (biff->popupState == 1) {
      if (biff->poptag > -1)
	gtk_timeout_remove (biff->poptag);
      biff->poptag = -1;
      if (biff->polltag > -1)
	gtk_timeout_remove (biff->polltag);
      if (biff->suspendMode == 0) {
	biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
      }
      biff->popupState = 0;
      return (FALSE);
    };
    
    /* Check for new mail */
    if (biff->protocol == PROTOCOL_FILE) {
      status = FILE_checkmail (biff);
      if (status != -1)
	FILE_getheaders (biff);
    }
    else if (biff->protocol == PROTOCOL_MAILDIR) {
      status = MAILDIR_checkmail (biff);
      if (status != -1)
	MAILDIR_getheaders (biff);
    }
    else if (biff->protocol == PROTOCOL_POP3) {
      status = POP3_checkmail (biff);
      if (status != -1)
	POP3_getheaders (biff);
    }
    else if (biff->protocol == PROTOCOL_IMAP4) {
      status = IMAP4_checkmail (biff);
      if (status != -1)
	IMAP4_getheaders (biff);
    }
    else if (biff->protocol == PROTOCOL_MH) {
      status = MH_checkmail (biff);
      if (status != -1)
	MH_getheaders (biff);
    }
    
    
    if (status != -1) {
      /* Update header list */
      list = GTK_CLIST (lookup_widget(biff->popup, "PopupList"));
      gtk_clist_clear (list);
      for (i=0; i<biff->unread; i++)
	gtk_clist_append (list, biff->headers[i]);
      gtk_clist_columns_autosize (list);
      
#ifdef USE_GNOME
      /* Update messages number in applet */
      if (biff->unread < MAX_HEADER_IN_POPUP)
	sprintf (unread, "%.2d", biff->unread);
      else
	sprintf  (unread, "++");
      gtk_label_set_text (GTK_LABEL(lookup_widget (biff->applet, "MailNumberLabel")), unread);
    }
    else {
      sprintf  (unread, "--");
      gtk_label_set_text (GTK_LABEL(lookup_widget (biff->applet, "MailNumberLabel")), unread);
    };
#else
    /* Update messages number in applet */
      updateWindow (biff, status);
    }
    else {
      updateWindow (biff, status);
    };
#endif

#ifdef USE_GNOME
    /* Update pixmap in applet */
    if (status == STATUS_NO_MAIL)
      gtk_pixmap_set (GTK_PIXMAP(lookup_widget(biff->applet, "StatusPixmap")), biff->nomailGdkPixmap, 0);
    else if ((status == STATUS_NEW_MAIL) || (status == STATUS_OLD_MAIL))
      gtk_pixmap_set (GTK_PIXMAP(lookup_widget(biff->applet, "StatusPixmap")), biff->mailGdkPixmap, 0);
#endif    

    if ((status == STATUS_NEW_MAIL) || ((status == STATUS_OLD_MAIL) && (biff->unread > 0))) {
      gtk_widget_hide (biff->popup);
      gtk_widget_show (biff->popup);
      gdk_window_get_size (biff->popup->window, &width, &height);
      if (biff->popupX < 0)
        x = gdk_screen_width()  - width  + biff->popupX;
      else
        x = biff->popupX;
      if (biff->popupY < 0)
        y = gdk_screen_height() - height + biff->popupY; 
      else
        y = biff->popupY;
      gdk_window_move (biff->popup->window, x, y);
      /* Each time popup is up, poll timer is disabled */
      if (biff->polltag > -1) {
        gtk_timeout_remove (biff->polltag);
        biff->polltag = -1;
      };
      biff->popupState = 1;
    }
  }
  else {
      gtk_widget_hide (biff->popup);
      if (biff->polltag > -1)
        gtk_timeout_remove (biff->polltag);
      if (biff->suspendMode == 0) {
        biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
      }
      biff->popupState = 0;
    }
    
  /* Double click on button 1 */
  if ((event->type==GDK_2BUTTON_PRESS) && (event->button == 1)) {
      };
  return (FALSE);
}

/********************************************************************************
 * Function: Callback on user button release on applet
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Set pop timer if popup is on
 * Return  : FALSE
 ********************************************************************************/
gboolean on_AppletEventBox_button_release_event (GtkWidget *widget, GdkEventButton *event, gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->popupState == 1) {
    if (biff->poptag > -1)
      gtk_timeout_remove (biff->poptag);
    biff->poptag = gtk_timeout_add (biff->poptime*1000, on_TimeoutPop_event, biff);
  };
  return (FALSE);
}

#ifdef USE_GNOME
/********************************************************************************
 * Function: Callback on "about" menu
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Show a gnome about window
 * Return  : /
 ********************************************************************************/
void on_Applet_AboutMenu_selection (GtkWidget *widget, gpointer user_data) {
static const gchar *author[]={"Nicolas Rougier <Nicolas.Rougier@loria.fr>",NULL};  
  GtkWidget *About;
  About = gnome_about_new (_("gbiff applet"),
			   VERSION,
			   _("(C) 2000, 2001"),
			   author,
			   _("Released under the GNU general public license.\nCheck mail periodically and display headers when new mail has arrived."),
			   NULL);
  gtk_widget_show(About);
}

/********************************************************************************
 * Function: Callback on panel size change
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Change applet look according to panel size
 * Return  : /
 ********************************************************************************/
void on_Applet_change_pixel_size (GtkWidget *widget, int size, gpointer user_data) {
#ifdef HAVE_PANEL_PIXEL_SIZE
  Biff *biff = (Biff *) user_data;

  biff->size = size;
if ((biff->size < PIXEL_SIZE_STANDARD) && ((biff->orient == ORIENT_DOWN) || (biff->orient == ORIENT_UP)))
    applet_change_orientation (biff, APPLET_HORIZONTAL);
  else
    applet_change_orientation (biff, APPLET_VERTICAL);
#endif
}

/********************************************************************************
 * Function: Callback on panel orientation change
 * Input   : GTK stuff + user_data which is a biff structure
 * Output  : Change applet look according to panel orientation
 * Return  : /
 ********************************************************************************/
void on_Applet_change_orient (GtkWidget *widget, PanelOrientType orient, gpointer user_data) {
#ifdef HAVE_PANEL_PIXEL_SIZE
  Biff *biff = (Biff *) user_data;

  biff->orient = orient;
if ((biff->size < PIXEL_SIZE_STANDARD) && ((biff->orient == ORIENT_DOWN) || (biff->orient == ORIENT_UP)))
    applet_change_orientation (biff, APPLET_HORIZONTAL);
  else
    applet_change_orientation (biff, APPLET_VERTICAL);
#endif
}
#endif


/* Timers */
/********************************************************************************
 * Function: Callback on poll timeout
 * Input   : user_data which is a biff structure
 * Output  : Check mail status and get headers if necessary
 *           Show the popup and set timers stuff
 * Return  : /
 ********************************************************************************/
gboolean on_TimeoutPoll_event (gpointer user_data) {
Biff *biff = (Biff *) user_data;
  int status = -1, i;
  GtkCList *list;
#ifdef USE_GNOME
  char unread[4];
#endif
  char command[1048];
  gint x, y, width, height;

  /* Check for new mail */
  if (biff->protocol == PROTOCOL_FILE) {
    status = FILE_checkmail (biff);
    if (status != -1)
      FILE_getheaders (biff);
  }
  else if (biff->protocol == PROTOCOL_POP3) {
status = POP3_checkmail (biff);
if (status != -1)
      POP3_getheaders (biff);
  }
  else if (biff->protocol == PROTOCOL_MAILDIR) {
    status = MAILDIR_checkmail (biff);
    if (status != -1)
      MAILDIR_getheaders (biff);
  }
  else if (biff->protocol == PROTOCOL_IMAP4) {
    status = IMAP4_checkmail (biff);
    if (status != -1)
      IMAP4_getheaders (biff);
  }
  else if (biff->protocol == PROTOCOL_MH) {
    status = MH_checkmail (biff);
    if (status != -1)
      MH_getheaders (biff);
  }
  if (status != -1) {
    /* Update header list */
    list = GTK_CLIST (lookup_widget(biff->popup, "PopupList"));
    gtk_clist_clear (list);
    for (i=0; i<biff->unread; i++)
      gtk_clist_append (list, biff->headers[i]);
    gtk_clist_columns_autosize (list);
#ifdef USE_GNOME
    /* Update messages number in applet */
    if (biff->unread < MAX_HEADER_IN_POPUP)
      sprintf (unread, "%.2d", biff->unread);
    else
      sprintf  (unread, "++");
    gtk_label_set_text (GTK_LABEL(lookup_widget (biff->applet, "MailNumberLabel")), unread);
  }
  else {
    sprintf  (unread, "--");
    gtk_label_set_text (GTK_LABEL(lookup_widget (biff->applet, "MailNumberLabel")), unread);
  };
#else
    updateWindow (biff, status);
  }
  else {
    updateWindow (biff, status);
  }
#endif

#ifdef USE_GNOME
  /* Update pixmap in applet */
  if (status == STATUS_NO_MAIL)
    gtk_pixmap_set (GTK_PIXMAP(lookup_widget(biff->applet, "StatusPixmap")), biff->nomailGdkPixmap, 0);
  else if ((status == STATUS_NEW_MAIL) || (status == STATUS_OLD_MAIL))
    gtk_pixmap_set (GTK_PIXMAP(lookup_widget(biff->applet, "StatusPixmap")), biff->mailGdkPixmap, 0);
#endif

  if ((status == STATUS_NEW_MAIL) && (!biff->optNoPopup)) {
    gtk_widget_hide (biff->popup);
    gtk_widget_show (biff->popup);
    gdk_window_get_size (biff->popup->window, &width, &height);
    if (biff->popupX < 0)
      x = gdk_screen_width()  - width  + biff->popupX;
    else
      x = biff->popupX;
    if (biff->popupY < 0)
      y = gdk_screen_height() - height + biff->popupY; 
    else
      y = biff->popupY;
    gdk_window_move (biff->popup->window, x, y);
    if (biff->polltag > -1) {
      gtk_timeout_remove (biff->polltag);
      biff->polltag = -1;
    };
    biff->popupState = 1;
    if (biff->poptag > -1)
      gtk_timeout_remove (biff->poptag);
    biff->poptag = gtk_timeout_add (biff->poptime*1000, on_TimeoutPop_event, biff);
    biff->popupState = 1;
    if (!biff->optNoSound) {
      /* Do not remove the 1024, it prevents buffer overflow */
      sprintf (command, "play %.1024s&", biff->sound);
      system (command);
    }
  }
  return (TRUE);
}

/********************************************************************************
 * Function: Callback on popup timeout
 * Input   : user_data which is a biff structure
 * Output  : Remove the timer, hide the popup and set a new poll timer if necessary
 * Return  : /
 ********************************************************************************/
gboolean on_TimeoutPop_event (gpointer user_data) {
  Biff *biff = (Biff *) user_data;

  if (biff->poptag > -1) {
    gtk_timeout_remove (biff->poptag);
    biff->poptag = -1;
  };
  gtk_widget_hide (biff->popup);
  if (biff->polltag == -1)
    biff->polltag = gtk_timeout_add (biff->polltime*1000, on_TimeoutPoll_event, biff);
  biff->popupState = 0;

  return (FALSE);
}


