/*
  GNOME Comics Organizer
  Copyright (C) 1999-2000 Maxximum Software
   
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include <glib.h>
#include "fileops.h"
#include "fileops-helper.h"
#include "functions.h"
#include "comiclist.h"
#include "debug.h"

const gchar *CO19Ident     = "CO19_File";
const gchar *GCO_0_1_Ident = "GCO_0_1__";
const gchar *GCO_0_3_Ident = "GCO_0_3__";

GList *LoadCO19(GList *ComicList, FILE *fp)
{
  int c;
  TComic *TempComic;
  TIssue *TempIssue;
  GString * TempString;

  while((c=my_getc())!=EOF)
    {
      my_ungetc (c);
      TempComic = Init_Comic();

      while((c=my_getc())!='\x00')
	{
	  TempComic->CTitle = g_string_append_c(TempComic->CTitle, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CPublisher = g_string_append_c(TempComic->CPublisher, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CType = g_string_append_c(TempComic->CType, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CVolume = g_string_append_c(TempComic->CVolume, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CGroup = g_string_append_c(TempComic->CGroup, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->Comment = g_string_append_c(TempComic->Comment, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->LinkFile = g_string_append_c(TempComic->LinkFile, c);
	}
      TempComic->IsLink = (TempComic->LinkFile->len>0);

      /* Now load the Issues */
      while((c=my_getc())!='\x00')
	{
	  my_ungetc (c);

	  TempIssue = Init_Issue();

  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->IssueNo = g_string_append_c(TempIssue->IssueNo, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Writer = g_list_append(TempIssue->Writer, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Penciller = g_list_append(TempIssue->Penciller, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Inker = g_list_append(TempIssue->Inker, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Title = g_list_append(TempIssue->Title, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Location = g_string_append_c(TempIssue->Location, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Price = g_string_append_c(TempIssue->Price, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->CurrentPrice = g_string_append_c(TempIssue->CurrentPrice, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Image = g_string_append_c(TempIssue->Image, c);
  	    } 

	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  if (strcmp(TempString->str, "Poor (P)")==0)
	    TempIssue->Condition = 0;
	  if (strcmp(TempString->str, "Fair (FR)")==0)
	    TempIssue->Condition = 1;
	  if (strcmp(TempString->str, "Good (G)")==0) 
	    TempIssue->Condition = 2;
	  if (strcmp(TempString->str, "Very Good (VG)")==0)
	    TempIssue->Condition = 3;
	  if (strcmp(TempString->str, "Fine (F)")==0)
	    TempIssue->Condition = 4;
	  if (strcmp(TempString->str, "Fine-Very Fine (VFV)")==0)
	    TempIssue->Condition = 5;
	  if (strcmp(TempString->str, "Very Fine (VF)")==0)
	    TempIssue->Condition = 6;
	  if (strcmp(TempString->str, "Near Mint (NM)")==0)
	    TempIssue->Condition = 7;
	  if (strcmp(TempString->str, "Mint (M)")==0)
	    TempIssue->Condition = 8;


  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Comment = g_string_append_c(TempIssue->Comment, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      /*
	      TempIssue->Date = g_string_append_c(TempIssue->Date, c);
	      */

	      /* FIXME - doesn't import dates... */
  	    } 

	  /* Now the TIssue structure is filled - then insert into list */
	  TempComic->IssueNumbers = g_list_insert_sorted(TempComic->IssueNumbers, TempIssue,
							 CompareTIssue_IssueNo);
	}
      
      /* Now the TComic structure is filled - then insert into list */
      ComicList = g_list_insert_sorted(ComicList, TempComic, CompareTComic_Title);
    }

  return ComicList;
}

GList *LoadGCO_0_1(GList *ComicList, FILE *fp)
{
  /* Read largest used UID and set it using SetUID() */
  /* Read ComicList (remember to read the UIDs of the individual Comics and Issues, etc.) */
  int c;
  guint32 LargestUID;
  TComic *TempComic;
  TIssue *TempIssue;
  GString * TempString;

  TempString = g_string_new(NULL);
  while((c=my_getc())!='\x00')
    {
      TempString = g_string_append_c(TempString, c);
    }
  LargestUID = atof(TempString->str);
  g_string_free(TempString, TRUE);

  while((c=my_getc())!=EOF)
    {
      my_ungetc (c);
      TempComic = Init_Comic();

      while((c=my_getc())!='\x00')
	{
	  TempComic->CTitle = g_string_append_c(TempComic->CTitle, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CPublisher = g_string_append_c(TempComic->CPublisher, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CType = g_string_append_c(TempComic->CType, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CVolume = g_string_append_c(TempComic->CVolume, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CGroup = g_string_append_c(TempComic->CGroup, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->Comment = g_string_append_c(TempComic->Comment, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->LinkFile = g_string_append_c(TempComic->LinkFile, c);
	}
      TempComic->IsLink = (TempComic->LinkFile->len>0);
      TempString = g_string_new(NULL);
      while((c=my_getc())!='\x00')
	{
	  TempString = g_string_append_c(TempString, c);
	}
      TempComic->UID = atof(TempString->str);
      g_string_free(TempString, TRUE);

      /* Now load the Issues */
      while((c=my_getc())!='\x00')
	{
	  my_ungetc (c);
	  TempIssue = Init_Issue();

  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->IssueNo = g_string_append_c(TempIssue->IssueNo, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Writer = g_list_append(TempIssue->Writer, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Penciller = g_list_append(TempIssue->Penciller, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Inker = g_list_append(TempIssue->Inker, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Title = g_list_append(TempIssue->Title, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Location = g_string_append_c(TempIssue->Location, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Price = g_string_append_c(TempIssue->Price, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->CurrentPrice = g_string_append_c(TempIssue->CurrentPrice, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Image = g_string_append_c(TempIssue->Image, c);
  	    } 
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->Condition = atof(TempString->str);
	  g_string_free(TempString, TRUE);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Comment = g_string_append_c(TempIssue->Comment, c);
  	    } 
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->Date = atof(TempString->str);
	  g_string_free(TempString, TRUE);
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->UID = atof(TempString->str);
	  g_string_free(TempString, TRUE);

	  /* Now the TIssue structure is filled - then insert into list */
	  TempComic->IssueNumbers = g_list_insert_sorted(TempComic->IssueNumbers, TempIssue,
							 CompareTIssue_IssueNo);
	}
      /* Now the TComic structure is filled - then insert into list */
      ComicList = g_list_insert_sorted(ComicList, TempComic, CompareTComic_Title);
    }

  SetUID(LargestUID);
  return ComicList;
}

GList *LoadGCO_0_3(GList *ComicList, FILE *fp)
{
  /* Read largest used UID and set it using SetUID() */
  /* Read ComicList (remember to read the UIDs of the individual Comics and Issues, etc.) */
  int c;
  guint32 LargestUID;
  TComic *TempComic;
  TIssue *TempIssue;
  GString * TempString;

  TempString = g_string_new(NULL);
  while((c=my_getc())!='\x00')
    {
      TempString = g_string_append_c(TempString, c);
    }
  LargestUID = atof(TempString->str);
  g_string_free(TempString, TRUE);

  while((c=my_getc())!=EOF)
    {
      my_ungetc (c);
      TempComic = Init_Comic();

      while((c=my_getc())!='\x00')
	{
	  TempComic->CTitle = g_string_append_c(TempComic->CTitle, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CPublisher = g_string_append_c(TempComic->CPublisher, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CType = g_string_append_c(TempComic->CType, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CVolume = g_string_append_c(TempComic->CVolume, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->CGroup = g_string_append_c(TempComic->CGroup, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->Comment = g_string_append_c(TempComic->Comment, c);
	}
      while((c=my_getc())!='\x00')
	{
	  TempComic->LinkFile = g_string_append_c(TempComic->LinkFile, c);
	}
      TempComic->IsLink = (TempComic->LinkFile->len>0);
      TempString = g_string_new(NULL);
      while((c=my_getc())!='\x00')
	{
	  TempString = g_string_append_c(TempString, c);
	}
      TempComic->UID = atof(TempString->str);
      g_string_free(TempString, TRUE);

      /* Now load the Issues */
      while((c=my_getc())!='\x00')
	{
	  my_ungetc (c);
	  TempIssue = Init_Issue();

  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->IssueNo = g_string_append_c(TempIssue->IssueNo, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Writer = g_list_append(TempIssue->Writer, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Penciller = g_list_append(TempIssue->Penciller, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Inker = g_list_append(TempIssue->Inker, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      my_ungetc (c);
	      TempString = g_string_new(NULL);

	      while((c=my_getc())!='\x02')
		{
		  TempString = g_string_append_c(TempString, c);
		}
	      TempIssue->Title = g_list_append(TempIssue->Title, TempString);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Location = g_string_append_c(TempIssue->Location, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Price = g_string_append_c(TempIssue->Price, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->CurrentPrice = g_string_append_c(TempIssue->CurrentPrice, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->PurchasePrice = g_string_append_c(TempIssue->PurchasePrice, c);
  	    } 
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Image = g_string_append_c(TempIssue->Image, c);
  	    } 
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->Condition = atof(TempString->str);
	  g_string_free(TempString, TRUE);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempIssue->Comment = g_string_append_c(TempIssue->Comment, c);
  	    } 
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->Date = atof(TempString->str);
	  g_string_free(TempString, TRUE);
	  TempString = g_string_new(NULL);
  	  while((c=my_getc())!='\x01')
  	    { 
	      TempString = g_string_append_c(TempString, c);
  	    }
	  TempIssue->UID = atof(TempString->str);
	  g_string_free(TempString, TRUE);

	  /* Now the TIssue structure is filled - then insert into list */
	  TempComic->IssueNumbers = g_list_insert_sorted(TempComic->IssueNumbers, TempIssue,
							 CompareTIssue_IssueNo);
	}
      /* Now the TComic structure is filled - then insert into list */
      ComicList = g_list_insert_sorted(ComicList, TempComic, CompareTComic_Title);
    }

  SetUID(LargestUID);
  return ComicList;
}

GList *LoadFile(GList *ComicList, FILE *fp)
{
  switch (FileCheck(fp)) 
    {
    case CO19: 
      return LoadCO19(ComicList, fp);
      break;
    case GCO_0_1:
      return LoadGCO_0_1(ComicList, fp);
      break;
    case GCO_0_3:
      return LoadGCO_0_3(ComicList, fp);
      break;
    case NotCO: 
      dprintf("File is not a recognized filetype...\n");
      return NULL;
    default:
      dprintf("Something unexpected happened...\n");
      return NULL;
    }
}

void SaveFile(GList *ComicList, FILE *fp)
{
  /* Write GCO_0_3_Ident */
  /* Write largest used UID */
  /* Write ComicList (remember to save the UIDs of the individual Comics and Issues, etc.) */
  int i, j, k, l, m, n;
  TComic *TempComic;
  TIssue *TempIssue;
  GString * TempString;

  my_putc('\x00', fp);
  my_puts(GCO_0_3_Ident, fp);
  my_putc('\x00', fp);
  my_puts(itoa(GetUID()+1), fp); 
  my_putc('\x00', fp);

  j = g_list_length(ComicList);
  for (i=0;i<j;i++)
    {
      /* Save the Comic info */
      TempComic = (g_list_nth(ComicList,i))->data;
      my_puts(TempComic->CTitle->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->CPublisher->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->CType->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->CVolume->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->CGroup->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->Comment->str, fp);
      my_putc('\x00', fp);
      my_puts(TempComic->LinkFile->str, fp);
      my_putc('\x00', fp);
      my_puts(itoa(TempComic->UID), fp); 
      my_putc('\x00', fp);

      /* Now save the Issues */
      k = g_list_length(TempComic->IssueNumbers);
      for (l=0;l<k;l++)
	{
	  TempIssue = (g_list_nth(TempComic->IssueNumbers,l))->data;
	  my_puts(TempIssue->IssueNo->str, fp);
	  my_putc('\x01', fp);

	  m = g_list_length(TempIssue->Writer);
	  for (n=0;n<m;n++)
	    {
	      TempString = (g_list_nth(TempIssue->Writer, n)->data);
	      my_puts(TempString->str, fp);
	      my_putc('\x02', fp);
	    }
	  my_putc('\x01', fp);

	  m = g_list_length(TempIssue->Penciller);
	  for (n=0;n<m;n++)
	    {
	      TempString = (g_list_nth(TempIssue->Penciller, n)->data);
	      my_puts(TempString->str, fp);
	      my_putc('\x02', fp);
	    }
	  my_putc('\x01', fp);

	  m = g_list_length(TempIssue->Inker);
	  for (n=0;n<m;n++)
	    {
	      TempString = (g_list_nth(TempIssue->Inker, n)->data);
	      my_puts(TempString->str, fp);
	      my_putc('\x02', fp);
	    }
	  my_putc('\x01', fp);

	  m = g_list_length(TempIssue->Title);
	  for (n=0;n<m;n++)
	    {
	      TempString = (g_list_nth(TempIssue->Title, n)->data);
	      my_puts(TempString->str, fp);
	      my_putc('\x02', fp);
	    }
	  my_putc('\x01', fp);

	  my_puts(TempIssue->Location->str, fp);
	  my_putc('\x01', fp);
	  my_puts(TempIssue->Price->str, fp);
	  my_putc('\x01', fp);
	  my_puts(TempIssue->CurrentPrice->str, fp);
	  my_putc('\x01', fp);
	  my_puts(TempIssue->PurchasePrice->str, fp);
	  my_putc('\x01', fp);
	  my_puts(TempIssue->Image->str, fp);
	  my_putc('\x01', fp);
	  my_puts(itoa(TempIssue->Condition), fp);
	  my_putc('\x01', fp);
	  my_puts(TempIssue->Comment->str, fp);
	  my_putc('\x01', fp);
	  my_puts(itoa(TempIssue->Date), fp);
	  my_putc('\x01', fp);
	  my_puts(itoa(TempIssue->UID), fp);
	  my_putc('\x01', fp);
	}
      my_putc('\x00', fp);
    }
}


TFileType FileCheck(FILE *fp)
{
  gshort i;
  gchar c;
  GString *s;

  s = g_string_new(NULL);
  for(i=0;i<=10;i++)
    {
      if ((c=my_getc())==EOF)
	g_print("Error...\n");
      if (c!=0) /* Idents are surrounded by #0's */
	s = g_string_append_c(s, c);
    }

  if (strcmp(s->str, CO19Ident)==0)
    return CO19;
  if (strcmp(s->str, GCO_0_1_Ident)==0)
    return GCO_0_1;
  if (strcmp(s->str, GCO_0_3_Ident)==0)
    return GCO_0_3;
  else
    return NotCO;
}

int ExportToASCIIFile(GList *clist, GString *fname)
{
  TComic *TempComic;
  TIssue *TempIssue;
  guint32 i,j,k,l,m,n;
  GString *String;
  FILE *fp;

  fp = fopen (fname->str, "w");
  if (fp == NULL)
    {
      /* Unable to open file */
      return 1;
    }

  fprintf(fp, "%i comic title(s)\n\n\n\n", j=g_list_length(clist));
  for(i=0;i<j;i++)
    {
      TempComic = (g_list_nth(clist,i))->data;
      fprintf(fp, "%s", TempComic->CTitle->str);
      if(TempComic->IsLink)
	fprintf(fp, " (is a link to %s)\n", TempComic->LinkFile->str);
      else
	fprintf(fp, " (Vol. %s, %s, %s)\n", TempComic->CVolume->str,
		TempComic->CType->str, TempComic->CPublisher->str);
      fprintf(fp, "Comment: %s\n", TempComic->Comment->str);
      fprintf(fp, "Comic UID: %i\n", TempComic->UID);

      if(TempComic->IssueNumbers!=NULL)
	{
	  fprintf(fp, "%i issue(s)\n", l=g_list_length(TempComic->IssueNumbers));

	  for(k=0;k<l;k++)
	    {
	      TempIssue = (g_list_nth(TempComic->IssueNumbers,k))->data;

	      fprintf(fp, "  Issue #%s:\n",TempIssue->IssueNo->str);

	      fprintf(fp, "    Writer(s)\n");
	      n=g_list_length(TempIssue->Writer);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Writer, m)->data;
		  fprintf(fp, "      %s\n", String->str);
		}

	      fprintf(fp, "    Penciller(s)\n");
	      n=g_list_length(TempIssue->Penciller);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Penciller, m)->data;
		  fprintf(fp, "      %s\n", String->str);
		}

	      fprintf(fp, "    Inker(s)\n");
	      n=g_list_length(TempIssue->Inker);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Inker, m)->data;
		  fprintf(fp, "      %s\n", String->str);
		}

	      fprintf(fp, "    StoryTitle(s)\n");
	      n=g_list_length(TempIssue->Title);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Title, m)->data;
		  fprintf(fp, "      %s\n", String->str);
		}

	      fprintf(fp, "    Location: %s\n", TempIssue->Location->str);
	      fprintf(fp, "    CoverPrice: %s\n", TempIssue->Price->str);
	      fprintf(fp, "    Current Price: %s\n", TempIssue->CurrentPrice->str);
	      fprintf(fp, "    Purchase Price: %s\n", TempIssue->PurchasePrice->str);
	      fprintf(fp, "    Image: %s\n", TempIssue->Image->str);
	      fprintf(fp, "    Condition: ");

	      switch (TempIssue->Condition)
		{
		case Mint:
		    fprintf(fp, "Mint\n");
		    break;
		case NearMint:      
		  fprintf(fp, "NearMint\n");
		  break;
		case VeryFine: 
		  fprintf(fp, "VeryFine\n");
		  break;
		case FineVeryFine: 
		  fprintf(fp, "FineVeryFine\n");
		  break;
		case Fine: 
		  fprintf(fp, "Fine\n");
		  break;
		case VeryGood:  
		  fprintf(fp, "VeryGood\n");
		  break;
		case Good:  
		  fprintf(fp, "Good\n");
		  break;
		case Fair:  
		  fprintf(fp, "Fair\n");
		  break;
		case Poor: 
		  fprintf(fp, "Poor\n");
		  break;
		default: 
		  fprintf(fp, "Unknown\n");
		  break;
		}

	      fprintf(fp, "    Comment: %s\n", TempIssue->Comment->str);
	      fprintf(fp, "    Issue UID: %i\n", TempIssue->UID);
	    }
	}
      else
	{
	  fprintf(fp, "no issues\n");
	}
      fprintf(fp, "\n\n");
    }
  
  fclose(fp);
  return 0;
}

void write_html_preamble (FILE *fp, char *title)
{
  fprintf (fp, "<html>");
  fprintf (fp, "<title>");
  fprintf (fp, title);
  fprintf (fp, "</title>");
  fprintf (fp, "<body bgcolor=\"white\">");
}

void write_html_end (FILE *fp)
{
  fprintf (fp, "</body>");
  fprintf (fp, "</html>");
}

void write_html_h1_start (FILE *fp)
{
  fprintf (fp, "<h1>");
}

void write_html_h1_end (FILE *fp)
{
  fprintf (fp, "</h1>");
}

void write_html_h2_start (FILE *fp)
{
  fprintf (fp, "<h2>");
}

void write_html_h2_end (FILE *fp)
{
  fprintf (fp, "</h2>");
}

void write_html_h3_start (FILE *fp)
{
  fprintf (fp, "<h3>");
}

void write_html_h3_end (FILE *fp)
{
  fprintf (fp, "</h3>");
}

void write_html_br (FILE *fp)
{
  fprintf (fp, "<br>");
}

void write_html_p (FILE *fp)
{
  fprintf (fp, "<p>");
}

void write_html_link (FILE *fp, GString *linkdest, GString *linkname)
{
  fprintf (fp, "<a href=\"%s\">%s</a>", linkdest->str, linkname->str);
}

void write_html_ul_start (FILE *fp)
{
  fprintf (fp, "<ul>");
}

void write_html_ul_end (FILE *fp)
{
  fprintf (fp, "</ul>");
}

void write_html_li_start (FILE *fp)
{
  fprintf (fp, "<li>");
}

void write_html_li_end (FILE *fp)
{
  fprintf (fp, "</li>");
}

void write_html_table_start (FILE *fp)
{
  fprintf (fp, "<table cellpadding=\"2\" cellspacing=\"2\" border=\"1\" width=\"75%%\">");
  fprintf (fp, "<tbody>");
}

void write_html_table_end (FILE *fp)
{
  fprintf (fp, "</table>");
}

void write_html_td_start (FILE *fp)
{
  fprintf (fp, "<td>");
}

void write_html_td_end (FILE *fp)
{
  fprintf (fp, "</td>");
}

void write_html_tr_start (FILE *fp)
{
  fprintf (fp, "<tr>");
}

void write_html_tr_end (FILE *fp)
{
  fprintf (fp, "</tr>");
}

void write_html_center_start (FILE *fp)
{
  fprintf (fp, "<center>");
}

void write_html_center_end (FILE *fp)
{
  fprintf (fp, "</center>");
}

void write_html_image (FILE *fp, GString *imgfile, GString *alt)
{
  if (strcmp (imgfile->str, "")!=0)
    {
      fprintf (fp, "<img src=\"%s\" alt=\"%s\">", imgfile->str, alt->str);
    }
}

gchar *generate_subdirname (GString *dirname, TComic *TempComic)
{
  GString *returnme;
  int i;

  returnme = g_string_new (NULL);

  g_string_append (returnme, dirname->str);

  /* sanitize string  */
  /* g_string_append (returnme, TempComic->CTitle->str); */
  for (i=0; i<strlen(TempComic->CTitle->str); i++)
    {
      if (TempComic->CTitle->str[i] == ':' ||
	  TempComic->CTitle->str[i] == '/' ||
	  TempComic->CTitle->str[i] == '\\' ||
	  TempComic->CTitle->str[i] == ';' ||
	  TempComic->CTitle->str[i] == '?' ||
	  TempComic->CTitle->str[i] == '&' ||
	  TempComic->CTitle->str[i] == ' ')
	{
	  g_string_append_c (returnme, '_');
	}
      else
	{
	  g_string_append_c (returnme, TempComic->CTitle->str[i]);
	}
    }
  g_string_append (returnme, "_");
  g_string_append (returnme, itoa (TempComic->UID));

  return returnme->str;
}

gchar *generate_comic_filename (GString *dirname, TComic *TempComic)
{
  /* The filename is generated from the subdirname by adding .html */
  GString *returnme;

  returnme = g_string_new (generate_subdirname (dirname, TempComic));

  g_string_append (returnme, ".html");

  return returnme->str;
}

gchar *generate_issue_filename (GString *dirname, TIssue *TempIssue)
{
  GString *returnme;
  int i;

  returnme = g_string_new (NULL);

  g_string_append (returnme, dirname->str);
  g_string_append (returnme, "/");

  /* sanitize string  */
  /* g_string_append (returnme, TempIssue->IssueNo->Str); */
  for (i=0; i<strlen(TempIssue->IssueNo->str); i++)
    {
      if (TempIssue->IssueNo->str[i] == ':' ||
	  TempIssue->IssueNo->str[i] == '/' ||
	  TempIssue->IssueNo->str[i] == '\\' ||
	  TempIssue->IssueNo->str[i] == ';' ||
	  TempIssue->IssueNo->str[i] == '?' ||
	  TempIssue->IssueNo->str[i] == '&' ||
	  TempIssue->IssueNo->str[i] == ' ')
	{
	  g_string_append_c (returnme, '_');
	}
      else
	{
	  g_string_append_c (returnme, TempIssue->IssueNo->str[i]);
	}
    }
  g_string_append (returnme, "_");
  g_string_append (returnme, itoa (TempIssue->UID));
  g_string_append (returnme, ".html");

  return returnme->str;
}

int ExportToHTMLFile(GList *clist, GString *dirname)
{
  TComic *TempComic;
  TIssue *TempIssue;
  guint32 i,j,k,l,m,n;
  GString *String, *indexname, *fname, *dname;
  FILE *indexfp = NULL;
  FILE *ifp  = NULL;
  FILE *cfp = NULL;

  indexname = g_string_new (dirname->str);
  g_string_append (indexname, "index.html");
  indexfp = fopen (indexname->str, "w");
  if (indexfp == NULL)
    {
      /* Unable to open file */
      return 1;
    }

  j=g_list_length(clist);

  write_html_preamble (indexfp, "My Comics");
  write_html_h1_start (indexfp);
  fprintf(indexfp, "%i comic title(s)\n\n\n\n", j);
  write_html_h1_end (indexfp);
  write_html_br (indexfp);
  fprintf (indexfp, "\n");


  fname = g_string_new (NULL);
  dname = g_string_new (NULL);

  for(i=0;i<j;i++)
    {
      TempComic = (g_list_nth(clist,i))->data;

      /* Construct unique filename */
      g_string_assign (fname, "");
      g_string_append (fname, generate_comic_filename (dirname, TempComic));
      
      /* open file */
      cfp = fopen (fname->str, "w");
      if (cfp == NULL)
	{
	  /* Unable to open file */
	  return 1;
	}
    
      write_html_preamble (cfp, TempComic->CTitle->str);

      String = g_string_new (TempComic->CTitle->str);
      g_string_append (String, " (Vol. ");
      g_string_append (String, TempComic->CVolume->str);
      g_string_append (String, ")");
      write_html_link (indexfp, fname, String);
      write_html_br (indexfp);
      fprintf (indexfp, "\n");

      write_html_h1_start (cfp);
      fprintf(cfp, "%s", TempComic->CTitle->str);
      if(TempComic->IsLink)
	fprintf(cfp, " (is a link to %s)\n", TempComic->LinkFile->str);
      else
	fprintf(cfp, " (Vol. %s, %s, %s)\n", TempComic->CVolume->str,
		TempComic->CType->str, TempComic->CPublisher->str);
      write_html_h1_end (cfp);
      write_html_br (cfp);
      fprintf(cfp, "Comment: %s\n", TempComic->Comment->str);
      write_html_p (cfp);

      if(TempComic->IssueNumbers!=NULL)
	{
	  write_html_h2_start (cfp);
	  fprintf(cfp, "%i issue(s)\n", l=g_list_length(TempComic->IssueNumbers));
	  write_html_h2_end (cfp);
	  write_html_br (cfp);

	  g_string_assign (dname, "");
	  g_string_append (dname, generate_subdirname (dirname, TempComic));
	  mkdir (dname->str, 0755); /* read/write/execute by user, read/execute by anyone else - umask permitting */
      
	  for(k=0;k<l;k++)
	    {
	      TempIssue = (g_list_nth(TempComic->IssueNumbers,k))->data;

	      g_string_assign (fname, "");
	      g_string_append (fname, generate_issue_filename (dname, TempIssue));


	      /* open file */
	      ifp = fopen (fname->str, "w");
	      if (ifp == NULL)
		{
		  /* Unable to open file */
		  return 1;
		}


	      String = g_string_new ("#");
	      g_string_append (String, TempIssue->IssueNo->str);
	      write_html_link (cfp, fname, String);
	      write_html_br (cfp);
	      fprintf (cfp, "\n");


	      g_string_assign (String, TempComic->CTitle->str);
	      g_string_append (String, " - #");
	      g_string_append (String, TempIssue->IssueNo->str);
	      write_html_preamble (ifp, String->str);
    
	      write_html_center_start (ifp);
	      write_html_h3_start (ifp);
	      fprintf(ifp, "  Issue #%s:\n", TempIssue->IssueNo->str);
	      write_html_h3_end (ifp);

	      write_html_image (ifp, TempIssue->Image, TempIssue->IssueNo);

	      write_html_table_start (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Writer(s)");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      n=g_list_length(TempIssue->Writer);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Writer, m)->data;
		  fprintf(ifp, "      %s\n", String->str);
		  write_html_br (ifp);
		}
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Penciller(s)");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      n=g_list_length(TempIssue->Penciller);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Penciller, m)->data;
		  fprintf(ifp, "      %s\n", String->str);
		  write_html_br (ifp);
		}
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);
	      
	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Inker(s)");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      n=g_list_length(TempIssue->Inker);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Inker, m)->data;
		  fprintf(ifp, "      %s\n", String->str);
		  write_html_br (ifp);
		}
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, " StoryTitle(s)");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      n=g_list_length(TempIssue->Title);
	      for(m=0;m<n;m++)
		{ 
		  String = g_list_nth(TempIssue->Title, m)->data;
		  fprintf(ifp, "      %s\n", String->str);
		  write_html_br (ifp);
		}
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Location");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      fprintf (ifp, "%s", TempIssue->Location->str);
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);


	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "CoverPrice");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "%s", TempIssue->Price->str);
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);
	      
	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Current Price");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "%s", TempIssue->CurrentPrice->str);
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Purchase Price");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "%s", TempIssue->PurchasePrice->str);
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Condition");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);

	      switch (TempIssue->Condition)
		{
		case Mint:
		  fprintf(ifp, "Mint\n");
		  break;
		case NearMint:      
		  fprintf(ifp, "NearMint\n");
		  break;
		case VeryFine: 
		  fprintf(ifp, "VeryFine\n");
		  break;
		case FineVeryFine: 
		  fprintf(ifp, "FineVeryFine\n");
		  break;
		case Fine: 
		  fprintf(ifp, "Fine\n");
		  break;
		case VeryGood:  
		  fprintf(ifp, "VeryGood\n");
		  break;
		case Good:  
		  fprintf(ifp, "Good\n");
		  break;
		case Fair:  
		  fprintf(ifp, "Fair\n");
		  break;
		case Poor: 
		  fprintf(ifp, "Poor\n");
		  break;
		default: 
		  fprintf(ifp, "Unknown\n");
		  break;
		}
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_tr_start (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "Comment");
	      write_html_td_end (ifp);
	      write_html_td_start (ifp);
	      fprintf(ifp, "%s", TempIssue->Comment->str);
	      write_html_td_end (ifp);
	      write_html_tr_end (ifp);

	      write_html_table_end (ifp);
	      write_html_center_end (ifp);

	      write_html_end (ifp);
	      fclose(ifp);
	    }
	}
      else
	{
	  write_html_ul_start (cfp);
	  write_html_li_start (cfp);
	  fprintf(ifp, "no issues\n");
	  write_html_li_end (cfp);
	  write_html_ul_end (cfp);
	}
      write_html_end (cfp);
      fclose(cfp);
    }
  
  write_html_end (indexfp);
  fclose(indexfp);
  return 0;
}
