
#line 1 "./zscan_rfc1035.rl"
/* Copyright © 2012 Brandon L Black <blblack@gmail.com> and Jay Reitz <jreitz@gmail.com>
 *
 * This file is part of gdnsd.
 *
 * gdnsd is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * gdnsd is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with gdnsd.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include "zscan_rfc1035.h"

#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <setjmp.h>
#include <errno.h>

#include "conf.h"
#include "ltree.h"
#include "ltarena.h"
#include <gdnsd/alloc.h>
#include <gdnsd/log.h>
#include <gdnsd/misc.h>

#ifndef INET6_ADDRSTRLEN
#define INET6_ADDRSTRLEN 46
#endif

/*
 * The longest possible tstart-based token is a maximum-length
 *  quoted TXT string, which given autosplit, can be up to
 *  roughly 64K, which means buffer size has to be that big
 *  to accommodate that.  There probably wouldn't be any bugs
 *  going down to something reasonable like 4K, but it would
 *  cause parse errors if anyone tried to use longer TXT strings.
 * Another important thing: for integers, we use strtoul() directly
 *  on the buffer itself.  In the normal case this works because
 *  there is always some non-integer after it in the buffer, which
 *  halts strtoul().  The corner case is if the last digit of an
 *  integer happened to be the last byte of the buffer.  This
 *  is why we allocate one extra buffer byte and set it to zero.
 */
#define MAX_BUFSIZE 65536

#define parse_error(_fmt, ...) \
    do {\
        log_err("rfc1035: Zone %s: Zonefile parse error at line %u: " _fmt,logf_dname(z->zone->dname),z->lcount,__VA_ARGS__);\
        siglongjmp(z->jbuf, 1);\
    } while(0)

#define parse_error_noargs(_fmt) \
    do {\
        log_err("rfc1035: Zone %s: Zonefile parse error at line %u: " _fmt,logf_dname(z->zone->dname),z->lcount);\
        siglongjmp(z->jbuf, 1);\
    } while(0)

typedef struct {
    uint8_t  ipv6[16];
    uint32_t ipv4;
    bool     in_paren;
    bool     zn_err_detect;
    bool     lhs_is_ooz;
    unsigned lcount;
    unsigned num_texts;
    unsigned def_ttl;
    unsigned uval;
    unsigned ttl;
    unsigned ttl_min;
    unsigned uv_1;
    unsigned uv_2;
    unsigned uv_3;
    unsigned uv_4;
    unsigned uv_5;
    unsigned rfc3597_data_len;
    unsigned rfc3597_data_written;
    unsigned limit_v4;
    unsigned limit_v6;
    uint8_t* rfc3597_data;
    zone_t* zone;
    const char* tstart;
    uint8_t  origin[256];
    uint8_t  lhs_dname[256];
    uint8_t  rhs_dname[256];
    union {
        uint8_t eml_dname[256];
        char    rhs_dyn[256];
    };
    uint8_t** texts;
    sigjmp_buf jbuf;
} zscan_t;

F_NONNULL
static void scanner(zscan_t* z, char* buf, const unsigned bufsize, const int fd);

/******** IP Addresses ********/

F_NONNULL
static void set_ipv4(zscan_t* z, const char* end) {
    dmn_assert(z); dmn_assert(end);
    char txt[16];
    unsigned len = end - z->tstart;
    memcpy(txt, z->tstart, len);
    txt[len] = 0;
    z->tstart = NULL;
    struct in_addr addr;
    int status = inet_pton(AF_INET, txt, &addr);
    if(status > 0)
        z->ipv4 = addr.s_addr;
    else
        parse_error("IPv4 address '%s' invalid", txt);
}

F_NONNULL
static void set_ipv6(zscan_t* z, const char* end) {
    dmn_assert(z);
    dmn_assert(end);
    char txt[INET6_ADDRSTRLEN + 1];
    unsigned len = end - z->tstart;
    memcpy(txt, z->tstart, len);
    txt[len] = 0;
    z->tstart = NULL;
    struct in6_addr v6a;
    int status = inet_pton(AF_INET6, txt, &v6a);
    if(status > 0)
        memcpy(z->ipv6, v6a.s6_addr, 16);
    else
        parse_error("IPv6 address '%s' invalid", txt);
}

F_NONNULL
static void set_uval(zscan_t* z) {
    errno = 0;
    z->uval = strtoul(z->tstart, NULL, 10);
    z->tstart = NULL;
    if(errno)
        parse_error("Integer conversion error: %s", dmn_logf_errno());
}

F_NONNULL
static void validate_origin_in_zone(zscan_t* z, const uint8_t* origin) {
    dmn_assert(z); dmn_assert(z->zone->dname); dmn_assert(origin);
    if(!dname_isinzone(z->zone->dname, origin))
        parse_error("Origin '%s' is not within this zonefile's zone (%s)", logf_dname(origin), logf_dname(z->zone->dname));
}

F_NONNULL
static void validate_lhs_not_ooz(zscan_t* z) {
    dmn_assert(z);
    if(z->lhs_is_ooz)
        parse_error("Domainname '%s' is not within this zonefile's zone (%s)", logf_dname(z->lhs_dname), logf_dname(z->zone->dname));
}

F_NONNULL
static void dname_set(zscan_t* z, uint8_t* dname, unsigned len, bool lhs) {
    dmn_assert(z); dmn_assert(dname); dmn_assert(z->zone->dname);
    dname_status_t catstat;
    dname_status_t status;

    if(len) {
        status = dname_from_string(dname, z->tstart, len);
    }
    else {
        dmn_assert(lhs);
        dname_copy(dname, z->origin);
        status = DNAME_VALID;
    }

    switch(status) {
        case DNAME_INVALID:
            parse_error_noargs("unparseable domainname");
            break;
        case DNAME_VALID:
            if(lhs) {
                const bool inzone = dname_isinzone(z->zone->dname, dname);
                z->lhs_is_ooz = !inzone;
                // in-zone LHS dnames are made relative to zroot
                if(inzone)
                    gdnsd_dname_drop_zone(dname, z->zone->dname);
            }
            break;
        case DNAME_PARTIAL:
            // even though in the lhs case we commonly trim
            //   back most or all of z->origin from dname, we
            //   still have to construct it just for validity checks
            catstat = dname_cat(dname, z->origin);
            if(catstat == DNAME_INVALID)
                parse_error_noargs("illegal domainname");
            dmn_assert(catstat == DNAME_VALID);
            if(lhs) {
                z->lhs_is_ooz = false;
                gdnsd_dname_drop_zone(dname, z->zone->dname);
            }
            break;
    }
}

/********** TXT ******************/

F_NONNULL
static void text_start(zscan_t* z) {
    dmn_assert(z);
    z->num_texts = 0;
    z->texts = NULL;
}

F_NONNULL
static void text_add_tok(zscan_t* z, const unsigned len, const bool big_ok) {
    dmn_assert(z);

    char text_temp[len + 1];
    text_temp[0] = 0;
    unsigned newlen = len;
    if(len)
        newlen = dns_unescape(text_temp, z->tstart, len);

    dmn_assert(newlen <= len);

    if(newlen > 255) {
        if(!big_ok || gconfig.disable_text_autosplit)
            parse_error_noargs("Text chunk too long (>255 unescaped)");
        if(newlen > 65500) parse_error_noargs("Text chunk too long (>65500 unescaped)");
        unsigned remainder = newlen % 255;
        unsigned num_whole_chunks = (newlen - remainder) / 255;
        const char* zptr = text_temp;
        const unsigned new_alloc = 1 + z->num_texts + num_whole_chunks + (remainder ? 1 : 0);
        z->texts = xrealloc(z->texts, new_alloc * sizeof(uint8_t*));
        for(unsigned i = 0; i < num_whole_chunks; i++) {
            uint8_t* chunk = z->texts[z->num_texts++] = xmalloc(256);
            *chunk++ = 255;
            memcpy(chunk, zptr, 255);
            zptr += 255;
        }
        if(remainder) {
            uint8_t* chunk = z->texts[z->num_texts++] = xmalloc(remainder + 1);
            *chunk++ = remainder;
            memcpy(chunk, zptr, remainder);
        }
        z->texts[z->num_texts] = NULL;
    }
    else {
        z->texts = xrealloc(z->texts, (z->num_texts + 2) * sizeof(uint8_t*));
        uint8_t* chunk = z->texts[z->num_texts++] = xmalloc(newlen + 1);
        *chunk++ = newlen;
        memcpy(chunk, text_temp, newlen);
        z->texts[z->num_texts] = NULL;
    }

    z->tstart = NULL;
}

// Input must have two bytes of text constrained to [0-9A-Fa-f]
F_NONNULL
static unsigned hexbyte(const char* intxt) {
    dmn_assert(intxt);
    dmn_assert(
        (intxt[0] >= '0' && intxt[0] <= '9')
        || (intxt[0] >= 'A' && intxt[0] <= 'F')
        || (intxt[0] >= 'a' && intxt[0] <= 'f')
    );
    dmn_assert(
        (intxt[1] >= '0' && intxt[1] <= '9')
        || (intxt[1] >= 'A' && intxt[1] <= 'F')
        || (intxt[1] >= 'a' && intxt[1] <= 'f')
    );

    unsigned out;

    if(intxt[0] <= '9')
        out = (intxt[0] - '0') << 4;
    else
        out = ((intxt[0] | 0x20) - ('a' - 10)) << 4;

    if(intxt[1] <= '9')
        out |= (intxt[1] - '0');
    else
        out |= ((intxt[1] | 0x20) - ('a' - 10));

    return out;
}

F_NONNULL
static void mult_uval(zscan_t* z, int fc) {
    dmn_assert(z);
    fc |= 0x20;
    switch(fc) {
        case 'm': z->uval *= 60; break;
        case 'h': z->uval *= 3600; break;
        case 'd': z->uval *= 86400; break;
        case 'w': z->uval *= 604800; break;
    }
}

F_NONNULL
static void set_dyna(zscan_t* z, const char* fpc) {
    dmn_assert(z);
    unsigned dlen = fpc - z->tstart;
    if(dlen > 255)
        parse_error_noargs("DYNA/DYNC plugin!resource string cannot exceed 255 chars");
    memcpy(z->rhs_dyn, z->tstart, dlen);
    z->rhs_dyn[dlen] = 0;
    z->tstart = NULL;
}

F_NONNULL
static void rec_soa(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(z->lhs_dname[0] != 1)
        parse_error_noargs("SOA record can only be defined for the root of the zone");
    if(ltree_add_rec_soa(z->zone, z->lhs_dname, z->rhs_dname, z->eml_dname, z->ttl, z->uv_1, z->uv_2, z->uv_3, z->uv_4, z->uv_5))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_a(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_a(z->zone, z->lhs_dname, z->ipv4, z->ttl, z->limit_v4, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_aaaa(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_aaaa(z->zone, z->lhs_dname, z->ipv6, z->ttl, z->limit_v6, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_ns(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_ns(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_cname(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_cname(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_ptr(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_ptr(z->zone, z->lhs_dname, z->rhs_dname, z->ttl))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_mx(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_mx(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_srv(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_srv(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1, z->uv_2, z->uv_3))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void texts_cleanup(zscan_t* z) {
    dmn_assert(z);
    free(z->texts);
    z->texts = NULL;
    z->num_texts = 0;
}

F_NONNULL
static void rec_naptr(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_naptr(z->zone, z->lhs_dname, z->rhs_dname, z->ttl, z->uv_1, z->uv_2, z->num_texts, z->texts))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_txt(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_txt(z->zone, z->lhs_dname, z->num_texts, z->texts, z->ttl))
        siglongjmp(z->jbuf, 1);
    texts_cleanup(z);
}

F_NONNULL
static void rec_dyna(zscan_t* z) {
    dmn_assert(z);
    if(ltree_add_rec_dynaddr(z->zone, z->lhs_dname, z->rhs_dyn, z->ttl, z->ttl_min, z->limit_v4, z->limit_v6, z->lhs_is_ooz))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_dync(zscan_t* z) {
    dmn_assert(z);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_dync(z->zone, z->lhs_dname, z->rhs_dyn, z->origin, z->ttl, z->ttl_min, z->limit_v4, z->limit_v6))
        siglongjmp(z->jbuf, 1);
}

F_NONNULL
static void rec_rfc3597(zscan_t* z) {
    dmn_assert(z);
    if(z->rfc3597_data_written < z->rfc3597_data_len)
        parse_error("RFC3597 generic RR claimed rdata length of %u, but only %u bytes of data present", z->rfc3597_data_len, z->rfc3597_data_written);
    validate_lhs_not_ooz(z);
    if(ltree_add_rec_rfc3597(z->zone, z->lhs_dname, z->uv_1, z->ttl, z->rfc3597_data_len, z->rfc3597_data))
        siglongjmp(z->jbuf, 1);
    z->rfc3597_data = NULL;
}

F_NONNULL
static void rfc3597_data_setup(zscan_t* z) {
    dmn_assert(z);
    z->rfc3597_data_len = z->uval;
    z->rfc3597_data_written = 0;
    z->rfc3597_data = xmalloc(z->uval);
}

F_NONNULL
static void rfc3597_octet(zscan_t* z) {
    dmn_assert(z);
    if(z->rfc3597_data_written == z->rfc3597_data_len)
       parse_error_noargs("RFC3597 generic RR: more rdata is present than the indicated length");
    z->rfc3597_data[z->rfc3597_data_written++] = hexbyte(z->tstart);
}

F_NONNULL
static void set_limit_v4(zscan_t* z) {
    dmn_assert(z);
    if(z->uval > 65535)
        parse_error("$ADDR_LIMIT_V4 value %u out of range (0-65535)", z->uval);
    z->limit_v4 = z->uval;
}

F_NONNULL
static void set_limit_v6(zscan_t* z) {
    dmn_assert(z);
    if(z->uval > 65535)
        parse_error("$ADDR_LIMIT_V6 value %u out of range (0-65535)", z->uval);
    z->limit_v6 = z->uval;
}

F_NONNULL
static void open_paren(zscan_t* z) {
    dmn_assert(z);
    if(z->in_paren)
        parse_error_noargs("Parenthetical error: double-open");
    z->in_paren = true;
}

F_NONNULL
static void close_paren(zscan_t* z) {
    dmn_assert(z);
    if(!z->in_paren)
        parse_error_noargs("Parenthetical error: unnecessary close");
    z->in_paren = false;
}


#line 484 "./zscan_rfc1035.c"
static const int zone_start = 530;
static const int zone_first_final = 530;
static const int zone_error = 0;

static const int zone_en_main = 530;


#line 695 "./zscan_rfc1035.rl"


F_NONNULL
static void scanner(zscan_t* z, char* buf, const unsigned bufsize, const int fd) {
    dmn_assert(z);

    (void)zone_en_main; // silence unused var warning from generated code

    char* read_at;

    const char* pe = NULL;
    const char* eof = NULL;
    int cs = zone_start;

    while(!eof) {
        unsigned have = 0;
        if(z->tstart != NULL) {
            dmn_assert(pe);
            dmn_assert(z->tstart < pe);
            dmn_assert(z->tstart != buf);
            have = pe - z->tstart;
            memmove(buf, z->tstart, have);
            z->tstart = buf;
        }

        const int space = bufsize - have;
        const char* p = read_at = buf + have;

        const int len = read(fd, read_at, space);
        if(len < 0)
            parse_error("read() failed: %s", dmn_logf_errno());

        pe = p + len;

        if(len < space)
            eof = pe;

#ifndef __clang_analyzer__
        // ^ ... because the ragel-generated code for the zonefile parser is
        //   so huge that it makes analyzer runs take forever.
        
#line 534 "./zscan_rfc1035.c"
	{
	short _widec;
	if ( p == pe )
		goto _test_eof;
	switch ( cs )
	{
case 530:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st15;
		case 32: goto st15;
		case 34: goto tr1836;
		case 36: goto st416;
		case 40: goto tr86;
		case 41: goto tr87;
		case 64: goto tr1838;
		case 92: goto tr1839;
		case 266: goto st531;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st403;
		case 525: goto st404;
		case 571: goto st405;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1835;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1835;
		} else if ( _widec >= 14 )
			goto tr1835;
	} else
		goto tr1835;
	goto st0;
tr1835:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st1;
tr1728:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st1;
tr1840:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st1;
st1:
	if ( ++p == pe )
		goto _test_eof1;
case 1:
#line 609 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1;
		case 32: goto tr1;
		case 40: goto tr3;
		case 41: goto tr4;
		case 92: goto st475;
		case 522: goto tr6;
		case 525: goto tr7;
		case 571: goto tr8;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st1;
		} else
			goto st1;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st1;
		} else if ( _widec >= 35 )
			goto st1;
	} else
		goto st1;
	goto st0;
tr1:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st2;
tr3:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr4:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr10:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr11:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1530:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st2;
tr1531:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1532:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1524:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st2;
tr1525:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1526:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1719:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st2;
tr1720:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1721:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
tr1729:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st2;
tr1730:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st2;
tr1731:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st2;
st2:
	if ( ++p == pe )
		goto _test_eof2;
case 2:
#line 754 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st2;
		case 32: goto st2;
		case 40: goto tr10;
		case 41: goto tr11;
		case 65: goto tr13;
		case 67: goto tr14;
		case 68: goto tr15;
		case 73: goto tr16;
		case 77: goto tr17;
		case 78: goto tr18;
		case 80: goto tr19;
		case 83: goto tr20;
		case 84: goto tr21;
		case 97: goto tr13;
		case 99: goto tr14;
		case 100: goto tr15;
		case 105: goto tr16;
		case 109: goto tr17;
		case 110: goto tr18;
		case 112: goto tr19;
		case 115: goto tr20;
		case 116: goto tr21;
		case 522: goto st410;
		case 525: goto st411;
		case 571: goto st412;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr12;
	goto st0;
st0:
cs = 0;
	goto _out;
tr12:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st3;
tr1501:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st3;
st3:
	if ( ++p == pe )
		goto _test_eof3;
case 3:
#line 831 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr25;
		case 32: goto tr25;
		case 40: goto tr26;
		case 41: goto tr27;
		case 47: goto tr28;
		case 68: goto tr30;
		case 72: goto tr30;
		case 77: goto tr30;
		case 87: goto tr30;
		case 100: goto tr30;
		case 104: goto tr30;
		case 109: goto tr30;
		case 119: goto tr30;
		case 522: goto tr31;
		case 525: goto tr32;
		case 571: goto tr33;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st3;
	goto st0;
tr35:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr36:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
tr25:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
tr26:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
tr27:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
tr1791:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st4;
tr1792:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st4;
tr1793:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st4;
tr1828:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
tr1829:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
tr1830:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st4;
st4:
	if ( ++p == pe )
		goto _test_eof4;
case 4:
#line 964 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st4;
		case 32: goto st4;
		case 40: goto tr35;
		case 41: goto tr36;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 73: goto st515;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st340;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 105: goto st515;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st340;
		case 522: goto st517;
		case 525: goto st518;
		case 571: goto st519;
	}
	goto st0;
tr13:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st5;
tr1446:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st5;
tr1502:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st5;
st5:
	if ( ++p == pe )
		goto _test_eof5;
case 5:
#line 1032 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st6;
		case 32: goto st6;
		case 40: goto tr50;
		case 41: goto tr51;
		case 65: goto st491;
		case 97: goto st491;
		case 522: goto st488;
		case 525: goto st489;
		case 571: goto st490;
	}
	goto st0;
tr50:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st6;
tr51:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st6;
tr1741:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st6;
tr1742:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st6;
tr1743:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st6;
st6:
	if ( ++p == pe )
		goto _test_eof6;
case 6:
#line 1094 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st6;
		case 32: goto st6;
		case 40: goto tr50;
		case 41: goto tr51;
		case 522: goto st488;
		case 525: goto st489;
		case 571: goto st490;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr56;
	goto st0;
tr56:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st7;
tr1744:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st7;
st7:
	if ( ++p == pe )
		goto _test_eof7;
case 7:
#line 1142 "./zscan_rfc1035.c"
	if ( (*p) == 46 )
		goto st8;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st486;
	goto st0;
st8:
	if ( ++p == pe )
		goto _test_eof8;
case 8:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st9;
	goto st0;
st9:
	if ( ++p == pe )
		goto _test_eof9;
case 9:
	if ( (*p) == 46 )
		goto st10;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st484;
	goto st0;
st10:
	if ( ++p == pe )
		goto _test_eof10;
case 10:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st11;
	goto st0;
st11:
	if ( ++p == pe )
		goto _test_eof11;
case 11:
	if ( (*p) == 46 )
		goto st12;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st482;
	goto st0;
st12:
	if ( ++p == pe )
		goto _test_eof12;
case 12:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st13;
	goto st0;
st13:
	if ( ++p == pe )
		goto _test_eof13;
case 13:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr66;
		case 32: goto tr66;
		case 40: goto tr67;
		case 41: goto tr68;
		case 266: goto tr70;
		case 269: goto tr71;
		case 315: goto tr72;
		case 522: goto tr73;
		case 525: goto tr74;
		case 571: goto tr75;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st480;
	goto st0;
tr77:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr78:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr66:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st14;
tr67:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr68:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr106:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr107:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr108:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr130:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st14;
tr120:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr121:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr122:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr131:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr132:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr150:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr151:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr152:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr161:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st14;
tr162:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr163:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr193:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st14;
tr195:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr196:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr210:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st14;
tr211:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr212:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr245:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st14;
tr247:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr248:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr262:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st14;
tr263:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr264:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr337:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr338:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr339:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr351:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr352:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr353:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr372:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr373:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr374:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr383:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st14;
tr384:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr385:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr504:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr505:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr506:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr518:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr519:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr520:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr539:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr540:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr541:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr550:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st14;
tr551:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr552:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr711:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr712:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr713:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr725:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr726:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr727:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr746:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr747:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr748:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr757:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st14;
tr758:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr759:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr789:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr790:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr791:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr803:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr804:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr805:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr824:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr825:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr826:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr835:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st14;
tr836:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr837:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr959:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st14;
tr960:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr961:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr970:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st14;
tr971:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr972:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1177:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1178:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1179:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1191:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1192:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1193:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1212:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1213:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1214:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1223:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st14;
tr1224:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1225:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1561:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st14;
tr1562:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1563:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1585:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st14;
tr1586:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1587:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1618:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1619:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1620:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1632:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1633:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1634:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1653:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1654:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1655:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1664:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st14;
tr1665:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1666:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1692:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st14;
tr1693:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1694:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1703:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st14;
tr1704:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1705:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
tr1757:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st14;
tr1758:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st14;
tr1759:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st14;
st14:
	if ( ++p == pe )
		goto _test_eof14;
case 14:
#line 2225 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st14;
		case 32: goto st14;
		case 40: goto tr77;
		case 41: goto tr78;
		case 266: goto st531;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st27;
		case 525: goto st28;
		case 571: goto st29;
	}
	goto st0;
tr70:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st531;
tr110:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st531;
tr133:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st531;
tr124:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st531;
tr153:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st531;
tr164:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st531;
tr198:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st531;
tr214:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st531;
tr250:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st531;
tr266:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st531;
tr341:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st531;
tr355:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st531;
tr375:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st531;
tr386:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st531;
tr508:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st531;
tr522:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st531;
tr542:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st531;
tr553:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st531;
tr715:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st531;
tr729:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st531;
tr749:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st531;
tr760:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st531;
tr793:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st531;
tr807:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st531;
tr827:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st531;
tr838:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st531;
tr964:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st531;
tr973:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st531;
tr1181:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st531;
tr1195:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st531;
tr1215:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st531;
tr1226:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st531;
tr1287:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st531;
tr1299:
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st531;
tr1321:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st531;
tr1336:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st531;
tr1348:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st531;
tr1398:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st531;
tr1416:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st531;
tr1565:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st531;
tr1589:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st531;
tr1622:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st531;
tr1636:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st531;
tr1656:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st531;
tr1667:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st531;
tr1697:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st531;
tr1706:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st531;
tr1762:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st531;
st531:
	if ( ++p == pe )
		goto _test_eof531;
case 531:
#line 2582 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1498;
		case 32: goto tr1498;
		case 34: goto tr1841;
		case 36: goto tr1842;
		case 40: goto tr1499;
		case 41: goto tr1500;
		case 64: goto tr1843;
		case 92: goto tr1844;
		case 266: goto tr133;
		case 269: goto tr134;
		case 315: goto tr135;
		case 522: goto tr1511;
		case 525: goto tr1512;
		case 571: goto tr1513;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1840;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1840;
		} else if ( _widec >= 14 )
			goto tr1840;
	} else
		goto tr1840;
	goto st0;
tr86:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st15;
tr87:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st15;
tr1498:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st15;
tr1499:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st15;
tr1500:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st15;
st15:
	if ( ++p == pe )
		goto _test_eof15;
case 15:
#line 2660 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st15;
		case 32: goto st15;
		case 40: goto tr86;
		case 41: goto tr87;
		case 65: goto tr13;
		case 67: goto tr14;
		case 68: goto tr15;
		case 73: goto tr16;
		case 77: goto tr17;
		case 78: goto tr18;
		case 80: goto tr19;
		case 83: goto tr20;
		case 84: goto tr21;
		case 97: goto tr13;
		case 99: goto tr14;
		case 100: goto tr15;
		case 105: goto tr16;
		case 109: goto tr17;
		case 110: goto tr18;
		case 112: goto tr19;
		case 115: goto tr20;
		case 116: goto tr21;
		case 266: goto st531;
		case 269: goto st25;
		case 315: goto st26;
		case 522: goto st403;
		case 525: goto st404;
		case 571: goto st405;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr12;
	goto st0;
tr14:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st16;
tr1447:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st16;
tr1503:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st16;
st16:
	if ( ++p == pe )
		goto _test_eof16;
case 16:
#line 2733 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st17;
		case 110: goto st17;
	}
	goto st0;
st17:
	if ( ++p == pe )
		goto _test_eof17;
case 17:
	switch( (*p) ) {
		case 65: goto st18;
		case 97: goto st18;
	}
	goto st0;
st18:
	if ( ++p == pe )
		goto _test_eof18;
case 18:
	switch( (*p) ) {
		case 77: goto st19;
		case 109: goto st19;
	}
	goto st0;
st19:
	if ( ++p == pe )
		goto _test_eof19;
case 19:
	switch( (*p) ) {
		case 69: goto st20;
		case 101: goto st20;
	}
	goto st0;
st20:
	if ( ++p == pe )
		goto _test_eof20;
case 20:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st21;
		case 32: goto st21;
		case 40: goto tr96;
		case 41: goto tr97;
		case 522: goto st41;
		case 525: goto st42;
		case 571: goto st43;
	}
	goto st0;
tr96:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st21;
tr97:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st21;
tr171:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st21;
tr173:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st21;
tr174:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st21;
st21:
	if ( ++p == pe )
		goto _test_eof21;
case 21:
#line 2829 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st21;
		case 32: goto st21;
		case 34: goto tr102;
		case 40: goto tr96;
		case 41: goto tr97;
		case 64: goto tr103;
		case 92: goto tr104;
		case 522: goto st41;
		case 525: goto st42;
		case 571: goto st43;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr101;
		} else
			goto tr101;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr101;
		} else if ( _widec >= 37 )
			goto tr101;
	} else
		goto tr101;
	goto st0;
tr101:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st22;
tr119:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st22;
tr170:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st22;
st22:
	if ( ++p == pe )
		goto _test_eof22;
case 22:
#line 2896 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr106;
		case 32: goto tr106;
		case 40: goto tr107;
		case 41: goto tr108;
		case 92: goto st23;
		case 266: goto tr110;
		case 269: goto tr111;
		case 315: goto tr112;
		case 522: goto tr113;
		case 525: goto tr114;
		case 571: goto tr115;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st22;
		} else
			goto st22;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st22;
		} else if ( _widec >= 35 )
			goto st22;
	} else
		goto st22;
	goto st0;
tr104:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st23;
tr123:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st23;
tr176:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st23;
st23:
	if ( ++p == pe )
		goto _test_eof23;
case 23:
#line 2964 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st24;
		case 13: goto st30;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st31;
	goto st22;
st24:
	if ( ++p == pe )
		goto _test_eof24;
case 24:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr120;
		case 32: goto tr120;
		case 40: goto tr121;
		case 41: goto tr122;
		case 92: goto tr123;
		case 266: goto tr124;
		case 269: goto tr125;
		case 315: goto tr126;
		case 522: goto tr127;
		case 525: goto tr128;
		case 571: goto tr129;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr119;
		} else
			goto tr119;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr119;
		} else if ( _widec >= 35 )
			goto tr119;
	} else
		goto tr119;
	goto st0;
tr71:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st25;
tr111:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr134:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st25;
tr125:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr154:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr165:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st25;
tr199:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st25;
tr215:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st25;
tr251:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st25;
tr267:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st25;
tr342:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr356:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr376:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr387:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st25;
tr509:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr523:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr543:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr554:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st25;
tr716:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr730:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr750:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr761:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st25;
tr794:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr808:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr828:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr839:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st25;
tr965:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st25;
tr974:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st25;
tr1182:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1196:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1216:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1227:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st25;
tr1288:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1300:
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1322:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1337:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1349:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st25;
tr1399:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st25;
tr1417:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st25;
tr1566:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st25;
tr1590:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st25;
tr1623:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1637:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1657:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1668:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st25;
tr1698:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st25;
tr1707:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st25;
tr1763:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st25;
st25:
	if ( ++p == pe )
		goto _test_eof25;
case 25:
#line 3350 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 266 )
		goto st531;
	goto st0;
tr72:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st26;
tr112:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr135:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st26;
tr126:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr155:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr166:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st26;
tr200:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st26;
tr216:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st26;
tr252:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st26;
tr268:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st26;
tr343:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr357:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr377:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr388:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st26;
tr510:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr524:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr544:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr555:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st26;
tr717:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr731:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr751:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr762:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st26;
tr795:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr809:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr829:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr840:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st26;
tr966:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st26;
tr975:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st26;
tr1183:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1197:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1217:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1228:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st26;
tr1289:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1301:
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1323:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1338:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1350:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 536 "./zscan_rfc1035.rl"
	{ rec_txt(z); }
	goto st26;
tr1400:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st26;
tr1418:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 539 "./zscan_rfc1035.rl"
	{ rec_rfc3597(z); }
	goto st26;
tr1567:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st26;
tr1591:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st26;
tr1624:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1638:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1658:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1669:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st26;
tr1699:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st26;
tr1708:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st26;
tr1764:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st26;
st26:
	if ( ++p == pe )
		goto _test_eof26;
case 26:
#line 3683 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 266: goto st531;
		case 269: goto st25;
	}
	if ( 128 <= _widec && _widec <= 383 )
		goto st26;
	goto st0;
tr73:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st27;
tr113:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr136:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st27;
tr127:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr156:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr167:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st27;
tr201:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st27;
tr217:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st27;
tr253:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st27;
tr269:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st27;
tr344:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr358:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr378:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr389:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st27;
tr511:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr525:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr545:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr556:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st27;
tr718:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr732:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr752:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr763:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st27;
tr796:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr810:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr830:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr841:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st27;
tr967:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st27;
tr976:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st27;
tr1184:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1198:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1218:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1229:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st27;
tr1568:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st27;
tr1592:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st27;
tr1625:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1639:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1659:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1670:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st27;
tr1700:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st27;
tr1709:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st27;
tr1765:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st27;
st27:
	if ( ++p == pe )
		goto _test_eof27;
case 27:
#line 4004 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr130;
		case 32: goto tr130;
		case 40: goto tr131;
		case 41: goto tr132;
		case 266: goto tr133;
		case 269: goto tr134;
		case 315: goto tr135;
		case 522: goto tr136;
		case 525: goto tr137;
		case 571: goto tr138;
	}
	goto st0;
tr74:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st28;
tr114:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr137:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st28;
tr128:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr157:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr168:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st28;
tr202:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st28;
tr218:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st28;
tr254:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st28;
tr270:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st28;
tr345:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr359:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr379:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr390:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st28;
tr512:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr526:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr546:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr557:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st28;
tr719:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr733:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr753:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr764:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st28;
tr797:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr811:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr831:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr842:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st28;
tr968:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st28;
tr977:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st28;
tr1185:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1199:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1219:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1230:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st28;
tr1569:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st28;
tr1593:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st28;
tr1626:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1640:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1660:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1671:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st28;
tr1701:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st28;
tr1710:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st28;
tr1766:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st28;
st28:
	if ( ++p == pe )
		goto _test_eof28;
case 28:
#line 4317 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st27;
	goto st0;
tr75:
#line 506 "./zscan_rfc1035.rl"
	{ set_ipv4(z, p); }
#line 528 "./zscan_rfc1035.rl"
	{ rec_a(z); }
	goto st29;
tr115:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr138:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st29;
tr129:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr158:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr169:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 531 "./zscan_rfc1035.rl"
	{ rec_cname(z); }
	goto st29;
tr203:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st29;
tr219:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 537 "./zscan_rfc1035.rl"
	{ rec_dyna(z); }
	goto st29;
tr255:
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st29;
tr271:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 525 "./zscan_rfc1035.rl"
	{ set_dyna(z, p); }
#line 538 "./zscan_rfc1035.rl"
	{ rec_dync(z); }
	goto st29;
tr346:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr360:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr380:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr391:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 533 "./zscan_rfc1035.rl"
	{ rec_mx(z); }
	goto st29;
tr513:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr527:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr547:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr558:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 535 "./zscan_rfc1035.rl"
	{ rec_naptr(z); }
	goto st29;
tr720:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr734:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr754:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr765:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 530 "./zscan_rfc1035.rl"
	{ rec_ns(z); }
	goto st29;
tr798:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr812:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr832:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr843:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 532 "./zscan_rfc1035.rl"
	{ rec_ptr(z); }
	goto st29;
tr969:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st29;
tr978:
#line 520 "./zscan_rfc1035.rl"
	{ z->uv_5 = z->uval; }
#line 527 "./zscan_rfc1035.rl"
	{ rec_soa(z); }
	goto st29;
tr1186:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1200:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1220:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1231:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 534 "./zscan_rfc1035.rl"
	{ rec_srv(z); }
	goto st29;
tr1570:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 522 "./zscan_rfc1035.rl"
	{ set_limit_v4(z); }
	goto st29;
tr1594:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 523 "./zscan_rfc1035.rl"
	{ set_limit_v6(z); }
	goto st29;
tr1627:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1641:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1661:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1672:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 495 "./zscan_rfc1035.rl"
	{
        validate_origin_in_zone(z, z->rhs_dname);
        dname_copy(z->origin, z->rhs_dname);
    }
	goto st29;
tr1702:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st29;
tr1711:
#line 513 "./zscan_rfc1035.rl"
	{ z->def_ttl = z->uval; }
	goto st29;
tr1767:
#line 507 "./zscan_rfc1035.rl"
	{ set_ipv6(z, p); }
#line 529 "./zscan_rfc1035.rl"
	{ rec_aaaa(z); }
	goto st29;
st29:
	if ( ++p == pe )
		goto _test_eof29;
case 29:
#line 4606 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st27;
		case 525: goto st28;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st29;
	goto st0;
st30:
	if ( ++p == pe )
		goto _test_eof30;
case 30:
	if ( (*p) == 10 )
		goto st24;
	goto st0;
st31:
	if ( ++p == pe )
		goto _test_eof31;
case 31:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st32;
	goto st0;
st32:
	if ( ++p == pe )
		goto _test_eof32;
case 32:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st22;
	goto st0;
tr102:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st33;
tr145:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st33;
tr172:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st33;
st33:
	if ( ++p == pe )
		goto _test_eof33;
case 33:
#line 4688 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st34;
		case 13: goto st35;
		case 34: goto st36;
		case 92: goto st37;
	}
	goto st33;
tr146:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st34;
st34:
	if ( ++p == pe )
		goto _test_eof34;
case 34:
#line 4704 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr146;
		case 13: goto tr147;
		case 34: goto tr148;
		case 92: goto tr149;
	}
	goto tr145;
tr147:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st35;
st35:
	if ( ++p == pe )
		goto _test_eof35;
case 35:
#line 4720 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st34;
	goto st0;
tr148:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st36;
st36:
	if ( ++p == pe )
		goto _test_eof36;
case 36:
#line 4732 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr150;
		case 32: goto tr150;
		case 40: goto tr151;
		case 41: goto tr152;
		case 266: goto tr153;
		case 269: goto tr154;
		case 315: goto tr155;
		case 522: goto tr156;
		case 525: goto tr157;
		case 571: goto tr158;
	}
	goto st0;
tr149:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st37;
st37:
	if ( ++p == pe )
		goto _test_eof37;
case 37:
#line 4775 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st34;
		case 13: goto st35;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st38;
	goto st33;
st38:
	if ( ++p == pe )
		goto _test_eof38;
case 38:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st39;
	goto st0;
st39:
	if ( ++p == pe )
		goto _test_eof39;
case 39:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st33;
	goto st0;
tr103:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st40;
tr175:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st40;
st40:
	if ( ++p == pe )
		goto _test_eof40;
case 40:
#line 4814 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr161;
		case 32: goto tr161;
		case 40: goto tr162;
		case 41: goto tr163;
		case 266: goto tr164;
		case 269: goto tr165;
		case 315: goto tr166;
		case 522: goto tr167;
		case 525: goto tr168;
		case 571: goto tr169;
	}
	goto st0;
tr177:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st41;
st41:
	if ( ++p == pe )
		goto _test_eof41;
case 41:
#line 4857 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr171;
		case 32: goto tr171;
		case 34: goto tr172;
		case 40: goto tr173;
		case 41: goto tr174;
		case 64: goto tr175;
		case 92: goto tr176;
		case 522: goto tr177;
		case 525: goto tr178;
		case 571: goto tr179;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr170;
		} else
			goto tr170;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr170;
		} else if ( _widec >= 37 )
			goto tr170;
	} else
		goto tr170;
	goto st0;
tr178:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st42;
st42:
	if ( ++p == pe )
		goto _test_eof42;
case 42:
#line 4914 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st41;
	goto st0;
tr179:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st43;
st43:
	if ( ++p == pe )
		goto _test_eof43;
case 43:
#line 4933 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st41;
		case 525: goto st42;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st43;
	goto st0;
tr15:
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st44;
tr1448:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st44;
tr1504:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st44;
st44:
	if ( ++p == pe )
		goto _test_eof44;
case 44:
#line 4994 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 89: goto st45;
		case 121: goto st45;
	}
	goto st0;
st45:
	if ( ++p == pe )
		goto _test_eof45;
case 45:
	switch( (*p) ) {
		case 78: goto st46;
		case 110: goto st46;
	}
	goto st0;
st46:
	if ( ++p == pe )
		goto _test_eof46;
case 46:
	switch( (*p) ) {
		case 65: goto st47;
		case 67: goto st65;
		case 97: goto st47;
		case 99: goto st65;
	}
	goto st0;
st47:
	if ( ++p == pe )
		goto _test_eof47;
case 47:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st48;
		case 32: goto st48;
		case 40: goto tr185;
		case 41: goto tr186;
		case 522: goto st62;
		case 525: goto st63;
		case 571: goto st64;
	}
	goto st0;
tr185:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st48;
tr186:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st48;
tr229:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st48;
tr230:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st48;
tr231:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st48;
st48:
	if ( ++p == pe )
		goto _test_eof48;
case 48:
#line 5083 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st48;
		case 32: goto st48;
		case 40: goto tr185;
		case 41: goto tr186;
		case 92: goto tr191;
		case 522: goto st62;
		case 525: goto st63;
		case 571: goto st64;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr190;
		} else
			goto tr190;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr190;
		} else if ( _widec >= 35 )
			goto tr190;
	} else
		goto tr190;
	goto st0;
tr190:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st49;
tr224:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st49;
tr228:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st49;
st49:
	if ( ++p == pe )
		goto _test_eof49;
case 49:
#line 5148 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr193;
		case 32: goto tr193;
		case 33: goto st50;
		case 40: goto tr195;
		case 41: goto tr196;
		case 92: goto st57;
		case 266: goto tr198;
		case 269: goto tr199;
		case 315: goto tr200;
		case 522: goto tr201;
		case 525: goto tr202;
		case 571: goto tr203;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st49;
		} else
			goto st49;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st49;
		} else if ( _widec >= 35 )
			goto st49;
	} else
		goto st49;
	goto st0;
tr225:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st50;
st50:
	if ( ++p == pe )
		goto _test_eof50;
case 50:
#line 5207 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 13: goto st0;
		case 59: goto st0;
		case 92: goto st52;
	}
	if ( (*p) < 32 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto st0;
	} else if ( (*p) > 34 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto st0;
	} else
		goto st0;
	goto st51;
tr209:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st51;
st51:
	if ( ++p == pe )
		goto _test_eof51;
case 51:
#line 5230 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr193;
		case 32: goto tr193;
		case 40: goto tr195;
		case 41: goto tr196;
		case 92: goto st52;
		case 266: goto tr198;
		case 269: goto tr199;
		case 315: goto tr200;
		case 522: goto tr201;
		case 525: goto tr202;
		case 571: goto tr203;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st51;
		} else
			goto st51;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st51;
		} else if ( _widec >= 35 )
			goto st51;
	} else
		goto st51;
	goto st0;
tr213:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st52;
st52:
	if ( ++p == pe )
		goto _test_eof52;
case 52:
#line 5288 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st53;
		case 13: goto st54;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st55;
	goto st51;
st53:
	if ( ++p == pe )
		goto _test_eof53;
case 53:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr210;
		case 32: goto tr210;
		case 40: goto tr211;
		case 41: goto tr212;
		case 92: goto tr213;
		case 266: goto tr214;
		case 269: goto tr215;
		case 315: goto tr216;
		case 522: goto tr217;
		case 525: goto tr218;
		case 571: goto tr219;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr209;
		} else
			goto tr209;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr209;
		} else if ( _widec >= 35 )
			goto tr209;
	} else
		goto tr209;
	goto st0;
st54:
	if ( ++p == pe )
		goto _test_eof54;
case 54:
	if ( (*p) == 10 )
		goto st53;
	goto st0;
st55:
	if ( ++p == pe )
		goto _test_eof55;
case 55:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st56;
	goto st0;
st56:
	if ( ++p == pe )
		goto _test_eof56;
case 56:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st51;
	goto st0;
tr191:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st57;
tr226:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st57;
tr232:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st57;
st57:
	if ( ++p == pe )
		goto _test_eof57;
case 57:
#line 5391 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st58;
		case 13: goto st59;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st60;
	goto st49;
st58:
	if ( ++p == pe )
		goto _test_eof58;
case 58:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr210;
		case 32: goto tr210;
		case 33: goto tr225;
		case 40: goto tr211;
		case 41: goto tr212;
		case 92: goto tr226;
		case 266: goto tr214;
		case 269: goto tr215;
		case 315: goto tr216;
		case 522: goto tr217;
		case 525: goto tr218;
		case 571: goto tr219;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr224;
		} else
			goto tr224;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr224;
		} else if ( _widec >= 35 )
			goto tr224;
	} else
		goto tr224;
	goto st0;
st59:
	if ( ++p == pe )
		goto _test_eof59;
case 59:
	if ( (*p) == 10 )
		goto st58;
	goto st0;
st60:
	if ( ++p == pe )
		goto _test_eof60;
case 60:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st61;
	goto st0;
st61:
	if ( ++p == pe )
		goto _test_eof61;
case 61:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st49;
	goto st0;
tr233:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st62;
st62:
	if ( ++p == pe )
		goto _test_eof62;
case 62:
#line 5485 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr229;
		case 32: goto tr229;
		case 40: goto tr230;
		case 41: goto tr231;
		case 92: goto tr232;
		case 522: goto tr233;
		case 525: goto tr234;
		case 571: goto tr235;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr228;
		} else
			goto tr228;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr228;
		} else if ( _widec >= 35 )
			goto tr228;
	} else
		goto tr228;
	goto st0;
tr234:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st63;
st63:
	if ( ++p == pe )
		goto _test_eof63;
case 63:
#line 5540 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st62;
	goto st0;
tr235:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st64;
st64:
	if ( ++p == pe )
		goto _test_eof64;
case 64:
#line 5559 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st62;
		case 525: goto st63;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st64;
	goto st0;
st65:
	if ( ++p == pe )
		goto _test_eof65;
case 65:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st66;
		case 32: goto st66;
		case 40: goto tr237;
		case 41: goto tr238;
		case 522: goto st80;
		case 525: goto st81;
		case 571: goto st82;
	}
	goto st0;
tr237:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st66;
tr238:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st66;
tr281:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st66;
tr282:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st66;
tr283:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st66;
st66:
	if ( ++p == pe )
		goto _test_eof66;
case 66:
#line 5665 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st66;
		case 32: goto st66;
		case 40: goto tr237;
		case 41: goto tr238;
		case 92: goto tr243;
		case 522: goto st80;
		case 525: goto st81;
		case 571: goto st82;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr242;
		} else
			goto tr242;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr242;
		} else if ( _widec >= 35 )
			goto tr242;
	} else
		goto tr242;
	goto st0;
tr242:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st67;
tr276:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st67;
tr280:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st67;
st67:
	if ( ++p == pe )
		goto _test_eof67;
case 67:
#line 5730 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr245;
		case 32: goto tr245;
		case 33: goto st68;
		case 40: goto tr247;
		case 41: goto tr248;
		case 92: goto st75;
		case 266: goto tr250;
		case 269: goto tr251;
		case 315: goto tr252;
		case 522: goto tr253;
		case 525: goto tr254;
		case 571: goto tr255;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st67;
		} else
			goto st67;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st67;
		} else if ( _widec >= 35 )
			goto st67;
	} else
		goto st67;
	goto st0;
tr277:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st68;
st68:
	if ( ++p == pe )
		goto _test_eof68;
case 68:
#line 5789 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 13: goto st0;
		case 59: goto st0;
		case 92: goto st70;
	}
	if ( (*p) < 32 ) {
		if ( 9 <= (*p) && (*p) <= 10 )
			goto st0;
	} else if ( (*p) > 34 ) {
		if ( 40 <= (*p) && (*p) <= 41 )
			goto st0;
	} else
		goto st0;
	goto st69;
tr261:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st69;
st69:
	if ( ++p == pe )
		goto _test_eof69;
case 69:
#line 5812 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr245;
		case 32: goto tr245;
		case 40: goto tr247;
		case 41: goto tr248;
		case 92: goto st70;
		case 266: goto tr250;
		case 269: goto tr251;
		case 315: goto tr252;
		case 522: goto tr253;
		case 525: goto tr254;
		case 571: goto tr255;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st69;
		} else
			goto st69;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st69;
		} else if ( _widec >= 35 )
			goto st69;
	} else
		goto st69;
	goto st0;
tr265:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st70;
st70:
	if ( ++p == pe )
		goto _test_eof70;
case 70:
#line 5870 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st71;
		case 13: goto st72;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st73;
	goto st69;
st71:
	if ( ++p == pe )
		goto _test_eof71;
case 71:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr262;
		case 32: goto tr262;
		case 40: goto tr263;
		case 41: goto tr264;
		case 92: goto tr265;
		case 266: goto tr266;
		case 269: goto tr267;
		case 315: goto tr268;
		case 522: goto tr269;
		case 525: goto tr270;
		case 571: goto tr271;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr261;
		} else
			goto tr261;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr261;
		} else if ( _widec >= 35 )
			goto tr261;
	} else
		goto tr261;
	goto st0;
st72:
	if ( ++p == pe )
		goto _test_eof72;
case 72:
	if ( (*p) == 10 )
		goto st71;
	goto st0;
st73:
	if ( ++p == pe )
		goto _test_eof73;
case 73:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st74;
	goto st0;
st74:
	if ( ++p == pe )
		goto _test_eof74;
case 74:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st69;
	goto st0;
tr243:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st75;
tr278:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st75;
tr284:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st75;
st75:
	if ( ++p == pe )
		goto _test_eof75;
case 75:
#line 5973 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st76;
		case 13: goto st77;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st78;
	goto st67;
st76:
	if ( ++p == pe )
		goto _test_eof76;
case 76:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr262;
		case 32: goto tr262;
		case 33: goto tr277;
		case 40: goto tr263;
		case 41: goto tr264;
		case 92: goto tr278;
		case 266: goto tr266;
		case 269: goto tr267;
		case 315: goto tr268;
		case 522: goto tr269;
		case 525: goto tr270;
		case 571: goto tr271;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr276;
		} else
			goto tr276;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr276;
		} else if ( _widec >= 35 )
			goto tr276;
	} else
		goto tr276;
	goto st0;
st77:
	if ( ++p == pe )
		goto _test_eof77;
case 77:
	if ( (*p) == 10 )
		goto st76;
	goto st0;
st78:
	if ( ++p == pe )
		goto _test_eof78;
case 78:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st79;
	goto st0;
st79:
	if ( ++p == pe )
		goto _test_eof79;
case 79:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st67;
	goto st0;
tr285:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st80;
st80:
	if ( ++p == pe )
		goto _test_eof80;
case 80:
#line 6067 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr281;
		case 32: goto tr281;
		case 40: goto tr282;
		case 41: goto tr283;
		case 92: goto tr284;
		case 522: goto tr285;
		case 525: goto tr286;
		case 571: goto tr287;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr280;
		} else
			goto tr280;
	} else if ( _widec > 31 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr280;
		} else if ( _widec >= 35 )
			goto tr280;
	} else
		goto tr280;
	goto st0;
tr286:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st81;
st81:
	if ( ++p == pe )
		goto _test_eof81;
case 81:
#line 6122 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st80;
	goto st0;
tr287:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st82;
st82:
	if ( ++p == pe )
		goto _test_eof82;
case 82:
#line 6141 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st80;
		case 525: goto st81;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st82;
	goto st0;
tr16:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st83;
tr1505:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
#line 515 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; z->ttl_min = z->def_ttl >> 1; z->uv_2 = 0; }
	goto st83;
st83:
	if ( ++p == pe )
		goto _test_eof83;
case 83:
#line 6202 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st84;
		case 110: goto st84;
	}
	goto st0;
st84:
	if ( ++p == pe )
		goto _test_eof84;
case 84:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st85;
		case 32: goto st85;
		case 40: goto tr290;
		case 41: goto tr291;
		case 522: goto st400;
		case 525: goto st401;
		case 571: goto st402;
	}
	goto st0;
tr290:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st85;
tr291:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st85;
tr1491:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st85;
tr1492:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st85;
tr1493:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st85;
st85:
	if ( ++p == pe )
		goto _test_eof85;
case 85:
#line 6271 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st85;
		case 32: goto st85;
		case 40: goto tr290;
		case 41: goto tr291;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st340;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st340;
		case 522: goto st400;
		case 525: goto st401;
		case 571: goto st402;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr295;
	goto st0;
tr295:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st86;
tr1494:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st86;
st86:
	if ( ++p == pe )
		goto _test_eof86;
case 86:
#line 6335 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr296;
		case 32: goto tr296;
		case 40: goto tr297;
		case 41: goto tr298;
		case 47: goto tr299;
		case 68: goto tr301;
		case 72: goto tr301;
		case 77: goto tr301;
		case 87: goto tr301;
		case 100: goto tr301;
		case 104: goto tr301;
		case 109: goto tr301;
		case 119: goto tr301;
		case 522: goto tr302;
		case 525: goto tr303;
		case 571: goto tr304;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st86;
	goto st0;
tr306:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr307:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
tr296:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
tr297:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
tr298:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
tr1443:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st87;
tr1444:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st87;
tr1445:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st87;
tr1484:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
tr1485:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
tr1486:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st87;
st87:
	if ( ++p == pe )
		goto _test_eof87;
case 87:
#line 6468 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st87;
		case 32: goto st87;
		case 40: goto tr306;
		case 41: goto tr307;
		case 65: goto st5;
		case 67: goto st16;
		case 68: goto st44;
		case 77: goto st88;
		case 78: goto st113;
		case 80: goto st216;
		case 83: goto st237;
		case 84: goto st340;
		case 97: goto st5;
		case 99: goto st16;
		case 100: goto st44;
		case 109: goto st88;
		case 110: goto st113;
		case 112: goto st216;
		case 115: goto st237;
		case 116: goto st340;
		case 522: goto st389;
		case 525: goto st390;
		case 571: goto st391;
	}
	goto st0;
tr17:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st88;
tr1449:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st88;
tr1506:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st88;
st88:
	if ( ++p == pe )
		goto _test_eof88;
case 88:
#line 6534 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 88: goto st89;
		case 120: goto st89;
	}
	goto st0;
st89:
	if ( ++p == pe )
		goto _test_eof89;
case 89:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st90;
		case 32: goto st90;
		case 40: goto tr313;
		case 41: goto tr314;
		case 522: goto st110;
		case 525: goto st111;
		case 571: goto st112;
	}
	goto st0;
tr313:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st90;
tr314:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st90;
tr402:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st90;
tr403:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st90;
tr404:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st90;
st90:
	if ( ++p == pe )
		goto _test_eof90;
case 90:
#line 6603 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st90;
		case 32: goto st90;
		case 40: goto tr313;
		case 41: goto tr314;
		case 522: goto st110;
		case 525: goto st111;
		case 571: goto st112;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr318;
	goto st0;
tr318:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st91;
tr405:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st91;
st91:
	if ( ++p == pe )
		goto _test_eof91;
case 91:
#line 6651 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr319;
		case 32: goto tr319;
		case 40: goto tr320;
		case 41: goto tr321;
		case 522: goto tr323;
		case 525: goto tr324;
		case 571: goto tr325;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st91;
	goto st0;
tr329:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr330:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
tr319:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st92;
tr393:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st92;
tr395:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr396:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
tr320:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st92;
tr321:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st92;
st92:
	if ( ++p == pe )
		goto _test_eof92;
case 92:
#line 6735 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st92;
		case 32: goto st92;
		case 34: goto tr328;
		case 40: goto tr329;
		case 41: goto tr330;
		case 64: goto tr331;
		case 92: goto tr332;
		case 522: goto st107;
		case 525: goto st108;
		case 571: goto st109;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr326;
		} else
			goto tr326;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr326;
		} else if ( _widec >= 37 )
			goto tr326;
	} else
		goto tr326;
	goto st0;
tr326:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st93;
tr350:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st93;
tr392:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st93;
st93:
	if ( ++p == pe )
		goto _test_eof93;
case 93:
#line 6802 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr337;
		case 32: goto tr337;
		case 40: goto tr338;
		case 41: goto tr339;
		case 92: goto st94;
		case 266: goto tr341;
		case 269: goto tr342;
		case 315: goto tr343;
		case 522: goto tr344;
		case 525: goto tr345;
		case 571: goto tr346;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st93;
		} else
			goto st93;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st93;
		} else if ( _widec >= 35 )
			goto st93;
	} else
		goto st93;
	goto st0;
tr332:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st94;
tr354:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st94;
tr398:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st94;
st94:
	if ( ++p == pe )
		goto _test_eof94;
case 94:
#line 6870 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st95;
		case 13: goto st96;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st97;
	goto st93;
st95:
	if ( ++p == pe )
		goto _test_eof95;
case 95:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr351;
		case 32: goto tr351;
		case 40: goto tr352;
		case 41: goto tr353;
		case 92: goto tr354;
		case 266: goto tr355;
		case 269: goto tr356;
		case 315: goto tr357;
		case 522: goto tr358;
		case 525: goto tr359;
		case 571: goto tr360;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr350;
		} else
			goto tr350;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr350;
		} else if ( _widec >= 35 )
			goto tr350;
	} else
		goto tr350;
	goto st0;
st96:
	if ( ++p == pe )
		goto _test_eof96;
case 96:
	if ( (*p) == 10 )
		goto st95;
	goto st0;
st97:
	if ( ++p == pe )
		goto _test_eof97;
case 97:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st98;
	goto st0;
st98:
	if ( ++p == pe )
		goto _test_eof98;
case 98:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st93;
	goto st0;
tr328:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st99;
tr367:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st99;
tr394:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st99;
st99:
	if ( ++p == pe )
		goto _test_eof99;
case 99:
#line 6973 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st100;
		case 13: goto st101;
		case 34: goto st102;
		case 92: goto st103;
	}
	goto st99;
tr368:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st100;
st100:
	if ( ++p == pe )
		goto _test_eof100;
case 100:
#line 6989 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr368;
		case 13: goto tr369;
		case 34: goto tr370;
		case 92: goto tr371;
	}
	goto tr367;
tr369:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st101;
st101:
	if ( ++p == pe )
		goto _test_eof101;
case 101:
#line 7005 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st100;
	goto st0;
tr370:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st102;
st102:
	if ( ++p == pe )
		goto _test_eof102;
case 102:
#line 7017 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr372;
		case 32: goto tr372;
		case 40: goto tr373;
		case 41: goto tr374;
		case 266: goto tr375;
		case 269: goto tr376;
		case 315: goto tr377;
		case 522: goto tr378;
		case 525: goto tr379;
		case 571: goto tr380;
	}
	goto st0;
tr371:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st103;
st103:
	if ( ++p == pe )
		goto _test_eof103;
case 103:
#line 7060 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st100;
		case 13: goto st101;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st104;
	goto st99;
st104:
	if ( ++p == pe )
		goto _test_eof104;
case 104:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st105;
	goto st0;
st105:
	if ( ++p == pe )
		goto _test_eof105;
case 105:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st99;
	goto st0;
tr331:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st106;
tr397:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st106;
st106:
	if ( ++p == pe )
		goto _test_eof106;
case 106:
#line 7099 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr383;
		case 32: goto tr383;
		case 40: goto tr384;
		case 41: goto tr385;
		case 266: goto tr386;
		case 269: goto tr387;
		case 315: goto tr388;
		case 522: goto tr389;
		case 525: goto tr390;
		case 571: goto tr391;
	}
	goto st0;
tr323:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st107;
tr399:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st107;
st107:
	if ( ++p == pe )
		goto _test_eof107;
case 107:
#line 7148 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr393;
		case 32: goto tr393;
		case 34: goto tr394;
		case 40: goto tr395;
		case 41: goto tr396;
		case 64: goto tr397;
		case 92: goto tr398;
		case 522: goto tr399;
		case 525: goto tr400;
		case 571: goto tr401;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr392;
		} else
			goto tr392;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr392;
		} else if ( _widec >= 37 )
			goto tr392;
	} else
		goto tr392;
	goto st0;
tr324:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st108;
tr400:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st108;
st108:
	if ( ++p == pe )
		goto _test_eof108;
case 108:
#line 7211 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st107;
	goto st0;
tr325:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st109;
tr401:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st109;
st109:
	if ( ++p == pe )
		goto _test_eof109;
case 109:
#line 7236 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st107;
		case 525: goto st108;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st109;
	goto st0;
tr406:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st110;
st110:
	if ( ++p == pe )
		goto _test_eof110;
case 110:
#line 7287 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr402;
		case 32: goto tr402;
		case 40: goto tr403;
		case 41: goto tr404;
		case 522: goto tr406;
		case 525: goto tr407;
		case 571: goto tr408;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr405;
	goto st0;
tr407:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st111;
st111:
	if ( ++p == pe )
		goto _test_eof111;
case 111:
#line 7329 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st110;
	goto st0;
tr408:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st112;
st112:
	if ( ++p == pe )
		goto _test_eof112;
case 112:
#line 7348 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st110;
		case 525: goto st111;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st112;
	goto st0;
tr18:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st113;
tr1450:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st113;
tr1507:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st113;
st113:
	if ( ++p == pe )
		goto _test_eof113;
case 113:
#line 7409 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 65: goto st114;
		case 83: goto st197;
		case 97: goto st114;
		case 115: goto st197;
	}
	goto st0;
st114:
	if ( ++p == pe )
		goto _test_eof114;
case 114:
	switch( (*p) ) {
		case 80: goto st115;
		case 112: goto st115;
	}
	goto st0;
st115:
	if ( ++p == pe )
		goto _test_eof115;
case 115:
	switch( (*p) ) {
		case 84: goto st116;
		case 116: goto st116;
	}
	goto st0;
st116:
	if ( ++p == pe )
		goto _test_eof116;
case 116:
	switch( (*p) ) {
		case 82: goto st117;
		case 114: goto st117;
	}
	goto st0;
st117:
	if ( ++p == pe )
		goto _test_eof117;
case 117:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st118;
		case 32: goto st118;
		case 40: goto tr415;
		case 41: goto tr416;
		case 522: goto st194;
		case 525: goto st195;
		case 571: goto st196;
	}
	goto st0;
tr415:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st118;
tr416:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st118;
tr693:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st118;
tr694:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st118;
tr695:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st118;
st118:
	if ( ++p == pe )
		goto _test_eof118;
case 118:
#line 7507 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st118;
		case 32: goto st118;
		case 40: goto tr415;
		case 41: goto tr416;
		case 522: goto st194;
		case 525: goto st195;
		case 571: goto st196;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr420;
	goto st0;
tr420:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st119;
tr696:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st119;
st119:
	if ( ++p == pe )
		goto _test_eof119;
case 119:
#line 7555 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr421;
		case 32: goto tr421;
		case 40: goto tr422;
		case 41: goto tr423;
		case 522: goto tr425;
		case 525: goto tr426;
		case 571: goto tr427;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st119;
	goto st0;
tr429:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr430:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
tr421:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st120;
tr686:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st120;
tr687:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr688:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
tr422:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st120;
tr423:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st120;
st120:
	if ( ++p == pe )
		goto _test_eof120;
case 120:
#line 7639 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st120;
		case 32: goto st120;
		case 40: goto tr429;
		case 41: goto tr430;
		case 522: goto st191;
		case 525: goto st192;
		case 571: goto st193;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr431;
	goto st0;
tr431:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st121;
tr689:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st121;
st121:
	if ( ++p == pe )
		goto _test_eof121;
case 121:
#line 7687 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr435;
		case 32: goto tr435;
		case 40: goto tr436;
		case 41: goto tr437;
		case 522: goto tr439;
		case 525: goto tr440;
		case 571: goto tr441;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st121;
	goto st0;
tr445:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr446:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
tr678:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st122;
tr680:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr681:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
tr435:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st122;
tr436:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st122;
tr437:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st122;
st122:
	if ( ++p == pe )
		goto _test_eof122;
case 122:
#line 7771 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st122;
		case 32: goto st122;
		case 34: goto tr444;
		case 40: goto tr445;
		case 41: goto tr446;
		case 92: goto tr447;
		case 522: goto st188;
		case 525: goto st189;
		case 571: goto st190;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr442;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr442;
		} else if ( _widec >= 14 )
			goto tr442;
	} else
		goto tr442;
	goto st0;
tr650:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st123;
tr442:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st123;
tr677:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st123;
st123:
	if ( ++p == pe )
		goto _test_eof123;
case 123:
#line 7838 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr452;
		case 32: goto tr452;
		case 40: goto tr453;
		case 41: goto tr454;
		case 92: goto st176;
		case 522: goto tr456;
		case 525: goto tr457;
		case 571: goto tr458;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st123;
		} else
			goto st123;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st123;
		} else if ( _widec >= 35 )
			goto st123;
	} else
		goto st123;
	goto st0;
tr462:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr463:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr639:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st124;
tr641:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr642:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr452:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st124;
tr453:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr454:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr651:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st124;
tr652:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr653:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
tr669:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st124;
tr670:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st124;
tr671:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st124;
st124:
	if ( ++p == pe )
		goto _test_eof124;
case 124:
#line 7967 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st124;
		case 32: goto st124;
		case 34: goto tr461;
		case 40: goto tr462;
		case 41: goto tr463;
		case 92: goto tr464;
		case 522: goto st173;
		case 525: goto st174;
		case 571: goto st175;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr459;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr459;
		} else if ( _widec >= 14 )
			goto tr459;
	} else
		goto tr459;
	goto st0;
tr459:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st125;
tr611:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st125;
tr638:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st125;
st125:
	if ( ++p == pe )
		goto _test_eof125;
case 125:
#line 8030 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr469;
		case 32: goto tr469;
		case 40: goto tr470;
		case 41: goto tr471;
		case 92: goto st161;
		case 522: goto tr473;
		case 525: goto tr474;
		case 571: goto tr475;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st125;
		} else
			goto st125;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st125;
		} else if ( _widec >= 35 )
			goto st125;
	} else
		goto st125;
	goto st0;
tr479:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr480:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr600:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st126;
tr602:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr603:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr469:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st126;
tr470:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr471:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr612:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st126;
tr613:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr614:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
tr630:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st126;
tr631:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st126;
tr632:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st126;
st126:
	if ( ++p == pe )
		goto _test_eof126;
case 126:
#line 8159 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st126;
		case 32: goto st126;
		case 34: goto tr478;
		case 40: goto tr479;
		case 41: goto tr480;
		case 92: goto tr481;
		case 522: goto st158;
		case 525: goto st159;
		case 571: goto st160;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr476;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr476;
		} else if ( _widec >= 14 )
			goto tr476;
	} else
		goto tr476;
	goto st0;
tr476:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st127;
tr572:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st127;
tr599:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st127;
st127:
	if ( ++p == pe )
		goto _test_eof127;
case 127:
#line 8222 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr486;
		case 32: goto tr486;
		case 40: goto tr487;
		case 41: goto tr488;
		case 92: goto st146;
		case 522: goto tr490;
		case 525: goto tr491;
		case 571: goto tr492;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st127;
		} else
			goto st127;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st127;
		} else if ( _widec >= 35 )
			goto st127;
	} else
		goto st127;
	goto st0;
tr496:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr497:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr560:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st128;
tr562:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr563:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr486:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st128;
tr487:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr488:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr573:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st128;
tr574:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr575:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
tr591:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st128;
tr592:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st128;
tr593:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st128;
st128:
	if ( ++p == pe )
		goto _test_eof128;
case 128:
#line 8351 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st128;
		case 32: goto st128;
		case 34: goto tr495;
		case 40: goto tr496;
		case 41: goto tr497;
		case 64: goto tr498;
		case 92: goto tr499;
		case 522: goto st143;
		case 525: goto st144;
		case 571: goto st145;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr493;
		} else
			goto tr493;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr493;
		} else if ( _widec >= 37 )
			goto tr493;
	} else
		goto tr493;
	goto st0;
tr493:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st129;
tr517:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st129;
tr559:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st129;
st129:
	if ( ++p == pe )
		goto _test_eof129;
case 129:
#line 8418 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr504;
		case 32: goto tr504;
		case 40: goto tr505;
		case 41: goto tr506;
		case 92: goto st130;
		case 266: goto tr508;
		case 269: goto tr509;
		case 315: goto tr510;
		case 522: goto tr511;
		case 525: goto tr512;
		case 571: goto tr513;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st129;
		} else
			goto st129;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st129;
		} else if ( _widec >= 35 )
			goto st129;
	} else
		goto st129;
	goto st0;
tr499:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st130;
tr521:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st130;
tr565:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st130;
st130:
	if ( ++p == pe )
		goto _test_eof130;
case 130:
#line 8486 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st131;
		case 13: goto st132;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st133;
	goto st129;
st131:
	if ( ++p == pe )
		goto _test_eof131;
case 131:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr518;
		case 32: goto tr518;
		case 40: goto tr519;
		case 41: goto tr520;
		case 92: goto tr521;
		case 266: goto tr522;
		case 269: goto tr523;
		case 315: goto tr524;
		case 522: goto tr525;
		case 525: goto tr526;
		case 571: goto tr527;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr517;
		} else
			goto tr517;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr517;
		} else if ( _widec >= 35 )
			goto tr517;
	} else
		goto tr517;
	goto st0;
st132:
	if ( ++p == pe )
		goto _test_eof132;
case 132:
	if ( (*p) == 10 )
		goto st131;
	goto st0;
st133:
	if ( ++p == pe )
		goto _test_eof133;
case 133:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st134;
	goto st0;
st134:
	if ( ++p == pe )
		goto _test_eof134;
case 134:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st129;
	goto st0;
tr495:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st135;
tr534:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st135;
tr561:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st135;
st135:
	if ( ++p == pe )
		goto _test_eof135;
case 135:
#line 8589 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st136;
		case 13: goto st137;
		case 34: goto st138;
		case 92: goto st139;
	}
	goto st135;
tr535:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st136;
st136:
	if ( ++p == pe )
		goto _test_eof136;
case 136:
#line 8605 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr535;
		case 13: goto tr536;
		case 34: goto tr537;
		case 92: goto tr538;
	}
	goto tr534;
tr536:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st137;
st137:
	if ( ++p == pe )
		goto _test_eof137;
case 137:
#line 8621 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st136;
	goto st0;
tr537:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st138;
st138:
	if ( ++p == pe )
		goto _test_eof138;
case 138:
#line 8633 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr539;
		case 32: goto tr539;
		case 40: goto tr540;
		case 41: goto tr541;
		case 266: goto tr542;
		case 269: goto tr543;
		case 315: goto tr544;
		case 522: goto tr545;
		case 525: goto tr546;
		case 571: goto tr547;
	}
	goto st0;
tr538:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st139;
st139:
	if ( ++p == pe )
		goto _test_eof139;
case 139:
#line 8676 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st136;
		case 13: goto st137;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st140;
	goto st135;
st140:
	if ( ++p == pe )
		goto _test_eof140;
case 140:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st141;
	goto st0;
st141:
	if ( ++p == pe )
		goto _test_eof141;
case 141:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st135;
	goto st0;
tr498:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st142;
tr564:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st142;
st142:
	if ( ++p == pe )
		goto _test_eof142;
case 142:
#line 8715 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr550;
		case 32: goto tr550;
		case 40: goto tr551;
		case 41: goto tr552;
		case 266: goto tr553;
		case 269: goto tr554;
		case 315: goto tr555;
		case 522: goto tr556;
		case 525: goto tr557;
		case 571: goto tr558;
	}
	goto st0;
tr566:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st143;
tr490:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st143;
tr577:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st143;
tr594:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st143;
st143:
	if ( ++p == pe )
		goto _test_eof143;
case 143:
#line 8772 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr560;
		case 32: goto tr560;
		case 34: goto tr561;
		case 40: goto tr562;
		case 41: goto tr563;
		case 64: goto tr564;
		case 92: goto tr565;
		case 522: goto tr566;
		case 525: goto tr567;
		case 571: goto tr568;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr559;
		} else
			goto tr559;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr559;
		} else if ( _widec >= 37 )
			goto tr559;
	} else
		goto tr559;
	goto st0;
tr567:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st144;
tr491:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st144;
tr578:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st144;
tr595:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st144;
st144:
	if ( ++p == pe )
		goto _test_eof144;
case 144:
#line 8843 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st143;
	goto st0;
tr568:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st145;
tr492:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st145;
tr579:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st145;
tr596:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st145;
st145:
	if ( ++p == pe )
		goto _test_eof145;
case 145:
#line 8876 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st143;
		case 525: goto st144;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st145;
	goto st0;
tr481:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st146;
tr576:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st146;
tr604:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st146;
st146:
	if ( ++p == pe )
		goto _test_eof146;
case 146:
#line 8937 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st147;
		case 13: goto st148;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st149;
	goto st127;
st147:
	if ( ++p == pe )
		goto _test_eof147;
case 147:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr573;
		case 32: goto tr573;
		case 40: goto tr574;
		case 41: goto tr575;
		case 92: goto tr576;
		case 522: goto tr577;
		case 525: goto tr578;
		case 571: goto tr579;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr572;
		} else
			goto tr572;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr572;
		} else if ( _widec >= 35 )
			goto tr572;
	} else
		goto tr572;
	goto st0;
st148:
	if ( ++p == pe )
		goto _test_eof148;
case 148:
	if ( (*p) == 10 )
		goto st147;
	goto st0;
st149:
	if ( ++p == pe )
		goto _test_eof149;
case 149:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st150;
	goto st0;
st150:
	if ( ++p == pe )
		goto _test_eof150;
case 150:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st127;
	goto st0;
tr478:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st151;
tr586:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st151;
tr601:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st151;
st151:
	if ( ++p == pe )
		goto _test_eof151;
case 151:
#line 9037 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st152;
		case 13: goto st153;
		case 34: goto st154;
		case 92: goto st155;
	}
	goto st151;
tr587:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st152;
st152:
	if ( ++p == pe )
		goto _test_eof152;
case 152:
#line 9053 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr587;
		case 13: goto tr588;
		case 34: goto tr589;
		case 92: goto tr590;
	}
	goto tr586;
tr588:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st153;
st153:
	if ( ++p == pe )
		goto _test_eof153;
case 153:
#line 9069 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st152;
	goto st0;
tr589:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st154;
st154:
	if ( ++p == pe )
		goto _test_eof154;
case 154:
#line 9081 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr591;
		case 32: goto tr591;
		case 40: goto tr592;
		case 41: goto tr593;
		case 522: goto tr594;
		case 525: goto tr595;
		case 571: goto tr596;
	}
	goto st0;
tr590:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st155;
st155:
	if ( ++p == pe )
		goto _test_eof155;
case 155:
#line 9121 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st152;
		case 13: goto st153;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st156;
	goto st151;
st156:
	if ( ++p == pe )
		goto _test_eof156;
case 156:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st157;
	goto st0;
st157:
	if ( ++p == pe )
		goto _test_eof157;
case 157:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st151;
	goto st0;
tr605:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st158;
tr473:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st158;
tr616:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st158;
tr633:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st158;
st158:
	if ( ++p == pe )
		goto _test_eof158;
case 158:
#line 9168 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr600;
		case 32: goto tr600;
		case 34: goto tr601;
		case 40: goto tr602;
		case 41: goto tr603;
		case 92: goto tr604;
		case 522: goto tr605;
		case 525: goto tr606;
		case 571: goto tr607;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr599;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr599;
		} else if ( _widec >= 14 )
			goto tr599;
	} else
		goto tr599;
	goto st0;
tr606:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st159;
tr474:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st159;
tr617:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st159;
tr634:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st159;
st159:
	if ( ++p == pe )
		goto _test_eof159;
case 159:
#line 9235 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st158;
	goto st0;
tr607:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st160;
tr475:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st160;
tr618:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st160;
tr635:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st160;
st160:
	if ( ++p == pe )
		goto _test_eof160;
case 160:
#line 9268 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st158;
		case 525: goto st159;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st160;
	goto st0;
tr464:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st161;
tr615:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st161;
tr643:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st161;
st161:
	if ( ++p == pe )
		goto _test_eof161;
case 161:
#line 9329 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st162;
		case 13: goto st163;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st164;
	goto st125;
st162:
	if ( ++p == pe )
		goto _test_eof162;
case 162:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr612;
		case 32: goto tr612;
		case 40: goto tr613;
		case 41: goto tr614;
		case 92: goto tr615;
		case 522: goto tr616;
		case 525: goto tr617;
		case 571: goto tr618;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr611;
		} else
			goto tr611;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr611;
		} else if ( _widec >= 35 )
			goto tr611;
	} else
		goto tr611;
	goto st0;
st163:
	if ( ++p == pe )
		goto _test_eof163;
case 163:
	if ( (*p) == 10 )
		goto st162;
	goto st0;
st164:
	if ( ++p == pe )
		goto _test_eof164;
case 164:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st165;
	goto st0;
st165:
	if ( ++p == pe )
		goto _test_eof165;
case 165:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st125;
	goto st0;
tr461:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st166;
tr625:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st166;
tr640:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st166;
st166:
	if ( ++p == pe )
		goto _test_eof166;
case 166:
#line 9429 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st167;
		case 13: goto st168;
		case 34: goto st169;
		case 92: goto st170;
	}
	goto st166;
tr626:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st167;
st167:
	if ( ++p == pe )
		goto _test_eof167;
case 167:
#line 9445 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr626;
		case 13: goto tr627;
		case 34: goto tr628;
		case 92: goto tr629;
	}
	goto tr625;
tr627:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st168;
st168:
	if ( ++p == pe )
		goto _test_eof168;
case 168:
#line 9461 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st167;
	goto st0;
tr628:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st169;
st169:
	if ( ++p == pe )
		goto _test_eof169;
case 169:
#line 9473 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr630;
		case 32: goto tr630;
		case 40: goto tr631;
		case 41: goto tr632;
		case 522: goto tr633;
		case 525: goto tr634;
		case 571: goto tr635;
	}
	goto st0;
tr629:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st170;
st170:
	if ( ++p == pe )
		goto _test_eof170;
case 170:
#line 9513 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st167;
		case 13: goto st168;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st171;
	goto st166;
st171:
	if ( ++p == pe )
		goto _test_eof171;
case 171:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st172;
	goto st0;
st172:
	if ( ++p == pe )
		goto _test_eof172;
case 172:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st166;
	goto st0;
tr644:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st173;
tr456:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st173;
tr655:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st173;
tr672:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st173;
st173:
	if ( ++p == pe )
		goto _test_eof173;
case 173:
#line 9560 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr639;
		case 32: goto tr639;
		case 34: goto tr640;
		case 40: goto tr641;
		case 41: goto tr642;
		case 92: goto tr643;
		case 522: goto tr644;
		case 525: goto tr645;
		case 571: goto tr646;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr638;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr638;
		} else if ( _widec >= 14 )
			goto tr638;
	} else
		goto tr638;
	goto st0;
tr645:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st174;
tr457:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st174;
tr656:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st174;
tr673:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st174;
st174:
	if ( ++p == pe )
		goto _test_eof174;
case 174:
#line 9627 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st173;
	goto st0;
tr646:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st175;
tr458:
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st175;
tr657:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 503 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, false); }
	goto st175;
tr674:
#line 504 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, false); }
	goto st175;
st175:
	if ( ++p == pe )
		goto _test_eof175;
case 175:
#line 9660 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st173;
		case 525: goto st174;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st175;
	goto st0;
tr654:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st176;
tr447:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st176;
tr682:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st176;
st176:
	if ( ++p == pe )
		goto _test_eof176;
case 176:
#line 9725 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st177;
		case 13: goto st178;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st179;
	goto st123;
st177:
	if ( ++p == pe )
		goto _test_eof177;
case 177:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr651;
		case 32: goto tr651;
		case 40: goto tr652;
		case 41: goto tr653;
		case 92: goto tr654;
		case 522: goto tr655;
		case 525: goto tr656;
		case 571: goto tr657;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr650;
		} else
			goto tr650;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr650;
		} else if ( _widec >= 35 )
			goto tr650;
	} else
		goto tr650;
	goto st0;
st178:
	if ( ++p == pe )
		goto _test_eof178;
case 178:
	if ( (*p) == 10 )
		goto st177;
	goto st0;
st179:
	if ( ++p == pe )
		goto _test_eof179;
case 179:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st180;
	goto st0;
st180:
	if ( ++p == pe )
		goto _test_eof180;
case 180:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st123;
	goto st0;
tr664:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st181;
tr444:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st181;
tr679:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st181;
st181:
	if ( ++p == pe )
		goto _test_eof181;
case 181:
#line 9829 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st182;
		case 13: goto st183;
		case 34: goto st184;
		case 92: goto st185;
	}
	goto st181;
tr665:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st182;
st182:
	if ( ++p == pe )
		goto _test_eof182;
case 182:
#line 9845 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr665;
		case 13: goto tr666;
		case 34: goto tr667;
		case 92: goto tr668;
	}
	goto tr664;
tr666:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st183;
st183:
	if ( ++p == pe )
		goto _test_eof183;
case 183:
#line 9861 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st182;
	goto st0;
tr667:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st184;
st184:
	if ( ++p == pe )
		goto _test_eof184;
case 184:
#line 9873 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr669;
		case 32: goto tr669;
		case 40: goto tr670;
		case 41: goto tr671;
		case 522: goto tr672;
		case 525: goto tr673;
		case 571: goto tr674;
	}
	goto st0;
tr668:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st185;
st185:
	if ( ++p == pe )
		goto _test_eof185;
case 185:
#line 9913 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st182;
		case 13: goto st183;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st186;
	goto st181;
st186:
	if ( ++p == pe )
		goto _test_eof186;
case 186:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st187;
	goto st0;
st187:
	if ( ++p == pe )
		goto _test_eof187;
case 187:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st181;
	goto st0;
tr683:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st188;
tr439:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st188;
st188:
	if ( ++p == pe )
		goto _test_eof188;
case 188:
#line 9952 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr678;
		case 32: goto tr678;
		case 34: goto tr679;
		case 40: goto tr680;
		case 41: goto tr681;
		case 92: goto tr682;
		case 522: goto tr683;
		case 525: goto tr684;
		case 571: goto tr685;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr677;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr677;
		} else if ( _widec >= 14 )
			goto tr677;
	} else
		goto tr677;
	goto st0;
tr684:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st189;
tr440:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st189;
st189:
	if ( ++p == pe )
		goto _test_eof189;
case 189:
#line 10011 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st188;
	goto st0;
tr685:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st190;
tr441:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st190;
st190:
	if ( ++p == pe )
		goto _test_eof190;
case 190:
#line 10036 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st188;
		case 525: goto st189;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st190;
	goto st0;
tr425:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st191;
tr690:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st191;
st191:
	if ( ++p == pe )
		goto _test_eof191;
case 191:
#line 10093 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr686;
		case 32: goto tr686;
		case 40: goto tr687;
		case 41: goto tr688;
		case 522: goto tr690;
		case 525: goto tr691;
		case 571: goto tr692;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr689;
	goto st0;
tr426:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st192;
tr691:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st192;
st192:
	if ( ++p == pe )
		goto _test_eof192;
case 192:
#line 10141 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st191;
	goto st0;
tr427:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st193;
tr692:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st193;
st193:
	if ( ++p == pe )
		goto _test_eof193;
case 193:
#line 10166 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st191;
		case 525: goto st192;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st193;
	goto st0;
tr697:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st194;
st194:
	if ( ++p == pe )
		goto _test_eof194;
case 194:
#line 10217 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr693;
		case 32: goto tr693;
		case 40: goto tr694;
		case 41: goto tr695;
		case 522: goto tr697;
		case 525: goto tr698;
		case 571: goto tr699;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr696;
	goto st0;
tr698:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st195;
st195:
	if ( ++p == pe )
		goto _test_eof195;
case 195:
#line 10259 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st194;
	goto st0;
tr699:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st196;
st196:
	if ( ++p == pe )
		goto _test_eof196;
case 196:
#line 10278 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st194;
		case 525: goto st195;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st196;
	goto st0;
st197:
	if ( ++p == pe )
		goto _test_eof197;
case 197:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st198;
		case 32: goto st198;
		case 40: goto tr701;
		case 41: goto tr702;
		case 522: goto st213;
		case 525: goto st214;
		case 571: goto st215;
	}
	goto st0;
tr701:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st198;
tr702:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st198;
tr767:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st198;
tr769:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st198;
tr770:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st198;
st198:
	if ( ++p == pe )
		goto _test_eof198;
case 198:
#line 10384 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st198;
		case 32: goto st198;
		case 34: goto tr707;
		case 40: goto tr701;
		case 41: goto tr702;
		case 64: goto tr708;
		case 92: goto tr709;
		case 522: goto st213;
		case 525: goto st214;
		case 571: goto st215;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr706;
		} else
			goto tr706;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr706;
		} else if ( _widec >= 37 )
			goto tr706;
	} else
		goto tr706;
	goto st0;
tr706:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st199;
tr724:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st199;
tr766:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st199;
st199:
	if ( ++p == pe )
		goto _test_eof199;
case 199:
#line 10451 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr711;
		case 32: goto tr711;
		case 40: goto tr712;
		case 41: goto tr713;
		case 92: goto st200;
		case 266: goto tr715;
		case 269: goto tr716;
		case 315: goto tr717;
		case 522: goto tr718;
		case 525: goto tr719;
		case 571: goto tr720;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st199;
		} else
			goto st199;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st199;
		} else if ( _widec >= 35 )
			goto st199;
	} else
		goto st199;
	goto st0;
tr709:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st200;
tr728:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st200;
tr772:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st200;
st200:
	if ( ++p == pe )
		goto _test_eof200;
case 200:
#line 10519 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st201;
		case 13: goto st202;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st203;
	goto st199;
st201:
	if ( ++p == pe )
		goto _test_eof201;
case 201:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr725;
		case 32: goto tr725;
		case 40: goto tr726;
		case 41: goto tr727;
		case 92: goto tr728;
		case 266: goto tr729;
		case 269: goto tr730;
		case 315: goto tr731;
		case 522: goto tr732;
		case 525: goto tr733;
		case 571: goto tr734;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr724;
		} else
			goto tr724;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr724;
		} else if ( _widec >= 35 )
			goto tr724;
	} else
		goto tr724;
	goto st0;
st202:
	if ( ++p == pe )
		goto _test_eof202;
case 202:
	if ( (*p) == 10 )
		goto st201;
	goto st0;
st203:
	if ( ++p == pe )
		goto _test_eof203;
case 203:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st204;
	goto st0;
st204:
	if ( ++p == pe )
		goto _test_eof204;
case 204:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st199;
	goto st0;
tr707:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st205;
tr741:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st205;
tr768:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st205;
st205:
	if ( ++p == pe )
		goto _test_eof205;
case 205:
#line 10622 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st206;
		case 13: goto st207;
		case 34: goto st208;
		case 92: goto st209;
	}
	goto st205;
tr742:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st206;
st206:
	if ( ++p == pe )
		goto _test_eof206;
case 206:
#line 10638 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr742;
		case 13: goto tr743;
		case 34: goto tr744;
		case 92: goto tr745;
	}
	goto tr741;
tr743:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st207;
st207:
	if ( ++p == pe )
		goto _test_eof207;
case 207:
#line 10654 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st206;
	goto st0;
tr744:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st208;
st208:
	if ( ++p == pe )
		goto _test_eof208;
case 208:
#line 10666 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr746;
		case 32: goto tr746;
		case 40: goto tr747;
		case 41: goto tr748;
		case 266: goto tr749;
		case 269: goto tr750;
		case 315: goto tr751;
		case 522: goto tr752;
		case 525: goto tr753;
		case 571: goto tr754;
	}
	goto st0;
tr745:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st209;
st209:
	if ( ++p == pe )
		goto _test_eof209;
case 209:
#line 10709 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st206;
		case 13: goto st207;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st210;
	goto st205;
st210:
	if ( ++p == pe )
		goto _test_eof210;
case 210:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st211;
	goto st0;
st211:
	if ( ++p == pe )
		goto _test_eof211;
case 211:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st205;
	goto st0;
tr708:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st212;
tr771:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st212;
st212:
	if ( ++p == pe )
		goto _test_eof212;
case 212:
#line 10748 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr757;
		case 32: goto tr757;
		case 40: goto tr758;
		case 41: goto tr759;
		case 266: goto tr760;
		case 269: goto tr761;
		case 315: goto tr762;
		case 522: goto tr763;
		case 525: goto tr764;
		case 571: goto tr765;
	}
	goto st0;
tr773:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st213;
st213:
	if ( ++p == pe )
		goto _test_eof213;
case 213:
#line 10791 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr767;
		case 32: goto tr767;
		case 34: goto tr768;
		case 40: goto tr769;
		case 41: goto tr770;
		case 64: goto tr771;
		case 92: goto tr772;
		case 522: goto tr773;
		case 525: goto tr774;
		case 571: goto tr775;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr766;
		} else
			goto tr766;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr766;
		} else if ( _widec >= 37 )
			goto tr766;
	} else
		goto tr766;
	goto st0;
tr774:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st214;
st214:
	if ( ++p == pe )
		goto _test_eof214;
case 214:
#line 10848 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st213;
	goto st0;
tr775:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st215;
st215:
	if ( ++p == pe )
		goto _test_eof215;
case 215:
#line 10867 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st213;
		case 525: goto st214;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st215;
	goto st0;
tr19:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st216;
tr1451:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st216;
tr1508:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st216;
st216:
	if ( ++p == pe )
		goto _test_eof216;
case 216:
#line 10928 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 84: goto st217;
		case 116: goto st217;
	}
	goto st0;
st217:
	if ( ++p == pe )
		goto _test_eof217;
case 217:
	switch( (*p) ) {
		case 82: goto st218;
		case 114: goto st218;
	}
	goto st0;
st218:
	if ( ++p == pe )
		goto _test_eof218;
case 218:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st219;
		case 32: goto st219;
		case 40: goto tr779;
		case 41: goto tr780;
		case 522: goto st234;
		case 525: goto st235;
		case 571: goto st236;
	}
	goto st0;
tr779:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st219;
tr780:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st219;
tr845:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st219;
tr847:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st219;
tr848:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st219;
st219:
	if ( ++p == pe )
		goto _test_eof219;
case 219:
#line 11006 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st219;
		case 32: goto st219;
		case 34: goto tr785;
		case 40: goto tr779;
		case 41: goto tr780;
		case 64: goto tr786;
		case 92: goto tr787;
		case 522: goto st234;
		case 525: goto st235;
		case 571: goto st236;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr784;
		} else
			goto tr784;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr784;
		} else if ( _widec >= 37 )
			goto tr784;
	} else
		goto tr784;
	goto st0;
tr784:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st220;
tr802:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st220;
tr844:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st220;
st220:
	if ( ++p == pe )
		goto _test_eof220;
case 220:
#line 11073 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr789;
		case 32: goto tr789;
		case 40: goto tr790;
		case 41: goto tr791;
		case 92: goto st221;
		case 266: goto tr793;
		case 269: goto tr794;
		case 315: goto tr795;
		case 522: goto tr796;
		case 525: goto tr797;
		case 571: goto tr798;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st220;
		} else
			goto st220;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st220;
		} else if ( _widec >= 35 )
			goto st220;
	} else
		goto st220;
	goto st0;
tr787:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st221;
tr806:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st221;
tr850:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st221;
st221:
	if ( ++p == pe )
		goto _test_eof221;
case 221:
#line 11141 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st222;
		case 13: goto st223;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st224;
	goto st220;
st222:
	if ( ++p == pe )
		goto _test_eof222;
case 222:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr803;
		case 32: goto tr803;
		case 40: goto tr804;
		case 41: goto tr805;
		case 92: goto tr806;
		case 266: goto tr807;
		case 269: goto tr808;
		case 315: goto tr809;
		case 522: goto tr810;
		case 525: goto tr811;
		case 571: goto tr812;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr802;
		} else
			goto tr802;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr802;
		} else if ( _widec >= 35 )
			goto tr802;
	} else
		goto tr802;
	goto st0;
st223:
	if ( ++p == pe )
		goto _test_eof223;
case 223:
	if ( (*p) == 10 )
		goto st222;
	goto st0;
st224:
	if ( ++p == pe )
		goto _test_eof224;
case 224:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st225;
	goto st0;
st225:
	if ( ++p == pe )
		goto _test_eof225;
case 225:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st220;
	goto st0;
tr785:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st226;
tr819:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st226;
tr846:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st226;
st226:
	if ( ++p == pe )
		goto _test_eof226;
case 226:
#line 11244 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st227;
		case 13: goto st228;
		case 34: goto st229;
		case 92: goto st230;
	}
	goto st226;
tr820:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st227;
st227:
	if ( ++p == pe )
		goto _test_eof227;
case 227:
#line 11260 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr820;
		case 13: goto tr821;
		case 34: goto tr822;
		case 92: goto tr823;
	}
	goto tr819;
tr821:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st228;
st228:
	if ( ++p == pe )
		goto _test_eof228;
case 228:
#line 11276 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st227;
	goto st0;
tr822:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st229;
st229:
	if ( ++p == pe )
		goto _test_eof229;
case 229:
#line 11288 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr824;
		case 32: goto tr824;
		case 40: goto tr825;
		case 41: goto tr826;
		case 266: goto tr827;
		case 269: goto tr828;
		case 315: goto tr829;
		case 522: goto tr830;
		case 525: goto tr831;
		case 571: goto tr832;
	}
	goto st0;
tr823:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st230;
st230:
	if ( ++p == pe )
		goto _test_eof230;
case 230:
#line 11331 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st227;
		case 13: goto st228;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st231;
	goto st226;
st231:
	if ( ++p == pe )
		goto _test_eof231;
case 231:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st232;
	goto st0;
st232:
	if ( ++p == pe )
		goto _test_eof232;
case 232:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st226;
	goto st0;
tr786:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st233;
tr849:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st233;
st233:
	if ( ++p == pe )
		goto _test_eof233;
case 233:
#line 11370 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr835;
		case 32: goto tr835;
		case 40: goto tr836;
		case 41: goto tr837;
		case 266: goto tr838;
		case 269: goto tr839;
		case 315: goto tr840;
		case 522: goto tr841;
		case 525: goto tr842;
		case 571: goto tr843;
	}
	goto st0;
tr851:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st234;
st234:
	if ( ++p == pe )
		goto _test_eof234;
case 234:
#line 11413 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr845;
		case 32: goto tr845;
		case 34: goto tr846;
		case 40: goto tr847;
		case 41: goto tr848;
		case 64: goto tr849;
		case 92: goto tr850;
		case 522: goto tr851;
		case 525: goto tr852;
		case 571: goto tr853;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr844;
		} else
			goto tr844;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr844;
		} else if ( _widec >= 37 )
			goto tr844;
	} else
		goto tr844;
	goto st0;
tr852:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st235;
st235:
	if ( ++p == pe )
		goto _test_eof235;
case 235:
#line 11470 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st234;
	goto st0;
tr853:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st236;
st236:
	if ( ++p == pe )
		goto _test_eof236;
case 236:
#line 11489 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st234;
		case 525: goto st235;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st236;
	goto st0;
tr20:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st237;
tr1452:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st237;
tr1509:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st237;
st237:
	if ( ++p == pe )
		goto _test_eof237;
case 237:
#line 11550 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 79: goto st238;
		case 82: goto st305;
		case 111: goto st238;
		case 114: goto st305;
	}
	goto st0;
st238:
	if ( ++p == pe )
		goto _test_eof238;
case 238:
	switch( (*p) ) {
		case 65: goto st239;
		case 97: goto st239;
	}
	goto st0;
st239:
	if ( ++p == pe )
		goto _test_eof239;
case 239:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st240;
		case 32: goto st240;
		case 40: goto tr858;
		case 41: goto tr859;
		case 522: goto st302;
		case 525: goto st303;
		case 571: goto st304;
	}
	goto st0;
tr858:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st240;
tr859:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st240;
tr1114:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st240;
tr1116:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st240;
tr1117:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st240;
st240:
	if ( ++p == pe )
		goto _test_eof240;
case 240:
#line 11630 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st240;
		case 32: goto st240;
		case 34: goto tr864;
		case 40: goto tr858;
		case 41: goto tr859;
		case 64: goto tr865;
		case 92: goto tr866;
		case 522: goto st302;
		case 525: goto st303;
		case 571: goto st304;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr863;
		} else
			goto tr863;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr863;
		} else if ( _widec >= 37 )
			goto tr863;
	} else
		goto tr863;
	goto st0;
tr863:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st241;
tr1080:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st241;
tr1113:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st241;
st241:
	if ( ++p == pe )
		goto _test_eof241;
case 241:
#line 11697 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr868;
		case 32: goto tr868;
		case 40: goto tr869;
		case 41: goto tr870;
		case 92: goto st289;
		case 522: goto tr872;
		case 525: goto tr873;
		case 571: goto tr874;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st241;
		} else
			goto st241;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st241;
		} else if ( _widec >= 35 )
			goto st241;
	} else
		goto st241;
	goto st0;
tr878:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr879:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1069:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st242;
tr1071:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1072:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr868:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st242;
tr869:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr870:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1081:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st242;
tr1082:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1083:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1099:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st242;
tr1100:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1101:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
tr1107:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st242;
tr1108:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st242;
tr1109:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st242;
st242:
	if ( ++p == pe )
		goto _test_eof242;
case 242:
#line 11842 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st242;
		case 32: goto st242;
		case 34: goto tr877;
		case 40: goto tr878;
		case 41: goto tr879;
		case 92: goto tr880;
		case 522: goto st286;
		case 525: goto st287;
		case 571: goto st288;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr875;
		} else
			goto tr875;
	} else if ( _widec > 35 ) {
		if ( _widec < 60 ) {
			if ( 37 <= _widec && _widec <= 58 )
				goto tr875;
		} else if ( _widec > 63 ) {
			if ( 65 <= _widec )
				goto tr875;
		} else
			goto tr875;
	} else
		goto tr875;
	goto st0;
tr875:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st243;
tr1041:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st243;
tr1068:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st243;
st243:
	if ( ++p == pe )
		goto _test_eof243;
case 243:
#line 11911 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr885;
		case 32: goto tr885;
		case 40: goto tr886;
		case 41: goto tr887;
		case 92: goto st274;
		case 522: goto tr889;
		case 525: goto tr890;
		case 571: goto tr891;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st243;
		} else
			goto st243;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st243;
		} else if ( _widec >= 35 )
			goto st243;
	} else
		goto st243;
	goto st0;
tr893:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr894:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1031:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st244;
tr1032:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1033:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr885:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st244;
tr886:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr887:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1042:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st244;
tr1043:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1044:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
tr1060:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st244;
tr1061:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st244;
tr1062:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st244;
st244:
	if ( ++p == pe )
		goto _test_eof244;
case 244:
#line 12040 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st244;
		case 32: goto st244;
		case 40: goto tr893;
		case 41: goto tr894;
		case 522: goto st271;
		case 525: goto st272;
		case 571: goto st273;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr895;
	goto st0;
tr895:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st245;
tr1034:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st245;
st245:
	if ( ++p == pe )
		goto _test_eof245;
case 245:
#line 12088 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr899;
		case 32: goto tr899;
		case 40: goto tr900;
		case 41: goto tr901;
		case 68: goto tr903;
		case 72: goto tr903;
		case 77: goto tr903;
		case 87: goto tr903;
		case 100: goto tr903;
		case 104: goto tr903;
		case 109: goto tr903;
		case 119: goto tr903;
		case 522: goto tr904;
		case 525: goto tr905;
		case 571: goto tr906;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st245;
	goto st0;
tr908:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr909:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr899:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st246;
tr1018:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st246;
tr1019:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr1020:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr900:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr901:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
tr1025:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st246;
tr1026:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st246;
tr1027:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st246;
st246:
	if ( ++p == pe )
		goto _test_eof246;
case 246:
#line 12196 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st246;
		case 32: goto st246;
		case 40: goto tr908;
		case 41: goto tr909;
		case 522: goto st267;
		case 525: goto st268;
		case 571: goto st269;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr910;
	goto st0;
tr910:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st247;
tr1021:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st247;
st247:
	if ( ++p == pe )
		goto _test_eof247;
case 247:
#line 12244 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr914;
		case 32: goto tr914;
		case 40: goto tr915;
		case 41: goto tr916;
		case 68: goto tr918;
		case 72: goto tr918;
		case 77: goto tr918;
		case 87: goto tr918;
		case 100: goto tr918;
		case 104: goto tr918;
		case 109: goto tr918;
		case 119: goto tr918;
		case 522: goto tr919;
		case 525: goto tr920;
		case 571: goto tr921;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st247;
	goto st0;
tr923:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr924:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr1005:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st248;
tr1006:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr1007:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr914:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st248;
tr915:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr916:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
tr1012:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st248;
tr1013:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st248;
tr1014:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st248;
st248:
	if ( ++p == pe )
		goto _test_eof248;
case 248:
#line 12352 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st248;
		case 32: goto st248;
		case 40: goto tr923;
		case 41: goto tr924;
		case 522: goto st263;
		case 525: goto st264;
		case 571: goto st265;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr925;
	goto st0;
tr925:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st249;
tr1008:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st249;
st249:
	if ( ++p == pe )
		goto _test_eof249;
case 249:
#line 12400 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr929;
		case 32: goto tr929;
		case 40: goto tr930;
		case 41: goto tr931;
		case 68: goto tr933;
		case 72: goto tr933;
		case 77: goto tr933;
		case 87: goto tr933;
		case 100: goto tr933;
		case 104: goto tr933;
		case 109: goto tr933;
		case 119: goto tr933;
		case 522: goto tr934;
		case 525: goto tr935;
		case 571: goto tr936;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st249;
	goto st0;
tr938:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr939:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr992:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st250;
tr993:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr994:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr929:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st250;
tr930:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr931:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
tr999:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st250;
tr1000:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st250;
tr1001:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st250;
st250:
	if ( ++p == pe )
		goto _test_eof250;
case 250:
#line 12508 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st250;
		case 32: goto st250;
		case 40: goto tr938;
		case 41: goto tr939;
		case 522: goto st259;
		case 525: goto st260;
		case 571: goto st261;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr940;
	goto st0;
tr940:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st251;
tr995:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st251;
st251:
	if ( ++p == pe )
		goto _test_eof251;
case 251:
#line 12556 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr944;
		case 32: goto tr944;
		case 40: goto tr945;
		case 41: goto tr946;
		case 68: goto tr948;
		case 72: goto tr948;
		case 77: goto tr948;
		case 87: goto tr948;
		case 100: goto tr948;
		case 104: goto tr948;
		case 109: goto tr948;
		case 119: goto tr948;
		case 522: goto tr949;
		case 525: goto tr950;
		case 571: goto tr951;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st251;
	goto st0;
tr953:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr954:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr979:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st252;
tr980:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr981:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr944:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st252;
tr945:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr946:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
tr986:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st252;
tr987:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st252;
tr988:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st252;
st252:
	if ( ++p == pe )
		goto _test_eof252;
case 252:
#line 12664 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st252;
		case 32: goto st252;
		case 40: goto tr953;
		case 41: goto tr954;
		case 522: goto st255;
		case 525: goto st256;
		case 571: goto st257;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr955;
	goto st0;
tr955:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st253;
tr982:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st253;
st253:
	if ( ++p == pe )
		goto _test_eof253;
case 253:
#line 12712 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr959;
		case 32: goto tr959;
		case 40: goto tr960;
		case 41: goto tr961;
		case 68: goto tr963;
		case 72: goto tr963;
		case 77: goto tr963;
		case 87: goto tr963;
		case 100: goto tr963;
		case 104: goto tr963;
		case 109: goto tr963;
		case 119: goto tr963;
		case 266: goto tr964;
		case 269: goto tr965;
		case 315: goto tr966;
		case 522: goto tr967;
		case 525: goto tr968;
		case 571: goto tr969;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st253;
	goto st0;
tr963:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st254;
st254:
	if ( ++p == pe )
		goto _test_eof254;
case 254:
#line 12767 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr970;
		case 32: goto tr970;
		case 40: goto tr971;
		case 41: goto tr972;
		case 266: goto tr973;
		case 269: goto tr974;
		case 315: goto tr975;
		case 522: goto tr976;
		case 525: goto tr977;
		case 571: goto tr978;
	}
	goto st0;
tr983:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st255;
tr949:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st255;
tr989:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st255;
st255:
	if ( ++p == pe )
		goto _test_eof255;
case 255:
#line 12820 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr979;
		case 32: goto tr979;
		case 40: goto tr980;
		case 41: goto tr981;
		case 522: goto tr983;
		case 525: goto tr984;
		case 571: goto tr985;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr982;
	goto st0;
tr984:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st256;
tr950:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st256;
tr990:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st256;
st256:
	if ( ++p == pe )
		goto _test_eof256;
case 256:
#line 12872 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st255;
	goto st0;
tr985:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st257;
tr951:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st257;
tr991:
#line 519 "./zscan_rfc1035.rl"
	{ z->uv_4 = z->uval; }
	goto st257;
st257:
	if ( ++p == pe )
		goto _test_eof257;
case 257:
#line 12901 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st255;
		case 525: goto st256;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st257;
	goto st0;
tr948:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st258;
st258:
	if ( ++p == pe )
		goto _test_eof258;
case 258:
#line 12954 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr986;
		case 32: goto tr986;
		case 40: goto tr987;
		case 41: goto tr988;
		case 522: goto tr989;
		case 525: goto tr990;
		case 571: goto tr991;
	}
	goto st0;
tr996:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st259;
tr934:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st259;
tr1002:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st259;
st259:
	if ( ++p == pe )
		goto _test_eof259;
case 259:
#line 13004 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr992;
		case 32: goto tr992;
		case 40: goto tr993;
		case 41: goto tr994;
		case 522: goto tr996;
		case 525: goto tr997;
		case 571: goto tr998;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr995;
	goto st0;
tr997:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st260;
tr935:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st260;
tr1003:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st260;
st260:
	if ( ++p == pe )
		goto _test_eof260;
case 260:
#line 13056 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st259;
	goto st0;
tr998:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st261;
tr936:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st261;
tr1004:
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st261;
st261:
	if ( ++p == pe )
		goto _test_eof261;
case 261:
#line 13085 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st259;
		case 525: goto st260;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st261;
	goto st0;
tr933:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st262;
st262:
	if ( ++p == pe )
		goto _test_eof262;
case 262:
#line 13138 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr999;
		case 32: goto tr999;
		case 40: goto tr1000;
		case 41: goto tr1001;
		case 522: goto tr1002;
		case 525: goto tr1003;
		case 571: goto tr1004;
	}
	goto st0;
tr1009:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st263;
tr919:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st263;
tr1015:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st263;
st263:
	if ( ++p == pe )
		goto _test_eof263;
case 263:
#line 13188 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1005;
		case 32: goto tr1005;
		case 40: goto tr1006;
		case 41: goto tr1007;
		case 522: goto tr1009;
		case 525: goto tr1010;
		case 571: goto tr1011;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1008;
	goto st0;
tr1010:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st264;
tr920:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st264;
tr1016:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st264;
st264:
	if ( ++p == pe )
		goto _test_eof264;
case 264:
#line 13240 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st263;
	goto st0;
tr1011:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st265;
tr921:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st265;
tr1017:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st265;
st265:
	if ( ++p == pe )
		goto _test_eof265;
case 265:
#line 13269 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st263;
		case 525: goto st264;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st265;
	goto st0;
tr918:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st266;
st266:
	if ( ++p == pe )
		goto _test_eof266;
case 266:
#line 13322 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1012;
		case 32: goto tr1012;
		case 40: goto tr1013;
		case 41: goto tr1014;
		case 522: goto tr1015;
		case 525: goto tr1016;
		case 571: goto tr1017;
	}
	goto st0;
tr904:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st267;
tr1022:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st267;
tr1028:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st267;
st267:
	if ( ++p == pe )
		goto _test_eof267;
case 267:
#line 13372 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1018;
		case 32: goto tr1018;
		case 40: goto tr1019;
		case 41: goto tr1020;
		case 522: goto tr1022;
		case 525: goto tr1023;
		case 571: goto tr1024;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1021;
	goto st0;
tr905:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st268;
tr1023:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st268;
tr1029:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st268;
st268:
	if ( ++p == pe )
		goto _test_eof268;
case 268:
#line 13424 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st267;
	goto st0;
tr906:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st269;
tr1024:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st269;
tr1030:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st269;
st269:
	if ( ++p == pe )
		goto _test_eof269;
case 269:
#line 13453 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st267;
		case 525: goto st268;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st269;
	goto st0;
tr903:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st270;
st270:
	if ( ++p == pe )
		goto _test_eof270;
case 270:
#line 13506 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1025;
		case 32: goto tr1025;
		case 40: goto tr1026;
		case 41: goto tr1027;
		case 522: goto tr1028;
		case 525: goto tr1029;
		case 571: goto tr1030;
	}
	goto st0;
tr1035:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st271;
tr889:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st271;
tr1046:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st271;
tr1063:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st271;
st271:
	if ( ++p == pe )
		goto _test_eof271;
case 271:
#line 13560 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1031;
		case 32: goto tr1031;
		case 40: goto tr1032;
		case 41: goto tr1033;
		case 522: goto tr1035;
		case 525: goto tr1036;
		case 571: goto tr1037;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1034;
	goto st0;
tr1036:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st272;
tr890:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st272;
tr1047:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st272;
tr1064:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st272;
st272:
	if ( ++p == pe )
		goto _test_eof272;
case 272:
#line 13616 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st271;
	goto st0;
tr1037:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st273;
tr891:
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st273;
tr1048:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 492 "./zscan_rfc1035.rl"
	{ dname_set(z, z->eml_dname, p - z->tstart, false); }
	goto st273;
tr1065:
#line 493 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->eml_dname, p - z->tstart - 1, false); }
	goto st273;
st273:
	if ( ++p == pe )
		goto _test_eof273;
case 273:
#line 13649 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st271;
		case 525: goto st272;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st273;
	goto st0;
tr880:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st274;
tr1045:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st274;
tr1073:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st274;
st274:
	if ( ++p == pe )
		goto _test_eof274;
case 274:
#line 13710 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st275;
		case 13: goto st276;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st277;
	goto st243;
st275:
	if ( ++p == pe )
		goto _test_eof275;
case 275:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1042;
		case 32: goto tr1042;
		case 40: goto tr1043;
		case 41: goto tr1044;
		case 92: goto tr1045;
		case 522: goto tr1046;
		case 525: goto tr1047;
		case 571: goto tr1048;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1041;
		} else
			goto tr1041;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1041;
		} else if ( _widec >= 35 )
			goto tr1041;
	} else
		goto tr1041;
	goto st0;
st276:
	if ( ++p == pe )
		goto _test_eof276;
case 276:
	if ( (*p) == 10 )
		goto st275;
	goto st0;
st277:
	if ( ++p == pe )
		goto _test_eof277;
case 277:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st278;
	goto st0;
st278:
	if ( ++p == pe )
		goto _test_eof278;
case 278:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st243;
	goto st0;
tr877:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st279;
tr1055:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st279;
tr1070:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st279;
st279:
	if ( ++p == pe )
		goto _test_eof279;
case 279:
#line 13810 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st280;
		case 13: goto st281;
		case 34: goto st282;
		case 92: goto st283;
	}
	goto st279;
tr1056:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st280;
st280:
	if ( ++p == pe )
		goto _test_eof280;
case 280:
#line 13826 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1056;
		case 13: goto tr1057;
		case 34: goto tr1058;
		case 92: goto tr1059;
	}
	goto tr1055;
tr1057:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st281;
st281:
	if ( ++p == pe )
		goto _test_eof281;
case 281:
#line 13842 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st280;
	goto st0;
tr1058:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st282;
st282:
	if ( ++p == pe )
		goto _test_eof282;
case 282:
#line 13854 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1060;
		case 32: goto tr1060;
		case 40: goto tr1061;
		case 41: goto tr1062;
		case 522: goto tr1063;
		case 525: goto tr1064;
		case 571: goto tr1065;
	}
	goto st0;
tr1059:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st283;
st283:
	if ( ++p == pe )
		goto _test_eof283;
case 283:
#line 13894 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st280;
		case 13: goto st281;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st284;
	goto st279;
st284:
	if ( ++p == pe )
		goto _test_eof284;
case 284:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st285;
	goto st0;
st285:
	if ( ++p == pe )
		goto _test_eof285;
case 285:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st279;
	goto st0;
tr1074:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st286;
tr872:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st286;
tr1085:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st286;
tr1102:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st286;
tr1110:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st286;
st286:
	if ( ++p == pe )
		goto _test_eof286;
case 286:
#line 13945 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1069;
		case 32: goto tr1069;
		case 34: goto tr1070;
		case 40: goto tr1071;
		case 41: goto tr1072;
		case 92: goto tr1073;
		case 522: goto tr1074;
		case 525: goto tr1075;
		case 571: goto tr1076;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1068;
		} else
			goto tr1068;
	} else if ( _widec > 35 ) {
		if ( _widec < 60 ) {
			if ( 37 <= _widec && _widec <= 58 )
				goto tr1068;
		} else if ( _widec > 63 ) {
			if ( 65 <= _widec )
				goto tr1068;
		} else
			goto tr1068;
	} else
		goto tr1068;
	goto st0;
tr1075:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st287;
tr873:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st287;
tr1086:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st287;
tr1103:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st287;
tr1111:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st287;
st287:
	if ( ++p == pe )
		goto _test_eof287;
case 287:
#line 14022 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st286;
	goto st0;
tr1076:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st288;
tr874:
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st288;
tr1087:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 490 "./zscan_rfc1035.rl"
	{ dname_set(z, z->rhs_dname, p - z->tstart, false); }
	goto st288;
tr1104:
#line 491 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->rhs_dname, p - z->tstart - 1, false); }
	goto st288;
tr1112:
#line 489 "./zscan_rfc1035.rl"
	{ dname_copy(z->rhs_dname, z->origin); }
	goto st288;
st288:
	if ( ++p == pe )
		goto _test_eof288;
case 288:
#line 14059 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st286;
		case 525: goto st287;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st288;
	goto st0;
tr866:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st289;
tr1084:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st289;
tr1119:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st289;
st289:
	if ( ++p == pe )
		goto _test_eof289;
case 289:
#line 14120 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st290;
		case 13: goto st291;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st292;
	goto st241;
st290:
	if ( ++p == pe )
		goto _test_eof290;
case 290:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1081;
		case 32: goto tr1081;
		case 40: goto tr1082;
		case 41: goto tr1083;
		case 92: goto tr1084;
		case 522: goto tr1085;
		case 525: goto tr1086;
		case 571: goto tr1087;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1080;
		} else
			goto tr1080;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1080;
		} else if ( _widec >= 35 )
			goto tr1080;
	} else
		goto tr1080;
	goto st0;
st291:
	if ( ++p == pe )
		goto _test_eof291;
case 291:
	if ( (*p) == 10 )
		goto st290;
	goto st0;
st292:
	if ( ++p == pe )
		goto _test_eof292;
case 292:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st293;
	goto st0;
st293:
	if ( ++p == pe )
		goto _test_eof293;
case 293:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st241;
	goto st0;
tr864:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st294;
tr1094:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st294;
tr1115:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st294;
st294:
	if ( ++p == pe )
		goto _test_eof294;
case 294:
#line 14220 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st295;
		case 13: goto st296;
		case 34: goto st297;
		case 92: goto st298;
	}
	goto st294;
tr1095:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st295;
st295:
	if ( ++p == pe )
		goto _test_eof295;
case 295:
#line 14236 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1095;
		case 13: goto tr1096;
		case 34: goto tr1097;
		case 92: goto tr1098;
	}
	goto tr1094;
tr1096:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st296;
st296:
	if ( ++p == pe )
		goto _test_eof296;
case 296:
#line 14252 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st295;
	goto st0;
tr1097:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st297;
st297:
	if ( ++p == pe )
		goto _test_eof297;
case 297:
#line 14264 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1099;
		case 32: goto tr1099;
		case 40: goto tr1100;
		case 41: goto tr1101;
		case 522: goto tr1102;
		case 525: goto tr1103;
		case 571: goto tr1104;
	}
	goto st0;
tr1098:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st298;
st298:
	if ( ++p == pe )
		goto _test_eof298;
case 298:
#line 14304 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st295;
		case 13: goto st296;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st299;
	goto st294;
st299:
	if ( ++p == pe )
		goto _test_eof299;
case 299:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st300;
	goto st0;
st300:
	if ( ++p == pe )
		goto _test_eof300;
case 300:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st294;
	goto st0;
tr865:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st301;
tr1118:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st301;
st301:
	if ( ++p == pe )
		goto _test_eof301;
case 301:
#line 14343 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1107;
		case 32: goto tr1107;
		case 40: goto tr1108;
		case 41: goto tr1109;
		case 522: goto tr1110;
		case 525: goto tr1111;
		case 571: goto tr1112;
	}
	goto st0;
tr1120:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st302;
st302:
	if ( ++p == pe )
		goto _test_eof302;
case 302:
#line 14383 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1114;
		case 32: goto tr1114;
		case 34: goto tr1115;
		case 40: goto tr1116;
		case 41: goto tr1117;
		case 64: goto tr1118;
		case 92: goto tr1119;
		case 522: goto tr1120;
		case 525: goto tr1121;
		case 571: goto tr1122;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1113;
		} else
			goto tr1113;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1113;
		} else if ( _widec >= 37 )
			goto tr1113;
	} else
		goto tr1113;
	goto st0;
tr1121:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st303;
st303:
	if ( ++p == pe )
		goto _test_eof303;
case 303:
#line 14440 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st302;
	goto st0;
tr1122:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st304;
st304:
	if ( ++p == pe )
		goto _test_eof304;
case 304:
#line 14459 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st302;
		case 525: goto st303;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st304;
	goto st0;
st305:
	if ( ++p == pe )
		goto _test_eof305;
case 305:
	switch( (*p) ) {
		case 86: goto st306;
		case 118: goto st306;
	}
	goto st0;
st306:
	if ( ++p == pe )
		goto _test_eof306;
case 306:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st307;
		case 32: goto st307;
		case 40: goto tr1125;
		case 41: goto tr1126;
		case 522: goto st337;
		case 525: goto st338;
		case 571: goto st339;
	}
	goto st0;
tr1125:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st307;
tr1126:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st307;
tr1256:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st307;
tr1257:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st307;
tr1258:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st307;
st307:
	if ( ++p == pe )
		goto _test_eof307;
case 307:
#line 14574 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st307;
		case 32: goto st307;
		case 40: goto tr1125;
		case 41: goto tr1126;
		case 522: goto st337;
		case 525: goto st338;
		case 571: goto st339;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1130;
	goto st0;
tr1130:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
tr1259:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st308;
st308:
	if ( ++p == pe )
		goto _test_eof308;
case 308:
#line 14622 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1131;
		case 32: goto tr1131;
		case 40: goto tr1132;
		case 41: goto tr1133;
		case 522: goto tr1135;
		case 525: goto tr1136;
		case 571: goto tr1137;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st308;
	goto st0;
tr1139:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1140:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1131:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st309;
tr1249:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st309;
tr1250:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1251:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
tr1132:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st309;
tr1133:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st309;
st309:
	if ( ++p == pe )
		goto _test_eof309;
case 309:
#line 14706 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st309;
		case 32: goto st309;
		case 40: goto tr1139;
		case 41: goto tr1140;
		case 522: goto st334;
		case 525: goto st335;
		case 571: goto st336;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1141;
	goto st0;
tr1141:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
tr1252:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st310;
st310:
	if ( ++p == pe )
		goto _test_eof310;
case 310:
#line 14754 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1145;
		case 32: goto tr1145;
		case 40: goto tr1146;
		case 41: goto tr1147;
		case 522: goto tr1149;
		case 525: goto tr1150;
		case 571: goto tr1151;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st310;
	goto st0;
tr1153:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st311;
tr1154:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st311;
tr1242:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st311;
tr1243:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st311;
tr1244:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st311;
tr1145:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st311;
tr1146:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st311;
tr1147:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st311;
st311:
	if ( ++p == pe )
		goto _test_eof311;
case 311:
#line 14838 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st311;
		case 32: goto st311;
		case 40: goto tr1153;
		case 41: goto tr1154;
		case 522: goto st331;
		case 525: goto st332;
		case 571: goto st333;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1155;
	goto st0;
tr1155:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st312;
tr1245:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st312;
st312:
	if ( ++p == pe )
		goto _test_eof312;
case 312:
#line 14886 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1159;
		case 32: goto tr1159;
		case 40: goto tr1160;
		case 41: goto tr1161;
		case 522: goto tr1163;
		case 525: goto tr1164;
		case 571: goto tr1165;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st312;
	goto st0;
tr1169:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st313;
tr1170:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st313;
tr1233:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st313;
tr1235:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st313;
tr1236:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st313;
tr1159:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st313;
tr1160:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st313;
tr1161:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st313;
st313:
	if ( ++p == pe )
		goto _test_eof313;
case 313:
#line 14970 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st313;
		case 32: goto st313;
		case 34: goto tr1168;
		case 40: goto tr1169;
		case 41: goto tr1170;
		case 64: goto tr1171;
		case 92: goto tr1172;
		case 522: goto st328;
		case 525: goto st329;
		case 571: goto st330;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1166;
		} else
			goto tr1166;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1166;
		} else if ( _widec >= 37 )
			goto tr1166;
	} else
		goto tr1166;
	goto st0;
tr1166:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
tr1190:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st314;
tr1232:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st314;
st314:
	if ( ++p == pe )
		goto _test_eof314;
case 314:
#line 15037 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1177;
		case 32: goto tr1177;
		case 40: goto tr1178;
		case 41: goto tr1179;
		case 92: goto st315;
		case 266: goto tr1181;
		case 269: goto tr1182;
		case 315: goto tr1183;
		case 522: goto tr1184;
		case 525: goto tr1185;
		case 571: goto tr1186;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st314;
		} else
			goto st314;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st314;
		} else if ( _widec >= 35 )
			goto st314;
	} else
		goto st314;
	goto st0;
tr1172:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st315;
tr1194:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st315;
tr1238:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st315;
st315:
	if ( ++p == pe )
		goto _test_eof315;
case 315:
#line 15105 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st316;
		case 13: goto st317;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st318;
	goto st314;
st316:
	if ( ++p == pe )
		goto _test_eof316;
case 316:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1191;
		case 32: goto tr1191;
		case 40: goto tr1192;
		case 41: goto tr1193;
		case 92: goto tr1194;
		case 266: goto tr1195;
		case 269: goto tr1196;
		case 315: goto tr1197;
		case 522: goto tr1198;
		case 525: goto tr1199;
		case 571: goto tr1200;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1190;
		} else
			goto tr1190;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1190;
		} else if ( _widec >= 35 )
			goto tr1190;
	} else
		goto tr1190;
	goto st0;
st317:
	if ( ++p == pe )
		goto _test_eof317;
case 317:
	if ( (*p) == 10 )
		goto st316;
	goto st0;
st318:
	if ( ++p == pe )
		goto _test_eof318;
case 318:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st319;
	goto st0;
st319:
	if ( ++p == pe )
		goto _test_eof319;
case 319:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st314;
	goto st0;
tr1168:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st320;
tr1207:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st320;
tr1234:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st320;
st320:
	if ( ++p == pe )
		goto _test_eof320;
case 320:
#line 15208 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st321;
		case 13: goto st322;
		case 34: goto st323;
		case 92: goto st324;
	}
	goto st320;
tr1208:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st321;
st321:
	if ( ++p == pe )
		goto _test_eof321;
case 321:
#line 15224 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1208;
		case 13: goto tr1209;
		case 34: goto tr1210;
		case 92: goto tr1211;
	}
	goto tr1207;
tr1209:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st322;
st322:
	if ( ++p == pe )
		goto _test_eof322;
case 322:
#line 15240 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st321;
	goto st0;
tr1210:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st323;
st323:
	if ( ++p == pe )
		goto _test_eof323;
case 323:
#line 15252 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1212;
		case 32: goto tr1212;
		case 40: goto tr1213;
		case 41: goto tr1214;
		case 266: goto tr1215;
		case 269: goto tr1216;
		case 315: goto tr1217;
		case 522: goto tr1218;
		case 525: goto tr1219;
		case 571: goto tr1220;
	}
	goto st0;
tr1211:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st324;
st324:
	if ( ++p == pe )
		goto _test_eof324;
case 324:
#line 15295 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st321;
		case 13: goto st322;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st325;
	goto st320;
st325:
	if ( ++p == pe )
		goto _test_eof325;
case 325:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st326;
	goto st0;
st326:
	if ( ++p == pe )
		goto _test_eof326;
case 326:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st320;
	goto st0;
tr1171:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st327;
tr1237:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st327;
st327:
	if ( ++p == pe )
		goto _test_eof327;
case 327:
#line 15334 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1223;
		case 32: goto tr1223;
		case 40: goto tr1224;
		case 41: goto tr1225;
		case 266: goto tr1226;
		case 269: goto tr1227;
		case 315: goto tr1228;
		case 522: goto tr1229;
		case 525: goto tr1230;
		case 571: goto tr1231;
	}
	goto st0;
tr1239:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st328;
tr1163:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st328;
st328:
	if ( ++p == pe )
		goto _test_eof328;
case 328:
#line 15383 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1233;
		case 32: goto tr1233;
		case 34: goto tr1234;
		case 40: goto tr1235;
		case 41: goto tr1236;
		case 64: goto tr1237;
		case 92: goto tr1238;
		case 522: goto tr1239;
		case 525: goto tr1240;
		case 571: goto tr1241;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1232;
		} else
			goto tr1232;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1232;
		} else if ( _widec >= 37 )
			goto tr1232;
	} else
		goto tr1232;
	goto st0;
tr1240:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st329;
tr1164:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st329;
st329:
	if ( ++p == pe )
		goto _test_eof329;
case 329:
#line 15446 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st328;
	goto st0;
tr1241:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st330;
tr1165:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 518 "./zscan_rfc1035.rl"
	{ z->uv_3 = z->uval; }
	goto st330;
st330:
	if ( ++p == pe )
		goto _test_eof330;
case 330:
#line 15471 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st328;
		case 525: goto st329;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st330;
	goto st0;
tr1246:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st331;
tr1149:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st331;
st331:
	if ( ++p == pe )
		goto _test_eof331;
case 331:
#line 15528 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1242;
		case 32: goto tr1242;
		case 40: goto tr1243;
		case 41: goto tr1244;
		case 522: goto tr1246;
		case 525: goto tr1247;
		case 571: goto tr1248;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1245;
	goto st0;
tr1247:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st332;
tr1150:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st332;
st332:
	if ( ++p == pe )
		goto _test_eof332;
case 332:
#line 15576 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st331;
	goto st0;
tr1248:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st333;
tr1151:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
	goto st333;
st333:
	if ( ++p == pe )
		goto _test_eof333;
case 333:
#line 15601 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st331;
		case 525: goto st332;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st333;
	goto st0;
tr1135:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st334;
tr1253:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st334;
st334:
	if ( ++p == pe )
		goto _test_eof334;
case 334:
#line 15658 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1249;
		case 32: goto tr1249;
		case 40: goto tr1250;
		case 41: goto tr1251;
		case 522: goto tr1253;
		case 525: goto tr1254;
		case 571: goto tr1255;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1252;
	goto st0;
tr1136:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st335;
tr1254:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st335;
st335:
	if ( ++p == pe )
		goto _test_eof335;
case 335:
#line 15706 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st334;
	goto st0;
tr1137:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st336;
tr1255:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st336;
st336:
	if ( ++p == pe )
		goto _test_eof336;
case 336:
#line 15731 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st334;
		case 525: goto st335;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st336;
	goto st0;
tr1260:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st337;
st337:
	if ( ++p == pe )
		goto _test_eof337;
case 337:
#line 15782 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1256;
		case 32: goto tr1256;
		case 40: goto tr1257;
		case 41: goto tr1258;
		case 522: goto tr1260;
		case 525: goto tr1261;
		case 571: goto tr1262;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1259;
	goto st0;
tr1261:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st338;
st338:
	if ( ++p == pe )
		goto _test_eof338;
case 338:
#line 15824 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st337;
	goto st0;
tr1262:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st339;
st339:
	if ( ++p == pe )
		goto _test_eof339;
case 339:
#line 15843 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st337;
		case 525: goto st338;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st339;
	goto st0;
tr21:
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st340;
tr1453:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st340;
tr1510:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 514 "./zscan_rfc1035.rl"
	{ z->ttl  = z->def_ttl; }
	goto st340;
st340:
	if ( ++p == pe )
		goto _test_eof340;
case 340:
#line 15904 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 88: goto st341;
		case 89: goto st368;
		case 120: goto st341;
		case 121: goto st368;
	}
	goto st0;
st341:
	if ( ++p == pe )
		goto _test_eof341;
case 341:
	switch( (*p) ) {
		case 84: goto st342;
		case 116: goto st342;
	}
	goto st0;
st342:
	if ( ++p == pe )
		goto _test_eof342;
case 342:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st343;
		case 32: goto st343;
		case 40: goto tr1267;
		case 41: goto tr1268;
		case 522: goto st365;
		case 525: goto st366;
		case 571: goto st367;
	}
	goto st0;
tr1267:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st343;
tr1268:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st343;
st343:
	if ( ++p == pe )
		goto _test_eof343;
case 343:
#line 15968 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1273;
		case 32: goto tr1273;
		case 34: goto tr1274;
		case 40: goto tr1275;
		case 41: goto tr1276;
		case 92: goto tr1277;
		case 522: goto tr1278;
		case 525: goto tr1279;
		case 571: goto tr1280;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1272;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1272;
		} else if ( _widec >= 14 )
			goto tr1272;
	} else
		goto tr1272;
	goto st0;
tr1293:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st344;
tr1330:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st344;
tr1342:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st344;
tr1272:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st344;
tr1357:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st344;
tr1315:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st344;
st344:
	if ( ++p == pe )
		goto _test_eof344;
case 344:
#line 16051 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1282;
		case 32: goto tr1282;
		case 34: goto tr1283;
		case 40: goto tr1284;
		case 41: goto tr1285;
		case 92: goto st350;
		case 266: goto tr1287;
		case 269: goto tr1288;
		case 315: goto tr1289;
		case 522: goto tr1290;
		case 525: goto tr1291;
		case 571: goto tr1292;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto st344;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st344;
		} else if ( _widec >= 14 )
			goto st344;
	} else
		goto st344;
	goto st0;
tr1296:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st345;
tr1297:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st345;
tr1343:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st345;
tr1345:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st345;
tr1346:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st345;
tr1282:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st345;
tr1284:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st345;
tr1285:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st345;
tr1316:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st345;
tr1318:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st345;
tr1319:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st345;
tr1331:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st345;
tr1333:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st345;
tr1334:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st345;
st345:
	if ( ++p == pe )
		goto _test_eof345;
case 345:
#line 16181 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st345;
		case 32: goto st345;
		case 34: goto tr1295;
		case 40: goto tr1296;
		case 41: goto tr1297;
		case 92: goto tr1298;
		case 266: goto tr1299;
		case 269: goto tr1300;
		case 315: goto tr1301;
		case 522: goto st352;
		case 525: goto st353;
		case 571: goto st354;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1293;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1293;
		} else if ( _widec >= 14 )
			goto tr1293;
	} else
		goto tr1293;
	goto st0;
tr1295:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1310:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st346;
tr1344:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1274:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1359:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1283:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1317:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
tr1332:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st346;
st346:
	if ( ++p == pe )
		goto _test_eof346;
case 346:
#line 16281 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st347;
		case 13: goto st348;
		case 34: goto st349;
		case 92: goto st358;
	}
	goto st346;
tr1311:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st347;
st347:
	if ( ++p == pe )
		goto _test_eof347;
case 347:
#line 16297 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1311;
		case 13: goto tr1312;
		case 34: goto tr1313;
		case 92: goto tr1314;
	}
	goto tr1310;
tr1312:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st348;
st348:
	if ( ++p == pe )
		goto _test_eof348;
case 348:
#line 16313 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st347;
	goto st0;
tr1313:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st349;
st349:
	if ( ++p == pe )
		goto _test_eof349;
case 349:
#line 16325 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1316;
		case 32: goto tr1316;
		case 34: goto tr1317;
		case 40: goto tr1318;
		case 41: goto tr1319;
		case 92: goto tr1320;
		case 266: goto tr1321;
		case 269: goto tr1322;
		case 315: goto tr1323;
		case 522: goto tr1324;
		case 525: goto tr1325;
		case 571: goto tr1326;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1315;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1315;
		} else if ( _widec >= 14 )
			goto tr1315;
	} else
		goto tr1315;
	goto st0;
tr1298:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st350;
tr1335:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st350;
tr1347:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st350;
tr1277:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st350;
tr1362:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st350;
tr1320:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st350;
st350:
	if ( ++p == pe )
		goto _test_eof350;
case 350:
#line 16411 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st351;
		case 13: goto st355;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st356;
	goto st344;
st351:
	if ( ++p == pe )
		goto _test_eof351;
case 351:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1331;
		case 32: goto tr1331;
		case 34: goto tr1332;
		case 40: goto tr1333;
		case 41: goto tr1334;
		case 92: goto tr1335;
		case 266: goto tr1336;
		case 269: goto tr1337;
		case 315: goto tr1338;
		case 522: goto tr1339;
		case 525: goto tr1340;
		case 571: goto tr1341;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1330;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1330;
		} else if ( _widec >= 14 )
			goto tr1330;
	} else
		goto tr1330;
	goto st0;
tr1351:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st352;
tr1290:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st352;
tr1324:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st352;
tr1339:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st352;
st352:
	if ( ++p == pe )
		goto _test_eof352;
case 352:
#line 16495 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1343;
		case 32: goto tr1343;
		case 34: goto tr1344;
		case 40: goto tr1345;
		case 41: goto tr1346;
		case 92: goto tr1347;
		case 266: goto tr1348;
		case 269: goto tr1349;
		case 315: goto tr1350;
		case 522: goto tr1351;
		case 525: goto tr1352;
		case 571: goto tr1353;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1342;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1342;
		} else if ( _widec >= 14 )
			goto tr1342;
	} else
		goto tr1342;
	goto st0;
tr1352:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st353;
tr1291:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st353;
tr1325:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st353;
tr1340:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st353;
st353:
	if ( ++p == pe )
		goto _test_eof353;
case 353:
#line 16565 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st352;
	goto st0;
tr1353:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st354;
tr1292:
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st354;
tr1326:
#line 502 "./zscan_rfc1035.rl"
	{ z->tstart++; text_add_tok(z, p - z->tstart - 1, true); }
	goto st354;
tr1341:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 501 "./zscan_rfc1035.rl"
	{ text_add_tok(z, p - z->tstart, true); }
	goto st354;
st354:
	if ( ++p == pe )
		goto _test_eof354;
case 354:
#line 16598 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st352;
		case 525: goto st353;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st354;
	goto st0;
st355:
	if ( ++p == pe )
		goto _test_eof355;
case 355:
	if ( (*p) == 10 )
		goto st351;
	goto st0;
st356:
	if ( ++p == pe )
		goto _test_eof356;
case 356:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st357;
	goto st0;
st357:
	if ( ++p == pe )
		goto _test_eof357;
case 357:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st344;
	goto st0;
tr1314:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st358;
st358:
	if ( ++p == pe )
		goto _test_eof358;
case 358:
#line 16670 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st347;
		case 13: goto st348;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st359;
	goto st346;
st359:
	if ( ++p == pe )
		goto _test_eof359;
case 359:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st360;
	goto st0;
st360:
	if ( ++p == pe )
		goto _test_eof360;
case 360:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st346;
	goto st0;
tr1273:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
tr1275:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
tr1276:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
tr1358:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
tr1360:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
tr1361:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st361;
st361:
	if ( ++p == pe )
		goto _test_eof361;
case 361:
#line 16737 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1273;
		case 32: goto tr1273;
		case 34: goto tr1274;
		case 40: goto tr1275;
		case 41: goto tr1276;
		case 92: goto tr1277;
		case 266: goto tr1299;
		case 269: goto tr1300;
		case 315: goto tr1301;
		case 522: goto tr1278;
		case 525: goto tr1279;
		case 571: goto tr1280;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1272;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1272;
		} else if ( _widec >= 14 )
			goto tr1272;
	} else
		goto tr1272;
	goto st0;
tr1278:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st362;
tr1363:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st362;
st362:
	if ( ++p == pe )
		goto _test_eof362;
case 362:
#line 16799 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1358;
		case 32: goto tr1358;
		case 34: goto tr1359;
		case 40: goto tr1360;
		case 41: goto tr1361;
		case 92: goto tr1362;
		case 266: goto tr1348;
		case 269: goto tr1349;
		case 315: goto tr1350;
		case 522: goto tr1363;
		case 525: goto tr1364;
		case 571: goto tr1365;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1357;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1357;
		} else if ( _widec >= 14 )
			goto tr1357;
	} else
		goto tr1357;
	goto st0;
tr1279:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st363;
tr1364:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st363;
st363:
	if ( ++p == pe )
		goto _test_eof363;
case 363:
#line 16861 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st362;
	goto st0;
tr1280:
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st364;
tr1365:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 500 "./zscan_rfc1035.rl"
	{ text_start(z); }
	goto st364;
st364:
	if ( ++p == pe )
		goto _test_eof364;
case 364:
#line 16886 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st362;
		case 525: goto st363;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st364;
	goto st0;
st365:
	if ( ++p == pe )
		goto _test_eof365;
case 365:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1358;
		case 32: goto tr1358;
		case 34: goto tr1359;
		case 40: goto tr1360;
		case 41: goto tr1361;
		case 92: goto tr1362;
		case 522: goto tr1363;
		case 525: goto tr1364;
		case 571: goto tr1365;
	}
	if ( _widec < 11 ) {
		if ( _widec <= 8 )
			goto tr1357;
	} else if ( _widec > 12 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1357;
		} else if ( _widec >= 14 )
			goto tr1357;
	} else
		goto tr1357;
	goto st0;
st366:
	if ( ++p == pe )
		goto _test_eof366;
case 366:
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st365;
	goto st0;
st367:
	if ( ++p == pe )
		goto _test_eof367;
case 367:
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st365;
		case 525: goto st366;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st367;
	goto st0;
st368:
	if ( ++p == pe )
		goto _test_eof368;
case 368:
	switch( (*p) ) {
		case 80: goto st369;
		case 112: goto st369;
	}
	goto st0;
st369:
	if ( ++p == pe )
		goto _test_eof369;
case 369:
	switch( (*p) ) {
		case 69: goto st370;
		case 101: goto st370;
	}
	goto st0;
st370:
	if ( ++p == pe )
		goto _test_eof370;
case 370:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1371;
	goto st0;
tr1371:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st371;
st371:
	if ( ++p == pe )
		goto _test_eof371;
case 371:
#line 17070 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1372;
		case 32: goto tr1372;
		case 40: goto tr1373;
		case 41: goto tr1374;
		case 522: goto tr1376;
		case 525: goto tr1377;
		case 571: goto tr1378;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st371;
	goto st0;
tr1380:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st372;
tr1381:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st372;
tr1372:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st372;
tr1436:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st372;
tr1437:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st372;
tr1438:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st372;
tr1373:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st372;
tr1374:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st372;
st372:
	if ( ++p == pe )
		goto _test_eof372;
case 372:
#line 17154 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st372;
		case 32: goto st372;
		case 40: goto tr1380;
		case 41: goto tr1381;
		case 92: goto st373;
		case 522: goto st386;
		case 525: goto st387;
		case 571: goto st388;
	}
	goto st0;
tr1439:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st373;
st373:
	if ( ++p == pe )
		goto _test_eof373;
case 373:
#line 17195 "./zscan_rfc1035.c"
	if ( (*p) == 35 )
		goto st374;
	goto st0;
st374:
	if ( ++p == pe )
		goto _test_eof374;
case 374:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st375;
		case 32: goto st375;
		case 40: goto tr1388;
		case 41: goto tr1389;
		case 522: goto st383;
		case 525: goto st384;
		case 571: goto st385;
	}
	goto st0;
tr1388:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st375;
tr1389:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st375;
tr1429:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st375;
tr1430:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st375;
tr1431:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st375;
st375:
	if ( ++p == pe )
		goto _test_eof375;
case 375:
#line 17262 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st375;
		case 32: goto st375;
		case 40: goto tr1388;
		case 41: goto tr1389;
		case 522: goto st383;
		case 525: goto st384;
		case 571: goto st385;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1393;
	goto st0;
tr1393:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st376;
tr1432:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st376;
st376:
	if ( ++p == pe )
		goto _test_eof376;
case 376:
#line 17310 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1394;
		case 32: goto tr1394;
		case 40: goto tr1395;
		case 41: goto tr1396;
		case 266: goto tr1398;
		case 269: goto tr1399;
		case 315: goto tr1400;
		case 522: goto tr1401;
		case 525: goto tr1402;
		case 571: goto tr1403;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st376;
	goto st0;
tr1405:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st377;
tr1406:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st377;
tr1422:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st377;
tr1423:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st377;
tr1424:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st377;
tr1394:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st377;
tr1395:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st377;
tr1396:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st377;
tr1412:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st377;
tr1413:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st377;
tr1414:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st377;
st377:
	if ( ++p == pe )
		goto _test_eof377;
case 377:
#line 17413 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st377;
		case 32: goto st377;
		case 40: goto tr1405;
		case 41: goto tr1406;
		case 522: goto st380;
		case 525: goto st381;
		case 571: goto st382;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1407;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1407;
	} else
		goto tr1407;
	goto st0;
tr1407:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st378;
tr1425:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st378;
tr1415:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st378;
st378:
	if ( ++p == pe )
		goto _test_eof378;
case 378:
#line 17473 "./zscan_rfc1035.c"
	if ( (*p) < 65 ) {
		if ( 48 <= (*p) && (*p) <= 57 )
			goto st379;
	} else if ( (*p) > 70 ) {
		if ( 97 <= (*p) && (*p) <= 102 )
			goto st379;
	} else
		goto st379;
	goto st0;
st379:
	if ( ++p == pe )
		goto _test_eof379;
case 379:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1412;
		case 32: goto tr1412;
		case 40: goto tr1413;
		case 41: goto tr1414;
		case 266: goto tr1416;
		case 269: goto tr1417;
		case 315: goto tr1418;
		case 522: goto tr1419;
		case 525: goto tr1420;
		case 571: goto tr1421;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1415;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1415;
	} else
		goto tr1415;
	goto st0;
tr1426:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st380;
tr1401:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st380;
tr1419:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st380;
st380:
	if ( ++p == pe )
		goto _test_eof380;
case 380:
#line 17547 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1422;
		case 32: goto tr1422;
		case 40: goto tr1423;
		case 41: goto tr1424;
		case 522: goto tr1426;
		case 525: goto tr1427;
		case 571: goto tr1428;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto tr1425;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1425;
	} else
		goto tr1425;
	goto st0;
tr1427:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st381;
tr1402:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st381;
tr1420:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st381;
st381:
	if ( ++p == pe )
		goto _test_eof381;
case 381:
#line 17605 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st380;
	goto st0;
tr1428:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st382;
tr1403:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 541 "./zscan_rfc1035.rl"
	{ rfc3597_data_setup(z); }
	goto st382;
tr1421:
#line 542 "./zscan_rfc1035.rl"
	{ rfc3597_octet(z); }
	goto st382;
st382:
	if ( ++p == pe )
		goto _test_eof382;
case 382:
#line 17634 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st380;
		case 525: goto st381;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st382;
	goto st0;
tr1433:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st383;
st383:
	if ( ++p == pe )
		goto _test_eof383;
case 383:
#line 17685 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1429;
		case 32: goto tr1429;
		case 40: goto tr1430;
		case 41: goto tr1431;
		case 522: goto tr1433;
		case 525: goto tr1434;
		case 571: goto tr1435;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1432;
	goto st0;
tr1434:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st384;
st384:
	if ( ++p == pe )
		goto _test_eof384;
case 384:
#line 17727 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st383;
	goto st0;
tr1435:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st385;
st385:
	if ( ++p == pe )
		goto _test_eof385;
case 385:
#line 17746 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st383;
		case 525: goto st384;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st385;
	goto st0;
tr1376:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st386;
tr1440:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st386;
st386:
	if ( ++p == pe )
		goto _test_eof386;
case 386:
#line 17803 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1436;
		case 32: goto tr1436;
		case 40: goto tr1437;
		case 41: goto tr1438;
		case 92: goto tr1439;
		case 522: goto tr1440;
		case 525: goto tr1441;
		case 571: goto tr1442;
	}
	goto st0;
tr1377:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st387;
tr1441:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st387;
st387:
	if ( ++p == pe )
		goto _test_eof387;
case 387:
#line 17850 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st386;
	goto st0;
tr1378:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st388;
tr1442:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st388;
st388:
	if ( ++p == pe )
		goto _test_eof388;
case 388:
#line 17875 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st386;
		case 525: goto st387;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st388;
	goto st0;
tr302:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st389;
tr1454:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st389;
tr1488:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st389;
st389:
	if ( ++p == pe )
		goto _test_eof389;
case 389:
#line 17944 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1443;
		case 32: goto tr1443;
		case 40: goto tr1444;
		case 41: goto tr1445;
		case 65: goto tr1446;
		case 67: goto tr1447;
		case 68: goto tr1448;
		case 77: goto tr1449;
		case 78: goto tr1450;
		case 80: goto tr1451;
		case 83: goto tr1452;
		case 84: goto tr1453;
		case 97: goto tr1446;
		case 99: goto tr1447;
		case 100: goto tr1448;
		case 109: goto tr1449;
		case 110: goto tr1450;
		case 112: goto tr1451;
		case 115: goto tr1452;
		case 116: goto tr1453;
		case 522: goto tr1454;
		case 525: goto tr1455;
		case 571: goto tr1456;
	}
	goto st0;
tr303:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st390;
tr1455:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st390;
tr1489:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st390;
st390:
	if ( ++p == pe )
		goto _test_eof390;
case 390:
#line 18018 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st389;
	goto st0;
tr304:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st391;
tr1456:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st391;
tr1490:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st391;
st391:
	if ( ++p == pe )
		goto _test_eof391;
case 391:
#line 18055 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st389;
		case 525: goto st390;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st391;
	goto st0;
tr299:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st392;
tr1487:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st392;
st392:
	if ( ++p == pe )
		goto _test_eof392;
case 392:
#line 18112 "./zscan_rfc1035.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1457;
	goto st0;
tr1457:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st393;
st393:
	if ( ++p == pe )
		goto _test_eof393;
case 393:
#line 18124 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1458;
		case 32: goto tr1458;
		case 40: goto tr1459;
		case 41: goto tr1460;
		case 68: goto tr1462;
		case 72: goto tr1462;
		case 77: goto tr1462;
		case 87: goto tr1462;
		case 100: goto tr1462;
		case 104: goto tr1462;
		case 109: goto tr1462;
		case 119: goto tr1462;
		case 522: goto tr1463;
		case 525: goto tr1464;
		case 571: goto tr1465;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st393;
	goto st0;
tr1467:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st394;
tr1468:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st394;
tr1472:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st394;
tr1473:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st394;
tr1474:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st394;
tr1458:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st394;
tr1459:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st394;
tr1460:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st394;
tr1478:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st394;
tr1479:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st394;
tr1480:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st394;
st394:
	if ( ++p == pe )
		goto _test_eof394;
case 394:
#line 18244 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st394;
		case 32: goto st394;
		case 40: goto tr1467;
		case 41: goto tr1468;
		case 68: goto st44;
		case 100: goto st44;
		case 522: goto st395;
		case 525: goto st396;
		case 571: goto st397;
	}
	goto st0;
tr1475:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st395;
tr1463:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st395;
tr1481:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st395;
st395:
	if ( ++p == pe )
		goto _test_eof395;
case 395:
#line 18300 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1472;
		case 32: goto tr1472;
		case 40: goto tr1473;
		case 41: goto tr1474;
		case 68: goto tr1448;
		case 100: goto tr1448;
		case 522: goto tr1475;
		case 525: goto tr1476;
		case 571: goto tr1477;
	}
	goto st0;
tr1476:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st396;
tr1464:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st396;
tr1482:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st396;
st396:
	if ( ++p == pe )
		goto _test_eof396;
case 396:
#line 18356 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st395;
	goto st0;
tr1477:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st397;
tr1465:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st397;
tr1483:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st397;
st397:
	if ( ++p == pe )
		goto _test_eof397;
case 397:
#line 18389 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st395;
		case 525: goto st396;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st397;
	goto st0;
tr1462:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st398;
st398:
	if ( ++p == pe )
		goto _test_eof398;
case 398:
#line 18442 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1478;
		case 32: goto tr1478;
		case 40: goto tr1479;
		case 41: goto tr1480;
		case 522: goto tr1481;
		case 525: goto tr1482;
		case 571: goto tr1483;
	}
	goto st0;
tr301:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st399;
st399:
	if ( ++p == pe )
		goto _test_eof399;
case 399:
#line 18484 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1484;
		case 32: goto tr1484;
		case 40: goto tr1485;
		case 41: goto tr1486;
		case 47: goto tr1487;
		case 522: goto tr1488;
		case 525: goto tr1489;
		case 571: goto tr1490;
	}
	goto st0;
tr1495:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st400;
st400:
	if ( ++p == pe )
		goto _test_eof400;
case 400:
#line 18525 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1491;
		case 32: goto tr1491;
		case 40: goto tr1492;
		case 41: goto tr1493;
		case 65: goto tr1446;
		case 67: goto tr1447;
		case 68: goto tr1448;
		case 77: goto tr1449;
		case 78: goto tr1450;
		case 80: goto tr1451;
		case 83: goto tr1452;
		case 84: goto tr1453;
		case 97: goto tr1446;
		case 99: goto tr1447;
		case 100: goto tr1448;
		case 109: goto tr1449;
		case 110: goto tr1450;
		case 112: goto tr1451;
		case 115: goto tr1452;
		case 116: goto tr1453;
		case 522: goto tr1495;
		case 525: goto tr1496;
		case 571: goto tr1497;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1494;
	goto st0;
tr1496:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st401;
st401:
	if ( ++p == pe )
		goto _test_eof401;
case 401:
#line 18583 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st400;
	goto st0;
tr1497:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st402;
st402:
	if ( ++p == pe )
		goto _test_eof402;
case 402:
#line 18602 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st400;
		case 525: goto st401;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st402;
	goto st0;
tr1511:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st403;
st403:
	if ( ++p == pe )
		goto _test_eof403;
case 403:
#line 18653 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1498;
		case 32: goto tr1498;
		case 40: goto tr1499;
		case 41: goto tr1500;
		case 65: goto tr1502;
		case 67: goto tr1503;
		case 68: goto tr1504;
		case 73: goto tr1505;
		case 77: goto tr1506;
		case 78: goto tr1507;
		case 80: goto tr1508;
		case 83: goto tr1509;
		case 84: goto tr1510;
		case 97: goto tr1502;
		case 99: goto tr1503;
		case 100: goto tr1504;
		case 105: goto tr1505;
		case 109: goto tr1506;
		case 110: goto tr1507;
		case 112: goto tr1508;
		case 115: goto tr1509;
		case 116: goto tr1510;
		case 266: goto tr133;
		case 269: goto tr134;
		case 315: goto tr135;
		case 522: goto tr1511;
		case 525: goto tr1512;
		case 571: goto tr1513;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1501;
	goto st0;
tr1512:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st404;
st404:
	if ( ++p == pe )
		goto _test_eof404;
case 404:
#line 18716 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st403;
	goto st0;
tr1513:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st405;
st405:
	if ( ++p == pe )
		goto _test_eof405;
case 405:
#line 18735 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st403;
		case 525: goto st404;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st405;
	goto st0;
tr1836:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st406;
tr1519:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st406;
tr1841:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st406;
st406:
	if ( ++p == pe )
		goto _test_eof406;
case 406:
#line 18796 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st407;
		case 13: goto st408;
		case 34: goto st409;
		case 92: goto st413;
	}
	goto st406;
tr1520:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st407;
st407:
	if ( ++p == pe )
		goto _test_eof407;
case 407:
#line 18812 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1520;
		case 13: goto tr1521;
		case 34: goto tr1522;
		case 92: goto tr1523;
	}
	goto tr1519;
tr1521:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st408;
st408:
	if ( ++p == pe )
		goto _test_eof408;
case 408:
#line 18828 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st407;
	goto st0;
tr1522:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st409;
st409:
	if ( ++p == pe )
		goto _test_eof409;
case 409:
#line 18840 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1524;
		case 32: goto tr1524;
		case 40: goto tr1525;
		case 41: goto tr1526;
		case 522: goto tr1527;
		case 525: goto tr1528;
		case 571: goto tr1529;
	}
	goto st0;
tr6:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st410;
tr1533:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st410;
tr1527:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st410;
tr1722:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st410;
tr1733:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st410;
st410:
	if ( ++p == pe )
		goto _test_eof410;
case 410:
#line 18898 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1530;
		case 32: goto tr1530;
		case 40: goto tr1531;
		case 41: goto tr1532;
		case 65: goto tr1502;
		case 67: goto tr1503;
		case 68: goto tr1504;
		case 73: goto tr1505;
		case 77: goto tr1506;
		case 78: goto tr1507;
		case 80: goto tr1508;
		case 83: goto tr1509;
		case 84: goto tr1510;
		case 97: goto tr1502;
		case 99: goto tr1503;
		case 100: goto tr1504;
		case 105: goto tr1505;
		case 109: goto tr1506;
		case 110: goto tr1507;
		case 112: goto tr1508;
		case 115: goto tr1509;
		case 116: goto tr1510;
		case 522: goto tr1533;
		case 525: goto tr1534;
		case 571: goto tr1535;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1501;
	goto st0;
tr7:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st411;
tr1534:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st411;
tr1528:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st411;
tr1723:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st411;
tr1734:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st411;
st411:
	if ( ++p == pe )
		goto _test_eof411;
case 411:
#line 18976 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st410;
	goto st0;
tr8:
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st412;
tr1535:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st412;
tr1529:
#line 488 "./zscan_rfc1035.rl"
	{ z->tstart++; dname_set(z, z->lhs_dname, p - z->tstart - 1, true); }
	goto st412;
tr1724:
#line 486 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, 0, true); }
	goto st412;
tr1735:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 487 "./zscan_rfc1035.rl"
	{ dname_set(z, z->lhs_dname, p - z->tstart, true); }
	goto st412;
st412:
	if ( ++p == pe )
		goto _test_eof412;
case 412:
#line 19013 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st410;
		case 525: goto st411;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st412;
	goto st0;
tr1523:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st413;
st413:
	if ( ++p == pe )
		goto _test_eof413;
case 413:
#line 19064 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st407;
		case 13: goto st408;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st414;
	goto st406;
st414:
	if ( ++p == pe )
		goto _test_eof414;
case 414:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st415;
	goto st0;
st415:
	if ( ++p == pe )
		goto _test_eof415;
case 415:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st406;
	goto st0;
tr1842:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st416;
st416:
	if ( ++p == pe )
		goto _test_eof416;
case 416:
#line 19097 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 65: goto st417;
		case 79: goto st441;
		case 84: goto st465;
		case 97: goto st417;
		case 111: goto st441;
		case 116: goto st465;
	}
	goto st0;
st417:
	if ( ++p == pe )
		goto _test_eof417;
case 417:
	switch( (*p) ) {
		case 68: goto st418;
		case 100: goto st418;
	}
	goto st0;
st418:
	if ( ++p == pe )
		goto _test_eof418;
case 418:
	switch( (*p) ) {
		case 68: goto st419;
		case 100: goto st419;
	}
	goto st0;
st419:
	if ( ++p == pe )
		goto _test_eof419;
case 419:
	switch( (*p) ) {
		case 82: goto st420;
		case 114: goto st420;
	}
	goto st0;
st420:
	if ( ++p == pe )
		goto _test_eof420;
case 420:
	if ( (*p) == 95 )
		goto st421;
	goto st0;
st421:
	if ( ++p == pe )
		goto _test_eof421;
case 421:
	switch( (*p) ) {
		case 76: goto st422;
		case 108: goto st422;
	}
	goto st0;
st422:
	if ( ++p == pe )
		goto _test_eof422;
case 422:
	switch( (*p) ) {
		case 73: goto st423;
		case 105: goto st423;
	}
	goto st0;
st423:
	if ( ++p == pe )
		goto _test_eof423;
case 423:
	switch( (*p) ) {
		case 77: goto st424;
		case 109: goto st424;
	}
	goto st0;
st424:
	if ( ++p == pe )
		goto _test_eof424;
case 424:
	switch( (*p) ) {
		case 73: goto st425;
		case 105: goto st425;
	}
	goto st0;
st425:
	if ( ++p == pe )
		goto _test_eof425;
case 425:
	switch( (*p) ) {
		case 84: goto st426;
		case 116: goto st426;
	}
	goto st0;
st426:
	if ( ++p == pe )
		goto _test_eof426;
case 426:
	if ( (*p) == 95 )
		goto st427;
	goto st0;
st427:
	if ( ++p == pe )
		goto _test_eof427;
case 427:
	switch( (*p) ) {
		case 86: goto st428;
		case 118: goto st428;
	}
	goto st0;
st428:
	if ( ++p == pe )
		goto _test_eof428;
case 428:
	switch( (*p) ) {
		case 52: goto st429;
		case 54: goto st435;
	}
	goto st0;
st429:
	if ( ++p == pe )
		goto _test_eof429;
case 429:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st430;
		case 32: goto st430;
		case 40: goto tr1555;
		case 41: goto tr1556;
		case 522: goto st432;
		case 525: goto st433;
		case 571: goto st434;
	}
	goto st0;
tr1555:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1556:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
tr1571:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st430;
tr1572:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st430;
tr1573:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st430;
st430:
	if ( ++p == pe )
		goto _test_eof430;
case 430:
#line 19274 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st430;
		case 32: goto st430;
		case 40: goto tr1555;
		case 41: goto tr1556;
		case 522: goto st432;
		case 525: goto st433;
		case 571: goto st434;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1560;
	goto st0;
tr1560:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st431;
tr1574:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st431;
st431:
	if ( ++p == pe )
		goto _test_eof431;
case 431:
#line 19322 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1561;
		case 32: goto tr1561;
		case 40: goto tr1562;
		case 41: goto tr1563;
		case 266: goto tr1565;
		case 269: goto tr1566;
		case 315: goto tr1567;
		case 522: goto tr1568;
		case 525: goto tr1569;
		case 571: goto tr1570;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st431;
	goto st0;
tr1575:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st432;
st432:
	if ( ++p == pe )
		goto _test_eof432;
case 432:
#line 19367 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1571;
		case 32: goto tr1571;
		case 40: goto tr1572;
		case 41: goto tr1573;
		case 522: goto tr1575;
		case 525: goto tr1576;
		case 571: goto tr1577;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1574;
	goto st0;
tr1576:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st433;
st433:
	if ( ++p == pe )
		goto _test_eof433;
case 433:
#line 19409 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st432;
	goto st0;
tr1577:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st434;
st434:
	if ( ++p == pe )
		goto _test_eof434;
case 434:
#line 19428 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st432;
		case 525: goto st433;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st434;
	goto st0;
st435:
	if ( ++p == pe )
		goto _test_eof435;
case 435:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st436;
		case 32: goto st436;
		case 40: goto tr1579;
		case 41: goto tr1580;
		case 522: goto st438;
		case 525: goto st439;
		case 571: goto st440;
	}
	goto st0;
tr1579:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st436;
tr1580:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st436;
tr1595:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st436;
tr1596:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st436;
tr1597:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st436;
st436:
	if ( ++p == pe )
		goto _test_eof436;
case 436:
#line 19534 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st436;
		case 32: goto st436;
		case 40: goto tr1579;
		case 41: goto tr1580;
		case 522: goto st438;
		case 525: goto st439;
		case 571: goto st440;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1584;
	goto st0;
tr1584:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st437;
tr1598:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st437;
st437:
	if ( ++p == pe )
		goto _test_eof437;
case 437:
#line 19582 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1585;
		case 32: goto tr1585;
		case 40: goto tr1586;
		case 41: goto tr1587;
		case 266: goto tr1589;
		case 269: goto tr1590;
		case 315: goto tr1591;
		case 522: goto tr1592;
		case 525: goto tr1593;
		case 571: goto tr1594;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st437;
	goto st0;
tr1599:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st438;
st438:
	if ( ++p == pe )
		goto _test_eof438;
case 438:
#line 19627 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1595;
		case 32: goto tr1595;
		case 40: goto tr1596;
		case 41: goto tr1597;
		case 522: goto tr1599;
		case 525: goto tr1600;
		case 571: goto tr1601;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1598;
	goto st0;
tr1600:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st439;
st439:
	if ( ++p == pe )
		goto _test_eof439;
case 439:
#line 19669 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st438;
	goto st0;
tr1601:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st440;
st440:
	if ( ++p == pe )
		goto _test_eof440;
case 440:
#line 19688 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st438;
		case 525: goto st439;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st440;
	goto st0;
st441:
	if ( ++p == pe )
		goto _test_eof441;
case 441:
	switch( (*p) ) {
		case 82: goto st442;
		case 114: goto st442;
	}
	goto st0;
st442:
	if ( ++p == pe )
		goto _test_eof442;
case 442:
	switch( (*p) ) {
		case 73: goto st443;
		case 105: goto st443;
	}
	goto st0;
st443:
	if ( ++p == pe )
		goto _test_eof443;
case 443:
	switch( (*p) ) {
		case 71: goto st444;
		case 103: goto st444;
	}
	goto st0;
st444:
	if ( ++p == pe )
		goto _test_eof444;
case 444:
	switch( (*p) ) {
		case 73: goto st445;
		case 105: goto st445;
	}
	goto st0;
st445:
	if ( ++p == pe )
		goto _test_eof445;
case 445:
	switch( (*p) ) {
		case 78: goto st446;
		case 110: goto st446;
	}
	goto st0;
st446:
	if ( ++p == pe )
		goto _test_eof446;
case 446:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st447;
		case 32: goto st447;
		case 40: goto tr1608;
		case 41: goto tr1609;
		case 522: goto st462;
		case 525: goto st463;
		case 571: goto st464;
	}
	goto st0;
tr1608:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st447;
tr1609:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st447;
tr1674:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st447;
tr1676:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st447;
tr1677:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st447;
st447:
	if ( ++p == pe )
		goto _test_eof447;
case 447:
#line 19839 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st447;
		case 32: goto st447;
		case 34: goto tr1614;
		case 40: goto tr1608;
		case 41: goto tr1609;
		case 64: goto tr1615;
		case 92: goto tr1616;
		case 522: goto st462;
		case 525: goto st463;
		case 571: goto st464;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1613;
		} else
			goto tr1613;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1613;
		} else if ( _widec >= 37 )
			goto tr1613;
	} else
		goto tr1613;
	goto st0;
tr1613:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st448;
tr1631:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st448;
tr1673:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st448;
st448:
	if ( ++p == pe )
		goto _test_eof448;
case 448:
#line 19906 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1618;
		case 32: goto tr1618;
		case 40: goto tr1619;
		case 41: goto tr1620;
		case 92: goto st449;
		case 266: goto tr1622;
		case 269: goto tr1623;
		case 315: goto tr1624;
		case 522: goto tr1625;
		case 525: goto tr1626;
		case 571: goto tr1627;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto st448;
		} else
			goto st448;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto st448;
		} else if ( _widec >= 35 )
			goto st448;
	} else
		goto st448;
	goto st0;
tr1616:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st449;
tr1635:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st449;
tr1679:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st449;
st449:
	if ( ++p == pe )
		goto _test_eof449;
case 449:
#line 19974 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st450;
		case 13: goto st451;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st452;
	goto st448;
st450:
	if ( ++p == pe )
		goto _test_eof450;
case 450:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1632;
		case 32: goto tr1632;
		case 40: goto tr1633;
		case 41: goto tr1634;
		case 92: goto tr1635;
		case 266: goto tr1636;
		case 269: goto tr1637;
		case 315: goto tr1638;
		case 522: goto tr1639;
		case 525: goto tr1640;
		case 571: goto tr1641;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1631;
		} else
			goto tr1631;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1631;
		} else if ( _widec >= 35 )
			goto tr1631;
	} else
		goto tr1631;
	goto st0;
st451:
	if ( ++p == pe )
		goto _test_eof451;
case 451:
	if ( (*p) == 10 )
		goto st450;
	goto st0;
st452:
	if ( ++p == pe )
		goto _test_eof452;
case 452:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st453;
	goto st0;
st453:
	if ( ++p == pe )
		goto _test_eof453;
case 453:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st448;
	goto st0;
tr1614:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st454;
tr1648:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st454;
tr1675:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st454;
st454:
	if ( ++p == pe )
		goto _test_eof454;
case 454:
#line 20077 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st455;
		case 13: goto st456;
		case 34: goto st457;
		case 92: goto st458;
	}
	goto st454;
tr1649:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st455;
st455:
	if ( ++p == pe )
		goto _test_eof455;
case 455:
#line 20093 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto tr1649;
		case 13: goto tr1650;
		case 34: goto tr1651;
		case 92: goto tr1652;
	}
	goto tr1648;
tr1650:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st456;
st456:
	if ( ++p == pe )
		goto _test_eof456;
case 456:
#line 20109 "./zscan_rfc1035.c"
	if ( (*p) == 10 )
		goto st455;
	goto st0;
tr1651:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st457;
st457:
	if ( ++p == pe )
		goto _test_eof457;
case 457:
#line 20121 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1653;
		case 32: goto tr1653;
		case 40: goto tr1654;
		case 41: goto tr1655;
		case 266: goto tr1656;
		case 269: goto tr1657;
		case 315: goto tr1658;
		case 522: goto tr1659;
		case 525: goto tr1660;
		case 571: goto tr1661;
	}
	goto st0;
tr1652:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st458;
st458:
	if ( ++p == pe )
		goto _test_eof458;
case 458:
#line 20164 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st455;
		case 13: goto st456;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st459;
	goto st454;
st459:
	if ( ++p == pe )
		goto _test_eof459;
case 459:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st460;
	goto st0;
st460:
	if ( ++p == pe )
		goto _test_eof460;
case 460:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st454;
	goto st0;
tr1615:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st461;
tr1678:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st461;
st461:
	if ( ++p == pe )
		goto _test_eof461;
case 461:
#line 20203 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1664;
		case 32: goto tr1664;
		case 40: goto tr1665;
		case 41: goto tr1666;
		case 266: goto tr1667;
		case 269: goto tr1668;
		case 315: goto tr1669;
		case 522: goto tr1670;
		case 525: goto tr1671;
		case 571: goto tr1672;
	}
	goto st0;
tr1680:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st462;
st462:
	if ( ++p == pe )
		goto _test_eof462;
case 462:
#line 20246 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1674;
		case 32: goto tr1674;
		case 34: goto tr1675;
		case 40: goto tr1676;
		case 41: goto tr1677;
		case 64: goto tr1678;
		case 92: goto tr1679;
		case 522: goto tr1680;
		case 525: goto tr1681;
		case 571: goto tr1682;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1673;
		} else
			goto tr1673;
	} else if ( _widec > 35 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1673;
		} else if ( _widec >= 37 )
			goto tr1673;
	} else
		goto tr1673;
	goto st0;
tr1681:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st463;
st463:
	if ( ++p == pe )
		goto _test_eof463;
case 463:
#line 20303 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st462;
	goto st0;
tr1682:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st464;
st464:
	if ( ++p == pe )
		goto _test_eof464;
case 464:
#line 20322 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st462;
		case 525: goto st463;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st464;
	goto st0;
st465:
	if ( ++p == pe )
		goto _test_eof465;
case 465:
	switch( (*p) ) {
		case 84: goto st466;
		case 116: goto st466;
	}
	goto st0;
st466:
	if ( ++p == pe )
		goto _test_eof466;
case 466:
	switch( (*p) ) {
		case 76: goto st467;
		case 108: goto st467;
	}
	goto st0;
st467:
	if ( ++p == pe )
		goto _test_eof467;
case 467:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st468;
		case 32: goto st468;
		case 40: goto tr1686;
		case 41: goto tr1687;
		case 522: goto st471;
		case 525: goto st472;
		case 571: goto st473;
	}
	goto st0;
tr1686:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st468;
tr1687:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st468;
tr1712:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st468;
tr1713:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st468;
tr1714:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st468;
st468:
	if ( ++p == pe )
		goto _test_eof468;
case 468:
#line 20446 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st468;
		case 32: goto st468;
		case 40: goto tr1686;
		case 41: goto tr1687;
		case 522: goto st471;
		case 525: goto st472;
		case 571: goto st473;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1691;
	goto st0;
tr1691:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st469;
tr1715:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st469;
st469:
	if ( ++p == pe )
		goto _test_eof469;
case 469:
#line 20494 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1692;
		case 32: goto tr1692;
		case 40: goto tr1693;
		case 41: goto tr1694;
		case 68: goto tr1696;
		case 72: goto tr1696;
		case 77: goto tr1696;
		case 87: goto tr1696;
		case 100: goto tr1696;
		case 104: goto tr1696;
		case 109: goto tr1696;
		case 119: goto tr1696;
		case 266: goto tr1697;
		case 269: goto tr1698;
		case 315: goto tr1699;
		case 522: goto tr1700;
		case 525: goto tr1701;
		case 571: goto tr1702;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st469;
	goto st0;
tr1696:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st470;
st470:
	if ( ++p == pe )
		goto _test_eof470;
case 470:
#line 20549 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1703;
		case 32: goto tr1703;
		case 40: goto tr1704;
		case 41: goto tr1705;
		case 266: goto tr1706;
		case 269: goto tr1707;
		case 315: goto tr1708;
		case 522: goto tr1709;
		case 525: goto tr1710;
		case 571: goto tr1711;
	}
	goto st0;
tr1716:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st471;
st471:
	if ( ++p == pe )
		goto _test_eof471;
case 471:
#line 20592 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1712;
		case 32: goto tr1712;
		case 40: goto tr1713;
		case 41: goto tr1714;
		case 522: goto tr1716;
		case 525: goto tr1717;
		case 571: goto tr1718;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1715;
	goto st0;
tr1717:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st472;
st472:
	if ( ++p == pe )
		goto _test_eof472;
case 472:
#line 20634 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st471;
	goto st0;
tr1718:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st473;
st473:
	if ( ++p == pe )
		goto _test_eof473;
case 473:
#line 20653 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st471;
		case 525: goto st472;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st473;
	goto st0;
tr1838:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st474;
tr1843:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st474;
st474:
	if ( ++p == pe )
		goto _test_eof474;
case 474:
#line 20710 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1719;
		case 32: goto tr1719;
		case 40: goto tr1720;
		case 41: goto tr1721;
		case 522: goto tr1722;
		case 525: goto tr1723;
		case 571: goto tr1724;
	}
	goto st0;
tr1839:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st475;
tr1732:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st475;
tr1844:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st475;
st475:
	if ( ++p == pe )
		goto _test_eof475;
case 475:
#line 20760 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 10: goto st476;
		case 13: goto st477;
	}
	if ( (*p) > 49 ) {
		if ( 50 <= (*p) && (*p) <= 57 )
			goto st0;
	} else if ( (*p) >= 48 )
		goto st478;
	goto st1;
st476:
	if ( ++p == pe )
		goto _test_eof476;
case 476:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1729;
		case 32: goto tr1729;
		case 40: goto tr1730;
		case 41: goto tr1731;
		case 92: goto tr1732;
		case 522: goto tr1733;
		case 525: goto tr1734;
		case 571: goto tr1735;
	}
	if ( _widec < 14 ) {
		if ( _widec > 8 ) {
			if ( 11 <= _widec && _widec <= 12 )
				goto tr1728;
		} else
			goto tr1728;
	} else if ( _widec > 33 ) {
		if ( _widec > 58 ) {
			if ( 60 <= _widec )
				goto tr1728;
		} else if ( _widec >= 35 )
			goto tr1728;
	} else
		goto tr1728;
	goto st0;
st477:
	if ( ++p == pe )
		goto _test_eof477;
case 477:
	if ( (*p) == 10 )
		goto st476;
	goto st0;
st478:
	if ( ++p == pe )
		goto _test_eof478;
case 478:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st479;
	goto st0;
st479:
	if ( ++p == pe )
		goto _test_eof479;
case 479:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st1;
	goto st0;
st480:
	if ( ++p == pe )
		goto _test_eof480;
case 480:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr66;
		case 32: goto tr66;
		case 40: goto tr67;
		case 41: goto tr68;
		case 266: goto tr70;
		case 269: goto tr71;
		case 315: goto tr72;
		case 522: goto tr73;
		case 525: goto tr74;
		case 571: goto tr75;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st481;
	goto st0;
st481:
	if ( ++p == pe )
		goto _test_eof481;
case 481:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr66;
		case 32: goto tr66;
		case 40: goto tr67;
		case 41: goto tr68;
		case 266: goto tr70;
		case 269: goto tr71;
		case 315: goto tr72;
		case 522: goto tr73;
		case 525: goto tr74;
		case 571: goto tr75;
	}
	goto st0;
st482:
	if ( ++p == pe )
		goto _test_eof482;
case 482:
	if ( (*p) == 46 )
		goto st12;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st483;
	goto st0;
st483:
	if ( ++p == pe )
		goto _test_eof483;
case 483:
	if ( (*p) == 46 )
		goto st12;
	goto st0;
st484:
	if ( ++p == pe )
		goto _test_eof484;
case 484:
	if ( (*p) == 46 )
		goto st10;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st485;
	goto st0;
st485:
	if ( ++p == pe )
		goto _test_eof485;
case 485:
	if ( (*p) == 46 )
		goto st10;
	goto st0;
st486:
	if ( ++p == pe )
		goto _test_eof486;
case 486:
	if ( (*p) == 46 )
		goto st8;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st487;
	goto st0;
st487:
	if ( ++p == pe )
		goto _test_eof487;
case 487:
	if ( (*p) == 46 )
		goto st8;
	goto st0;
tr1745:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st488;
st488:
	if ( ++p == pe )
		goto _test_eof488;
case 488:
#line 20976 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1741;
		case 32: goto tr1741;
		case 40: goto tr1742;
		case 41: goto tr1743;
		case 522: goto tr1745;
		case 525: goto tr1746;
		case 571: goto tr1747;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto tr1744;
	goto st0;
tr1746:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st489;
st489:
	if ( ++p == pe )
		goto _test_eof489;
case 489:
#line 21018 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st488;
	goto st0;
tr1747:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st490;
st490:
	if ( ++p == pe )
		goto _test_eof490;
case 490:
#line 21037 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st488;
		case 525: goto st489;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st490;
	goto st0;
st491:
	if ( ++p == pe )
		goto _test_eof491;
case 491:
	switch( (*p) ) {
		case 65: goto st492;
		case 97: goto st492;
	}
	goto st0;
st492:
	if ( ++p == pe )
		goto _test_eof492;
case 492:
	switch( (*p) ) {
		case 65: goto st493;
		case 97: goto st493;
	}
	goto st0;
st493:
	if ( ++p == pe )
		goto _test_eof493;
case 493:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st494;
		case 32: goto st494;
		case 40: goto tr1751;
		case 41: goto tr1752;
		case 522: goto st512;
		case 525: goto st513;
		case 571: goto st514;
	}
	goto st0;
tr1751:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st494;
tr1752:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st494;
tr1783:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st494;
tr1784:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st494;
tr1785:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st494;
st494:
	if ( ++p == pe )
		goto _test_eof494;
case 494:
#line 21161 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st494;
		case 32: goto st494;
		case 40: goto tr1751;
		case 41: goto tr1752;
		case 522: goto st512;
		case 525: goto st513;
		case 571: goto st514;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 58 )
			goto tr1756;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1756;
	} else
		goto tr1756;
	goto st0;
tr1756:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st495;
tr1786:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st495;
st495:
	if ( ++p == pe )
		goto _test_eof495;
case 495:
#line 21215 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 58: goto st496;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st495;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st495;
	} else
		goto st495;
	goto st0;
st496:
	if ( ++p == pe )
		goto _test_eof496;
case 496:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 58: goto st496;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st497;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st495;
	} else
		goto st495;
	goto st0;
st497:
	if ( ++p == pe )
		goto _test_eof497;
case 497:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 46: goto st498;
		case 58: goto st496;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st510;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st495;
	} else
		goto st495;
	goto st0;
st498:
	if ( ++p == pe )
		goto _test_eof498;
case 498:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st499;
	goto st0;
st499:
	if ( ++p == pe )
		goto _test_eof499;
case 499:
	if ( (*p) == 46 )
		goto st500;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st508;
	goto st0;
st500:
	if ( ++p == pe )
		goto _test_eof500;
case 500:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st501;
	goto st0;
st501:
	if ( ++p == pe )
		goto _test_eof501;
case 501:
	if ( (*p) == 46 )
		goto st502;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st506;
	goto st0;
st502:
	if ( ++p == pe )
		goto _test_eof502;
case 502:
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st503;
	goto st0;
st503:
	if ( ++p == pe )
		goto _test_eof503;
case 503:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st504;
	goto st0;
st504:
	if ( ++p == pe )
		goto _test_eof504;
case 504:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st505;
	goto st0;
st505:
	if ( ++p == pe )
		goto _test_eof505;
case 505:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	goto st0;
st506:
	if ( ++p == pe )
		goto _test_eof506;
case 506:
	if ( (*p) == 46 )
		goto st502;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st507;
	goto st0;
st507:
	if ( ++p == pe )
		goto _test_eof507;
case 507:
	if ( (*p) == 46 )
		goto st502;
	goto st0;
st508:
	if ( ++p == pe )
		goto _test_eof508;
case 508:
	if ( (*p) == 46 )
		goto st500;
	if ( 48 <= (*p) && (*p) <= 57 )
		goto st509;
	goto st0;
st509:
	if ( ++p == pe )
		goto _test_eof509;
case 509:
	if ( (*p) == 46 )
		goto st500;
	goto st0;
st510:
	if ( ++p == pe )
		goto _test_eof510;
case 510:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 46: goto st498;
		case 58: goto st496;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st511;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st495;
	} else
		goto st495;
	goto st0;
st511:
	if ( ++p == pe )
		goto _test_eof511;
case 511:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1757;
		case 32: goto tr1757;
		case 40: goto tr1758;
		case 41: goto tr1759;
		case 46: goto st498;
		case 58: goto st496;
		case 266: goto tr1762;
		case 269: goto tr1763;
		case 315: goto tr1764;
		case 522: goto tr1765;
		case 525: goto tr1766;
		case 571: goto tr1767;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 57 )
			goto st495;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto st495;
	} else
		goto st495;
	goto st0;
tr1787:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st512;
st512:
	if ( ++p == pe )
		goto _test_eof512;
case 512:
#line 21647 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1783;
		case 32: goto tr1783;
		case 40: goto tr1784;
		case 41: goto tr1785;
		case 522: goto tr1787;
		case 525: goto tr1788;
		case 571: goto tr1789;
	}
	if ( _widec < 65 ) {
		if ( 48 <= _widec && _widec <= 58 )
			goto tr1786;
	} else if ( _widec > 70 ) {
		if ( 97 <= _widec && _widec <= 102 )
			goto tr1786;
	} else
		goto tr1786;
	goto st0;
tr1788:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st513;
st513:
	if ( ++p == pe )
		goto _test_eof513;
case 513:
#line 21695 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st512;
	goto st0;
tr1789:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st514;
st514:
	if ( ++p == pe )
		goto _test_eof514;
case 514:
#line 21714 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st512;
		case 525: goto st513;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st514;
	goto st0;
tr1794:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st515;
st515:
	if ( ++p == pe )
		goto _test_eof515;
case 515:
#line 21765 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st516;
		case 110: goto st516;
	}
	goto st0;
st516:
	if ( ++p == pe )
		goto _test_eof516;
case 516:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st87;
		case 32: goto st87;
		case 40: goto tr306;
		case 41: goto tr307;
		case 522: goto st389;
		case 525: goto st390;
		case 571: goto st391;
	}
	goto st0;
tr31:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st517;
tr1795:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st517;
tr1832:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st517;
st517:
	if ( ++p == pe )
		goto _test_eof517;
case 517:
#line 21832 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1791;
		case 32: goto tr1791;
		case 40: goto tr1792;
		case 41: goto tr1793;
		case 65: goto tr1446;
		case 67: goto tr1447;
		case 68: goto tr1448;
		case 73: goto tr1794;
		case 77: goto tr1449;
		case 78: goto tr1450;
		case 80: goto tr1451;
		case 83: goto tr1452;
		case 84: goto tr1453;
		case 97: goto tr1446;
		case 99: goto tr1447;
		case 100: goto tr1448;
		case 105: goto tr1794;
		case 109: goto tr1449;
		case 110: goto tr1450;
		case 112: goto tr1451;
		case 115: goto tr1452;
		case 116: goto tr1453;
		case 522: goto tr1795;
		case 525: goto tr1796;
		case 571: goto tr1797;
	}
	goto st0;
tr32:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st518;
tr1796:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st518;
tr1833:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st518;
st518:
	if ( ++p == pe )
		goto _test_eof518;
case 518:
#line 21908 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st517;
	goto st0;
tr33:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st519;
tr1797:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st519;
tr1834:
#line 511 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uval; }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st519;
st519:
	if ( ++p == pe )
		goto _test_eof519;
case 519:
#line 21945 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st517;
		case 525: goto st518;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st519;
	goto st0;
tr28:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st520;
tr1831:
#line 516 "./zscan_rfc1035.rl"
	{ z->uv_1 = z->uval; }
	goto st520;
st520:
	if ( ++p == pe )
		goto _test_eof520;
case 520:
#line 22002 "./zscan_rfc1035.c"
	if ( 48 <= (*p) && (*p) <= 57 )
		goto tr1798;
	goto st0;
tr1798:
#line 483 "./zscan_rfc1035.rl"
	{ z->tstart = p; }
	goto st521;
st521:
	if ( ++p == pe )
		goto _test_eof521;
case 521:
#line 22014 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1799;
		case 32: goto tr1799;
		case 40: goto tr1800;
		case 41: goto tr1801;
		case 68: goto tr1803;
		case 72: goto tr1803;
		case 77: goto tr1803;
		case 87: goto tr1803;
		case 100: goto tr1803;
		case 104: goto tr1803;
		case 109: goto tr1803;
		case 119: goto tr1803;
		case 522: goto tr1804;
		case 525: goto tr1805;
		case 571: goto tr1806;
	}
	if ( 48 <= _widec && _widec <= 57 )
		goto st521;
	goto st0;
tr1808:
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st522;
tr1809:
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st522;
tr1815:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st522;
tr1816:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st522;
tr1817:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st522;
tr1799:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st522;
tr1800:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st522;
tr1801:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st522;
tr1822:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st522;
tr1823:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 543 "./zscan_rfc1035.rl"
	{ open_paren(z); }
	goto st522;
tr1824:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
#line 544 "./zscan_rfc1035.rl"
	{ close_paren(z); }
	goto st522;
st522:
	if ( ++p == pe )
		goto _test_eof522;
case 522:
#line 22134 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st522;
		case 32: goto st522;
		case 40: goto tr1808;
		case 41: goto tr1809;
		case 68: goto st44;
		case 73: goto st523;
		case 100: goto st44;
		case 105: goto st523;
		case 522: goto st525;
		case 525: goto st526;
		case 571: goto st527;
	}
	goto st0;
tr1818:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st523;
st523:
	if ( ++p == pe )
		goto _test_eof523;
case 523:
#line 22178 "./zscan_rfc1035.c"
	switch( (*p) ) {
		case 78: goto st524;
		case 110: goto st524;
	}
	goto st0;
st524:
	if ( ++p == pe )
		goto _test_eof524;
case 524:
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto st394;
		case 32: goto st394;
		case 40: goto tr1467;
		case 41: goto tr1468;
		case 522: goto st395;
		case 525: goto st396;
		case 571: goto st397;
	}
	goto st0;
tr1819:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st525;
tr1804:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st525;
tr1825:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st525;
st525:
	if ( ++p == pe )
		goto _test_eof525;
case 525:
#line 22241 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1815;
		case 32: goto tr1815;
		case 40: goto tr1816;
		case 41: goto tr1817;
		case 68: goto tr1448;
		case 73: goto tr1818;
		case 100: goto tr1448;
		case 105: goto tr1818;
		case 522: goto tr1819;
		case 525: goto tr1820;
		case 571: goto tr1821;
	}
	goto st0;
tr1820:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st526;
tr1805:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st526;
tr1826:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st526;
st526:
	if ( ++p == pe )
		goto _test_eof526;
case 526:
#line 22299 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( 10 <= (*p) && (*p) <= 10 ) {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	if ( _widec == 522 )
		goto st525;
	goto st0;
tr1821:
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	goto st527;
tr1806:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st527;
tr1827:
#line 517 "./zscan_rfc1035.rl"
	{ z->uv_2 = z->uval; }
#line 512 "./zscan_rfc1035.rl"
	{ z->ttl  = z->uv_1; z->ttl_min = z->uv_2 ? z->uv_2 : z->uv_1 >> 1; }
	goto st527;
st527:
	if ( ++p == pe )
		goto _test_eof527;
case 527:
#line 22332 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 11 ) {
		if ( (*p) > 9 ) {
			if ( 10 <= (*p) && (*p) <= 10 ) {
				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 12 ) {
		if ( (*p) > 13 ) {
			if ( 14 <= (*p) )
 {				_widec = (short)(128 + ((*p) - -128));
				if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
			}
		} else if ( (*p) >= 13 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 522: goto st525;
		case 525: goto st526;
	}
	if ( 384 <= _widec && _widec <= 639 )
		goto st527;
	goto st0;
tr1803:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st528;
st528:
	if ( ++p == pe )
		goto _test_eof528;
case 528:
#line 22385 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1822;
		case 32: goto tr1822;
		case 40: goto tr1823;
		case 41: goto tr1824;
		case 522: goto tr1825;
		case 525: goto tr1826;
		case 571: goto tr1827;
	}
	goto st0;
tr30:
#line 508 "./zscan_rfc1035.rl"
	{ set_uval(z); }
#line 509 "./zscan_rfc1035.rl"
	{ mult_uval(z, (*p)); }
	goto st529;
st529:
	if ( ++p == pe )
		goto _test_eof529;
case 529:
#line 22427 "./zscan_rfc1035.c"
	_widec = (*p);
	if ( (*p) < 13 ) {
		if ( 10 <= (*p) && (*p) <= 10 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else if ( (*p) > 13 ) {
		if ( 59 <= (*p) && (*p) <= 59 ) {
			_widec = (short)(128 + ((*p) - -128));
			if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
		}
	} else {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 545 "./zscan_rfc1035.rl"
 z->in_paren  ) _widec += 256;
	}
	switch( _widec ) {
		case 9: goto tr1828;
		case 32: goto tr1828;
		case 40: goto tr1829;
		case 41: goto tr1830;
		case 47: goto tr1831;
		case 522: goto tr1832;
		case 525: goto tr1833;
		case 571: goto tr1834;
	}
	goto st0;
	}
	_test_eof1: cs = 1; goto _test_eof; 
	_test_eof2: cs = 2; goto _test_eof; 
	_test_eof3: cs = 3; goto _test_eof; 
	_test_eof4: cs = 4; goto _test_eof; 
	_test_eof5: cs = 5; goto _test_eof; 
	_test_eof6: cs = 6; goto _test_eof; 
	_test_eof7: cs = 7; goto _test_eof; 
	_test_eof8: cs = 8; goto _test_eof; 
	_test_eof9: cs = 9; goto _test_eof; 
	_test_eof10: cs = 10; goto _test_eof; 
	_test_eof11: cs = 11; goto _test_eof; 
	_test_eof12: cs = 12; goto _test_eof; 
	_test_eof13: cs = 13; goto _test_eof; 
	_test_eof14: cs = 14; goto _test_eof; 
	_test_eof531: cs = 531; goto _test_eof; 
	_test_eof15: cs = 15; goto _test_eof; 
	_test_eof16: cs = 16; goto _test_eof; 
	_test_eof17: cs = 17; goto _test_eof; 
	_test_eof18: cs = 18; goto _test_eof; 
	_test_eof19: cs = 19; goto _test_eof; 
	_test_eof20: cs = 20; goto _test_eof; 
	_test_eof21: cs = 21; goto _test_eof; 
	_test_eof22: cs = 22; goto _test_eof; 
	_test_eof23: cs = 23; goto _test_eof; 
	_test_eof24: cs = 24; goto _test_eof; 
	_test_eof25: cs = 25; goto _test_eof; 
	_test_eof26: cs = 26; goto _test_eof; 
	_test_eof27: cs = 27; goto _test_eof; 
	_test_eof28: cs = 28; goto _test_eof; 
	_test_eof29: cs = 29; goto _test_eof; 
	_test_eof30: cs = 30; goto _test_eof; 
	_test_eof31: cs = 31; goto _test_eof; 
	_test_eof32: cs = 32; goto _test_eof; 
	_test_eof33: cs = 33; goto _test_eof; 
	_test_eof34: cs = 34; goto _test_eof; 
	_test_eof35: cs = 35; goto _test_eof; 
	_test_eof36: cs = 36; goto _test_eof; 
	_test_eof37: cs = 37; goto _test_eof; 
	_test_eof38: cs = 38; goto _test_eof; 
	_test_eof39: cs = 39; goto _test_eof; 
	_test_eof40: cs = 40; goto _test_eof; 
	_test_eof41: cs = 41; goto _test_eof; 
	_test_eof42: cs = 42; goto _test_eof; 
	_test_eof43: cs = 43; goto _test_eof; 
	_test_eof44: cs = 44; goto _test_eof; 
	_test_eof45: cs = 45; goto _test_eof; 
	_test_eof46: cs = 46; goto _test_eof; 
	_test_eof47: cs = 47; goto _test_eof; 
	_test_eof48: cs = 48; goto _test_eof; 
	_test_eof49: cs = 49; goto _test_eof; 
	_test_eof50: cs = 50; goto _test_eof; 
	_test_eof51: cs = 51; goto _test_eof; 
	_test_eof52: cs = 52; goto _test_eof; 
	_test_eof53: cs = 53; goto _test_eof; 
	_test_eof54: cs = 54; goto _test_eof; 
	_test_eof55: cs = 55; goto _test_eof; 
	_test_eof56: cs = 56; goto _test_eof; 
	_test_eof57: cs = 57; goto _test_eof; 
	_test_eof58: cs = 58; goto _test_eof; 
	_test_eof59: cs = 59; goto _test_eof; 
	_test_eof60: cs = 60; goto _test_eof; 
	_test_eof61: cs = 61; goto _test_eof; 
	_test_eof62: cs = 62; goto _test_eof; 
	_test_eof63: cs = 63; goto _test_eof; 
	_test_eof64: cs = 64; goto _test_eof; 
	_test_eof65: cs = 65; goto _test_eof; 
	_test_eof66: cs = 66; goto _test_eof; 
	_test_eof67: cs = 67; goto _test_eof; 
	_test_eof68: cs = 68; goto _test_eof; 
	_test_eof69: cs = 69; goto _test_eof; 
	_test_eof70: cs = 70; goto _test_eof; 
	_test_eof71: cs = 71; goto _test_eof; 
	_test_eof72: cs = 72; goto _test_eof; 
	_test_eof73: cs = 73; goto _test_eof; 
	_test_eof74: cs = 74; goto _test_eof; 
	_test_eof75: cs = 75; goto _test_eof; 
	_test_eof76: cs = 76; goto _test_eof; 
	_test_eof77: cs = 77; goto _test_eof; 
	_test_eof78: cs = 78; goto _test_eof; 
	_test_eof79: cs = 79; goto _test_eof; 
	_test_eof80: cs = 80; goto _test_eof; 
	_test_eof81: cs = 81; goto _test_eof; 
	_test_eof82: cs = 82; goto _test_eof; 
	_test_eof83: cs = 83; goto _test_eof; 
	_test_eof84: cs = 84; goto _test_eof; 
	_test_eof85: cs = 85; goto _test_eof; 
	_test_eof86: cs = 86; goto _test_eof; 
	_test_eof87: cs = 87; goto _test_eof; 
	_test_eof88: cs = 88; goto _test_eof; 
	_test_eof89: cs = 89; goto _test_eof; 
	_test_eof90: cs = 90; goto _test_eof; 
	_test_eof91: cs = 91; goto _test_eof; 
	_test_eof92: cs = 92; goto _test_eof; 
	_test_eof93: cs = 93; goto _test_eof; 
	_test_eof94: cs = 94; goto _test_eof; 
	_test_eof95: cs = 95; goto _test_eof; 
	_test_eof96: cs = 96; goto _test_eof; 
	_test_eof97: cs = 97; goto _test_eof; 
	_test_eof98: cs = 98; goto _test_eof; 
	_test_eof99: cs = 99; goto _test_eof; 
	_test_eof100: cs = 100; goto _test_eof; 
	_test_eof101: cs = 101; goto _test_eof; 
	_test_eof102: cs = 102; goto _test_eof; 
	_test_eof103: cs = 103; goto _test_eof; 
	_test_eof104: cs = 104; goto _test_eof; 
	_test_eof105: cs = 105; goto _test_eof; 
	_test_eof106: cs = 106; goto _test_eof; 
	_test_eof107: cs = 107; goto _test_eof; 
	_test_eof108: cs = 108; goto _test_eof; 
	_test_eof109: cs = 109; goto _test_eof; 
	_test_eof110: cs = 110; goto _test_eof; 
	_test_eof111: cs = 111; goto _test_eof; 
	_test_eof112: cs = 112; goto _test_eof; 
	_test_eof113: cs = 113; goto _test_eof; 
	_test_eof114: cs = 114; goto _test_eof; 
	_test_eof115: cs = 115; goto _test_eof; 
	_test_eof116: cs = 116; goto _test_eof; 
	_test_eof117: cs = 117; goto _test_eof; 
	_test_eof118: cs = 118; goto _test_eof; 
	_test_eof119: cs = 119; goto _test_eof; 
	_test_eof120: cs = 120; goto _test_eof; 
	_test_eof121: cs = 121; goto _test_eof; 
	_test_eof122: cs = 122; goto _test_eof; 
	_test_eof123: cs = 123; goto _test_eof; 
	_test_eof124: cs = 124; goto _test_eof; 
	_test_eof125: cs = 125; goto _test_eof; 
	_test_eof126: cs = 126; goto _test_eof; 
	_test_eof127: cs = 127; goto _test_eof; 
	_test_eof128: cs = 128; goto _test_eof; 
	_test_eof129: cs = 129; goto _test_eof; 
	_test_eof130: cs = 130; goto _test_eof; 
	_test_eof131: cs = 131; goto _test_eof; 
	_test_eof132: cs = 132; goto _test_eof; 
	_test_eof133: cs = 133; goto _test_eof; 
	_test_eof134: cs = 134; goto _test_eof; 
	_test_eof135: cs = 135; goto _test_eof; 
	_test_eof136: cs = 136; goto _test_eof; 
	_test_eof137: cs = 137; goto _test_eof; 
	_test_eof138: cs = 138; goto _test_eof; 
	_test_eof139: cs = 139; goto _test_eof; 
	_test_eof140: cs = 140; goto _test_eof; 
	_test_eof141: cs = 141; goto _test_eof; 
	_test_eof142: cs = 142; goto _test_eof; 
	_test_eof143: cs = 143; goto _test_eof; 
	_test_eof144: cs = 144; goto _test_eof; 
	_test_eof145: cs = 145; goto _test_eof; 
	_test_eof146: cs = 146; goto _test_eof; 
	_test_eof147: cs = 147; goto _test_eof; 
	_test_eof148: cs = 148; goto _test_eof; 
	_test_eof149: cs = 149; goto _test_eof; 
	_test_eof150: cs = 150; goto _test_eof; 
	_test_eof151: cs = 151; goto _test_eof; 
	_test_eof152: cs = 152; goto _test_eof; 
	_test_eof153: cs = 153; goto _test_eof; 
	_test_eof154: cs = 154; goto _test_eof; 
	_test_eof155: cs = 155; goto _test_eof; 
	_test_eof156: cs = 156; goto _test_eof; 
	_test_eof157: cs = 157; goto _test_eof; 
	_test_eof158: cs = 158; goto _test_eof; 
	_test_eof159: cs = 159; goto _test_eof; 
	_test_eof160: cs = 160; goto _test_eof; 
	_test_eof161: cs = 161; goto _test_eof; 
	_test_eof162: cs = 162; goto _test_eof; 
	_test_eof163: cs = 163; goto _test_eof; 
	_test_eof164: cs = 164; goto _test_eof; 
	_test_eof165: cs = 165; goto _test_eof; 
	_test_eof166: cs = 166; goto _test_eof; 
	_test_eof167: cs = 167; goto _test_eof; 
	_test_eof168: cs = 168; goto _test_eof; 
	_test_eof169: cs = 169; goto _test_eof; 
	_test_eof170: cs = 170; goto _test_eof; 
	_test_eof171: cs = 171; goto _test_eof; 
	_test_eof172: cs = 172; goto _test_eof; 
	_test_eof173: cs = 173; goto _test_eof; 
	_test_eof174: cs = 174; goto _test_eof; 
	_test_eof175: cs = 175; goto _test_eof; 
	_test_eof176: cs = 176; goto _test_eof; 
	_test_eof177: cs = 177; goto _test_eof; 
	_test_eof178: cs = 178; goto _test_eof; 
	_test_eof179: cs = 179; goto _test_eof; 
	_test_eof180: cs = 180; goto _test_eof; 
	_test_eof181: cs = 181; goto _test_eof; 
	_test_eof182: cs = 182; goto _test_eof; 
	_test_eof183: cs = 183; goto _test_eof; 
	_test_eof184: cs = 184; goto _test_eof; 
	_test_eof185: cs = 185; goto _test_eof; 
	_test_eof186: cs = 186; goto _test_eof; 
	_test_eof187: cs = 187; goto _test_eof; 
	_test_eof188: cs = 188; goto _test_eof; 
	_test_eof189: cs = 189; goto _test_eof; 
	_test_eof190: cs = 190; goto _test_eof; 
	_test_eof191: cs = 191; goto _test_eof; 
	_test_eof192: cs = 192; goto _test_eof; 
	_test_eof193: cs = 193; goto _test_eof; 
	_test_eof194: cs = 194; goto _test_eof; 
	_test_eof195: cs = 195; goto _test_eof; 
	_test_eof196: cs = 196; goto _test_eof; 
	_test_eof197: cs = 197; goto _test_eof; 
	_test_eof198: cs = 198; goto _test_eof; 
	_test_eof199: cs = 199; goto _test_eof; 
	_test_eof200: cs = 200; goto _test_eof; 
	_test_eof201: cs = 201; goto _test_eof; 
	_test_eof202: cs = 202; goto _test_eof; 
	_test_eof203: cs = 203; goto _test_eof; 
	_test_eof204: cs = 204; goto _test_eof; 
	_test_eof205: cs = 205; goto _test_eof; 
	_test_eof206: cs = 206; goto _test_eof; 
	_test_eof207: cs = 207; goto _test_eof; 
	_test_eof208: cs = 208; goto _test_eof; 
	_test_eof209: cs = 209; goto _test_eof; 
	_test_eof210: cs = 210; goto _test_eof; 
	_test_eof211: cs = 211; goto _test_eof; 
	_test_eof212: cs = 212; goto _test_eof; 
	_test_eof213: cs = 213; goto _test_eof; 
	_test_eof214: cs = 214; goto _test_eof; 
	_test_eof215: cs = 215; goto _test_eof; 
	_test_eof216: cs = 216; goto _test_eof; 
	_test_eof217: cs = 217; goto _test_eof; 
	_test_eof218: cs = 218; goto _test_eof; 
	_test_eof219: cs = 219; goto _test_eof; 
	_test_eof220: cs = 220; goto _test_eof; 
	_test_eof221: cs = 221; goto _test_eof; 
	_test_eof222: cs = 222; goto _test_eof; 
	_test_eof223: cs = 223; goto _test_eof; 
	_test_eof224: cs = 224; goto _test_eof; 
	_test_eof225: cs = 225; goto _test_eof; 
	_test_eof226: cs = 226; goto _test_eof; 
	_test_eof227: cs = 227; goto _test_eof; 
	_test_eof228: cs = 228; goto _test_eof; 
	_test_eof229: cs = 229; goto _test_eof; 
	_test_eof230: cs = 230; goto _test_eof; 
	_test_eof231: cs = 231; goto _test_eof; 
	_test_eof232: cs = 232; goto _test_eof; 
	_test_eof233: cs = 233; goto _test_eof; 
	_test_eof234: cs = 234; goto _test_eof; 
	_test_eof235: cs = 235; goto _test_eof; 
	_test_eof236: cs = 236; goto _test_eof; 
	_test_eof237: cs = 237; goto _test_eof; 
	_test_eof238: cs = 238; goto _test_eof; 
	_test_eof239: cs = 239; goto _test_eof; 
	_test_eof240: cs = 240; goto _test_eof; 
	_test_eof241: cs = 241; goto _test_eof; 
	_test_eof242: cs = 242; goto _test_eof; 
	_test_eof243: cs = 243; goto _test_eof; 
	_test_eof244: cs = 244; goto _test_eof; 
	_test_eof245: cs = 245; goto _test_eof; 
	_test_eof246: cs = 246; goto _test_eof; 
	_test_eof247: cs = 247; goto _test_eof; 
	_test_eof248: cs = 248; goto _test_eof; 
	_test_eof249: cs = 249; goto _test_eof; 
	_test_eof250: cs = 250; goto _test_eof; 
	_test_eof251: cs = 251; goto _test_eof; 
	_test_eof252: cs = 252; goto _test_eof; 
	_test_eof253: cs = 253; goto _test_eof; 
	_test_eof254: cs = 254; goto _test_eof; 
	_test_eof255: cs = 255; goto _test_eof; 
	_test_eof256: cs = 256; goto _test_eof; 
	_test_eof257: cs = 257; goto _test_eof; 
	_test_eof258: cs = 258; goto _test_eof; 
	_test_eof259: cs = 259; goto _test_eof; 
	_test_eof260: cs = 260; goto _test_eof; 
	_test_eof261: cs = 261; goto _test_eof; 
	_test_eof262: cs = 262; goto _test_eof; 
	_test_eof263: cs = 263; goto _test_eof; 
	_test_eof264: cs = 264; goto _test_eof; 
	_test_eof265: cs = 265; goto _test_eof; 
	_test_eof266: cs = 266; goto _test_eof; 
	_test_eof267: cs = 267; goto _test_eof; 
	_test_eof268: cs = 268; goto _test_eof; 
	_test_eof269: cs = 269; goto _test_eof; 
	_test_eof270: cs = 270; goto _test_eof; 
	_test_eof271: cs = 271; goto _test_eof; 
	_test_eof272: cs = 272; goto _test_eof; 
	_test_eof273: cs = 273; goto _test_eof; 
	_test_eof274: cs = 274; goto _test_eof; 
	_test_eof275: cs = 275; goto _test_eof; 
	_test_eof276: cs = 276; goto _test_eof; 
	_test_eof277: cs = 277; goto _test_eof; 
	_test_eof278: cs = 278; goto _test_eof; 
	_test_eof279: cs = 279; goto _test_eof; 
	_test_eof280: cs = 280; goto _test_eof; 
	_test_eof281: cs = 281; goto _test_eof; 
	_test_eof282: cs = 282; goto _test_eof; 
	_test_eof283: cs = 283; goto _test_eof; 
	_test_eof284: cs = 284; goto _test_eof; 
	_test_eof285: cs = 285; goto _test_eof; 
	_test_eof286: cs = 286; goto _test_eof; 
	_test_eof287: cs = 287; goto _test_eof; 
	_test_eof288: cs = 288; goto _test_eof; 
	_test_eof289: cs = 289; goto _test_eof; 
	_test_eof290: cs = 290; goto _test_eof; 
	_test_eof291: cs = 291; goto _test_eof; 
	_test_eof292: cs = 292; goto _test_eof; 
	_test_eof293: cs = 293; goto _test_eof; 
	_test_eof294: cs = 294; goto _test_eof; 
	_test_eof295: cs = 295; goto _test_eof; 
	_test_eof296: cs = 296; goto _test_eof; 
	_test_eof297: cs = 297; goto _test_eof; 
	_test_eof298: cs = 298; goto _test_eof; 
	_test_eof299: cs = 299; goto _test_eof; 
	_test_eof300: cs = 300; goto _test_eof; 
	_test_eof301: cs = 301; goto _test_eof; 
	_test_eof302: cs = 302; goto _test_eof; 
	_test_eof303: cs = 303; goto _test_eof; 
	_test_eof304: cs = 304; goto _test_eof; 
	_test_eof305: cs = 305; goto _test_eof; 
	_test_eof306: cs = 306; goto _test_eof; 
	_test_eof307: cs = 307; goto _test_eof; 
	_test_eof308: cs = 308; goto _test_eof; 
	_test_eof309: cs = 309; goto _test_eof; 
	_test_eof310: cs = 310; goto _test_eof; 
	_test_eof311: cs = 311; goto _test_eof; 
	_test_eof312: cs = 312; goto _test_eof; 
	_test_eof313: cs = 313; goto _test_eof; 
	_test_eof314: cs = 314; goto _test_eof; 
	_test_eof315: cs = 315; goto _test_eof; 
	_test_eof316: cs = 316; goto _test_eof; 
	_test_eof317: cs = 317; goto _test_eof; 
	_test_eof318: cs = 318; goto _test_eof; 
	_test_eof319: cs = 319; goto _test_eof; 
	_test_eof320: cs = 320; goto _test_eof; 
	_test_eof321: cs = 321; goto _test_eof; 
	_test_eof322: cs = 322; goto _test_eof; 
	_test_eof323: cs = 323; goto _test_eof; 
	_test_eof324: cs = 324; goto _test_eof; 
	_test_eof325: cs = 325; goto _test_eof; 
	_test_eof326: cs = 326; goto _test_eof; 
	_test_eof327: cs = 327; goto _test_eof; 
	_test_eof328: cs = 328; goto _test_eof; 
	_test_eof329: cs = 329; goto _test_eof; 
	_test_eof330: cs = 330; goto _test_eof; 
	_test_eof331: cs = 331; goto _test_eof; 
	_test_eof332: cs = 332; goto _test_eof; 
	_test_eof333: cs = 333; goto _test_eof; 
	_test_eof334: cs = 334; goto _test_eof; 
	_test_eof335: cs = 335; goto _test_eof; 
	_test_eof336: cs = 336; goto _test_eof; 
	_test_eof337: cs = 337; goto _test_eof; 
	_test_eof338: cs = 338; goto _test_eof; 
	_test_eof339: cs = 339; goto _test_eof; 
	_test_eof340: cs = 340; goto _test_eof; 
	_test_eof341: cs = 341; goto _test_eof; 
	_test_eof342: cs = 342; goto _test_eof; 
	_test_eof343: cs = 343; goto _test_eof; 
	_test_eof344: cs = 344; goto _test_eof; 
	_test_eof345: cs = 345; goto _test_eof; 
	_test_eof346: cs = 346; goto _test_eof; 
	_test_eof347: cs = 347; goto _test_eof; 
	_test_eof348: cs = 348; goto _test_eof; 
	_test_eof349: cs = 349; goto _test_eof; 
	_test_eof350: cs = 350; goto _test_eof; 
	_test_eof351: cs = 351; goto _test_eof; 
	_test_eof352: cs = 352; goto _test_eof; 
	_test_eof353: cs = 353; goto _test_eof; 
	_test_eof354: cs = 354; goto _test_eof; 
	_test_eof355: cs = 355; goto _test_eof; 
	_test_eof356: cs = 356; goto _test_eof; 
	_test_eof357: cs = 357; goto _test_eof; 
	_test_eof358: cs = 358; goto _test_eof; 
	_test_eof359: cs = 359; goto _test_eof; 
	_test_eof360: cs = 360; goto _test_eof; 
	_test_eof361: cs = 361; goto _test_eof; 
	_test_eof362: cs = 362; goto _test_eof; 
	_test_eof363: cs = 363; goto _test_eof; 
	_test_eof364: cs = 364; goto _test_eof; 
	_test_eof365: cs = 365; goto _test_eof; 
	_test_eof366: cs = 366; goto _test_eof; 
	_test_eof367: cs = 367; goto _test_eof; 
	_test_eof368: cs = 368; goto _test_eof; 
	_test_eof369: cs = 369; goto _test_eof; 
	_test_eof370: cs = 370; goto _test_eof; 
	_test_eof371: cs = 371; goto _test_eof; 
	_test_eof372: cs = 372; goto _test_eof; 
	_test_eof373: cs = 373; goto _test_eof; 
	_test_eof374: cs = 374; goto _test_eof; 
	_test_eof375: cs = 375; goto _test_eof; 
	_test_eof376: cs = 376; goto _test_eof; 
	_test_eof377: cs = 377; goto _test_eof; 
	_test_eof378: cs = 378; goto _test_eof; 
	_test_eof379: cs = 379; goto _test_eof; 
	_test_eof380: cs = 380; goto _test_eof; 
	_test_eof381: cs = 381; goto _test_eof; 
	_test_eof382: cs = 382; goto _test_eof; 
	_test_eof383: cs = 383; goto _test_eof; 
	_test_eof384: cs = 384; goto _test_eof; 
	_test_eof385: cs = 385; goto _test_eof; 
	_test_eof386: cs = 386; goto _test_eof; 
	_test_eof387: cs = 387; goto _test_eof; 
	_test_eof388: cs = 388; goto _test_eof; 
	_test_eof389: cs = 389; goto _test_eof; 
	_test_eof390: cs = 390; goto _test_eof; 
	_test_eof391: cs = 391; goto _test_eof; 
	_test_eof392: cs = 392; goto _test_eof; 
	_test_eof393: cs = 393; goto _test_eof; 
	_test_eof394: cs = 394; goto _test_eof; 
	_test_eof395: cs = 395; goto _test_eof; 
	_test_eof396: cs = 396; goto _test_eof; 
	_test_eof397: cs = 397; goto _test_eof; 
	_test_eof398: cs = 398; goto _test_eof; 
	_test_eof399: cs = 399; goto _test_eof; 
	_test_eof400: cs = 400; goto _test_eof; 
	_test_eof401: cs = 401; goto _test_eof; 
	_test_eof402: cs = 402; goto _test_eof; 
	_test_eof403: cs = 403; goto _test_eof; 
	_test_eof404: cs = 404; goto _test_eof; 
	_test_eof405: cs = 405; goto _test_eof; 
	_test_eof406: cs = 406; goto _test_eof; 
	_test_eof407: cs = 407; goto _test_eof; 
	_test_eof408: cs = 408; goto _test_eof; 
	_test_eof409: cs = 409; goto _test_eof; 
	_test_eof410: cs = 410; goto _test_eof; 
	_test_eof411: cs = 411; goto _test_eof; 
	_test_eof412: cs = 412; goto _test_eof; 
	_test_eof413: cs = 413; goto _test_eof; 
	_test_eof414: cs = 414; goto _test_eof; 
	_test_eof415: cs = 415; goto _test_eof; 
	_test_eof416: cs = 416; goto _test_eof; 
	_test_eof417: cs = 417; goto _test_eof; 
	_test_eof418: cs = 418; goto _test_eof; 
	_test_eof419: cs = 419; goto _test_eof; 
	_test_eof420: cs = 420; goto _test_eof; 
	_test_eof421: cs = 421; goto _test_eof; 
	_test_eof422: cs = 422; goto _test_eof; 
	_test_eof423: cs = 423; goto _test_eof; 
	_test_eof424: cs = 424; goto _test_eof; 
	_test_eof425: cs = 425; goto _test_eof; 
	_test_eof426: cs = 426; goto _test_eof; 
	_test_eof427: cs = 427; goto _test_eof; 
	_test_eof428: cs = 428; goto _test_eof; 
	_test_eof429: cs = 429; goto _test_eof; 
	_test_eof430: cs = 430; goto _test_eof; 
	_test_eof431: cs = 431; goto _test_eof; 
	_test_eof432: cs = 432; goto _test_eof; 
	_test_eof433: cs = 433; goto _test_eof; 
	_test_eof434: cs = 434; goto _test_eof; 
	_test_eof435: cs = 435; goto _test_eof; 
	_test_eof436: cs = 436; goto _test_eof; 
	_test_eof437: cs = 437; goto _test_eof; 
	_test_eof438: cs = 438; goto _test_eof; 
	_test_eof439: cs = 439; goto _test_eof; 
	_test_eof440: cs = 440; goto _test_eof; 
	_test_eof441: cs = 441; goto _test_eof; 
	_test_eof442: cs = 442; goto _test_eof; 
	_test_eof443: cs = 443; goto _test_eof; 
	_test_eof444: cs = 444; goto _test_eof; 
	_test_eof445: cs = 445; goto _test_eof; 
	_test_eof446: cs = 446; goto _test_eof; 
	_test_eof447: cs = 447; goto _test_eof; 
	_test_eof448: cs = 448; goto _test_eof; 
	_test_eof449: cs = 449; goto _test_eof; 
	_test_eof450: cs = 450; goto _test_eof; 
	_test_eof451: cs = 451; goto _test_eof; 
	_test_eof452: cs = 452; goto _test_eof; 
	_test_eof453: cs = 453; goto _test_eof; 
	_test_eof454: cs = 454; goto _test_eof; 
	_test_eof455: cs = 455; goto _test_eof; 
	_test_eof456: cs = 456; goto _test_eof; 
	_test_eof457: cs = 457; goto _test_eof; 
	_test_eof458: cs = 458; goto _test_eof; 
	_test_eof459: cs = 459; goto _test_eof; 
	_test_eof460: cs = 460; goto _test_eof; 
	_test_eof461: cs = 461; goto _test_eof; 
	_test_eof462: cs = 462; goto _test_eof; 
	_test_eof463: cs = 463; goto _test_eof; 
	_test_eof464: cs = 464; goto _test_eof; 
	_test_eof465: cs = 465; goto _test_eof; 
	_test_eof466: cs = 466; goto _test_eof; 
	_test_eof467: cs = 467; goto _test_eof; 
	_test_eof468: cs = 468; goto _test_eof; 
	_test_eof469: cs = 469; goto _test_eof; 
	_test_eof470: cs = 470; goto _test_eof; 
	_test_eof471: cs = 471; goto _test_eof; 
	_test_eof472: cs = 472; goto _test_eof; 
	_test_eof473: cs = 473; goto _test_eof; 
	_test_eof474: cs = 474; goto _test_eof; 
	_test_eof475: cs = 475; goto _test_eof; 
	_test_eof476: cs = 476; goto _test_eof; 
	_test_eof477: cs = 477; goto _test_eof; 
	_test_eof478: cs = 478; goto _test_eof; 
	_test_eof479: cs = 479; goto _test_eof; 
	_test_eof480: cs = 480; goto _test_eof; 
	_test_eof481: cs = 481; goto _test_eof; 
	_test_eof482: cs = 482; goto _test_eof; 
	_test_eof483: cs = 483; goto _test_eof; 
	_test_eof484: cs = 484; goto _test_eof; 
	_test_eof485: cs = 485; goto _test_eof; 
	_test_eof486: cs = 486; goto _test_eof; 
	_test_eof487: cs = 487; goto _test_eof; 
	_test_eof488: cs = 488; goto _test_eof; 
	_test_eof489: cs = 489; goto _test_eof; 
	_test_eof490: cs = 490; goto _test_eof; 
	_test_eof491: cs = 491; goto _test_eof; 
	_test_eof492: cs = 492; goto _test_eof; 
	_test_eof493: cs = 493; goto _test_eof; 
	_test_eof494: cs = 494; goto _test_eof; 
	_test_eof495: cs = 495; goto _test_eof; 
	_test_eof496: cs = 496; goto _test_eof; 
	_test_eof497: cs = 497; goto _test_eof; 
	_test_eof498: cs = 498; goto _test_eof; 
	_test_eof499: cs = 499; goto _test_eof; 
	_test_eof500: cs = 500; goto _test_eof; 
	_test_eof501: cs = 501; goto _test_eof; 
	_test_eof502: cs = 502; goto _test_eof; 
	_test_eof503: cs = 503; goto _test_eof; 
	_test_eof504: cs = 504; goto _test_eof; 
	_test_eof505: cs = 505; goto _test_eof; 
	_test_eof506: cs = 506; goto _test_eof; 
	_test_eof507: cs = 507; goto _test_eof; 
	_test_eof508: cs = 508; goto _test_eof; 
	_test_eof509: cs = 509; goto _test_eof; 
	_test_eof510: cs = 510; goto _test_eof; 
	_test_eof511: cs = 511; goto _test_eof; 
	_test_eof512: cs = 512; goto _test_eof; 
	_test_eof513: cs = 513; goto _test_eof; 
	_test_eof514: cs = 514; goto _test_eof; 
	_test_eof515: cs = 515; goto _test_eof; 
	_test_eof516: cs = 516; goto _test_eof; 
	_test_eof517: cs = 517; goto _test_eof; 
	_test_eof518: cs = 518; goto _test_eof; 
	_test_eof519: cs = 519; goto _test_eof; 
	_test_eof520: cs = 520; goto _test_eof; 
	_test_eof521: cs = 521; goto _test_eof; 
	_test_eof522: cs = 522; goto _test_eof; 
	_test_eof523: cs = 523; goto _test_eof; 
	_test_eof524: cs = 524; goto _test_eof; 
	_test_eof525: cs = 525; goto _test_eof; 
	_test_eof526: cs = 526; goto _test_eof; 
	_test_eof527: cs = 527; goto _test_eof; 
	_test_eof528: cs = 528; goto _test_eof; 
	_test_eof529: cs = 529; goto _test_eof; 

	_test_eof: {}
	if ( p == eof )
	{
	switch ( cs ) {
	case 531: 
#line 548 "./zscan_rfc1035.rl"
	{ z->lcount++; }
	break;
#line 23000 "./zscan_rfc1035.c"
	}
	}

	_out: {}
	}

#line 737 "./zscan_rfc1035.rl"

#endif // __clang_analyzer__

        if(cs == zone_error) {
            parse_error_noargs("General parse error");
        }
        else if(eof && cs < zone_first_final) {
            if(eof > buf && *(eof - 1) != '\n')
                parse_error_noargs("Trailing incomplete or unparseable record at end of file (missing newline at end of file?)");
            else
                parse_error_noargs("Trailing incomplete or unparseable record at end of file");
        }
    }
}

// This is broken out into a separate function (called via
//   function pointer to eliminate the possibility of
//   inlining on non-gcc compilers, I hope) to avoid issues with
//   setjmp and all of the local auto variables in zscan_rfc1035() below.
typedef bool (*sij_func_t)(zscan_t*,char*,const unsigned,const int);
F_NONNULL F_NOINLINE
static bool _scan_isolate_jmp(zscan_t* z, char* buf, const unsigned bufsize, const int fd) {
    dmn_assert(z); dmn_assert(buf); dmn_assert(fd >= 0);

    volatile bool failed = true;

    if(!sigsetjmp(z->jbuf, 0)) {
        scanner(z, buf, bufsize, fd);
        failed = false;
    }
    else {
        failed = true;
    }

    return failed;
}

bool zscan_rfc1035(zone_t* zone, const char* fn) {
    dmn_assert(zone);
    dmn_assert(zone->dname);
    dmn_assert(fn);
    log_debug("rfc1035: Scanning zone '%s'", logf_dname(zone->dname));

    const int fd = open(fn, O_RDONLY);
    if(fd < 0) {
        log_err("rfc1035: Cannot open file '%s' for reading: %s", fn, dmn_logf_errno());
        return true;
    }

    unsigned bufsize = MAX_BUFSIZE;
    {
        struct stat fdstat;
        if(!fstat(fd, &fdstat)) {
#ifdef HAVE_POSIX_FADVISE
            (void)posix_fadvise(fd, 0, fdstat.st_size, POSIX_FADV_SEQUENTIAL);
#endif
            if(fdstat.st_size < (int)bufsize)
                bufsize = fdstat.st_size;
        }
        else {
            log_warn("rfc1035: fstat(%s) failed for advice, not critical...", fn);
        }
    }

    zscan_t* z = xcalloc(1, sizeof(zscan_t));
    z->lcount = 1;
    z->def_ttl = gconfig.zones_default_ttl;
    z->zone = zone;
    dname_copy(z->origin, zone->dname);
    z->lhs_dname[0] = 1; // set lhs to relative origin initially

    char* buf = xmalloc(bufsize + 1);
    buf[bufsize] = 0;

    sij_func_t sij = &_scan_isolate_jmp;
    bool failed = sij(z, buf, bufsize, fd);

    if(close(fd)) {
        log_err("rfc1035: Cannot close file '%s': %s", fn, dmn_logf_errno());
        failed = true;
    }

    free(buf);

    if(z->texts) {
        for(unsigned i = 0; i < z->num_texts; i++)
            if(z->texts[i])
                free(z->texts[i]);
        free(z->texts);
    }
    if(z->rfc3597_data)
        free(z->rfc3597_data);
    free(z);

    return failed;
}

