/**********************************************************************
** This program is part of the kinetics library and is
**           copyright (C) 1995 Upinder S. Bhalla.
** It is made available under the terms of the
**           GNU Library General Public License. 
** See the file COPYRIGHT for the full notice.
**********************************************************************/
static char rcsid[] = "$Id: pool.c,v 1.4 2000/06/12 05:07:03 mhucka Exp $";
 
/*
** $Log: pool.c,v $
** Revision 1.4  2000/06/12 05:07:03  mhucka
** Removed nested comments; added NOTREACHED comments where appropriate.
**
** Revision 1.3  1998/07/21 19:49:11  dhb
** Fixed ANSI C isms.
**
 * Revision 1.2  1998/07/15 06:45:33  dhb
 * Upi update
 *
 * Revision 1.1  1997/07/24 17:49:40  dhb
 * Initial revision
 *
 * Revision 1.5  1994/10/26  20:46:09  bhalla
 * Fixed bug with conserve option, where the nTotal got set to twice nInit
 *
 * Revision 1.4  1994/09/29  18:20:21  bhalla
 * -- Split up the SUMTOTAL msg into separate SUMTOTAL and CONSERVE
 * msgs, so that these two cases become more distinct and can coexist.
 * This vastly cleans up the calculation for SUMTOTAL, which usually
 * needs some form of CONSERVE as well.
 *
 * -- Changed the format for slave_enable. It now handles 3 flags:
 * NSLAVE = 0x01, CONCSLAVE = 0x02 and BUFFER = 0x04. The BUFFER
 * option is new, and simplifies the frequent process of using a pool
 * as a buffer by bypassing the whole calculations and just setting
 * n to nInit and Co to CoInit.
 *
 * Revision 1.3  1994/08/15  18:31:55  bhalla
 * Added option to use SLAVE msg to control concen rather than n,
 * using the slave_enable flag set to 2.
 *
 * Revision 1.2  1994/08/05  19:23:48  bhalla
 * conversion to using n rather than conc.
 * Also added new option for using SUMTOTAL msg to handle conservation
 * relationships
 *
 * Revision 1.1  1994/06/13  22:55:39  bhalla
 * Initial revision
 * */

#include "kin_ext.h"

#ifdef __STDC__
static int do_reset(struct pool_type *pool);
static int check_msg_type(struct pool_type *pool, MsgIn *msg);
static void do_oldmsgmode(struct pool_type *pool, double* A, double* B,
	double* conserve, double* sumtotal);
#else
static int do_reset();
static int check_msg_type();
static void do_oldmsgmode();
#endif
static double **msgdata;
static int alloced_nmsgs = 0;
static int used_nmsgs;
#define ALLOCED_NMSGS_INCREMENT 10000

/*
** A pool of a reagent. This version does not carry out any
** reactions, but interfaces to the kin and enz elements which do.
** This way there are no limitations on the number of reactions a
** pool can be involved in
*/

#define POOL_SUMTOTAL	0x01
#define POOL_CONSERVE	0x02
#define POOL_NSLAVE		0x01
#define POOL_CONCSLAVE	0x02
#define POOL_BUFFER		0x04

PoolFunc(pool,action)
register struct pool_type *pool;
Action		*action;
{
MsgIn	*msg;
int j;
double	A,B;
double	dt;
double	sumtotal;
double	conserve;

    if(debug > 1){
		ActionHeader("ReactionPool",pool,action);
    }

    SELECT_ACTION(action){
	/* the CoRemaining stuff should actually be calculated
	** in a separate phase, at INIT, but it would be a very
	** expensive undertaking in terms of CPU cycles.
	*/
	    case PROCESS:
			/* slave_enable of 3 is the same as buffering,
			** so we bypass everything else */
			if (pool->slave_enable & POOL_BUFFER) {
				pool->n = pool->nInit;
				pool->Co = pool->CoInit;
				break;
			}
			A = B = 0;
			sumtotal = 0;
			conserve = 0;
			if (pool->oldmsgmode) {
				do_oldmsgmode(pool, &A, &B, &conserve, &sumtotal);
			} else {
		
				/* TYPE = REAC */
				/* Keep things together to help cache coherence */
				double **data = pool->msgdata;
				for (j = 0; j < pool->msgcount[0]; j+=2, data++) {
					A += **data++; /* data is already a double* */
					B += **data; /* data is already a double* */
				}
			
				/* TYPE = MM_PRD */
				for (j = 0; j < pool->msgcount[1]; j++, data++)
					A += **data; /* data is already a double* */
			
				/* TYPE = SLAVE */
				if (pool->was_slaved) {
					for (j = 0; j < pool->msgcount[2]; j++, data++)
						pool->n = **data;
					if (pool->slave_enable == POOL_CONCSLAVE)
						pool->n *= pool->vol;
					/* Convert the n's to Co's */
					pool->nRemaining = pool->nTotal - pool->n;
					if (pool->vol > 0) {
						pool->Co = pool->n / pool->vol;
						pool->CoRemaining =
							pool->nRemaining / pool->vol;
					}
					break;
				}
			
				/* TYPE = REMAINING */
				for (j = 0; j < pool->msgcount[3]; j++, data++)
					pool->nRemaining -= **data;
			
				/* TYPE = CONSERVE */
				for (j = 0; j < pool->msgcount[4]; j++, data++)
					conserve += **data;
			
				/* TYPE = VOL */
				for (j = 0; j < pool->msgcount[5]; j++, data++) {
					pool->vol = **data;
					if (pool->vol <= 0) 
						pool->vol = 1;
				}
			
				/* TYPE = CONSERVE */
				for (j = 0; j < pool->msgcount[6]; j++, data++)
					sumtotal += **data;
			}
	
			if (pool->consv_flag) {
				if (pool->consv_flag & POOL_SUMTOTAL)  {
					pool->nTotal = sumtotal;
					if (pool->consv_flag & ~POOL_CONSERVE)  {
						pool->n = sumtotal;
					}
				}
				if (pool->consv_flag & POOL_CONSERVE)  {
					pool->n = pool->nTotal - conserve;
					if (pool->n < 0)
						pool->n = 0;
				}
			} else {
				dt = Clockrate(pool);
				if (pool->n > 1.0e-10 && B > 1e-20) {
					/* Exp Euler */
					double D = exp(-B * dt / pool->n);
					pool->n *= D + (A/B) * (1 - D);
				} else {
					/* Euler */
					pool->n += (A - B) * dt;
				}
				if (pool->n < pool->nMin) 
					pool->n = pool->nMin;
			}
			/* Convert the n's to Co's */
			pool->nRemaining = pool->nTotal - pool->n;
			pool->Co = pool->n / pool->vol;
			pool->CoRemaining = pool->CoTotal - pool->Co;
		break;
    case RESET:
		do_reset(pool);
    break;
	case CREATE:
		pool->vol = 1; /* need to initialize this */
		pool->nMin = pool->CoMin = 0;
		pool->oldmsgmode = 0;
	break;
    case SET :
		if (action->argc == 2) {
			char *field = action->argv[0];
			double newval;
			if (strcmp(field,"vol") == 0 ||
				strcmp(field,"n") == 0 ||
				strcmp(field,"nInit") == 0 ||
				strcmp(field,"nRemaining") == 0 ||
				strcmp(field,"nTotal") == 0 ||
				strcmp(field,"nMin") == 0 ||
				strcmp(field,"Co") == 0 ||
				strcmp(field,"CoInit") == 0 ||
				strcmp(field,"CoRemaining") == 0 ||
				strcmp(field,"CoTotal") == 0 ||
				strcmp(field,"CoMin") == 0) {
					newval = atof(action->argv[1]);
					if (newval < 0) {
						/* Error(); */
						printf("Cannot set field=%s of pool=%s \n to negative value, using 0.0\n",
						field,Pathname(pool));
						/* return(1); */
						newval = 0.0;
					}
				}
				if (strcmp(field,"vol") == 0 && newval > 0) {
					pool->vol = newval;
					if (pool->keepconc) {
						pool->n = pool->Co * pool->vol;
						pool->nRemaining = pool->CoRemaining* pool->vol;
						pool->nTotal = pool->CoTotal * pool->vol;
						pool->nInit = pool->CoInit * pool->vol;
						pool->nMin = pool->CoMin * pool->vol;
					} else {
						pool->Co = pool->n / pool->vol;
						pool->CoRemaining = pool->nRemaining/pool->vol;
						pool->CoTotal = pool->nTotal / pool->vol;
						pool->CoInit = pool->nInit / pool->vol;
						pool->CoMin = pool->nMin / pool->vol;
					}
					return(1);
				}
				if (pool->vol > 0) {
					if (strcmp(field,"n") == 0)
						pool->Co = newval / pool->vol;
					if (strcmp(action->argv[0],"nInit") == 0)
						pool->CoInit = newval / pool->vol;
					if (strcmp(action->argv[0],"nRemaining") == 0)
						pool->CoRemaining = newval / pool->vol;
					if (strcmp(action->argv[0],"nMin") == 0)
						pool->CoMin = newval / pool->vol;
					if (strcmp(action->argv[0],"nTotal") == 0)
						pool->CoTotal = newval / pool->vol;

					if (strcmp(field,"Co") == 0)
						pool->n = newval * pool->vol;
					if (strcmp(action->argv[0],"CoInit") == 0)
						pool->nInit = newval * pool->vol;
					if (strcmp(action->argv[0],"CoRemaining") == 0)
						pool->nRemaining = newval * pool->vol;
					if (strcmp(action->argv[0],"CoMin") == 0)
						pool->nMin = newval * pool->vol;
					if (strcmp(action->argv[0],"CoTotal") == 0)
						pool->nTotal = newval * pool->vol;
				}
				/* Do the normal set on all the fields */
				return(0);
		}
        return(0); /* do the normal set */
	/* NOTREACHED */
		break;
	}
}

#ifdef __STDC__
static void do_oldmsgmode(struct pool_type *pool, double* A, double* B,
	double* conserve, double* sumtotal) {
#else
static void do_oldmsgmode(pool, A, B, conserve, sumtotal)
struct pool_type *pool;
double* A;
double* B;
double* conserve;
double* sumtotal;
{
#endif
	MsgIn	*msg;
        MSGLOOP(pool,msg) {
        case 0:		/* TYPE = REAC */
   			*A += MSGVALUE(msg,0);
   		   	*B += MSGVALUE(msg,1);
        	break;
		case 1:		/* TYPE = MM_PRD. */
   		   	*A += MSGVALUE(msg,0);
		break;
		case 2:		/* TYPE = SLAVE */
					/* n = value to follow */
					/* msg 0 = new value of n */
					if (pool->slave_enable) {
						pool->was_slaved = 1;
						/* hack to use concen for SLAVE
						** rather than n */
						if (pool->slave_enable == 
							POOL_CONCSLAVE)
							pool->n =
								pool->vol * MSGVALUE(msg,0);
						else
							pool->n = MSGVALUE(msg,0);
					}
		break;
		case 3:		/* TYPE = REMAINING */
				pool->nRemaining -= MSGVALUE(msg,0);
		break;
		case 4:		/* TYPE = CONSERVE */
				*conserve += MSGVALUE(msg,0);
		break;
		case 5:		/* TYPE = VOL */
				pool->vol = MSGVALUE(msg,0);
		break;
		case 6:		/* TYPE = SUMTOTAL */
				*sumtotal += MSGVALUE(msg,0);
		break;
   	}
}

#ifdef __STDC__
static int check_msg_type(struct pool_type *pool, MsgIn *msg) {
#else
static int check_msg_type(pool, msg)
struct pool_type *pool;
MsgIn *msg;
{
#endif
	if (MSGSLOT(msg)[0].func != DoubleMessageData) {
		Warning();
		/* Complain */
		printf("Message source for REAC msg to %s is not a double\n",
		Pathname(pool));
		printf("Reverting to slow old messaging\n");
		pool->oldmsgmode = 1;
	}
}

do_reset_alloc_msgdata() {
	used_nmsgs = 0;
	if (alloced_nmsgs == 0) {
		alloced_nmsgs = ALLOCED_NMSGS_INCREMENT;
		msgdata = (double **)calloc(alloced_nmsgs, sizeof(double*));
	}
	SimReset();
}

#ifdef __STDC__
double** alloc_msgdata(int n, double** data) {
#else
double** alloc_msgdata(n, data)
int n;
double** data;
{
#endif

	memcpy((void *)(msgdata + used_nmsgs), (void *)data,
		n * sizeof(double *));
	used_nmsgs += n;
	if (used_nmsgs >= alloced_nmsgs) {
		alloced_nmsgs += ALLOCED_NMSGS_INCREMENT;
		if (msgdata)
			free(msgdata);
		msgdata = (double **)calloc(alloced_nmsgs, sizeof(double));
		Error();
		printf("Realloced message table\n");
	}
	return msgdata + (used_nmsgs - n);
}

#ifdef __STDC__
static int do_reset(struct pool_type *pool) {
#else
static int do_reset(pool)
struct pool_type *pool;
{
#endif
	MsgIn	*msg;
	double	sumtotal;
	double	conserve;
	double	*tempdata[200]; /* over 200 msgs would be odd */
	double	**data;
	int i,j;

		if (pool->vol <= 0)
			pool->vol = 1;
		if (pool->keepconc) {
			pool->n = pool->Co * pool->vol;
			pool->nRemaining = pool->CoRemaining* pool->vol;
			pool->nTotal = pool->CoTotal * pool->vol;
			pool->nInit = pool->CoInit * pool->vol;
			pool->nMin = pool->CoMin * pool->vol;
		} else {
			pool->Co = pool->n / pool->vol;
			pool->CoRemaining = pool->nRemaining/pool->vol;
			pool->CoTotal = pool->nTotal / pool->vol;
			pool->CoInit = pool->nInit / pool->vol;
			pool->CoMin = pool->nMin / pool->vol;
		}
		pool->consv_flag = 0;
		pool->was_slaved = 0;
		sumtotal = 0;
		conserve = pool->nInit;

		for (i = 0 ; i < 7; i++)
			pool->msgcount[i] = 0;

		/* During RESET, we need to get the msg pointers. */
		data = tempdata;
		MSGLOOP(pool,msg) {
			
   			case 0:		/* TYPE = REAC */
				*data++ = (double*)(MSGPTR(msg, 0));
				*data++ = (double*)(MSGPTR(msg, 1));
				pool->msgcount[0] += 2;
			break;
			default:
			check_msg_type(pool,msg);
			break;
		}
		MSGLOOP(pool,msg) {
   			case 1:		/* TYPE = MM_PRD */
				*data++ = (double*)(MSGPTR(msg, 0));
				pool->msgcount[1]++;
			break;
		}
		if (!(pool->slave_enable & POOL_BUFFER)) {
			MSGLOOP(pool,msg) {
			case 2:		/* TYPE = SLAVE */
				if (pool->slave_enable == POOL_CONCSLAVE ||
					pool->slave_enable == POOL_NSLAVE) {
					pool->was_slaved = 1;
					*data++ = (double*)(MSGPTR(msg, 0));
					pool->msgcount[2]++;
				}
			break;
			}
		}
		MSGLOOP(pool,msg) {
			case 3:		/* TYPE = REMAINING */
				*data++ = (double*)(MSGPTR(msg, 0));
				pool->msgcount[3]++;
			break;
		}
		MSGLOOP(pool,msg) {
			case 4:		/* TYPE = CONSERVE */
				*data++ = (double*)(MSGPTR(msg, 0));
				pool->msgcount[4]++;
				pool->consv_flag |= POOL_CONSERVE;
				conserve += MSGVALUE(msg,1);
			break;
		}
		MSGLOOP(pool,msg) {
			case 5:		/* TYPE = VOL */
				*data++ = (double*)(MSGPTR(msg, 0));
				pool->msgcount[5]++;
			break;
		}
		MSGLOOP(pool,msg) {
			case 6:		/* TYPE = SUMTOTAL */
				*data++ = (double*)(MSGPTR(msg, 0));
				pool->msgcount[6]++;

				pool->consv_flag |= POOL_SUMTOTAL;
				sumtotal += MSGVALUE(msg,1);
			break;
		}

		for (i = 0, j= 0 ; i < 7; i++)
			j += pool->msgcount[i];
		pool->msgdata = alloc_msgdata(j, tempdata);
		
		if (pool->consv_flag & POOL_SUMTOTAL) { /*This controls nTotal*/
			pool->nTotal = sumtotal;
		}
		if (pool->consv_flag & POOL_CONSERVE) { /* This controls n. */
			/* add up all the relevant nInits. */
			if (pool->consv_flag & POOL_SUMTOTAL) {
				if (sumtotal < conserve &&
					!(pool->slave_enable & POOL_BUFFER)) {
					printf("Warning: Initial values for sumtotal %g are less than for conserve=%g on %s\n",
					sumtotal,conserve, Pathname(pool));
					conserve = sumtotal;
				}
				if (!(pool->slave_enable & POOL_BUFFER))
					pool->nInit = sumtotal - conserve;
			} else {
				pool->nTotal = conserve;
			}
		}

		if (pool->nTotal < pool->nInit)
			pool->nTotal = pool->nInit;
		pool->n = pool->nInit;
		/* redo the Co values that might have changed in the RESET */
		pool->Co = pool->n / pool->vol;
		pool->CoTotal = pool->nTotal / pool->vol;
}

#undef POOL_SUMTOTAL
#undef POOL_CONSERVE
#undef POOL_NSLAVE
#undef POOL_CONCSLAVE
#undef POOL_BUFFER

/* This function is declared as an extern void() */
void copyleft_kin()
{
	printf("The kinetics library is copylefted under the LGPL, see kinetics/COPYRIGHT.\n\n");
}
