/****************************************************************************
** clparam.h
**
** Michael S. Borella <mike@borella.net>
**
** Class definition for a single command line parameter
**
** $Id: clparam.h,v 1.4 2000/10/25 22:52:05 mborella Exp $
**
*****************************************************************************/

#ifndef CLPARAM_H
#define CLPARAM_H

#include <list>
#include <string>
#include "logfile.h"

// valid types of parameters
enum param_t
{
  null_t = 0, int_t, float_t, char_t, string_t, flag_t
};

//---------------------------------------------------------------------------
//
// class Clparam
//
//---------------------------------------------------------------------------

class Clparam
{
private:
  char          _short_rep;       // single letter representation
  string        _long_rep;        // string representation
  string        _internal_name;
  string        _external_name;
  param_t       _type;
  string        _default_value;
  string        _low_range;
  string        _high_range;
  string        _callback;
  list<string>  _descriptions;
  int           _longopt_value;
  bool          _long_only;

public:
  // Constructor, destructor and copy constructor
  Clparam();
  ~Clparam(){}
  Clparam& operator=(const Clparam &);

  // functions to set the values
  void    short_rep(char c)         { _short_rep = c; }
  void    long_rep(string s)        { _long_rep = s; }
  void    internal_name(string s)   { _internal_name = s; }
  void    external_name(string s)   { _external_name = s; }
  void    external_name(char c)     { _external_name += c; }
  void    type(param_t t)           { _type = t; }
  void    default_value(string s)   { _default_value = s; }
  void    low_range(string s)       { _low_range = s; }
  void    high_range(string s)      { _high_range = s; }
  void    callback(string s)        { _callback = s; }
  void    add_description(string s) { _descriptions.push_back(s); }
  void    longopt_value(int i)      { _longopt_value = i; }
  void    long_only(bool b)         { _long_only = b; }

  // functions to retrieve the values
  char    short_rep()            { return _short_rep; }
  string  long_rep()             { return _long_rep; }
  string  internal_name()        { return _internal_name; }
  string  external_name()        { return _external_name; }
  param_t type()                 { return _type; }
  string  default_value()        { return _default_value; }
  string  low_range()            { return _low_range; }
  string  high_range()           { return _high_range; }
  string  callback()             { return _callback; }
  int     longopt_value()        { return _longopt_value; }
  bool    long_only()            { return _long_only; }
  int     get_num_desc()         { return _descriptions.size(); }
  string  get_description(int);
  

  // logdump - dump contents to a Logfile
  void    logdump(Logfile&);
};

#endif
