/*  GFC-UI: GTK+ Foundation Classes (User Interface Library)
 *  Copyright (C) 2004 The GFC Development Team.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/// @file gfc/gdk/displaymanager.hh
/// @brief A GdkDisplayManager C++ wrapper interface.
///
/// Provides DisplayManager, an object that Maintains a list of all open Gdk::Displays.

#ifndef GFC_GDK_DISPLAY_MANAGER_HH
#define GFC_GDK_DISPLAY_MANAGER_HH

#ifndef GFC_G_OBJECT_HH
#include <gfc/glib/object.hh>
#endif

#ifndef GFC_GDK_DISPLAY_HH
#include <gfc/gdk/display.hh>
#endif

#ifndef GFC_GDK_TYPES_HH
#include <gfc/gdk/types.hh>
#endif

#ifndef _CPP_VECTOR
#include <vector>
#endif

namespace GFC {

namespace Gdk {

/// @class DisplayManager displaymanager.hh gfc/gdk/displaymanager.hh
/// @brief A GdkDisplayManager C++ wrapper class.
///
/// The purpose of the DisplayManager singleton object is to offer notification
/// when displays appear or disappear or the default display changes.

class DisplayManager : public G::Object
{
	friend class G::Object;

	DisplayManager(const DisplayManager&);
	DisplayManager& operator=(const DisplayManager&);

protected:
/// @name Constructors
/// @{

	explicit DisplayManager(GdkDisplayManager *display_manager, bool owns_reference = true);
	///< Construct a new DisplayManager from an existing GdkDisplayManager.
	///< @param display_manager A pointer to a GdkDisplayManager.
	///< @param owns_reference Set false if the initial reference count is floating, set true if it's not.
	///<
	///< <BR>The <EM>display_manager</EM> can be a newly created GdkDisplayManager or an existing
	///< GdkDisplayManager. (see G::Object::Object).

/// @}
/// @name Signal Prototypes
/// @{

	typedef G::Signal<void, Display&> DisplayOpenedSignalType;
	typedef G::SignalProxy<TypeInstance, DisplayOpenedSignalType> DisplayOpenedSignalProxy;
	static const DisplayOpenedSignalType display_opened_signal;
	///< Display opened signal (see sig_display_opened()). Calls a slot with the signature:
	///< @code
	///< void function(Display& display);
	///< // display: The Display being opened.
	///< @endcode
	
/// @}

public:
/// @name Constructors
/// @{

	virtual ~DisplayManager();
	///< Destructor.

/// @}

	static DisplayManager* get();
	///< Returns the global DisplayManager singleton object.

/// @name Accessors
/// @{

	GdkDisplayManager* gdk_display_manager() const;
	///< Get a pointer to the GdkDisplayManager structure.

	operator GdkDisplayManager* () const;
	///< Conversion operator; safely converts a DisplayManager to a GdkDisplayManager pointer.

	Display* get_default_display() const;
	///< Gets the default Display.
	///< @return A Display, or null if there is no default display.

	bool list_displays(std::vector<Display*>& displays) const;
	///< List all currently open displays.
	///< @param displays A reference to a vector of Display* to hold the list of opened displays.
	///< @return <EM>true</EM> if the vector is not empty.

/// @}
/// @name Methods
/// @{

	void set_default_display(const Display& display);
	///< Sets display as the default display.
	///< @param display A Display.

/// @}
/// @name Signal Proxies
/// @{

	const DisplayOpenedSignalProxy sig_display_opened();
	///< Connect to the display_opened_signal; emitted when a Display is opened.

/// @}
};

} // namespace Gdk

} // namespace GFC

#include <gfc/gdk/inline/displaymanager.inl>

#endif // GFC_GDK_DISPLAY_MANAGER_HH

