// MM2ENG.H : calculations for reduced protein models (molecular mechanics).

// Copyright (C) 1998 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "config.h"	// this is target-dependent...

#ifndef MM2ENG_H
#define MM2ENG_H

struct mm2_bt1;
struct mm2_bt2;
struct mm2_bt3;
struct mm2_bt4;

struct mm2_bt1_data;
struct mm2_bt2_data;

struct mm2_nbt1;

struct mm2_nbt3_nd;
struct mm2_nbt3_ipd;

struct mm2_nbt3_nl;
struct mm2_nbt3_coi;
struct mm2_nbt3_ips;
struct mm2_nbt3_arc;

#define SIZE_NL		100
#define SIZE_NT		100
#define SIZE_COI	100
#define SIZE_IPD	50
#define SIZE_IPS	100
#define SIZE_ARC	100

#define INDEX_FLAG 0x8000000	// index of the point
#define ORDER_FLAG 0x4000000	// 0 = starting point, 1 = ending point
#define FLAG_MASK ~(INDEX_FLAG | ORDER_FLAG)

#define BETA 52.0	// beta-angle of the bt4-term...

class mm2_eng;

/*################################################################################################*/

#include "engine.h"
#include "mm2mdl.h"

#include <vector>
using namespace std;

/*################################################################################################*/

#define TYPE_LOOP	0x00
#define TYPE_HELIX	0x01
#define TYPE_STRAND	0x02

#define TYPE_SIDE	0x10
#define TYPE_BRIDGE	0x11
#define TYPE_TERM	0x12

struct mm2_bt1		// bond stretching
{
	i32s atmi[2];
	
	f64 opt; f64 fc;
	
	i32s GetIndex(i32s p1, bool p2)
	{
		return atmi[p2 ? p1 : !p1];
	}
};

struct mm2_bt2		// angle bending
{
	i32s atmi[3];
	
	i32s index1[2];
	bool dir1[2];
	
	i32s type;
	
	f64 opt; f64 fc[2];
};

struct mm2_bt3		// main-chain torsion interactions + peptide bond dipoles
{
	i32s atmi[4];
	
	i32s index2[2];
	i32s index1[4];
	bool dir1[4];
	
	i32s tor_type;
	
	f64 torc[3];
	f64 tors[3];
	
	// dipole term starts here... dipole term starts here... dipole term starts here...
	// dipole term starts here... dipole term starts here... dipole term starts here...
	// dipole term starts here... dipole term starts here... dipole term starts here...
	
	f64 pbdd;
	
	f64 bv[2][3];
	f64 dbv[2][3][3][3];
	
	f64 dv[3];
	f64 ddv[4][3][3];
	
	i32s dip_type;
	
	bool skip;
	
	f64 dipc[3];
	f64 dips[3];
	f64 dipk[2];
};

struct mm2_bt4		// chirality-conserving special side-chain term
{
	i32s index1;
	i32s index2;
	
	f64 opt; f64 fc;
	
	f64 fscos[3];
	f64 fssin[3];
};

struct mm2_bt1_data
{
	f64 data1;		// len????
	f64 data2[2][3];	// dlen????
};

struct mm2_bt2_data
{
	f64 data1;		// ang????
	f64 data2[3][3];	// dang????
};

struct mm2_nbt1			// non-bonded interactions
{
	i32s atmi[2];
	
	f64 data[2];
	
	bool operator==(const mm2_nbt1 & p1) const
	{
		if ((atmi[0] == p1.atmi[0]) && (atmi[1] == p1.atmi[1])) return true;
		if ((atmi[0] == p1.atmi[1]) && (atmi[1] == p1.atmi[0])) return true;
		
		return false;
	}
};

struct mm2_nbt3_nd		// SASA neighbor data
{
	i32s index;
	f64 distance;
	
	// these are sorted in reverse order, from large to small...
	// these are sorted in reverse order, from large to small...
	// these are sorted in reverse order, from large to small...
	
	bool operator<(const mm2_nbt3_nd & p1) const
	{
		return (distance > p1.distance);
	}
};

struct mm2_nbt3_ipd		// SASA intersection point data
{
	f64 angle;
	i32u ipdata;
	
	bool operator<(const mm2_nbt3_ipd & p1) const
	{
		return (angle < p1.angle);
	}
};

struct mm2_nbt3_nl		// SASA neighbor list
{
	i32s index_count;
	i32s index[SIZE_NL];
};

struct mm2_nbt3_coi		// SASA circle of intersection
{
	i32s index;
	
	i32s ipd_count;
	mm2_nbt3_ipd ipdt[SIZE_IPD];
	
	f64 refv[3];
	
	f64 dist;
	f64 dv[3]; f64 ddv[3][3];
	
	f64 g; f64 dg[3];
	f64 ct; f64 dct[3];
	
	bool flag;
	
	void AddIPD(f64 * p1, i32u p2)
	{
		ipdt[ipd_count].ipdata = p2;
		
		if (!ipd_count)
		{
			f64 t1a[3];
			t1a[0] = dv[0] * p1[0];
			t1a[1] = dv[1] * p1[1];
			t1a[2] = dv[2] * p1[2];
			
			f64 t1b = t1a[0] + t1a[1] + t1a[2];
			
			refv[0] = p1[0] - dv[0] * t1b;
			refv[1] = p1[1] - dv[1] * t1b;
			refv[2] = p1[2] - dv[2] * t1b;
			
			f64 t1c = sqrt(refv[0] * refv[0] + refv[1] * refv[1] + refv[2] * refv[2]);
			refv[0] /= t1c; refv[1] /= t1c; refv[2] /= t1c;
			
			ipdt[ipd_count].angle = 0.0;
		}
		else
		{
			f64 t1a[3];
			t1a[0] = dv[0] * p1[0];
			t1a[1] = dv[1] * p1[1];
			t1a[2] = dv[2] * p1[2];
			
			f64 t1b = t1a[0] + t1a[1] + t1a[2];
			
			f64 t2a[3];
			t2a[0] = p1[0] - dv[0] * t1b;
			t2a[1] = p1[1] - dv[1] * t1b;
			t2a[2] = p1[2] - dv[2] * t1b;
			
			f64 t1c = sqrt(t2a[0] * t2a[0] + t2a[1] * t2a[1] + t2a[2] * t2a[2]);
			t2a[0] /= t1c; t2a[1] /= t1c; t2a[2] /= t1c;
			
			f64 t1d = refv[0] * t2a[0] + refv[1] * t2a[1] + refv[2] * t2a[2];
			if (t1d < -1.0) t1d = -1.0;	// domain check...
			if (t1d > +1.0) t1d = +1.0;	// domain check...
			
			f64 t9a = acos(t1d);
			
			f64 t3a[3];
			t3a[0] = dv[1] * t2a[2] - dv[2] * t2a[1];
			t3a[1] = dv[2] * t2a[0] - dv[0] * t2a[2];
			t3a[2] = dv[0] * t2a[1] - dv[1] * t2a[0];
			
			f64 t9b = refv[0] * t3a[0] + refv[1] * t3a[1] + refv[2] * t3a[2];
			
			if (t9b < 0.0) ipdt[ipd_count].angle = -t9a;
			else ipdt[ipd_count].angle = +t9a;
		}
		
		ipd_count++;
		if (ipd_count >= SIZE_IPD) { cout << "IPD overflow!!!" << endl; exit(EXIT_FAILURE); }
	}
};

struct mm2_nbt3_ips		// SASA intersection points
{
	i32s coi[2];
	
	f64 ipv[2][3];
	f64 dipv[2][2][3][3];
};

struct mm2_nbt3_arc		// SASA positively oriented arc
{
	i32s coi;
	i32s index[2][2];
	
	i32u ipdata[2];
	
	f64 tv[2][3];
	f64 dtv[2][2][3][3];
	
	bool flag;
};

/*################################################################################################*/

void CopyCRD(mm2_mdl *, mm2_eng *, i32u);
void CopyCRD(mm2_eng *, mm2_mdl *, i32u);

class mm2_eng : public engine
{
	private:
	
	mm2_mdl & mdl;
	
//	protected:
	public:		// this might be needed sometimes in TARGET1...
	
	f64_a3 * crd;
	f64_a3 * d1;
	
	vector<mm2_atm> index_vector;
	
	vector<mm2_bt1> bt1_vector;
	vector<mm2_bt2> bt2_vector;
	vector<mm2_bt3> bt3_vector;
	vector<mm2_bt4> bt4_vector;
	
	vector<mm2_bt1_data> bt1data;
	vector<mm2_bt2_data> bt2data;
	
	vector<mm2_nbt1> nbt1_vector;
	
	mm2_nbt3_nl * nbt3_nl;
	i32s * dist1; f64 * dist2;
	
	i32s * nbtp;
	f64 * sasa;
	
	f64 * mass;
	
	f64 * vdwr;
	f64 * vdwr1;
	f64 * vdwr2;
	
	public:
	
	f64 energy;
	f64 energy_bt1;
	f64 energy_bt2;
	f64 energy_bt3;
	f64 energy_bt4;
	f64 energy_nbt1a;
	f64 energy_nbt1b;
	f64 energy_nbt2a;
	f64 energy_nbt2b;
	f64 energy_nbt3;
	
	f64 constraints;
	f64 constraints_OLD_FIXME;
	
	public:
	
	friend void CopyCRD(mm2_mdl *, mm2_eng *, i32u);
	friend void CopyCRD(mm2_eng *, mm2_mdl *, i32u);
	
	mm2_eng(mm2_mdl &);
	~mm2_eng(void);
	
	mm2_mdl * GetModel(void) { return & mdl; }
	
	void Compute(i32s);	// virtual
	void Check(i32s);	// virtual

	f64 GetGradientVectorLength(void);	// virtual
	
	protected:
	
	bool InitNBT1(mm2_nbt1 *);
	void InitLenJon(mm2_nbt1 *, f64, f64);
	
	i32s GetIndex(vector<mm2_atm> &, mm2_atm);
	
	void ComputeBT1(i32s);
	void ComputeBT2(i32s);
	void ComputeBT3(i32s);
	void ComputeBT4(i32s);
	
	void ComputeNBT1(i32s);
	void ComputeNBT2(i32s);
	void ComputeNBT3(i32s);
	
	public:
	
	friend class mm2_geomopt;
	friend class mm2_moldyn;
};

/*################################################################################################*/

#endif	// MM2ENG_H

// eof
