// QM1ALG.CPP

// Copyright (C) 2001 Tommi Hassinen.

// This program is free software; you can redistribute it and/or modify it
// under the terms of the license (GNU GPL) which comes with this package.

/*################################################################################################*/

#include "qm1alg.h"

/*################################################################################################*/

qm1_geomopt::qm1_geomopt(qm1_eng * p1, i32s p2, f64 p3, f64 * p4) : conjugate_gradient(p2, p3)
{
	eng = p1; tss_target = p4;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		for (i32u n2 = 0;n2 < 3;n2++)
		{
			AddVar(& eng->crd[n1 * 3 + n2], & eng->d1[n1 * 3 + n2]);
		}
	}
}

qm1_geomopt::~qm1_geomopt(void)
{
}

f64 qm1_geomopt::GetValue(void)
{
	eng->Compute(0);	// request energy
	
	if (tss_target != NULL) AddConst(0, eng, tss_target);
	return eng->energy;
}

f64 qm1_geomopt::GetGradient(void)
{
	eng->Compute(1);	// request energy and gradient
	
	if (tss_target != NULL) AddConst(1, eng, tss_target);
	return eng->energy;
}

// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
// eng -> p2???? tss_target -> p3?????
void qm1_geomopt::AddConst(i32s p1, qm1_eng * eng, f64 * tss_target)
{
	const i32u store = eng->GetModel()->atom_list.size() * 3;
	tss_target[store + 1] = 0.0;
	
	for (i32u n1 = 0;n1 < eng->GetModel()->atom_list.size();n1++)
	{
		f64 t1a[3]; f64 t1b = 0.0;
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t9a = eng->crd[n1 * 3 + n2];
			f64 t9b = tss_target[n1 * 3 + n2];
			
			t1a[n2] = t9a - t9b;
			t1b += t1a[n2] * t1a[n2];
		}
		
		f64 t1c = sqrt(t1b);
		for (i32s n2 = 0;n2 < 3;n2++)
		{
			f64 t9a = t1a[n2] / t1c;
			
			t1a[n2] = +t9a;
		}
		
		// f = a(x)^2
		// df/dx = 2a(x)
		
		f64 t2a = t1c;
		f64 t2b = tss_target[store] * t2a * t2a;
		
		tss_target[store + 1] += t2a * t2a;
		eng->energy += t2b;
		
		if (p1 > 0)
		{
			f64 t2c = 2.0 * tss_target[store] * t2a;
			
			for (i32s n2 = 0;n2 < 3;n2++)
			{
				f64 t2d = t1a[n2] * t2c;
				
				eng->d1[n1 * 3 + n2] += t2d;
			}
		}
	}
}

/*################################################################################################*/

qm1_simple_ts_search::qm1_simple_ts_search(qm1_mdl * p1)
{
	mdl = p1;
	
	// check that there are at least 2 crd-sets present (the reactants and products)...
	// check that there are at least 2 crd-sets present (the reactants and products)...
	// check that there are at least 2 crd-sets present (the reactants and products)...
	
	if (mdl->GetCRDSetCount() < 2)
	{
		cout << "for transition state search, at least 2 coordinate sets are needed:" << endl;
		cout << "one for reactants and one for products. the third set is optional, and" << endl;
		cout << "it can be used as expected ts stucture at initial part of the search." << endl;
		
		exit(EXIT_FAILURE);
	}
	
	target[0] = new f64[mdl->atom_list.size() * 3 + 2];
	target[0][mdl->atom_list.size() * 3 + 0] = 0.0; target[0][mdl->atom_list.size() * 3 + 1] = 0.0;
	
	target[1] = new f64[mdl->atom_list.size() * 3 + 2];
	target[1][mdl->atom_list.size() * 3 + 0] = 0.0; target[1][mdl->atom_list.size() * 3 + 1] = 0.0;
	
	if (mdl->GetCRDSetCount() > 2)
	{
		SetTarget(0, 2);	// use the estimate...
		SetTarget(1, 2);	// use the estimate...
	}
	else
	{
		SetTarget(0, 1);	// use products as target for reactants...
		SetTarget(1, 0);	// use reactants as target for products...
	}
	
	// if needed, add the following crd-sets:
	// 0 : the current reactant-like state
	// 1 : the current product-like state
	// 2 : last stored reactant-like state
	// 3 : last stored product-like state
	
	i32s new_crd_sets = 4 - mdl->GetCRDSetCount();
	if (new_crd_sets > 0) mdl->PushCRDSets(new_crd_sets);

	mdl->CopyCRDSet(0, 2);
	mdl->CopyCRDSet(1, 3);
	
	mdl->SetCRDSetVisible(0, true);
	mdl->SetCRDSetVisible(1, true);
	mdl->SetCRDSetVisible(2, !true);
	mdl->SetCRDSetVisible(3, !true);
	
	// update the energies... THERE ARE NO CONSTRAINTS YET...
	
	mdl->DiscardCurrentEng();
	for (i32s n1 = 0;n1 < 2;n1++)
	{
		mdl->current_eng = mdl->CreateDefaultEngine();
		if (!mdl->current_eng) exit(EXIT_FAILURE);	// sanity check failed...
		
		CopyCRD(mdl, mdl->current_eng, n1);
		
		mdl->current_eng->Compute(0);
		energy[n1] = mdl->current_eng->energy;
		
		mdl->DiscardCurrentEng();
	}
	
cout << "r-energy = " << energy[0] << "   "; cout << "p-energy = " << energy[1] << "   ";
cout << (energy[0] < energy[1] ? "r" : "p") << " is lower " << fabs(energy[0] - energy[1]) << endl;

}

qm1_simple_ts_search::~qm1_simple_ts_search(void)
{
	delete[] target[0];
	delete[] target[1];
}

void qm1_simple_ts_search::UpdateTargets(void)
{
	const i32u store = mdl->atom_list.size() * 3;
	
	SetTarget(0, 1);	// use products as target for reactants...
	SetTarget(1, 0);	// use reactants as target for products...
	
	for (i32s n1 = 0;n1 < 2;n1++)
	{
		f64 tmp1 = target[n1][store] * target[n1][store + 1];
		
		mdl->current_eng = mdl->CreateDefaultEngine();
		if (!mdl->current_eng) exit(EXIT_FAILURE);	// sanity check failed...
		
		CopyCRD(mdl, mdl->current_eng, n1);
		
		mdl->current_eng->Compute(0);
		qm1_geomopt::AddConst(0, mdl->current_eng, target[n1]);		// this is ugly...
		mdl->DiscardCurrentEng();
		
		target[n1][store] = tmp1 / target[n1][store + 1];
	}
	
cout << "r-energy = " << energy[0] << "   "; cout << "p-energy = " << energy[1] << "   ";
cout << (energy[0] < energy[1] ? "r" : "p") << " is lower " << fabs(energy[0] - energy[1]) << endl;

}

void qm1_simple_ts_search::Run(i32s index)
{
	const i32u store = mdl->atom_list.size() * 3;
	
	i32s automatic = (index < 0 ? -index : false);
	if (automatic) index = (energy[0] < energy[1] ? 0 : 1);
	
	f64 old1 = target[index][store]; f64 old2 = target[index][store + 1];
	
	if (automatic)
	{
		switch (automatic)
		{
			case 1:		target[index][store] = old1 * 1.125; break;
			case 2:		target[index][store] = old1 * 1.250; break;
			default:	target[index][store] = old1 * 1.500; break;
		}
	}
	else
	{
		cout << "last weight was " << target[index][store] << " : next ??? ";
		cin >> target[index][store];
	}
	
	// perform constrained optimization...
	
	mdl->current_eng = mdl->CreateDefaultEngine();
	if (!mdl->current_eng) exit(EXIT_FAILURE);	// sanity check failed...
	
	CopyCRD(mdl, mdl->current_eng, index);
	
	qm1_geomopt * opt = new qm1_geomopt(mdl->current_eng, 20, 0.0125, target[index]);	// optimal settings?!?!?
	
	f64 prev = energy[index]; i32s failed_steps = 0;
	for (i32s n1 = 0;n1 < 100;n1++)
	{
		opt->TakeCGStep(conjugate_gradient::Newton2An);
		cout << n1 << " " << opt->optval << " " << opt->optstp << endl;
		
		f64 delta = prev - opt->optval;
		if (delta > 0.001) failed_steps = 0;
		else failed_steps++;
		
		prev = opt->optval;
		
		if (failed_steps >= 20) break;
	}
	
	delete opt;
	
	// copy the optimized structure and update energy...
	
	mdl->current_eng->Compute(0);
	qm1_geomopt::AddConst(0, mdl->current_eng, target[index]);	// this is ugly...
	
	f64 old_e = energy[index];
	f64 new_e = mdl->current_eng->energy;
	f64 delta_e = new_e - old_e;
	
	if (delta_e < 0.0)
	{
		target[index][store] = old1;
		target[index][store + 1] = old2;
	}
	else
	{
		CopyCRD(mdl->current_eng, mdl, index);
		energy[index] = new_e;
	}
	
	mdl->DiscardCurrentEng();
	
	cout << "old E = " << old_e << " new E = " << new_e << " delta = " << delta_e << endl;
	if (delta_e < 0.0) cout << "DELTA WAS NEGATIVE - THE RESULT WAS REJECTED!!!" << endl;
	
cout << "r-energy = " << energy[0] << "   "; cout << "p-energy = " << energy[1] << "   ";
cout << (energy[0] < energy[1] ? "r" : "p") << " is lower " << fabs(energy[0] - energy[1]) << endl;

}

void qm1_simple_ts_search::SetTarget(i32s index, i32s cset)
{
	iter_qm1al it1; i32u counter = 0;
	for (it1 = mdl->GetAtomsBegin();it1 != mdl->GetAtomsEnd();it1++)
	{
		for (i32s n1 = 0;n1 < 3;n1++)
		{
			f64 tmp1 = (* it1).crd_vector[cset].data[n1];
			target[index][counter++] = tmp1;
		}
	}
}

/*################################################################################################*/

// eof
