/* GtkEditor - a source editor widget for GTK
 * Copyright (C) 1998 Thomas Mailund.
 *
 * The editor widget was written by Thomas Mailund, so bugs should be
 * reported to <mailund@daimi.au.dk>, not the gtk ppl.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <setjmp.h>
#include <guile/gh.h>
#include "gtkeditor.h"
#include "guile.h"


/* ==<error handling>===================================================== */
/* The error handling in this module is a bit hairy because we need to
 * send exceptions back to guile whenever we get an error, and at the
 * same time need to free allocated memory and other resources.
 *
 * This is how it is done.  Only the _wrapper functions are to
 * interface directly to guile.  If they call any other function they
 * should make sure to defer interrupts first.  It must also make sure
 * to set the jumpenv.
 *
 * Functions called later in the process should all take as paramter a
 * jmp_buf* called 'jumpenv'.  This is the point to jump to should an
 * error occur.  Futher more they should have local variables 'int
 * jumpstate' and (if they call futher, and needs to free something
 * should an error occur) 'jmp_buf env'.  Finally such functions
 * should keep a label 'ERROR' at the end.  Put all cleanup after an
 * error after this label.
 */

/* the possible states for a setjmp/longjmp call */
enum _JUMP_RESULTS {
  JUMPSTATE_OK,
  JUMPSTATE_WRONG_TYPE,
  JUMPSTATE_MISC_ERROR
};

#define SETJMP() do { \
           if ((jumpstate=setjmp(env)) != 0) goto ERROR; } while (0)
#define TRY(EXP,RESULT) if (!(EXP)) { jumpstate = (RESULT); goto ERROR; }
#define PROPAGATE() longjmp (*jumpenv, jumpstate)

/* ==<editor-handle managment>============================================ */
/* in guile editor widgets are referenced by a "editor-handle" which
 * is an integer.  This handle is an index into the EdHandles.handles
 * array.  To avoid memory leak we reuse slots, and to do this
 * efficiently we keep a freelist.  The first element in this list is
 * pointed to by 'freelist'
 *
 * Index 0 is a special index.  It is used as the default handle.  No
 * editor can become default by simple install but has to be made so
 * explicitly.  If you do not want to use a default, simply let it
 * remain NULL.
 */
#define INIT_HANDLES_SIZE 5	/* probably more than most will need... */
#define IS_LEGAL_IDX(IDX) (((IDX)>=0) && ((IDX)<EdHandles.used))
#define IN_FREELIST(PTR) ((EdHandles.handles-(GtkEditor**)(PTR))<EdHandles.size)
static struct {
  GtkEditor **handles;		/* pointers to the widgets installed */
  guint size;			/* allocated size of handles */
  guint used;			/* handles in use (+handles in freelist) */
  GtkEditor **freelist;		/* the first free slot in [0,used] */
} EdHandles;

/* installes 'editor' in the handle array.  Returns the index for
 * later removal.
 */
guint
gtk_editor_guile_install_editor (GtkEditor *editor)
{
  guint idx;

  if (EdHandles.freelist == NULL) {
    /* no more handles in freelist...need "use" one more */
    if (EdHandles.used == EdHandles.size) {
      /* no more room...get some more */
      EdHandles.size *= 2;
      EdHandles.handles = g_renew (GtkEditor*, EdHandles.handles, EdHandles.size);
    }
    idx = EdHandles.used;
    EdHandles.handles[idx] = editor;
    EdHandles.used++;

  } else {
    /* take next in freelist */
    GtkEditor **next = (GtkEditor**)(*(EdHandles.freelist));
    idx = EdHandles.freelist-EdHandles.handles;
    *(EdHandles.freelist) = editor;
    EdHandles.freelist = next;
  }

    return idx;
}

/* removes the handle with index 'idx' from the handles. */
void
gtk_editor_guile_remove_editor (guint idx)
{
  g_return_if_fail (IS_LEGAL_IDX (idx));

  if (IN_FREELIST (EdHandles.handles[idx])) {
    g_warning ("%d is already in freelist!", idx);
  } else {
    /* remove */
    EdHandles.handles[idx] = (GtkEditor*)EdHandles.freelist;
    EdHandles.freelist = &(EdHandles.handles[idx]);
  }
}

/* sets the default handle to point to editor. */
void
gtk_editor_guile_set_default (GtkEditor *editor)
{
  EdHandles.handles[0] = editor;
}

/* right...well, it clears it :) */
void
gtk_editor_guile_clear_default (void)
{
  EdHandles.handles[0] = NULL;
}

/* ==<misc. utils. functions>============================================= */
GdkColor*
get_color (GtkEditor *editor, jmp_buf *jumpenv, SCM list)
{
  GdkColor *color;
  GdkColormap *cmap;
  SCM red, green, blue;
  int jumpstate;

  TRY (gh_list_p (list), JUMPSTATE_WRONG_TYPE);

  if (gh_null_p (list)) {
    return NULL;
  }

  TRY ((gh_length (list) == 3), JUMPSTATE_WRONG_TYPE);

  red = gh_car (list); list = gh_cdr (list);
  green = gh_car (list); list = gh_cdr (list);
  blue = gh_car (list);

  TRY ((gh_exact_p (red) && gh_exact_p (green) && gh_exact_p (blue)),
       JUMPSTATE_WRONG_TYPE);

  color = g_new (GdkColor,1);
  color->red = gh_scm2int (red);
  color->green = gh_scm2int (green);
  color->blue = gh_scm2int (blue);

  cmap = gtk_widget_get_colormap (GTK_WIDGET (editor));
  gdk_color_alloc(cmap, color);  

  return color;

 ERROR:
  PROPAGATE();
}

GList*
get_stentry (GtkEditor *editor, jmp_buf *jumpenv, SCM list)
{
  GList *new = NULL;
  char *name = NULL, *start = NULL, *stop = NULL;
  gboolean nest;
  GdkColor *fore = NULL, *back = NULL;
  char *fontname = NULL;
  GdkFont *font = NULL;
  SCM sname, sstart, sstop, snest;
  SCM sfont, sfore, sback;
  int jumpstate = JUMPSTATE_MISC_ERROR;
  jmp_buf env;

  TRY (gh_list_p (list), JUMPSTATE_WRONG_TYPE);
  TRY ((gh_length (list) == 7), JUMPSTATE_WRONG_TYPE);

  /* get the stuff */
  sname = gh_car (list); list = gh_cdr (list);
  sstart = gh_car (list); list = gh_cdr (list);
  sstop = gh_car (list); list = gh_cdr (list);
  snest = gh_car (list); list = gh_cdr (list);
  sfont = gh_car (list); list = gh_cdr (list);
  sfore = gh_car (list); list = gh_cdr (list);
  sback = gh_car (list);

  TRY (gh_string_p (sname), JUMPSTATE_WRONG_TYPE);
  name = gh_scm2newstr (sname, NULL);

  TRY (gh_string_p (sstart), JUMPSTATE_WRONG_TYPE);
  start = gh_scm2newstr (sstart, NULL);

  TRY (gh_string_p (sstop), JUMPSTATE_WRONG_TYPE);
  stop = gh_scm2newstr (sstop, NULL);

  TRY (gh_boolean_p (snest), JUMPSTATE_WRONG_TYPE);
  nest = gh_scm2bool (snest);

  TRY (gh_string_p (sfont), JUMPSTATE_WRONG_TYPE);
  fontname = gh_scm2newstr (sfont, NULL);
  font = gdk_font_load (fontname);
  free (fontname); fontname = NULL;

  /* get ready for jump */
  SETJMP();
  fore = get_color (editor, &env, sfore);
  back = get_color (editor, &env, sback);

  /* finally, get the entry */
  new =  gtk_editor_stentry_new (name, start, stop, nest,
				 font, fore, back, NULL);

  /* The strings has been copied, so we must free them here. */
  free (name); name = NULL;
  free (start); start = NULL;
  free (stop); stop = NULL;

  if (!new) {
    /* who knows what happend, better just get out of here! */
    goto ERROR;
  }

  return new;

 ERROR:
  /* clean up and get the hell out of here */
  free (name);
  free (start);
  free (stop);
  free (fontname);
  if (font) gdk_font_unref (font);
  g_free (fore);
  g_free (back);
  PROPAGATE();			/* propagate error */
}

GList*
get_stentries (GtkEditor *editor, jmp_buf *jumpenv, SCM list)
{
  GList *entries = NULL;
  GList *tmp = NULL, *new = NULL;
  int jumpstate;
  jmp_buf env;
  SCM next;

  /* setup jumppoint */
  SETJMP();

  /* first case is special case */
  if (!gh_null_p (list)) {
    next = gh_car (list);

    new = get_stentry (editor, &env, next);
    entries = new;
    tmp = new;

    list = gh_cdr (list);
  }

  while (!gh_null_p (list)) {
    next = gh_car (list);

    new = get_stentry (editor, &env, next);
    tmp->next = new;
    tmp = new;

    list = gh_cdr (list);
  }

  return entries;

 ERROR:
  /* bad news, clean up and resume error handling */
  gtk_editor_free_stentries (entries);
  PROPAGATE();			/* propagate error */
}

GList*
get_pentry (GtkEditor *editor, jmp_buf *jumpenv, SCM list)
{
  GList *new = NULL;
  char *name = NULL, *pattern = NULL;
  GdkColor *fore = NULL, *back = NULL;
  char *fontname = NULL;
  GdkFont *font = NULL;
  SCM sname, spattern;
  SCM sfont, sfore, sback;
  int jumpstate = JUMPSTATE_MISC_ERROR;
  jmp_buf env;

  TRY (gh_list_p (list), JUMPSTATE_WRONG_TYPE);
  TRY ((gh_length (list) == 5), JUMPSTATE_WRONG_TYPE);

  /* get the stuff */
  sname = gh_car (list); list = gh_cdr (list);
  spattern = gh_car (list); list = gh_cdr (list);
  sfont = gh_car (list); list = gh_cdr (list);
  sfore = gh_car (list); list = gh_cdr (list);
  sback = gh_car (list);

  TRY (gh_string_p (sname), JUMPSTATE_WRONG_TYPE);
  name = gh_scm2newstr (sname, NULL);

  TRY (gh_string_p (spattern), JUMPSTATE_WRONG_TYPE);
  pattern = gh_scm2newstr (spattern, NULL);

  TRY (gh_string_p (sfont), JUMPSTATE_WRONG_TYPE);
  fontname = gh_scm2newstr (sfont, NULL);
  font = gdk_font_load (fontname);
  free (fontname); fontname = NULL;

  /* get ready for jump */
  SETJMP();
  fore = get_color (editor, &env, sfore);
  back = get_color (editor, &env, sback);

  /* finally, get the entry */
  new =  gtk_editor_pentry_new (name, pattern,
				font, fore, back, NULL);

  /* The strings has been copied, so we must free them here. */
  free (name); name = NULL;
  free (pattern); pattern = NULL;

  if (!new) {
    /* who knows what happend, better just get out of here! */
    goto ERROR;
  }

  return new;

 ERROR:
  /* clean up and get the hell out of here */
  free (name);
  free (pattern);
  free (fontname);
  if (font) gdk_font_unref (font);
  g_free (fore);
  g_free (back);
  PROPAGATE();			/* propagate error */
}


GList*
get_pentries (GtkEditor *editor, jmp_buf *jumpenv, SCM list)
{
  GList *entries = NULL;
  GList *tmp = NULL, *new = NULL;
  int jumpstate;
  jmp_buf env;
  SCM next;

  /* setup jumppoint */
  SETJMP();

  /* first case is special case */
  if (!gh_null_p (list)) {
    next = gh_car (list);

    new = get_pentry (editor, &env, next);
    entries = new;
    tmp = new;

    list = gh_cdr (list);
  }

  while (!gh_null_p (list)) {
    next = gh_car (list);

    new = get_pentry (editor, &env, next);
    tmp->next = new;
    tmp = new;

    list = gh_cdr (list);
  }

  return entries;

 ERROR:
  /* bad news, clean up and resume error handling */
  gtk_editor_free_pentries (entries);
  PROPAGATE();			/* propagate error */
}


/* ==<exported functions>================================================= */
/* --<hiliting>----------------------------------------------------------- */
SCM
install_stable_wrapper (SCM editor, SCM entries)
{
  GtkEditor *buffer;
  GList *stentries;
  guint edhandle;
  jmp_buf env;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-install-stable wrong type",
			1, editor);
  }

  if (!gh_list_p (entries)) {
    scm_wrong_type_arg ("gtk-editor-install-stable wrong type",
			2, entries);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-install-stable",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-install-stable",
		    "illegal editor handle", editor);
  }

  /* we've got the editor widget now */
  gh_defer_ints ();
  if (setjmp(env) == 0) {
    stentries = get_stentries (buffer, &env, entries);

    /* if we get this far, we're safe to install */
    gtk_editor_install_stable (buffer, stentries);
    gtk_editor_free_stentries (stentries);
    gh_allow_ints ();

  } else {
    /* misc error */
    gh_allow_ints ();
    scm_misc_error ("gtk-editor-install-stable", "illegal entry",
		    editor);
  }

  return SCM_UNDEFINED;
}

SCM
install_patterns_wrapper (SCM editor, SCM entries)
{
  GtkEditor *buffer;
  GList *pentries;
  guint edhandle;
  jmp_buf env;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-install-patterns wrong type",
			1, editor);
  }

  if (!gh_list_p (entries)) {
    scm_wrong_type_arg ("gtk-editor-install-patterns wrong type",
			2, entries);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-install-patterns",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-install-patterns",
		    "illegal editor handle", editor);
  }

  /* we've got the editor widget now */
  gh_defer_ints ();
  if (setjmp(env) == 0) {
    pentries = get_pentries (buffer, &env, entries);

    /* if we get this far, we're safe to install */
    gtk_editor_install_patterns (buffer, pentries);
    gtk_editor_free_pentries (pentries);
    gh_allow_ints ();

  } else {
    /* misc error */
    gh_allow_ints ();
    scm_misc_error ("gtk-editor-install-patterns", "illegal entry",
		    editor);
  }

  return SCM_UNDEFINED;
}

/* --<searching>---------------------------------------------------------- */
SCM
search_from_point_wrapper (SCM editor, SCM sstring, SCM scasein)
{
  int edhandle;
  GtkEditor *buffer;
  char *string;
  gboolean casein;

  gboolean result;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-search-from-point wrong type",
			1, editor);
  }

  if (!gh_string_p (sstring)) {
    scm_wrong_type_arg ("gtk-editor-search-from-point wrong type", 2,
			sstring);
  }

  if (!gh_boolean_p (scasein)) {
    scm_wrong_type_arg ("gtk-editor-search-from-point wrong type",
			3, scasein);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-search-from-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-search-from-point",
		    "illegal editor handle", editor);
  }

  /* since there is *no* kind of consistency between point and
   * position we need to do this */
  gtk_sctext_set_point (GTK_SCTEXT (buffer),
		      gtk_editable_get_position (GTK_EDITABLE (buffer)));

  gh_defer_ints ();
  string = gh_scm2newstr (sstring, NULL);
  casein = gh_scm2bool (scasein);
  result = gtk_editor_search_from_point (buffer, string, casein);
  free (string);
  gh_allow_ints ();

  return gh_bool2scm (result);
}

SCM
search_back_from_point_wrapper (SCM editor, SCM sstring, SCM scasein)
{
  int edhandle;
  GtkEditor *buffer;
  char *string;
  gboolean casein;

  gboolean result;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-search-back-from-point wrong type",
			1, editor);
  }

  if (!gh_string_p (sstring)) {
    scm_wrong_type_arg ("gtk-editor-search-back-from-point wrong type",
			2, sstring);
  }

  if (!gh_boolean_p (scasein)) {
    scm_wrong_type_arg ("gtk-editor-search-back-from-point wrong type",
			3, scasein);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-search-back-from-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-search-back-from-point",
		    "illegal editor handle", editor);
  }

  /* since there is *no* kind of consistency between point and
   * position we need to do this */
  gtk_sctext_set_point (GTK_SCTEXT (buffer),
		      gtk_editable_get_position (GTK_EDITABLE (buffer)));

  gh_defer_ints ();
  string = gh_scm2newstr (sstring, NULL);
  casein = gh_scm2bool (scasein);
  result = gtk_editor_search_back_from_point (buffer, string, casein);
  free (string);
  gh_allow_ints ();

  return gh_bool2scm (result);
}

SCM
regex_search_from_point_wrapper (SCM editor, SCM spattern)
{
  int edhandle;
  GtkEditor *buffer;
  char *pattern;

  gboolean result;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-regex-search-from-point wrong type",
			1, editor);
  }

  if (!gh_string_p (spattern)) {
    scm_wrong_type_arg ("gtk-editor-regex-search-from-point wrong type",
			2, spattern);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-regex-search-from-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-regex-search-from-point",
		    "illegal editor handle", editor);
  }

  /* since there is *no* kind of consistency between point and
   * position we need to do this */
  gtk_sctext_set_point (GTK_SCTEXT (buffer),
		      gtk_editable_get_position (GTK_EDITABLE (buffer)));


  gh_defer_ints ();
  pattern = gh_scm2newstr (spattern, NULL);
  result = gtk_editor_regex_search_from_point (buffer, pattern);
  free (pattern);
  gh_allow_ints ();

  return gh_bool2scm (result);
}

SCM
regex_search_back_from_point_wrapper (SCM editor, SCM spattern)
{
  int edhandle;
  GtkEditor *buffer;
  char *pattern;

  gboolean result;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-regex-search-back-from-point wrong type",
			1, editor);
  }

  if (!gh_string_p (spattern)) {
    scm_wrong_type_arg ("gtk-editor-regex-search-back-from-point wrong type",
			2, spattern);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-regex-search-back-from-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-regex-search-back-from-point",
		    "illegal editor handle", editor);
  }

  /* since there is *no* kind of consistency between point and
   * position we need to do this */
  gtk_sctext_set_point (GTK_SCTEXT (buffer),
		      gtk_editable_get_position (GTK_EDITABLE (buffer)));


  gh_defer_ints ();
  pattern = gh_scm2newstr (spattern, NULL);
  result = gtk_editor_regex_search_back_from_point (buffer, pattern);
  free (pattern);
  gh_allow_ints ();

  return gh_bool2scm (result);
}

/* --<editing>------------------------------------------------------------ */
SCM
select_wrapper (SCM editor, SCM sfrom, SCM sto)
{
  int edhandle;
  GtkEditor *buffer;
  int from, to;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-select-region wrong type",
			1, editor);
  }

  if (!gh_exact_p (sfrom)) {
    scm_wrong_type_arg ("gtk-editor-select-region wrong type",
			2, sfrom);
  }

  if (!gh_exact_p (sto)) {
    scm_wrong_type_arg ("gtk-editor-select-region wrong type",
			3, sto);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-select-region",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-select-region",
		    "illegal editor handle", editor);
  }
  
  from = gh_scm2int (sfrom);
  to = gh_scm2int (sto);

  gtk_editable_select_region (GTK_EDITABLE (buffer), from, to);

  return SCM_UNDEFINED;
}

SCM
delete_selection_wrapper (SCM editor)
{
  int edhandle;
  GtkEditor *buffer;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-delete-selection wrong type",
			1, editor);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-delete-selection",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-delete-selection",
		    "illegal editor handle", editor);
  }

  gtk_editable_delete_selection (GTK_EDITABLE (buffer));
  
  return SCM_UNDEFINED;
}

SCM
get_string_wrapper (SCM editor, SCM sfrom, SCM sto)
{
  int edhandle;
  GtkEditor *buffer;
  int from, to;
  char *string;
  SCM result;
  
  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-get-string wrong type",
			1, editor);
  }

  if (!gh_exact_p (sfrom)) {
    scm_wrong_type_arg ("gtk-editor-get-string wrong type",
			2, sfrom);
  }

  if (!gh_exact_p (sto)) {
    scm_wrong_type_arg ("gtk-editor-get-string wrong type",
			3, sto);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-get-string",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-get-string",
		    "illegal editor handle", editor);
  }
  
  from = gh_scm2int (sfrom);
  to = gh_scm2int (sto);

  gh_defer_ints ();
  string = gtk_editable_get_chars (GTK_EDITABLE (buffer), from, to);
  result = gh_str02scm (string);
  g_free (string);
  gh_allow_ints ();

  return result;
}

SCM
get_selection_wrapper (SCM editor)
{
  int edhandle;
  GtkEditor *buffer;
  char *string;
  SCM result;
  
  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-get-selection wrong type",
			1, editor);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-get-selection",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-get-selection",
		    "illegal editor handle", editor);
  }
  
  gh_defer_ints ();
  if (GTK_EDITABLE (buffer)->has_selection) {
    string = gtk_editable_get_chars (GTK_EDITABLE (buffer),
				     GTK_EDITABLE (buffer)->selection_start_pos,
				     GTK_EDITABLE (buffer)->selection_end_pos);
    result = gh_str02scm (string);
    g_free (string);
  } else {
    result = gh_str02scm ("");
  }
  gh_allow_ints ();

  return result;
}


SCM
insert_string_wrapper (SCM editor, SCM sstring, SCM spos)
{
  int edhandle;
  GtkEditor *buffer;
  char *string;
  gint pos;
  
  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-insert-string wrong type",
			1, editor);
  }

  if (!gh_string_p (sstring)) {
    scm_wrong_type_arg ("gtk-editor-insert-string wrong type",
			2, sstring);
  }

  if (!gh_exact_p (spos)) {
    scm_wrong_type_arg ("gtk-editor-insert-string wrong type",
			3, spos);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-insert-string",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-insert-string",
		    "illegal editor handle", editor);
  }

  pos = gh_scm2int (spos);
  if (pos > gtk_sctext_get_length (GTK_SCTEXT (buffer))) {
    scm_misc_error ("gtk-editor-insert-string",
		    "point out of bound", spos);
  }
  
  gh_defer_ints ();
  string = gh_scm2newstr (sstring, NULL);
  gtk_editable_insert_text (GTK_EDITABLE (buffer), string,
			    strlen (string), &pos);
  gtk_editable_set_position (GTK_EDITABLE (buffer),
			     pos + gtk_editable_get_position (GTK_EDITABLE (buffer)));
  free (string);
  gh_allow_ints ();

  return SCM_UNDEFINED;
}

SCM
insert_string_at_point_wrapper (SCM editor, SCM sstring)
{
  int edhandle;
  GtkEditor *buffer;
  char *string;
  gint pos;
  
  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-insert-string-at-point wrong type",
			1, editor);
  }

  if (!gh_string_p (sstring)) {
    scm_wrong_type_arg ("gtk-editor-insert-string-at-point wrong type",
			2, sstring);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-insert-string-at-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-insert-string-at-point",
		    "illegal editor handle", editor);
  }

  pos = gtk_editable_get_position (GTK_EDITABLE (buffer));
  
  gh_defer_ints ();
  string = gh_scm2newstr (sstring, NULL);

  gtk_editable_insert_text (GTK_EDITABLE (buffer), string,
			    strlen (string), &pos);

  free (string);
  gh_allow_ints ();

  return SCM_UNDEFINED;
}


/* --<movement>----------------------------------------------------------- */
SCM
set_point_wrapper (SCM editor, SCM spos)
{
  int edhandle;
  GtkEditor *buffer;
  int pos;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-set-point wrong type",
			1, editor);
  }

  if (!gh_exact_p (spos)) {
    scm_wrong_type_arg ("gtk-editor-set-point wrong type",
			2, spos);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-set-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-set-point",
		    "illegal editor handle", editor);
  }
  
  pos = gh_scm2int (spos);
  gtk_editable_set_position (GTK_EDITABLE (buffer), pos);

  return SCM_UNDEFINED;
}

SCM
get_point_wrapper (SCM editor)
{
  int edhandle;
  GtkEditor *buffer;

  /* check types */
  if (!gh_exact_p (editor)) {
    scm_wrong_type_arg ("gtk-editor-get-point wrong type",
			1, editor);
  }

  /* make sure editor is legal handle */
  edhandle = gh_scm2int (editor);
  if (!IS_LEGAL_IDX(edhandle)) {
    scm_misc_error ("gtk-editor-get-point",
		    "illegal editor handle", editor);
  }

  /* get editor */
  buffer = EdHandles.handles[edhandle];
  if (IN_FREELIST (buffer)) {
    scm_misc_error ("gtk-editor-get-point",
		    "illegal editor handle", editor);
  }
  
  return gh_int2scm (gtk_editable_get_position (GTK_EDITABLE (buffer)));
}


/* ==<install functions>================================================== */
void
gtk_editor_guile_install_hilite_functions (void)
{
  gh_new_procedure2_0 ("gtk-editor-install-stable", install_stable_wrapper);
  gh_new_procedure2_0 ("gtk-editor-install-patterns", install_patterns_wrapper);
}

void
gtk_editor_guile_install_search_functions (void)
{
  gh_new_procedure3_0 ("gtk-editor-search-from-point",
		       search_from_point_wrapper);
  gh_new_procedure3_0 ("gtk-editor-search-back-from-point", 
		       search_back_from_point_wrapper);
  gh_new_procedure2_0 ("gtk-editor-regex-search-from-point",
		       regex_search_from_point_wrapper);
  gh_new_procedure2_0 ("gtk-editor-regex-search-back-from-point",
		       regex_search_back_from_point_wrapper);
}

void
gtk_editor_guile_install_editing_functions (void)
{
  gh_new_procedure3_0 ("gtk-editor-select-region",
		       select_wrapper);
  gh_new_procedure1_0 ("gtk-editor-delete-selection",
		       delete_selection_wrapper);
  gh_new_procedure3_0 ("gtk-editor-get-string",
		       get_string_wrapper);
  gh_new_procedure1_0 ("gtk-editor-get-selection",
		       get_selection_wrapper);
  gh_new_procedure3_0 ("gtk-editor-insert-string",
		       insert_string_wrapper);
  gh_new_procedure2_0 ("gtk-editor-insert-string-at-point",
		       insert_string_at_point_wrapper);
}

void
gtk_editor_guile_install_movement_functions (void)
{
  gh_new_procedure2_0 ("gtk-editor-set-point", set_point_wrapper);
  gh_new_procedure1_0 ("gtk-editor-get-point", get_point_wrapper);
}

/* ==<init>=============================================================== */
/* guile_init initializes datastructures needed for interfacing to
 * guile.  It does *not* start up guile nor install any functions to
 * guile.  This has to be done in the application using the editor
 * widget.  This function should be called *before* any other function
 * in gtkeditor-guile!
 */
void
gtk_editor_guile_init (void)
{
  /* editor-handle managment */
  EdHandles.handles = g_new (GtkEditor*, INIT_HANDLES_SIZE);
  EdHandles.handles[0] = NULL;	/* no default yet! */
  EdHandles.size = INIT_HANDLES_SIZE;
  EdHandles.used = 1;		/* the default is always "used" */
  EdHandles.freelist = NULL;
}
