/* $Header: /cvs/gnome/gIDE/src/gI_file.c,v 1.5 1999/12/05 07:17:51 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>
#include <gnome.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#include "structs.h"
#include "gide.h"
#include "gI_document.h"
#include "gI_file.h"
#include "gI_window.h"
#include "gI_compile.h"
#include "gI_common.h"
#include "gI_edit.h"
#include "gI_menus.h"
#include "gI_project.h"

#ifdef HAVE_GTKTEXT_PATCH
#include <gtksctext.h>
#include <gtkeditor/gtkeditor.h>
#include "gI_globpatterns.h"
#endif


/* globals */
static glong noc;
static GtkWidget *print_window = NULL;
static GtkWidget *e_print_other;
static glong print_lpr = TRUE;
static glong file_exit_save = FALSE;
static glong file_close_save = FALSE;
static GtkWidget *r_print_all;
static GtkWidget *r_print_sel;
glong   ro_not_change = 0;

/* externs */
extern gI_window *main_window;
extern gI_config *cfg;
extern gchar gide_path[];

#ifdef HAVE_GTKTEXT_PATCH
/* declared in gtkeditor-regex.h and syntaxtable.h */
void            _gtk_editor_destroy_patterns   (GtkEditorHilitePatterns *patterns);
void            _gtk_editor_destroy_stable (GtkEditorSyntaxTable *stable);
#endif


/*
 ---------------------------------------------------------------------
     Function: file_strip_name()
     Desc: strip path from filename (unix-style)
 ---------------------------------------------------------------------
*/
gchar *file_strip_name( gchar *filename )
{
	gchar *striped = filename;
	gint i;

    	if( !filename )
        	return( NULL );
    	if( strlen( filename ) == 0 )
        	return( NULL );
    
    	for(i=strlen(filename);i>=0;i--)
	{
		if( filename[i] == '/' )
        	{
            		striped = &filename[i+1];
            		break;
        	}
	}

	return( striped );
}


/*
 ---------------------------------------------------------------------
     Function: file_new()
     Desc: Callback-Function /File/New
 ---------------------------------------------------------------------
*/
void file_new( GtkWidget *widget, gpointer data )
{
	gchar HelpString[MAXLEN];
    	gchar *ptr;
    
    	if( cfg->use_redit )
    	{
        	strcpy( HelpString, cfg->redit );
        	ptr = strstr( HelpString, "%s" );
        	if( ptr )
        	{
            		*ptr = ' ';
            		ptr++;
            		*ptr = ' ';
        	}

        	sprintf( HelpString, "%s &", HelpString );
        	system( HelpString );
    	}
    	else
    	{
        	gI_document_new( main_window );
    	}
}


/*
 ---------------------------------------------------------------------
     Function: file_open()
     Desc: Callback-Function /File/Open
 ---------------------------------------------------------------------
*/
void file_open( GtkWidget *widget, gpointer data )
{
	GList *files = gI_file_sel_new( _("Open File..."), FALSE, TRUE );

	while( files )
	{
		gchar *filename = (gchar *) files->data;

    		file_open_by_name( main_window, filename );

		files = g_list_next( files );
	}

	gI_file_sel_free_list( files );
}


void file_open_by_name( gI_window *window, gchar *filename )
{
    	FILE *file;
    	gchar buf[STRLEN];
    	gI_document *document;
    	gchar HelpString[STRLEN];
    	gchar *ptr;

    	/* only open the file, if its not already opened */
    	if( check_file_open( filename ) )
    	{
		goto_file( filename );
		return;
    	}

    	/* bullshit below, better ignore it */
    	if( cfg->use_redit )
    	{
        	strcpy( HelpString, cfg->redit );
        	ptr = strstr( HelpString, "%s" );
        	if( ptr )
        	{
            		*ptr = '\0';
            		strcat( HelpString, filename );
            		ptr = strstr( cfg->redit, "%s" );
            		ptr++;ptr++;
            		strcat( HelpString, ptr );
        	}
       		strcat( HelpString, " &" );
        	system( HelpString );
        	return;
    	}
    
    	file = fopen( filename, "r" );
    	if( file == NULL )
    	{
        	g_snprintf( buf, STRLEN, _("Unable to Open '%s'"), filename );
        	gI_error_dialog( buf );
        	return;
    	}


    	document = gI_document_get_current( window );
    	if( document )
    	{
        	if( document->filename != NULL || document->changed )
        	{
            		document = gI_document_new( window );
        	}
    	}
    	else
    	{
		document = gI_document_new( window );
    	}
   
    	gI_text_freeze( document->text );

    	while( fgets( buf, sizeof(buf), file ) )
    	{
        	gI_text_insert( document->text, NULL, NULL, NULL,
                         	 buf, strlen( buf ) );
    	}

    	gI_text_thaw( document->text );

    	fclose( file );

#ifdef HAVE_GTKTEXT_PATCH
	if( cfg->highlight )
      	{
		gI_HilitePattern *pat = gI_glob_lookup( g_basename (filename) );
		if (pat)
	  	{
	    		/* If it fails...we just return to default pattern... 
	     		 * so there's no reson to test for return value here. */
	    		gI_hilite_install_pattern( GTK_EDITOR( document->text ),
				      pat );
	  	}
		gtk_editor_hilite_buffer( GTK_EDITOR( document->text ) );
      	}
    	else
      	{
		gtk_editor_hilite_when_idle( GTK_EDITOR( document->text ), FALSE );
	}
#endif

    	document->filename = (gchar *) realloc( document->filename, (strlen( filename ) + 1) * sizeof( gchar ) );
    	strcpy( document->filename, filename );
    	if( strstr( filename, "/" ) )
	{
		gtk_label_set_text( GTK_LABEL( document->label ), file_strip_name( filename ) );
	}
	else
	{
		gtk_label_set_text( GTK_LABEL( document->label ), filename );
	}

	document->last_mod = get_last_mod( filename );

    	/*gI_document_set_unchanged( document );*/

    	if( !get_write_stat( document->filename ) )
    	{
        	document->read_only = 1;

        	/* disconnect signal from menu item */

        	/* gtk_signal_disconnect_by_func( GTK_OBJECT( menus_get_item_widget( "/Edit/Read Only" ) ), GTK_SIGNAL_FUNC( edit_read_only ), NULL ); */

        	gI_window_set_visible_read_only( TRUE );

        	ro_not_change = 1;

        	/* set check menu item state & correct editable flag in widget */
        	gtk_check_menu_item_set_state( GTK_CHECK_MENU_ITEM( menus_get_item_widget( "/Edit/Read Only" ) ), document->read_only );
        	if( document->read_only )
		{
			gI_text_set_editable( document->text, FALSE );
		}
		else
		{
			gI_text_set_editable( document->text, TRUE );
		}
        	ro_not_change = 0;

        	/* gtk_signal_connect( GTK_OBJECT( menus_get_item_widget( "/Edit/Read Only" ) ), "activate", GTK_SIGNAL_FUNC( edit_read_only ), NULL ); */
    	} 

    	gI_window_set_statusbar( document );
}


void file_open_cancel( GtkWidget *widget, GtkFileSelection *sel )
{
	if( GTK_WIDGET_VISIBLE( sel ) )
        	gtk_widget_hide( GTK_WIDGET( sel ) );
}


/*
 ---------------------------------------------------------------------
     Function: file_reload()
     Desc: Callback-Function /File/Reload
 ---------------------------------------------------------------------
*/
void file_reload( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	FILE *file;
	gchar buf[STRLEN];
	gint pos;

	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

    	if( current->filename == NULL )
        	return;
    
    	if( current->changed == TRUE && (gint) data != 1 )
    	{
		if( gI_ask_dialog( _("The file has been changed,\nDo You want to reload it?")) != 0 )  /* 0 == GNOME_YES */
		{
	        	return;
		}
    	}

    	pos = gI_text_get_point( current->text );
    
    	gI_document_clear_text( current );

    	file = fopen( current->filename, "r" );
    	if( file == NULL )
    	{
        	g_snprintf( buf, STRLEN, _("Unable to Open '%s'"), current->filename );
        	gI_error_dialog( buf);
        	return; 
    	}
    
        gI_text_freeze( current->text );

    	while( fgets( buf, 1024, file ) )
	{
		gI_text_insert( current->text, NULL, NULL, NULL,
                         buf, strlen( buf ) );
	}

	fclose( file );

#ifdef HAVE_GTKTEXT_PATCH
    	if( cfg->highlight )
    	{
        	gtk_editor_hilite_buffer( GTK_EDITOR( current->text ) );
   	}
#endif

    	gI_text_thaw( current->text );

    	if( gI_text_get_length( current->text ) >= pos )
        	gI_text_set_point( current->text, pos );

    	gI_document_set_unchanged( current );

    	gI_window_set_statusbar( current );

    	gtk_widget_grab_focus( current->text );

	/* update last mod time */
	current->last_mod = get_last_mod( current->filename );
}


/*
 ---------------------------------------------------------------------
     Function: file_save()
     Desc: Callback-Function /File/Save
 ---------------------------------------------------------------------
*/
void file_save( GtkWidget *widget, gpointer data )
{
	FILE *file;
	gI_document *current;
    	glong length,i,current_no;
	gchar *c;

    	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

    	if( current->filename == NULL )
    	{
        	file_save_as( widget, data );
        	return;
    	}
    
    	file = fopen( current->filename, "w" );
    	if( file == NULL )
    	{
        	gI_error_dialog( _("Unable to Save File") );
        	return;
    	}

    	length = gI_text_get_length( current->text );
    	for(i=1;i<=length;i++)
	{
        	c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
        	fputc( c[0], file );
		g_free( c );
	}

    	fclose( file );

	current->last_mod = get_last_mod( current->filename );

    	gI_document_set_unchanged( current );

    	gI_window_set_statusbar( current );

    	if( file_exit_save )
    	{
        	if( gI_document_get_no( main_window )-1 == 0 )
            		gtk_main_quit();
    	}
    	if( file_close_save )
    	{
        	current_no = gtk_notebook_current_page( GTK_NOTEBOOK( main_window->notebook ) );

        	gtk_notebook_remove_page( GTK_NOTEBOOK( main_window->notebook ), current_no );
        	g_list_remove( main_window->documents, current );

    		free_op_list( current->op_list );

		/* free memory */
		g_free( current->filename );
		g_free( current );	
    	}
}


/*
 ---------------------------------------------------------------------
     Function: file_save_as()
     Desc: Callback-Function /File/Save As
 ---------------------------------------------------------------------
*/
void file_save_as( GtkWidget *widget, gpointer data )
{
	GList *files = gI_file_sel_new( _("Save As..."), TRUE, FALSE );
	gI_document *current = gI_document_get_current( main_window );

	if( files != NULL )
	{
		gchar *filename = files->data;
		if( !filename ) 
		{
			return;
		}

        	current->filename = (gchar *) realloc( current->filename, (strlen( filename ) + 1) * sizeof( gchar ) );
		strcpy( current->filename, filename );
	        file_save_by_name( main_window, filename );
	}
	else
	{
		return;
	}	

	gI_file_sel_free_list( files );

	if( file_exit_save )
	{
		if( gI_document_get_no( main_window ) == 0 )
			gtk_main_quit();
	}

	if( file_close_save )
	{
		glong current_no = gtk_notebook_get_current_page( GTK_NOTEBOOK( main_window->notebook ) );

		gtk_notebook_remove_page( GTK_NOTEBOOK( main_window->notebook ), current_no );
		g_list_remove( main_window->documents, current );

		free_op_list( current->op_list );

		/* free memory */
		g_free( current->filename );
		g_free( current );	
	}
}


static void file_save_overwrite( GtkWidget *widget, GtkWidget *dialog )
{
	gI_document *current;
	gchar *filename;
	glong length, i;
	FILE *file;
	gchar *c;

	if( dialog )
	{
		gtk_widget_destroy( dialog );
	}

	current = gI_document_get_current( main_window );
	g_return_if_fail( current != NULL );

	filename = current->filename;

	file = fopen( filename, "w" );
	if( file == NULL )
	{
		gI_error_dialog( _("Unable to Save File") );
		return;
    	}
    
    	length = gI_text_get_length( current->text );

    	for(i=1;i<=length;i++)
	{
        	c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
        	fputc( c[0], file );
		g_free( c );
	}

    	fclose( file );

	current->last_mod = get_last_mod( current->filename );

    	/*
	    current->changed = FALSE;
	    current->changed_id = gtk_signal_connect( GTK_OBJECT( current->text ), "changed",
                                              GTK_SIGNAL_FUNC( gI_document_changed ), current );
	    current->label = gtk_label_new( current->filename );
	*/

    	gI_document_set_unchanged( current );

    	gI_window_set_statusbar( current );
}


static void file_save_abort( GtkWidget *widget, GtkWidget *dialog )
{
	if( dialog )
	{
		gtk_widget_destroy( dialog );

	}
}


void file_save_by_name( gI_window *window, gchar *filename )
{
	FILE *file;
	gI_document *current;
	glong length,i;
	gchar warnmsg[STRLEN];
	gchar *c;
	gint btn;

	if( file_exist( filename ) != 0 )
	{
		/* overwrite dialog */
		sprintf( warnmsg, _("The File %s already exists, do you really want to overwrite it?"), filename );
	        btn = gI_ask_dialog( warnmsg);
        
        	switch (btn)
		{
			/* YES */
			case 0:
		    	{
				file_save_overwrite( NULL, NULL );
			    	break;
			}

			/* NO */
	    		case 1:
			{
				file_save_abort( NULL, NULL );
		    		break;
			}	

			/* CANCEL/CLOSE */
	    		case 2:
	    		case -1:
			{
				break;
			}
		}
	
		return;
	}

	file = fopen( filename, "w" );
	if( file == NULL )
    	{
        	gI_error_dialog( _("Unable to Save File") );
        	return;
    	}
    
    	current = gI_document_get_current( window );
	g_return_if_fail( current != NULL );	

	length = gI_text_get_length( current->text );
    	for(i=1;i<=length;i++)
	{
        	c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
	        fputc( c[0], file );
		g_free( c );
	}

	fclose( file );

	/* correct last modification time after save */
	current->last_mod = get_last_mod( current->filename );	

	/*
	current->changed = FALSE;
	current->changed_id = gtk_signal_connect( GTK_OBJECT( current->text ), "changed",
                                              GTK_SIGNAL_FUNC( gI_document_changed ), current );
	current->label = gtk_label_new( current->filename );
	*/

#ifdef HAVE_GTKTEXT_PATCH
	if( cfg->highlight )
        {
	        gI_HilitePattern *pat = gI_glob_lookup( g_basename (filename) );
                if (pat)
	        {
		        /* If it fails...we just return to default pattern...
	 		 * so there's no reson to test for return value here. */
                        gI_hilite_install_pattern( GTK_EDITOR( current->text ), pat );
		}
		else
		{
			/* syntaxtable.h says, that i should not use this functions, so
			 * is there another legal way to just remove the syntax table and the
			 * patterns ??? */
			_gtk_editor_destroy_stable( GTK_EDITOR( current->text )->stable );
			_gtk_editor_destroy_patterns( GTK_EDITOR( current->text )->patterns );
			GTK_EDITOR( current->text )->stable = NULL;
			GTK_EDITOR( current->text )->patterns = NULL;
		}

		gtk_editor_hilite_buffer( GTK_EDITOR( current->text ) );
       }
       else
       {
	       gtk_editor_hilite_when_idle( GTK_EDITOR( current->text ), FALSE );
       }
#endif

    	gI_document_set_unchanged( current );

    	gI_window_set_statusbar( current );
}


void file_save_cancel( GtkWidget *widget, GtkFileSelection *sel )
{
	glong current_no;
	gI_document *current;

	current = gI_document_get_current( main_window );
	g_return_if_fail( current != NULL );

	if( GTK_WIDGET_VISIBLE( sel ) )
	{
		gtk_widget_hide( GTK_WIDGET( sel ) );
	}

	if( file_exit_save )
	{
		if( gI_document_get_no( main_window ) == 0 )
            		gtk_main_quit();
    	}

	if( file_close_save )
	{
        	current_no = gtk_notebook_get_current_page( GTK_NOTEBOOK( main_window->notebook ) );

        	gtk_notebook_remove_page( GTK_NOTEBOOK( main_window->notebook ), current_no );
        	g_list_remove( main_window->documents, current );

    		free_op_list( current->op_list );

		/* free memory */
		g_free( current->filename );
		g_free( current );	
    	}
}


static void print_destroy( GtkWidget *widget, gpointer data )
{
	gtk_widget_destroy( print_window );
	print_window = NULL;
}


static void print_show_lpr( GtkWidget *widget, gpointer data )
{
	gtk_widget_set_sensitive( e_print_other, FALSE );
	print_lpr = TRUE;
}


static void print_show_other( GtkWidget *widget, gpointer data )
{
	gtk_widget_set_sensitive( e_print_other, TRUE );
	print_lpr = FALSE;
}


static void _file_print( GtkWidget *widget, gpointer data )
{
	FILE *file;
	gI_document *current;
	glong length,i;
	gchar filename[MAXLEN];
	gchar tmpstring[MAXLEN];
	gchar _tmpstring[MAXLEN];
	gchar *ptr;
	gchar *var_ptr;
	gchar *c;

	current = gI_document_get_current( main_window );
	g_return_if_fail( current != NULL );

	if( print_lpr )
    	{
        	/* print using lpr */
        	if( current->changed || current->filename == NULL || GTK_TOGGLE_BUTTON( r_print_sel )->active )
        	{
            		/* strcat still vuln. to overflows, write ANSI-comp. (SK_)strncat */
            		g_snprintf( filename, MAXLEN, "%s/", cfg->tmpdir );
            		if( current->filename != NULL )
                		strcat( filename, file_strip_name( current->filename ) );

			g_snprintf( tmpstring, MAXLEN, ".print.%d%d", (int) time(0), (int)getpid() );
            		strcat( filename, tmpstring );

            		file = fopen( filename, "w" );
            		if( file == NULL )
            		{
                		/* error dialog */
                		g_snprintf( tmpstring, MAXLEN, _("Unable to Open '%s'"), filename );
                		gI_error_dialog( tmpstring);
                		return;
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_sel )->active )
            		{
                		if( !GTK_EDITABLE( current->text )->has_selection )
                		{
                    			gI_error_dialog( _("No selection in current document!") );
                    			return;
                		}
                
                		for(i=GTK_EDITABLE( current->text )->selection_start_pos+1;
                    		    i<=GTK_EDITABLE( current->text )->selection_end_pos;
                    		    i++)
				{
                    			c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
					fputc( c[0], file );
					g_free( c );
				}
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_all )->active )
            		{
                		length = gI_text_get_length( current->text );

    				for(i=1;i<=length;i++)
				{
			        	c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
			        	fputc( c[0], file );
					g_free( c );
				}
            		}

            		fclose( file );

            		g_snprintf( tmpstring, MAXLEN, "lpr %s", filename );
            		system( tmpstring );

            		remove( filename );
       		}
        	else
        	{
            		g_snprintf( tmpstring, MAXLEN, "lpr %s", current->filename );
            		system( tmpstring );
        	}

		print_destroy( NULL, NULL );
    	}
    	else
    	{
        	/* print using specified command */
        	ptr = gtk_entry_get_text( GTK_ENTRY( e_print_other ) );
        	if( !ptr || isempty( ptr ) )
        	{
            		return;
        	}

       		/* parse for "file variable" */
        	var_ptr = strstr( ptr, "%s" );
        	if( !var_ptr )
        	{
            		return;
        	}

        	*var_ptr = ' ';
        	var_ptr++;
        	*var_ptr = ' ';

        	var_ptr--;

        	/* backup the line */
        	strcpy( _tmpstring, var_ptr );

        	*var_ptr = '\0';

        	if( current->changed || current->filename == NULL )
        	{
            		/* write to temp file */
            		g_snprintf( filename, MAXLEN, "%s/", cfg->tmpdir );
            		g_snprintf( tmpstring, MAXLEN, ".print.%d%d", (int) time(0), (int) getpid() );
            		strcat( filename, tmpstring );

            		file = fopen( filename, "w" );
            		if( file == NULL )
            		{
                		/* error dialog */
                		g_snprintf( tmpstring, MAXLEN, _("Unable to Open '%s'"), filename );
                		gI_error_dialog( tmpstring);
                		return;
            		}

            		if( GTK_TOGGLE_BUTTON( r_print_all )->active )
            		{
                		length = gI_text_get_length( current->text );

    				for(i=1;i<=length;i++)
				{
			        	c = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
			        	fputc( c[0], file );
					g_free( c );
				}
            		}

            		fclose( file );

            		/* build string */
            		strcpy( tmpstring, ptr );
            		strcat( tmpstring, filename );
            		strcat( tmpstring, _tmpstring );

            		/* execute */
            		system( tmpstring );

            		remove( filename );
        	}
        	else
        	{
            		/* build string */
            		strcpy( tmpstring, ptr );
            		strcat( tmpstring, current->filename );
            		strcat( tmpstring, _tmpstring );

            		/* execute */
            		system( tmpstring );
        	}

        	print_destroy( NULL, NULL );
    	}
}


/*
 ---------------------------------------------------------------------
     Function: file_print()
     Desc: Callback-Function /File/Print
 ---------------------------------------------------------------------
*/
void file_print( GtkWidget *widget, gpointer data )
{
	GtkWidget *hbox;
	GtkWidget *vbox;
	GtkWidget *hsep;
	GtkWidget *button;
	GSList *type_group;
	GtkWidget *label;
 	GtkWidget *r_print_lpr;
    	GtkWidget *r_print_other;
    	GSList *print_group;

    	if( print_window )
        	return;

    	print_window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
    	gtk_widget_set_usize( print_window, 300, 260 );
    	gtk_window_set_title( GTK_WINDOW( print_window ), _("Print...") );
    	gtk_signal_connect( GTK_OBJECT( print_window ), "destroy",
                            GTK_SIGNAL_FUNC( print_destroy ), NULL );

    	vbox = gtk_vbox_new( FALSE, 0 );
    	gtk_container_add( GTK_CONTAINER( print_window ), vbox );
    	gtk_widget_show( vbox );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

    	label = gtk_label_new( _("Print Using: ") );
    	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
    	gtk_widget_show( label );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	r_print_lpr = gtk_radio_button_new_with_label( NULL, "lpr" );
    	type_group = gtk_radio_button_group( GTK_RADIO_BUTTON( r_print_lpr ) );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_lpr, FALSE, TRUE, 5 );
    	gtk_signal_connect( GTK_OBJECT( r_print_lpr ), "clicked",
			    GTK_SIGNAL_FUNC( print_show_lpr ), NULL );
    	gtk_widget_show( r_print_lpr );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	r_print_other = gtk_radio_button_new_with_label( type_group, _("Command") );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_other, FALSE, TRUE, 5 );
    	gtk_signal_connect( GTK_OBJECT( r_print_other ), "clicked",
                            GTK_SIGNAL_FUNC( print_show_other ), NULL );
    	gtk_widget_show( r_print_other );

    	e_print_other = gtk_entry_new_with_max_length( 255 );
    	gtk_box_pack_start( GTK_BOX( hbox ), e_print_other, FALSE, TRUE, 10 );
    	gtk_widget_show( e_print_other );
    	gtk_widget_set_sensitive( e_print_other, FALSE );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

   	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 10 );
    	gtk_widget_show( hbox );

    	label = gtk_label_new( _("Print Area: ") );
    	gtk_box_pack_start( GTK_BOX( hbox ), label, FALSE, TRUE, 5 );
    	gtk_widget_show( label );
    
    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );
    
    	r_print_all = gtk_radio_button_new_with_label( NULL, "all" );
    	print_group = gtk_radio_button_group( GTK_RADIO_BUTTON( r_print_all ) );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_all, FALSE, TRUE, 5 );
    	gtk_widget_show( r_print_all );

    	r_print_sel = gtk_radio_button_new_with_label( print_group, "selection" );
    	gtk_box_pack_start( GTK_BOX( hbox ), r_print_sel, FALSE, TRUE, 10 );
    	gtk_widget_show( r_print_sel );

    	hsep = gtk_hseparator_new();
    	gtk_box_pack_start( GTK_BOX( vbox ), hsep, FALSE, TRUE, 10 );
    	gtk_widget_show( hsep );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	button = gnome_stock_button( GNOME_STOCK_BUTTON_OK );
    	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
                       	    GTK_SIGNAL_FUNC( _file_print ), NULL );
    	gtk_widget_show( button );

    	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
    	gtk_box_pack_start( GTK_BOX( hbox ), button, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
                       	    GTK_SIGNAL_FUNC( print_destroy ), NULL );
    	gtk_widget_show( button );

    	gtk_widget_show( print_window );
}


/*
 ---------------------------------------------------------------------
     Function: file_close()
     Desc: Callback-Function /File/Close
 ---------------------------------------------------------------------
*/
void file_close( GtkWidget *widget, gpointer data)
{
	gI_document *current;
	gint current_no;

	current = gI_document_get_current( main_window );
	if( !current )
	{
		return;
	}

    	if( current->changed )
    	{
		if( !file_close_changed_dialog_modal(current) )  /* Return false means dont close until */ 
	        	return; /* save done or user press cancel */            
    	}
    
    	current_no = gtk_notebook_get_current_page( GTK_NOTEBOOK( main_window->notebook ) );

    	gtk_notebook_remove_page( GTK_NOTEBOOK( main_window->notebook ), current_no );
    	free_op_list( current->op_list );

    	g_list_remove( main_window->documents, current );

    	/* free memory */
    	g_free( current->filename );
   	g_free( current );	

    	if( gI_document_get_no( main_window ) == 0 )
    	{
        	gtk_window_set_title( GTK_WINDOW( main_window->window ), "gIDE " VERSION );
        	gI_window_clear_statusbar( main_window );
		gI_window_set_visible_read_only( FALSE ); 
    	}
}


/* -------------------------------------*/
/* file_close_changed_dialog_modal()                       */
/* return TRUE if user press NO and file is NOT saved      */
/* Return FALSE if file will save or user press cancel button*/
/* Use this instead of file_close_changed_dialog */
/* --------------------------------------*/
gint file_close_changed_dialog_modal(gI_document *current)
{
	gchar HelpString[STRLEN];
	gint user_reply;

	if( !current->filename )
	{
		gchar *label, *label_dup;
		gchar *ptr;

		gtk_label_get( GTK_LABEL( current->label ), &label );

		label_dup = g_strdup( label );
		label = label_dup;

		ptr = strchr( label, '*' );
		if( ptr )
			*ptr = '\0';

        	current->filename = g_strdup( label );

		g_free( label_dup );
	}

	g_snprintf( HelpString, STRLEN, _("\nThe File %s has been changed!\nDo You want to save it?\n"), file_strip_name( current->filename ) );
	
	user_reply = gI_ask_dialog( HelpString );
	if( user_reply == 2 )   /* CANCEL */
		return FALSE;

   	if( user_reply == 0 )   /* YES */
   	{
		file_close_save = TRUE;
		if( current->filename == NULL )
			file_save_as( NULL, NULL );
		else
			file_save( NULL, NULL );
    	    
		file_close_save = FALSE;
		return( FALSE );
	}

	return( TRUE );
}


static void write_history( gI_document *document, FILE *history )
{
	if( !document )
        	return;

	if( !document->filename )
        	return;

	fprintf( history, "%s\n", document->filename );
}


/*
 ---------------------------------------------------------------------
     Function: file_exit()
     Desc: Callback-Function /File/Exit
 ---------------------------------------------------------------------
*/
void file_exit( GtkWidget *widget, gpointer data )
{
	gint nod,i,changed=0;
	gI_document *current;
	FILE *history;
    	gchar historywp[MAXLEN];
    	gI_project *project;

    	/* close opened project */
    	if( (project = gI_project_get_current()) )
    	{
        	project_close( NULL, NULL );
    	}

    	/* save filenames */
    	g_snprintf( historywp, MAXLEN, "%s/%s", gide_path, HISTORY );
    	history = fopen( historywp, "w" );
    	if( history )
    	{
        	g_list_foreach( main_window->documents, (GFunc) write_history, (gpointer) history );
        	fclose( history );
    	}

    	nod = gI_document_get_no( main_window );

    	for(i=1;i<=nod;i++)
    	{
        	gtk_notebook_set_page( GTK_NOTEBOOK( main_window->notebook ), i );
		current = gI_document_get_current( main_window );
        	if( current->changed == TRUE )
        	{
            		changed = TRUE;
            		break;
        	}
    	}

    	if( changed )
	{
		file_exit_is_changed( main_window );
	}
	else
	{
		gtk_main_quit();

	}
}


void file_exit_is_changed( gI_window *window )
{
	gint nod,i;
    	gI_document *current;

    	nod = gI_document_get_no( main_window );
	noc = gI_document_get_changed( main_window ); 
    
    	for(i=0;i<nod;i++)
    	{
        	gtk_notebook_set_page( GTK_NOTEBOOK( window->notebook ), i );
        	current = gI_document_get_current( window );
        	if( current->changed == TRUE )
        	{
            		file_exit_changed_dialog( current );
        	}
    	}
}


void file_exit_changed_dialog( gI_document *document )
{
/***
    GtkWidget *dialog;
    GtkWidget *vbox, *hbox, *separator, *label, *ok, *cancel;
***/
    	gchar HelpString[STRLEN];

    	if( !document->filename )
	{
		gchar *label, *label_dup;
		gchar *ptr;

		gtk_label_get( GTK_LABEL( document->label ), &label );

		label_dup = g_strdup( label );
		label = label_dup;

		ptr = strchr( label, '*' );
		if( ptr )
			*ptr = '\0';

        	document->filename = g_strdup( label );

		g_free( label_dup );
	}

    	g_snprintf( HelpString, STRLEN, _("\nThe File\n'%s'\nhas been changed!\nDo You want to save it?\n"), file_strip_name( document->filename ) );

	if( gI_ask_dialog( HelpString ) == 0 )  /* YES */
	{
		file_exit_changed_ok( NULL, NULL );
	}
	else
	{
		file_exit_changed_cancel( NULL, NULL );
	}

/***
    	dialog = gtk_window_new( GTK_WINDOW_DIALOG );
    	gtk_widget_set_usize( dialog, 250, 175 );

    	vbox = gtk_vbox_new( FALSE, 0 );
    	gtk_container_add( GTK_CONTAINER( dialog ), vbox );
    	gtk_widget_show( vbox );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	if( !document->filename )
        	document->filename = g_strdup( "Untitled" );
    

    	g_snprintf( HelpString, STRLEN, "\nThe File %s has been changed!\nDo You want to save it?\n", file_strip_name( document->filename ) );
    	label = gtk_label_new( HelpString );
    	gtk_container_add( GTK_CONTAINER( hbox ), label );
    	gtk_widget_show( label );
    
    	separator = gtk_hseparator_new();
    	gtk_box_pack_start( GTK_BOX( vbox ), separator, FALSE, TRUE, 5 );
    	gtk_widget_show( separator );

    	hbox = gtk_hbox_new( FALSE, 0 );
    	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 5 );
    	gtk_widget_show( hbox );

    	ok = gtk_button_new_with_label( "Save" );
    	gtk_box_pack_start( GTK_BOX( hbox ), ok, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( ok ), "clicked",
                       	    GTK_SIGNAL_FUNC( file_exit_changed_ok ), document );
    	gtk_widget_show( ok );
    
    	cancel = gtk_button_new_with_label( "Discard" );
    	gtk_box_pack_start( GTK_BOX( hbox ), cancel, TRUE, TRUE, 15 );
    	gtk_signal_connect( GTK_OBJECT( cancel ), "clicked",
                            GTK_SIGNAL_FUNC( file_exit_changed_cancel ), document );
    	gtk_widget_show( cancel );
    
    	gtk_widget_show( dialog );
***/

}


void file_exit_changed_ok( GtkWidget *widget, gpointer data )
{
	/* save */
	file_exit_save = TRUE;
	file_save( NULL, NULL );
	file_exit_save = FALSE;
}


void file_exit_changed_cancel( GtkWidget *widget, gpointer data )
{
	noc--;

	if( noc < 0 )
	{
		g_warning( "NOC < 0: Something is going wrong (%s/%d)!\n", __FILE__, __LINE__ );
   	}

    	if( noc == 0 )
	{
		gtk_main_quit();
	}
}


static void _file_autosave( gI_document *document, gpointer data )
{
	FILE *docfile;
	gchar str[STRLEN];
	glong length,i;
	gchar *c;

    	/* don't autosave, if no document is given */
    	if( !document )
        	return;

    	/* don't autosave, if the document has no filename (i.e. "Untitled") */
    	if( !document->filename )
        	return;

    	/* don't autosave, if document is unchanged */
    	if( !document->changed )
        	return;

    	docfile = fopen( document->filename, "w" );
    	if( !docfile )
    	{
        	g_snprintf( str, STRLEN, _("Unable to Save '%s'!"), document->filename );
        	gI_error_dialog( str);
        	return;
    	}

    	length = gI_text_get_length( document->text );
    	for(i=1;i<=length;i++)
	{
        	c = gtk_editable_get_chars( GTK_EDITABLE( document->text ), i-1, i );
		fputc( c[0], docfile );
		g_free( c );
	}

    	fclose( docfile );

    	gI_document_set_unchanged( document );

	document->last_mod = get_last_mod( document->filename );

	gI_window_set_statusbar( document );
}


void file_autosave( GtkWidget *widget, gpointer data )
{
	gchar sb_msg[STRLEN];

	/*g_print( "autosave!!\n" );*/
	/* remove timeout */
	gtk_timeout_remove( main_window->timeout_id );

    	/*push message to statusbar*/
    	g_snprintf( sb_msg, STRLEN, _("Autosaving...") );
    	gtk_statusbar_push( GTK_STATUSBAR( main_window->statusbar ), 2, sb_msg );

    	g_list_foreach( main_window->documents, (GFunc) _file_autosave, NULL );

	/* re-add timeout */
	main_window->timeout_id = gtk_timeout_add( cfg->autosave_freq * 1000, (GtkFunction) file_autosave, NULL ); 
}


gint file_check_if_exist( gchar *fname, gint askcreate )
{
	gchar txt[512];
	FILE *file;
	struct stat sta;
	gint e;
        
	e=stat(fname,&sta);
	if( !e && S_ISREG(sta.st_mode) )
	{   	/* file exist and is a regular file */
		return( 1 );	
	}

	if( askcreate && e==-1 )
	{
	    sprintf( txt, _("The file\n'%s'\ndoes not exist!\nDo you want to create the file?"), fname );
	    if( gI_ask_dialog( txt ) == 0 ) /*GNOME_YES )*/
	    {
	        file = fopen( fname, "w" );	/* this is not a good way!! */
		if( file ) 
		{
			fclose( file );
		}

		return( 1 );
	    }
	}
	else
	{
		if( !e )
		{ 
		    sprintf( txt, _("The file '%s' is not a regular file!"), fname );
                    gI_error_dialog( txt );
                }
	}

	return( 0 );
}


/*
 * this function checks if a document has been changed from
 * outside (i.e. the file mod time has changed)
 * return: 0 - not changed
 * 	   1 - changed
 *	   2 - does not (longer) exist
 */
glong check_doc_changed( gI_document *doc )
{
	if( !doc )
		return( 0 );

	if( !doc->filename )
		return( 0 );

	if( !file_exist( doc->filename ) )
		return( 2 );

	if( get_last_mod( doc->filename ) != doc->last_mod )
		return( 1 );

	return( 0 );
}


