/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <locale.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"

#include "gimv_image.h"
#include "argparse.h"
#include "prefs.h"
#include "fileload.h"
#include "fileutil.h"
#include "icon.h"
#include "image_window.h"
#include "thumbnail_window.h"


/* private functions */
void               gimageview_initialize   (gint     argc,
					    gchar   *argv[]);
static OpenFiles  *get_files_from_argument (gint     argc,
					    gchar   *argv[],
					    gint     remaining);
static gint        idle_open_image_startup (gpointer data);
static gint        idle_slideshow_startup  (gpointer data);


/* global variables */
GlobValue glob_val = {
   FALSE,
};


void
gimageview_initialize (gint argc, gchar *argv[])
{
   gint num;
   gchar *gtkrc;

   /* set locale */
   setlocale (LC_ALL, "");
   bindtextdomain (PACKAGE,LOCALEDIR);
   textdomain (PACKAGE);

   /* Gtk & Imlib Initialize */
   gtk_set_locale();
   gtk_init(&argc, &argv);
   gtkrc = g_strconcat (g_getenv ("HOME"), "/", RC_DIR, "/", GTK_RC, NULL);
   gtk_rc_parse (gtkrc);
   gimv_image_init ();

   /* load config */
   prefs_load_config ();
   prefs_bind_key ();

   icon_init (conf.iconset);

   /* FIXME!! */
   disp_mode_labels = thumbview_get_disp_mode_labels (&num);

   g_free (gtkrc);
}


void
gimageview_quit ()
{
   ThumbWindow *tw_tmp;
   GList *node;

   node = g_list_last (ThumbWinList);

   if (node) {
      tw_tmp = node->data;
      if (tw_tmp && conf.thumbwin_save_win_state)
	 thumbwin_store_win_state_to_config (tw_tmp);
   }

   /* save config to disk */
   prefs_save_config ();

   gtk_main_quit ();
}


static OpenFiles *
get_files_from_argument (gint argc, gchar *argv[], gint remaining)
{
   OpenFiles *files = NULL;
   gchar *path;
   gint i;

   files = files_loader_new ();

#ifdef GIMV_DEBUG
   g_print ("Program start\n");
   file_disp_loading_status (files);
#endif /* GIMV_DEBUG */

   for (i = remaining; i < argc; i++) {
      if (conf.conv_rel_path_to_abs) {
	 path = fileutil_conv_rel_path_to_abs (argv[i]);
      } else {
	 path = g_strdup (argv[i]);
      }

      /* store dir list */
      if (isdir (path)) {
	 files->dirlist = g_list_append (files->dirlist, path);

      /* detect image file by file name extension */
      } else if (file_exists (path)
		 && (!conf.detect_filetype_by_ext ||
		     gimv_image_detect_type_by_ext (path)))
      {
	 files->filelist = g_list_append (files->filelist, path);
      } else {
	 g_free (path);
      }
   }

   return files;
}


static gint
idle_open_image_startup (gpointer data)
{
   gboolean quit_main = FALSE;
   OpenFiles *files = (OpenFiles *) data;
   LoadStatus status = FALSE;

   status = open_image_files(files);
   if (conf.startup_read_dir) {
      status = open_dirs (files, NULL, LOAD_CACHE, conf.scan_dir_recursive)
	 || status;
   }
   if (!status) {
      if (!conf.startup_no_warning) {
	 g_warning (_("No image files specified!!"));
	 if (!conf.startup_read_dir)
	 g_warning (_("If you want to scan directory, use \"-d\" option."));
      }
      /* quit if no window opened */
      if (!ThumbWinList && !ImageWinList) {
	 if (!conf.startup_no_warning)
	    g_warning (_("No window opened!! Quiting..."));
	 quit_main = TRUE;
      }
   }

   files_loader_delete (files);

   /* reset config */
   prefs_load_config ();

   if (quit_main)
      gtk_main_quit ();

   return FALSE;
}


static gint
idle_slideshow_startup (gpointer data)
{
   SlideShow *slideshow = (SlideShow *) data;

   /* reset config */
   prefs_load_config ();

   imagewin_exec_slideshow (slideshow);

   return FALSE;
}



/******************************************************************************
 *
 *   main
 *
 ******************************************************************************/
gint
main (gint argc, gchar *argv[])
{
   ThumbWindow *tw = NULL;
   gint remaining;
   OpenFiles *files = NULL;

   gimageview_initialize (argc, argv);

   /* override config by argument while start up*/
   arg_parse (argc, argv, &remaining);

   /* open thumbnail window if specified by config or argument */
   if (conf.startup_open_thumbwin || argc == remaining) {
      tw = thumbwin_open_window();
   } /* else if () {
      arg_help(argv, stderr);
      exit (1);
      } */

   /* set OpenFiles struct data for opening files */
   files = get_files_from_argument (argc, argv, remaining);
   if (!files && !ThumbWinList && !ImageWinList) exit (1);

   /* exec slide show if specified by argument */
   if (glob_val.startup_exec_slideshow) {
      SlideShow *slideshow = g_new0 (SlideShow, 1);
      slideshow->filelist = files->filelist;
      files->filelist = NULL;

      if (g_list_length (slideshow->filelist) < 1)
	 arg_help (argv, stderr);

      gtk_init_add (idle_slideshow_startup, slideshow);

      files_loader_delete (files);

   /* check filelist & dirlist and open image files */
   } else {
      gtk_init_add (idle_open_image_startup, files);
   }

   /* main roop */
   gtk_main ();

   return 0;
}
