/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>

#include "gimageview.h"
#include "gimv_image.h"
#include "prefs.h"
#include "fileutil.h"

/* loader */
#include "bmp.h"
#include "mag.h"
#include "pcx.h"
#include "tga.h"
#include "xbm.h"

#if HAVE_GDK_PIXBUF
#   include "pixbuf_utils.h"
#endif /* HAVE_GDK_PIXBUF */


/* for file name extension list */
typedef struct _img_ext
{
   gchar *ext;
   gchar *type;
} img_ext;


/*
 *  img_ext_list:
 *     @ File name extensions list.
 *     @ In the future, this will be placed at external file
 */
static img_ext img_ext_list[] =
{
   {"tif",  IMG_TIF},
   {"tiff", IMG_TIF},
   {"jpg",  IMG_JPG},
   {"jpeg", IMG_JPG},
   {"png",  IMG_PNG},
   {"xbm",  IMG_XBM},
   {"xpm",  IMG_XPM},
   {"xwd",  IMG_XWD},
   {"ppm",  IMG_PPM},
   {"pgm",  IMG_PGM},
   {"pbm",  IMG_PBM},
   {"bmp",  IMG_BMP},
   {"pcx",  IMG_PCX},
   /*
   {"pic",  IMG_PICT},
   {"pct",  IMG_PICT},
   {"pict", IMG_PICT},
   */
   {"gif",  IMG_GIF},
   {"ico",  IMG_UNKNOWN},
   {"icon", IMG_UNKNOWN},
   {"mag",  IMG_MAG},
   {"tga",  IMG_TGA},
   {NULL,   NULL}
};


void
gimv_image_init ()
{
#if HAVE_GDK_PIXBUF
   return;
#else
#if HAVE_GDK_IMLIB
   /*
   GdkImlibInitParams *imlib_param;

   imlib_param = g_new0 (GdkImlibInitParams, 1);

   imlib_param->flags = PARAMS_REMAP | PARAMS_FASTRENDER;
   imlib_param->remap = 0;
   imlib_param->fastrender = 0;
   gdk_imlib_init_params(imlib_param);
   g_free (imlib_param);
   */
   gdk_imlib_init ();
   gtk_widget_push_visual (gdk_imlib_get_visual());
   gtk_widget_push_colormap (gdk_imlib_get_colormap());
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


GimvImage *
gimv_image_load_file (const gchar *filename)
{
   GimvImage *image;
   guchar *rgb_data;
   gint width, height;

   rgb_data = bmp_load (filename, &width, &height);
   if (rgb_data) {
      image = gimv_image_create_from_data (rgb_data, width, height);
      return image;
   }

   rgb_data = tga_load (filename, &width, &height);
   if (rgb_data) {
      image = gimv_image_create_from_data (rgb_data, width, height);
      return image;
   }

#if HAVE_GDK_PIXBUF
   image = gdk_pixbuf_new_from_file ((char *)filename);
#else
#if HAVE_GDK_IMLIB
   image = gdk_imlib_load_image ((char *)filename);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   if (!image) {
      rgb_data = pcx_load (filename, &width, &height);
      if (rgb_data)
	 image = gimv_image_create_from_data (rgb_data, width, height);
   }

   if (!image) {
      rgb_data = xbm_load (filename, &width, &height);
      if (rgb_data)
	 image = gimv_image_create_from_data (rgb_data, width, height);
   }

   if (!image) {
      rgb_data = xvpics_load (filename, &width, &height);
      if (rgb_data)
	 image = gimv_image_create_from_data (rgb_data, width, height);
   }

   if (!image) {
      rgb_data = mag_load (filename, &width, &height);
      if (rgb_data)
	 image = gimv_image_create_from_data (rgb_data, width, height);
   }

   return image;
}


#if HAVE_GDK_PIXBUF
static void
free_rgb_buffer (guchar *pixels, gpointer data)
{
   g_free(pixels);
}
#endif /* HAVE_GDK_PIXBUF */


GimvImage *
gimv_image_create_from_data (guchar *data, gint width, gint height)
{
   GimvImage *image;

#if HAVE_GDK_PIXBUF
   image = gdk_pixbuf_new_from_data (data, GDK_COLORSPACE_RGB, FALSE, 8,
				     width, height, 3 * width, free_rgb_buffer, NULL);
#else
#if HAVE_GDK_IMLIB
   image = gdk_imlib_create_image_from_data (data, NULL, width, height);
   g_free (data);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return image;
}


GimvImage *
gimv_image_create_from_drawable (GdkWindow *win, gint x, gint y,
				 gint width, gint height)
{
   GimvImage *image;

#if HAVE_GDK_PIXBUF
   GdkColormap *cmap = gdk_colormap_get_system ();
   image = gdk_pixbuf_get_from_drawable (NULL, win, cmap, x , y, 0, 0, width, height);
#else
#if HAVE_GDK_IMLIB
   image = gdk_imlib_create_image_from_drawable (win, NULL, x, y, width, height);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return image;
}


GimvImage *
gimv_image_rotate_90 (GimvImage *src_image,
		      gboolean counter_clockwise)
{
   GimvImage *dest_image;

#if HAVE_GDK_PIXBUF
   dest_image = pixbuf_copy_rotate_90 (src_image, counter_clockwise);
   gdk_pixbuf_unref (src_image);
#else
#if HAVE_GDK_IMLIB
   if (counter_clockwise) {
      gdk_imlib_flip_image_horizontal (src_image);
      gdk_imlib_rotate_image (src_image, 1);
   } else {
      gdk_imlib_flip_image_vertical (src_image);
      gdk_imlib_rotate_image (src_image, 1);
   }
   dest_image = src_image;
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return dest_image;
}


GimvImage *
gimv_image_rotate_180 (GimvImage *src_image)
{
   GimvImage *dest_image;

#if HAVE_GDK_PIXBUF
   dest_image = pixbuf_copy_mirror (src_image, TRUE, TRUE);
   gdk_pixbuf_unref (src_image);
#else
#if HAVE_GDK_IMLIB
   gdk_imlib_flip_image_vertical (src_image);
   gdk_imlib_flip_image_horizontal (src_image);
   dest_image = src_image;
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return dest_image;
}


void
gimv_image_get_pixmap_and_mask (GimvImage *image,
				GdkPixmap **pixmap_return,
				GdkBitmap **mask_return)
{
#if HAVE_GDK_PIXBUF
   gdk_pixbuf_render_pixmap_and_mask (image, pixmap_return, mask_return, 64);
#else
#if HAVE_GDK_IMLIB
   *pixmap_return = gdk_imlib_move_image (image);
   *mask_return = gdk_imlib_move_mask (image);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


void
gimv_image_free_pixmap_and_mask (GdkPixmap *pixmap,
				 GdkBitmap *mask)
{
#if HAVE_GDK_PIXBUF
   if (pixmap) gdk_pixmap_unref (pixmap);
   if (mask) gdk_bitmap_unref (mask);
#else
#if HAVE_GDK_IMLIB
   gdk_imlib_free_pixmap (pixmap);
   gdk_imlib_free_bitmap (mask);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


GimvImage *
gimv_image_scale_and_dump (GimvImage *src_image,
			   gint width, gint height)
{
   GimvImage *dest_image;

#if HAVE_GDK_PIXBUF
   dest_image = gdk_pixbuf_scale_simple (src_image, width, height,
					 conf.interpolation);
#else
#if HAVE_GDK_IMLIB
   dest_image = gdk_imlib_clone_scaled_image (src_image, width, height);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return dest_image;
}


void
gimv_image_scale_get_pixmap (GimvImage *src_image,
			     gint width, gint height,
			     GdkPixmap **pixmap_return,
			     GdkBitmap **mask_return)
{
   GimvImage *dest_image;

#if HAVE_GDK_PIXBUF
   dest_image = gdk_pixbuf_scale_simple (src_image, width, height,
					 conf.interpolation);
   gimv_image_get_pixmap_and_mask (dest_image, pixmap_return, mask_return);
   gdk_pixbuf_unref (dest_image);
#else
#if HAVE_GDK_IMLIB
   gdk_imlib_render (src_image, width, height);
   dest_image = src_image;
   gimv_image_get_pixmap_and_mask (dest_image, pixmap_return, mask_return);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


void
gimv_image_get_size (GimvImage *image, gint *width, gint *height)
{
#if HAVE_GDK_PIXBUF
   *width  = gdk_pixbuf_get_width  (image);
   *height = gdk_pixbuf_get_height (image);
#else
#if HAVE_GDK_IMLIB
   *width  = image->rgb_width;
   *height = image->rgb_height;
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


gint
gimv_image_width (GimvImage *image)
{
#if HAVE_GDK_PIXBUF
   return gdk_pixbuf_get_width  (image);
#else
#if HAVE_GDK_IMLIB
   return image->rgb_width;
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


gint
gimv_image_height (GimvImage *image)
{
#if HAVE_GDK_PIXBUF
   return gdk_pixbuf_get_height (image);
#else
#if HAVE_GDK_IMLIB
   return image->rgb_height;
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


void
gimv_image_kill (GimvImage *image)
{
   if (!image) return;

#if HAVE_GDK_PIXBUF
   gdk_pixbuf_unref (image);
#else
#if HAVE_GDK_IMLIB
   gdk_imlib_kill_image (image);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */
}


/*
 *  gimv_image_detect_type_by_ext:
 *     @ Detect image format by filename extension.
 *
 *  str    : Image file name.
 *  Return : Image type by string.
 */
gchar *
gimv_image_detect_type_by_ext (const gchar *str)
{
   gchar *ext = fileutil_get_filename_extention (str);
   gint i;

   if (ext) {
      for (i = 0; img_ext_list[i].ext; i++) {
	 if (!g_strcasecmp(ext, img_ext_list[i].ext)) {
	    return img_ext_list[i].type;
	 }
      }
   }
   return NULL;
}


/* FIXME!! Implement as plugin */
gboolean 
gimv_image_save_file_as_png (GimvImage *image, const char *filename)
{
   gboolean retval;

#if HAVE_GDK_PIXBUF
   retval = pixbuf_to_file_as_png (image, filename);
#else
#if HAVE_GDK_IMLIB
   gchar tmp_file[BUF_SIZE];

   g_snprintf (tmp_file, BUF_SIZE, "%s.png", filename);
   retval = gdk_imlib_save_image (image, tmp_file, NULL);
   rename (tmp_file, filename);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return retval;
}


gboolean 
gimv_image_save_file_as_ppm (GimvImage *image, const char *filename)
{
   gboolean retval;

#if HAVE_GDK_PIXBUF
   retval = pixbuf_to_file_as_ppm (image, filename);
#else
#if HAVE_GDK_IMLIB
   retval = gdk_imlib_save_image_to_ppm (image, (char *) filename);
#endif /* HAVE_GDK_IMLIB */
#endif /* HAVE_GDK_PIXBUF */

   return retval;
}
/* End FIXME!! */



/******************************************************************************
 *
 * xvpics thumbnail support
 *
 ******************************************************************************/
guchar *
xvpics_load (const gchar *filename, gint *widthp, gint *heightp)
{
   FILE *file;
   gchar buffer[BUF_SIZE];
   guchar *data, *rgb_data;
   gint width, height, depth;
   gint i;

   file = fopen (filename, "rt");
   if (!file) return NULL;

   fgets (buffer, BUF_SIZE, file);
   if (strncmp (buffer, "P7 332", 6) != 0) {
      fclose (file);
      return NULL;
   }

   while (fgets(buffer, BUF_SIZE, file) && buffer[0] == '#');

   if (sscanf(buffer, "%d %d %d", &width, &height, &depth) != 3) {
      fclose (file);
      return NULL;
   }

   data = g_new0 (guchar, width * height);
   fread (data, 1, width * height, file);

   rgb_data = g_new(guchar, width * height * 3);
   for(i = 0; i < width * height; i++) {
      rgb_data[i * 3 + 0] = (data[i] >> 5) * 36;
      rgb_data[i * 3 + 1] = ((data[i] & 28) >> 2) * 36;
      rgb_data[i * 3 + 2] = (data[i] & 3) * 85;
   }

   fclose (file);
   g_free (data);
   *widthp = width;
   *heightp = height;

   return rgb_data;
}


GimvImage *
gimv_image_get_xv_thumbnail (const gchar *thumb_filename)
{
   GimvImage *image;
   gint width, height;
   guchar *rgb_data;

   rgb_data
      = xvpics_load (thumb_filename, &width, &height);

   if(!rgb_data) return NULL;

   image = gimv_image_create_from_data (rgb_data, width, height);
   return image;
}
