/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gdk/gdkkeysyms.h>

#include "gimageview.h"

#include "icon.h"
#include "image_view.h"
#include "image_window.h"
#include "prefs.h"
#include "prefs_win.h"
#include "menu.h"
#include "help.h"


GList *ImageWinList;
ImageWindow *shared_img_win;


static void  create_imageview_menus     (ImageWindow  *iw);
static GtkWidget *create_toolbar        (ImageWindow  *iw,
					 GtkWidget    *container);

static void imagewin_set_win_size       (ImageWindow  *iw);
static void imagewin_set_window_title   (ImageWindow  *iw);
static void imagewin_set_statusbar_info (ImageWindow  *iw);

/* for image view module */
static void imagewin_render_cb          (gpointer      data);
static void imagewin_aspect_cb          (gpointer      data);

static gboolean timeout_slideshow       (SlideShow    *slideshow);

/* callback functions for menubar */
static void cb_file_select          (gpointer     data,
				     guint        action,
				     GtkWidget   *widget);
static void cb_open_thumbnail       (gpointer     data,
				     guint        action,
				     GtkWidget   *widget);
static void cb_toggle_buffer        (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_window_close         (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_toggle_menubar       (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_toggle_toolbar       (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_toggle_statusbar     (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_toggle_fullscreen    (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_fit_to_image         (ImageWindow *iw,
				     guint        action,
				     GtkWidget   *widget);
static void cb_options              ();

/* callback functions for toolbar */
static void cb_toolbar_open_button  (GtkWidget *widget);
static void cb_toolbar_prefs_button (GtkWidget *widget);
static void cb_toolbar_no_zoom      (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_zoom_in      (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_zoom_out     (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_zoom_fit     (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_zoom         (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_rotate       (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_keep_aspect  (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_fit_window   (GtkWidget *widget, ImageWindow *iw);
static void cb_toolbar_fullscreen   (GtkWidget *widget, ImageWindow *iw);
static gint cb_scale_spinner_key_press (GtkWidget *widget, 
					GdkEventKey *event,
					ImageWindow *iw);

/* other callback functions */
static void cb_window_closed (GtkWidget   *widget,
			      ImageWindow *iw);

/* for main menu */
GtkItemFactoryEntry imagewin_menu_items[] =
{
   {N_("/_File"),                       NULL,         NULL,              0,  "<Branch>"},
   {N_("/_File/_Open"),                 NULL,         cb_file_select,    0,  NULL},
   {N_("/_File/_Open Thumbnail Window"),NULL,         cb_open_thumbnail, 0,  NULL},
   {N_("/_File/---"),                   NULL,         NULL,              0,  "<Separator>"},
   {N_("/_File/Memory _Buffer"),        NULL,         cb_toggle_buffer,  0,  "<ToggleItem>"},
   {N_("/_File/---"),                   NULL,         NULL,              0,  "<Separator>"},
   {N_("/_File/Close"),                 NULL,         cb_window_close,   0,  NULL},
   {N_("/_File/Quit"),                  NULL,         gimageview_quit,   0,  NULL},

   {N_("/_Edit"),                       NULL,         NULL,              0,  "<Branch>"},
   {N_("/_Edit/Options"),               NULL,         cb_options,        0,  NULL},

   {N_("/_View"),                       NULL,         NULL,              0,  "<Branch>"},

   {N_("/_Help"),                       NULL,         NULL,              0,  "<LastBranch>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "View" sub menu */
GtkItemFactoryEntry imagewin_view_items [] =
{
   {N_("/tear"),         NULL,  NULL,                 0,  "<Tearoff>"},
   {N_("/Zoom"),         NULL,  NULL,                 0,  "<Branch>"},
   {N_("/Rotate"),       NULL,  NULL,                 0,  "<Branch>"},
   {N_("/---"),          NULL,  NULL,                 0,  "<Separator>"},
   {N_("/Menu Bar"),     NULL,  cb_toggle_menubar,    0,  "<ToggleItem>"},
   {N_("/Tool Bar"),     NULL,  cb_toggle_toolbar,    0,  "<ToggleItem>"},
   {N_("/Status Bar"),   NULL,  cb_toggle_statusbar,  0,  "<ToggleItem>"},
   {N_("/---"),          NULL,  NULL,                 0,  "<Separator>"},
   {N_("/Full Screen"),  NULL,  cb_toggle_fullscreen, 0,  "<ToggleItem>"},
   {N_("/Fit to Image"), NULL,  cb_fit_to_image,      0,  NULL},
   {NULL, NULL, NULL, 0, NULL},
};


/*
 *  create_imageview_menus:
 *     @ create main menu and popup menu.
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
create_imageview_menus (ImageWindow *iw)
{
   guint n_menu_items;
   GtkWidget *help;

   /* main menu */
   n_menu_items = sizeof (imagewin_menu_items)
                     / sizeof (imagewin_menu_items[0]) - 1;
   iw->menubar = menubar_create (iw->window, imagewin_menu_items, n_menu_items,
				 "<ImageviewMain>", iw);
   gtk_container_add(GTK_CONTAINER(iw->menubar_handle), iw->menubar);
   gtk_menu_bar_set_shadow_type (GTK_MENU_BAR(iw->menubar), GTK_SHADOW_NONE);
   gtk_widget_show (iw->menubar);

   /* sub menu */
   n_menu_items = sizeof(imagewin_view_items)
                     / sizeof(imagewin_view_items[0]) - 1;
   iw->view_menu = menu_create_items(iw->window, imagewin_view_items,
				     n_menu_items, "<ViewSubMenu>", iw);

   iw->zoom_menu   = imageview_create_zoom_menu (iw->window, iw->iv);
   iw->rotate_menu = imageview_create_rotate_menu (iw->window, iw->iv);
   help = gimvhelp_create_menu (iw->window);

   /* attach sub menus to parent menu */
   menu_set_submenu (iw->menubar,   "/View",   iw->view_menu);
   menu_set_submenu (iw->menubar,   "/Help",   help);
   menu_set_submenu (iw->view_menu, "/Zoom",   iw->zoom_menu);
   menu_set_submenu (iw->view_menu, "/Rotate", iw->rotate_menu);

   /* popup menu */
   iw->iv->imageview_popup = iw->view_menu;

   /* initialize menubar check items */
   menu_check_item_set_active (iw->menubar,   "/File/Memory Buffer", iw->iv->buffer);

   menu_check_item_set_active (iw->view_menu, "/Menu Bar",   iw->show_menubar);
   menu_check_item_set_active (iw->view_menu, "/Tool Bar",   iw->show_toolbar);
   menu_check_item_set_active (iw->view_menu, "/Status Bar", iw->show_statusbar);
   menu_check_item_set_active (iw->view_menu, "/Full Screen", iw->fullscreen);
   if (!iw->show_menubar)
      gtk_widget_hide (iw->menubar_handle);
   if (!iw->show_toolbar)
      gtk_widget_hide (iw->toolbar_handle);
   if (!iw->show_statusbar)
      gtk_widget_hide (iw->status_bar_container);
}


/*
 *  create_toolbar:
 *     @ create toolbar and buttons.
 *
 *  im        : Pointer to ImageWindow structure.
 *  container : Container to store toolbar.
 *  Return    : Pointer to toolbar widget.
 */
static GtkWidget *
create_toolbar (ImageWindow *iw, GtkWidget *container)
{
   GtkWidget *toolbar;
   GtkWidget *button;
   GtkWidget *iconw;
   GtkAdjustment *adj;
   GtkWidget *spinner;

   toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
   gtk_toolbar_set_button_relief (GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
   gtk_toolbar_set_space_style (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_SPACE_LINE);
   gtk_toolbar_set_space_size  (GTK_TOOLBAR(toolbar), 16);


   /* file open button */
   iconw = icon_get_widget ("nfolder");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Open"),
				    _("File Open"), _("File Open"),
				    iconw, cb_toolbar_open_button, NULL);

   /* preference button */
   iconw = icon_get_widget ("prefs");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Prefs"),
				    _("Preference"), _("Preference"),
				    iconw, cb_toolbar_prefs_button, NULL);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* no zoom button */
   iconw = icon_get_widget ("no_zoom");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("No Zoom"),
				    _("No Zoom"), _("No Zoom"),
				    iconw, cb_toolbar_no_zoom, iw);

   /* zoom in button */
   iconw = icon_get_widget ("zoom_in");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Zoom in"),
				    _("Zoom in"), _("Zoom in"),
				    iconw, cb_toolbar_zoom_in, iw);

   /* zoom out button */
   iconw = icon_get_widget ("zoom_out");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Zoom out"),
				    _("Zoom out"), _("Zoom out"),
				    iconw, cb_toolbar_zoom_out, iw);

   /* zoom fit button */
   iconw = icon_get_widget ("zoom_fit");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Zoom fit"),
				    _("Zoom fit"), _("Zoom fit"),
				    iconw, cb_toolbar_zoom_fit, iw);

   /* x scale spinner */
   adj = (GtkAdjustment *) gtk_adjustment_new (iw->iv->x_scale,
					       MIN_THUMB_SIZE, MAX_THUMB_SIZE,
					       1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_NONE);
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), spinner,
			      _("X Scale"), _("X Scale"));
   gtk_signal_connect (GTK_OBJECT (adj), "value_changed",
		       GTK_SIGNAL_FUNC (cb_toolbar_keep_aspect), iw);
   iw->button.xscale = spinner;
   gtk_signal_connect (GTK_OBJECT(spinner), "key-press-event",
		       GTK_SIGNAL_FUNC(cb_scale_spinner_key_press), iw);

   /* y scale spinner */
   adj = (GtkAdjustment *) gtk_adjustment_new (iw->iv->y_scale,
					       MIN_THUMB_SIZE, MAX_THUMB_SIZE,
					       1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_NONE);
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), spinner,
			      _("Y Scale"), _("Y Scale"));
   iw->button.yscale = spinner;

   /* zoom button */
   iconw = icon_get_widget ("zoom");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Zoom"),
				    _("Zoom"), _("Zoom"),
				    iconw, cb_toolbar_zoom, iw);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* rotate button */
   iconw = icon_get_widget ("rotate");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Rotate"),
				    _("Rotate CW"), _("Rotate CW"),
				    iconw, cb_toolbar_rotate, iw);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* resize button */
   iconw = icon_get_widget ("resize");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Resize"),
				    _("Fit Window Size to Image"),
				    _("Fit Window Size to Image"),
				    iconw, cb_toolbar_fit_window, iw);

   /* fullscreen button */
   iconw = icon_get_widget ("fullscreen");
   /* button = gtk_toggle_button_new (); */
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Full"),
				    _("Fullscreen"),
				    _("Fullscreen"),
				    iconw, cb_toolbar_fullscreen, iw);

   return toolbar;
}


/*
 *  imageview_set_win_size:
 *     @ Set image window size to image size.
 *
 *  im    : Pointer to ImageWindow structure.
 */
static void
imagewin_set_win_size (ImageWindow *iw)
{
   gint x_size, y_size, width, height;
   gfloat x_scale, y_scale;
   gint menubar_height, toolbar_height, statusbar_height;

   if (!iw->window || iw->fullscreen)
      return;

   if (iw->iv->rotate == 0 || iw->iv->rotate == 2) {
      x_scale = iw->iv->x_scale;
      y_scale = iw->iv->y_scale;
      width =  iw->iv->info->width;
      height = iw->iv->info->height;
   } else {
      x_scale = iw->iv->y_scale;
      y_scale = iw->iv->x_scale;
      width  = iw->iv->info->height;
      height = iw->iv->info->width;
   }

   x_size = width  * x_scale / 100.0 + 0.5;
   y_size = height * y_scale / 100.0 + 0.5;

   if (iw->show_menubar)
      menubar_height = iw->menubar_handle->allocation.height;
   else
      menubar_height = 0;

   if (iw->show_toolbar)
      toolbar_height = iw->toolbar_handle->allocation.height;
   else
      toolbar_height = 0;

   if (iw->show_statusbar)
      statusbar_height = iw->status_bar_container->allocation.height;
   else
      statusbar_height = 0;

   x_size = x_size;
   y_size = y_size + menubar_height + toolbar_height  + statusbar_height;

   if (x_size < 1)
      x_size = conf.imgwin_width;
   if (y_size < 1)
      y_size = conf.imgwin_height;

   iw->iv->x_pos = 0;
   iw->iv->y_pos = 0;

   gdk_window_resize (iw->window->window, x_size, y_size);
   imageview_draw_image (iw->iv);
}


/*
 *  imageview_set_window_title:
 *     @ set window title (using image file name).
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
imagewin_set_window_title (ImageWindow *iw)
{
   gchar buf[BUF_SIZE];
   gchar *filename = NULL, *dirname = NULL;

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (ImageViewList, iw->iv)) return;

   if (iw->iv->info->filename) {
      filename = g_basename (iw->iv->info->filename);
      dirname = g_dirname (iw->iv->info->filename);
   }

   if (iw->iv->buffer)
      g_snprintf (buf, BUF_SIZE, "%s (%s) - %s - ",
		  filename, dirname, PROG_NAME);
   else
      g_snprintf (buf, BUF_SIZE, "* %s (%s) - %s - *",
		  filename, dirname, PROG_NAME);

   gtk_window_set_title (GTK_WINDOW (iw->window), buf);

   g_free (dirname);
}


/*
 *  imageview_set_statusbar_info:
 *     @ set statubar strings.
 *
 *  im : Pointer to ImageWindow structure.
 */
static void
imagewin_set_statusbar_info (ImageWindow *iw)
{
   gchar buf[BUF_SIZE];

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (ImageViewList, iw->iv)) return;

   if (iw->iv->buffer)
      g_snprintf (buf, BUF_SIZE, "%s (Buffer ON)", iw->iv->info->filename);
   else
      g_snprintf (buf, BUF_SIZE, "%s (Buffer OFF)", iw->iv->info->filename);

   gtk_statusbar_push (GTK_STATUSBAR (iw->status_bar1), 1, buf);
   g_snprintf (buf, BUF_SIZE, "%s %d x %d", iw->iv->info->format, 
	       iw->iv->info->width, iw->iv->info->height);
   gtk_statusbar_push (GTK_STATUSBAR (iw->status_bar2), 1, buf);
}


static void
imagewin_render_cb (gpointer data)
{
   ImageWindow *iw = data;

   g_return_if_fail (iw);
   if (!g_list_find (ImageWinList, iw)) return;

   g_return_if_fail (iw->iv);
   if (!g_list_find (ImageViewList, iw->iv)) return;

   /* set statu bar */
   imagewin_set_statusbar_info (iw);

   /* set title */
   imagewin_set_window_title (iw);

   /* resize window */
   if (!iw->fullscreen && conf.imgwin_fit_to_image)
      imagewin_set_win_size (iw);

   if (iw->button.xscale)
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.xscale), iw->iv->x_scale);
   if (iw->button.yscale)
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.yscale), iw->iv->y_scale);
}


static void
imagewin_aspect_cb (gpointer data)
{
   ImageWindow *iw = data;

   gtk_widget_set_sensitive (iw->button.yscale, !iw->iv->keep_aspect);
}



/******************************************************************************
 *
 *   For Slide Show.
 *
 ******************************************************************************/
/*
 *  timeout_slideshow:
 *     @ Time out function for slideshow.
 *
 *  slideshow : Poiter to SlideShow struct.
 *  Return    : FALSE if end of slideshow or canceled by user.
 */
static gboolean
timeout_slideshow (SlideShow *slideshow)
{
   if ((!slideshow->pos && !slideshow->repeat) || !slideshow->running) {
      slideshow->running = FALSE;
      return FALSE;
   } else if (!slideshow->pos && slideshow->repeat)
      slideshow->pos = slideshow->filelist;

   imagewin_change_image_file (slideshow->iw, slideshow->pos->data);

   slideshow->pos = g_list_next (slideshow->pos);

   return TRUE;
}


/*
 *  imageview_exec_slideshow:
 *     @ Execute slide show;
 *
 *  slideshow : Poiter to SlideShow struct.
 */
void
imagewin_exec_slideshow (SlideShow *slideshow)
{
   slideshow->repeat = conf.slideshow_repeat;
   slideshow->pos = g_list_first (slideshow->filelist);

   if (!slideshow->pos)
      return;

   slideshow->iw = imagewin_initialize (slideshow->pos->data);

   /* override some parameters */
   slideshow->iw->iv->fit_to_frame = conf.slideshow_fit_to_win;
   slideshow->iw->iv->x_scale = conf.slideshow_img_scale;
   slideshow->iw->iv->y_scale = conf.slideshow_img_scale;
   slideshow->iw->iv->keep_aspect = conf.slideshow_keep_aspect;

   slideshow->iw->show_menubar   = conf.slideshow_menubar;
   slideshow->iw->show_toolbar   = conf.slideshow_toolbar;
   slideshow->iw->show_statusbar = conf.slideshow_statusbar;
   slideshow->iw->show_scrollbar = TRUE;
   slideshow->iw->fullscreen = conf.slideshow_fullscreen;

   imagewin_create_window (slideshow->iw);

   slideshow->iw->slideshow = slideshow;

   slideshow->running = TRUE;
   gtk_timeout_add (conf.slideshow_interval * 1000,
		    (GtkFunction) timeout_slideshow, slideshow);
}



/******************************************************************************
 *
 *   Callback functions for menubar.
 *
 ******************************************************************************/
/*
 *  cb_file_select:
 *     @ Callback function for main menu item (/File/Open).
 *       Open file browser.
 *
 *  data   :
 *  action :
 *  widget :
 */
static void
cb_file_select (gpointer data, guint action, GtkWidget *widget)
{
   create_filebrowser (NULL);
}


/*
 *  cb_open_thumbnail:
 *     @ Callback function for main menu item (/File/Open Thumbnail Window).
 *       Open new thumbnail window.
 *
 *  data   :
 *  action :
 *  widget :
 */
static void
cb_open_thumbnail (gpointer data, guint action, GtkWidget *widget)
{
   thumbwin_open_window();
}


/*
 *  cb_toggle_buffer:
 *     @ Callback function for main menu item (/File/Memory Buffer).
 *       Keep original image on memory or not.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_buffer (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      imageview_load_image_buf (iw->iv);
      iw->iv->buffer = TRUE;
   } else {
      iw->iv->buffer = FALSE;
      imageview_free_image_buf (iw->iv);
   }

   imagewin_set_window_title (iw);
   imagewin_set_statusbar_info (iw);
}


/*
 *  cb_window_close:
 *     @ Callback function for main menu item (/File/Close).
 *       Image window close.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_window_close (ImageWindow *iw, guint action, GtkWidget *widget)
{
   gtk_widget_destroy (iw->window);
}


/*
 *  cb_toggle_menubar:
 *     @ Callback function for main menu item (/View/Menu Bar).
 *       Show/Hide menu bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_menubar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->menubar_handle);
      iw->show_menubar = TRUE;
   } else {
      gtk_widget_hide (iw->menubar_handle);
      iw->show_menubar = FALSE;
   }

   imageview_show_image (iw->iv, iw->iv->rotate);
}


/*
 *  cb_toggle_toolbar:
 *     @ Callback function for main menu item (/View/Tool Bar).
 *       Show/Hide tool bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_toolbar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->toolbar_handle);
      iw->show_toolbar = TRUE;
   } else {
      gtk_widget_hide (iw->toolbar_handle);
      iw->show_toolbar = FALSE;
   }

   imageview_show_image (iw->iv, iw->iv->rotate);
}


/*
 *  cb_toggle_statusbar:
 *     @ Callback function for main menu item (/View/Status Bar).
 *       Show/Hide status bar.
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_statusbar (ImageWindow *iw, guint action, GtkWidget *widget)
{
   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (iw->status_bar_container);
      iw->show_statusbar = TRUE;
   } else {
      gtk_widget_hide (iw->status_bar_container);
      iw->show_statusbar = FALSE;
   }

   imageview_show_image (iw->iv, iw->iv->rotate);
}


/*
 *  cb_toggle_fullscreen:
 *     @ Callback function for main menu item (/View/Full Screen).
 *
 *  im     : Pointer to ImageWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_fullscreen (ImageWindow *iw, guint action, GtkWidget *widget)
{
   GdkWindow *gdk_window = iw->window->window;
   gint client_x, client_y, root_x, root_y;
   gint width, height;

   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gdk_window_get_origin (gdk_window, &root_x, &root_y);
      gdk_window_get_geometry (gdk_window, &client_x, &client_y,
			       &width, &height, NULL);

      gdk_window_move_resize (gdk_window, -client_x, -client_y,
			      gdk_screen_width (), gdk_screen_height ());

      iw->fullscreen = TRUE;
      iw->win_x = root_x - client_x;
      iw->win_y = root_y - client_y;
      iw->win_width  = width;
      iw->win_height = height;
   } else {
      gdk_window_move_resize (gdk_window, iw->win_x, iw->win_y,
			      iw->win_width, iw->win_height);
      iw->fullscreen = FALSE;
   }
}


/*
 *  cb_fit_to_image:
 *     @ Callback function for main menu item (/View/Full Screen).
 *     @ Fit window size to image.
 */
static void
cb_fit_to_image (ImageWindow *iw, guint action, GtkWidget *widget) {
   imagewin_set_win_size (iw);
}


/*
 *  cb_options:
 *     @ Callback function for main menu item (/Preference/Options).
 *     @ Open preference window.
 */
static void
cb_options () {
   prefs_open_window ();
}


/******************************************************************************
 *
 *  callback functions for toolbar.
 *
 ******************************************************************************/
/*
 *  cb_toolbar_open_button:
 *     @ Callback function for open button.
 *     @ Open file browser. 
 *
 *  widget :
 *  im     : Pointer to ImageWindow struct.
 */
static void
cb_toolbar_open_button (GtkWidget *widget)
{
   create_filebrowser (NULL);
}


/*
 *  cb_toolbar_prefs_button:
 *     @ Callback function for prefs button.
 *     @ Open preference button. 
 *
 *  widget :
 */
static void
cb_toolbar_prefs_button (GtkWidget *widget)
{
      prefs_open_window();
}


/*
 *  cb_toolbar_no_zoom:
 *     @ Callback function for toolbar no zoom button.
 *       Zoom image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_no_zoom (GtkWidget *widget, ImageWindow *iw)
{
   imageview_zoom (iw->iv, ZOOM_100, 0, 0);
}


/*
 *  cb_toolbar_zoom_in:
 *     @ Callback function for toolbar zoom in button.
 *       Zoom image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_zoom_in (GtkWidget *widget, ImageWindow *iw)
{
   imageview_zoom (iw->iv, ZOOM_IN, 0, 0);
}


/*
 *  cb_toolbar_zoom_out:
 *     @ Callback function for toolbar zoom out button.
 *       Zoom image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_zoom_out (GtkWidget *widget, ImageWindow *iw)
{
   imageview_zoom (iw->iv, ZOOM_OUT, 0, 0);
}


/*
 *  cb_toolbar_zoom_fit:
 *     @ Callback function for toolbar zoom fit button.
 *       Zoom image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void cb_toolbar_zoom_fit (GtkWidget *widget, ImageWindow *iw)
{
   imageview_zoom (iw->iv, ZOOM_FIT, 0, 0);
}


/*
 *  cb_toolbar_zoom:
 *     @ Callback function for toolbar zoom button.
 *       Zoom image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_zoom (GtkWidget *widget, ImageWindow *iw)
{
   iw->iv->x_scale = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(iw->button.xscale));
   iw->iv->y_scale = gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(iw->button.yscale));
   imageview_show_image (iw->iv, iw->iv->rotate);
}


/*
 *  cb_toolbar_rotate:
 *     @ Callback function for toolbar rotate button.
 *       Rotate image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_rotate (GtkWidget *widget, ImageWindow *iw)
{
   guint angle;

   /* convert to absolute angle */
   angle = ( iw->iv->rotate + 3 ) % 4;

   imageview_show_image (iw->iv, angle);
}


/*
 *  cb_toolbar_keep_aspect:
 *     @ Callback function for toolbar spin button.
 *     @ Set y_scale value to x_scale value if keep aspect.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_keep_aspect (GtkWidget *widget, ImageWindow *iw)
{
   gfloat scale;

   if (!iw->iv->keep_aspect)
      return;

   scale = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (iw->button.xscale));
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (iw->button.yscale), scale);
}


/*
 *  cb_toolbar_fit_window:
 *     @ Callback function for toolbar fit window button.
 *     @ Fit window size to image.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_fit_window (GtkWidget *widget, ImageWindow *iw)
{
   imagewin_set_win_size (iw);
}


/*
 *  cb_toolbar_fullscreen:
 *     @ Callback function for toolbar fullscreen button.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_toolbar_fullscreen (GtkWidget *widget, ImageWindow *iw)
{
   menu_check_item_set_active (iw->view_menu, "/Full Screen", !iw->fullscreen);
}


static gint
cb_scale_spinner_key_press (GtkWidget *widget, 
			    GdkEventKey *event,
			    ImageWindow *iw)
{
   g_return_val_if_fail (iw, FALSE);
        
   switch (event->keyval) {
   case GDK_Escape:
      gtk_window_set_focus (GTK_WINDOW (iw->window), NULL);
      return TRUE;
   }

   return TRUE;
}


/******************************************************************************
 *
 *  other callback functions.
 *
 ******************************************************************************/
/*
 *  cb_window_closed:
 *     @ Callback function for Image Window destroy event.
 *
 *  widget :
 *  im     : Pointer to ImageWindow structure.
 */
static void
cb_window_closed (GtkWidget *widget, ImageWindow *iw)
{
   if (!iw)
      return;

   if (iw->iv) {
      iw->iv = NULL;
   }

   /* update linked list */
   ImageWinList = g_list_remove (ImageWinList, iw);

   if (iw == shared_img_win)
      shared_img_win = NULL;

   /* if session is slideshow */
   if (iw->slideshow) {
      GList *node;

      node = iw->slideshow->filelist;
      while (node) {
	 g_free (node->data);
	 node = g_list_next (node);
      }
      if (iw->slideshow->running)
	 iw->slideshow->running = FALSE;
      else
	 g_free (iw->slideshow);
   }

   /* free mem for image view window info */
   g_free(iw);

   /* quit when last window */
   if (!g_list_first(ImageWinList) && !g_list_first(ThumbWinList)) {
      gimageview_quit();
   }
}



/******************************************************************************
 *
 *   public functions.
 *
 ******************************************************************************/
void
imagewin_change_image_file (ImageWindow *iw, const gchar *newfile)
{
   imageview_change_image_file (iw->iv, newfile);

   if (conf.imgwin_raise_window)
      gdk_window_raise (iw->window->window);

   if (iw->slideshow)
      iw->iv->fit_to_frame = conf.slideshow_fit_to_win;

   if (iw->iv->fit_to_frame && conf.imgwin_fit_to_image && !iw->fullscreen)
      gdk_window_resize (iw->window->window, conf.imgwin_width, conf.imgwin_height);

   imageview_show_image(iw->iv, iw->iv->rotate);
}


/*
 *   imagewin_initialize:
 *      @ allocate new ImageWindow & ImageView struct, and initialize it.
 *
 *   filename : image file name for open.
 *   Return   : Pointer to ImageWindow struct.
 */
ImageWindow *
imagewin_initialize (const gchar *filename)
{
   ImageWindow *iw;
   ImageView   *iv;

   iv = imageview_create (filename);
   if (!iv)
      return NULL;

   iw = g_new0 (ImageWindow, 1);
   if (!iw)
      return NULL;

   iw->iv = iv;
   iw->show_menubar   = conf.imgwin_show_menubar;
   iw->show_toolbar   = conf.imgwin_show_toolbar;
   iw->show_statusbar = conf.imgwin_show_statusbar;
   iw->fullscreen     = FALSE;

   iw->iv->render_cb_func = imagewin_render_cb;
   iw->iv->render_cb_data = iw;
   iw->iv->aspect_cb_func = imagewin_aspect_cb;
   iw->iv->aspect_cb_data = iw;

   ImageWinList = g_list_append (ImageWinList, iw);

   return iw;
}


/*
 *   imagewin_create_widnow.
 *      @ Create new iamge window. 
 *        ImageWindow struct must be initialized before call this function.
 *
 *   iw     : Pointer to ImageWindow structure.
 *   Return : Return TRUE if success.
 */
gboolean
imagewin_create_window (ImageWindow *iw)
{
   GtkWidget *vbox, *hbox;

   if (!iw)
      return FALSE;

   while (gtk_events_pending()) gtk_main_iteration();

   /* Image Window */
   iw->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
   gtk_window_set_policy(GTK_WINDOW(iw->window), TRUE, TRUE, FALSE);
   gtk_window_set_default_size (GTK_WINDOW(iw->window),
				conf.imgwin_width, conf.imgwin_height);
   gtk_widget_show (iw->window);
   gtk_signal_connect (GTK_OBJECT (iw->window), "destroy",
		       GTK_SIGNAL_FUNC (cb_window_closed), iw);
   imagewin_set_window_title (iw);

   /* Main vbox */
   vbox = gtk_vbox_new (FALSE, 0);
   iw->main_vbox = vbox;
   gtk_container_add (GTK_CONTAINER (iw->window), vbox);
   gtk_widget_show (vbox);

   /* Menu Bar */
   iw->menubar_handle = gtk_handle_box_new();
   gtk_container_set_border_width (GTK_CONTAINER(iw->menubar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), iw->menubar_handle, FALSE, FALSE, 0);
   gtk_widget_show (iw->menubar_handle);

   /* toolbar */
   iw->toolbar_handle = gtk_handle_box_new();
   gtk_container_set_border_width (GTK_CONTAINER(iw->toolbar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), iw->toolbar_handle, FALSE, FALSE, 0);
   gtk_widget_show (iw->toolbar_handle);
   iw->toolbar = create_toolbar(iw, iw->toolbar_handle);
   gtk_container_add(GTK_CONTAINER(iw->toolbar_handle), iw->toolbar);
   gtk_widget_show_all (iw->toolbar);

   gtk_toolbar_set_style (GTK_TOOLBAR(iw->toolbar), conf.imgwin_toolbar_style);

   /* draw area */
   gtk_widget_show_all (iw->iv->event_box);
   gtk_box_pack_start (GTK_BOX (iw->main_vbox), iw->iv->event_box,
		       TRUE, TRUE, 0 );

   /* status bar */
   hbox = gtk_hbox_new (FALSE, 0);
   iw->status_bar_container = hbox;
   gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   gtk_widget_show (hbox);

   iw->status_bar1 = gtk_statusbar_new ();
   gtk_container_border_width (GTK_CONTAINER (iw->status_bar1), 1);
   gtk_widget_set_usize(iw->status_bar1, 200, -1);
   gtk_box_pack_start (GTK_BOX (hbox), iw->status_bar1, TRUE, TRUE, 0);
   gtk_statusbar_push(GTK_STATUSBAR (iw->status_bar1), 1, "New Window");
   gtk_widget_show (iw->status_bar1);

   iw->status_bar2 = gtk_statusbar_new ();
   gtk_container_border_width (GTK_CONTAINER (iw->status_bar2), 1);
   gtk_widget_set_usize(iw->status_bar2, 50, -1);
   gtk_box_pack_end (GTK_BOX (hbox), iw->status_bar2, TRUE, TRUE, 0);
   gtk_widget_show (iw->status_bar2);

   /* create menus */
   create_imageview_menus (iw);

   /* Show Image */
   imageview_show_image(iw->iv, iw->iv->rotate);

   return TRUE;
}


/*
 *   imagewin_open_widnow.
 *      @ Initialize ImageWindow & ImageView struct and create new image window. 
 *
 *   filename : Image file name for open.
 *              If this value is NULL, create blank canvas.
 *   Return   : New ImageWindow struct.
 */
ImageWindow *
imagewin_open_window (const gchar *filename)
{
   ImageWindow *iw;

   iw = imagewin_initialize (filename);
   imagewin_create_window (iw);

   return iw;
}
