/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <unistd.h>

#include "gimageview.h"

#include "dirview.h"
#include "dnd.h"
#include "fileload.h"
#include "fileutil.h"
#include "gfileutil.h"
#include "gimv_image.h"
#include "gtkutils.h"
#include "image_window.h"
#include "menu.h"
#include "prefs.h"
#include "thumbnail.h"
#include "thumbnail_support.h"
#include "thumbnail_view.h"
#include "thumbnail_window.h"
#include "thumbview_detail.h"
#include "thumbview_list.h"
#include "thumbview_table.h"

typedef enum
{
   OPEN_IMAGE_AUTO,
   OPEN_IMAGE_PREVIEW,
   OPEN_IMAGE_NEW_WIN,
   OPEN_IMAGE_SHARED_WIN,
   OPEN_IMAGE_EXTERNAL
} OpenImageType;


/* callback functions */
static void   cb_open_image            (ThumbView      *tv,
					OpenImageType   action,
					GtkWidget      *menuitem);
static void   cb_recreate_thumbnail    (ThumbView      *tv,
					guint           action,
					GtkWidget      *menuitem);
static void   cb_remove_thumbnail      (ThumbView      *tv,
					guint           action,
					GtkWidget      *menuitem);
static void   cb_rename_file           (ThumbView      *tv,
					guint           action,
					GtkWidget      *menuitem);
static void   cb_remove_file           (ThumbView      *tv,
					guint           action,
					GtkWidget      *menuitem);


/* compare functions */
static int comp_func_spel  (gconstpointer data1, gconstpointer data2);
static int comp_func_size  (gconstpointer data1, gconstpointer data2);
static int comp_func_atime (gconstpointer data1, gconstpointer data2);
static int comp_func_mtime (gconstpointer data1, gconstpointer data2);
static int comp_func_ctime (gconstpointer data1, gconstpointer data2);
static int comp_func_type  (gconstpointer data1, gconstpointer data2);

/* other private functions */
static gboolean check_mode_num                       (gint         num);
static void     thumbview_destroy_bg_widgets         (ThumbView   *tv);
static void     thumbview_remove_mode_data           (ThumbView   *tv);
static void     thumbview_remove_thumbnail_mode_data (Thumbnail   *thumb);
static GList   *thumbview_add_thumb_data             (ThumbView   *tv,
						      GList       *filelist);
static GList   *thumbview_add_thumb_data             (ThumbView   *tv,
						      GList       *filelist);
static gchar   *get_uri_list                         (GList       *thumblist);
static void     open_image                           (ThumbWindow *tw,
						      Thumbnail   *thumb,
						      OpenImageType type);
static GtkWidget *create_progs_submenu               (ThumbView *tv);


GList *ThumbViewList;


ThumbDispMode thumb_view_modes[] =
{
   {N_("Thumbnail"),
    thumbtable_create,
    thumbtable_append_thumb_frames,
    thumbtable_add_thumbnail,
    thumbtable_refresh_thumbnail,
    thumbtable_redraw,
    thumbtable_resize,
    thumbtable_adjust,
    thumbtable_destroy_table,
    thumbtable_remove_thumbview_data,
    thumbtable_remove_thumbnail_data,
    thumbtable_set_selection},

   {N_("List (Icon)"),
    listview_create,
    list_view_append_thumb_frames,
    listview_add_thumbnail,
    listview_refresh_thumbnail,
    listview_redraw,
    listview_resize,
    listview_adjust,
    NULL,
    listview_remove_thumbview_data,
    listview_remove_thumbnail_data,
    listview_set_selection},

   {N_("List (Thumbnail)"),
    listview_create,
    list_view_append_thumb_frames,
    listview_add_thumbnail,
    listview_refresh_thumbnail,
    listview_redraw,
    listview_resize,
    listview_adjust,
    NULL,
    listview_remove_thumbview_data,
    listview_remove_thumbnail_data,
    listview_set_selection},

   {N_("Detail"),
    detailview_create,
    detailview_append_thumb_frames,
    detailview_add_thumbnail,
    detailview_refresh_thumbnail,
    detailview_redraw,
    detailview_resize,
    detailview_adjust,
    NULL,
    detailview_remove_thumbview_data,
    NULL,
    detailview_set_selection},

   {N_("Detail + Icon"),
    detailview_create,
    detailview_append_thumb_frames,
    detailview_add_thumbnail,
    detailview_refresh_thumbnail,
    detailview_redraw,
    detailview_resize,
    detailview_adjust,
    NULL,
    detailview_remove_thumbview_data,
    NULL,
    detailview_set_selection},

   {N_("Detail + Thumbnail"),
    detailview_create,
    detailview_append_thumb_frames,
    detailview_add_thumbnail,
    detailview_refresh_thumbnail,
    detailview_redraw,
    detailview_resize,
    detailview_adjust,
    NULL,
    detailview_remove_thumbview_data,
    NULL,
    detailview_set_selection},
};


/* reference popup menu for each thumbnail */
static GtkItemFactoryEntry thumb_button_popup_items [] =
{
   {N_("/Open in New Window"),       NULL,  cb_open_image,    OPEN_IMAGE_NEW_WIN,     NULL},
   {N_("/Open in Shared Window"),    NULL,  cb_open_image,    OPEN_IMAGE_SHARED_WIN,  NULL},
   {N_("/Open in External Program"), NULL,  NULL,             0,           "<Branch>"},
   {N_("/---"),                      NULL,  NULL,             0,           "<Separator>"},
   {N_("/Update Thumbnail"),         NULL,  cb_recreate_thumbnail,  0,     NULL},
   {N_("/Remove from List"),         NULL,  cb_remove_thumbnail,    0,     NULL},
   {N_("/---"),                      NULL,  NULL,             0,           "<Separator>"},
   {N_("/Rename"),                   NULL,  cb_rename_file,   0,           NULL},
   {N_("/Remove file"),              NULL,  cb_remove_file,   0,           NULL},
   {NULL, NULL, NULL, 0, NULL},
};


static guint    button = 0;
static gboolean pressed = FALSE;
static gboolean dragging = FALSE;


/******************************************************************************
 *
 *   Callback functions.
 *
 ******************************************************************************/
/*
 *  cb_open_image:
 *     @ Callback function for popup menu item at thumbnail button ("/Open").
 *     @ Open images on image window. If shared image window is existing,
 *       use it.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_open_image (ThumbView *tv, OpenImageType action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   if (action == OPEN_IMAGE_SHARED_WIN && shared_img_win) {
      thumb = thumblist->data;
      open_image (tv->thumb_window, thumb, action);
   } else if (action == OPEN_IMAGE_SHARED_WIN && !shared_img_win) {
      thumb = thumblist->data;
      open_image (tv->thumb_window, thumb, action);
   } else {
      node = thumblist;
      while (node) {
	 thumb = node->data;
	 open_image (tv->thumb_window, thumb, action);
	 node = g_list_next (node);
      }
   }

   g_list_free (thumblist);
}


/*
 *  cb_open_image:
 *     @ Callback function for popup menu item at thumbnail button
 *       ("/Open in External Program").
 *     @ Open images by external program
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_open_image_by_external (GtkWidget *menuitem, ThumbView *tv)
{
   Thumbnail *thumb;
   GList *thumblist, *node;
   gint action;
   gchar *cmd = NULL, *tmpstr = NULL, **pair;

   g_return_if_fail (menuitem && tv);

   thumblist = node = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   action = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (menuitem), "num"));

   /* find command */
   if (action < sizeof (conf.progs) / sizeof (conf.progs[0])) {
      pair = g_strsplit (conf.progs[action], ",", 2);
      if (!pair[1]) {
	 g_strfreev (pair);
	 return;
      } else {
	 cmd = g_strdup (pair[1]);
      }
      g_strfreev (pair);
   } else {
      return;
   }

   /* create command string */
   while (node) {
      thumb = node->data;
      tmpstr = g_strconcat (cmd, " ", "\"", thumb->info->filename, "\"",  NULL);
      g_free (cmd);
      cmd = tmpstr;
      node = g_list_next (node);
   }
   tmpstr = g_strconcat (cmd, " &", NULL);
   g_free (cmd);
   cmd = tmpstr;

   /* exec command */
   if (cmd) {
      system (cmd);
      g_free (cmd);
   }

   g_list_free (thumblist);
}


/*
 *  cb_recreate_thumbnail:
 *     @ Callback function for popup menu item at thumbnail button
 *       ("/Recreate Thumbnail").
 *     @ Create thumbnail from original image. 
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_recreate_thumbnail (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   node = thumblist;
   while (node) {
      thumb = node->data;
      node = g_list_next (node);
      if (!thumb) continue;

      thumbview_refresh_thumbnail (thumb);
   }

   g_list_free (thumblist);
}


/*
 *  cb_remove_thumbnail:
 *     @ Callback function of popup menu. ("/Remove from List")
 *     @ Remove specified thumbnail from list.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_remove_thumbnail (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist, *node;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   node = thumblist;
   while (node) {
      thumb = node->data;
      node = g_list_next (node);
      if (!thumb) continue;

      tv->thumblist = g_list_remove (tv->thumblist, thumb);
      thumbview_free_thumbnail (thumb);
   }

   thumbview_redraw (tv, tv->disp_mode, tv->container);

   g_list_free (thumblist);
}


/*
 *  cb_rename_file:
 *     @ Callback function of popup menu. ("/Rename file")
 *     @ Rename specified thumbnail from list.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_rename_file (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   Thumbnail *thumb;
   GList *thumblist;
   gchar *src_file, *dest_file, *dest_path, *src_cache_path, *dest_cache_path;
   gchar message[BUF_SIZE], *dirname;
   ConfirmType confirm;
   gboolean exist;
   struct stat dest_st;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist || g_list_length (thumblist) > 1) return;

   thumb = thumblist->data;
   if (!thumb) return;

   src_file  = g_basename(thumb->info->filename);
   dest_file = gtkutil_popup_textentry (_("Rename a file"), _("New file name: "),
				       src_file);
   if (!dest_file) return;
   if (!strcmp (src_file, dest_file)) {
      g_free (dest_file);
      return;
   }

   dirname = g_dirname (thumb->info->filename);
   dest_path = g_strconcat (dirname, "/", dest_file, NULL);
   g_free (dirname);
   exist = !lstat(dest_path, &dest_st);
   if (exist) {
      g_snprintf (message, BUF_SIZE,
		  _("File exist : %s\n\n"
		  "Overwrite?"), dest_path);
      confirm = gtkutil_confirm_dialog (_("File exist!!"), message, FALSE);
      if (confirm == CONFIRM_NO) {
	 g_free (dest_path);
	 return;
      }
   }

   /* rename file!! */
   if (rename (thumb->info->filename, dest_path) < 0) {
      g_snprintf (message, BUF_SIZE,
		  _("Faild to rename file :\n%s"), thumb->info->filename);
      gtkutil_message_dialog ("Error!!", message);
   }

   /* rename cache */
   if (thumb->cache_type > 0) {
      src_cache_path 
	 = thumbsupport_get_thumb_cache_path (thumb->info->filename,  thumb->cache_type);
      dest_cache_path
	 = thumbsupport_get_thumb_cache_path (dest_path, thumb->cache_type);
      if (rename (src_cache_path, dest_cache_path) < 0)
	 g_print (_("Faild to rename cache file :%s\n"), dest_path);
      g_free (src_cache_path);
      g_free (dest_cache_path);
   }

   thumbview_refresh_list (tv);

   g_free (dest_file);
   g_free (dest_path);
}


/*
 *  cb_remove_file:
 *     @ Callback function of popup menu. ("/Remove file")
 *     @ Remove specified file.
 *
 *  thumb    :
 *  action   :
 *  menuitem :
 */
static void
cb_remove_file (ThumbView *tv, guint action, GtkWidget *menuitem)
{
   ConfirmType confirm;
   Thumbnail *thumb;
   GtkWidget *progress_win;
   GList *thumblist, *node;
   gboolean cancel = FALSE;
   gint pos, length;
   gfloat progress;
   gchar message[BUF_SIZE], *dirname;

   g_return_if_fail (tv);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   length = g_list_length (thumblist);
   g_snprintf (message, BUF_SIZE,
	       _("Delete these %d files.\n"
		 "OK?"),
	       length);
   confirm = gtkutil_confirm_dialog (_("Confirm Deleting Files"), message, FALSE);
   if (confirm != CONFIRM_YES) {
      g_list_free (thumblist);
      return;
   }

   progress_win = gtkutil_create_progress_window (_("Delete File"), _("Deleting Files"),
						  &cancel, 300, -1);
   gtk_grab_add (progress_win);

   node = thumblist;
   while (node) {
      thumb = node->data;
      node = g_list_next (node);
      if (!thumb) continue;

      while (gtk_events_pending()) gtk_main_iteration();

      pos = g_list_position (thumblist, node);
      progress = (gfloat) pos / (gfloat) length;
      g_snprintf (message, BUF_SIZE, _("Deleting %s ..."), thumb->info->filename);
      gtkutil_progress_window_update (progress_win, NULL,
				      message, NULL, progress);

      dirname = g_dirname (thumb->info->filename);
      if (!iswritable (dirname)) {
	 g_snprintf (message, BUF_SIZE,
		     _("Permission denied : %s"), dirname);
	 gtkutil_message_dialog ("Error!!", message);
      } else if (remove (thumb->info->filename) < 0) {       /* remove file!! */
	 g_snprintf (message, BUF_SIZE,
		     _("Faild to delete file :\n%s"), thumb->info->filename);
	 gtkutil_message_dialog (_("Error!!"), message);
      }

      g_free (dirname);

      /* cancel */
      if (cancel) break;
   }

   gtk_grab_remove (progress_win);
   gtk_widget_destroy (progress_win);

   thumbview_refresh_list (tv);

   g_list_free (thumblist);
}


/*
 *  thumbview_thumb_button_press_cb:
 *     @ Callback function for mouse button pressed on thumbnail button event.
 *     @ If middle button pressed, open the image on image window.
 *     @ If right buton pressed, open popup menu.
 *
 *  widget :
 *  event  :
 *  thumb  :
 */
void
thumbview_thumb_button_press_cb (GtkWidget *widget,
				 GdkEventButton *event,
				 Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;
   GtkItemFactory *ifactory;
   GtkWidget *menuitem;
   GList *thumblist = NULL, *node;
   gchar *dirname;

   g_return_if_fail (thumb && event);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   thumbwin_notebook_drag_src_unset (tw);   /* FIXMEEEEEEEE!! */

   button = event->button;
   pressed = TRUE;

   /* reset selection */
   if (event->type == GDK_BUTTON_PRESS
	&& (event->button == 2 || event->button == 3))
   {
      if (!thumb->selected) {
	 thumbview_set_selection_all (tv, FALSE);
	 thumbview_set_selection (thumb, TRUE);
      }
   }

   if (event->type == GDK_BUTTON_PRESS && event->button == 1) {
      if (tw->show_preview || shared_img_win)
	 open_image (tw, thumb, OPEN_IMAGE_AUTO);
   } else if (event->type == GDK_2BUTTON_PRESS && event->button == 1) {
      if (conf.imgwin_open_new_win)
	 open_image (tw, thumb, OPEN_IMAGE_NEW_WIN);
      else
	 open_image (tw, thumb, OPEN_IMAGE_SHARED_WIN);
   } else if (event->type == GDK_BUTTON_PRESS && event->button == 3) {
      GtkWidget *popup_menu, *progs_submenu;
      guint n_menu_items;

      if (tv->popup_menu) {
	 gtk_widget_unref (tv->popup_menu);
	 tv->popup_menu = NULL;
      }

      thumblist = node = thumbview_get_selection_list (tv);
      if (!thumblist) return;

      /* create popup menu */
      n_menu_items = sizeof(thumb_button_popup_items)
	 / sizeof(thumb_button_popup_items[0]) - 1;
      popup_menu = menu_create_items(thumb->thumb_view->thumb_window->window,
				     thumb_button_popup_items, n_menu_items,
				     "<ThumbButtonPop>", tv);

      progs_submenu = create_progs_submenu (tv);
      menu_set_submenu (popup_menu, "/Open in External Program", progs_submenu);

      /* set sensitive */
      ifactory = gtk_item_factory_from_widget (popup_menu);
      if (g_list_length (thumblist) > 1) {
	 menuitem = gtk_item_factory_get_item (ifactory, "/Open in Shared Window");
	 gtk_widget_set_sensitive (menuitem, FALSE);
      }

      if (tv->progress || tv->mode == THUMB_MODE_DIR) {
	 menuitem = gtk_item_factory_get_item (ifactory, "/Remove from List");
	 gtk_widget_set_sensitive (menuitem, FALSE);
      }

      dirname = g_dirname (thumb->info->filename);
      if (g_list_length (thumblist) > 1 || !iswritable (dirname)) {	 
	 menuitem = gtk_item_factory_get_item (ifactory, "/Rename");
	 gtk_widget_set_sensitive (menuitem, FALSE);
      }
      g_free (dirname);

      /* popup */
      gtk_menu_popup(GTK_MENU(popup_menu), NULL, NULL,
                NULL, NULL, event->button, event->time);

      tv->popup_menu = popup_menu;

      g_list_free (thumblist);
   }
}


void
thumbview_thumb_button_release_cb (GtkWidget *widget,
				   GdkEventButton *event,
				   Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;

   g_return_if_fail (thumb && event);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   tw = tv->thumb_window;
   g_return_if_fail (tw);

   thumbwin_notebook_drag_src_reset (tw);   /* FIXMEEEEEEEEE!!! */

   if(pressed && !dragging) {
      if (event->button == 2) {
	 open_image (tw, thumb, OPEN_IMAGE_AUTO);
      }
   }

   button   = 0;
   pressed  = FALSE;
   dragging = FALSE;
}


gint
thumbview_motion_notify_cb (GtkWidget *widget,
			    GdkEventMotion *event,
			    Thumbnail *thumb)
{
   ThumbView   *tv;
   ThumbWindow *tw;

   g_return_val_if_fail (thumb && event, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   tw = tv->thumb_window;
   g_return_val_if_fail (tw, FALSE);

   if (!pressed)
      return FALSE;

   dragging = TRUE;

   return TRUE;
}


void
thumbview_drag_begin_cb (GtkWidget *widget,
			 GdkDragContext *context,
			 gpointer data)
{
   ThumbView *tv = data;
   Thumbnail *thumb;
   GList *thumblist;
   GdkPixmap *pixmap;
   GdkBitmap *mask;
   GdkColormap *colormap;

   g_return_if_fail (tv && widget);

   thumblist = thumbview_get_selection_list (tv);
   if (!thumblist) return;

   thumb = thumblist->data;
   if (g_list_length (thumblist) == 1 && thumb->icon) {
      gtk_pixmap_get (GTK_PIXMAP (thumb->icon), &pixmap, &mask);
      colormap = gdk_colormap_get_system ();
      gtk_drag_set_icon_pixmap (context, colormap, pixmap, mask, -7, -7);
   }
}


void
thumbview_drag_data_get_cb (GtkWidget *widget,
			    GdkDragContext *context,
			    GtkSelectionData *seldata,
			    guint info,
			    guint time,
			    gpointer data)
{
   ThumbView *tv = data;
   GList *thumblist;
   gchar *uri_list;

   g_return_if_fail (tv && widget);

   thumblist = thumbview_get_selection_list (tv);

   if (!thumblist) {
      gtkutil_message_dialog (_("Error!!"), _("No files specified!!"));
      return;
   }

   switch (info) {
   case TARGET_URI_LIST:
      if (!thumblist) return;
      uri_list = get_uri_list (thumblist);
      gtk_selection_data_set(seldata, seldata->target,
			     8, uri_list, strlen(uri_list));
      g_free (uri_list);
      break;
   default:
      break;
   }
}


/*
 *  thumbview_drag_data_received_cb:
 *     @ reference "drag_data_received" event callback function.
 */
void
thumbview_drag_data_received_cb (GtkWidget *widget,
				 GdkDragContext *context,
				 gint x, gint y,
				 GtkSelectionData *seldata,
				 guint info,
				 guint time,
				 gpointer data)
{
   ThumbView *tv = data;
   OpenFiles *files;
   GList *list;
   GtkWidget *src_widget;

   g_return_if_fail (tv && widget);

   src_widget = gtk_drag_get_source_widget (context);
   if (src_widget == widget) return;

   if (tv->mode == THUMB_MODE_DIR && iswritable (tv->dirname)) {
      dnd_file_operation (tv->dirname, context, seldata, time, tv->thumb_window);
   } else if (tv->mode == THUMB_MODE_COLLECTION) {
      list = dnd_get_file_list (seldata->data);
      files = files_loader_new ();
      files->filelist = list;
      thumbview_append_thumbnail (tv, files, FALSE);
      files_loader_delete (files);
   }
}


void
thumbview_drag_end_cb (GtkWidget *widget, GdkDragContext *drag_context,
		       gpointer data)
{
   ThumbView *tv = data;

   g_return_if_fail (tv);

   if (conf.dnd_refresh_list_always)
      thumbview_refresh_list (tv);
}


void
thumbview_drag_data_delete_cb (GtkWidget *widget, GdkDragContext *drag_context,
		     gpointer data)
{
   ThumbView *tv = data;

   g_return_if_fail (tv);

   if (!conf.dnd_refresh_list_always)
      thumbview_refresh_list (tv);
}



/******************************************************************************
 *
 *   Compare functions.
 *
 ******************************************************************************/
/* sort by spel */
static int
comp_func_spel (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;
   return strcmp ((gchar *) thumb1->info->filename,
		  (gchar *) thumb2->info->filename);
}


/* sort by file size */
static int
comp_func_size (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;
   return thumb1->info->st.st_size - thumb2->info->st.st_size;
}


/* sort by time of las access */
static int
comp_func_atime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;
   return thumb1->info->st.st_atime - thumb2->info->st.st_atime;
}


/* sort by time of last modification */
static int
comp_func_mtime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;
   return thumb1->info->st.st_mtime - thumb2->info->st.st_mtime;
}


/* sort by time of last change */
static int
comp_func_ctime (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;
   return thumb1->info->st.st_ctime - thumb2->info->st.st_ctime;
}


/* sort by time of file name extension */
static int
comp_func_type (gconstpointer data1, gconstpointer data2)
{
   Thumbnail *thumb1, *thumb2;
   gchar *ext1, *ext2;

   thumb1 = (Thumbnail *) data1;
   thumb2 = (Thumbnail *) data2;

   ext1 = strrchr (thumb1->info->filename, '.');
   if (ext1)
      ext1++;
   ext2 = strrchr (thumb2->info->filename, '.');
   if (ext2)
      ext2++;
   return g_strcasecmp ((gchar *) ext1, (gchar *) ext2);
}



/******************************************************************************
 *
 *   Other Private Functions.
 *
 ******************************************************************************/
static gboolean
check_mode_num (gint num)
{
   gint length = thumbview_get_disp_modes_length ();

   if (num > length || num < 0)
      return FALSE;
   else
      return TRUE;
}


static void
thumbview_destroy_bg_widgets (ThumbView *tv)
{
   gint i, n_modes;

   if (!tv) return;

   n_modes = thumbview_get_disp_modes_length ();
   for (i = 0; i < n_modes; i++) {
      if (i != tv->disp_mode && thumb_view_modes[i].destroy_func)
	 thumb_view_modes[i].destroy_func (tv);
   }
}


static void
thumbview_remove_mode_data (ThumbView *tv)
{
   gint i, n_modes;

   if (!tv) return;

   n_modes = thumbview_get_disp_modes_length ();
   for (i = 0; i < n_modes; i++) {
      if (thumb_view_modes[i].thumbview_data_remove_func)
	 thumb_view_modes[i].thumbview_data_remove_func (tv);
   }
}


static void
thumbview_remove_thumbnail_mode_data (Thumbnail *thumb)
{
   gint i, n_modes;

   if (!thumb) return;

   n_modes = thumbview_get_disp_modes_length ();
   for (i = 0; i < n_modes; i++) {
      if (thumb_view_modes[i].thumbnail_data_remove_func)
	 thumb_view_modes[i].thumbnail_data_remove_func (thumb);
   }
}


/*
 *  thumbview_add_thumb_data:
 *     @ Store file info to new Thumbnail struct, and add to thumblist GList
 *       in ThumbView struct. 
 *
 *  tv       : Pointer to ThumbView struct.
 *  filelist : file list.
 *  Return   : Pointer to head of added thumbnail list.
 */
static GList *
thumbview_add_thumb_data (ThumbView *tv, GList *filelist)
{
   Thumbnail  *thumb;
   GList *node, *retval = NULL;
   struct stat st;

   if (!tv || !filelist) return NULL;

   node = filelist;
   while (node) {
      if (!stat(node->data, &st)) {
	 thumb = g_new0 (Thumbnail, 1);
	 thumb->thumb_view = tv;
	 thumb->info = image_info_get ((const gchar *) node->data);
	 thumb->sync = FALSE;
	 thumb->mode_data = g_hash_table_new (g_str_hash, g_str_equal);
	 /* FIXME!! */
	 //thumb->button = NULL;
	 tv->thumblist = g_list_append (tv->thumblist, thumb);

	 /* update file num info */
	 tv->thumb_window->filenum++;
	 tv->thumb_window->filesize += st.st_size;
	 tv->filenum++;
	 tv->filesize += st.st_size;

	 if (!retval)
	    retval = g_list_find (tv->thumblist, thumb);
      }
      node = g_list_next (node);
   }
   return retval;
}


static gchar *
get_uri_list (GList *thumblist)
{
   gchar *path;
   gchar *uri;
   GList *node;
   Thumbnail *thumb;
 
   if (!thumblist) return NULL;

   uri = g_strdup ("");
   node = thumblist;
   while (node) {
      thumb = node->data;

      path = g_strconcat (uri, "file:", thumb->info->filename, "\r\n", NULL);
      g_free (uri);

      uri = g_strdup (path);
      g_free (path);

      node = g_list_next (node);
   }

   return uri;
}


static GList *
next_image (ImageView *iv, gpointer list_owner,
	    GList *current, gpointer data)
{
   ImageWindow *iw = data;
   GList *next;
   Thumbnail *thumb;
   ThumbView *tv;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (current, NULL);

   next = g_list_next (current);
   if (!next) next = g_list_first (current);
   g_return_val_if_fail (next, NULL);

   thumb = next->data;
   g_return_val_if_fail (thumb, NULL);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, NULL);

   thumbview_set_selection_all (tv, FALSE);
   thumbview_set_selection (thumb, TRUE);
   thumbview_adjust (tv, thumb);

   if (iw)
      imagewin_change_image_file (iw, thumb->info->filename);
   else
      imageview_change_image_file (iv, thumb->info->filename);
   imageview_show_image (iv, iv->rotate);

   return next;
}


static GList *
prev_image (ImageView *iv, gpointer list_owner,
	    GList *current, gpointer data)
{
   ImageWindow *iw = data;
   GList *prev;
   Thumbnail *thumb;
   ThumbView *tv;

   g_return_val_if_fail (iv, NULL);
   g_return_val_if_fail (current, NULL);

   prev = g_list_previous (current);
   if (!prev) prev = g_list_last (current);
   g_return_val_if_fail (prev, NULL);

   thumb = prev->data;
   g_return_val_if_fail (thumb, NULL);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, NULL);

   thumbview_set_selection_all (tv, FALSE);
   thumbview_set_selection (thumb, TRUE);
   thumbview_adjust (tv, thumb);

   if (iw)
      imagewin_change_image_file (iw, thumb->info->filename);
   else
      imageview_change_image_file (iv, thumb->info->filename);
   imageview_show_image (iv, iv->rotate);

   return prev;
}


static void
remove_list (ImageView *iv, gpointer list_owner, gpointer data)
{
   ThumbView *tv = list_owner;
   GList *node;

   g_return_if_fail (iv);
   g_return_if_fail (tv);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return;

   tv->related_image_view = g_list_remove (tv->related_image_view, iv);
}


static void
open_image (ThumbWindow *tw, Thumbnail *thumb, OpenImageType type)
{
   ThumbView *tv;
   ImageWindow *iw = NULL;
   ImageView   *iv;
   GList *current, *node;

   g_return_if_fail (tw && thumb);

   tv = thumb->thumb_view;
   g_return_if_fail (tv);

   if ((type == OPEN_IMAGE_AUTO && tw->show_preview)
       || type == OPEN_IMAGE_PREVIEW)
   {
      iv = tw->iv;
      imageview_change_image_file (iv, thumb->info->filename);
      imageview_show_image (iv, iv->rotate);

   } else if (shared_img_win &&
	      ((type == OPEN_IMAGE_AUTO && !conf.imgwin_open_new_win)
	      ||type == OPEN_IMAGE_SHARED_WIN))
   {
      iw = shared_img_win;
      imagewin_change_image_file (iw, thumb->info->filename);
      iv = iw->iv;

   } else {
      iw = imagewin_open_window (thumb->info->filename);
      if (iw && (type == OPEN_IMAGE_SHARED_WIN
	   || (type == OPEN_IMAGE_AUTO && !conf.imgwin_open_new_win)))
      {
	 shared_img_win = iw;
      }
      iv = iw->iv;
   }

   current = g_list_find (tv->thumblist, thumb);
   imageview_set_list (iv, tv->thumblist, current,
		       (gpointer) tv,
		       next_image, iw,
		       prev_image, iw,
		       NULL, NULL,
		       remove_list, NULL);
   node = g_list_find (tv->related_image_view, iv);
   if (!node) {
      gint num;
      tv->related_image_view = g_list_append (tv->related_image_view, iv);
      num = g_list_length (tv->related_image_view);
   }
}


static GtkWidget *
create_progs_submenu (ThumbView *tv)
{
   GtkWidget *menu;
   GtkWidget *menu_item;
   gint i, conf_num = sizeof (conf.progs) / sizeof (conf.progs[0]);
   gchar **pair;

   menu = gtk_menu_new();

   /* count items num */
   for (i = 0; i < conf_num; i++) {
      if (!conf.progs[i]) continue;

      pair = g_strsplit (conf.progs[i], ",", 2);

      if (pair[0] && pair[1]) {
	 menu_item = gtk_menu_item_new_with_label (pair[0]);
	 gtk_object_set_data (GTK_OBJECT (menu_item), "num", GINT_TO_POINTER (i));
	 gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
			     GTK_SIGNAL_FUNC (cb_open_image_by_external), tv);
	 gtk_menu_append (GTK_MENU (menu), menu_item);
	 gtk_widget_show (menu_item);
      }

      g_strfreev (pair);
   }

   return menu;
}


/******************************************************************************
 *
 *   Public Functions.
 *
 ******************************************************************************/
/*
 *  thumbview_get_disp_modes_length:
 *     @ Return number of display modes.
 *
 *  Return : number of display modes
 */
gint
thumbview_get_disp_modes_length ()
{
   gint length;

   length = sizeof (thumb_view_modes) / sizeof (thumb_view_modes[0]);

   return length;
}


/*
 *  thumbview_label_to_num:
 *     @ Convert text thumbnail view display mode ID to number one.
 *
 *  label : thumbnail view display mode ID by string.
 */
gint
thumbview_label_to_num(gchar *label)
{
   gint i, length;

   length = thumbview_get_disp_modes_length ();
   for (i = 0; i < length; i++) {
      if (!strcmp(thumb_view_modes[i].label, label))
	 return i;
   }
   return -1;
}


/*
 *  thumbview_num_to_label:
 *     @ Convert number thumbnail view display mode ID to text one.
 *
 *  num : thumbnail view display mode ID by int.
 */
gchar *
thumbview_num_to_label (gint num)
{
   gint length;

   length = thumbview_get_disp_modes_length ();
   if (check_mode_num (num))
      return thumb_view_modes[num].label;
   else
      return NULL;
}


/*
 *  thumbview_get_disp_mode_labels:
 *     @ return display mode labels.
 *
 *  length_ret: labels array length for return.
 *  Return : array of display mode label strings.
 */
gchar **
thumbview_get_disp_mode_labels (gint *length_ret)
{
   gint i, num;
   gchar **labels;

   num = *length_ret = sizeof (thumb_view_modes) / sizeof (thumb_view_modes[0]);
   labels = g_new0 (gchar *, num + 1);
   for (i = 0; i < num; i++)
      labels[i] = g_strdup (thumb_view_modes[i].label);
   labels[num] = NULL;

   return labels;
}


/*
 *  thumbview_free_thumbnail:
 *     @ Free Thumbnail struct.
 *
 *  thumb : Pointer to Thumbnail struct to delete.
 */
void
thumbview_free_thumbnail (Thumbnail *thumb)
{
   if (!thumb) return;

   if (thumb->pixmap) {
      gtk_widget_unref (thumb->pixmap);
      thumb->pixmap = NULL;
   }
   if (thumb->icon) {
      gtk_widget_unref (thumb->icon);
      thumb->icon = NULL;
   }
   image_info_unref (thumb->info);
   thumb->info = NULL;
   thumbview_remove_thumbnail_mode_data (thumb);
   g_hash_table_destroy (thumb->mode_data);
   g_free (thumb);
}


/*
 *  thumbview_free:
 *     @ Free ThumbView struct.
 *
 *  tv : Pointer to ThumbView struct to free.
 */
void
thumbview_free (ThumbView *tv)
{
   ThumbWindow *tw;
   GList *node, *node_obs;
   gint num;

   if (!tv) return;

   tw = tv->thumb_window;
   if (tw) {
      tw->filenum -= tv->filenum;
      tw->filesize -= tv->filesize;
   }

   ThumbViewList = g_list_remove (ThumbViewList, tv);

   if (GTK_BIN (tv->container)->child)
      gtk_widget_destroy (GTK_BIN (tv->container)->child);   

   if (tv->progress && tv->progress->status != WINDOW_DESTROYED) {
      tv->progress->status = CONTAINER_DESTROYED;
   }

   /* destroy relation */
   node = tv->related_image_view;
   num = g_list_length (tv->related_image_view);
   while (node) {
      ImageView *iv = node->data;
      node = g_list_next (node);
      imageview_remove_list (iv, (gpointer) tv);
   }

   thumbview_destroy_bg_widgets (tv);

   node = g_list_first (tv->thumblist);

   while (node) {
      Thumbnail *thumb = node->data;

      thumbview_free_thumbnail (thumb);
      node->data = NULL;

      node_obs = node;
      node = g_list_next (node_obs);

      tv->thumblist = g_list_remove (tv->thumblist, node_obs);
   }

   if (tv->popup_menu) {
      gtk_widget_unref (tv->popup_menu);
      tv->popup_menu = NULL;
   }

   thumbview_remove_mode_data (tv);
   g_hash_table_destroy (tv->disp_mode_data);
   g_list_free (tv->thumblist);
   g_free (tv->dirname);
   g_free (tv->tabtitle);
   g_free (tv);
}


/*
 *  thumbview_delete:
 *     @ Delete specified thumbnail table.
 *
 *  tv : Pointer to ThumbView struct to delete.
 */
void
thumbview_delete (ThumbView *tv)
{
   if (!tv) return;
   thumbview_free (tv);
}


/*
 *  thumbview_find_opened_dir:
 *
 *  path   : directory name for find.
 *  Return : Pointer to ThumbView struct. If not found, return NULL.
 */
ThumbView *
thumbview_find_opened_dir (const gchar *path)
{
   ThumbView *tv;
   GList *node;

   if (!path) return NULL;

   node = g_list_first (ThumbViewList);
   while (node) {
      tv = node->data;
      if (tv->mode == THUMB_MODE_DIR && !strcmp (path, tv->dirname))
	 return tv;
      node = g_list_next (node);
   }

   return NULL;
}


/*
 *  thumbview_sort_data:
 *     @ sort thumbnail list.
 *
 *  tv : Pointer to ThumbView struct to sort thumbnails.
 */
void
thumbview_sort_data (ThumbView *tv)
{
   GtkWidget *menuitem;
   GtkItemFactory *ifactory;
   gboolean reverse = FALSE;

   if (!tv) return;

   ifactory = gtk_item_factory_from_widget (tv->thumb_window->sort_menu);
   menuitem = gtk_item_factory_get_item(ifactory, "/Reverse Order");
   if (menuitem)
      reverse = GTK_CHECK_MENU_ITEM(menuitem)->active;

   /* sort thumbnail */
   if (tv->thumb_window->sortitem == NAME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_spel);
   else if (tv->thumb_window->sortitem == SIZE)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_size);
   else if (tv->thumb_window->sortitem == ATIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_atime);
   else if (tv->thumb_window->sortitem == MTIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_mtime);
   else if (tv->thumb_window->sortitem == CTIME)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_ctime);
   else if (tv->thumb_window->sortitem == TYPE)
      tv->thumblist = g_list_sort (tv->thumblist, comp_func_type);

   if (reverse)
      tv->thumblist = g_list_reverse (tv->thumblist);
}


/*
 *  thumbview_load_thumbnails:
 *    @ Load specified thumbnails and draw to the thumbnail container.
 *
 *  tv     : Pointer to the ThumbView struct.
 *  start  : The thumbnail list.
 *  Return : TRUE if success.
 */
gboolean
thumbview_load_thumbnails (ThumbView *tv, GList *start, gint dest_mode)
{
   OpenFiles *files = tv->progress;
   GList *node;

   if (!tv || !start || !check_mode_num (dest_mode))
      return FALSE;

   node = start;

   tv->progress->window = tv->thumb_window->window;
   tv->progress->progressbar = tv->thumb_window->progressbar;
   tv->progress->num = g_list_length (start);

   while (node && files->status < CANCEL) {
      Thumbnail *thumb = (Thumbnail *) node->data;

      if (tv->progress->pos % conf.thumbwin_redraw_interval == 0)
	 while (gtk_events_pending()) gtk_main_iteration();

      create_thumbnail_auto (thumb, tv->ThumbnailSize, files->thumb_load_type);

      if (thumb_view_modes[dest_mode].add_thumb_func)
	 thumb_view_modes[dest_mode].add_thumb_func (thumb, dest_mode,
						     files->thumb_load_type);


      if(files->status < 0) {
	 ThumbViewList = g_list_remove (ThumbViewList, tv);
	 thumbview_free (tv);
	 break;
      } else {
	 /* update progress info */
	 tv->progress->now_file = thumb->info->filename;
	 tv->progress->pos = g_list_position (start, node) + 1;
	 thumbwin_set_statusbar_page_info (tv->thumb_window, THUMB_CURRENT_PAGE);
      }

      node = g_list_next (node);
   }

   return TRUE;
}


/*
 *  thumbview_append_thumbnail:
 *     @ Store file info to new Thumbnail struct, and add to thumblist GList
 *       in ThumbView struct. 
 *
 *  tv     : Pointer to ThumbView struct for open thumbnails.
 *  fieles : Pointer to OpenFiles struct.
 *  force  : Force append thumbnail unless any mode.
 *  Return : Retrun TRUE if success.
 */
gboolean
thumbview_append_thumbnail (ThumbView *tv, OpenFiles *files, gboolean force)
{
   GList *start_pos;

   if (!tv || !files || (tv->mode == THUMB_MODE_DIR && !force)
       || !check_mode_num (tv->disp_mode))
      return FALSE;

   tv->progress = files;

   start_pos = thumbview_add_thumb_data (tv, files->filelist);

   thumb_view_modes[tv->disp_mode].add_thumb_frames_func (tv, start_pos, tv->disp_mode);

   thumbview_load_thumbnails (tv, start_pos, tv->disp_mode);

   tv->progress = NULL;

   if (files->status >= 0)
      thumbwin_set_statusbar_page_info (tv->thumb_window, THUMB_CURRENT_PAGE);

   return TRUE;
}


/*
 *  thumbview_refresh_thumbnail:
 *     @ Create thumbnail from original image file.
 *
 *  thumb  : Pointer to the Thumbnail struct to refresh.
 *  Return : True if success.
 */
gboolean
thumbview_refresh_thumbnail (Thumbnail *thumb)
{
   ThumbView *tv = thumb->thumb_view;
   gboolean retval = FALSE;

   if (!thumb || !check_mode_num (tv->disp_mode))
      return FALSE;

   create_thumbnail_auto (thumb, tv->ThumbnailSize, CREATE_THUMB);

   if (thumb_view_modes[tv->disp_mode].refresh_thumb_func)
      retval = thumb_view_modes[tv->disp_mode].refresh_thumb_func (thumb);

   return retval;
}


/*
 *  thumbview_refresh_list:
 *     @ Create thumbnail from original image file.
 *
 *  thumb  : Pointer to the Thumbnail struct to refresh.
 *  Return : True if success.
 */
gboolean
thumbview_refresh_list (ThumbView *tv)
{
   OpenFiles *files;
   GList *thumbnode, *node;
   Thumbnail *thumb;
   gchar *filename;
   gboolean exist = FALSE;
   struct stat st;
   gint flags;

   if (!tv || !check_mode_num (tv->disp_mode))
      return FALSE;

   files = files_loader_new ();

   if (tv->mode == THUMB_MODE_DIR) {
      g_return_val_if_fail (tv->dirname, FALSE);

      flags = GETDIR_FOLLOW_SYMLINK;
      if (conf.read_dotfile)
	 flags = flags | GETDIR_READ_DOT;
      if (conf.detect_filetype_by_ext)
	 flags = flags | GETDIR_DETECT_EXT;
      get_dir (tv->dirname, flags, &files->filelist, NULL);

      thumbnode = g_list_first (tv->thumblist);
      while (thumbnode) {
	 thumb = thumbnode->data;
	 thumbnode = g_list_next (thumbnode);

	 /* remove same file from files->filelist */
 	 node = g_list_first (files->filelist);
	 while (node) {
	    filename = node->data;
	    if (!strcmp (filename, thumb->info->filename)) {
	       /* check modification time */
	       if (!stat (filename, &st)
		   && (thumb->info->st.st_mtime != st.st_mtime))
	       {
		  exist = FALSE;
	       } else {
		  exist = TRUE;
		  files->filelist = g_list_remove (files->filelist, filename);
		  g_free (filename);
	       }
	       break;
	    }
	    node = g_list_next (node);
	 }

	 /* FIXME!! */
	 /* remove obsolete data */
	 if (!exist) {
	    tv->thumblist = g_list_remove (tv->thumblist, thumb);
	    tv->filenum--;
	    tv->thumb_window->filenum--;
	    tv->filesize -= thumb->info->st.st_size;
	    tv->thumb_window->filesize -= thumb->info->st.st_size;
	    thumbview_free_thumbnail (thumb);
	 }
	 /* END FIXME!! */

	 exist = FALSE;
      }

      thumbview_redraw (tv, tv->disp_mode, tv->container);

      /* append new files */
      if (files->filelist) {
	 thumbview_append_thumbnail (tv, files, TRUE);
      }

   } else if (tv->mode == THUMB_MODE_COLLECTION) {
   }

   files_loader_delete (files);

   thumbview_redraw (tv, tv->disp_mode, tv->container);
   thumbwin_set_statusbar_page_info (tv->thumb_window, THUMB_CURRENT_PAGE);

   return TRUE;
}


/*
 *  thumbview_redraw:
 *     @ Redraw thumbnail view or switch display mode or move to new container.
 *
 *  tv         : Pointer to the Thumbview struct.
 *  mode       : new display mode for switching display mode.
 *  scroll_win : new container for moveng thumbnail view.
 */
GtkWidget *
thumbview_redraw (ThumbView *tv, gint mode, GtkWidget *scroll_win)
{
   GtkWidget *widget = NULL;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   if (!check_mode_num (tv->disp_mode) || !check_mode_num (mode))
      return NULL;

   if (!scroll_win)
      scroll_win = tv->container;

   /* sort thumbnail list */
   thumbview_sort_data (tv);

   /* remove (or move to background) current widget */
   if (thumb_view_modes[tv->disp_mode].redraw_func)
      thumb_view_modes[tv->disp_mode].redraw_func (tv, mode, NULL);
   /* create (or move to foreground) new widget */
   if (thumb_view_modes[mode].redraw_func)
      widget = thumb_view_modes[mode].redraw_func (tv, mode, scroll_win);

   tv->disp_mode = mode;
   tv->container = scroll_win;

   return widget;
}


/*
 *  thumbview_redraw:
 *     @ Switch display mode.
 *
 *  tv         : Pointer to the Thumbview struct.
 *  mode       : new display mode for switching display mode.
 */
GtkWidget *
thumbview_change_mode (ThumbView *tv, gint mode)
{
   GtkWidget *thumbview;
   GList *node;

   g_return_val_if_fail (tv, NULL);

   node = g_list_find (ThumbViewList, tv);
   if (!node) return NULL;

   if (!check_mode_num (mode))
      return NULL;

   thumbview = thumbview_redraw (tv, mode, tv->container);

   return thumbview;
}


/*
 *  thumbview_resize:
 *     @ Resize thumbnail view widget. 
 *
 *  tv     : Pointer to the ThumbView struct.
 *  Return : New thumbnail view widget.
 */
GtkWidget *
thumbview_resize (ThumbView *tv)
{
   GtkWidget *retval;

   if (!tv || !check_mode_num (tv->disp_mode))
      return NULL;

   if (!thumb_view_modes[tv->disp_mode].resize_func)
      return NULL;

   retval = thumb_view_modes[tv->disp_mode].resize_func (tv);

   return retval;
}


/*
 *  thumbview_adjust:
 *     @  
 *
 *  tv     : Pointer to the ThumbView struct.
 *  thumb  : Pointer to the Thumbnail struct.
 */
void
thumbview_adjust (ThumbView *tv, Thumbnail *thumb)
{
   if (!tv || !check_mode_num (tv->disp_mode)) return;

   if (thumb_view_modes[tv->disp_mode].adjust_func)
      thumb_view_modes[tv->disp_mode].adjust_func (tv, thumb);

   return;
}


GList *
thumbview_get_selection_list (ThumbView *tv)
{
   GList *list = NULL, *node;
   Thumbnail *thumb;

   g_return_val_if_fail (tv, NULL);

   node = tv->thumblist;
   if (!node) return NULL;

   while (node) {
      thumb = node->data;

      if (thumb->selected)
	 list = g_list_append (list, thumb);

      node = g_list_next (node);
   }

   return list;
}


gboolean
thumbview_set_selection (Thumbnail *thumb, gboolean select)
{
   ThumbView *tv;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv && check_mode_num (tv->disp_mode), FALSE);

   if (thumb_view_modes[tv->disp_mode].set_selection_func)
      thumb_view_modes[tv->disp_mode].set_selection_func (thumb, select);
   else
      return FALSE;

   return TRUE;
}


gboolean
thumbview_set_selection_all (ThumbView *tv, gboolean select)
{
   Thumbnail *thumb;
   GList *node;

   g_return_val_if_fail (tv, FALSE);
   g_return_val_if_fail (tv->thumblist, FALSE);

   node = tv->thumblist;
   while (node) {
      thumb = node->data;
      thumbview_set_selection (thumb, select);
      node = g_list_next (node);
   }

   return TRUE;
}


gboolean
thumbview_set_selection_multiple (Thumbnail *thumb,
				  gboolean reverse, gboolean clear)
{
   ThumbView *tv;
   Thumbnail *thumb_tmp;
   GList *node, *current_node;
   gboolean retval = FALSE;

   g_return_val_if_fail (thumb, FALSE);

   tv = thumb->thumb_view;
   g_return_val_if_fail (tv, FALSE);

   node = current_node = g_list_find (tv->thumblist, thumb);
   if (reverse)
      node = g_list_previous (node);
   else
      node = g_list_next (node);

   while (node) {
      thumb_tmp = node->data;
      if (thumb_tmp->selected) {
	 if (clear)
	    thumbview_set_selection_all (tv, FALSE);
	 while (TRUE) {
	    thumb_tmp = node->data;
	    thumbview_set_selection (thumb_tmp, TRUE);
	    if (node == current_node) break;
	    if (reverse)
	       node = g_list_next (node);
	    else
	       node = g_list_previous (node);
	 }
	 retval = TRUE;
	 break;
      }
      if (reverse)
	 node = g_list_previous (node);
      else
	 node = g_list_next (node);
   }

   return retval;
}


/*
 *  thumbview_initialize:
 *     @ initialize specified thumbnail view widget. 
 *
 *  tv        : Pointer to the ThumbView struct.
 *  dest_mode : Thumbnail display mode to initialize.
 */
static GtkWidget *
thumbview_initialize (ThumbView *tv, gint dest_mode)
{
   GtkWidget *widget;

   g_return_val_if_fail (tv && check_mode_num (dest_mode), FALSE);
   g_return_val_if_fail (thumb_view_modes[dest_mode].create_func, FALSE);

   widget = thumb_view_modes[dest_mode].create_func (tv, dest_mode);

   return widget;
}


/*
 *  thumbview_create
 *     @ Create a thumbnail view. 
 *
 *  files     : Pointer to OpenFiles struct that store some infomation of
 *              opening imagee files.
 *  tw        : Pointer to the ThumbWindow struct.
 *  container : Pointer to GtkWidget to store thumbnail table.
 *  mode      : Thumbnail View mode (DIR or COLLECTION).
 *  Return    : Pointer to the ThumbView struct.
 */
ThumbView *
thumbview_create (OpenFiles *files, ThumbWindow *tw,
		  GtkWidget *container, ThumbViewMode mode)
{
   ThumbView *tv = NULL;
   GtkWidget *widget;
   gchar *tmpstr;
   gint current_page, this_page;
   gchar buf[BUF_SIZE];

   g_return_val_if_fail (files, NULL);

   if (files->status >= CANCEL)
      return NULL;

   files->status = THUMB_LOADING;

   /* allocate new ThumbView struct and initialize */
   tv = g_new0 (ThumbView, 1);
   tv->progress = files;
   tv->thumb_window = tw;
   tv->container = container;
   tv->filenum = 0;
   tv->filesize = 0;
   tv->mode = mode;
   tv->disp_mode = tw->thumb_disp_mode;
   if (mode == THUMB_MODE_COLLECTION) {
      g_snprintf (buf, BUF_SIZE, _("Collection %d"), collection_page_count++);
      tv->tabtitle = g_strdup (buf);
   } else if (mode == THUMB_MODE_DIR) {
      if (files->dirname[strlen (files->dirname) - 1] != '/')
	 tv->dirname = g_strconcat (files->dirname, "/", NULL);
      else
	 tv->dirname = g_strdup (files->dirname);

      if (conf.thumbwin_tab_fullpath) {
	 tv->tabtitle = fileutil_conv_path_to_short_home (tv->dirname);
      } else {
	 tmpstr = g_dirname (tv->dirname);
	 tv->tabtitle = fileutil_dir_basename (tmpstr);
	 g_free (tmpstr);
      }
   }
   tv->disp_mode_data = g_hash_table_new (g_str_hash, g_str_equal);
   tv->related_image_view = NULL;

   /* get thumbnai size from toolbar */
   tv->ThumbnailSize =
      gtk_spin_button_get_value_as_int (GTK_SPIN_BUTTON(tw->button.size_spin));

   ThumbViewList = g_list_append (ThumbViewList, tv);
   thumbwin_location_entry_set_text (tw, NULL);

   /* set widget sensitive */
   current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (tw->notebook));
   this_page = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook), container);

   if (current_page == this_page) {
      tw->status = THUMB_WIN_STATUS_LOADING;
      /* thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING); */
      /* adjust dirtree */
      if (tw->show_dirview && tv->mode == THUMB_MODE_DIR && tv->dirname)
	 dirview_change_dir (tw->dv, tv->dirname);
   } else {
      tw->status = THUMB_WIN_STATUS_LOADING_BG;
      thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING_BG);
   }

   /* fetch infomation about image files */
   thumbview_add_thumb_data (tv, files->filelist);
   tv->thumblist = g_list_sort (tv->thumblist, comp_func_spel);

   /* load thumbnails */
   widget = thumbview_initialize (tv, tv->disp_mode);
   if (GTK_WIDGET_CLASS (GTK_OBJECT (widget)->klass)->set_scroll_adjustments_signal) {
      gtk_container_add (GTK_CONTAINER (tv->container), widget);
   } else {
      gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (tv->container),
					      widget);     
      g_print ("done\n");
   }
   thumbview_redraw (tv, tv->disp_mode, tv->container);
   thumbview_load_thumbnails (tv, tv->thumblist, tv->disp_mode);

   if (files->status > 0 && files->status < CANCEL) {
      files->status = THUMB_LOAD_DONE;
   }

   if (files->status > 0) {
      tv->progress = NULL;
      /* reset status bar, widget sensitive, tab label */
      tw->status = THUMB_WIN_STATUS_NORMAL;
      /* thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_NORMAL); */
      thumbwin_set_tab_label_text (tv->container, tv->tabtitle);
      thumbwin_set_statusbar_page_info (tw, THUMB_CURRENT_PAGE);
   }

   return tv;
}
