/*
 * GImageView
 * Copyright (C) 2001 Takuro Ashie
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "gimageview.h"

#include "auto-completion.h"
#include "dnd.h"
#include "gedo-hpaned.h"
#include "gedo-vpaned.h"
#include "fileutil.h"
#include "help.h"
#include "icon.h"
#include "image_window.h"
#include "dirview.h"
#include "menu.h"
#include "prefs.h"
#include "prefs_win.h"
#include "thumbnail_view.h"
#include "thumbnail_window.h"


typedef enum
{
   OPEN_FILE,
   OPEN_THUMBWIN,
   OPEN_THUMBTAB,
   OPEN_COLLECTION,
   OPEN_PREFS,
   OPEN_GIMV_INFO
} OpenItem;


typedef enum
{
   CLOSE_FILE,
   CLOSE_THUMBWIN,
   CLOSE_THUMBTAB,
   CLOSE_PREFS,
   CLOSE_GIMV_INFO,
   CLOSE_ALL
} CloseItem;


typedef enum
{
   CURRENT,
   NEXT,
   PREV,
   FIRST,
   LAST
} SwitchPage;


typedef enum
{
   MENUBAR,
   TOOLBAR,
   TAB,
   STATUSBAR,
   DIR_TOOLBAR
} ShowItem;


typedef enum
{
   LEFT,
   RIGHT
} MoveTabItem;


/* Parts of Thumbnail Window and utility */
static void       create_thumbwin_menus              (ThumbWindow *tw);
static void       thumbwin_sync_widget_state_to_menu (ThumbWindow *tw);
static GtkWidget *create_toolbar                     (ThumbWindow *tw,
						      GtkWidget   *container);
static void       thumbwin_get_layout                (ThumbwinComposeType *compose,
						      gint         layout);
static void       thumbwin_pane_set_visible          (ThumbWindow *tw,
						      ThumbwinWidgetType item);
static GtkWidget *thumbnail_window_contents_new      (ThumbWindow *tw);
static GtkWidget *thumbnail_view_new                 (ThumbWindow *tw);
static GtkWidget *image_preview_new                  (ThumbWindow *tw);


/* Callback Functions for main menu */
static void cb_open               (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_close              (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_select_all         (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_refresh_list       (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_reload             (ThumbWindow   *tw,
				   ThumbLoadType  type,
				   GtkWidget     *widget);
static void cb_move_tab           (ThumbWindow   *tw,
				   SortItem       sortitem,
				   GtkWidget     *widget);
static void cb_cut_out_tab        (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_switch_layout      (ThumbWindow   *tw,
				   gint          action,
				   GtkWidget     *widget);
static void cb_switch_tab_pos     (ThumbWindow   *tw,
				   GtkPositionType pos,
				   GtkWidget     *widget);
static void cb_slideshow          (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_switch_page        (ThumbWindow   *tw,
				   SwitchPage     action,
				   GtkWidget     *widget);
static void cb_toggle_view        (ThumbWindow   *tw,
				   ThumbwinWidgetType item,
				   GtkWidget     *widget);
static void cb_toggle_show        (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_toggle_fullscreen  (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);
static void cb_sort_item          (ThumbWindow   *tw,
				   SortItem       sortitem,
				   GtkWidget     *widget);
static void cb_reverse_order      (ThumbWindow   *tw,
				   guint          action,
				   GtkWidget     *widget);

/* callback function for toolbar buttons */
static gint cb_location_entry_key_press (GtkWidget   *widget, 
					 GdkEventKey *event,
					 ThumbWindow *tw);
static void cb_open_button              (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_prefs_button             (GtkWidget   *widget);
static void cb_previous_button          (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_next_button              (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_refresh_button           (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_skip_button              (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_stop_button              (GtkWidget   *widget,
					 ThumbWindow *tw);
static gint cb_size_spinner_key_press   (GtkWidget *widget, 
					 GdkEventKey *event,
					 ThumbWindow *tw);
static void cb_quit_button              (GtkWidget   *widget,
					 ThumbWindow *tw);
static void cb_display_mode_menu        (GtkWidget   *widget,
					 ThumbWindow *tw);

/* other callback functions */
static int  cb_thumbwin_delete            (GtkWidget       *widget,
					   GdkEvent        *event,
					   ThumbWindow     *tw);
static void cb_thumbwin_destroy           (GtkWidget       *widget,
					   ThumbWindow     *tw);
static void cb_thumb_notebook_switch_page (GtkNotebook     *notebook,
					   GtkNotebookPage *page,
					   gint             pagenum,
					   ThumbWindow     *tw);
static void cb_resize_window              ();
static void cb_tab_close_button_clicked   (GtkWidget       *button,
					   ThumbWindow     *tw);
static void cb_notebook_drag_begin        (GtkWidget *widget,
					   GdkDragContext *context,
					   gpointer data);
static void cb_notebook_drag_data_get     (GtkWidget *widget,
					   GdkDragContext *context,
					   GtkSelectionData *seldata,
					   guint info,
					   guint time,
					   gpointer data);
static void cb_notebook_drag_data_received(GtkWidget *widget,
					   GdkDragContext *context,
					   gint x, gint y,
					   GtkSelectionData *seldata,
					   guint info,
					   guint time,
					   gpointer data);
static void cb_tab_drag_data_received     (GtkWidget *widget,
					   GdkDragContext *context,
					   gint x, gint y,
					   GtkSelectionData *seldata,
					   guint info,
					   guint time,
					   gpointer data);


/******************************************************************************
 *
 *   Parts of thumbnail window.
 *
 ******************************************************************************/
/* for main manu */
GtkItemFactoryEntry thumbwin_menu_items[] =
{
   {N_("/_File"),               NULL, NULL,              0,              "<Branch>"},
   {N_("/_File/_Open"),         NULL, cb_open,           OPEN_FILE,      NULL},
   {N_("/_File/New Window"),    NULL, cb_open,           OPEN_THUMBWIN,  NULL},
   {N_("/_File/New Tab"),       NULL, cb_open,           OPEN_THUMBTAB,  NULL},
   {N_("/_File/New Collection"),NULL, cb_open,           OPEN_COLLECTION,NULL},
   {N_("/_File/---"),           NULL, NULL,              0,              "<Separator>"},
   {N_("/_File/Close Tab"),     NULL, cb_close,          CLOSE_THUMBTAB, NULL},
   {N_("/_File/Close Window"),  NULL, cb_close,          CLOSE_THUMBWIN, NULL},
   {N_("/_File/Quit"),          NULL, cb_close,          CLOSE_ALL,      NULL},

   {N_("/_Edit"),                      NULL, NULL,            0,            "<Branch>"},
   {N_("/_Edit/Select All"),           NULL, cb_select_all,   TRUE,         NULL},
   {N_("/_Edit/Unselect All"),         NULL, cb_select_all,   FALSE,        NULL},
   {N_("/_Edit/---"),                  NULL, NULL,            0,            "<Separator>"},
   {N_("/_Edit/Refresh List"),         NULL, cb_refresh_list, 0,            NULL},
   {N_("/_Edit/Reload Cache"),         NULL, cb_reload,       LOAD_CACHE,   NULL},
   {N_("/_Edit/Update All Thumbnail"), NULL, cb_reload,       CREATE_THUMB, NULL},
   {N_("/_Edit/---"),                  NULL, NULL,            0,            "<Separator>"},
   {N_("/_Edit/Options"),              NULL, cb_open,         OPEN_PREFS,   NULL},

   {N_("/_View"), NULL, NULL, 0, "<Branch>"},

   {N_("/_Tab"),  NULL, NULL, 0, "<Branch>"},
   {N_("/_Tab/Tab Position"),        NULL, NULL,          0,           "<Branch>"},  
   {N_("/_Tab/Tab Position/Top"),    NULL, cb_switch_tab_pos, GTK_POS_TOP,    "<RadioItem>"},  
   {N_("/_Tab/Tab Position/Bottom"), NULL, cb_switch_tab_pos, GTK_POS_BOTTOM, "/Tab/Tab Position/Top"},  
   {N_("/_Tab/Tab Position/Left"),   NULL, cb_switch_tab_pos, GTK_POS_LEFT,   "/Tab/Tab Position/Bottom"},  
   {N_("/_Tab/Tab Position/Right"),  NULL, cb_switch_tab_pos, GTK_POS_RIGHT,  "/Tab/Tab Position/Left"},  
   {N_("/_Tab/---"),                 NULL, NULL,            0,           "<Separator>"},
   {N_("/_Tab/First Page"),          NULL, cb_switch_page,  FIRST,       NULL},
   {N_("/_Tab/Last Page"),           NULL, cb_switch_page,  LAST,        NULL},
   {N_("/_Tab/Next Page"),           NULL, cb_switch_page,  NEXT,        NULL},
   {N_("/_Tab/Previous Page"),       NULL, cb_switch_page,  PREV,        NULL},
   {N_("/_Tab/---"),                 NULL, NULL,            0,           "<Separator>"},
   {N_("/_Tab/Move tab forward"),    NULL, cb_move_tab,     LEFT,        NULL},
   {N_("/_Tab/Move tab backward"),   NULL, cb_move_tab,     RIGHT,       NULL},
   {N_("/_Tab/Detach tab"),          NULL, cb_cut_out_tab,  0,           NULL},

   {N_("/_Help"),        NULL, NULL,    0,              "<LastBranch>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "View" sub menu */
GtkItemFactoryEntry thumbwin_view_items [] =
{
   {N_("/Sort File List"),  NULL, NULL,              0,           "<Branch>"},
   {N_("/Layout"),          NULL, NULL,              0,           "<Branch>"},  
   {N_("/Layout/Layout0"),  NULL, cb_switch_layout,  0,           "<RadioItem>"},  
   {N_("/Layout/Layout1"),  NULL, cb_switch_layout,  1,           "/Layout/Layout0"},  
   {N_("/Layout/Layout2"),  NULL, cb_switch_layout,  2,           "/Layout/Layout1"},  
   {N_("/Layout/Layout3"),  NULL, cb_switch_layout,  3,           "/Layout/Layout2"},  
   {N_("/Layout/Layout4"),  NULL, cb_switch_layout,  4,           "/Layout/Layout3"},  
   {N_("/Layout/Custom"),   NULL, cb_switch_layout, -1,           "/Layout/Layout4"},  
   {N_("/Slideshow"),       NULL, cb_slideshow,      0,           NULL},
   {N_("/---"),             NULL, NULL,              0,           "<Separator>"},
   {N_("/Directory View"),  NULL, cb_toggle_view,    WIDGET_DIRVIEW, "<ToggleItem>"},
   {N_("/Preview"),         NULL, cb_toggle_view,    WIDGET_PREVIEW, "<ToggleItem>"},
   {N_("/Menu Bar"),        NULL, cb_toggle_show,    MENUBAR,     "<ToggleItem>"},
   {N_("/Tool Bar"),        NULL, cb_toggle_show,    TOOLBAR,     "<CheckItem>"},
   {N_("/DirView Tool Bar"),NULL, cb_toggle_show,    DIR_TOOLBAR, "<CheckItem>"},
   {N_("/Tab"),             NULL, cb_toggle_show,    TAB,         "<CheckItem>"},
   {N_("/Status Bar"),      NULL, cb_toggle_show,    STATUSBAR,   "<CheckItem>"},
   {N_("/---"),             NULL, NULL,              0,           "<Separator>"},
   {N_("/Full Screen"),     NULL, cb_toggle_fullscreen,    0,     "<ToggleItem>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for "Sort File List" sub menu */
GtkItemFactoryEntry thumbwin_sort_items [] =
{
   {N_("/by Name"),              NULL, cb_sort_item,      NAME,   "<RadioItem>"},
   {N_("/by Access Time"),       NULL, cb_sort_item,      ATIME,  "/by Name"},
   {N_("/by Modification Time"), NULL, cb_sort_item,      MTIME,  "/by Access Time"},
   {N_("/by Change Time"),       NULL, cb_sort_item,      CTIME,  "/by Modification Time"},
   {N_("/by Size"),              NULL, cb_sort_item,      SIZE,   "/by Change Time"},
   {N_("/by Type"),              NULL, cb_sort_item,      TYPE,   "/by Size"},
   {N_("/---"),                  NULL, NULL,              0,      "<Separator>"},
   {N_("/Reverse Order"),        NULL, cb_reverse_order,  0,      "<CheckItem>"},
   {NULL, NULL, NULL, 0, NULL},
};


/* for layout */
ThumbwinComposeType compose_type[] =
{
   {TRUE,  FALSE, FALSE, {1, 2, 3}},
   {TRUE,  FALSE, TRUE,  {3, 1, 2}},
   {FALSE, TRUE,  TRUE,  {3, 1, 2}},
   {TRUE,  FALSE, TRUE,  {2, 1, 3}},
   {FALSE, TRUE,  TRUE,  {2, 1, 3}},
};
gint compose_type_num = sizeof (compose_type) / sizeof (compose_type[0]);


/* for drag and drop tab */
static GtkTargetEntry tab_dnd_dest_types[] = {
   {"GIMV_TAB",      0, TARGET_TAB},
   {"text/uri-list", 0, TARGET_URI_LIST},
};
static const gint tab_dnd_dest_types_num
   = sizeof(tab_dnd_dest_types) / sizeof(tab_dnd_dest_types[0]);


static GtkTargetEntry tab_dnd_src_types[] = {
   {"GIMV_TAB",      0, TARGET_TAB},
};
static const gint tab_dnd_src_types_num
   = sizeof(tab_dnd_src_types) / sizeof(tab_dnd_src_types[0]);


GList      *ThumbWinList;
GtkWidget  *newest_tab;
gchar     **disp_mode_labels = NULL;
gint        collection_page_count = 0;

/*
 *  create_thumbwin_menu:
 *     @ create main menu and popup menu.
 *
 *  tw : Pointer to ThumbWindow structure.
 */
static void
create_thumbwin_menus (ThumbWindow *tw)
{
   GtkWidget *item;
   GtkItemFactory *ifactory;
   guint n_menu_items;
   GtkWidget *help;

   /* main menu */
   n_menu_items = sizeof (thumbwin_menu_items)
                     / sizeof (thumbwin_menu_items[0]) - 1;
   tw->menubar = menubar_create (tw->window, thumbwin_menu_items,
				 n_menu_items, "<ThumbVieMain>", tw);

   gtk_container_add(GTK_CONTAINER(tw->menubar_handle), tw->menubar);
   gtk_menu_bar_set_shadow_type(GTK_MENU_BAR(tw->menubar), GTK_SHADOW_NONE);
   gtk_widget_show (tw->menubar);

   /* sub menu */
   n_menu_items = sizeof(thumbwin_view_items)
                     / sizeof(thumbwin_view_items[0]) - 1;
   tw->view_menu = menu_create_items(tw->window, thumbwin_view_items,
				     n_menu_items, "<ViewSubMenu>", tw);
   n_menu_items = sizeof(thumbwin_sort_items)
                     / sizeof(thumbwin_sort_items[0]) - 1;
   tw->sort_menu = menu_create_items(tw->window, thumbwin_sort_items,
				     n_menu_items, "<SortSubMenu>", tw);
   help = gimvhelp_create_menu (tw->window);

   /* attach sub menus to parent menu */
   menu_set_submenu (tw->menubar,   "/View", tw->view_menu);
   menu_set_submenu (tw->menubar,   "/Help", help);
   menu_set_submenu (tw->view_menu, "/Sort File List", tw->sort_menu);

   ifactory = gtk_item_factory_from_widget (tw->menubar);
   tw->menuitem.file = gtk_item_factory_get_item (ifactory, "/File");
   tw->menuitem.edit = gtk_item_factory_get_item (ifactory, "/Edit");
   tw->menuitem.view = gtk_item_factory_get_item (ifactory, "/View");
   tw->menuitem.tab_top    = gtk_item_factory_get_item (ifactory, "/Tab/Tab Position/Top");
   tw->menuitem.tab_bottom = gtk_item_factory_get_item (ifactory, "/Tab/Tab Position/Bottom");
   tw->menuitem.tab_left   = gtk_item_factory_get_item (ifactory, "/Tab/Tab Position/Left");
   tw->menuitem.tab_right  = gtk_item_factory_get_item (ifactory, "/Tab/Tab Position/Right");

   ifactory = gtk_item_factory_from_widget (tw->view_menu);
   tw->menuitem.layout[0] = gtk_item_factory_get_item (ifactory, "/Layout/Layout0");
   tw->menuitem.layout[1] = gtk_item_factory_get_item (ifactory, "/Layout/Layout1");
   tw->menuitem.layout[2] = gtk_item_factory_get_item (ifactory, "/Layout/Layout2");
   tw->menuitem.layout[3] = gtk_item_factory_get_item (ifactory, "/Layout/Layout3");
   tw->menuitem.layout[4] = gtk_item_factory_get_item (ifactory, "/Layout/Layout4");
   tw->menuitem.dirview   = gtk_item_factory_get_item (ifactory, "/Directory View");
   tw->menuitem.preview   = gtk_item_factory_get_item (ifactory, "/Preview");
   tw->menuitem.menubar   = gtk_item_factory_get_item (ifactory, "/Menu Bar");
   tw->menuitem.toolbar   = gtk_item_factory_get_item (ifactory, "/Tool Bar");
   tw->menuitem.dir_toolbar = gtk_item_factory_get_item (ifactory, "/DirView Tool Bar");
   tw->menuitem.statusbar = gtk_item_factory_get_item (ifactory, "/Status Bar");
   tw->menuitem.tab       = gtk_item_factory_get_item (ifactory, "/Tab");
   tw->menuitem.fullscr   = gtk_item_factory_get_item (ifactory, "/Full Screen");

   ifactory = gtk_item_factory_from_widget (tw->sort_menu);
   tw->menuitem.sort_name   = gtk_item_factory_get_item (ifactory, "/by Name");
   tw->menuitem.sort_access = gtk_item_factory_get_item (ifactory, "/by Access Time");
   tw->menuitem.sort_time   = gtk_item_factory_get_item (ifactory, "/by Modification Time");
   tw->menuitem.sort_change = gtk_item_factory_get_item (ifactory, "/by Change Time");
   tw->menuitem.sort_size   = gtk_item_factory_get_item (ifactory, "/by Size");
   tw->menuitem.sort_type   = gtk_item_factory_get_item (ifactory, "/by Type");
   tw->menuitem.sort_rev    = gtk_item_factory_get_item (ifactory, "/Reverse Order");

   /* initialize check menu items */
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				      (tw->menuitem.layout[tw->layout_type]),
				   TRUE);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.dirview),
				   conf.thumbwin_show_dir_view);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.preview),
				   conf.thumbwin_show_preview);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.menubar),
				   conf.thumbwin_show_menubar);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.toolbar),
				   conf.thumbwin_show_toolbar);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.dir_toolbar),
				   conf.dirview_show_toolbar);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.statusbar),
				   conf.thumbwin_show_statusbar);
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (tw->menuitem.tab),
				   conf.thumbwin_show_tab);

   switch (tw->tab_pos) {
   case GTK_POS_TOP:
      item = tw->menuitem.tab_top;
      break;
   case GTK_POS_BOTTOM:
      item = tw->menuitem.tab_bottom;
      break;
   case GTK_POS_LEFT:
      item = tw->menuitem.tab_left;
      break;
   case GTK_POS_RIGHT:
      item = tw->menuitem.tab_right;
      break;
   default:
      item = tw->menuitem.tab_top;
      break;
   }
   gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (item), TRUE);
}


static void
thumbwin_sync_widget_state_to_menu (ThumbWindow *tw)
{
   thumbwin_pane_set_visible (tw, WIDGET_DIRVIEW);
   thumbwin_pane_set_visible (tw, WIDGET_PREVIEW);

   if (!GTK_CHECK_MENU_ITEM(tw->menuitem.menubar)->active)
      gtk_widget_hide (tw->menubar_handle);
   if (!GTK_CHECK_MENU_ITEM(tw->menuitem.toolbar)->active)
      gtk_widget_hide (tw->toolbar_handle);
   if (!GTK_CHECK_MENU_ITEM(tw->menuitem.statusbar)->active)
      gtk_widget_hide (tw->status_bar_container);
   gtk_notebook_set_show_tabs (GTK_NOTEBOOK(tw->notebook),
			       GTK_CHECK_MENU_ITEM(tw->menuitem.tab)->active);
}


/*
 *  create_toolbar:
 *     @ create toolbar and buttons.
 *
 *  tw        : Pointer to ThumbWindow structure.
 *  container : Widget to store toolbar.
 *  Return    : Pointer to toolbar widget.
 */
static GtkWidget *
create_toolbar (ThumbWindow *tw, GtkWidget *container)
{
   GtkWidget *toolbar;
   GtkWidget *button;
   GtkWidget *iconw;   
   GtkAdjustment *adj;
   GtkWidget *spinner;

   toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_BOTH);
   gtk_toolbar_set_button_relief(GTK_TOOLBAR(toolbar), GTK_RELIEF_NONE);
   gtk_toolbar_set_space_style (GTK_TOOLBAR(toolbar), GTK_TOOLBAR_SPACE_LINE);
   gtk_toolbar_set_space_size  (GTK_TOOLBAR(toolbar), 16);

   /* file open button */
   iconw = icon_get_widget ("nfolder");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Open"),
				    _("File Open"),
				    _("File Open"),
				    iconw, cb_open_button, tw);
   tw->button.fileopen = button;

   /* preference button */
   iconw = icon_get_widget ("prefs");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Prefs"),
				    _("Preference"), _("Preference"),
				    iconw, cb_prefs_button, NULL);
   tw->button.prefs = button;

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* previous button */
   iconw = icon_get_widget ("leftarrow");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Prev"),
				    _("Go to previous page"), 
				    _("Go to previous page"),
				    iconw, cb_previous_button, tw);
   tw->button.prev = button;

   /* next button */
   iconw = icon_get_widget ("rightarrow");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Next"),
				    _("Go to next page"), 
				    _("Go to next page"),
				    iconw, cb_next_button, tw);
   tw->button.next = button;

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* refresh button */
   iconw = icon_get_widget ("refresh");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Reload"),
				    _("Reload Cache"), 
				    _("Reload Cache"),
				    iconw, cb_refresh_button, tw);
   tw->button.refresh = button;

   /* skip button */
   iconw = icon_get_widget ("skip");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Skip"),
				    _("Skip creating current thumbnail table"), 
				    _("Skip creating current thumbnail table"),
				    iconw, cb_skip_button, tw);
   gtk_widget_set_sensitive (button, FALSE);
   tw->button.skip = button;

   /* stop button */
   iconw = icon_get_widget ("stop");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Stop"),
				    _("Stop creating thumbnails"), 
				    _("Stop creating thumbnails"),
				    iconw, cb_stop_button, tw);
   gtk_widget_set_sensitive (button, FALSE);
   tw->button.stop = button;

   /* thumbnail size spinner */
   adj = (GtkAdjustment *) gtk_adjustment_new (conf.thumbwin_thumb_size,
					       MIN_THUMB_SIZE, MAX_THUMB_SIZE,
					       1.0, 5.0, 0.0);
   spinner = gtk_spin_button_new (adj, 0, 0);
   tw->button.size_spin = spinner;
   gtk_spin_button_set_wrap (GTK_SPIN_BUTTON (spinner), TRUE);
   gtk_spin_button_set_shadow_type (GTK_SPIN_BUTTON (spinner),
				    GTK_SHADOW_NONE);
   gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), spinner,
			      _("Thumbnail Size"), _("Thumbnail Size"));
   gtk_signal_connect (GTK_OBJECT(spinner), "key-press-event",
		       GTK_SIGNAL_FUNC(cb_size_spinner_key_press), tw);
   gtk_widget_show (spinner);

   gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));

   /* quit button */
   iconw = icon_get_widget ("close");
   button = gtk_toolbar_append_item(GTK_TOOLBAR (toolbar), _("Quit"),
				    _("Exit this program"),
				    _("Exit this program"),
				    iconw, cb_quit_button, NULL);
   tw->button.quit = button;
   gtk_widget_hide (tw->button.quit);

   return toolbar;
}


static void
thumbwin_get_layout (ThumbwinComposeType *compose, gint layout)
{
   gint i;

   if (layout >= compose_type_num)
      layout = 0;

   if (layout > -1) {
      memcpy (compose, &compose_type[layout], sizeof (ThumbwinComposeType));
   } else {
      compose->pane1_horizontal = conf.thumbwin_pane1_horizontal;
      compose->pane2_horizontal = conf.thumbwin_pane2_horizontal;
      compose->pane2_attach_to_child1 = conf.thumbwin_pane2_attach_1;
      for (i = 0; i < 3; i++)
	 compose->widget_type[i] = conf.thumbwin_widget[i];
   }
}


static void
thumbwin_pane_set_visible (ThumbWindow *tw, ThumbwinWidgetType item)
{
   ThumbwinComposeType compose;
   ThumbwinWidgetType type;
   GtkWidget *gpane;
   gint i, val = -1, *size, pane, child;
   gboolean show, *current_state;

   g_return_if_fail (tw);

   if (tw->layout_type >= compose_type_num) return;
   thumbwin_get_layout (&compose, tw->layout_type);

   if (item == WIDGET_DIRVIEW) {
      type = WIDGET_DIRVIEW;
      current_state = &tw->show_dirview;
      show = GTK_CHECK_MENU_ITEM(tw->menuitem.dirview)->active;
   } else if (item == WIDGET_PREVIEW) {
      type = WIDGET_PREVIEW;
      current_state = &tw->show_preview;
      show = GTK_CHECK_MENU_ITEM(tw->menuitem.preview)->active;
   } else {
      return;
   }

   /* find attached pane and child */
   for (i = 0; i < 3; i++) {
      if (compose.widget_type[i] == type)
	 val = i;
   }
   if (val < 0) return;

   if (val == 0) {
      pane = 1;
      if (compose.pane2_attach_to_child1)
	 child = 2;
      else
	 child = 1;
   } else if (val == 1) {
      pane = 2;
      child = 1;
   } else if (val == 2) {
      pane = 2;
      child = 2;
   } else {
      goto FUNC_END;
   }

   if (pane == 1) {
      gpane = tw->pane1;
      size = &tw->pane_size1;
   } else {
      gint pane1_hide_item = GEDO_PANED(tw->pane1)->child_hidden;
      gint pane2_hide_item = GEDO_PANED(tw->pane2)->child_hidden;

      /*
       *  case that select child of pane1 regardless of selected item.
       *  directory view and preview are attach to same pane,
       *  and one of them is hiden.
       */
      if (compose.widget_type[0] == WIDGET_THUMBVIEW
	  && GEDO_PANED(tw->pane2)->child_hidden != 0
	 && ((pane1_hide_item == 0 && !show) || (pane1_hide_item != 0 && show)))
      {
	 gpane = tw->pane1;
	 size = &tw->pane_size1;
	 if (GEDO_PANED(tw->pane1)->child1 == tw->pane2) {
	    child = 1;
	 } else if (GEDO_PANED(tw->pane1)->child2 == tw->pane2) {
	    child = 2;
	 } else {
	    goto FUNC_END;
	 }

	 if (show && (val == pane2_hide_item)) {
	    gedo_paned_split (GEDO_PANED(tw->pane2));
	    if (pane2_hide_item == 1)
	       gedo_paned_hide_child2 (GEDO_PANED (tw->pane2));
	    else if (pane2_hide_item == 2)
	       gedo_paned_hide_child1 (GEDO_PANED (tw->pane2));
	 }
      } else {
	 gpane = tw->pane2;
	 size = &tw->pane_size2;
      }
   }

   if (show) {
      gedo_paned_split (GEDO_PANED (gpane));
      gedo_paned_set_position (GEDO_PANED (gpane), *size);
   } else {
      *size = gedo_paned_get_position (GEDO_PANED (gpane));
      if (child == 1)
	 gedo_paned_hide_child1 (GEDO_PANED (gpane));
      else
	 gedo_paned_hide_child2 (GEDO_PANED (gpane));
   }

 FUNC_END:
   *current_state = show;
}


/*
 *  thumbnail_window_contents_new:
 *     @ main contents of thumbnail window
 *
 *  tw        : Pointer to ThumbWindow structure.
 *  Return    : Pointer to main contents widget.
 */
static GtkWidget *
thumbnail_window_contents_new (ThumbWindow *tw)
{
   GtkWidget *widget[3];
   ThumbwinComposeType compose;
   gint i;

   if (tw->layout_type >= compose_type_num)
      tw->layout_type = 0;

   thumbwin_get_layout (&compose, tw->layout_type);

   /* create each widget */
   tw->dv = dirview_create (NULL, tw->window, tw);
   tw->dirview = tw->dv->container;
   tw->notebook = thumbnail_view_new (tw);
   tw->preview = image_preview_new (tw);

   for (i = 0; i < 3; i++) {
      switch (compose.widget_type[i]) {
      case WIDGET_DIRVIEW:
	 widget[i] = tw->dirview;
	 break;
      case WIDGET_THUMBVIEW:
	 widget[i] = tw->notebook;
	 break;
      case WIDGET_PREVIEW:
	 widget[i] = tw->preview;
	 break;
      default:
	 break;
      }
   }

   /* compose */
   if (compose.pane2_horizontal)
      tw->pane2 = gedo_hpaned_new ();
   else
      tw->pane2 = gedo_vpaned_new ();
   if (compose.pane1_horizontal)
      tw->pane1 = gedo_hpaned_new ();
   else
      tw->pane1 = gedo_vpaned_new ();

   if (compose.pane2_attach_to_child1) {
      gedo_paned_add1 (GEDO_PANED (tw->pane1), tw->pane2);
      gedo_paned_add2 (GEDO_PANED (tw->pane1), widget[0]);
   } else {
      gedo_paned_add1 (GEDO_PANED (tw->pane1), widget[0]);
      gedo_paned_add2 (GEDO_PANED (tw->pane1), tw->pane2);
   }
   gedo_paned_add1 (GEDO_PANED (tw->pane2), widget[1]);
   gedo_paned_add2 (GEDO_PANED (tw->pane2), widget[2]);

   gedo_paned_set_position (GEDO_PANED (tw->pane1),
			    tw->pane_size1);
   gedo_paned_set_position (GEDO_PANED (tw->pane2),
			    tw->pane_size2);

   /* show widget */
   gtk_widget_show (widget[0]);
   gtk_widget_show (widget[1]);
   gtk_widget_show (widget[2]);
   gtk_widget_show (tw->pane2);

   return tw->pane1;
}


/*
 *  thumbnail_view_new:
 *     @ Top of image area. 
 *
 *  tw     : Pointer to ThumbWindow structure.
 *  Return : Pointer to notebook widget(thumbnail view).
 */
static GtkWidget *
thumbnail_view_new (ThumbWindow *tw)
{
   GtkWidget *notebook;

   notebook = gtk_notebook_new ();

   gtk_notebook_set_tab_pos (GTK_NOTEBOOK (notebook),
			     tw->tab_pos);
   gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
   gtk_notebook_popup_enable (GTK_NOTEBOOK (notebook));
   gtk_notebook_set_tab_hborder (GTK_NOTEBOOK (notebook), 0);
   gtk_notebook_set_tab_vborder (GTK_NOTEBOOK (notebook), 0);
   gtk_signal_connect (GTK_OBJECT(notebook), "switch-page",
		       GTK_SIGNAL_FUNC(cb_thumb_notebook_switch_page), tw);

   gtk_object_set_data (GTK_OBJECT (notebook), "thumbwin", tw);
   dnd_src_set  (notebook, tab_dnd_src_types, tab_dnd_src_types_num);
   gtk_signal_connect (GTK_OBJECT (notebook), "drag_begin",
		       GTK_SIGNAL_FUNC (cb_notebook_drag_begin), tw);
   gtk_signal_connect (GTK_OBJECT (notebook), "drag_data_get",
		       GTK_SIGNAL_FUNC (cb_notebook_drag_data_get), tw);

   dnd_dest_set (notebook, tab_dnd_dest_types, tab_dnd_dest_types_num);
   gtk_signal_connect (GTK_OBJECT (notebook), "drag_data_received",
		       GTK_SIGNAL_FUNC (cb_notebook_drag_data_received), tw);

   return notebook;
}


static void
image_preview_clicked (ImageView *iv, GdkEventButton *event, gpointer data)
{
   g_return_if_fail (iv);
   g_return_if_fail (event);

   if (event->button == 1) {
      if (event->state & GDK_SHIFT_MASK)
         imageview_prev (iv);
      else
         imageview_next (iv);
   } else if (event->button == 2) {
      if (iv->info && iv->info->filename)
	 imagewin_open_window (iv->info->filename);
   }
}


/*
 *  image_preview_new:
 *     @ Bottom of image area. 
 *
 *  Return : Pointer to notebook widget (thumbnail view).
 */
static GtkWidget *
image_preview_new (ThumbWindow *tw)
{
   GtkWidget *notebook;

   notebook = gtk_notebook_new ();
   gtk_notebook_set_scrollable (GTK_NOTEBOOK (notebook), TRUE);
   gtk_notebook_set_show_tabs (GTK_NOTEBOOK(notebook), FALSE);

   tw->iv = imageview_create (NULL);
   gtk_widget_show (tw->iv->event_box);
   gtk_notebook_append_page (GTK_NOTEBOOK(notebook), tw->iv->event_box, NULL);

   imageview_create_popup_menu (tw->window, tw->iv);

   tw->iv->button_clicked_cb_func = image_preview_clicked;
   tw->iv->button_clicked_cb_data = tw;

   return notebook;
}



/******************************************************************************
 *
 *   Callback functions for menubar.
 *
 ******************************************************************************/
/*
 *  cb_open:
 *     @ Callback function for main menu item 
 *       (/File/Open ~, /Edit/Prefs, /Help/About).
 *
 *  tw     : Pointer to the ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_open (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbView *tv;
   GtkWidget *newtab;
   OpenFiles *files;

   switch (action) {
   case OPEN_FILE:
      if (tw->open_dialog)
	 gdk_window_raise (tw->open_dialog->window);
      else
	 tw->open_dialog = (GtkWidget *) create_filebrowser (tw);
      break;
   case OPEN_THUMBWIN:
      thumbwin_open_window();
      break;
   case OPEN_THUMBTAB:
      thumbwin_create_new_tab (tw);
      break;
   case OPEN_COLLECTION:
      files = files_loader_new ();
      tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
      if (!(tw->pagenum) || tv) {
	 newtab = thumbwin_create_new_tab (tw);
      } else {
	 gint pagenum;
	 pagenum = gtk_notebook_get_current_page (GTK_NOTEBOOK(tw->notebook));
	 newtab = gtk_notebook_get_nth_page (GTK_NOTEBOOK(tw->notebook), pagenum);
      }
      tv = thumbview_create (files, tw, newtab, THUMB_MODE_COLLECTION);
      thumbwin_set_statusbar_page_info (tw, THUMB_CURRENT_PAGE);
      files_loader_delete (files);
      break;
   case OPEN_PREFS:
      prefs_open_window();
      break;
   case OPEN_GIMV_INFO:
      gimvhelp_info ();
      break;
   default:
      break;
   }
}


/*
 *  cb_close:
 *     @ Callback function for main menu item (/File/).
 *
 *  tw     : Pointer to the ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_close (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   switch (action) {
   case CLOSE_THUMBWIN:
      if (g_list_length (ThumbWinList) == 1 && conf.thumbwin_save_win_state)
	 thumbwin_store_win_state_to_config (tw);
      gtk_widget_destroy (tw->window);
      break;
   case CLOSE_THUMBTAB:
      thumbwin_close_tab (tw, THUMB_CURRENT_PAGE);
      break;
   case CLOSE_ALL:
      gimageview_quit ();
      break;
   default:
      break;
   }
}


static void
cb_select_all (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbView *tv;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   thumbview_set_selection_all (tv, action);
}


static void
cb_refresh_list (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbView *tv;

   if (!tw) return;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   thumbview_refresh_list (tv);
}

/*
 *  cb_reload:
 *     @ Callback function for main menu item
 *       (/Edit/Reload Cache, /Edit/Recreate Thumbnail).
 *     @ Reload thumnail cache. If cache not exist,
 *       create thumbnail from original images.
 *       If type == CREATE_THUMB, recreate all thumbnail from original image.
 *
 *  tw     : Pointer to the ThumbWindow struct.
 *  type   : reload type.
 *  widget :
 */
static void
cb_reload (ThumbWindow *tw, ThumbLoadType type, GtkWidget *widget)
{
   thumbwin_reload_thumbnail (tw, type);
}


/*
 *  cb_move_tab:
 *     @ Callback function for main menu item (/Edit/Move tab to ~).
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  item   : where to move.
 *  widget :
 */
static void
cb_move_tab (ThumbWindow *tw, MoveTabItem item, GtkWidget *widget)
{
   GtkWidget *page_container;
   gint current_page;

   current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK (tw->notebook));
   if (current_page < 0) return;

   page_container = gtk_notebook_get_nth_page (GTK_NOTEBOOK (tw->notebook),
					       current_page);

   switch (item) {
   case LEFT:
      gtk_notebook_reorder_child (GTK_NOTEBOOK (tw->notebook),
				  page_container, current_page - 1);
      break;
   case RIGHT:
      gtk_notebook_reorder_child (GTK_NOTEBOOK (tw->notebook),
				  page_container, current_page + 1);
      break;
   default:
      break;
   }
}


/*
 *  cb_cut_out_tab:
 *     @ Callback function for main menu item (/Edit/Cut out current page).
 *     @ Move current notebook page to new thumbnail window.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_cut_out_tab (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbWindow *tw_new;
   ThumbView  *tv;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   /* open new window */
   tw_new = thumbwin_open_window ();

   thumbwin_detach_tab (tw_new, tw, tv);
}


/*
 *  cb_sort_item:
 *     @ Callback function for main menu item (/View/Sort File List/).
 *     @ Choose sort item and exec sort.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_sort_item (ThumbWindow *tw, SortItem sortitem, GtkWidget *widget)
{
   ThumbView  *tv;
   GtkWidget *thumbview;

   tw->sortitem = sortitem;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   thumbview = thumbview_redraw (tv, tv->disp_mode, tv->container);
}


/*
 *  cb_reverse_order:
 *     @ Callback function for main menu item (/View/Sort File List/Reverse Order).
 *     @ Reverse sort order.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_reverse_order (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbView *tv;
   GtkWidget *thumbview;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   thumbview = thumbview_redraw (tv, tv->disp_mode, tv->container);
}


/*
 *  cb_swicth_layout:
 *     @ Callback function for main menu item (/View/Layout/).
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action : ignore
 *  widget : pointer to menu item widget
 */
static void
cb_switch_layout (ThumbWindow *tw, gint action, GtkWidget *widget)
{
   ThumbView *tv;
   Thumbnail *thumb = NULL;
   GtkWidget *tab, *new_main_contents;
   GList *pages_list = NULL, *node;
   gint i, pagenum, current_page;
   gfloat progress;

   if (tw->layout_type == action) return;

   thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_NO_SENSITIVE);

   if (action >= compose_type_num)
      tw->layout_type = 0;
   else
      tw->layout_type = action;

   if (tw->show_preview && tw->thumb_pos)
      thumb = tw->thumb_pos->data;

   /* get pages list in this window */
   for (i = 0; i < tw->pagenum; i++) {
      tv = thumbwin_find_thumbtable (tw, i);
      pages_list = g_list_append (pages_list, tv);
   }

   current_page = gtk_notebook_get_current_page (GTK_NOTEBOOK(tw->notebook));
   pagenum = tw->pagenum;

   /* create new main contents */
   if (tw->iv) {
      tw->iv = NULL;
      tw->thumb_pos = NULL;
      tw->preview = NULL;
   }
   if (tw->dv) {
      tw->dv = NULL;
      tw->dirview = NULL;
   }
   new_main_contents = thumbnail_window_contents_new (tw);

   /* move pages */
   if (pages_list) {
      node = g_list_first (pages_list);

      for (i = 0; i < pagenum; i++) {
	 tv = node->data;
	 tab = thumbwin_create_new_tab (tw);
	 if (tv && tab) {
	    thumbview_redraw (tv, tv->disp_mode, tab); /* move to new tab */ 
	    tv->container = tab;
	    thumbwin_set_tab_label_text (tv->container, tv->tabtitle);
	 }
	 node = g_list_next (node);
	 if (pagenum > 4) {
	    progress = (gfloat) i / (gfloat) pagenum;
	    gtk_progress_bar_update (GTK_PROGRESS_BAR(tw->progressbar), progress);
	    while (gtk_events_pending()) gtk_main_iteration();
	 }
      }
      gtk_progress_bar_update (GTK_PROGRESS_BAR(tw->progressbar), 0.0);
   }

   gtk_widget_destroy (tw->main_contents);

   /* attach new main contents */
   gtk_container_add (GTK_CONTAINER (tw->main_vbox), new_main_contents);
   thumbwin_sync_widget_state_to_menu (tw);  /* reset widget state */
   gtk_widget_show (new_main_contents);
   tw->main_contents = new_main_contents;
   tw->pagenum = pagenum;

   gtk_notebook_set_page (GTK_NOTEBOOK (tw->notebook), current_page);
   thumbwin_set_statusbar_page_info (tw, THUMB_CURRENT_PAGE);

   tv = thumbwin_find_thumbtable (tw, current_page);
   if (tv) {
      tw->thumb_disp_mode = tv->disp_mode;
      gtk_option_menu_set_history (GTK_OPTION_MENU (tw->disp_mode_menu),
				   tw->thumb_disp_mode);
   }

   /* reset preview image */
   if (tw->show_preview && thumb) {
      imageview_change_image_file (tw->iv, thumb->info->filename);
      imageview_show_image (tw->iv, tw->iv->rotate);
      tw->thumb_pos = g_list_find (thumb->thumb_view->thumblist, thumb);
   }

   /* adjust dirview */
   while (gtk_events_pending()) gtk_main_iteration();
   if (tv && tw->show_dirview && tv->mode == THUMB_MODE_DIR && tv->dirname)
      dirview_change_dir (tw->dv, tv->dirname);

   thumbview_resize (tv);
   thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_NORMAL);
}


static void
cb_switch_tab_pos (ThumbWindow *tw, GtkPositionType pos, GtkWidget *widget)
{
   g_return_if_fail (tw);

   tw->tab_pos = pos;
   gtk_notebook_set_tab_pos (GTK_NOTEBOOK (tw->notebook),
			     tw->tab_pos);
}


/*
 *  cb_slideshow:
 *     @ Callback function for main menu item (/View/Slideshow).
 *     @ Exec slideshow useing current page's file list.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_slideshow (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   ThumbView *tv;
   Thumbnail *thumb;
   SlideShow *slideshow;
   GList *node;
   gchar *filename;

   slideshow = g_new0 (SlideShow, 1);

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   node = tv->thumblist;
   while (node) {
      thumb = node->data;
      filename = g_strdup (thumb->info->filename);
      slideshow->filelist = g_list_append (slideshow->filelist, filename);
     node = g_list_next (node);
   }

   imagewin_exec_slideshow (slideshow);
}


/*
 *  cb_switch_page:
 *     @ Callback function for main menu item (/View/Next Page, Previous Page).
 *     @ Go to next, previous page.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action :
 *  widget :
 */
static void
cb_switch_page (ThumbWindow *tw, SwitchPage action, GtkWidget *widget)
{
   switch (action) {
      case FIRST:
	 gtk_notebook_set_page (GTK_NOTEBOOK (tw->notebook), 0);
	 break;
      case LAST:
	 gtk_notebook_set_page (GTK_NOTEBOOK (tw->notebook), -1);
	 break;
      case NEXT:
	 gtk_notebook_next_page (GTK_NOTEBOOK (tw->notebook));
	 break;
      case PREV:
	 gtk_notebook_prev_page (GTK_NOTEBOOK (tw->notebook));
	 break;
      default:
	 break;
   }
}


/*
 *  cb_toggle_view:
 *     @ Callback function for main menu item (/View/Directory View).
 *     @ Show/hide directory view widget.
 *     @ Move all pages from old notebook to new one.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  action : ignore
 *  widget : pointer to menu item widget
 */
static void
cb_toggle_view (ThumbWindow *tw, ThumbwinWidgetType item, GtkWidget *widget)
{
   g_return_if_fail (tw && (tw->layout_type < compose_type_num));
   thumbwin_pane_set_visible (tw, item);
}


/*
 *  cb_toggle_show:
 *     @ Callback function for main menu item (/View/...).
 *     @ Show/hide widget.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  sitem  : Specify widget to show or hide.
 *  widget :
 */
static void
cb_toggle_show (ThumbWindow *tw, ShowItem sitem, GtkWidget *widget)
{
   GtkWidget *item = NULL;

   switch (sitem) {
   case MENUBAR:
      item = tw->menubar_handle;
      break;
   case TOOLBAR:
      item = tw->toolbar_handle;
      break;
   case TAB:
      gtk_notebook_set_show_tabs (GTK_NOTEBOOK(tw->notebook), 
				  GTK_CHECK_MENU_ITEM(widget)->active);
      break;
   case STATUSBAR:
      item = tw->status_bar_container;
      break;
   case DIR_TOOLBAR:
      if (!tw->dv) return;
      if (GTK_CHECK_MENU_ITEM(widget)->active) {
	 dirview_show_toolbar (tw->dv);
      } else {
	 dirview_hide_toolbar (tw->dv);
      }
      return;
   default:
      return;
      break;
   }

   if (sitem == TAB)
      return;

   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gtk_widget_show (item);
   } else {
      gtk_widget_hide (item);
   }
}


/*
 *  cb_toggle_fullscreen:
 *     @ Callback function for main menu item (/View/Full Screen).
 *
 *  tw     : Pointer to ThumbWindow structure.
 *  action :
 *  widget :
 */
static void
cb_toggle_fullscreen (ThumbWindow *tw, guint action, GtkWidget *widget)
{
   GdkWindow *gdk_window = tw->window->window;
   gint client_x, client_y, root_x, root_y;
   gint width, height;

   if (GTK_CHECK_MENU_ITEM(widget)->active) {
      gdk_window_get_origin (gdk_window, &root_x, &root_y);
      gdk_window_get_geometry (gdk_window, &client_x, &client_y,
			       &width, &height, NULL);

      gdk_window_move_resize (gdk_window, -client_x, -client_y,
			      gdk_screen_width (), gdk_screen_height ());

      tw->fullscreen = TRUE;
      tw->win_x = root_x - client_x;
      tw->win_y = root_y - client_y;
      tw->win_width  = width;
      tw->win_height = height;
   }
   else {
      gdk_window_move_resize (gdk_window, tw->win_x, tw->win_y,
			      tw->win_width, tw->win_height);
      tw->fullscreen = FALSE;
   }
}



/******************************************************************************
 *
 *  Callback functions for toolbar buttons.
 *
 ******************************************************************************/
static gint
cb_location_entry_key_press (GtkWidget *widget, 
                             GdkEventKey *event,
                             ThumbWindow *tw)
{
   gchar *path, *text;
   gint   n, len;

   g_return_val_if_fail (tw, FALSE);
        
   switch (event->keyval) {
   case GDK_Return:
   case GDK_KP_Enter:
      path = gtk_entry_get_text (GTK_ENTRY (widget));

      if (!isdir (path)) return TRUE;

      len = strlen (path);
      if (path[len - 1] != '/') {
	 text = g_strconcat (path, "/", NULL);
      } else {
	 text = g_strdup (path);
      }
      open_dir_images (text, tw, LOAD_CACHE, conf.scan_dir_recursive);
      g_free (text);
      return TRUE;

   case GDK_Tab:
      path = gtk_entry_get_text (GTK_ENTRY (widget));
      n = auto_compl_get_n_alternatives (path);

      if (n < 1) return TRUE;

      text = auto_compl_get_common_prefix ();

      if (n == 1) {
	 auto_compl_hide_alternatives ();
	 gtk_entry_set_text (GTK_ENTRY (widget), text);
	 if (text[strlen(text) - 1] != '/')
	    gtk_entry_append_text (GTK_ENTRY (widget), "/");
	 dirview_change_dir (tw->dv, text);
      } else {
	 gtk_entry_set_text (GTK_ENTRY (widget), text);
	 auto_compl_show_alternatives (widget);
      }
	 
      if (text)
	 g_free (text);
   return TRUE;

   case GDK_Escape:
      gtk_window_set_focus (GTK_WINDOW (tw->window), NULL);
      return TRUE;

   case GDK_Right:
   case GDK_Left:
   case GDK_Up:
   case GDK_Down:
      return TRUE;
   }

   return TRUE;
}


/*
 *  cb_open_button:
 *     @ Callback function for open button.
 *     @ Open file browser. 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct.
 */
static void
cb_open_button (GtkWidget *widget, ThumbWindow *tw)
{
   if (tw->open_dialog)
      gdk_window_raise (tw->open_dialog->window);
   else
      tw->open_dialog = (GtkWidget *) create_filebrowser (tw);
}


/*
 *  cb_refresh_button:
 *     @ Callback function for refresh button.
 *     @ Reload thumbnail cache. 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct.
 */
static void
cb_refresh_button (GtkWidget *widget, ThumbWindow *tw)
{
   /* ThumbView *tv; */

   if (!tw) return;

   thumbwin_reload_thumbnail (tw, LOAD_CACHE);

   /*
   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   thumbview_refresh_list (tv);
   */
}


/*
 *  cb_previous_button:
 *     @ Callback function for previous button.
 *     @ Switch to previous page. 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct.
 */
static void
cb_previous_button (GtkWidget *widget, ThumbWindow *tw)
{
   gtk_notebook_prev_page (GTK_NOTEBOOK (tw->notebook));
}



/*
 *  cb_next_button:
 *     @ Callback function for next button.
 *     @ Switch to next page. 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct.
 */
static void
cb_next_button (GtkWidget *widget, ThumbWindow *tw)
{
   gtk_notebook_next_page (GTK_NOTEBOOK (tw->notebook));
}


/*
 *  cb_skip_button:
 *     @ Callback function for skip button.
 *     @ Skip creating current thumbnail table. 
 *
 *  widget :
 *  tw     : Pointer to ThumbWindow struct.
 */
static void
cb_skip_button (GtkWidget *widget, ThumbWindow *tw)
{
   ThumbView *tv;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);

   if (tv && tv->progress) {
      tv->progress->status = CANCEL;
   }
}


static gint
cb_size_spinner_key_press (GtkWidget *widget, 
			   GdkEventKey *event,
			   ThumbWindow *tw)
{
   g_return_val_if_fail (tw, FALSE);

   switch (event->keyval) {
   case GDK_Escape:
      gtk_window_set_focus (GTK_WINDOW (tw->window), NULL);
      return TRUE;
   }

   return TRUE;
}


/*
 *  cb_stop_button:
 *     @ Callback function for stop button.
 *     @ Stop loading thumbnails. 
 *
 *  widget :
 *  tw     : Pointer to ThumbWindow struct.
 */
static void
cb_stop_button (GtkWidget *widget, ThumbWindow *tw)
{
   ThumbView *tv;
   gint i;

   for (i = 0; i < tw->pagenum; i++) {
      tv = thumbwin_find_thumbtable (tw, i);

      if (tv && tv->thumb_window == tw && tv->progress) {
	 tv->progress->status = STOP;
      }
   }

   files_loader_stop ();
}


/*
 *  cb_prefs_button:
 *     @ Callback function for prefs button.
 *     @ Open preference button. 
 *
 *  widget :
 */
static void
cb_prefs_button (GtkWidget *widget)
{
   if (prefs_window)
      gdk_window_raise (prefs_window->window);
   else
      prefs_open_window();
}


static void
cb_quit_button (GtkWidget *widget, ThumbWindow *tw)
{
   gimageview_quit ();
}


/*
 *  cb_display_mode_menu:
 *     @ Callback function for display mode option menu.
 *     @ Change list mode <-> thumbnail table mode 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct.
 */
static void
cb_display_mode_menu (GtkWidget *widget, ThumbWindow *tw)
{
   ThumbView *tv;
   GtkWidget *thumbview;
   gint disp_mode;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   disp_mode = GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (widget), "num"));
   tw->thumb_disp_mode = disp_mode;

   thumbview = thumbview_change_mode (tv, disp_mode);
}



/******************************************************************************
 *
 *   other collback functions for thumbnail window.
 *
 ******************************************************************************/
/*
 *  idle_thumbwin_resize:
 *     @ Called from cb_resize_window ().
 *
 *  data   : Pointer to ThumbView struct.
 *  Return : False. (do only once)
 */
static gint
idle_thumbwin_resize (gpointer data)
{
   ThumbView *tv = data;

   if (!tv) return FALSE;

   thumbview_resize (tv);

   return FALSE;
}


#if 0
static gint
idle_thumbwin_delete (gpointer data)
{
   ThumbView *tv = data;

   g_return_val_if_fail (tv, FALSE);

   gtk_widget_destroy (tv->thumb_window->window);

   return FALSE;
}
#endif


/*
 *  cb_thumbwin_delete:
 *     @ Callback function for thumbnail window delete event.
 *     @ do not destroy thumbnail window while loading.
 *
 *  widget :
 *  event  :
 *  tw     : Pointer to Thumbnail struct.
 *  Return :
 */
static gint
cb_thumbwin_delete (GtkWidget *widget, GdkEvent *event, ThumbWindow *tw)
{
   ThumbView *tv;
   GList *node;

   node = g_list_first (ThumbViewList);
   while (node) {
      tv = node->data;
      node = g_list_next (node);

      if (tv->thumb_window == tw) {
	 /* do not destroy this window while loading */
	 if (tv->progress) {
	    return TRUE;
	 }
      }
   }

   if (g_list_length (ThumbWinList) == 1 && conf.thumbwin_save_win_state)
      thumbwin_store_win_state_to_config (tw);

   return FALSE;
}


/*
 *  cb_thumbwin_destroy:
 *     @ Callback function for thumbnail window destroy event.
 *     @ Free memory resource. 
 *
 *  widget :
 *  tw     : Pointer to Thumbnail struct to destory.
 */
static void
cb_thumbwin_destroy (GtkWidget *widget, ThumbWindow *tw)
{
   ThumbView *tv;
   GList *node;

   /* for inhibit Gtk-WARNING */
   tw->location_entry = NULL;
   tw->button.size_spin = NULL;
   tw->disp_mode_menu = NULL;

   node = g_list_first (ThumbViewList);
   while (node) {
      tv = node->data;
      node = g_list_next (node);

      if (tv->thumb_window == tw) {
	 if (tv->progress) {
	    tv->progress->status = WINDOW_DESTROYED;
	 } else {
	    ThumbViewList = g_list_remove (ThumbViewList, tv);
	    thumbview_free (tv);
	 }
      }

   }

   ThumbWinList = g_list_remove (ThumbWinList, tw);
   g_free(tw);

   /* quit when last window */
   if (!g_list_first(ImageWinList) && !g_list_first(ThumbWinList)) {
      gimageview_quit();
   }

#ifdef GIMV_DEBUG
   g_print ("Window destroyed!!\n");
#endif /* GIMV_DEBUG */
}


/*
 *  cb_thumb_notebook_switch_page:
 *     @ Callback function for note book switch page event.
 *
 *  notebook : Pointer to the notebook widget.
 *  page     : Pointer to the notebook page.
 *  pagenum  : page number of new page.
 *  tw       : Pointer to Thumbnail struct.
 */
static void
cb_thumb_notebook_switch_page (GtkNotebook *notebook, GtkNotebookPage *page,
			       gint pagenum, ThumbWindow *tw)
{
   ThumbView *tv = NULL;
   gint disp_mode = DEFAULT_DISP_MODE_NUM;
   GList *node;

   g_return_if_fail (notebook && tw);
   node = g_list_find (ThumbWinList, tw);
   if (!node) return;

   if (files_loader_query_loading ())
      thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING);

   thumbwin_set_statusbar_page_info (tw, pagenum);
   tv = thumbwin_find_thumbtable (tw, pagenum);

   if (tv) {
      thumbview_resize (tv);
      if (tv->dirname)
	 gtk_entry_set_text (GTK_ENTRY (tw->location_entry), tv->dirname);
      else
	 gtk_entry_set_text (GTK_ENTRY (tw->location_entry), g_getenv("HOME"));
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (tw->button.size_spin),
				 tv->ThumbnailSize);
      gtk_option_menu_set_history (GTK_OPTION_MENU (tw->disp_mode_menu),
				   tv->disp_mode);
      disp_mode = tv->disp_mode;

      if (tw->status == THUMB_WIN_STATUS_LOADING
	  || tw->status == THUMB_WIN_STATUS_LOADING_BG
	  || files_loader_query_loading ())
      {
	 if (tv->progress)
	    thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING);
	 else
	    thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING_BG);
      }

      if (tw->show_dirview && tv->mode == THUMB_MODE_DIR && tv->dirname)
	 dirview_change_dir (tw->dv, tv->dirname);

   } else {   /* if empty page */
      if (tw->location_entry)
	 gtk_entry_set_text (GTK_ENTRY (tw->location_entry), g_getenv("HOME"));

      if (tw->button.size_spin)
	 gtk_spin_button_set_value (GTK_SPIN_BUTTON (tw->button.size_spin),
				    conf.thumbwin_thumb_size);
      if (tw->disp_mode_menu) {
	 disp_mode = thumbview_label_to_num (conf.thumbwin_disp_mode);
	 if (disp_mode < 0)
	    disp_mode = DEFAULT_DISP_MODE_NUM;
	 gtk_option_menu_set_history (GTK_OPTION_MENU (tw->disp_mode_menu),
				      disp_mode);
      }

      if (tw->status == THUMB_WIN_STATUS_LOADING
	  || tw->status == THUMB_WIN_STATUS_LOADING_BG)
      {
	 thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING_BG);
      }
   }

   tw->thumb_disp_mode = disp_mode;

   /* for preview */
   tw->thumb_pos = NULL;
}


/*
 *  cb_resize_window:
 *     @ Callback function for thumbnail window configure event.
 *     @ Resize thumbnail table.
 *
 *  widget :
 *  event  : 
 *  tw     : Pointer to ThumbWindow struct.
 */
static void
cb_resize_window (GtkWidget *widget, GdkEventConfigure *event, ThumbWindow *tw)
{
   ThumbView *tv;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   gtk_idle_add (idle_thumbwin_resize, tv);
}


/*
 *  cb_tab_close_button_clicked:
 *     @ Callback function for close tab button clicked.
 *     @ Close tab.
 *
 *  widget :
 *  event  : 
 *  tw     : Pointer to ThumbWindow struct.
 */
static void
cb_tab_close_button_clicked (GtkWidget *button, ThumbWindow *tw)
{
   GtkWidget *container;
   gint num;

   container = gtk_object_get_data (GTK_OBJECT (button), "page-container");
   num = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook), container);

   thumbwin_close_tab (tw, num);
   thumbwin_set_statusbar_page_info (tw, THUMB_CURRENT_PAGE);
}


static void
cb_notebook_drag_begin (GtkWidget *widget,
			GdkDragContext *context,
			gpointer data)
{
   GdkColormap *colormap;
   Icon *icon;

   icon = icon_get ("paper");
   colormap = gdk_colormap_get_system ();
   gtk_drag_set_icon_pixmap (context, colormap,
			     icon->pixmap, icon->mask,
			     0, 0);
}


static void
cb_notebook_drag_data_get (GtkWidget *widget,
			   GdkDragContext *context,
			   GtkSelectionData *seldata,
			   guint info,
			   guint time,
			   gpointer data)
{
   switch (info) {
   case TARGET_TAB:
      gtk_selection_data_set(seldata, seldata->target,
			     8, "dummy", strlen("dummy"));
      break;
   }
}


static void
cb_notebook_drag_data_received (GtkWidget *widget,
				GdkDragContext *context,
				gint x, gint y,
				GtkSelectionData *seldata,
				guint info,
				guint time,
				gpointer data)
{
   ThumbWindow *tw = data;
   ThumbView *tv;
   GtkWidget *src_widget;

   switch (info) {
   case TARGET_TAB:
      src_widget = gtk_drag_get_source_widget (context);
      if (src_widget && GTK_IS_NOTEBOOK (src_widget)) {
	 ThumbWindow *tw_src, *tw_dest = tw;

	 tw_src = gtk_object_get_data (GTK_OBJECT (src_widget), "thumbwin");
	 if (!tw_src) return;

	 /* if deferent window, detach tab */
	 if (tw_src != tw_dest) {
	    tv = thumbwin_find_thumbtable (tw_src, THUMB_CURRENT_PAGE);
	    if (!tv) return;

	    thumbwin_detach_tab (tw_dest, tw_src, tv);
	 }
      }
      break;
   }
}


static void
cb_tab_drag_data_received (GtkWidget *widget,
			   GdkDragContext *context,
			   gint x, gint y,
			   GtkSelectionData *seldata,
			   guint info,
			   guint time,
			   gpointer data)
{
   ThumbWindow *tw = data;
   ThumbView *tv;
   GtkWidget *page_container;
   GtkWidget *src_widget;
   gint pagenum;

   g_return_if_fail (widget && tw);

   page_container = gtk_object_get_data (GTK_OBJECT (widget), "page-container");
   if (!page_container) return;

   switch (info) {
   case TARGET_URI_LIST:
      pagenum = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook),
				       page_container);

      tv = thumbwin_find_thumbtable (tw, pagenum);
      if (!tv) return;

      thumbview_drag_data_received_cb (widget, context, x, y,
				       seldata, info, time, tv);
      break;
   case TARGET_TAB:
      src_widget = gtk_drag_get_source_widget (context);
      if (src_widget && GTK_IS_NOTEBOOK (src_widget)) {
	 GtkNotebook *notebook = GTK_NOTEBOOK (src_widget);
	 GtkWidget *src_page, *newtab;
	 ThumbWindow *tw_src, *tw_dest = tw;
	 gint src_pagenum, dest_pagenum;

	 tw_src = gtk_object_get_data (GTK_OBJECT (src_widget), "thumbwin");
	 if (!tw_src) return;

	 dest_pagenum = gtk_notebook_page_num (GTK_NOTEBOOK (tw_dest->notebook),
					       page_container);

	 /* if same window, reorder page order */
	 if (tw_src == tw_dest) {
	    src_pagenum = gtk_notebook_get_current_page (notebook);
	    if (src_pagenum < 0) return;

	    src_page = gtk_notebook_get_nth_page (notebook, src_pagenum);
	    if (!src_page) return;

	    gtk_notebook_reorder_child (notebook, src_page, dest_pagenum);

	 /* if deferent window, detach tab */
	 } else {
	    tv = thumbwin_find_thumbtable (tw_src, THUMB_CURRENT_PAGE);
	    if (!tv) return;

	    if (dest_pagenum < 0) return;

	    newtab = thumbwin_detach_tab (tw_dest, tw_src, tv);
	    gtk_notebook_reorder_child (GTK_NOTEBOOK (tw_dest->notebook),
					newtab, dest_pagenum);	    
	 }
      }
      break;
   }
}



/******************************************************************************
 *
 *   Utility.
 *
 ******************************************************************************/
/*
 *  thumbwin_find_thumbtable:
 *     @ Find ThumbView struct of specidied page from ThumbViewList linked list.
 *
 *  tw       : Pointer to ThumbWindow struct.
 *  pagenum  : Notebook page number for find.
 *  Return   : Poiter to ThumbView struct.
 */
ThumbView *
thumbwin_find_thumbtable (ThumbWindow *tw, gint pagenum)
{
   ThumbView *tv;
   GtkWidget *tab;
   GList *node;
   gint i, num;

   g_return_val_if_fail (tw, NULL);
   node = g_list_find (ThumbWinList, tw);
   if (!node) return NULL;

   if (pagenum == THUMB_CURRENT_PAGE)
      pagenum = gtk_notebook_get_current_page (GTK_NOTEBOOK(tw->notebook));
   tab = gtk_notebook_get_nth_page (GTK_NOTEBOOK(tw->notebook), pagenum);
   ThumbViewList = g_list_first (ThumbViewList);
   node = ThumbViewList;
   num = g_list_length (ThumbViewList);
   for (i = 0; i < num; i++) {
      tv = node->data;
      if (tv->container == tab) {
	 return tv;
      }
     node = g_list_next (node);
   }
   return NULL;
}


/*
 *  thumbwin_set_statusbar_page_info:
 *     @ Set thumbnail window status bar infomation.
 *
 *  tw       : Pointer to ThumbWindow struct.
 *  pagenum  : Notebook page number for set status bar.
 */
void
thumbwin_set_statusbar_page_info (ThumbWindow *tw, gint pagenum)
{
   ThumbView *tv;
   gint page, tv_filenum, tv_filesize;
   gchar buf[BUF_SIZE];
   gchar *path;
   gfloat progress;
   GList *node;

   g_return_if_fail (tw);
   node = g_list_find (ThumbWinList, tw);
   if (!node) return;

   if (pagenum == THUMB_CURRENT_PAGE)
      page = gtk_notebook_get_current_page (GTK_NOTEBOOK(tw->notebook));
   else
      page = pagenum;

   tv = thumbwin_find_thumbtable (tw, page);

   if (!tv) {
      gtk_progress_set_show_text(GTK_PROGRESS(tw->progressbar), FALSE);
      g_snprintf (buf, BUF_SIZE, _("Empty"));
      tv_filenum = 0;
      tv_filesize = 0;
      progress = 0.0;
   } else if (tv->progress){
      /*
      if (tv->dirname)
	 gtk_entry_set_text (GTK_ENTRY (tw->location_entry), tv->dirname);
      */
      gtk_progress_set_show_text(GTK_PROGRESS(tw->progressbar), TRUE);
      g_snprintf (buf, BUF_SIZE, _("%d/%d files"), tv->progress->pos, tv->progress->num);
      gtk_progress_set_format_string(GTK_PROGRESS(tw->progressbar), buf);
      g_snprintf (buf, BUF_SIZE, "%s", tv->progress->now_file);
      progress = (gfloat) tv->progress->pos / (gfloat) tv->progress->num;
      tv_filenum = tv->filenum;
      tv_filesize = tv->filesize;
   } else {
      /*
      if (tv->dirname)
	 gtk_entry_set_text (GTK_ENTRY (tw->location_entry), tv->dirname);
      */
      gtk_progress_set_show_text(GTK_PROGRESS(tw->progressbar), FALSE);
      tv_filenum = tv->filenum;
      tv_filesize = tv->filesize;
      if (tv->mode == THUMB_MODE_DIR) {
	 path = fileutil_conv_path_to_short_home (tv->dirname);
	 g_snprintf (buf, BUF_SIZE, _("DIR View: %s"), path);
	 g_free (path);
      } else {
	 g_snprintf (buf, BUF_SIZE, _("Collection View"));
      }
      progress = 0.0;
   }

   if ((!tv || !tv->progress) && tw->status == THUMB_WIN_STATUS_LOADING_BG) {
      gtk_progress_set_show_text(GTK_PROGRESS(tw->progressbar), TRUE);
      gtk_progress_set_format_string(GTK_PROGRESS(tw->progressbar),
				     _("Loading in another tab..."));
   }

   /* update status bar */
   gtk_statusbar_pop (GTK_STATUSBAR (tw->status_bar1), 1);
   gtk_statusbar_push (GTK_STATUSBAR (tw->status_bar1), 1, buf);
   g_snprintf (buf, BUF_SIZE, _("%d/%d page  %d/%d files  %d/%d kB"),
	       page + 1, tw->pagenum, tv_filenum, tw->filenum,
	       tv_filesize / 1024, tw->filesize / 1024);
   gtk_statusbar_pop (GTK_STATUSBAR (tw->status_bar2), 1);
   gtk_statusbar_push (GTK_STATUSBAR (tw->status_bar2), 1, buf);
   /* update progress bar */
   gtk_progress_bar_update (GTK_PROGRESS_BAR(tw->progressbar), progress);
}


/*
 *  thumbwin_set_sensitive:
 *     @ Set widget sensitivity for thumbnail window.
 *
 *  tw     : Pointer to ThumbWindow struct.
 *  status : Status of window(Loading thumbnails or not).
 */
#define SET_SENSITIVE_WIDGETS_NUM 18

void
thumbwin_set_sensitive (ThumbWindow *tw, ThumbwinStatus status)
{
   gboolean thumbwin_status [][SET_SENSITIVE_WIDGETS_NUM] =
   {
      /* status: normal */
      {TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,  TRUE,
       TRUE,  TRUE,  TRUE,  FALSE, FALSE, TRUE,  TRUE,  TRUE},
      /* status: loading */
      {TRUE,  FALSE, FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
       FALSE, FALSE, TRUE,  TRUE,  TRUE,  FALSE, FALSE, FALSE},
      /* status: loading on BG */
      {TRUE,  FALSE, FALSE, TRUE,  FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
       FALSE, FALSE, FALSE, TRUE,  TRUE,  FALSE, FALSE, TRUE},
      /* status: take all sensitive */
      {FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE,
       FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE, FALSE},
   };
   GtkWidget *widget[SET_SENSITIVE_WIDGETS_NUM];
   gint i, num;

   num = sizeof (thumbwin_status) / sizeof (thumbwin_status[0]);
   if (status > num - 1 || status < 0)
      status = 0;

   widget[0] = tw->menubar;
   widget[1] = tw->menuitem.file;
   widget[2] = tw->menuitem.edit;
   widget[3] = tw->view_menu;
   widget[4] = tw->menuitem.sort_name;
   widget[5] = tw->menuitem.sort_access;
   widget[6] = tw->menuitem.sort_time;
   widget[7] = tw->menuitem.sort_change;
   widget[8] = tw->menuitem.sort_size;
   widget[9] = tw->menuitem.sort_type;
   widget[10] = tw->menuitem.sort_rev;

   widget[11] = tw->button.fileopen;
   widget[12] = tw->button.refresh;
   widget[13] = tw->button.skip;
   widget[14] = tw->button.stop;
   widget[15] = tw->button.prefs;
   widget[16] = tw->button.quit;
   widget[17] = tw->disp_mode_menu;

   for (i = 0; i < SET_SENSITIVE_WIDGETS_NUM; i++) {
      gtk_widget_set_sensitive (widget[i], thumbwin_status[status][i]);
   }
}

#undef SET_SENSITIVE_WIDGETS_NUM


/*
 *  thumbwin_set_tab_label_text:
 *     @ Set tab label text.
 *     @ Must use this function instead of gtk_noteboo_set_tab_label_text ().
 *
 *  page_container : the page container.
 *  title          : the label text.
 */
void
thumbwin_set_tab_label_text (GtkWidget *page_container, const gchar *title)
{
   GtkWidget *tablabel;

   tablabel = gtk_object_get_data (GTK_OBJECT (page_container), "tab-label");
   if (tablabel)
      gtk_label_set_text (GTK_LABEL (tablabel), title);
   gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (page_container->parent),
				     page_container, title);
}


void
cb_pagecontainer_button_press (GtkWidget *widget,
			       GdkEventButton *event,
			       ThumbWindow *tw)
{
   g_return_if_fail (tw);

   thumbwin_notebook_drag_src_unset (tw);   /* FIXMEEEEEEEE!! */
}


void
cb_pagecontainer_thumb_button_release (GtkWidget *widget,
				       GdkEventButton *event,
				       ThumbWindow *tw)
{
   g_return_if_fail (tw);

   thumbwin_notebook_drag_src_reset (tw);   /* FIXMEEEEEEEEE!!! */
}


/*
 *  thumbwin_create_new_tab:
 *     @ Create new notebook page.
 *
 *  tw     : Pointer to ThumbWindow struct for create new tab.
 *  Return : Pointer to page container widget.
 */
GtkWidget *
thumbwin_create_new_tab (ThumbWindow *tw)
{
   GtkWidget *hbox, *pixmap, *button;
   GtkWidget *tablabel;
   GtkWidget *scrolled_window;
   GtkScrolledWindow *scrollwin;
   gint pagenum;
   gchar buf[BUF_SIZE];

   /* page container */
   scrolled_window = gtk_scrolled_window_new (NULL, NULL);
   scrollwin = GTK_SCROLLED_WINDOW (scrolled_window);
   gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scrolled_window),
				   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
   gtk_widget_show (scrolled_window);
   gtk_signal_connect (GTK_OBJECT(scrolled_window), "button_press_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_button_press), tw);
   gtk_signal_connect (GTK_OBJECT(scrolled_window), "button_release_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_thumb_button_release), tw);
   gtk_signal_connect (GTK_OBJECT(scrollwin->hscrollbar), "button_press_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_button_press), tw);
   gtk_signal_connect (GTK_OBJECT(scrollwin->hscrollbar), "button_release_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_thumb_button_release), tw);
   gtk_signal_connect (GTK_OBJECT(scrollwin->vscrollbar), "button_press_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_button_press), tw);
   gtk_signal_connect (GTK_OBJECT(scrollwin->vscrollbar), "button_release_event",
		       GTK_SIGNAL_FUNC(cb_pagecontainer_thumb_button_release), tw);

   /* page label widget */
   hbox = gtk_hbox_new (FALSE, 0);
   gtk_object_set_data (GTK_OBJECT (hbox), "page-container", scrolled_window);

   tablabel = gtk_label_new (_("New Tab"));
   gtk_misc_set_alignment (GTK_MISC (tablabel), 0.00, 0.5);
   gtk_misc_set_padding (GTK_MISC (tablabel), 4, 0);
   gtk_box_pack_start(GTK_BOX(hbox), tablabel, TRUE, TRUE, 0);

   if (conf.thumbwin_show_tab_close) {
      button = gtk_button_new();
      gtk_widget_set_usize (button, 16, 16);
      gtk_button_set_relief((GtkButton *) button, GTK_RELIEF_NONE);
      pixmap = icon_get_widget ("small_close");
      gtk_object_set_data (GTK_OBJECT (button), "page-container", scrolled_window);
      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (cb_tab_close_button_clicked), tw);

      gtk_container_add(GTK_CONTAINER(button), pixmap);
      gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
   }

   gtk_widget_show_all (hbox);

   dnd_dest_set (hbox, tab_dnd_dest_types, tab_dnd_dest_types_num);
   gtk_signal_connect (GTK_OBJECT (hbox), "drag_data_received",
		       GTK_SIGNAL_FUNC (cb_tab_drag_data_received), tw);

   /* create page */
   gtk_notebook_append_page (GTK_NOTEBOOK(tw->notebook), scrolled_window, hbox);
   gtk_object_set_data (GTK_OBJECT (scrolled_window), "tab-label", tablabel);

   tw->pagenum++;

   /* set label text */
   pagenum = gtk_notebook_page_num (GTK_NOTEBOOK (tw->notebook), scrolled_window);
   g_snprintf (buf, BUF_SIZE, _("NewTab %d"), tw->newpage_count++);
   gtk_label_set_text (GTK_LABEL (tablabel), buf);
   gtk_notebook_set_menu_label_text (GTK_NOTEBOOK (tw->notebook),
				     scrolled_window, buf);

   /* switch to this page */
   if (conf.thumbwin_move_to_newtab || tw->pagenum == 1)
      gtk_notebook_set_page (GTK_NOTEBOOK(tw->notebook), pagenum);

   newest_tab = scrolled_window;
   return scrolled_window;
}


GtkWidget *
thumbwin_detach_tab (ThumbWindow *tw_dest, ThumbWindow *tw_src, ThumbView *tv)
{
   GtkWidget *newtab;
   GtkWidget *thumbview;

   g_return_val_if_fail (tw_dest && tw_src && tv, FALSE);
   g_return_val_if_fail (tw_dest != tw_src, FALSE);
   g_return_val_if_fail (tw_src == tv->thumb_window, FALSE);

   newtab = thumbwin_create_new_tab (tw_dest);
   thumbwin_set_tab_label_text (newtab, tv->tabtitle);

   /* move the table to new window */
   thumbview = thumbview_redraw (tv, tv->disp_mode, newtab);

   /* reset widget */
   gtk_spin_button_set_value (GTK_SPIN_BUTTON (tw_dest->button.size_spin),
			      tv->ThumbnailSize);
   gtk_option_menu_set_history (GTK_OPTION_MENU (tw_dest->disp_mode_menu),
				tv->disp_mode);

   /* update struct data */
   tv->container = newtab;
   tv->thumb_window = tw_dest;
   tw_src->filenum -= tv->filenum;
   tw_src->filesize -= tv->filesize;
   tw_dest->filenum += tv->filenum;
   tw_dest->filesize += tv->filesize;

   thumbwin_close_tab (tw_src, THUMB_CURRENT_PAGE);

   thumbwin_set_statusbar_page_info (tw_dest, THUMB_CURRENT_PAGE);

   /* change current directory of dirview */
   if (tv && tw_dest->show_dirview && tv->mode == THUMB_MODE_DIR && tv->dirname)
      dirview_change_dir (tw_dest->dv, tv->dirname);

   return newtab;
}


/*
 *  thumbwin_close_tab:
 *     @ Close a notebook page
 *
 *  tw   : Pointer to ThumbWindow struct for close tab.
 *  page : Page num to close.
 */
void
thumbwin_close_tab (ThumbWindow *tw, gint page)
{
   gint page_tmp;
   ThumbView *tv;

   tv = thumbwin_find_thumbtable (tw, page);

   if (tv && tv->progress)
      return;

   if (tv)
      thumbview_delete (tv);

   if (page == THUMB_CURRENT_PAGE)
      page_tmp = gtk_notebook_get_current_page (GTK_NOTEBOOK(tw->notebook));
   else
      page_tmp = page;

   gtk_notebook_remove_page (GTK_NOTEBOOK(tw->notebook), page_tmp);

   if (tw->pagenum > 0)
      tw->pagenum--;

   thumbwin_set_statusbar_page_info (tw, page);
}


/*
 *  thumbwin_reload_thumbnail:
 *     @ Reload thumbnai cache or rebuild thumbnail from original image.
 *
 *  tw   : Poitner to the ThumbWindow struct.
 *  type : Type of reload (load cache or rebuild thumbnail).
 */
void
thumbwin_reload_thumbnail (ThumbWindow *tw, ThumbLoadType type)
{
   ThumbView *tv;
   Thumbnail  *thumb;
   OpenFiles  *files;
   gchar *dirname, *filename;
   GList *node;

   tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
   if (!tv) return;

   if (tv->mode == THUMB_MODE_DIR) {
      dirname = strdup (tv->dirname);
      thumbview_delete (tv);
      open_dir_images (dirname, tw, type, conf.scan_dir_recursive);
      g_free (dirname);
   } else if (tv->mode == THUMB_MODE_COLLECTION) {
      files = files_loader_new ();
      node = tv->thumblist;
      while (node) {
	 thumb = (Thumbnail *) node->data;
	 filename = g_strdup ((gchar *) thumb->info->filename);
	 files->filelist = g_list_append (files->filelist, filename);
	 node = g_list_next (node);
      }
      files->status = END;
      files->thumb_load_type = type;

      thumbview_delete (tv);
      open_image_files_in_thumbnail_view (files, tw);
      files_loader_delete (files);
   }

   tw->thumb_pos = NULL;
}


void
thumbwin_location_entry_set_text (ThumbWindow *tw, const gchar *location)
{
   ThumbView *tv;
   gchar *text;

   g_return_if_fail (tw);
   if (!location) {
      tv = thumbwin_find_thumbtable (tw, THUMB_CURRENT_PAGE);
      if (tv) {
	 if (tv->mode == THUMB_MODE_DIR && tv->dirname)
	    text = g_strdup (tv->dirname);
	 else
	    text = g_strdup (g_getenv ("HOME"));
      } else {
	    text = g_strdup (g_getenv ("HOME"));
      }
   } else {
      text = g_strdup (location);
   }

   gtk_entry_set_text (GTK_ENTRY (tw->location_entry), text);

   g_free (text);
}


/*
 *  thumbwin_store_win_state_to_config:
 *     @ Save window state paramator such as window size.
 *
 *  tw : Pointer to ThumbWindow struct.
 */
void
thumbwin_store_win_state_to_config (ThumbWindow *tw)
{
   if (!tw || !tw->window) return;

   if (!tw->fullscreen) {
      conf.thumbwin_width  = tw->window->allocation.width;
      conf.thumbwin_height = tw->window->allocation.height;
   } else {
      conf.thumbwin_width  = tw->win_width;
      conf.thumbwin_height = tw->win_height;
   }

   if (tw->show_preview)
      conf.thumbwin_pane_size2
	 = gedo_paned_get_position (GEDO_PANED (tw->pane2));
   if (tw->show_dirview)
      conf.thumbwin_pane_size1
	 = gedo_paned_get_position (GEDO_PANED (tw->pane1));

   conf.thumbwin_show_preview  = tw->show_preview;
   conf.thumbwin_show_dir_view = tw->show_dirview;

   conf.thumbwin_show_menubar   = GTK_CHECK_MENU_ITEM (tw->menuitem.menubar)->active;
   conf.thumbwin_show_toolbar   = GTK_CHECK_MENU_ITEM (tw->menuitem.toolbar)->active;
   conf.thumbwin_show_statusbar = GTK_CHECK_MENU_ITEM (tw->menuitem.statusbar)->active;
   conf.thumbwin_show_tab       = GTK_CHECK_MENU_ITEM (tw->menuitem.tab)->active;

   conf.thumbwin_layout_type = tw->layout_type;
   conf.thumbwin_tab_pos = tw->tab_pos;
}


/* FIXMEEEEEEEEEEE!!! (TOT */
void
thumbwin_notebook_drag_src_unset (ThumbWindow *tw)
{
   g_return_if_fail (tw);

   gtk_drag_source_unset (tw->notebook);
}


void
thumbwin_notebook_drag_src_reset (ThumbWindow *tw)
{
   g_return_if_fail (tw);

   dnd_src_set  (tw->notebook, tab_dnd_src_types, tab_dnd_src_types_num);
}
/* END FIXMEEEEEEEEEEE!!! (TOT */


/******************************************************************************
 *
 *   thumbwin_open window.
 *      @ Create new thumbnail window
 *
 *   Return : Pointer to ThumbWindow structure.
 *
 ******************************************************************************/
ThumbWindow *
thumbwin_open_window ()
{
   ThumbWindow *tw;

   GtkWidget *vbox, *hbox;
   GtkWidget *separator;
   GtkWidget *entry;
   gint num, disp_mode;
   gboolean loading;

   tw = g_new0 (ThumbWindow, 1);
   tw->filenum = 0;
   tw->pagenum = 0;
   tw->filesize = 0;
   tw->newpage_count = 0;
   tw->open_dialog = NULL;
   tw->pane_size1 = conf.thumbwin_pane_size1;
   tw->pane_size2 = conf.thumbwin_pane_size2;
   tw->show_dirview = conf.thumbwin_show_dir_view;
   tw->show_preview = conf.thumbwin_show_preview;
   tw->layout_type = conf.thumbwin_layout_type;
   tw->tab_pos = conf.thumbwin_tab_pos;
   ThumbWinList = g_list_append (ThumbWinList, tw);

   /* window */
   tw->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

   gtk_window_set_title (GTK_WINDOW (tw->window), PROG_NAME" -Thumbnail Window-");
   gtk_window_set_default_size (GTK_WINDOW(tw->window),
				conf.thumbwin_width, conf.thumbwin_height);
   gtk_window_set_policy(GTK_WINDOW(tw->window), TRUE, TRUE, FALSE);

   gtk_signal_connect (GTK_OBJECT (tw->window), "configure_event",
		       GTK_SIGNAL_FUNC (cb_resize_window), tw);
   gtk_signal_connect (GTK_OBJECT (tw->window), "delete_event",
		       GTK_SIGNAL_FUNC (cb_thumbwin_delete), tw);
   gtk_signal_connect (GTK_OBJECT (tw->window), "destroy",
		       GTK_SIGNAL_FUNC (cb_thumbwin_destroy), tw);

   /* Main vbox */
   vbox = gtk_vbox_new (FALSE, 0);
   tw->main_vbox = vbox;
   gtk_container_add (GTK_CONTAINER (tw->window), vbox);
   gtk_widget_show (vbox);

   /* Menu Bar */
   tw->menubar_handle = gtk_handle_box_new();
   gtk_container_set_border_width (GTK_CONTAINER(tw->menubar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), tw->menubar_handle, FALSE, FALSE, 0);
   gtk_widget_show (tw->menubar_handle);

   /* toolbar */
   tw->toolbar_handle = gtk_handle_box_new();
   gtk_container_set_border_width (GTK_CONTAINER(tw->toolbar_handle), 2);
   gtk_box_pack_start(GTK_BOX(vbox), tw->toolbar_handle, FALSE, FALSE, 0);
   gtk_widget_show (tw->toolbar_handle);

   hbox = gtk_hbox_new (FALSE, 0);
   gtk_container_add(GTK_CONTAINER(tw->toolbar_handle), hbox);
   gtk_widget_show (hbox);

   tw->toolbar = create_toolbar (tw, tw->toolbar_handle);
   gtk_box_pack_start (GTK_BOX (hbox), tw->toolbar, FALSE, FALSE, 0);
   gtk_widget_show (tw->toolbar);
   gtk_toolbar_set_style (GTK_TOOLBAR(tw->toolbar), conf.thumbwin_toolbar_style);

   /* location entry */
   tw->location_entry = entry = gtk_entry_new ();
   gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
   gtk_signal_connect_after (GTK_OBJECT(entry), "key-press-event",
			     GTK_SIGNAL_FUNC(cb_location_entry_key_press), tw);
   gtk_widget_show (entry);

   /* option menu for display mode */
   if (!disp_mode_labels)
      disp_mode_labels = thumbview_get_disp_mode_labels (&num);

   disp_mode = thumbview_label_to_num (conf.thumbwin_disp_mode);
   if (disp_mode < 0)
      disp_mode = DEFAULT_DISP_MODE_NUM;

   tw->disp_mode_menu = create_option_menu ((const gchar **)disp_mode_labels,
					    disp_mode,
					    cb_display_mode_menu, tw);
   tw->thumb_disp_mode = disp_mode;
   gtk_box_pack_end (GTK_BOX (hbox), tw->disp_mode_menu, FALSE, FALSE, 0);
   gtk_widget_show (tw->disp_mode_menu);

   /* separator */
   separator = gtk_hseparator_new ();
   gtk_box_pack_start (GTK_BOX (vbox), separator, FALSE, FALSE, 0);
   gtk_widget_show (separator);

   /* main contents */
   tw->main_contents = thumbnail_window_contents_new (tw);
   gtk_container_add (GTK_CONTAINER (vbox), tw->main_contents);
   gtk_widget_show (tw->main_contents);

   /* status bar */
   hbox = gtk_hbox_new (FALSE, 0);
   tw->status_bar_container = hbox;
   gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);
   gtk_widget_show (hbox);

   tw->status_bar1 = gtk_statusbar_new ();
   gtk_container_border_width (GTK_CONTAINER (tw->status_bar1), 1);
   gtk_widget_set_usize(tw->status_bar1, 150, -1);
   gtk_box_pack_start (GTK_BOX (hbox), tw->status_bar1, TRUE, TRUE, 0);
   gtk_statusbar_push(GTK_STATUSBAR (tw->status_bar1), 1, "New Window");
   gtk_widget_show (tw->status_bar1);

   tw->status_bar2 = gtk_statusbar_new ();
   gtk_container_border_width (GTK_CONTAINER (tw->status_bar2), 1);
   gtk_widget_set_usize(tw->status_bar2, 150, -1);
   gtk_box_pack_start (GTK_BOX (hbox), tw->status_bar2, TRUE, TRUE, 0);
   gtk_widget_show (tw->status_bar2);

   tw->progressbar = gtk_progress_bar_new();
   gtk_box_pack_end (GTK_BOX (hbox), tw->progressbar, FALSE, FALSE, 0);
   gtk_widget_show (tw->progressbar);

   /* create menus */
   create_thumbwin_menus (tw);
   thumbwin_sync_widget_state_to_menu (tw);

   gtk_widget_show (tw->window);

   /* create new tab */
   /* thumbwin_create_new_tab (tw); */

   loading = files_loader_query_loading ();
   if (loading) thumbwin_set_sensitive (tw, THUMB_WIN_STATUS_LOADING);

   return tw;
}
