-------------------------------------------------------------------------------
--
--  Filename        : $Source: /cvsroot/gnade/gnade/tools/odbc.adb,v $
--  Description     : ODBC functions for ODBC explorer                       --
--  Author          : Michael Erdmann                                        --
--  Created         : 8.8.2001                                               --
--  Last Modified By: $Author: merdmann $
--  Last Modified On: $Date: 2003/04/07 19:39:26 $
--  Status          : $State: Exp $
--
--  Copyright (C) 2000 - 2003 Michael Erdmann                                --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  Author: Michael Erdmann <michael.erdmann@snafu.de>                       --
--                                                                           --
--  GNADE is implemented to work with GNAT, the GNU Ada compiler.            --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--                                                                           --
--  In contains the following sections:                                      --
--                                                                           --
--  Section 1. General support procedures                                    --
--  Section 2. Connection Management                                         --
--  Section 3. Query Management                                              --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  None                                                                     --
--                                                                           --
--  References                                                               --
--  ==========                                                               --
--  None                                                                     --
--                                                                           --
-------------------------------------------------------------------------------

--* Ada
with Ada.Strings.Unbounded;    use Ada.Strings.Unbounded;
with Ada.Strings.Fixed;        use Ada.Strings.Fixed;
with Ada.Strings;              use Ada.Strings;

with Ada.Text_IO;              use Ada.Text_IO;
with Ada.Exceptions;           use Ada.Exceptions;

with System.Address_To_Access_Conversions;
with System.Storage_Elements;
with Unchecked_Deallocation;



with GNU.DB.SQLCLI;            use GNU.DB.SQLCLI;
with GNU.DB.SQLCLI.Bind;
with GNU.DB.SQLCLI.Info;       use GNU.DB.SQLCLI.Info;
with GNU.DB.SQLCLI.Info.Debug;

with GNU.DB.SQLCLI.Environment_Attribute;
use  GNU.DB.SQLCLI.Environment_Attribute;
with GNU.DB.SQLCLI.Environment_Attribute.Debug;

with GNU.DB.SQLCLI.Connection_Attribute;
use  GNU.DB.SQLCLI.Connection_Attribute;
with GNU.DB.SQLCLI.Connection_Attribute.Debug;

pragma Warnings (Off);
with GNU.DB.SQLCLI.Desc;
with GNU.DB.SQLCLI.Generic_Attr.Integer_Attribute;
pragma Warnings (On);

use GNU.DB.SQLCLI;

package body ODBC is

   type Object_Data is record
         EnvironmentHandle : SQLHENV;
         ConnectionHandle  : SQLHDBC;
         StatementHandle   : SQLHSTMT;

         ServerName        : Unbounded_String := To_Unbounded_String("");
         UserName          : Unbounded_String := To_Unbounded_String("");
         Authentication    : Unbounded_String := To_Unbounded_String("");

         -- data realted to the last query
         Cols              : Integer := 0;
         Rows              : Integer := 0;

         Field_Name        : String_Array_Access := null;

         Last_Error        : Unbounded_String := Null_Unbounded_String;
         Last_Message      : Unbounded_String := Null_Unbounded_String;
      end record;

   --- ********************************************************************* ---
   --- ** SECTION 1.       S U P P O R T    P R O C E D U R E S          *** ---
   --- **                  ====================================          *** ---
   --- **                                                                *** ---
   --- ********************************************************************* ---

   ----------------
   -- Initialize --
   ----------------
   procedure Initialize(
      This : in out Object ) is
      Data : Object_Data_Access := null;
   begin
      This.Data := new Object_Data;
      Data := This.Data;

      SQLAllocHandle (
         SQL_HANDLE_ENV,
         SQL_NULL_HANDLE,
         Data.EnvironmentHandle
      );

      SQLSetEnvAttr  (
         Data.EnvironmentHandle,
         Environment_Attribute_ODBC_Version'                        --'
                   (Attribute => SQL_ATTR_ODBC_VERSION,
                    Value     => SQL_OV_ODBC3)
      );
   end Initialize;

   --------------
   -- Finalize --
   --------------
   procedure Finalize(
      This : in out Object ) is
      Data : Object_Data_Access := This.Data;
   begin
      SQLFreeHandle (SQL_HANDLE_ENV, Data.EnvironmentHandle);
   end Finalize;

   -------------
   -- Columns --
   -------------
   function Columns(
      This : in Object ) return Integer is
      Data : Object_Data_Access := This.Data;
   begin
      return Data.Cols;
   end Columns;

   -------------------
   -- Affected_Rows --
   -------------------
   function Affected_Rows(
      This : in Object ) return Integer is
      Data : Object_Data_Access := This.Data;
   begin
      return Data.Rows;
   end Affected_Rows;

   ---------------
   -- To_String --
   ---------------
   function To_String(
      Result : in Result_Record;
      Pos    : in Integer ) return String is
   begin
      return To_String( Result(Pos) );
   end To_String;

   -----------------
   -- Data_Source --
   -----------------
   procedure Data_Source(
      This : in out Object;
      Name : in String ) is
      Data : Object_Data_Access := This.Data;
   begin
      Data.ServerName := To_Unbounded_String( Name );
   end Data_Source;

   -------------------
   -- Authorization --
   -------------------
   procedure Authorization(
      This     : in out Object;
      Login    : in String;
      Password : in String ) is
      Data     : Object_Data_Access := This.Data;
   begin
      Data.UserName       := To_Unbounded_String( Login );
      Data.Authentication := To_Unbounded_String( Password );
   end Authorization;

   ---------------
   -- Get_Error --
   ---------------
   procedure Get_Error(
      This     : in out Object;
      Error    : out Unbounded_String;
      Message  : out Unbounded_String ) is
      Data     : Object_Data_Access := This.Data;
   begin
      Error   := Data.Last_Error;
      Message := Data.Last_Message;
   end Get_Error;

   ----------------
   -- Field_Name --
   ----------------
   function Field_Name(
      This : in Object;
      Pos  : in Positive ) return String is
      Data : Object_Data_Access := This.Data;
   begin
      if Pos in Data.Field_Name'Range then
         return To_String(Data.Field_Name( Pos ));
      else
         return "";
      end if;
   end Field_Name;

   ----------
   -- Free --
   ----------
   procedure Free(
      S : in out String_Array_Access ) is

      procedure XFree is
         new Unchecked_Deallocation( String_Array, String_Array_Access );
   begin
      XFree( S ) ;
      S := null;
   end Free;



   --- ********************************************************************* ---
   --- **                                                                *** ---
   --- ** SECTION 2.   C O N N E C T I O N   M A N A G E M E N T         *** ---
   --- **                                                                *** ---
   --- ********************************************************************* ---

   -------------
   -- Connect --
   -------------
   procedure Connect(
      This   : in out Object;
      Source : in String := "" ) is
      -- connect to the data source and set all connection related
      -- attributes.
      Data : Object_Data_Access := This.Data;
   begin
      if Source /= "" then
         Data.ServerName := To_Unbounded_String(Source);
      end if;

      SQLAllocHandle (
         SQL_HANDLE_DBC,
         Data.EnvironmentHandle,
         Data.ConnectionHandle);

      --SQLSetEnvAttr  (
      --   Data.EnvironmentHandle,
      --   Environment_Attribute_ODBC_Version'
      --             (Attribute => SQL_ATTR_ODBC_VERSION,             
      --              Value     => SQL_OV_ODBC3)                         --'
      --);

      SQLConnect (
         ConnectionHandle => Data.ConnectionHandle,
         ServerName       => To_String( Data.ServerName ),
         UserName         => To_String( Data.UserName ),
         Authentication   => To_String( Data.Authentication)
      );
   end Connect;

   ----------------
   -- Disconnect --
   ----------------
   procedure Disconnect(
      This : in out Object ) is
      -- diconnect ffrom the data source
      Data : Object_Data_Access := This.Data;
   begin
      SQLDisconnect (Data.ConnectionHandle);
      SQLFreeHandle (SQL_HANDLE_DBC, Data.ConnectionHandle);
   end Disconnect;

   --- ********************************************************************* ---
   --- **                                                                *** ---
   --- ** SECTION 3.  Q U E R Y   M A N A G E M E N T                    *** ---
   --- **                                                                *** ---
   --- ********************************************************************* ---

   -----------
   -- Query --
   -----------
   procedure Query(
      This         : in out Object;
      -- execute the query and evaluate the number of columns and
      -- rows.
      Query_String  : in String ) is
      Data          : Object_Data_Access := This.Data;

      DataType      : aliased SQL_DATA_TYPE;
      ColumnSize    : aliased SQLUINTEGER;
      DecimalDigits : aliased SQLSMALLINT;
      Nullable      : aliased SQL_NULLABLE_INFO;
      RC            : aliased SQLRETURN;
   begin
      Data.Last_Error   := Null_Unbounded_String;
      Data.Last_Message := Null_Unbounded_String;

      SQLAllocHandle (
         SQL_HANDLE_STMT,
         Data.ConnectionHandle,
         Data.StatementHandle
      );

      SQLExecDirect ( Data.StatementHandle, Query_String );

      Data.Rows := Integer( SQLRowCount(Data.StatementHandle) );
      Data.Cols := Integer( SQLNumResultCols ( Data.StatementHandle ));

--      Put_Line("Data.cols =" & Integer'Image(Data.Cols));
--      Put_Line("Data.rows =" & Integer'Image(Data.Rows));

      Data.Field_Name := new String_Array( 1..Integer(Data.Cols) );

      for Col in 1..Data.Cols loop
         Data.Field_Name( Col ) := To_Unbounded_String(
            SQLDescribeCol(
                     Data.StatementHandle,
                     SQL_Column_Number(Col),
                     SQLSMALLINT( 1024 ),
                     DataType'Access,
                     ColumnSize'Access,
                     DecimalDigits'Access,
                     Nullable'Access,
                     RC'Access )
            );
      end loop;

   exception
      when Error : others =>
         Data.Last_Error   := To_Unbounded_String(Exception_Name (Error));
         Data.Last_Message := To_Unbounded_String(Exception_Message (Error));
         raise;
   end Query;

   -----------
   -- Fetch --
   -----------
   function Fetch(
      This : in Object ) return Result_Record is
      -- fetch the next record from the result set and return a record
      Data          : Object_Data_Access  := This.Data;
      Result        : String_Array_Access := null;

      RC            : SQLRETURN;
      Buffer        : aliased String( 1..512 ) := (others =>' ');
      Buffer_Length : aliased SQLINTEGER;
   begin
      Data.Last_Error   := Null_Unbounded_String;
      Data.Last_Message := Null_Unbounded_String;

      SQLFetch( Data.StatementHandle );

      Result := new String_Array(1..Data.Cols);

      for Col in 1..Data.Cols loop
         RC := SQLGetData (
                  Data.StatementHandle,
                  ColumnNumber   => SQL_Column_Number(Col),
                  TargetType     => SQL_C_CHAR,
                  TargetValue    => To_SQLPOINTER(Buffer'Address),
                  BufferLength   => Buffer'Length,
                  StrLen_Or_Ind  => Buffer_Length'Access
               );
         Result(Col) :=
            To_Unbounded_String(Buffer(1..Integer(Buffer_Length)));
      end loop;

      return Result_Record( Result );

   exception
      when No_Data =>
         SQLFreeHandle (SQL_HANDLE_STMT, Data.StatementHandle);
         return null;

      when Error : others =>
         Data.Last_Error   := To_Unbounded_String(Exception_Name (Error));
         Data.Last_Message := To_Unbounded_String(Exception_Message (Error));

         SQLFreeHandle (SQL_HANDLE_STMT, Data.StatementHandle);
         return null;
   end Fetch;

   --- ********************************************************************* ---
   --- **                                                                *** ---
   --- ** SECTION 4.    D A T A B A S E   M A N A G E M E N T            *** ---
   --- **                                                                *** ---
   --- ********************************************************************* ---

   ----------------
   -- Procedures --
   ----------------
   function Procedures(
      This         : in Object;
      Catalog_Name : in String;
      Schema_Name  : in String ) return String_Array_Access is
      -- get a list of all table names
      Data            : Object_Data_Access := This.Data;
      StatementHandle : SQLHSTMT;
      Result          : String_Array_Access := null;
      Rows            : Integer;
      RC              : SQLRETURN;

      Pattern         : String := SQL_ALL_PATTERN;
      Catalog         : aliased String := 40 * ' ';
      Catalog_Len     : aliased SQLINTEGER;
      Schema          : aliased String := 40 * ' ';
      Schema_Len      : aliased SQLINTEGER;
      Name            : aliased String := 40 * ' ';
      Name_Len        : aliased SQLINTEGER;
      Remarks         : aliased String := 40 * ' ';
      Remarks_Len     : aliased SQLINTEGER;
   begin
      Data.Last_Error   := Null_Unbounded_String;
      Data.Last_Message := Null_Unbounded_String;

      SQLAllocHandle (
         SQL_HANDLE_STMT,
         Data.ConnectionHandle,
         StatementHandle
      );

      SQLProcedures (
         StatementHandle,
         CatalogName => Catalog_Name,
         SchemaName  => Schema_Name
      );

      Rows := Integer( SQLRowCount(StatementHandle) );
      Put_Line("Rows:" & Integer'Image(Rows));

      Result := new String_Array( 1..Integer(Rows) );

      Put_Line("Number of Procedures: " & Integer'Image(Rows));

      SQLBindCol (StatementHandle, 1, Catalog'Access, Catalog_Len'Access);
      SQLBindCol (StatementHandle, 2, Schema'Access,  Schema_Len'Access );
      SQLBindCol (StatementHandle, 3, Name'Access,    Name_Len'Access   );
      SQLBindCol (StatementHandle, 7, Remarks'Access, Remarks_Len'Access);

      loop
         RC := SQLFetch( Data.StatementHandle );
         exit when RC /= SQL_SUCCESS;

         Put_Line("Procedure: " & Name(1..Integer(Name_Len)) );
      end loop;

      SQLFreeHandle (SQL_HANDLE_STMT, StatementHandle);
      return Result;

   exception
      when Error : others =>
         Put_Line("Exception in Precedures");
         Data.Last_Error   := To_Unbounded_String(Exception_Name (Error));
         Data.Last_Message := To_Unbounded_String(Exception_Message (Error));

         SQLFreeHandle (SQL_HANDLE_STMT, StatementHandle);
         raise;
   end Procedures;

   ------------
   -- Tables --
   ------------
   function Tables(
      This         : in Object;
      Catalog_Name : in String;
      Schema_Name  : in String  ) return String_Array_Access is
      -- get a list of all table names
      Data            : Object_Data_Access := This.Data;
      StatementHandle : SQLHSTMT;
      Result          : String_Array_Access := null;
      Rows            : Integer;
      RC              : SQLRETURN;
      Pattern         : String := SQL_ALL_PATTERN;

      Catalog         : aliased String := 40 * ' ';
      Catalog_Len     : aliased SQLINTEGER;
      Schema          : aliased String := 40 * ' ';
      Schema_Len      : aliased SQLINTEGER;
      Name            : aliased String := 40 * ' ';
      Name_Len        : aliased SQLINTEGER;
   begin
      Data.Last_Error   := Null_Unbounded_String;
      Data.Last_Message := Null_Unbounded_String;

      SQLAllocHandle (
         SQL_HANDLE_STMT,
         Data.ConnectionHandle,
         StatementHandle
      );

      SQLTables (StatementHandle,
         CatalogName => Catalog_Name,
         SchemaName  => Schema_Name,
         TableName   => Pattern
      );

      Rows := Integer( SQLRowCount(StatementHandle) );
      Put_Line("Number of Tables: " & Integer'Image(Rows));

      Result := new String_Array( 1..Integer(Rows) );

      SQLBindCol (StatementHandle, 1, Catalog'Access, Catalog_Len'Access);
      SQLBindCol (StatementHandle, 2, Schema'Access,  Schema_Len'Access );
      SQLBindCol (StatementHandle, 3, Name'Access,    Name_Len'Access   );
      loop
         RC := SQLFetch( Data.StatementHandle );
         exit when RC /= SQL_SUCCESS;

         Put_Line("Table: " & Name(1..Integer(Name_Len)) );
      end loop;

      SQLFreeHandle (SQL_HANDLE_STMT, StatementHandle);
      return Result;

   exception
      when Error : others =>
         Data.Last_Error   := To_Unbounded_String(Exception_Name (Error));
         Data.Last_Message := To_Unbounded_String(Exception_Message (Error));

         SQLFreeHandle (SQL_HANDLE_STMT, StatementHandle);
         raise;
   end Tables;

   ------------------
   -- Data_Source --
   ------------------
   function Data_Source(
      This  : in Object;
      First : in Boolean := False ) return Source_Information is
      -- return the data source available in this data base object
      Data        : Object_Data_Access := This.Data;
      Direction   : SQL_EXTENDED_FETCH_DIRECTION := SQL_FETCH_NEXT;

      ServerName  : String( 1..256 );
      Description : String( 1..256 );
      RC          : SQLRETURN;
      Result      : Source_Information := Null_Source_Information;

   begin
      if First then
         Direction := SQL_FETCH_FIRST;
      end if;

      SQLDataSources(
         Data.EnvironmentHandle,
         Direction,
         ServerName,
         Description,
         RC);

      if RC = SQL_SUCCESS then
         Result.Server      := To_Unbounded_String( Trim(ServerName,Right) );
         Result.Description := To_Unbounded_String( Trim(Description,Right) );
      end if;

      return Result;
   end Data_Source;


end ODBC;


