/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *     (This file copied from apt-get, copyright someone else)
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */
using namespace std;

#include "cache.h"
#include "dialogprogress.h"
#include "acquirestatus.h"
#include "childprocess.h"
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/dpkgpm.h>
#include <apt-pkg/strutl.h>
#include <signal.h>
#include <errno.h>


static string broken_string(GAptCache* cache);
static string stats_string(GAptCache* cache);

static vector<string> removed_names   (GAptCache* cache);
static vector<string> installed_names (GAptCache* cache);
static vector<string> kept_names      (GAptCache* cache);
static vector<string> upgraded_names  (GAptCache* cache);
static vector<string> held_names      (GAptCache* cache);


static GAptCacheFile* cf = 0;

GAptCacheFile* 
gnome_apt_cache_file(void)
{
  return cf;
}

static void
failed_to_open_dialog()
{
  gnome_apt_fatal_dialog(_("Fatal error opening the package cache file\n"
                           "which describes the packages on your system."));
}

void       
gnome_apt_cache_file_init()
{
  g_return_if_fail(cf == 0);

  cf = new GAptCacheFile;

  if (cf->Open() && cf->cache()) {
    return;
  }
  else {
    failed_to_open_dialog();
    if (cf) delete cf;
    cf = 0;
  }
}

void
gnome_apt_cache_file_shutdown()
{
#ifdef GNOME_ENABLE_DEBUG
  ga_debug("Shutting down the cache");
#endif

  if (cf == 0) return;

  delete cf;
  cf = 0;
}

GAptCache::GAptCache(pkgCache *pCache)
  : pkgDepCache(pCache)
{

}

void
GAptCacheFile::clear()
{
  // It is vital to set all the view to 0 
  //  before we actually destroy the cache. 
  // This allows them to extract any info from the 
  //  previous cache that needs to be saved.
  set<CacheView*>::iterator i = views_.begin();
  while (i != views_.end())
    {
      (*i)->set_cache(0);
      ++i;
    }
  delete records_;
  records_ = 0;
  delete cache_;
  cache_ = 0;
  delete map_;
  map_ = 0;
  delete file_;
  file_ = 0;
}

// GAptCacheFile::Open - Open the cache file				/*{{{*/
// ---------------------------------------------------------------------
/* This routine generates the caches and then opens the dependency cache
   and verifies that the system is OK. */
// can call this multiple times.
bool 
GAptCacheFile::Open()
{
  string progressmessage;

  if (cache_ != 0){
    progressmessage = _("Reopening Apt's Package Cache File");
    clear();
  }
  else {
    progressmessage = _("Creating Apt's Package Cache File");
    // This is here to prevent double-locking when reopening the cache
    // file
    if (_system->Lock() == false)
      return false;
  }

  // Create a progress class
  OpDialogProgress* Progress = new OpDialogProgress(progressmessage.c_str());

  if (_error->PendingError() == true)
    return false;
      
  // Read the source list
  pkgSourceList List;
  if (List.ReadMainList() == false)
    return _error->Error(_("The list of sources could not be read."));
   
   // Build all of the caches
  pkgMakeStatusCache(List,*Progress);
  if (_error->PendingError() == true)
    return _error->Error(_("The package lists or status file could not be parsed or opened."));

  Progress->Done();
   
  // Open the cache file
  file_ = new FileFd(_config->FindFile("Dir::Cache::pkgcache"),FileFd::ReadOnly);
  if (_error->PendingError() == true) {
    delete file_;
    file_ = 0;
    return false;
  }
   
  map_ = new MMap(*file_,MMap::Public | MMap::ReadOnly);
  if (_error->PendingError() == true) {
    delete map_;
    map_ = 0;
    return false;
  }

  pkgCache *pkgcache = new pkgCache(map_);
  if (_error->PendingError() == true) {
    delete pkgcache;
    return false;
  }

  cache_ = new GAptCache(pkgcache);
  cache_->Init(Progress);
  if (_error->PendingError() == true) {
    delete cache_;
    cache_ = 0;
    return false;
  }

  Progress->Done();

  delete Progress;
  Progress = 0;

   // Check that the system is OK
  if (cache_->DelCount() != 0 || cache_->InstCount() != 0)
    return _error->Error("Internal Error, non-zero counts"
                         " (del count %ld, inst count %ld)",
                         cache_->DelCount(), cache_->InstCount());
   
   // Apply corrections for half-installed packages
  if (pkgApplyStatus(*cache_) == false)
    return false;

#if 0 
  // There's no need to refuse to run if stuff is broken; 
  //  we only have to refuse to Complete Run.
   
   // 
  if (cache_->BrokenCount() != 0)
    {
      bool do_fix = false;
       
      // Attempt to fix broken things
      if (_config->FindB("APT::Get::Fix-Broken",false) == true) {
        do_fix = true;
      }
      else {
        // We aren't doing this right; the dialog could be huge. 
        //  need to stuff in a CList.
        string message = _("Some packages are broken. Should Apt try to fix them?\n"
                           "(If they aren't fixed, Apt cannot be used.)\n");
        message += broken_string(cache_);
         
        // give them another chance
        GtkWidget* dialog = 
          gnome_message_box_new(message.c_str(),
                                GNOME_MESSAGE_BOX_QUESTION,
                                GNOME_STOCK_BUTTON_YES,
                                GNOME_STOCK_BUTTON_NO, 
                                NULL);
        gnome_dialog_set_default(GNOME_DIALOG(dialog), 0);
        gnome_apt_setup_dialog(dialog);
        gint response = gnome_dialog_run(GNOME_DIALOG(dialog));
        if (response == 0) {
          do_fix = true;
        }
        else return _error->Error(_("Unmet dependencies. Cannot continue."));
      }

      if (do_fix) {
        if (pkgFixBroken(*cache_) == false || cache_->BrokenCount() != 0) {
          return _error->Error(_("Unable to correct dependencies - something is very wrong."));
        }
        else {
          GtkWidget* dialog = gnome_ok_dialog("Apt had to fix the dependencies for some broken packages.\n"
                                              "Things seem to be fine now.");
          gnome_apt_setup_dialog(dialog);
        }
     
        if (pkgMinimizeUpgrade(*cache_) == false)
          return _error->Error(_("Unable to minimize the upgrade set"));      
      }      
    } // end of "if stuff is broken"

#endif // #if 0

  // Not sure this is a good idea; I'm assuming it's expensive to 
  //  create the Records, and not too much memory will be used.
  records_ = new pkgRecords(*cache_);
  if (_error->PendingError() == true) {
    delete records_;
    records_ = 0;
    return _error->Error(_("Couldn't open the package files"));
  }

  // All done! Cache is open...


  // Refill our list of sections

#if 0
  sections_.clear();
        pkgCache::PkgIterator i = cache_->PkgBegin();

      while (!i.end()) {
        Category * c = 0;
        const char* section = i.Section();
        
        if (section == 0) {           // apparently this can happen.
          if (nosection_bucket == 0) {
            nosection_bucket = new Category(_("no section"), this);
            category_list_->add_node(nosection_bucket);
          }
          c = nosection_bucket;
        }        
        else {
          map<string,Category*>::iterator sec = sections.find(string(section));
          
          if (sec == sections.end()) {
            c = new Category(section, this);
            sections[string(section)] = c;           // temporary quick tree
            category_list_->add_node(c);
          }
          else {
            c = sec->second;
          }
        }
        
        g_assert(c != 0);
        
        Item* item = new Pkg(Item::PackageItem, i, this, c);
        
        c->add(item);
        
        ++i;
      }
#endif
  

  set<CacheView*>::iterator i = views_.begin();
  while (i != views_.end())
    {
      (*i)->set_cache(cache_);
      ++i;
    }   
   
  return true;
}
									/*}}}*/
bool 
GAptCacheFile::UpdateAndReopen()
{
  // Get the source list
  pkgSourceList List;
  if (List.ReadMainList() == false)
    return false;

   // Lock the list directory
  if (_config->FindB("Debug::NoLocking",false) == false)
    {
      FileFd Lock(GetLock(_config->FindDir("Dir::State::Lists") + "lock"));
      if (_error->PendingError() == true)
        return _error->Error("Unable to lock the list directory");
    }
   
  // Create the download object
  GAcqStatus Stat;
  pkgAcquire Fetcher(&Stat);
   
   // Populate it with the source selection
  if (List.GetIndexes(&Fetcher) == false)
    return false;
   
  // Run it
  pkgAcquire::RunResult res = Fetcher.Run();
  switch (res)
    {
    case pkgAcquire::Failed:
      return false;
      break;
    case pkgAcquire::Cancelled:
      return true; // bail out, but not an error
      break;
    case pkgAcquire::Continue:
      /* nothing, continue */
      break;
    default:
      g_assert_not_reached();
      break;
    }

   // Clean out any old list files
  if (Fetcher.Clean(_config->FindDir("Dir::State::lists")) == false ||
      Fetcher.Clean(_config->FindDir("Dir::State::lists") + "partial/") == false)
    return false;
   
   // Prepare the cache.   
  if (Open() == false) {
    failed_to_open_dialog(); // fatal error
    return false;
  }

   
  return true;
}

// Show what we are going to do and get confirmation
class Preview {
public:
  Preview(GAptCache* cache) : cache_(cache) {}
  ~Preview() {}

  bool confirm();

private:
  GAptCache* cache_;

  GtkCTreeNode* add_category(GtkWidget* tree,
                             const char* category_name,
                             vector<string> names, 
                             GtkCTreeNode* category);
};

GtkCTreeNode* 
Preview::add_category(GtkWidget* tree,
                      const char* category_name,
                      vector<string> names, 
                      GtkCTreeNode* category)
{
  const gchar* text[1] = { 0 };
  
  if (!names.empty())
    {
      text[0] = category_name;

      category = gtk_ctree_insert_node(GTK_CTREE(tree),
                                       0,
                                       category,
                                       const_cast<char**>(text),
                                       0, // spacing
                                       0, 0, 0, 0,
                                       FALSE, // not a leaf
                                       FALSE); // expanded
      
      GtkCTreeNode* name = 0;
      vector<string>::iterator i = names.begin();
      while (i != names.end()) 
        {
          text[0] = (*i).c_str();

          name = gtk_ctree_insert_node(GTK_CTREE(tree),
                                       category,
                                       name,
                                       const_cast<char**>(text),
                                       0, // spacing
                                       0, 0, 0, 0,
                                       TRUE,
                                       FALSE);
          ++i;
        }
      
    }

  return category;
}


bool
Preview::confirm()
{
  // Check for removing essential
  pkgCache::PkgIterator i = cache_->PkgBegin();
  string essential;
  bool *added = new bool[cache_->Head().PackageCount];

  // Set whole array to false
  memset(static_cast<void*>(added), 
         '\0', 
         cache_->Head().PackageCount*sizeof(bool)); 

  while (!i.end())
    {
      if ((i->Flags & pkgCache::Flag::Essential) != pkgCache::Flag::Essential)
        {
          ++i;
          continue;
        }
      
      // The essential package is being removed
      if ((*cache_)[i].Delete() == true)
        {
          if (added[i->ID] == false)
            {
              added[i->ID] = true;
              essential += string(i.Name()) + "\n";
            }
        }
      
      if (i->CurrentVer == 0)
        {
          ++i;
          continue;
        }
      
      // Print out any essential package depenendents that are to be removed
      pkgCache::DepIterator d = i.CurrentVer().DependsList();
      while (!d.end())
        {
          // Skip everything but depends
          if (d->Type != pkgCache::Dep::PreDepends &&
              d->Type != pkgCache::Dep::Depends)
            {
              ++d;
              continue;
            }
          
          pkgCache::PkgIterator P = d.SmartTargetPkg();
          if ((*cache_)[P].Delete() == true)
            {
              if (added[P->ID] == true)
                {
                  ++d;
                  continue;
                }
              added[P->ID] = true;
              
              char buf[300];
              g_snprintf(buf,299,"%s (due to %s) ",P.Name(),i.Name());
              essential += buf;
            }	 
          ++d;
        }

      ++i;
    }
   
  delete [] added;
  if (essential.empty() == false)
    {
      string stern_warning = _("WARNING: The following essential packages will be removed\n");
      stern_warning += essential;
      stern_warning += _("This should NOT be done unless you know exactly what you are doing!\n"
                         "Are you sure you want to break your system?");
      
      GtkWidget* dialog = gnome_question_dialog_modal(stern_warning.c_str(),
                                                      NULL,
                                                      NULL);
  
      gnome_dialog_set_default(GNOME_DIALOG(dialog), GNOME_NO);
      gnome_apt_setup_dialog(dialog);    

      int reply = gnome_dialog_run(GNOME_DIALOG(dialog));
      
      if (reply != GNOME_YES)
        return false;
    }
  
  string statsmessage = _("Apt is ready to perform the changes\n"
                          "you have requested\n"
                          "Click OK to proceed.\n\n");

  statsmessage += stats_string(cache_);
    
  GtkWidget* stats = gtk_label_new(statsmessage.c_str());
  
  GtkWidget* dialog = gnome_dialog_new(_("Complete run"),
                                       GNOME_STOCK_BUTTON_OK,
                                       GNOME_STOCK_BUTTON_CANCEL,
                                       NULL);

  gnome_apt_setup_dialog(dialog);
  gnome_dialog_set_close(GNOME_DIALOG(dialog), TRUE);
  gtk_window_set_policy(GTK_WINDOW(dialog),TRUE,TRUE,FALSE);
  gtk_widget_set_usize(dialog, 290, 380);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
                     stats, FALSE, FALSE, 0);

  GtkWidget* frame = gtk_frame_new(_("Summary of changes"));

  gtk_container_border_width(GTK_CONTAINER(frame), GNOME_PAD);

  GtkWidget* sw = gtk_scrolled_window_new(NULL,NULL);

  gtk_container_border_width(GTK_CONTAINER(sw), GNOME_PAD);

  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), 
                                 GTK_POLICY_AUTOMATIC,
                                 GTK_POLICY_AUTOMATIC);

  GtkWidget* tree = gtk_ctree_new(1,0);

  gtk_container_add(GTK_CONTAINER(sw), tree);

  gtk_container_add(GTK_CONTAINER(frame), sw);

  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox),
                     frame, TRUE, TRUE, 0);

  GtkCTreeNode* category = 0;
  
  category = add_category(tree, _("To be installed"),
                          installed_names(cache_), category);
  category = add_category(tree, _("To be deleted"),
                          removed_names(cache_), category);
  category = add_category(tree, _("To be kept"),
                          kept_names(cache_), category);
  category = add_category(tree, _("To be upgraded"),
                          upgraded_names(cache_), category);
  category = add_category(tree, _("To be held"), 
                          held_names(cache_), category);

  gtk_widget_show_all(GNOME_DIALOG(dialog)->vbox);
  
  int reply = gnome_dialog_run(GNOME_DIALOG(dialog));
  
  if (reply == 0) 
    return true;
  
  return false;
}

bool
GAptCacheFile::Install()
{
  if (cache_->DelCount() == 0 && 
      cache_->InstCount() == 0 && 
      cache_->BadCount() == 0)
    {
      GtkWidget* dialog = gnome_ok_dialog(_("There are no packages to be deleted or installed."));
      gnome_apt_setup_dialog(dialog);
      return true;  // it's not an error, just silly.
    }

   // Sanity check
  if (cache_->BrokenCount() != 0)
    {
      string message = _("There are broken packages (unmet dependencies),"
                         " so I can't continue.\n"
                         "You might try the `Fix Broken' menu item.");
      message += broken_string(cache_);
      GtkWidget* dialog = gnome_error_dialog(message.c_str());
      gnome_apt_setup_dialog(dialog);
      return false;
    }
 
  Preview prev(cache_);
  
  if (prev.confirm()) 
    {
      // Run the simulator ..
      if (_config->FindB("APT::Gnome::Simulate") == true)
        {
          pkgSimulate PM(cache_);
          return PM.DoInstall();
        }

      // Create the text record parser
      pkgRecords Recs(*cache_);
      if (_error->PendingError() == true)
        return false;
   
      // Lock the archive directory
      if (_config->FindB("Debug::NoLocking",false) == false)
        {
          FileFd Lock(GetLock(_config->FindDir("Dir::Cache::Archives") + "lock"));
          if (_error->PendingError() == true)
            return _error->Error(_("Unable to lock the download directory"));
        }
   
      // Create the download object
      GAcqStatus Stat;
      pkgAcquire Fetcher(&Stat);

      // Read the source list
      pkgSourceList List;
      if (List.ReadMainList() == false)
        return _error->Error(_("The list of sources could not be read."));
   
      // Create the package manager and prepare to download
      pkgDPkgPM PM(cache_);
      if (PM.GetArchives(&Fetcher,&List,&Recs) == false)
        return false;

      // Display statistics
      double FetchBytes = Fetcher.FetchNeeded();
      double DebBytes = Fetcher.TotalNeeded();
      if (DebBytes != cache_->DebSize())
        {
          cerr << DebBytes << ',' << cache_->DebSize() << endl;
          cerr << "How odd.. The sizes didn't match, email apt@packages.debian.org" << endl;
        }

      string confirm;

      gchar buf[512];

      // Number of bytes
      if (DebBytes != FetchBytes)
        g_snprintf(buf, 511, _("Need to get %sB/%sB of archives\n"), 
                   SizeToStr(FetchBytes).c_str(),
                   SizeToStr(DebBytes).c_str());
      else
        g_snprintf(buf, 511, _("Need to get %sB of archives\n"), 
                   SizeToStr(DebBytes).c_str());
      
      confirm += buf;

      // Size delta
      if (cache_->UsrSize() >= 0)
        g_snprintf(buf, 511, _("%sB will be used\n"),
                   SizeToStr(cache_->UsrSize()).c_str());
      else
        g_snprintf(buf, 511, _("%sB will be freed\n"),
                   SizeToStr(-1*cache_->UsrSize()).c_str());

      confirm += buf;

      // I guess this is just a safety check?
      if (_error->PendingError() == true)
        return false;
   
      
      confirm += _("Continue?");

      GtkWidget* dialog = gnome_question_dialog_modal(confirm.c_str(),
                                                      NULL,
                                                      NULL);
      gnome_apt_setup_dialog(dialog);

      int reply = gnome_dialog_run(GNOME_DIALOG(dialog));
      
      if (reply != GNOME_YES)
        return true; // not an error, just a bailout.
   
      // Run it
      pkgAcquire::RunResult res = Fetcher.Run();
      switch (res)
        {
        case pkgAcquire::Failed:
          return false;
          break;
        case pkgAcquire::Cancelled:
          return true; // bail out, but not an error
          break;
        case pkgAcquire::Continue:
          /* nothing, continue */
          break;
        default:
          g_assert_not_reached();
          break;
        }

      string errors;

      // Print out errors
      vector<pkgAcquire::Item*>::iterator i = Fetcher.ItemsBegin(); 
      while (i != Fetcher.ItemsEnd())
        {
          if ((*i)->Status == pkgAcquire::Item::StatDone &&
              (*i)->Complete == true) {
            ++i;
            continue;
          }
      
          g_snprintf(buf, 511, _("Failed to fetch %s:\n"
                                 "%s\n"),
                     (*i)->DescURI().c_str(), 
                     (*i)->ErrorText.c_str());


          errors += buf;

          ++i;
        }

      if (!errors.empty())
        {
          errors += _("\nTry to fix the problem?");
          
          dialog = gnome_question_dialog_modal(errors.c_str(),
                                               NULL,
                                               NULL);
          gnome_apt_setup_dialog(dialog);

          reply = gnome_dialog_run(GNOME_DIALOG(dialog));
          
          if (reply != GNOME_YES)
            return true; // not an error, just a bailout.
        }
      
      // Try to deal with missing package files
      if (PM.FixMissing() == false)
        {
          return _error->Error(_("Unable to correct missing packages."));
        }
   
      ChildDialog cd(_("Running dpkg"),
                     _("Interact with the package installer in the window below."));

      // Give up the lock on the cache, just while the child is running
      _system->UnLock();

      int pid = cd.fork();
      switch (pid) {
      case -1:
        {
          string message = _("Unable to run the dpkg package manager\n");
          message += g_strerror(errno);
          GtkWidget* dialog = gnome_error_dialog(message.c_str());
          gnome_apt_setup_dialog(dialog);
          return false;
        }
        break;
      case 0:
        {
#if 0
          // Could invoke a GDK signal handler, which would be very
          // bad. (these break DoInstall, SIGCHLD probably)
          signal(SIGCHLD, SIG_DFL);
          signal(SIGPIPE,SIG_DFL);
          signal(SIGQUIT,SIG_DFL);
          signal(SIGINT,SIG_DFL);
          signal(SIGWINCH,SIG_DFL);
          signal(SIGCONT,SIG_DFL);
          signal(SIGTSTP,SIG_DFL);
#endif

          // No GUI after this!
          cout << _("  GNOME Apt: Running dpkg...") << endl;
          cout << flush;
      
          pkgPackageManager::OrderResult Res = PM.DoInstall();

          if (Res == pkgPackageManager::Failed || 
              _error->PendingError() == true)
            {
              _error->DumpErrors();
              cerr << _("  GNOME Apt: Install Failed. It is now safe to close this window.") << endl;
              cout << flush;
              cerr << flush;
              _exit(EXIT_FAILURE);
            }
          else if (Res == pkgPackageManager::Completed)
            {
              cout << _("  GNOME Apt: Install successful. It is now safe to close this window.") << endl;
              cout << flush;
              cerr << flush;
              _exit(EXIT_SUCCESS);
            }
          else
            {
              cout << _("  GNOME Apt: Ugh, I think you need to swap media, and gnome-apt doesn't know what do do about this right now. Apologies... will be fixed in a future version.") << endl;
              cout << flush;
              cerr << flush;
              _exit(EXIT_FAILURE);
            }
          g_assert_not_reached();
        }
        break;
      default:
        {
          // wait for dialog to close - dpkg should then be killed
          bool retval = cd.run();
          
          // get lock back
          if (_system->Lock() == false)
            gnome_apt_fatal_dialog(_("Failed to get the lock back from dpkg"));

   
          // Reopen the cache.   
          if (Open() == false) {
            failed_to_open_dialog(); // fatal error
            return false;
          }
          
          return retval;
        }
        break;
      }
      g_assert_not_reached();
      return false; // not reached
    }
  else 
    {
      return true; // not an error, just a cancel
    }

  return true;
}


void
GAptCacheFile::MarkUpgrades()
{
  if (pkgAllUpgrade(*cache_) == false)
    {
      g_warning("Internal error, AllUpgrade broke stuff");
    }
}   

void
GAptCacheFile::SmartMarkUpgrades()
{
  if (pkgDistUpgrade(*cache_) == false)
    {
      string broken = _("Unable to figure out an upgrade path, consider\n"
                        "making some changes manually then trying again.\n");
      broken += broken_string(cache_);
      GtkWidget* dialog = gnome_error_dialog(broken.c_str());
      gnome_apt_setup_dialog(dialog);
    }
}

void
GAptCacheFile::Fix()
{
  pkgProblemResolver fixer(cache_);

  fixer.InstallProtect();
  if (fixer.Resolve(true) == false)
    {
      gnome_apt_error_dialog(_("Error - some problems were unresolvable. If you are using\n"
                               "an unstable version of Debian, it's possible one or more\n"
                               "needed packages are not on the server.\n"
                               "Otherwise one or more packages may simply be broken and\n"
                               "uninstallable\n"));
    }
}

static vector<string> 
removed_names   (GAptCache* cache)
{
  vector<string> retval;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end())
    {
      if ((*cache)[i].Delete())
        retval.push_back(string(i.Name()));
      
      ++i;
    }

  return retval;
}

static vector<string> 
installed_names (GAptCache* cache)
{
  vector<string> retval;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end())
    {
      if ((*cache)[i].NewInstall())
        retval.push_back(string(i.Name()));

      ++i;
    }

  return retval;
}

static vector<string> 
kept_names      (GAptCache* cache)
{
  vector<string> retval;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end())
    {
      // Not interesting
      if ((*cache)[i].Upgrade() == true || 
          (*cache)[i].Upgradable() == false ||
          i->CurrentVer == 0 || 
          (*cache)[i].Delete() == true)
        ; // nothing

      else retval.push_back(string(i.Name()));

      ++i;
    }

  return retval;
}

static vector<string> 
upgraded_names  (GAptCache* cache)
{
  vector<string> retval;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end())
    {
      // Not interesting
      if ((*cache)[i].Upgrade() == false || 
          (*cache)[i].NewInstall() == true)
        ; // nothing

      else retval.push_back(string(i.Name()));

      ++i;
    }

  return retval;
}

static vector<string> 
held_names      (GAptCache* cache)
{
  vector<string> retval;

  pkgCache::PkgIterator i = cache->PkgBegin();
  while (!i.end())
    {
      if ((*cache)[i].InstallVer != static_cast<pkgCache::Version*>(i.CurrentVer())
          && i->SelectedState == pkgCache::State::Hold)
        retval.push_back(string(i.Name()));
      
      ++i;
    }
  
  return retval;
}

static string 
stats_string(GAptCache* cache)
{
  gchar buf[512];

  gulong upgrade = 0;
  gulong install = 0;

  pkgCache::PkgIterator i = cache->PkgBegin(); 
  while (!i.end()) 
    {
      if      ( (*cache)[i].NewInstall() ) ++install;
      else if ( (*cache)[i].Upgrade()    ) ++upgrade;

      ++i;
    }
  
  g_snprintf(buf, 511, _("%lu packages upgraded\n%lu packages newly installed\n"
                          "%lu packages to remove\n%lu packages not upgraded\n"
                         "%lu packages not fully installed or removed."),
             upgrade, install, cache->DelCount(), cache->KeepCount(),
             cache->BadCount());
  
  return string(buf);
}

static string
broken_string(GAptCache* cache)
{
  string out;

  out += "The following packages have unmet dependencies:\n";

  pkgCache::PkgIterator I = cache->PkgBegin();
  for (;I.end() != true; I++)
    {
      if ((*cache)[I].InstBroken() == false)
        continue;
	  
      // Print out each package and the failed dependencies
      out += "  ";
      out +=  I.Name();
      out += ":";
      int Indent = strlen(I.Name()) + 3;
      bool First = true;
      if ((*cache)[I].InstVerIter(*cache).end() == true)
        {
          out += "\n";
          continue;
        }
      
      for (pkgCache::DepIterator D = (*cache)[I].InstVerIter(*cache).DependsList(); D.end() == false;)
        {
          // Compute a single dependency element (glob or)
          pkgCache::DepIterator Start;
          pkgCache::DepIterator End;
          D.GlobOr(Start,End);
	 
          if (cache->IsImportantDep(End) == false || 
              ((*cache)[End] & pkgDepCache::DepGInstall) == pkgDepCache::DepGInstall)
            continue;
	 
          if (First == false)
            for (int J = 0; J != Indent; J++)
              out += ' ';
          First = false;

          out += ' ';
          out += End.DepType();
          out += ": ";
          out += End.TargetPkg().Name();
	 
          // Show a quick summary of the version requirements
          if (End.TargetVer() != 0) {
            out += " (";
            out += End.CompType();
            out += " ";
            out += End.TargetVer();
            out += ")";
          }
	 
          /* Show a summary of the target package if possible. In the case
             of virtual packages we show nothing */
	 
          pkgCache::PkgIterator Targ = End.TargetPkg();
          if (Targ->ProvidesList == 0)
            {
              out += " but ";
              pkgCache::VerIterator Ver = (*cache)[Targ].InstVerIter(*cache);
              if (Ver.end() == false) {
                out += Ver.VerStr();
                out += " is installed";
              }
              else
                {
                  if ((*cache)[Targ].CandidateVerIter(*cache).end() == true)
                    {
                      if (Targ->ProvidesList == 0)
                        out += "it is not installable";
                      else
                        out += "it is a virtual package";
                    }		  
                  else
                    out += "it is not installed";
                }	       
            }
	 
          out += "\n";
        }	    
    }   
  return out;
}
