//  gdeb (Gdeb class)
//  Copyright (C) 1998 Lalo Martins <lalo@debian.org>

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#include <gnome.h>
#include <ctype.h>

#include "gdeb.h"
#include "debianlogo.xpm"

using namespace std;

static GnomeStockPixmapEntryData stock_debian_logo = {
  GNOME_STOCK_PIXMAP_TYPE_DATA,
  114, 159,
  "debian_logo",
  debianlogo_xpm
};

static bool registered_logo = false;

// we keep an instance of this for when we don't have any
//  package
class NoPackage : public GDeb::Package
{
public:
  // Stuff to display
  virtual string Name() { return _("No package selected"); }
  virtual string Version() { return ""; }
  virtual string Description_Short()  { return ""; }
  virtual string Description_Long()  { return ""; }
  virtual string Section()  { return ""; }
  virtual string Priority()  { return ""; }
  virtual string Size()  { return ""; }
  virtual string Installed_Size()  { return ""; }
  virtual string Maintainer()  { return ""; }
  virtual string Status()  { return ""; }
  virtual string Current_Version()  { return ""; }
  virtual string Architecture()  { return ""; }
  virtual string Source()  { return ""; }
  virtual string Filename()  { return ""; }
  virtual bool   Essential() { return false; }
  virtual bool   Downloadable() { return false; }
  virtual bool   Automatic() { return false; }
  virtual vector<string> Provides() { return vector<string>(0); }
  virtual vector<string> PreDepends()  { return vector<string>(0); }
  virtual vector<string> Depends()  { return vector<string>(0); }
  virtual vector<string> Recommends()  { return vector<string>(0); }
  virtual vector<string> Suggests()  { return vector<string>(0); }
  virtual vector<string> Conflicts()  { return vector<string>(0); }
  virtual vector<string> Replaces()  { return vector<string>(0); }
  
  // Actions - should not be called on a nopackage
  virtual void Install() { g_warning(__FUNCTION__); }
  virtual void Delete() { g_warning(__FUNCTION__); }
  virtual void Keep() { g_warning(__FUNCTION__); }

};

static NoPackage* no_package = 0;

// rebuilds the Containers (for example, if configuration changed)
void 
GDeb::rebuild ()
{
  if (toplevel_ == 0)
    {
      toplevel_ = gtk_vbox_new(FALSE, 5);
    }


  if (conf_.show_title) 
    {
      if (title_ == 0)
        {
          title_ = gtk_hbox_new(FALSE, 10);
          gtk_box_pack_start(GTK_BOX(toplevel_), title_, FALSE, FALSE, 0);
        }
    }
  else 
    {
      if (title_) gtk_widget_destroy(title_);
      title_ = 0;
      // child widgets
      logo_ = 0;
      short_desc_ = 0;
      package_ = 0;
    }


  if (title_ != 0 && logo_ == 0 && package_ == 0 && short_desc_ == 0)
    {
      if (registered_logo == false)
        {
          gnome_stock_pixmap_register (stock_debian_logo.label,
                                       0,
                                       reinterpret_cast<GnomeStockPixmapEntry*>(&stock_debian_logo));
          registered_logo = true;
        }

      GtkWidget* logo_frame = gtk_frame_new(NULL);
      gtk_frame_set_shadow_type(GTK_FRAME(logo_frame), GTK_SHADOW_IN);
      logo_ = gnome_stock_pixmap_widget(toplevel_, stock_debian_logo.label);
      gtk_container_add(GTK_CONTAINER(logo_frame), logo_);
      gtk_box_pack_start(GTK_BOX(title_), logo_frame, FALSE, FALSE, 
                         GNOME_PAD_SMALL);

      GtkWidget* vbox = gtk_vbox_new(FALSE, 3);
      
      gtk_box_pack_end(GTK_BOX(title_), vbox, TRUE, TRUE, 0);

      package_ = gtk_label_new("");
      gtk_widget_set_name(package_, "GDeb_Package");
      short_desc_ = gtk_label_new("");
      gtk_widget_set_name(short_desc_, "GDeb_ShortDesc");

      gtk_box_pack_start (GTK_BOX(vbox), package_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), short_desc_, TRUE, TRUE, 2);

      // Status

      GtkWidget* hbox = gtk_hbox_new(0, FALSE);
      status_ = gtk_label_new("");
      gtk_widget_set_name(status_, "GDeb_Status");
      GtkWidget* status_label = gtk_label_new(_("Status:"));
      gtk_widget_set_name(status_label, "GDeb_Status_Tag");
      gtk_misc_set_alignment(GTK_MISC(status_label), 
                             0.0, 0.5);
      gtk_misc_set_alignment(GTK_MISC(status_), 
                             1.0, 0.5);
      gtk_box_pack_start (GTK_BOX(hbox), status_label, TRUE, TRUE, 0);
      gtk_box_pack_end   (GTK_BOX(hbox), status_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 2);

      // Installed size

      hbox = gtk_hbox_new(0, FALSE);
      installed_size_ = gtk_label_new("");
      gtk_widget_set_name(installed_size_, "GDeb_InstalledSize");
      GtkWidget* installed_size_label = gtk_label_new(_("Installed Size:"));
      gtk_widget_set_name(installed_size_label, "GDeb_InstalledSize_Tag");
      gtk_misc_set_alignment(GTK_MISC(installed_size_label), 
                             0.0, 0.5);
      gtk_misc_set_alignment(GTK_MISC(installed_size_), 
                             1.0, 0.5);
      gtk_box_pack_start (GTK_BOX(hbox), installed_size_label, TRUE, TRUE, 0);
      gtk_box_pack_end   (GTK_BOX(hbox), installed_size_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 2);

      // Section

      hbox = gtk_hbox_new(0, FALSE);
      section_ = gtk_label_new("");
      gtk_widget_set_name(section_, "GDeb_Section");
      GtkWidget* section_label = gtk_label_new(_("Section:"));
      gtk_widget_set_name(section_label, "GDeb_Section_Tag");
      gtk_misc_set_alignment(GTK_MISC(section_label), 
                             0.0, 0.5);
      gtk_misc_set_alignment(GTK_MISC(section_), 
                             1.0, 0.5);
      gtk_box_pack_start (GTK_BOX(hbox), section_label, TRUE, TRUE, 0);
      gtk_box_pack_end   (GTK_BOX(hbox), section_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 2);


      // Priority

      hbox = gtk_hbox_new(0, FALSE);
      priority_ = gtk_label_new("");
      gtk_widget_set_name(priority_, "GDeb_Priority");
      GtkWidget* priority_label = gtk_label_new(_("Priority:"));
      gtk_widget_set_name(priority_label, "GDeb_Priority_Tag");
      gtk_misc_set_alignment(GTK_MISC(priority_label), 
                             0.0, 0.5);
      gtk_misc_set_alignment(GTK_MISC(priority_), 
                             1.0, 0.5);
      gtk_box_pack_start (GTK_BOX(hbox), priority_label, TRUE, TRUE, 0);
      gtk_box_pack_end   (GTK_BOX(hbox), priority_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 2);


      // Download Size

      hbox = gtk_hbox_new(0, FALSE);
      size_ = gtk_label_new("");
      gtk_widget_set_name(size_, "GDeb_Download");
      GtkWidget* size_label = gtk_label_new(_("Download:"));
      gtk_widget_set_name(size_label, "GDeb_Download_Tag");
      gtk_misc_set_alignment(GTK_MISC(size_label), 
                             0.0, 0.5);
      gtk_misc_set_alignment(GTK_MISC(size_), 
                             1.0, 0.5);
      gtk_box_pack_start (GTK_BOX(hbox), size_label, TRUE, TRUE, 0);
      gtk_box_pack_end   (GTK_BOX(hbox), size_, TRUE, TRUE, 2);
      gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 2);

    }

  // We always create the notebook; the config option 
  //  determines whether it has tabs. Good idea from Lalo
  if (notebook_ == 0)
    {
      notebook_ = gtk_notebook_new();

      gtk_notebook_popup_enable(GTK_NOTEBOOK(notebook_));

      gtk_box_pack_end(GTK_BOX(toplevel_), notebook_, TRUE, TRUE, 0);
    }

  gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook_), conf_.use_notebook);
  
  // Eventually, we will do this differently for advanced/normal mode
  //  and consider vertical/horizontal
  if (pages_[PageDescription] == 0)
    {
      GtkWidget* desc = gtk_vbox_new(FALSE, 2);
      GtkWidget* label = gtk_label_new(_("Description"));

      GtkWidget* sw = gtk_scrolled_window_new(0,0);

      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), 
                                     GTK_POLICY_AUTOMATIC,
                                     GTK_POLICY_AUTOMATIC);

      long_desc_ = gtk_text_new(0,0);

      gtk_container_add(GTK_CONTAINER(sw), long_desc_);

      gtk_box_pack_start(GTK_BOX(desc), sw, TRUE, TRUE, 0);

      gtk_notebook_append_page(GTK_NOTEBOOK(notebook_), desc, label);

      pages_[PageDescription] = desc;
    }
  
  if (pages_[PageRelated] == 0)
    {
      GtkWidget* related = gtk_vbox_new(FALSE, 2);
      GtkWidget* label = gtk_label_new(_("Related"));

      gtk_notebook_append_page(GTK_NOTEBOOK(notebook_), related, label);

      pages_[PageRelated] = related;
    }

  if (pages_[PageSpecial] == 0)
    {
      GtkWidget* special = gtk_vbox_new(FALSE, 2);
      GtkWidget* label = gtk_label_new(_("Special"));

      gtk_notebook_append_page(GTK_NOTEBOOK(notebook_), special, label);

      pages_[PageSpecial] = special;
    }

  if (pages_[PageOther] == 0)
    {
      GtkWidget* other = gtk_vbox_new(FALSE, 2);
      GtkWidget* label = gtk_label_new(_("Other"));

      gtk_notebook_append_page(GTK_NOTEBOOK(notebook_), other, label);

      pages_[PageOther] = other;
    }

  refresh();
}

void
GDeb::refresh()
{
  Package* p = 0;
  if (pkg_ == 0)
    { 
      if (no_package == 0)
        no_package = new NoPackage;

      p = no_package;
    }
  else 
    {
      p = pkg_;
    }

  if (package_ != 0) 
    {
      string s = p->Name();
      s += " ";
      s += p->Version();

      gtk_label_set(GTK_LABEL(package_), s.c_str());
    }

  if (short_desc_ != 0) 
    {
      string s = p->Description_Short();
      
      // world's lamest word wrap
      if (s.size() > 30) {
        guint pos = s.size()/2;
        guint prev = pos;
        while (prev >= 0)
          {
            if (isspace(s[prev])) break;
            --prev;
          }
        guint next = pos;
        while (next < s.size())
          {
            if (isspace(s[next])) break;
            ++next;
          }

        guint which = 0;
        if ( (next - pos) <= (pos - prev) )
          {
            which = next;
          }
        else which = prev;

        s.insert(which, "\n");
      }

      gtk_label_set(GTK_LABEL(short_desc_), s.c_str());
    }

  if (status_ != 0)
    {
      string s = p->Status();
      gtk_label_set(GTK_LABEL(status_), s.c_str());
    }

  if (installed_size_ != 0)
    {
      string s = p->Installed_Size();
      gtk_label_set(GTK_LABEL(installed_size_), s.c_str());
    }

  if (section_ != 0)
    {
      string s = p->Section();
      gtk_label_set(GTK_LABEL(section_), s.c_str());
    }

  if (priority_ != 0)
    {
      string s = p->Priority();
      gtk_label_set(GTK_LABEL(priority_), s.c_str());
    }

  if (size_ != 0)
    {
      string s = p->Size();
      gtk_label_set(GTK_LABEL(size_), s.c_str());
    }

  if (long_desc_ != 0)
    {
      gtk_text_freeze(GTK_TEXT(long_desc_));

      gtk_editable_delete_text(GTK_EDITABLE(long_desc_),
                               0, 
                               gtk_text_get_length(GTK_TEXT(long_desc_)));
                              
      gint position = 0;

      string s = p->Description_Long();

      gtk_editable_insert_text(GTK_EDITABLE(long_desc_),
                               s.c_str(),
                               s.size(),
                               &position);
      
      gtk_text_thaw(GTK_TEXT(long_desc_));
    } 
}


// constructor
GDeb::GDeb (const string & app_name)
  : 
     toplevel_(0),
     title_(0),
     logo_(0),
     package_(0),
     short_desc_(0),
     
     status_(0),
     installed_size_(0),
     section_(0),
     priority_(0),
     size_(0),

     notebook_(0),

     long_desc_(0),

     pkg_(0),
     
     name_(app_name)
{
  int i = 0;
  while (i < static_cast<int>(PageTypeEnd))
    {
      pages_[i] = 0;
      ++i;
    }

  rebuild();
}

// destructor
GDeb::~GDeb()
{
  if (no_package != 0)
    {
      delete no_package;
      no_package = 0;
    }

  // We don't nuke the widgets, since we assume they were
  //  placed in some container.
}

// get the toplevel widget
GtkWidget* 
GDeb::widget()
{
  return toplevel_;
}

// sets the current page
void 
GDeb::set_page (PageType p)
{
  gtk_notebook_set_page(GTK_NOTEBOOK(notebook_), p);
}

// changes the package data
void 
GDeb::set_package (Package* pkg)
{
  // OK to set pkg_ to 0
  pkg_ = pkg;
  refresh();
}

void 
GDeb::set_config (const Config & c) 
{
  // If we change configuration such that widgets will need replacing,
  //  we need to destroy the old widgets and set their slots to 0, so 
  //  rebuild will replace them.

  if (conf_.adv_mode != c.adv_mode)
    {
      // Need to destroy all pages and recreate.

      int i = 0;
      while (i < static_cast<int>(PageTypeEnd))
        {
          if (pages_[i]) 
            gtk_widget_destroy(pages_[i]);
          pages_[i] = 0;
          ++i;
        }  
    }
  
  conf_ = c;
  rebuild();
}
