/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 *   Gnome Apt frontend
 *
 *   Copyright (C) 1998 Havoc Pennington <hp@pobox.com>
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "pkgutil.h"
#include "pkgtree.h"

// Chain from Package* to PkgIterator

void 
Util::remove(pkgCache::Package* p, GAptPkgTree* tree) 
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  remove(i, tree);
}

void 
Util::keep(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  keep(i, tree);
}
  
void 
Util::install(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  install(i, tree);
}

void 
Util::unremove(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  unremove(i, tree);
}

void 
Util::unkeep(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  unkeep(i, tree);
}
  
void 
Util::uninstall(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_if_fail(p != 0);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  uninstall(i, tree);
}
  
bool 
Util::installed(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return installed(i, tree);
}
  
bool 
Util::removed(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return removed(i, tree);
}
  
bool 
Util::kept(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return kept(i, tree);
}
  
bool 
Util::can_change_install(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return can_change_install(i, tree);
}
  
bool 
Util::can_change_remove(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return can_change_remove(i, tree);
}
  
bool 
Util::can_change_keep(pkgCache::Package* p, GAptPkgTree* tree)
{
  g_return_val_if_fail(p != 0, false);
  pkgCache::PkgIterator i(*(tree->cache()), p);
  return can_change_keep(i, tree);
}


// Real functions (use PkgIterator)

void 
Util::remove(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  if (i->Flags & pkgCache::Flag::Essential)
    {
      GtkWidget* dialog = 
        gnome_question_dialog_modal(_("You are trying to remove an essential package.\n"
                                      "This WILL break your system, and is a very, very bad idea.\n"
                                      "Are you sure you want to go ahead?"),
                                    NULL,
                                    NULL);

      gnome_apt_setup_dialog(dialog);      
      gnome_dialog_set_default(GNOME_DIALOG(dialog), GNOME_NO);

      int reply = gnome_dialog_run(GNOME_DIALOG(dialog));

      if (reply != GNOME_YES)
        return;
    }

  tree->cache()->MarkDelete(i);
  tree->package_change_notify();
}

void 
Util::unremove(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  keep(i,tree);
  tree->package_change_notify();
}

void 
Util::keep(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  tree->cache()->MarkKeep(i); // this has a default arg we might care about
  tree->package_change_notify();
}

void 
Util::unkeep(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  if (can_change_install(i,tree))
    {
      if (!installed(i,tree))
        install(i,tree); 
      else
        uninstall(i,tree);
    }
  else if (can_change_remove(i,tree))
    {
      if (removed(i,tree))
        unremove(i,tree);
      else 
        remove(i,tree);
    }

  tree->package_change_notify();
}

void 
Util::install(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  tree->cache()->MarkInstall(i, true); // auto-install enabled
  
  pkgDepCache::StateCache & state = (*tree->cache())[i];
  if (state.Install() == false)
    g_warning("gnome-apt bug: We permitted install, but install did not work");

  // In the future, if the user wants gnome-apt to ask about
  // installing dependencies too, we uncomment the following and
  // MarkInstall(,false) above

#if 0
  if (state.InstBroken() == true)
    {
      // FIXME need to list what we'd have to do to fix things.

      string message = _("Installing this package would break some dependencies.\n"
                         "Should I add and remove packages to resolve the problem?");

      GtkWidget* dialog = 
        gnome_question_dialog_modal(message.c_str(),
                                    NULL,
                                    NULL);

      gnome_apt_setup_dialog(dialog);

      int reply = gnome_dialog_run(GNOME_DIALOG(dialog));

      if (reply == GNOME_YES) 
        {
          tree->cache()->MarkInstall(i, true); // force things this time
        }
      else 
        {
          // revert to previous state - possible infinite loop?
          uninstall(i, tree);
          return;
        }
    }
#endif

  tree->package_change_notify();
}

void 
Util::uninstall(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  keep(i,tree);
  tree->package_change_notify();
}
  
bool 
Util::installed(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  pkgDepCache::StateCache & state = (*tree->cache())[i];

  return state.Install();
}
  
bool 
Util::removed(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  pkgDepCache::StateCache & state = (*tree->cache())[i];

  return state.Delete();
}
  
bool 
Util::kept(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  pkgDepCache::StateCache & state = (*tree->cache())[i];

  return state.Keep();
}
  
bool 
Util::can_change_install(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  // This is actually redundant with state.CandidateVer != 0
  //  perhaps.
  if (i->ProvidesList != 0) return false; // virtual package

  pkgDepCache::StateCache & state = (*tree->cache())[i];

  // We render the install check if it's not installed and there's
  // something to install, or it's installed but upgradeable.

  if (state.Upgradable()) 
    {
      pkgCache::VerIterator vi = state.CandidateVerIter(*tree->cache());
      if (vi.end()) return false; // nothing to upgrade to

      return true;
    }
  // otherwise
  else
    {
      return false;
    }
}
  
bool 
Util::can_change_remove(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  if (i->ProvidesList != 0) return false; // virtual package

  // Render if we're installed
  return (i->CurrentVer != 0);
}
  
bool 
Util::can_change_keep(pkgCache::PkgIterator& i, GAptPkgTree* tree)
{
  if (i->ProvidesList != 0) return false; // virtual package

  pkgDepCache::StateCache & state = (*tree->cache())[i];

  // Render if we're installed and upgradeable, or installed and deleteable
  return (i->CurrentVer != 0);
}


Util::StatusType 
Util::status_type(pkgCache::PkgIterator & i, GAptCache* cache)
{
  pkgDepCache::StateCache & state = (*cache)[i];

  if      (state.NowBroken())  return  StatusNowBroken;
  else if (state.InstBroken()) return  StatusInstBroken;
  else if (state.NewInstall()) return  StatusNewInstall;
  else if (state.Upgrade())    return  StatusUpgrade;
  else if (state.Downgrade())  return  StatusDowngrade;
  else if (state.Delete())     return  StatusDelete;
  else if (state.Install())    return  StatusInstall;
  else if (state.Status == 2)  return  StatusNotInstalled;
  else if (state.Keep())       return  StatusKept;
  else if (state.Held())       return  StatusHeld;

  else                         return  StatusTypeEnd;
}


const char*
Util::status_string(StatusType st)
{
  switch (st) {
  case StatusNowBroken:
    return _("Broken");
    break;
  case StatusInstBroken:
    return _("Would be broken");
    break;
  case StatusNewInstall:
    return _("To be installed");
    break;
  case StatusUpgrade:
    return _("To be upgraded");
    break;
  case StatusDowngrade:
    return _("To be downgraded");
    break;
  case StatusDelete:
    return _("To be deleted");
    break;
  case StatusInstall:
    return _("Installed");
    break;
  case StatusNotInstalled:
    return _("Not installed");
    break;
  case StatusKept:
    return _("Kept");
    break;
  case StatusHeld:
    return _("Held");
    break;

  case StatusTypeEnd:
  default:
    return "";
    break;
  }
#ifdef GNOME_ENABLE_DEBUG 
  g_warning("Status string: not reached");
#endif
  return ""; //not reached
}

static const pkgCache::State::VerPriority MAGIC_NOVERSION_PRIORITY = 
static_cast<pkgCache::State::VerPriority>(100);

const char* 
Util::priority_string(pkgCache::PkgIterator & i, 
                      GAptCache* cache,
                      pkgCache::State::VerPriority* p)
{
  pkgCache::State::VerPriority priority = static_cast<pkgCache::State::VerPriority>(0);

  const char* pri_string = 0;

  pkgCache::VerIterator vi = i.CurrentVer();

  if (!vi.end()) {
    priority = 
      static_cast<pkgCache::State::VerPriority>(vi->Priority);
    if (priority == 0) 
      {
        pri_string = _("No priority available");
      }
    else
      {
        pri_string = vi.PriorityType();
      }
  }
  else {
    // See if we have the info for the candidate version.
    pkgDepCache::StateCache & state = (*cache)[i];
    vi = state.CandidateVerIter(*cache);
    //    vi = i.TargetVer();
    if (!vi.end()) 
      {
        priority = 
          static_cast<pkgCache::State::VerPriority>(vi->Priority);
        if (priority == 0) 
          {
            pri_string = _("No priority available");
          }
        else
          {
            pri_string = vi.PriorityType();
          }
      }
    else 
      {
        priority = MAGIC_NOVERSION_PRIORITY;
        pri_string = _("No version available");
      }
  }

  if (p) *p = priority;
  return pri_string;
}

// I'm quite unsure about this.
pkgCache::DepIterator 
Util::depends(pkgCache::PkgIterator & i, GAptCache* cache)
{
  pkgDepCache::StateCache & state = (*cache)[i];  

  // there's no default constructor for VerIterator, 
  //  so we have to create the InstVerIter in all cases.
  // sucks
  pkgCache::VerIterator vi = state.InstVerIter(*cache);

  StatusType status = status_type(i, cache);

  if (status == StatusNotInstalled)
    {
      // There is no Install Version, so we show deps
      //  for the version that would be installed.
      vi = state.CandidateVerIter(*cache);      
    }
  else if (status == StatusDelete)
    {
      // The InstallVersion is nothing, so we use the current version
      vi = i.CurrentVer();
    }

  if (!vi.end())
    {
      pkgCache::DepIterator di = vi.DependsList();
      return di;
    }
  else 
    {
      pkgCache::DepIterator di;
      g_return_val_if_fail(di.end(), di);
      return di; // di.end() == true
    }
}

pkgCache::VerIterator 
Util::display_version(pkgCache::PkgIterator & i,
                      GAptCache* cache)
{
  pkgDepCache::StateCache & state = (*cache)[i];

  // do our best to find a version - kind of hard to say 
  //  which we should find. 
  pkgCache::VerIterator vi = i.CurrentVer();
  if (vi.end()) {
    vi = state.CandidateVerIter(*cache);
  }

  return vi;
}

pkgRecords::Parser* 
Util::pkg_parser(pkgCache::PkgIterator & i,
                 GAptCache* cache)
{
  pkgCache::VerIterator vi = display_version(i,cache);
  if (vi.end()) return 0;
  
  pkgRecords* r = gnome_apt_cache_file()->records();
  
  g_return_val_if_fail(r != 0, 0);
  
  pkgRecords::Parser & parser = r->Lookup(vi.FileList());

  return &parser;
}
