/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"


#define DEFAULT_ANONYMOUS_PW "you@yourhost"

GnomeCmdData *data = NULL;


struct _GnomeCmdDataPrivate
{
	GList               *ftp_servers;            // GnomeCmdFtpServer pointers
	gchar               *ftp_anonymous_password;
	GnomeCmdSizeDispMode size_disp_mode;
	GnomeCmdDateFormat   date_format;
	GnomeCmdLayout       layout;
	GnomeCmdCvsUpdateColors *cvs_update_colors;
	GnomeCmdLayoutColors *layout_colors;
	gboolean            type_filter[8];
	gboolean            hidden_filter;
	GList               *cvs_servers;
	gint                main_win_width, main_win_height;
	gboolean            case_sens_sort;
	gchar               *font;
	gchar               *cvsroot;
	gboolean            cvs_over_ssh;
	GnomeCmdRightMouseButtonMode right_mouse_button_mode;
	gint                embterm_height;
	gboolean            show_toolbar;
	gboolean            use_mime;
	guint               mime_icon_size;
	GdkInterpType       mime_scale_quality;
	gchar               *mime_icon_dir;
	
	gchar *viewer;
	gchar *editor;
	gchar *differ;
	gchar *cvs_differ;
	gchar *term;
};



static GdkColor *gdk_color_new (gushort r, gushort g, gushort b)
{
	GdkColor *c = g_new (GdkColor, 1);
	c->pixel = 0;
	c->red = r;
	c->green = g;
	c->blue = b;
	return c;
}


static void set_string (const gchar *path, const gchar *value)
{
	gnome_config_set_string (path, value);
}

static void set_int    (const gchar *path, int value)
{
	gnome_config_set_int (path, value);
}

static void set_bool (const gchar *path, gboolean value)
{
	gnome_config_set_bool (path, value);
}

static gchar* get_string (const gchar *path, gchar *def)
{
	gboolean b = FALSE;
	gchar *value = gnome_config_get_string_with_default (path, &b);
	if (b)
		return g_strdup (def);	
	return value;
}

static gint get_int (const gchar *path, int def)
{
	gboolean b = FALSE;
	gint value = gnome_config_get_int_with_default (path, &b);
	if (b)
		return def;
	return value;
}

static gboolean get_bool (const gchar *path, gboolean def)
{
	gboolean b = FALSE;
	gboolean value = gnome_config_get_bool_with_default (path, &b);
	if (b)
		return def;
	return value;
}


static void set_color (const gchar *path, GdkColor *color)
{
	gchar *color_str;
	color_str = g_strdup_printf ("%d %d %d", color->red, color->green, color->blue);
	set_string (path, color_str);
	g_free (color_str);
}


static void get_color (const gchar *path, GdkColor *color)
{
	gint red, green, blue;
	gchar *def = g_strdup_printf ("%d %d %d",
								  color->red, color->green, color->blue);
	gchar *color_str = get_string (path, def);
	if (sscanf (color_str, "%u %u %u", &red, &green, &blue) != 3)
		warn_print ("Illegal color in config file\n");

	if (color_str != def)
		g_free (color_str);
	
	color->red   = (gushort)red;
	color->green = (gushort)green;
	color->blue  = (gushort)blue;

	g_free (def);
}


static void create_conf_dir ()
{
	gchar *dir = g_strdup_printf ("%s/%s", g_get_home_dir (), ".gnome-commander");
	if (mkdir (dir, S_IRUSR|S_IWUSR|S_IXUSR) == 0)
		debug_print ("Creating new config directory %s\n", dir);
	g_free (dir);
}


static void write_ftp_servers ()
{
	gchar *path;
	FILE *fd;
	
	create_conf_dir ();
	path = g_strdup_printf ("%s/.gnome-commander/ftp-servers", g_get_home_dir());
	fd = fopen (path, "w");
	if (fd != NULL)
	{
		GList *tmp = data->priv->ftp_servers;
		while (tmp)
		{
			GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)tmp->data;
			if (server)
			{
				gchar *alias = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_alias (server));
				gchar *hname = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_host_name (server));
				gchar *uname = gnome_vfs_escape_string (
					gnome_cmd_ftp_server_get_user_name (server));
				gushort port = gnome_cmd_ftp_server_get_host_port (server);
				
				fprintf (fd, "%s %s %d %s\n", alias, hname, port, uname);

				g_free (alias);
				g_free (hname);
				g_free (uname);
			}
			tmp = tmp->next;
		}

		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for writing\n", path);

	g_free (path);
}

/*
static void write_cvs_servers ()
{
	gchar *path;
	FILE *fd;

	create_conf_dir ();
	path = g_strdup_printf ("%s/.gnome-commander/cvs-servers", g_get_home_dir());
	fd = fopen (path, "w");
	if (fd != NULL)
	{
		GList *tmp = data->priv->cvs_servers;
		while (tmp)
		{
			GnomeCmdCvsServer *server = (GnomeCmdCvsServer*)tmp->data;
			if (server)
			{
				gchar *alias = gnome_vfs_escape_string (
					gnome_cmd_cvs_server_get_alias (server));
				gchar *string = gnome_vfs_escape_string (
					gnome_cmd_cvs_server_get_string (server));
				
				fprintf (fd, "%s %s\n", alias, string);

				g_free (alias);
				g_free (string);
			}
			tmp = tmp->next;
		}

		fclose (fd);
	}
	else
		warn_print ("Failed to open the file %s for writing\n", path);

	g_free (path);
}
*/

static void load_ftp_servers ()
{
	gchar *path;
	FILE *fd;

	data->priv->ftp_servers = NULL;
	path = g_strdup_printf ("%s/.gnome-commander/ftp-servers", g_get_home_dir());
	fd = fopen (path, "r");
	if (fd != NULL)
	{
		int ret;
		gchar alias[256], host[256], user[256];
		gchar *alias2, *host2, *user2;
		gint port;
		
		do
		{
			ret = fscanf (fd, "%s %s %d %s\n", alias, host, &port, user);
			
			if (ret == 4)
			{
				alias2 = gnome_vfs_unescape_string (alias, NULL);
				host2  = gnome_vfs_unescape_string (host, NULL);
				user2  = gnome_vfs_unescape_string (user, NULL);
				
				data->priv->ftp_servers = g_list_append (
					data->priv->ftp_servers,
					gnome_cmd_ftp_server_new_with_values (
						alias2, host2, (gshort)port, user2));

				g_free (alias2);
				g_free (host2);
				g_free (user2);
			}
		} while (ret == 4);
	}
	else
		warn_print ("Failed to open the file %s for reading\n", path);

	g_free (path);
}

/*
static void load_cvs_servers ()
{
	gchar *path;
	FILE *fd;

	data->priv->cvs_servers = NULL;
	path = g_strdup_printf ("%s/.gnome-commander/cvs-servers", g_get_home_dir());
	fd = fopen (path, "r");
	if (fd != NULL)
	{
		int ret;
		gchar alias[256], string[256];
		gchar *alias2, *string2;
		
		do
		{
			ret = fscanf (fd, "%s %s\n", alias, string);
			
			if (ret == 2)
			{
				alias2  = gnome_vfs_unescape_string (alias, NULL);
				string2 = gnome_vfs_unescape_string (string, NULL);
				
				data->priv->cvs_servers = g_list_append (
					data->priv->cvs_servers,
					gnome_cmd_cvs_server_new_with_values (
						alias2, string2));

				g_free (alias2);
				g_free (string2);
			}
		} while (ret == 2);
	}
	else
		warn_printerr ("Failed to open the file %s for reading\n", path);

	g_free (path);
}
*/

void
gnome_cmd_data_free                      (void)
{
	if (data)
	{
		if (data->priv)
		{
			/* free the ftpservers */
			GList *tmp = data->priv->ftp_servers;
			while (tmp)
			{
				GnomeCmdFtpServer *server = (GnomeCmdFtpServer*)tmp->data;
				if (server)
					gnome_cmd_ftp_server_free (server);
				else
					g_warning ("GnomeCmdData: NULL ftpserver in list\n");

				tmp = tmp->next;
			}

			/* free the ftpserver list */
			if (data->priv->ftp_servers)
				g_list_free (data->priv->ftp_servers);

			/* free the anonymous password string */
			if (data->priv->ftp_anonymous_password)
				g_free (data->priv->ftp_anonymous_password);

			/* free the dateformat string */
			if (data->priv->date_format)
				g_free (data->priv->date_format);

			/* free the font name string */
			if (data->priv->font)
				g_free (data->priv->font);

			/* free cvsroot string */
			if (data->priv->cvsroot)
				g_free (data->priv->cvsroot);
			
			/* free the external programs strings */
			if (data->priv->viewer)
				g_free (data->priv->viewer);
			if (data->priv->editor)
				g_free (data->priv->editor);
			if (data->priv->differ)
				g_free (data->priv->differ);
			if (data->priv->cvs_differ)
				g_free (data->priv->cvs_differ);
			if (data->priv->term)
				g_free (data->priv->term);

			/* free cvs colors */
			g_free (data->priv->cvs_update_colors->modified);
			g_free (data->priv->cvs_update_colors->updated);
			g_free (data->priv->cvs_update_colors->unknown);
			g_free (data->priv->cvs_update_colors->added);
			g_free (data->priv->cvs_update_colors->removed);
			g_free (data->priv->cvs_update_colors->conflict);
			g_free (data->priv->cvs_update_colors);

			/* free layout colors */
			g_free (data->priv->layout_colors->norm_fg);
			g_free (data->priv->layout_colors->norm_bg);
			g_free (data->priv->layout_colors->sel_fg);
			g_free (data->priv->layout_colors->sel_bg);
			g_free (data->priv->layout_colors->curs_fg);
			g_free (data->priv->layout_colors->curs_bg);
			g_free (data->priv->layout_colors->term_fg);
			g_free (data->priv->layout_colors->term_bg);
			g_free (data->priv->layout_colors);
			
			g_free (data->priv);
		}

		g_free (data);
	}
}


void
gnome_cmd_data_save                      (void)
{
	set_string ("/gnome-commander/ftp/anonymous_password",
				data->priv->ftp_anonymous_password);
	set_int    ("/gnome-commander/options/size_disp_mode",
				data->priv->size_disp_mode);
	set_string ("/gnome-commander/options/date_disp_mode",
				data->priv->date_format);
	set_int    ("/gnome-commander/options/layout",
				data->priv->layout);

	set_bool   ("/gnome-commander/options/show_unknown",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_UNKNOWN]);
	set_bool   ("/gnome-commander/options/show_regular",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_REGULAR]);
	set_bool   ("/gnome-commander/options/show_directory",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_DIRECTORY]);
	set_bool   ("/gnome-commander/options/show_fifo",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_FIFO]);
	set_bool   ("/gnome-commander/options/show_socket",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_SOCKET]);
	set_bool   ("/gnome-commander/options/show_char_device",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE]);
	set_bool   ("/gnome-commander/options/show_block_device",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_BLOCK_DEVICE]);
	set_bool   ("/gnome-commander/options/show_symbolic_link",
				data->priv->type_filter[GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK]);

	set_bool   ("/gnome-commander/options/hidden_filter",
				data->priv->hidden_filter);

	set_bool   ("/gnome-commander/sort/case_sensitive",
				data->priv->case_sens_sort);

	set_color  ("/gnome-commander/colors/cvs_update_modified",
				data->priv->cvs_update_colors->modified);
	set_color  ("/gnome-commander/colors/cvs_update_updated",
				data->priv->cvs_update_colors->updated);
	set_color  ("/gnome-commander/colors/cvs_update_unknown",
				data->priv->cvs_update_colors->unknown);
	set_color  ("/gnome-commander/colors/cvs_update_added",
				data->priv->cvs_update_colors->added);
	set_color  ("/gnome-commander/colors/cvs_update_removed",
				data->priv->cvs_update_colors->removed);
	set_color  ("/gnome-commander/colors/cvs_update_conflict",
				data->priv->cvs_update_colors->conflict);
	
	set_color  ("/gnome-commander/colors/norm_fg",
				data->priv->layout_colors->norm_fg);
	set_color  ("/gnome-commander/colors/norm_bg",
				data->priv->layout_colors->norm_bg);
	set_color  ("/gnome-commander/colors/sel_fg",
				data->priv->layout_colors->sel_fg);
	set_color  ("/gnome-commander/colors/sel_bg",
				data->priv->layout_colors->sel_bg);
	set_color  ("/gnome-commander/colors/curs_fg",
				data->priv->layout_colors->curs_fg);
	set_color  ("/gnome-commander/colors/curs_bg",
				data->priv->layout_colors->curs_bg);
	set_color  ("/gnome-commander/colors/term_fg",
				data->priv->layout_colors->term_fg);
	set_color  ("/gnome-commander/colors/term_bg",
				data->priv->layout_colors->term_bg);

	set_string ("/gnome-commander/options/font",
				data->priv->font);

	set_string ("/gnome-commander/cvs/cvsroot",
				data->priv->cvsroot);
	set_bool   ("/gnome-commander/cvs/cvs_over_ssh",
				data->priv->cvs_over_ssh);
	set_int    ("/gnome-commander/options/right_mouse_button_mode",
				data->priv->right_mouse_button_mode);
	set_int    ("/gnome-commander/options/embterm_height",
				data->priv->embterm_height);
	set_bool   ("/gnome-commander/options/show_toolbar",
				data->priv->show_toolbar);
	set_bool   ("/gnome-commander/options/use_mime",
				data->priv->use_mime);
	set_int    ("/gnome-commander/options/mime_icon_size",
				data->priv->mime_icon_size);
	set_int    ("/gnome-commander/options/mime_scale_quality",
				data->priv->mime_scale_quality);
	set_string ("/gnome-commander/options/mime_icon_dir",
				data->priv->mime_icon_dir);
	
	set_string ("/gnome-commander/programs/viewer",
				data->priv->viewer);
	set_string ("/gnome-commander/programs/editor",
				data->priv->editor);
	set_string ("/gnome-commander/programs/differ",
				data->priv->differ);
	set_string ("/gnome-commander/programs/cvs_differ",
				data->priv->cvs_differ);
	set_string ("/gnome-commander/programs/term",
				data->priv->term);


	write_ftp_servers ();
//	write_cvs_servers ();
	
	gnome_config_sync ();
}


void
gnome_cmd_data_load                      (void)
{
	data = g_new (GnomeCmdData, 1);
	data->priv = g_new (GnomeCmdDataPrivate, 1);

	data->priv->cvs_update_colors = g_new (GnomeCmdCvsUpdateColors, 1);
	data->priv->cvs_update_colors->modified = gdk_color_new (0,0xffff,0);
	data->priv->cvs_update_colors->updated = gdk_color_new (0,0,0xffff);
	data->priv->cvs_update_colors->unknown = gdk_color_new (0xaaaa,0xaaaa,0xaaaa);
	data->priv->cvs_update_colors->added = gdk_color_new (0xffff,0xffff,0);
	data->priv->cvs_update_colors->removed = gdk_color_new (0xffff,0,0xffff);
	data->priv->cvs_update_colors->conflict = gdk_color_new (0xffff,0,0);

	data->priv->layout_colors = g_new (GnomeCmdLayoutColors, 1);
	data->priv->layout_colors->norm_fg = gdk_color_new (0xffff,0xffff,0xffff);
	data->priv->layout_colors->norm_bg = gdk_color_new (0,0,0x6666);
	data->priv->layout_colors->sel_fg = gdk_color_new (0xffff,0xffff,0);
	data->priv->layout_colors->sel_bg = gdk_color_new (0,0,0x6666);
	data->priv->layout_colors->curs_fg = gdk_color_new (0,0,0);
	data->priv->layout_colors->curs_bg = gdk_color_new (0xcccc,0xcccc,0xcccc);
	data->priv->layout_colors->term_fg = gdk_color_new (0xcccc,0xcccc,0xcccc);
	data->priv->layout_colors->term_bg = gdk_color_new (0,0,0x6666);

	data->priv->ftp_anonymous_password = 
		get_string ("/gnome-commander/ftp/anonymous_password", "you@provider.com");

	data->priv->size_disp_mode =
		get_int ("/gnome-commander/options/size_disp_mode",
				 GNOME_CMD_SIZE_DISP_MODE_GROUPED);

	data->priv->date_format = 
		get_string ("/gnome-commander/options/date_disp_mode", "%H:%M %d/%m %Y");

	data->priv->layout = 
		get_int ("/gnome-commander/options/layout", GNOME_CMD_LAYOUT_ICONS);


	data->priv->type_filter[GNOME_VFS_FILE_TYPE_UNKNOWN] =
		get_bool ("/gnome-commander/options/show_unknown", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_REGULAR] = 
		get_bool ("/gnome-commander/options/show_regular", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_DIRECTORY] = 
		get_bool ("/gnome-commander/options/show_directory", FALSE);

	data->priv->type_filter[GNOME_VFS_FILE_TYPE_FIFO] = 
		get_bool ("/gnome-commander/options/show_fifo", FALSE);
	
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_SOCKET] =
		get_bool ("/gnome-commander/options/show_socket", FALSE);
							 
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE] =
		get_bool ("/gnome-commander/options/show_char_device", FALSE);
							 
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_BLOCK_DEVICE] =
		get_bool ("/gnome-commander/options/show_block_device", FALSE);
	
	data->priv->type_filter[GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK] =
		get_bool ("/gnome-commander/options/show_symbolic_link", FALSE);

	data->priv->hidden_filter =
		get_bool ("/gnome-commander/options/hidden_filter", TRUE);

	data->priv->case_sens_sort =
		get_bool ("/gnome-commander/sort/case_sensitive", TRUE);

	data->priv->main_win_width =
		get_int ("/gnome-commander-size/main_win/width", 600);
	
	data->priv->main_win_height =
		get_int ("/gnome-commander-size/main_win/height", 400);

	get_color ("/gnome-commander/colors/cvs_update_modified",
			   data->priv->cvs_update_colors->modified);
	get_color ("/gnome-commander/colors/cvs_update_updated",
			   data->priv->cvs_update_colors->updated);
	get_color ("/gnome-commander/colors/cvs_update_unknown",
			   data->priv->cvs_update_colors->unknown);
	get_color ("/gnome-commander/colors/cvs_update_added",
			   data->priv->cvs_update_colors->added);
	get_color ("/gnome-commander/colors/cvs_update_removed",
			   data->priv->cvs_update_colors->removed);
	get_color ("/gnome-commander/colors/cvs_update_conflict",
			   data->priv->cvs_update_colors->conflict);

	get_color  ("/gnome-commander/colors/norm_fg",
				data->priv->layout_colors->norm_fg);
	get_color  ("/gnome-commander/colors/norm_bg",
				data->priv->layout_colors->norm_bg);
	get_color  ("/gnome-commander/colors/sel_fg",
				data->priv->layout_colors->sel_fg);
	get_color  ("/gnome-commander/colors/sel_bg",
				data->priv->layout_colors->sel_bg);
	get_color  ("/gnome-commander/colors/curs_fg",
				data->priv->layout_colors->curs_fg);
	get_color  ("/gnome-commander/colors/curs_bg",
				data->priv->layout_colors->curs_bg);
	get_color  ("/gnome-commander/colors/term_fg",
				data->priv->layout_colors->term_fg);
	get_color  ("/gnome-commander/colors/term_bg",
				data->priv->layout_colors->term_bg);

	data->priv->font = get_string (
		"/gnome-commander/options/font",
		"-adobe-helvetica-medium-r-normal-*-10-*-*-*-p-*-iso8859-1");

	data->priv->cvsroot = get_string ("/gnome-commander/cvs/cvsroot",
									  "");
	data->priv->cvs_over_ssh = get_bool ("/gnome-commander/cvs/cvs_over_ssh",
										 FALSE);
	data->priv->right_mouse_button_mode = get_int (
		"/gnome-commander/options/right_mouse_button_mode",
		RIGHT_BUTTON_POPUPS_MENU);
	data->priv->embterm_height = get_int (
		"/gnome-commander/options/embterm_height", 1);
	data->priv->show_toolbar = get_bool (
		"/gnome-commander/options/show_toolbar", TRUE);
	data->priv->use_mime = get_bool (
		"/gnome-commander/options/use_mime", TRUE);
	data->priv->mime_icon_size = get_int (
		"/gnome-commander/options/mime_icon_size", 16);
	data->priv->mime_scale_quality = get_int (
		"/gnome-commander/options/mime_scale_quality", 3);
	data->priv->mime_icon_dir = get_string (
		"/gnome-commander/options/mime_icon_dir", "/usr/share/pixmaps/nautilus");
	
	data->priv->viewer = get_string ("/gnome-commander/programs/viewer",
									 "emacs %s");
	data->priv->editor = get_string ("/gnome-commander/programs/editor",
									 "emacs %s");
	data->priv->differ = get_string ("/gnome-commander/programs/differ",
									 "gtk-diff %s %s");
	data->priv->cvs_differ = get_string (
		"/gnome-commander/programs/cvs_differ",
		"gtk-cvsdiff %s");
	data->priv->term = get_string (
		"/gnome-commander/programs/term",
		"gnome-terminal --command \"%s\"");


	load_ftp_servers ();
//	load_cvs_servers ();
}


void
gnome_cmd_data_add_ftp_server            (GnomeCmdFtpServer *server)
{
	if (server == NULL)
	{
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->ftp_servers = g_list_append (data->priv->ftp_servers, server);
}


void
gnome_cmd_data_remove_ftp_server         (GnomeCmdFtpServer *server)
{
	if (server == NULL)
	{
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->ftp_servers = g_list_remove (data->priv->ftp_servers, server);
}


GList*
gnome_cmd_data_get_ftp_servers           (void)
{
	return data->priv->ftp_servers;
}


void
gnome_cmd_data_set_ftp_servers           (GList *servers)
{
	data->priv->ftp_servers = servers;
}


const gchar *
gnome_cmd_data_get_ftp_anonymous_password (void)
{
	return data->priv->ftp_anonymous_password;
}


void
gnome_cmd_data_set_ftp_anonymous_password (const gchar *pw)
{
	if (data->priv->ftp_anonymous_password)
		g_free (data->priv->ftp_anonymous_password);

	data->priv->ftp_anonymous_password = g_strdup (pw);
}


GnomeCmdSizeDispMode
gnome_cmd_data_get_size_disp_mode        (void)
{
	return data->priv->size_disp_mode;
}


void
gnome_cmd_data_set_size_disp_mode        (GnomeCmdSizeDispMode mode)
{
	data->priv->size_disp_mode = mode;
}

GnomeCmdDateFormat
gnome_cmd_data_get_date_format           (void)
{
	return data->priv->date_format;
}

void
gnome_cmd_data_set_date_format           (GnomeCmdDateFormat format)
{
	if (data->priv->date_format)
		g_free (data->priv->date_format);

	data->priv->date_format = g_strdup (format);
}



GnomeCmdLayout
gnome_cmd_data_get_layout                (void)
{
	return data->priv->layout;
}


void
gnome_cmd_data_set_layout                (GnomeCmdLayout layout)
{
	data->priv->layout = layout;
}


GnomeCmdCvsUpdateColors*
gnome_cmd_data_get_cvs_update_colors    (void)
{
	return data->priv->cvs_update_colors;
}


GnomeCmdLayoutColors*
gnome_cmd_data_get_layout_colors    (void)
{
	return data->priv->layout_colors;
}


void
gnome_cmd_data_set_type_filter          (GnomeVFSFileType type, gboolean value)
{
	data->priv->type_filter[type] = value;
}


gboolean
gnome_cmd_data_get_type_filter          (GnomeVFSFileType type)
{
	return data->priv->type_filter[type];
}


void
gnome_cmd_data_set_hidden_filter        (gboolean value)
{
	data->priv->hidden_filter = value;
}


gboolean
gnome_cmd_data_get_hidden_filter        (void)
{
	return data->priv->hidden_filter;
}

/*
void
gnome_cmd_data_add_cvs_server           (GnomeCmdCvsServer *cvs_server)
{
	if (cvs_server == NULL)
	{
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->cvs_servers = g_list_append (data->priv->cvs_servers, cvs_server);
}


void
gnome_cmd_data_remove_cvs_server        (GnomeCmdCvsServer *cvs_server)
{
	if (cvs_server == NULL)
	{
		g_warning ("GnomeCmdData: server != NULL test failed\n");
		return;
	}

	data->priv->cvs_servers = g_list_remove (data->priv->cvs_servers, cvs_server);
}


GList*
gnome_cmd_data_get_cvs_servers          (void)
{
	return data->priv->cvs_servers;
}
*/

void
gnome_cmd_data_set_position             (gint width, gint height)
{
	data->priv->main_win_width = width;
	data->priv->main_win_height = height;
}


void
gnome_cmd_data_get_position             (gint *width, gint *height)
{
	*width = data->priv->main_win_width;
	*height = data->priv->main_win_height;
}


void
gnome_cmd_data_save_position            (void)
{
	set_int ("/gnome-commander-size/main_win/width", data->priv->main_win_width);
	set_int ("/gnome-commander-size/main_win/height", data->priv->main_win_height);
}


void
gnome_cmd_data_set_viewer               (const gchar *command)
{
	if (data->priv->viewer)
		g_free (data->priv->viewer);
	data->priv->viewer = g_strdup (command);
}


void
gnome_cmd_data_set_editor               (const gchar *command)
{
	if (data->priv->editor)
		g_free (data->priv->editor);
	data->priv->editor = g_strdup (command);
}


void
gnome_cmd_data_set_differ               (const gchar *command)
{
	if (data->priv->differ)
		g_free (data->priv->differ);
	data->priv->differ = g_strdup (command);
}


void
gnome_cmd_data_set_cvs_differ           (const gchar *command)
{
	if (data->priv->cvs_differ)
		g_free (data->priv->cvs_differ);
	data->priv->cvs_differ = g_strdup (command);
}


void
gnome_cmd_data_set_term                (const gchar *term)
{
	if (data->priv->term)
		g_free (data->priv->term);
	
	data->priv->term = g_strdup (term);
}


const gchar *
gnome_cmd_data_get_viewer               (void)
{
	return data->priv->viewer;
}


const gchar *
gnome_cmd_data_get_editor               (void)
{
	return data->priv->editor;
}


const gchar *
gnome_cmd_data_get_differ               (void)
{
	return data->priv->differ;
}


const gchar *
gnome_cmd_data_get_cvs_differ           (void)
{
	return data->priv->cvs_differ;
}


const gchar *
gnome_cmd_data_get_term                (void)
{
	return data->priv->term;
}


gboolean
gnome_cmd_data_get_case_sens_sort       (void)
{
	return data->priv->case_sens_sort;
}


void
gnome_cmd_data_set_case_sens_sort       (gboolean value)
{
	data->priv->case_sens_sort = value;
}


const gchar *gnome_cmd_data_get_font          (void)
{
	return data->priv->font;
}


void gnome_cmd_data_set_font            (const gchar *font)
{
	if (data->priv->font != NULL)
		g_free (data->priv->font);
	data->priv->font = g_strdup (font);
}


void
gnome_cmd_data_set_cvsroot              (const gchar *cvsroot)
{
	if (data->priv->cvsroot != NULL)
		g_free (data->priv->cvsroot);
	data->priv->cvsroot = g_strdup (cvsroot);
}


void
gnome_cmd_data_set_cvs_over_ssh         (gboolean value)
{
	data->priv->cvs_over_ssh = value;
}

const gchar *
gnome_cmd_data_get_cvsroot              (void)
{
	return data->priv->cvsroot;
}

gboolean
gnome_cmd_data_get_cvs_over_ssh         (void)
{
	return data->priv->cvs_over_ssh;
}


void
gnome_cmd_data_set_right_mouse_button_mode (GnomeCmdRightMouseButtonMode mode)
{
	data->priv->right_mouse_button_mode = mode;
}


GnomeCmdRightMouseButtonMode
gnome_cmd_data_get_right_mouse_button_mode (void)
{
	return data->priv->right_mouse_button_mode;
}


gint
gnome_cmd_data_get_embterm_height      (void)
{
	return data->priv->embterm_height;
}


void
gnome_cmd_data_set_embterm_height      (gint height)
{
	data->priv->embterm_height = height;
}


void
gnome_cmd_data_set_show_toolbar        (gboolean value)
{
	data->priv->show_toolbar = value;
}


gboolean
gnome_cmd_data_get_show_toolbar        (void)
{
	return data->priv->show_toolbar;
}


gboolean
gnome_cmd_data_get_use_mime            (void)
{
	return data->priv->use_mime;
}


void
gnome_cmd_data_set_use_mime            (gboolean value)
{
	data->priv->use_mime = value;
}


guint
gnome_cmd_data_get_mime_icon_size      (void)
{
	return data->priv->mime_icon_size;
}


void
gnome_cmd_data_set_mime_icon_size      (guint size)
{
	data->priv->mime_icon_size = size;
}


GdkInterpType
gnome_cmd_data_get_mime_scale_quality  (void)
{
	return data->priv->mime_scale_quality;
}


void
gnome_cmd_data_set_mime_scale_quality  (GdkInterpType quality)
{
	data->priv->mime_scale_quality = quality;
}


const gchar *
gnome_cmd_data_get_mime_icon_dir       (void)
{
	return data->priv->mime_icon_dir;
}


void
gnome_cmd_data_set_mime_icon_dir       (const gchar *dir)
{
	if (data->priv->mime_icon_dir)
		g_free (data->priv->mime_icon_dir);
	data->priv->mime_icon_dir = g_strdup (dir);
}

