/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include "gnome-cmd-includes.h"

#define MAX_OPEN_WITH_APPS 20

typedef struct
{
	GList *files;
	GnomeVFSMimeApplication *app;
} OpenWithData;


static GtkMenuClass *parent_class = NULL;

struct _GnomeCmdFilePopmenuPrivate
{
	GList *data_list;
};


static void
exec_with_app (GList *files, GnomeVFSMimeApplication *app)
{
	mime_exec_multiple (files, app);
}


/* Used by exec_default for each list of files that shares the same default application
 * This is a hash-table callback function
 */
static void
htcb_exec_with_app (const gchar* key,
					OpenWithData *data,
					gpointer user_data)
{
	exec_with_app (data->files, data->app);
	g_list_free (data->files);
	gnome_vfs_mime_application_free (data->app);
}


/* Executes a list of files with the same application
 *
 */
static void
cb_exec_with_app (GtkMenuItem *menu_item,
				  OpenWithData *data)
{
	 exec_with_app (data->files, data->app);
}


/* Iterates through all files and gets their default application.
 * All files with the same default app are grouped together and opened in one call.
 */
static void
cb_exec_default (GtkMenuItem *menu_item,
				 GList *files)
{
	GHashTable *hash;

	if (!gnome_cmd_data_get_use_mime ()) {
		gnome_error_dialog (_("You need to enable MIME support in the options dialog to use this"));
		return;
	}

	hash = g_hash_table_new (g_str_hash, g_str_equal);
	while (files) {
		GnomeCmdFile *finfo = (GnomeCmdFile*)files->data;
		GnomeVFSMimeApplication *app =
			gnome_vfs_mime_get_default_application (finfo->info->mime_type);
		if (app) {
			OpenWithData *data = (OpenWithData*)g_hash_table_lookup (hash, app->id);
			if (!data) {
				data = g_new (OpenWithData, 1);
				data->app = app;
				data->files = NULL;
				g_hash_table_insert (hash, app->id, data);
			}
			else
				gnome_vfs_mime_application_free (app);			
			
			data->files = g_list_append (data->files, finfo);
		}
		
		files = files->next;
	}

	g_hash_table_foreach (hash, (GHFunc)htcb_exec_with_app, NULL);
	g_hash_table_destroy (hash);
}


/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	OpenWithData *data;
	GList *tmp;
	GnomeCmdFilePopmenu *menu = GNOME_CMD_FILE_POPMENU (object);

	tmp = menu->priv->data_list;
	while (tmp) {
		data = (OpenWithData*)tmp->data;
		gnome_vfs_mime_application_free (data->app);
		g_free (data);
		tmp = tmp->next;
	}
	g_list_free (menu->priv->data_list);
	
	g_free (menu->priv);
	
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdFilePopmenuClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gtk_menu_get_type ());
	object_class->destroy = destroy;
	widget_class->map = map;
}


static void
init (GnomeCmdFilePopmenu *menu)
{
	menu->priv = g_new (GnomeCmdFilePopmenuPrivate, 1);
}




/***********************************
 * Public functions
 ***********************************/

GtkWidget*
gnome_cmd_file_popmenu_new (GList *files)
{	
	int i;
	GList *apps, *apps_copy;
	GnomeCmdFilePopmenu *menu;
	GnomeCmdFile *finfo;
	
	static GnomeUIInfo apps_uiinfo[MAX_OPEN_WITH_APPS];
	
	static GnomeUIInfo popmenu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("_Open"),
			NULL,
			cb_exec_default, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_SUBTREE, N_("_Open with..."),
			NULL,
			apps_uiinfo, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("_Cut"),
			NULL,
			file_cap_cut, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_CUT,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Copy"),
			NULL,
			file_cap_copy, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_COPY,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Paste"),
			NULL,
			file_cap_paste, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PASTE,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Delete"),
			NULL,
			file_delete, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("_View"),
			NULL,
			file_view, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Edit"),
			NULL,
			file_edit, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Chmod"),
			NULL,
			file_chmod, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Chown"),
			NULL,
			file_chown, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Properties"),
			NULL,
			file_properties, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};

	g_return_val_if_fail (files != NULL, NULL);

	menu = gtk_type_new (gnome_cmd_file_popmenu_get_type ());
	
	finfo = (GnomeCmdFile*)files->data;


	/* Fill the "Open with..." menu with applications
	 *
	 */
	i = -1;
	menu->priv->data_list = NULL;

	if (gnome_cmd_data_get_use_mime ()) {
		apps_copy = apps = gnome_vfs_mime_get_short_list_applications (
			finfo->info->mime_type);
		while (apps) {
			GnomeVFSMimeApplication *app = (GnomeVFSMimeApplication*)apps->data;

			if (app) {
				/* Don't add programs that can only handle one file when multiple files
				 * are selected.
				 */ 
				if (!(g_list_length (files) > 1 && !app->can_open_multiple_files)) {
					OpenWithData *data = g_new (OpenWithData, 1);
				
					i++;
					apps_uiinfo[i].type = GNOME_APP_UI_ITEM;
					apps_uiinfo[i].label = app->name;
					apps_uiinfo[i].moreinfo = cb_exec_with_app;

					data->files = files;
					data->app = app;
					apps_uiinfo[i].user_data = data;

					menu->priv->data_list = g_list_append (
						menu->priv->data_list, data);
				}
			}

			if (i >= MAX_OPEN_WITH_APPS)
				break;
		
			apps = apps->next;
		}
		g_list_free (apps_copy);
	}

	apps_uiinfo[i+1].type = GNOME_APP_UI_ENDOFINFO;

	
	/* Set default callback data 
	 */
	i = 0;
	while (popmenu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (popmenu_uiinfo[i].type == GNOME_APP_UI_ITEM) {
			popmenu_uiinfo[i].user_data = main_win;
		}
		i++;
	}

	
	/* Set the callback data of the "Open" menu-item
	 */
	popmenu_uiinfo[0].user_data = files;

	
	/* Fill the menu
	 */
	gnome_app_fill_menu (GTK_MENU_SHELL (menu), popmenu_uiinfo,
						 NULL, FALSE, 0);

	return GTK_WIDGET (menu);
}



GtkType
gnome_cmd_file_popmenu_get_type         (void)
{
	static GtkType dlg_type = 0;

	if (dlg_type == 0)
	{
		GtkTypeInfo dlg_info =
		{
			"GnomeCmdFilePopmenu",
			sizeof (GnomeCmdFilePopmenu),
			sizeof (GnomeCmdFilePopmenuClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		dlg_type = gtk_type_unique (gtk_menu_get_type (), &dlg_info);
	}
	return dlg_type;
}

