/*
    GNOME Commander - A GNOME based file manager 
    Copyright (C) 2001-2002 Marcus Bjurman

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/ 
#include <config.h>
#include <pthread.h>
#include <sys/wait.h>
#include <sys/types.h>
#include <unistd.h>
#include <zvt/zvtterm.h>
#include "gnome-cmd-includes.h"

#include "../pixmaps/menu_ftp_connect.xpm"
#include "../pixmaps/menu_ftp_disconnect.xpm"

#define LEFT 0
#define RIGHT 1

struct _GnomeCmdMainWinPrivate
{
	gboolean current_fs; // either LEFT or RIGHT
	
	GtkWidget *main_win;
	GtkWidget *vbox;
	GtkWidget *menubar;
	GtkWidget *hpaned;
	GtkWidget *vpaned;
	GtkWidget *buttonbox;
	GtkWidget *file_selector[2];
	GtkAccelGroup *accel_group;

	GtkWidget *zvt_term;

	GtkWidget *help_btn;
	GtkWidget *run_btn;
	GtkWidget *view_btn;
	GtkWidget *edit_btn;
	GtkWidget *copy_btn;
	GtkWidget *move_btn;
	GtkWidget *mkdir_btn;
	GtkWidget *delete_btn;
	GtkWidget *find_btn;
	GtkWidget *quit_btn;

	GtkWidget *toolbar;
};

static GnomeAppClass *parent_class = NULL;



/*****************************************************************************
	Buttonbar callbacks
*****************************************************************************/

static void
on_help_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	help_help (NULL, mw);
}


static void
on_run_clicked                        (GtkButton       *button,
									   GnomeCmdMainWin *mw)
{
	GnomeCmdDir *dir = gnome_cmd_file_selector_get_directory (
		gnome_cmd_main_win_get_active_fs (mw));
	GtkWidget *dialog = gnome_cmd_run_dialog_new (dir);
	gtk_widget_ref (dialog);
	gtk_object_set_data_full (GTK_OBJECT (mw),
							  "gnome_cmd_run_dialog", dialog,
							  (GtkDestroyNotify)gtk_widget_unref);
	gtk_widget_show (dialog);
}


static void
on_view_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_view (NULL, mw);
}


static void
on_edit_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_edit (NULL, mw);
}


static void
on_copy_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_copy (NULL, mw);
}


static void
on_move_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_move (NULL, mw);
}


static void
on_mkdir_clicked                       (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_mkdir (NULL, mw);
}


static void
on_delete_clicked                      (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	file_delete (NULL, mw);
}


static void
on_search_clicked                      (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
	edit_search (NULL, mw);
}


static void
on_quit_clicked                        (GtkButton       *button,
                                        GnomeCmdMainWin *mw)
{
    gtk_main_quit ();
}


static void
view_show_embterm                       (GtkMenuItem     *menuitem,
										 GnomeCmdMainWin *mw)
{
	gnome_cmd_main_win_show_embterm (mw);
}


static void
view_max_term_size                      (GtkMenuItem     *menuitem,
										 GnomeCmdMainWin *mw)
{
	gtk_paned_set_position (GTK_PANED (mw->priv->vpaned), 0);
	gnome_cmd_zvt_term_set_focus (GNOME_CMD_ZVT_TERM (mw->priv->zvt_term));
}


/*****************************************************************************
	Misc widgets callbacks
*****************************************************************************/

static void
on_main_win_realize                    (GtkWidget       *widget,
                                        GnomeCmdMainWin *mw)
{
	IMAGE_init (widget);

	gnome_cmd_file_selector_set_active (
		gnome_cmd_main_win_get_left_fs (mw), TRUE);
	gdk_window_set_icon (GTK_WIDGET (main_win)->window, NULL,
						 IMAGE_get_pixmap (PIXMAP_LOGO),
						 IMAGE_get_mask (PIXMAP_LOGO));
}


static void
on_main_win_size_allocate              (GtkWidget       *widget,
                                        GtkAllocation   *allocation,
                                        GnomeCmdMainWin *mw)
{
	gtk_paned_set_position (GTK_PANED (mw->priv->hpaned), allocation->width/2 - 5);
}


static void
on_left_fs_select                      (GtkCList *list,
										GdkEventButton *event,
                                        GnomeCmdMainWin *mw)
{
		mw->priv->current_fs = LEFT;
		
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT]), FALSE);
}



static void
on_right_fs_select                     (GtkCList *list,
										GdkEventButton *event,
                                        GnomeCmdMainWin *mw)
{
		mw->priv->current_fs = RIGHT;
		
		gnome_cmd_file_selector_set_active (
			GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT]), FALSE);
}



static void
on_fs_keypressed                    (GtkWidget *list,
									 GdkEventKey *event,
									 GnomeCmdMainWin *mw)
{
	switch (event->keyval)
	{
		case GDK_Tab:
			/* hack to avoid the deafult handling of the tab-key */
			event->keyval = GDK_VoidSymbol;
			mw->priv->current_fs = !mw->priv->current_fs;
			gnome_cmd_file_selector_set_active (
				gnome_cmd_main_win_get_active_fs (mw), TRUE);
			gnome_cmd_file_selector_set_active (
				gnome_cmd_main_win_get_inactive_fs (mw), FALSE);
			break;
		case GDK_1:
			if (event->state & GDK_MOD1_MASK)
			{
				gboolean ret;
				GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_left_fs (mw);
 				popup_list (GNOME_CMD_COMBO (fs->con_combo), fs->list);
				gtk_widget_grab_focus (GNOME_CMD_COMBO (fs->con_combo)->list);
				gtk_signal_emit_by_name (
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					"button-press-event",
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					&ret);
				gtk_signal_emit_by_name (
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					"button-release-event",
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					&ret);
			}
			break;
		case GDK_2:
			if (event->state & GDK_MOD1_MASK)
			{
				gboolean ret;
				GnomeCmdFileSelector *fs = gnome_cmd_main_win_get_right_fs (mw);
 				popup_list (GNOME_CMD_COMBO (fs->con_combo), fs->list);
				gtk_widget_grab_focus (GNOME_CMD_COMBO (fs->con_combo)->list);
				gtk_signal_emit_by_name (
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					"button-press-event",
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					&ret);
				gtk_signal_emit_by_name (
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					"button-release-event",
					GTK_OBJECT (GNOME_CMD_COMBO (fs->con_combo)->button),
					&ret);
			}
			break;
	}
}


static void
on_fs_list_resize_column            (GtkCList        *clist,
                                     gint             column,
                                     gint             width,
                                     GtkCList        *other_clist)
{
	static gboolean column_resize_lock = FALSE;

    /* the lock is used so that we dont get into the situation where
       the left list triggers the right witch triggers the left ... */
    if (!column_resize_lock)
    {
		column_resize_lock = TRUE;
		gtk_clist_set_column_width (other_clist, column, width);
		column_resize_lock = FALSE;
    }
}


static void
on_size_allocate                  (GtkWidget *widget,
								   GtkAllocation *allocation,
								   gpointer user_data)
{
	gnome_cmd_data_set_position (allocation->width, allocation->height);
}

/*******************************************************************
	Menus
********************************************************************/




/*******************************
 * Gtk class implementation
 *******************************/

static void
destroy (GtkObject *object)
{
	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(*GTK_OBJECT_CLASS (parent_class)->destroy) (object);

	gtk_main_quit ();
}


static void
map (GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS (parent_class)->map != NULL)
		GTK_WIDGET_CLASS (parent_class)->map (widget);
}


static void
class_init (GnomeCmdMainWinClass *class)
{
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	object_class = GTK_OBJECT_CLASS (class);
	widget_class = GTK_WIDGET_CLASS (class);

	parent_class = gtk_type_class (gnome_app_get_type ());

	object_class->destroy = destroy;

	widget_class->map = map;
}


static void
init (GnomeCmdMainWin *mw)
{
	gint main_win_width, main_win_height;
	
	static GnomeUIInfo file_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Run"),
			NULL,
			on_run_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Create directory"),
			NULL,
			on_mkdir_clicked, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("View"),
			NULL,
			file_view, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Edit"),
			NULL,
			file_edit, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Diff"),
			NULL,
			file_diff, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Chown/Chgrp"),
			NULL,
			file_chown, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Chmod"),
			NULL,
			file_chmod, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Properties..."),
			NULL,
			file_properties, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_P, GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Quit"),
			NULL,
			file_exit, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_QUIT,
			GDK_Q, GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_END
	};

	static GnomeUIInfo mark_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Toggle"),
			NULL,
			mark_toggle, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Toggle and step down"),
			NULL,
			mark_toggle_and_step, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_Insert, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Select all"),
			NULL,
			mark_select_all, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_A, GDK_CONTROL_MASK, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Unselect all"),
			NULL,
			mark_unselect_all, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_U, GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_END
	};
	
	//static GnomeUIInfo commands_menu_uiinfo[] =
	//{
	//};

	static GnomeUIInfo edit_menu_uiinfo[] =
	{
		GNOMEUIINFO_MENU_CUT_ITEM (file_cap_cut, NULL),
		GNOMEUIINFO_MENU_COPY_ITEM (file_cap_copy, NULL),
		GNOMEUIINFO_MENU_PASTE_ITEM (file_cap_paste, NULL),
		{
			GNOME_APP_UI_ITEM, N_("Delete"),
			NULL,
			file_delete, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_Delete, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Search..."),
			NULL,
			edit_search, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SEARCH,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};

	static GnomeUIInfo view_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Maximize terminal size"),
			NULL,
			view_max_term_size, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_M, GDK_CONTROL_MASK, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Show embedded terminal"),
			NULL,
			view_show_embterm, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			GDK_T, GDK_CONTROL_MASK, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Refresh"),
			NULL,
			view_refresh, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_REFRESH,
			GDK_R, GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_END
	};
	
	static GnomeUIInfo options_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Options..."),
			NULL,
			options_edit, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
			GDK_O, GDK_CONTROL_MASK, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Save options"),
			NULL,
			options_save, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Save position"),
			NULL,
			options_save_position, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_SAVE,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};
/*
	static GnomeUIInfo cvs_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Update"),
			NULL,
			cvs_update, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_update.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Commit..."),
			NULL,
			cvs_commit, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_commit.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Diff"),
			NULL,
			cvs_diff, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_diff.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Log"),
			NULL,
			cvs_log, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_log.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Add"),
			NULL,
			cvs_add, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_add.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Remove"),
			NULL,
			cvs_remove, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_remove.xpm",
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Checkout..."),
			NULL,
			cvs_checkout, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_checkout.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("_Import..."),
			NULL,
			cvs_import, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_import.xpm",
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Login..."),
			NULL,
			cvs_login, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_login.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Logout"),
			NULL,
			cvs_logout, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME, "gnome-commander/menu_cvs_logout.xpm",
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Options..."),
			NULL,
			cvs_options, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};
*/
	static GnomeUIInfo ftp_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Connect..."),
			NULL,
			ftp_connect, NULL, NULL,
			GNOME_APP_PIXMAP_DATA, menu_ftp_connect_xpm,
			GDK_F, GDK_CONTROL_MASK, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_("Disconnect"),
			NULL,
			ftp_disconnect, NULL, NULL,
			GNOME_APP_PIXMAP_DATA, menu_ftp_disconnect_xpm,
			GDK_D, GDK_CONTROL_MASK, NULL
		},
		/*
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, N_("Options..."),
			NULL,
			ftp_options, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_PROP,
			0, 0, NULL
		},
		*/
		GNOMEUIINFO_END
	};

	static GnomeUIInfo help_menu_uiinfo[] =
	{
		{
			GNOME_APP_UI_ITEM, N_("Help..."),
			NULL,
			help_help, NULL, NULL,
			GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_MENU_BOOK_RED,
			GDK_H, GDK_CONTROL_MASK, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		GNOMEUIINFO_MENU_ABOUT_ITEM (help_about, NULL),
		GNOMEUIINFO_END
	};

	static GnomeUIInfo menubar_uiinfo[] =
	{
		GNOMEUIINFO_MENU_FILE_TREE (file_menu_uiinfo),
		GNOMEUIINFO_MENU_EDIT_TREE (edit_menu_uiinfo),
		{
			GNOME_APP_UI_SUBTREE, N_("_Mark"),
			NULL,
			mark_menu_uiinfo, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_MENU_VIEW_TREE (view_menu_uiinfo),
		GNOMEUIINFO_MENU_SETTINGS_TREE (options_menu_uiinfo),
/*		
		{
			GNOME_APP_UI_SUBTREE, N_("_CVS"),
			NULL,
			cvs_menu_uiinfo, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
*/		
		{
			GNOME_APP_UI_SUBTREE, N_("F_TP"),
			NULL,
			ftp_menu_uiinfo, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_MENU_HELP_TREE (help_menu_uiinfo),
		GNOMEUIINFO_END
	};

	static GnomeUIInfo toolbar_uiinfo[] =
	{
		/*
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("cvs update"),
			cvs_update, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME,
			"gnome-commander/menu_cvs_update.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("cvs commit"),
			cvs_commit, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME,
			"gnome-commander/menu_cvs_commit.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("cvs diff"),
			cvs_diff, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME,
			"gnome-commander/menu_cvs_diff.xpm",
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("cvs log"),
			cvs_log, NULL, NULL,
			GNOME_APP_PIXMAP_FILENAME,
			"gnome-commander/menu_cvs_log.xpm",
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		*/
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("ftp connect"),
			ftp_connect, NULL, NULL,
			GNOME_APP_PIXMAP_DATA, menu_ftp_connect_xpm,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, N_(NULL),
			N_("ftp disconnect"),
			ftp_disconnect, NULL, NULL,
			GNOME_APP_PIXMAP_DATA, menu_ftp_disconnect_xpm,
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};

	int i;

	/* It is very important that this global variable gets assigned here so that
	 * child widgets to this window can use that variable when initializing
	 */
	main_win = GNOME_CMD_MAIN_WIN (mw);

	i = 0;
	while (file_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (file_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| file_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			file_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (mark_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (mark_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| mark_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			mark_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (edit_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (edit_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| edit_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			edit_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (view_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (view_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| view_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			view_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (options_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (options_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| options_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			options_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}
/*	
	i = 0;
	while (cvs_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (cvs_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| edit_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			cvs_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}
*/
	i = 0;
	while (ftp_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (ftp_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| ftp_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			ftp_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (help_menu_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (help_menu_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| help_menu_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			help_menu_uiinfo[i].user_data = mw;
		}
		i++;
	}

	i = 0;
	while (toolbar_uiinfo[i].type != GNOME_APP_UI_ENDOFINFO) {
		if (toolbar_uiinfo[i].type == GNOME_APP_UI_ITEM
			|| toolbar_uiinfo[i].type == GNOME_APP_UI_ITEM_CONFIGURABLE) {
			toolbar_uiinfo[i].user_data = mw;
		}
		i++;
	}
	
	mw->priv = g_new (GnomeCmdMainWinPrivate, 1);
	mw->priv->current_fs = LEFT;
	mw->priv->accel_group = gtk_accel_group_new ();

	gnome_cmd_data_get_position (&main_win_width, &main_win_height);

	gtk_object_set_data (GTK_OBJECT (main_win), "main_win", main_win);
	gtk_window_set_title (GTK_WINDOW (main_win), "GNOME Commander");
	gtk_window_set_position (GTK_WINDOW (main_win), GTK_WIN_POS_CENTER);
	gtk_window_set_policy (GTK_WINDOW (main_win), TRUE, TRUE, FALSE);
	gtk_widget_set_usize (GTK_WIDGET (main_win), main_win_width, main_win_height);

	mw->priv->vbox = gtk_vbox_new (FALSE, 0);
	gtk_widget_ref (mw->priv->vbox);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "vbox", mw->priv->vbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->vbox);
	gnome_app_set_contents (GNOME_APP (main_win), mw->priv->vbox);
	gnome_app_create_menus (GNOME_APP (mw), menubar_uiinfo);
	gnome_app_create_toolbar (GNOME_APP (mw), toolbar_uiinfo);

	mw->priv->vpaned = gtk_vpaned_new ();
	gtk_widget_ref (mw->priv->vpaned);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "paned", mw->priv->vpaned,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->vpaned);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->vpaned, TRUE, TRUE, 0);
	gtk_paned_set_handle_size (GTK_PANED (mw->priv->vpaned), 8);
	gtk_paned_set_gutter_size (GTK_PANED (mw->priv->vpaned), 10);
	
	mw->priv->hpaned = gtk_hpaned_new ();
	gtk_widget_ref (mw->priv->hpaned);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "paned", mw->priv->hpaned,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->hpaned);
	gtk_paned_pack1 (GTK_PANED (mw->priv->vpaned), mw->priv->hpaned,
					 TRUE, TRUE);
	gtk_paned_set_handle_size (GTK_PANED (mw->priv->hpaned), 8);
	gtk_paned_set_gutter_size (GTK_PANED (mw->priv->hpaned), 10);
//	gtk_paned_set_position (GTK_PANED (mw->priv->hpaned), 320);

	mw->priv->zvt_term = gnome_cmd_zvt_term_new ();
	gtk_widget_ref (mw->priv->zvt_term);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "zvt_term", mw->priv->zvt_term,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->zvt_term);
	gtk_paned_pack2 (GTK_PANED (mw->priv->vpaned), mw->priv->zvt_term,
					 FALSE, TRUE);

	mw->priv->buttonbox = gtk_hbox_new (TRUE, 0);
	gtk_widget_ref (mw->priv->buttonbox);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "buttonbox", mw->priv->buttonbox,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->buttonbox);
	gtk_box_pack_start (GTK_BOX (mw->priv->vbox), mw->priv->buttonbox, FALSE, FALSE, 0);

	mw->priv->help_btn = gtk_button_new_with_label (_("F1 Help"));
	gtk_widget_ref (mw->priv->help_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "help_btn", mw->priv->help_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->help_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->help_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->help_btn, "clicked", mw->priv->accel_group,
								GDK_F1, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->run_btn = gtk_button_new_with_label (_("F2 Run"));
	gtk_widget_ref (mw->priv->run_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "run_btn", mw->priv->run_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->run_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->run_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->run_btn, "clicked", mw->priv->accel_group,
								GDK_F2, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->view_btn = gtk_button_new_with_label (_("F3 View"));
	gtk_widget_ref (mw->priv->view_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "view_btn", mw->priv->view_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->view_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->view_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->view_btn, "clicked", mw->priv->accel_group,
								GDK_F3, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->edit_btn = gtk_button_new_with_label (_("F4 Edit"));
	gtk_widget_ref (mw->priv->edit_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "edit_btn", mw->priv->edit_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->edit_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->edit_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->edit_btn, "clicked", mw->priv->accel_group,
								GDK_F4, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->copy_btn = gtk_button_new_with_label (_("F5 Copy"));
	gtk_widget_ref (mw->priv->copy_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "copy_btn", mw->priv->copy_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->copy_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->copy_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->copy_btn, "clicked", mw->priv->accel_group,
								GDK_F5, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->move_btn = gtk_button_new_with_label (_("F6 Move"));
	gtk_widget_ref (mw->priv->move_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "move_btn", mw->priv->move_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->move_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->move_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->move_btn, "clicked", mw->priv->accel_group,
								GDK_F6, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->mkdir_btn = gtk_button_new_with_label (_("F7 Mkdir"));
	gtk_widget_ref (mw->priv->mkdir_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "mkdir_btn", mw->priv->mkdir_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->mkdir_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->mkdir_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->mkdir_btn, "clicked", mw->priv->accel_group,
								GDK_F7, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->delete_btn = gtk_button_new_with_label (_("F8 Delete"));
	gtk_widget_ref (mw->priv->delete_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "delete_btn", mw->priv->delete_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->delete_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->delete_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->delete_btn, "clicked", mw->priv->accel_group,
								GDK_F8, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->find_btn = gtk_button_new_with_label (_("F9 Search"));
	gtk_widget_ref (mw->priv->find_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "find_btn", mw->priv->find_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->find_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->find_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->find_btn, "clicked", mw->priv->accel_group,
								GDK_F9, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->quit_btn = gtk_button_new_with_label (_("F10 Quit"));
	gtk_widget_ref (mw->priv->quit_btn);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "quit_btn", mw->priv->quit_btn,
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->quit_btn);
	gtk_box_pack_start (GTK_BOX (mw->priv->buttonbox), mw->priv->quit_btn, TRUE, TRUE, 0);
	gtk_widget_add_accelerator (mw->priv->quit_btn, "clicked", mw->priv->accel_group,
								GDK_F10, 0,
								GTK_ACCEL_VISIBLE);

	mw->priv->file_selector[LEFT] = gnome_cmd_file_selector_new ();
	gtk_widget_ref (mw->priv->file_selector[LEFT]);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "left_file_selector",
							  mw->priv->file_selector[LEFT],
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->file_selector[LEFT]);
	gtk_paned_pack1 (GTK_PANED (mw->priv->hpaned), mw->priv->file_selector[LEFT],
					 TRUE, TRUE);

	mw->priv->file_selector[RIGHT] = gnome_cmd_file_selector_new ();
	gtk_widget_ref (mw->priv->file_selector[RIGHT]);
	gtk_object_set_data_full (GTK_OBJECT (main_win), "left_file_selector",
							  mw->priv->file_selector[RIGHT],
							  (GtkDestroyNotify) gtk_widget_unref);
	gtk_widget_show (mw->priv->file_selector[RIGHT]);
	gtk_paned_pack2 (GTK_PANED (mw->priv->hpaned), mw->priv->file_selector[RIGHT],
					 TRUE, TRUE);

	gnome_cmd_connection_create_local ();

	gnome_cmd_file_selector_set_connection (gnome_cmd_main_win_get_left_fs (mw),
											gnome_cmd_connection_get_local ());
	gnome_cmd_file_selector_set_connection (gnome_cmd_main_win_get_right_fs (mw),
											gnome_cmd_connection_get_local ());

	gtk_widget_set_style (GTK_BIN (mw->priv->help_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->run_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->view_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->edit_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->copy_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->move_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->mkdir_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->delete_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->find_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->quit_btn)->child, main_style);

	gtk_button_set_relief (GTK_BUTTON (mw->priv->help_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->run_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->view_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->edit_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->copy_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->move_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->mkdir_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->delete_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->find_btn), GTK_RELIEF_HALF);
	gtk_button_set_relief (GTK_BUTTON (mw->priv->quit_btn), GTK_RELIEF_HALF);

	
	gtk_signal_connect (GTK_OBJECT (main_win), "realize",
						GTK_SIGNAL_FUNC (on_main_win_realize), mw);
	gtk_signal_connect (GTK_OBJECT (main_win), "size_allocate",
						GTK_SIGNAL_FUNC (on_main_win_size_allocate), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->help_btn), "clicked",
						GTK_SIGNAL_FUNC (on_help_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->run_btn), "clicked",
						GTK_SIGNAL_FUNC (on_run_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->view_btn), "clicked",
						GTK_SIGNAL_FUNC (on_view_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->edit_btn), "clicked",
						GTK_SIGNAL_FUNC (on_edit_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->copy_btn), "clicked",
						GTK_SIGNAL_FUNC (on_copy_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->move_btn), "clicked",
						GTK_SIGNAL_FUNC (on_move_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->mkdir_btn), "clicked",
						GTK_SIGNAL_FUNC (on_mkdir_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->delete_btn), "clicked",
						GTK_SIGNAL_FUNC (on_delete_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->find_btn), "clicked",
						GTK_SIGNAL_FUNC (on_search_clicked), mw);
	gtk_signal_connect (GTK_OBJECT (mw->priv->quit_btn), "clicked",
						GTK_SIGNAL_FUNC (on_quit_clicked), mw);


	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list),
		"resize_column", GTK_SIGNAL_FUNC (on_fs_list_resize_column),
		GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list),
		"resize_column", GTK_SIGNAL_FUNC (on_fs_list_resize_column),
		GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list),
		"button_press_event", GTK_SIGNAL_FUNC (on_left_fs_select), mw);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list),
		"button_press_event", GTK_SIGNAL_FUNC (on_right_fs_select), mw);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT])->list),
		"key-press-event", GTK_SIGNAL_FUNC (on_fs_keypressed), mw);
	gtk_signal_connect (
		GTK_OBJECT (GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT])->list),
		"key-press-event", GTK_SIGNAL_FUNC (on_fs_keypressed), mw);
	gtk_signal_connect (
		GTK_OBJECT (mw), "size-allocate",
		GTK_SIGNAL_FUNC (on_size_allocate), mw);
	

	gtk_window_add_accel_group (GTK_WINDOW (main_win), mw->priv->accel_group);
}


GtkType
gnome_cmd_main_win_get_type         (void)
{
	static GtkType mw_type = 0;

	if (mw_type == 0)
	{
		GtkTypeInfo mw_info =
		{
			"GnomeCmdMainWin",
			sizeof (GnomeCmdMainWin),
			sizeof (GnomeCmdMainWinClass),
			(GtkClassInitFunc) class_init,
			(GtkObjectInitFunc) init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		mw_type = gtk_type_unique (gnome_app_get_type (), &mw_info);
	}
	return mw_type;
}



GtkWidget*
gnome_cmd_main_win_new              ()
{
	GnomeCmdMainWin *mw;

	mw = gtk_type_new (gnome_cmd_main_win_get_type ());

	return GTK_WIDGET (mw);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_active_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[mw->priv->current_fs]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_inactive_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[!mw->priv->current_fs]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_left_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[LEFT]);
}


GnomeCmdFileSelector*
gnome_cmd_main_win_get_right_fs         (GnomeCmdMainWin *mw)
{
	return GNOME_CMD_FILE_SELECTOR (mw->priv->file_selector[RIGHT]);
}


void
gnome_cmd_main_win_update_style          (GnomeCmdMainWin *mw)
{
	g_return_if_fail (mw != NULL);
	g_return_if_fail (mw->priv != NULL);
	
	gtk_widget_set_style (GTK_BIN (mw->priv->help_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->run_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->view_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->edit_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->copy_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->move_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->mkdir_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->delete_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->find_btn)->child, main_style);
	gtk_widget_set_style (GTK_BIN (mw->priv->quit_btn)->child, main_style);

	IMAGE_clear_mime_cache ();

	gnome_cmd_file_selector_update_style (gnome_cmd_main_win_get_left_fs (mw));
	gnome_cmd_file_selector_update_style (gnome_cmd_main_win_get_right_fs (mw));

	gnome_cmd_zvt_term_update (GNOME_CMD_ZVT_TERM (mw->priv->zvt_term));	
}


void
gnome_cmd_main_win_hide_embterm          (GnomeCmdMainWin *mw)
{
	g_return_if_fail (mw != NULL);
	g_return_if_fail (mw->priv != NULL);
	
	gtk_paned_set_position (GTK_PANED (mw->priv->vpaned),
							GTK_WIDGET (mw)->allocation.height);	
	gnome_cmd_file_selector_set_active (gnome_cmd_main_win_get_active_fs (mw),
										TRUE);	
}


void
gnome_cmd_main_win_show_embterm          (GnomeCmdMainWin *mw)
{
	g_return_if_fail (mw != NULL);
	g_return_if_fail (mw->priv != NULL);
	
	gtk_paned_set_position (GTK_PANED (mw->priv->vpaned), -1);
	gnome_cmd_zvt_term_set_focus (GNOME_CMD_ZVT_TERM (mw->priv->zvt_term));
}



