#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# DESCRIPTION:
# A wxPython based user interface driver for GNUe forms.
#
# NOTES:
#
# $Id: _base.py 6851 2005-01-03 20:59:28Z jcater $

from wxPython.wx import *

from gnue.common import events
from types import UnicodeType
from gnue.forms.GFForm import *
from gnue.forms.uidrivers._base.widgets._base import *
from gnue.forms.uidrivers.wx.common import *


#############################################################################
#############################################################################
#
# UI Objects
#
# A set of standard UI objects that allow us to move more code into UIbase
# These map generic functionality to the specific UI widget set in use by
# this driver.
#
#############################################################################
#############################################################################

#
# UIHelper
#
# Provides a UI widget set of std functions to reduce the number of functions
# require in each UIclass listed below.
#
class UIHelper(UIWidget):
  """
  Implements the common behaviour of WX widgets
  """

  # --------------------------------------------------------------------------
  # Add a widget to the cross reference tables
  # --------------------------------------------------------------------------
  def _addToCrossRef(self, widget,gfobject, uiobject):
    id = widget.GetId()
    self._uiDriver._IdToTkObj[id] = widget
    self._uiDriver._IdToGFObj[id] = gfobject
    self._uiDriver._IdToUIObj[id] = uiobject
    # BAH! This isn't working
    ##widget.__origBackgroundColor = widget.GetBackgroundColour()
    widget.__origBackgroundColor = wxWHITE

  # --------------------------------------------------------------------------
  # Remove a widget from the cross reference tables
  # --------------------------------------------------------------------------
  def _deleteFromCrossRef(self, widget, object):
    id = widget.GetId()
    try:
      del self._uiDriver._IdToTkObj[id]
      del self._uiDriver._IdToGFObj[id]
      del self._uiDriver._IdToUIObj[id]
    except:
      pass

  # --------------------------------------------------------------------------
  # Show all ui-widgets managed by this object
  # --------------------------------------------------------------------------
  def show(self):
    """
    Calls the show() on all wx-widgets managed by this instance.
    This is needed if a row-count greater than 0 is given.
    """
    for widget in self.widgets:
      widget.Show(1)
      widget.Raise()

  # --------------------------------------------------------------------------
  # Hide all ui-widgets managed by this object
  # --------------------------------------------------------------------------
  def hide(self):
    """
    Calls the hide() on all wx-widgets managed by this instance.
    This is needed if a row-count greater than 0 is given.
    """
    for widget in self.widgets:
      widget.Show(0)

  def showModal(self):
    for widget in self.widgets:
      widget.ShowModal()

  def destroy(self):
    for widget in self.widgets:
      self.Destroy()


  # --------------------------------------------------------------------------
  # Set the focus to a given widget
  # --------------------------------------------------------------------------
  def indexedFocus(self, index):
    """
    Sets the focus to the wx widget specified by index.
    The focused widget will have a highlight color if it is set in gnue.conf.
    """
    widget = self.widgets[index]
    widget.__focused = True

    #Make the focused widget's color different
    #if focus color is set in gnue.conf.
    #We store the widget's original color so we can set it back to normal
    #when it loses focus.
    #TODO: CheckButtons don't get colored...
    if self._uiDriver._focusColour:
      try:
        widget.__color
      except:
        widget.__color = widget.GetBackgroundColour()
      widget.SetBackgroundColour(self._uiDriver._focusColour)
    widget.SetFocus()


  # --------------------------------------------------------------------------
  # On lose of the focus we un-select ComboBox-Entries
  # --------------------------------------------------------------------------
  def loseFocus(self):
    """
    Releases focus from the widget that currently had it.
    Resets the widget's color to normal if focus highlighting is used.
    """
    widget = self.widgets[self._uiDriver._uiFocusIndex]
    widget.__focused = False
    try:
      widget.__color
      widget.SetBackgroundColour(widget.__color)
    except:
      pass


  # --------------------------------------------------------------------------
  # Set the value of a widget
  # --------------------------------------------------------------------------
  def setValue(self, value, index=0, enabled=True):
    """
    Sets the value of a widget and optionally enables or disables the widget.
    """
    # These must be here or dropdown style controls
    # will get events they shouldn't and break.
    widget = self.widgets[index]
    widget.SetEvtHandlerEnabled(FALSE)

    # Check if foreign key changed
    object = self._uiDriver._IdToGFObj[widget.GetId()]
    try:
      if (object.style == "dropdown" or object.style == "listbox") and \
          not object._field._allowedValues == widget._origAllowedValues:
        try:
          # Not in wx 2.2.x
          widget.Freeze()
        except AttributeError:
          pass
        widget._origAllowedValues = object._field._allowedValues
        widget.Clear()
        for value in object._field._allowedValuesDescr:
          widget.Append(wxEncode(value))
        widget.SetValue("")
        try:
          # Not in wx 2.2.x
          widget.Thaw()
        except AttributeError:
          pass
    except AttributeError:
      pass

    widget.SetValue(wxEncode(value))
    widget.Enable(enabled)
    widget.SetEvtHandlerEnabled(TRUE)

    if enabled:
      widget.__color = color = widget.__origBackgroundColor
      try:
        if widget.__focused and self._uiDriver._focusColour:
          color = self._uiDriver._focusColour
      except AttributeError:
        pass
    else:
      color = self._uiDriver._disabledColour

    widget.SetBackgroundColour(color)
    widget.Refresh()


  # --------------------------------------------------------------------------
  # Set the cursor's location in a widget
  # --------------------------------------------------------------------------
  def setCursorPosition(self, position, index=0):
    """
    Sets the cursor's position to the given location inside a capable widget.
    """
    try:
      self.widgets[index].SetInsertionPoint(position)
    except AttributeError:
      pass  # For label-style entries

  # --------------------------------------------------------------------------
  # Set the selection inside a widget
  # --------------------------------------------------------------------------
  def setSelectedArea(self, selection1, selection2, index=0):
    """
    Sets the selection start/end inside a capable widget.
    """
    try:
      self.widgets[index].SetSelection(selection1, selection2)
    except (AttributeError, TypeError):
      pass  # For label-style & dropdown entries


  # --------------------------------------------------------------------------
  # Create a new widget and add it to the cross reference table
  # --------------------------------------------------------------------------
  def createWidget(self, event, spacer):
    """
    Creates a new WX widget and adds it to the cross reference table.
    """
    object = event.object
    if hasattr(object,'Char__y'):
      self.itemY = (object.Char__y + spacer + (spacer * object._gap)) * event.widgetHeight
    else:
      self.itemY = -1

    newWidget = self._createWidget(event, spacer)
    initFont(newWidget)
    if event.initialize:
      self._addToCrossRef(newWidget,event.object,self)
    return newWidget

  # --------------------------------------------------------------------------
  # Make sure we leave everything clean and tidy
  # --------------------------------------------------------------------------
  def cleanup(self, object):
    """
    Cleans up everything, ie deletes all widgets in this instance.
    """
    for widget in self.widgets[:]:
      id = widget.GetId()
      try:
        del self._uiDriver._IdToTkObj[id]
        del self._uiDriver._IdToGFObj[id]
        del self._uiDriver._IdToUIObj[id]
      except KeyError:
        pass
      self.widgets.pop(0)
      self._deleteFromCrossRef(widget, object)
